/*!
  \file
  \brief V~[^p̃EBhE

  \author Satofumi KAMIMURA

  $Id: SimulationWindow.cpp 1239 2009-08-22 08:54:02Z satofumi $
*/

#include "SimulationWindow.h"
#include "SimulationMain.h"
#include "TicksControlWidget.h"
#include "DrawEnvironmentWidget.h"
#include "DeviceManager.h"
#include "ExecutionType.h"
#include "fileExist.h"
#include <QTimer>
#include <QShortcut>
#include <QSettings>
#include <QMessageBox>

using namespace qrk;
using namespace std;


namespace
{
  enum {
    RedrawFps = 20,             // [msec]
  };

  const char* Organization = "Hyakuren soft.";
  const char* Application = "QRK Simulator";


  class ArgsInformation
  {
    typedef enum {
      Normal,
      Recording,
      Playback,
    } Mode;

  public:
    bool simulation_;
    Mode mode_;


    ArgsInformation(void) : simulation_(false), mode_(Normal)
    {
    }
  };
}


struct SimulationWindow::pImpl
{
  SimulationWindow* window_;
  DrawEnvironmentWidget draw_environment_widget_;
  TicksControlWidget ticks_control_widget_;

  DeviceManager device_manager_;
  SimulationMain simulation_main_;

  QTimer redraw_timer_;


  pImpl(SimulationWindow* window,
        MainFunction* function, int argc, char *argv[])
    : window_(window), simulation_main_(function, argc, argv)
  {
    ArgsInformation args;

    // V~[VݒǂݏoĔf
    parseArgs(args, argc, argv);
    ExecutionType* execution_type = ExecutionType::object();
    if (args.simulation_) {
      execution_type->setType(ExecutionType::Simulation);

    } else if (args.mode_ == ArgsInformation::Playback) {
      execution_type->setType(ExecutionType::Playback);

    } else if (args.mode_ == ArgsInformation::Recording) {
      execution_type->setType(ExecutionType::Recording);

    } else {
      execution_type->setType(ExecutionType::RealDevice);
    }

    // CvO̎s
    simulation_main_.start();
  }


  void initializeForm(void)
  {
    window_->dummy_label_->hide();
    window_->main_layout_->addWidget(&draw_environment_widget_);

    // TicksControlWidget
    connect(&ticks_control_widget_, SIGNAL(play()),
            window_, SLOT(playPressed()));
    connect(&ticks_control_widget_, SIGNAL(pause()),
            window_, SLOT(pausePressed()));
    connect(&ticks_control_widget_, SIGNAL(moreFaster()),
            window_, SLOT(moreFasterPressed()));
    connect(&ticks_control_widget_, SIGNAL(moreFaster()),
            window_, SLOT(moreSlowerPressed()));

    // Ip̃VOiڑ
    connect(&ticks_control_widget_, SIGNAL(quit()),
            window_, SLOT(quitHandler()));
    connect(window_->action_quit_, SIGNAL(triggered()),
            window_, SLOT(quitHandler()));

    // About ̕\
    connect(window_->action_about_, SIGNAL(triggered()),
            window_, SLOT(aboutApplication()));

    // TuEBhE̕\/\
    connect(&ticks_control_widget_, SIGNAL(closed()),
            window_, SLOT(ticksControlClosed()));
    connect(window_->action_ticks_control_, SIGNAL(toggled(bool)),
            window_, SLOT(ticksControlChecked(bool)));

    // `^C}[̓Jn
    connect(&redraw_timer_, SIGNAL(timeout()), window_, SLOT(redraw()));
    redraw_timer_.setInterval(RedrawFps);
    redraw_timer_.start();
  }


  void loadSettings(void)
  {
    QSettings settings(Organization, Application);

    // EBhEʒu
    window_->restoreGeometry(settings.value("window_geometry").toByteArray());
    ticks_control_widget_.
      restoreGeometry(settings.value("ticks_geometry").toByteArray());

    // TuEBhE̕\/\
    bool ticks_control_show =
      settings.value("ticks_control_showo", false).toBool();
    ticks_control_widget_.setVisible(ticks_control_show);
    window_->action_ticks_control_->setChecked(ticks_control_show);
  }


  void saveSettings(void)
  {
    QSettings settings(Organization, Application);

    // EBhEʒu
    settings.setValue("window_geometry", window_->saveGeometry());
    settings.setValue("ticks_geometry", ticks_control_widget_.saveGeometry());

    // TuEBhE̕\/\
    settings.setValue("ticks_control_showo", ticks_control_widget_.isVisible());
  }


  void closeAll(void)
  {
    ticks_control_widget_.close();
    // !!!
  }
};


SimulationWindow::SimulationWindow(MainFunction* function,
                                   int argc, char *argv[])
  : pimpl(new pImpl(this, function, argc, argv))
{
  setupUi(this);
  pimpl->initializeForm();
  pimpl->loadSettings();

  // Ctrl-q, Ctrl-w, Alt-F4 ŏI
  (void) new QShortcut(Qt::ALT + Qt::Key_F4, this, SLOT(close()));
  (void) new QShortcut(Qt::CTRL + Qt::Key_W, this, SLOT(close()));
}


SimulationWindow::~SimulationWindow(void)
{
}


void SimulationWindow::parseArgs(ArgsInformation& args, int argc, char *argv[])
{
  // !!! -v, --version
  // !!! -h, --help
  for (int i = 1; i < argc; ++i) {
    char* token = argv[i];
    if ((! strcmp("-s", token)) || (! strcmp("--simulation", token))) {
      args.simulation_ = true;
    } else if ((! strcmp("-p", token)) || (! strcmp("--playback", token))) {
      args.mode_ = ArgsInformation::Playback;
    } else if ((! strcmp("-r", token)) || (! strcmp("--recording", token))) {
      args.mode_ = ArgsInformation::Recording;
    }
  }

  // ȂƂ "simulation_only.txt" t@C݂
  // V~[VƂē삳
  if ((args.mode_ == ArgsInformation::Normal) &&
      fileExist("simulation_only.txt")) {
    args.simulation_ = true;
  }

}


void SimulationWindow::quitHandler(void)
{
  close();
}


void SimulationWindow::closeEvent(QCloseEvent* event)
{
  static_cast<void>(event);
  pimpl->saveSettings();

  pimpl->simulation_main_.event_scheduler_.terminate();

  pimpl->simulation_main_.terminate();
  pimpl->simulation_main_.wait(100);

  pimpl->device_manager_.terminate();

  pimpl->closeAll();
}


void SimulationWindow::playPressed(void)
{
  pimpl->simulation_main_.event_scheduler_.play();
}


void SimulationWindow::pausePressed(void)
{
  pimpl->simulation_main_.event_scheduler_.pause();
}


void SimulationWindow::moreFasterPressed(void)
{
  pimpl->simulation_main_.event_scheduler_.moreFaster();
}


void SimulationWindow::moreSlowerPressed(void)
{
  pimpl->simulation_main_.event_scheduler_.moreSlower();
}


void SimulationWindow::redraw(void)
{
  pimpl->simulation_main_.event_scheduler_.lock();
  pimpl->draw_environment_widget_.redraw();
  pimpl->simulation_main_.event_scheduler_.unlock();
}


void SimulationWindow::aboutApplication(void)
{
  QMessageBox::about(this, tr("About QRK Simulator"),
                     tr("<h2>QRK Simulator</h2>"
                        "<p>QRK Robot & Sensor simulator</p>"
                        "<p>Report bugs to "
                        "&lt;satofumi@users.sourceforge.jp&gt;</p>"));
}


void SimulationWindow::ticksControlClosed(void)
{
  action_ticks_control_->setChecked(false);
}


void SimulationWindow::ticksControlChecked(bool checked)
{
  pimpl->ticks_control_widget_.setVisible(checked);
}
