/*!
  \file
  \brief SCIP vgR

  \author Satofumi KAMIMURA

  $Id: ScipHandler.cpp 1219 2009-08-14 02:59:46Z satofumi $
*/

#include "ScipHandler.h"
#include "RangeFinderParameter.h"
#include "Connection.h"
#include "ConnectionUtils.h"
#include "ticks.h"
#include "log_printf.h"
#include "delay.h"
#include <cstring>

using namespace qrk;
using namespace std;


namespace
{
  /*!
    \brief Mf[^̃^Cv
  */
  typedef enum {
    TypeUnknown,                //!< s
    QT,                         //!< ~R}hM
    GD,                         //!< f[^擾( 3 byte \)
    GS,                         //!< f[^擾( 2 byte \)
    MD,                         //!< f[^擾( 3 byte \)
    MS,                         //!< f[^擾( 2 byte \)
    ME,                         //!< M R}h̉
    Mx_Reply,                   //!< Mx R}h̍ŏ̉bZ[W
    InvalidData,                //!< Mf[^̃G[
  } CaptureType;


  /*!
    \brief f[^擾ݒ
  */
  class CaptureSettings
  {
  public:
    CaptureType type;           //!< Mf[^̎
    int error_code;             //!< G[R[h
    long timestamp;             //!< ^CX^v [msec]
    int capture_first;          //!< 擾JnCfbNX
    int capture_last;           //!< 擾ICfbNX
    int skip_lines;             //!< ԈC
    int skip_frames;            //!< f[^擾Ԋu
    int remain_times;           //!< f[^擾̎c
    int data_byte;              //!< f[^oCg


    CaptureSettings(void)
      : type(TypeUnknown), error_code(-1), timestamp(-1),
        capture_first(-1), capture_last(-1),
        skip_lines(-1), skip_frames(-1), remain_times(-1), data_byte(-1)
    {
    }
  };


  void clearReceived(vector<long>& data, CaptureType& type,
                     int& line_count, int& timeout,
                     string& remain_string, string& left_packet_data,
                     int& store_times)
  {
    data.clear();
    type = TypeUnknown;
    line_count = 0;
    timeout = 0;
    remain_string.clear();
    left_packet_data.clear();
    store_times = 0;
  }
}


struct ScipHandler::pImpl
{
  enum {
    ContinuousTimeout = 100,
    FirstTimeout = 1000,

    BufferSize = 64 + 1 + 1 + 16, // f[^ + `FbNT + s + \

    ResponseTimeout = -1,
    MismatchResponse = -2,
    SendFail = -3,
    ChecksumFail = -4,
    Scip11Response = -14,
  };


  typedef enum {
    Laser_Unknown = 0,
    Laser_On,
    Laser_Off,
  } LaserState;


  string error_message_;
  Connection* connection_;
  LaserState laser_state_;
  bool mx_capturing_;

  bool isPreCommand_QT_;


  pImpl(void)
    : error_message_("not connected."), connection_(NULL),
      laser_state_(Laser_Unknown), mx_capturing_(false),
      isPreCommand_QT_(false)
  {
  }


  bool connect(const char* device, long baudrate)
  {
    if (! connection_->connect(device, baudrate)) {
      error_message_ = connection_->what();
      return false;
    }

    long try_baudrates[] = { 115200, 19200, 38400, };
    size_t try_size = sizeof(try_baudrates) / sizeof(try_baudrates[0]);

    // ڑ{[[gz̐擪vfƓꊷ
    for (size_t i = 1; i < try_size; ++i) {
      if (baudrate == try_baudrates[i]) {
        swap(try_baudrates[0], try_baudrates[i]);
        break;
      }
    }

    // w̃{[[gŐڑAԂ邩ǂ
    for (size_t i = 0; i < try_size; ++i) {

      // zXg̃{[[gύX
      if (! connection_->setBaudrate(try_baudrates[i])) {
        error_message_ = connection_->what();
        return false;
      }

      // O񕪂̎MpPbgǂݎ̂Ă
      connection_->clear();

      // QT ̔s
      int return_code = -1;
      char qt_expected_response[] = { 0, -1 };
      // return_code g߁AsetLaserOutput() p QT 𑗐M
      if (response(return_code, "QT\n", qt_expected_response)) {
        laser_state_ = Laser_Off;
        return changeBothBaudrate(baudrate);

      } else if (return_code == ResponseTimeout) {
        // {[[gĂāAʐMłȂƂ݂Ȃ
        error_message_ = "baudrate is not detected.";
        continue;

      } else if (return_code == MismatchResponse) {
        // MD/MS ̉Ƃ݂ȂAMf[^ǂݔ΂
	connection_->clear();
        skip(connection_, ContinuousTimeout);
        return changeBothBaudrate(baudrate);

      } else if (return_code == Scip11Response) {
        // SCIP1.1 vgȐꍇ̂݁ASCIP2.0 𑗐M
        char scip20_expected_response[] = { 0, -1 };
        if (! response(return_code, "SCIP2.0\n", scip20_expected_response)) {
          error_message_ =
            "SCIP1.1 protocol is not supported. Please update URG firmware.";
          return false;
        }
        return changeBothBaudrate(baudrate);

      } else if (return_code == 0xE) {
        // TM [hƂ݂ȂATM2 𔭍s
        char tm2_expected_response[] = { 0, -1 };
        if (response(return_code, "TM2\n", tm2_expected_response)) {
          laser_state_ = Laser_Off;
          return changeBothBaudrate(baudrate);
        }
      }
    }

    return false;
  }


  bool response(int& return_code, const char send_command[],
                char expected_response[],
                vector<string>* lines = NULL)
  {
    return_code = -1;
    if (! connection_) {
      error_message_ = "no connection.";
      return false;
    }

    size_t send_size = strlen(send_command);
    int actual_send_size = connection_->send(send_command, send_size);
    if (strncmp(send_command, "QT\n", send_size)) {
      isPreCommand_QT_ = false;
    }
    if (actual_send_size != static_cast<int>(send_size)) {
      return_code = SendFail;
      return false;
    }

    // GR[obN̎M
    char buffer[BufferSize];
    int recv_size = readline(connection_, buffer, BufferSize, FirstTimeout);
    if (recv_size < 0) {
      error_message_ = "response timeout: "
        + string(send_command).substr(0, strlen(send_command) - 1);
      return_code = ResponseTimeout;
      return false;
    }

    // VAڑŃ{[[gύX 0x00 ́AOƂ
    if (! ((recv_size == 1) && (buffer[0] == 0x00))) {
      if ((recv_size != static_cast<int>(send_size - 1)) ||
          (strncmp(buffer, send_command, recv_size))) {
        error_message_ = "mismatch response: " + string(buffer);
        return_code = MismatchResponse;
        return false;
      }
    }

    // ̎M
    // SCIP1.1 vgR̉́At return_code Ɋi[
    recv_size = readline(connection_, buffer, BufferSize, ContinuousTimeout);
    if (recv_size < 0) {
      error_message_ = "response timeout.";
      return_code = ResponseTimeout;
      return false;
    }
    if (recv_size == 3) {
      // RȂ΁ASCIP2.0 Ƃ݂Ȃă`FbNTmF
      if (! checkSum(buffer, recv_size - 1, buffer[recv_size - 1])) {
	return_code = ChecksumFail;
	return false;
      }
      buffer[2] = '\0';
      return_code = strtol(buffer, NULL, 16);

    } else if (recv_size == 1) {
      // PȂ΁ASCIP1.1 Ƃ݂Ȃ 16iϊlɕĕԂ
      buffer[1] = '\0';
      return_code = -strtol(buffer, NULL, 16);
    }

    // f[^̈̎M
    // PsǂݏoAŝ݂ȂΏIƂ݂Ȃ
    do {
      recv_size = readline(connection_, buffer, BufferSize, ContinuousTimeout);
      if (lines && (recv_size > 0)) {
        lines->push_back(buffer);
      }
    } while (recv_size > 0);

    for (int i = 0; expected_response[i] != -1; ++i) {
      if (return_code == expected_response[i]) {
        return true;
      }
    }
    return false;
  }


  bool changeBothBaudrate(long baudrate)
  {
    // ɖڕWΏۂ̃{[[glȂ΁AƂ݂Ȃ
    // ̊֐́AScipHandler::connect() AȌłȂƌĂ΂Ȃ
    if (connection_->baudrate() == baudrate) {
      return true;
    }

    // URG ̃{[[gύX
    int pre_ticks = ticks();
    if (! changeBaudrate(baudrate)) {
      return false;
    }

    // VAʐM̏ꍇA{[[gύXAP҂Kv
    int reply_msec = ticks() - pre_ticks;
    delay((reply_msec * 4 / 3) + 10);

    // zXg̃{[[gύX
    return connection_->setBaudrate(baudrate);
  }


  bool changeBaudrate(long baudrate)
  {
    if (! ((baudrate == 19200) || (baudrate == 38400) ||
           (baudrate == 57600) || (baudrate == 115200))) {
      baudrate = 115200;
      //error_message_ = "Invalid baudrate value.";
      //return false;
    }

    // SS 𑗐MAURG ̃{[[gύX
    char send_buffer[] = "SSxxxxxx\n";
    snprintf(send_buffer, 10, "SS%06ld\n", baudrate);
    int return_code = -1;
    // !!! ɐݒΏۂ̃{[[gȀꍇ̖߂l ss_expected... ɒǉ
    char ss_expected_response[] = { 0, 3, 4, -1 };
    if (! response(return_code, send_buffer, ss_expected_response)) {
      error_message_ = "Baudrate change fail.";
      return false;
    }

    return true;
  }


  bool loadParameter(RangeFinderParameter& parameter)
  {
    // PP ̑Mƃf[^̎M
    int return_code = -1;
    char pp_expected_response[] = { 0, -1 };
    vector<string> lines;
    if (! response(return_code, "PP\n", pp_expected_response, &lines)) {
      error_message_ = "PP fail.";
      return false;
    }

    // PP e̊i[
    enum { PP_Lines = 8 };
    if (lines.size() != PP_Lines) {
      error_message_ = "Invalid PP response.";
      return false;
    }

    for (size_t i = 0; i < PP_Lines; ++i) {
      string& line = lines[i];
      size_t n = line.size();

      // PP ͖ ';' ̓`FbNT̕Ɋ܂܂Ȃ
      if ((! checkSum(&line[0], n - 2, line[n - 1], false)) &&
          (! checkSum(&line[0], n - 1, line[n - 1]))) {
        error_message_ = "PP checksum error.";
        return false;
      }
    }

    int modl_length =
      static_cast<int>(lines[RangeFinderParameter::MODL].size());
    // ŏ̃^OƁA`FbNTԂ
    if (modl_length > (5 + 2)) {
      modl_length -= (5 + 2);
    }
    parameter.model = lines[RangeFinderParameter::MODL].substr(5, modl_length);

    parameter.distance_min = substr2int(lines[RangeFinderParameter::DMIN], 5);
    parameter.distance_max = substr2int(lines[RangeFinderParameter::DMAX], 5);
    parameter.area_total = substr2int(lines[RangeFinderParameter::ARES], 5);
    parameter.area_min = substr2int(lines[RangeFinderParameter::AMIN], 5);
    parameter.area_max = substr2int(lines[RangeFinderParameter::AMAX], 5);
    parameter.area_front = substr2int(lines[RangeFinderParameter::AFRT], 5);
    parameter.scan_rpm = substr2int(lines[RangeFinderParameter::SCAN], 5);

    return true;
  }


  int substr2int(const string& line, int from_n, int length = string::npos)
  {
    return atoi(line.substr(from_n, length).c_str());
  }


  bool receiveData(std::vector<long>& data, long* timestamp)
  {
    string remain_string;
    string left_packet_data;
    char buffer[BufferSize];

    int store_times = 0;
    int line_count = 0;
    data.clear();

    CaptureSettings settings;

    error_message_ = "no response.";

    // !!! ȉ̏B֐ׂ
    CaptureType type = TypeUnknown;
    int timeout = FirstTimeout;
    int line_size = 0;
    while ((line_size = readline(connection_,
                                 buffer, BufferSize, timeout)) > 0) {
      //fprintf(stderr, "% 3d: %s\n", line_count, buffer);

      // `FbNT̊mF
      if (line_count != 0) {
        // GR[obNɂ̓`FbNT񂪂Ȃ̂ŁA
        if (! checkSum(buffer, line_size - 1, buffer[line_size - 1])) {
          log_printf("checksum error: %s\n", buffer);

          // ݂邾̃pPbgǂݔ΂
          error_message_ = "invalid packet.";
          clearReceived(data, type, line_count, timeout,
                        remain_string, left_packet_data, store_times);
          continue;
        }
      }

      if (line_count == 0) {

        // GR[obN
        string line = buffer;
        if ((! line.compare(0, 2, "GD")) || (! line.compare(0, 2, "GS"))) {
          if (! parseGdEchoback(settings, line)) {
            break;
          }
          type = (line[1] = 'D') ? GD : GS;
          data.reserve(settings.capture_last + 1);

        } else if ((! line.compare(0, 2, "MD")) ||
                   (! line.compare(0, 2, "MS"))) {
          if (! parseMdEchoback(settings, line)) {
            break;
          }
          type = (line[1] = 'D') ? MD : MS;
          laser_state_ = Laser_On;
          data.reserve(settings.capture_last + 1);

        } else if (! line.compare(0, 2, "ME")) {
          if (! parseMeEchoback(settings, line)) {
            break;
          }
          type = ME;
          laser_state_ = Laser_On;
          data.reserve(settings.capture_last + 1);

        } else if (! line.compare(0, 2, "QT")) {
          settings.remain_times = 0;
          laser_state_ = Laser_Off;
          mx_capturing_ = false;

        } else {
          clearReceived(data, type, line_count, timeout,
                        remain_string, left_packet_data, store_times);
          continue;
        }

      } else if (line_count == 1) {
        // R[h
        if (line_size != 3) {
          clearReceived(data, type, line_count, timeout,
                        remain_string, left_packet_data, store_times);
          continue;
        }
        buffer[2] = '\0';
        settings.error_code = atoi(buffer);

        if ((settings.error_code == 0) &&
            ((type == MD) || (type == MS) || (type == ME))) {
          type = Mx_Reply;
        }

      } else if (line_count == 2) {
        // ^CX^v
        if (timestamp) {
          *timestamp = encode(buffer, 4);
        }
      } else {
        if (line_count == 3) {
          // Mf[^Ȃ擪̗̈A_~[f[^Ŗ߂
          for (int i = 0; i < settings.capture_first; ++i) {
            data.push_back(0);
            if (type == ME) {
              // ME M̂Ƃ́Axf[^߂
              data.push_back(0);
            }
          }
        }
        // f[^
        left_packet_data =
          addLengthData(data, string(buffer),
                        left_packet_data, settings.data_byte,
                        store_times, settings.skip_lines);
      }
      ++line_count;
      timeout = ContinuousTimeout;
    }

    // ME  "܂Ƃ߂鐔" ݒȏ̃f[^ԂoOɑΏ
    size_t expected_n = settings.capture_last * ((type == ME) ? 2 : 1);
    if (expected_n < data.size()) {
      data.erase(data.begin() + expected_n, data.end());
    }

    // type f[^擾̂Ƃ́AɎMAmF
    int expected_store_times =
      settings.capture_last - settings.capture_first + 1;
    if (store_times != expected_store_times) {
      error_message_ = "capture fail.";
      return false;
    }

    return true;
  }


  bool parseGdEchoback(CaptureSettings& settings, const string& line)
  {
    if (line.size() != 12) {
      error_message_ = "Invalid Gx packet has arrived.";
      return false;
    }

    settings.capture_first = substr2int(line, 2, 4);
    settings.capture_last = substr2int(line, 6, 4);
    int skip_lines = substr2int(line, 10, 2);
    settings.skip_lines = (skip_lines == 0) ? 1 : skip_lines;
    settings.data_byte = (line[1] == 'D') ? 3 : 2;

    return true;
  }


  bool parseMdEchoback(CaptureSettings& settings, const string& line)
  {
    if (line.size() != 15) {
      error_message_ = "Invalid Mx packet has arrived.";
      return false;
    }

    settings.capture_first = substr2int(line, 2, 4);
    settings.capture_last = substr2int(line, 6, 4);
    int skip_lines = substr2int(line, 10, 2);
    settings.skip_lines = (skip_lines == 0) ? 1 : skip_lines;
    settings.skip_frames = substr2int(line, 12, 1);
    settings.remain_times = substr2int(line, 13, 2);
    settings.data_byte = (line[1] == 'D') ? 3 : 2;

    if (settings.remain_times == 1) {
      // Ō̃f[^擾ŁA[Uɂ
      // {́Ãf[^擾Ƀ[U͏Ă
      // 1 Ŕ肷ƁA擾񐔂 1 ̂Ƃɂɓ삷
      mx_capturing_ = false;

    } else if (settings.remain_times > 0) {
      mx_capturing_ = true;
    }

    return true;
  }


  bool parseMeEchoback(CaptureSettings& settings, const string& line)
  {
    if (line.size() != 15) {
      error_message_ = "Invalid ME packet has arrived.";
      return false;
    }

    settings.capture_first = substr2int(line, 2, 4);
    settings.capture_last = substr2int(line, 6, 4);
    int skip_lines = substr2int(line, 10, 2);
    settings.skip_lines = (skip_lines == 0) ? 1 : skip_lines;
    settings.skip_frames = substr2int(line, 12, 1);
    settings.remain_times = substr2int(line, 13, 2);
    settings.data_byte = 3;

    if (settings.remain_times == 1) {
      mx_capturing_ = false;

    } else {
      mx_capturing_ = true;
    }

    return true;
  }


  string addLengthData(vector<long>& data,
                       const string& line,
                       const string& left_packet_data,
                       const size_t data_byte, int& store_times,
                       const int skip_lines = 1)
  {
    if (line.empty()) {
      // s̏ꍇA߂
      return left_packet_data;
    }

    // [Bɏ镪
    string left_byte = left_packet_data;

    size_t data_size = (left_byte.size() + (line.size() - 1)) / data_byte;
    size_t n = data_size * data_byte - left_byte.size();
    for (size_t i = 0; i < n; ++i) {
      left_byte.push_back(line[i]);
      if (left_byte.size() >= data_byte) {
        // f[^ɕϊāAi[
        long length = encode(&left_byte[0], data_byte);
        for (int j = 0; j < skip_lines; ++j) {
          data.push_back(length);
          ++store_times;
        }
        left_byte.clear();
      }
    }
    left_byte += line.substr(n, (line.size() - n) - 1);

    return left_byte;
  }


  int send(const char data[], int size)
  {
    if (size >= 2) {
      // MR}h[U𔭌邩𔻒肷
      if ((! strncmp("MD", data, 2)) || (! strncmp("ME", data, 2)) ||
          (! strncmp("MS", data, 2))) {
        laser_state_ = Laser_On;
        mx_capturing_ = true;
        isPreCommand_QT_ = false;
      }
    }
    return connection_->send(data, size);
  }


  bool setLaserOutput(bool on, bool force)
  {
    // !!! force gȂȂ΁A폜

    if (((on == true) && (laser_state_ == Laser_On)) ||
        ((on == false) && (laser_state_ == Laser_Off))) {
      if (! force) {
        // [Uo݂͂̏ԂƓȂΖ߂
        // ݒtO true ̂Ƃ͖߂炸ɐݒs
        return true;
      }
    }
    if ((!on) && isPreCommand_QT_) {
      return false;
    }

    if (on) {
      int return_code = -1;
      char expected_response[] = { 0, -1 };
      if (! response(return_code, "BM\n", expected_response)) {
        error_message_ = "BM fail.";
        return false;
      }
      laser_state_ = Laser_On;
      return true;

    } else {
      // "QT"
      if (! mx_capturing_) {
        // 邽߂ QT ł́A҂ׂ
        int return_code = -1;
        char qt_expected_response[] = { 0, -1 };
        if (! response(return_code, "QT\n", qt_expected_response)) {
          return false;
        }
        laser_state_ = Laser_Off;
        isPreCommand_QT_ = true;
        return true;

      } else {
        // MD 𒆒f邽߂ QT ł́A҂Ă͂ȂȂ
        // ́AMXbhŏ
        connection_->send("QT\n", 3);
        isPreCommand_QT_ = true;
      }

      return true;
    }
  }
};


ScipHandler::ScipHandler(void) : pimpl(new pImpl)
{
}


ScipHandler::~ScipHandler(void)
{
}


bool ScipHandler::checkSum(char* buffer, int size, char actual_sum,
                           bool print_error)
{
  char expected_sum = 0x00;
  for (int i = 0; i < size; ++i) {
    expected_sum += buffer[i];
  }
  expected_sum = (expected_sum & 0x3f) + 0x30;

  if (print_error && (expected_sum != actual_sum)) {
    log_printf("checksum error!\n");
    for (int i = 0; i < size; ++i) {
      log_printf("%c", buffer[i]);
    }
    log_printf(", %d\n", size);
  }

  return (expected_sum == actual_sum) ? true : false;
}


const char* ScipHandler::what(void) const
{
  return pimpl->error_message_.c_str();
}


void ScipHandler::setConnection(Connection* connection)
{
  pimpl->connection_ = connection;
}


Connection* ScipHandler::connection(void) const
{
  return pimpl->connection_;
}


bool ScipHandler::connect(const char* device, long baudrate)
{
  return pimpl->connect(device, baudrate);
}


bool ScipHandler::loadParameter(RangeFinderParameter& parameter)
{
  return pimpl->loadParameter(parameter);
}


int ScipHandler::send(const char data[], int size)
{
  return pimpl->send(data, size);
}


bool ScipHandler::receiveData(std::vector<long>& data, long* timestamp)
{
  return pimpl->receiveData(data, timestamp);
}


void ScipHandler::laserOn(void)
{
  pimpl->setLaserOutput(true, false);
}


void ScipHandler::laserOff(void)
{
  pimpl->setLaserOutput(false, false);
}


long ScipHandler::encode(const char* data, size_t size)
{
  int value = 0;

  for (size_t i = 0; i < size; ++i) {
    value <<= 6;
    value &= ~0x3f;
    value |= data[i] - 0x30;
  }
  return value;
}
