# coding: utf-8
'''
FDTD Solver using Rectlinear Grid on Weave
'''

import numpy, scipy.weave, pylaf

class RectilinearYee:
    def __init__(self,shape=(30,30,30),dt=1.):
        self.shape             = shape
        opts                   = {'dtype':numpy.double}
        dx                     = numpy.ones(shape[0],**opts)
        dy                     = numpy.ones(shape[1],**opts)
        dz                     = numpy.ones(shape[2],**opts)
        self.axes              = [dx,dy,dz]
        self.time              = [- .5 * dt, 0.] # th, te
        self.clear_materials()
        self.clear_fields()
        self.dt                = pylaf.Port(dt)
        self.courant           = pylaf.Port(self.calc_courant())
    def clear_materials(self):
        shape    = self.shape
        self.epr = numpy.ones(shape,dtype=numpy.double)
        self.mur = numpy.ones(shape,dtype=numpy.double)
        self.sgm = numpy.zeros(shape,dtype=numpy.double)
    def clear_fields(self):
        shape, opts = self.shape, {'dtype':numpy.double}
        hx          = numpy.zeros([shape[0]+1,shape[1]  ,shape[2]  ],**opts)
        hy          = numpy.zeros([shape[0]  ,shape[1]+1,shape[2]  ],**opts)
        hz          = numpy.zeros([shape[0]  ,shape[1]  ,shape[2]+1],**opts)
        ex          = numpy.zeros([shape[0]  ,shape[1]+1,shape[2]+1],**opts)
        ey          = numpy.zeros([shape[0]+1,shape[1]  ,shape[2]+1],**opts)
        ez          = numpy.zeros([shape[0]+1,shape[1]+1,shape[2]  ],**opts)
        self.h      = [hx,hy,hz]
        self.e      = [ex,ey,ez]
    def calc_courant(self):
        max_epr = self.epr.max()
        vc = 299.792458 / numpy.sqrt(max_epr)
        dx, dy, dz = self.axes
        tc = 1./(vc * numpy.sqrt(1./dx.min()**2 + 1./dy.min()**2 + 1./dz.min()**2))
        return tc

class RectilinearFDTD:
    @classmethod
    def calchx(self,yee):
        code =\
        """
        #define mu 1.256637061
        #define Vhx(i,j,k)  (hx+(i)*Nhx[1]*Nhx[2]+(j)*Nhx[2]+k)
        #define Vey(i,j,k)  (ey+(i)*Ney[1]*Ney[2]+(j)*Ney[2]+k)
        #define Vez(i,j,k)  (ez+(i)*Nez[1]*Nez[2]+(j)*Nez[2]+k)
        #define Vdx(i)      (dx+i)
        #define Vdy(j)      (dy+j)
        #define Vdz(k)      (dz+k)
        #define Vmur(i,j,k) (mur+(i)*Nmur[1]*Nmur[2]+(j)*Nmur[2]+k)
        for (int k = 0; k < Nhx[2]; k++)
            for (int j = 0; j < Nhx[1]; j++)
                for (int i = 0; i < Nhx[0]; i++) {
                    double DX  = *Vdx(i  ) + *Vdx(i-1);
                    double MUR =(*Vdx(i  ) * *Vmur(i-1,j  ,k  )
                               + *Vdx(i-1) * *Vmur(i  ,j  ,k  ))/DX;
                    double dtmu = dt / mu / MUR;
                    double c1 = dtmu / *Vdy(j);
                    double c2 = dtmu / *Vdz(k);
                    *Vhx(i,j,k) -= c1 * (*Vez(i  ,j+1,k  ) - *Vez(i  ,j  ,k  ))
                                 - c2 * (*Vey(i  ,j  ,k+1) - *Vey(i  ,j  ,k  ));
                }
        """
        ex, ey, ez = yee.e
        hx, hy, hz = yee.h
        dx, dy, dz = yee.axes
        mur        = yee.mur
        dt         = yee.dt.get()
        scipy.weave.inline(code,['hx','ey','ez','dx','dy','dz','dt','mur'])
    @classmethod
    def calchy(self,yee):
        code =\
        """
        #define mu 1.256637061
        #define Vhy(i,j,k)  (hy+(i)*Nhy[1]*Nhy[2]+(j)*Nhy[2]+k)
        #define Vez(i,j,k)  (ez+(i)*Nez[1]*Nez[2]+(j)*Nez[2]+k)
        #define Vex(i,j,k)  (ex+(i)*Nex[1]*Nex[2]+(j)*Nex[2]+k)
        #define Vdx(i)      (dx+i)
        #define Vdy(j)      (dy+j)
        #define Vdz(k)      (dz+k)
        #define Vmur(i,j,k) (mur+(i)*Nmur[1]*Nmur[2]+(j)*Nmur[2]+k)
        for (int k = 0; k < Nhy[2]; k++)
            for (int j = 0; j < Nhy[1]; j++)
                for (int i = 0; i < Nhy[0]; i++) {
                    double DY  = *Vdy(j  ) + *Vdy(j-1);
                    double MUR =(*Vdy(j  ) * *Vmur(i  ,j-1,k  )
                               + *Vdy(j-1) * *Vmur(i  ,j  ,k  ))/DY;
                    double dtmu = dt / mu / MUR;
                    double c1 = dtmu / *Vdz(k);
                    double c2 = dtmu / *Vdx(i);
                    *Vhy(i,j,k) -= c1 * (*Vex(i  ,j  ,k+1) - *Vex(i  ,j  ,k  ))
                                 - c2 * (*Vez(i+1,j  ,k  ) - *Vez(i  ,j  ,k  ));
                }
        """
        ex, ey, ez = yee.e
        hx, hy, hz = yee.h
        dx, dy, dz = yee.axes
        mur        = yee.mur
        dt         = yee.dt.get()
        scipy.weave.inline(code,['hy','ez','ex','dx','dy','dz','dt','mur'])
    @classmethod
    def calchz(self,yee):
        code =\
        """
        #define mu 1.256637061
        #define Vhz(i,j,k)  (hz+(i)*Nhz[1]*Nhz[2]+(j)*Nhz[2]+k)
        #define Vex(i,j,k)  (ex+(i)*Nex[1]*Nex[2]+(j)*Nex[2]+k)
        #define Vey(i,j,k)  (ey+(i)*Ney[1]*Ney[2]+(j)*Ney[2]+k)
        #define Vdx(i)      (dx+i)
        #define Vdy(j)      (dy+j)
        #define Vdz(k)      (dz+k)
        #define Vmur(i,j,k) (mur+(i)*Nmur[1]*Nmur[2]+(j)*Nmur[2]+k)
        for (int k = 0; k < Nhz[2]; k++)
            for (int j = 0; j < Nhz[1]; j++)
                for (int i = 0; i < Nhz[0]; i++) {
                    double DZ  = *Vdz(k  ) + *Vdz(k-1);
                    double MUR =(*Vdz(k  ) * *Vmur(i  ,j  ,k-1)
                               + *Vdz(k-1) * *Vmur(i  ,j  ,k  ))/DZ;
                    double dtmu = dt / mu / MUR;
                    double c1 = dtmu / *Vdx(i);
                    double c2 = dtmu / *Vdy(j);
                    *Vhz(i,j,k) -= c1 * (*Vey(i+1,j  ,k  ) - *Vey(i  ,j  ,k  ))
                                 - c2 * (*Vex(i  ,j+1,k  ) - *Vex(i  ,j  ,k  ));
                }
        """
        ex, ey, ez = yee.e
        hx, hy, hz = yee.h
        dx, dy, dz = yee.axes
        mur        = yee.mur
        dt         = yee.dt.get()
        scipy.weave.inline(code,['hz','ex','ey','dx','dy','dz','dt','mur'])
    @classmethod
    def calcex(cls,yee):
        code =\
        """
        #define EP 0.000008854187817
        #define Vhy(i,j,k)  (hy+(i)*Nhy[1]*Nhy[2]+(j)*Nhy[2]+k)
        #define Vhz(i,j,k)  (hz+(i)*Nhz[1]*Nhz[2]+(j)*Nhz[2]+k)
        #define Vex(i,j,k)  (ex+(i)*Nex[1]*Nex[2]+(j)*Nex[2]+k)
        #define Vdy(j)      (dy+j)
        #define Vdz(k)      (dz+k)
        #define Vepr(i,j,k) (epr+(i)*Nepr[1]*Nepr[2]+(j)*Nepr[2]+k)
        #define Vsgm(i,j,k) (sgm+(i)*Nsgm[1]*Nsgm[2]+(j)*Nsgm[2]+k)
        for (int k = 1; k < Nex[2]-1; k++)
            for (int j = 1; j < Nex[1]-1; j++)
                for (int i = 0; i < Nex[0]  ; i++) {
                    double DY   = .5*(*Vdy(j  )+*Vdy(j-1));
                    double DZ   = .5*(*Vdz(k  )+*Vdz(k-1));
                    double DYDZ = (*Vdy(j  )+*Vdy(j-1))*(*Vdz(k  )+*Vdz(k-1));
                    double ER   =(*Vdy(j-1) * *Vdz(k-1) * *Vepr(i  ,j  ,k  )
                                + *Vdy(j-1) * *Vdz(k  ) * *Vepr(i  ,j  ,k-1)
                                + *Vdy(j  ) * *Vdz(k-1) * *Vepr(i  ,j-1,k  )
                                + *Vdy(j  ) * *Vdz(k  ) * *Vepr(i  ,j-1,k-1))/DYDZ;
                    double SGM  =(*Vdy(j-1) * *Vdz(k-1) * *Vsgm(i  ,j  ,k  )
                                + *Vdy(j-1) * *Vdz(k  ) * *Vsgm(i  ,j  ,k-1)
                                + *Vdy(j  ) * *Vdz(k-1) * *Vsgm(i  ,j-1,k  )
                                + *Vdy(j  ) * *Vdz(k  ) * *Vsgm(i  ,j-1,k-1))/DYDZ;
                    double c0   = (2.*EP*ER-SGM*dt)/(2.*EP*ER+SGM*dt);
                    double c1   = 2*dt/(2.*EP*ER+SGM*dt)/DY;
                    double c2   = 2*dt/(2.*EP*ER+SGM*dt)/DZ;
                    *Vex(i,j,k) =
                          c0 *  *Vex(i,j,k)
                        + c1 * (*Vhz(i  ,j  ,k  ) - *Vhz(i  ,j-1,k  ))
                        - c2 * (*Vhy(i  ,j  ,k  ) - *Vhy(i  ,j  ,k-1));
                }
        """
        ex, ey, ez = yee.e
        hx, hy, hz = yee.h
        dx, dy, dz = yee.axes
        dt         = yee.dt.get()
        epr        = yee.epr
        sgm        = yee.sgm
        scipy.weave.inline(code,['hy','hz','ex','dy','dz','dt','epr','sgm'])
        
    @classmethod
    def calcey(cls,yee):
        code =\
        """
        #define EP 0.000008854187817
        #define Vhz(i,j,k)  (hz+(i)*Nhz[1]*Nhz[2]+(j)*Nhz[2]+k)
        #define Vhx(i,j,k)  (hx+(i)*Nhx[1]*Nhx[2]+(j)*Nhx[2]+k)
        #define Vey(i,j,k)  (ey+(i)*Ney[1]*Ney[2]+(j)*Ney[2]+k)
        #define Vdz(k)      (dz+k)
        #define Vdx(i)      (dx+i)
        #define Vepr(i,j,k) (epr+(i)*Nepr[1]*Nepr[2]+(j)*Nepr[2]+k)
        #define Vsgm(i,j,k) (sgm+(i)*Nsgm[1]*Nsgm[2]+(j)*Nsgm[2]+k)
        for (int k = 1; k < Ney[2]-1; k++)
            for (int j = 0; j < Ney[1]  ; j++)
                for (int i = 1; i < Ney[0]-1; i++) {
                    double DZ   = .5*(*Vdz(k  )+*Vdz(k-1));
                    double DX   = .5*(*Vdx(i  )+*Vdx(i-1));
                    double DZDX = (*Vdz(k  )+*Vdz(k-1))*(*Vdx(i  )+*Vdx(i-1));
                    double ER   =(*Vdz(k-1) * *Vdx(i-1) * *Vepr(i  ,j  ,k  )
                                + *Vdz(k-1) * *Vdx(i  ) * *Vepr(i-1,j  ,k  )
                                + *Vdz(k  ) * *Vdx(i-1) * *Vepr(i  ,j  ,k-1)
                                + *Vdz(k  ) * *Vdx(i  ) * *Vepr(i-1,j  ,k-1))/DZDX;
                    double SGM  =(*Vdz(k-1) * *Vdx(i-1) * *Vsgm(i  ,j  ,k  )
                                + *Vdz(k-1) * *Vdx(i  ) * *Vsgm(i-1,j  ,k  )
                                + *Vdz(k  ) * *Vdx(i-1) * *Vsgm(i  ,j  ,k-1)
                                + *Vdz(k  ) * *Vdx(i  ) * *Vsgm(i-1,j  ,k-1))/DZDX;
                    double c0   = (2.*EP*ER-SGM*dt)/(2.*EP*ER+SGM*dt);
                    double c1   = 2*dt/(2.*EP*ER+SGM*dt)/DZ;
                    double c2   = 2*dt/(2.*EP*ER+SGM*dt)/DX;
                    *Vey(i,j,k) =
                          c0 *  *Vey(i,j,k)
                        + c1 * (*Vhx(i  ,j  ,k  ) - *Vhx(i  ,j  ,k-1))
                        - c2 * (*Vhz(i  ,j  ,k  ) - *Vhz(i-1,j  ,k  ));
                }
        """
        ex, ey, ez = yee.e
        hx, hy, hz = yee.h
        dx, dy, dz = yee.axes
        dt         = yee.dt.get()
        epr        = yee.epr
        sgm        = yee.sgm
        scipy.weave.inline(code,['hz','hx','ey','dz','dx','dt','epr','sgm'])
        
    @classmethod
    def calcez(cls,yee):
        code =\
        """
        #define EP 0.000008854187817
        #define Vhx(i,j,k)  (hx+(i)*Nhx[1]*Nhx[2]+(j)*Nhx[2]+k)
        #define Vhy(i,j,k)  (hy+(i)*Nhy[1]*Nhy[2]+(j)*Nhy[2]+k)
        #define Vez(i,j,k)  (ez+(i)*Nez[1]*Nez[2]+(j)*Nez[2]+k)
        #define Vdx(i)      (dx+i)
        #define Vdy(j)      (dy+j)
        #define Vepr(i,j,k) (epr+(i)*Nepr[1]*Nepr[2]+(j)*Nepr[2]+k)
        #define Vsgm(i,j,k) (sgm+(i)*Nsgm[1]*Nsgm[2]+(j)*Nsgm[2]+k)
        for (int k = 0; k < Nez[2]  ; k++)
            for (int j = 1; j < Nez[1]-1; j++)
                for (int i = 1; i < Nez[0]-1; i++) {
                    double DX   = .5*(*Vdx(i)+*Vdx(i-1));
                    double DY   = .5*(*Vdy(j)+*Vdy(j-1));
                    double DXDY = (*Vdx(i  )+*Vdx(i-1))*(*Vdy(j  )+*Vdy(j-1));
                    double ER   =(*Vdx(i-1) * *Vdy(j-1) * *Vepr(i  ,j  ,k  )
                                + *Vdx(i-1) * *Vdy(j  ) * *Vepr(i  ,j-1,k  )
                                + *Vdx(i  ) * *Vdy(j-1) * *Vepr(i-1,j  ,k  )
                                + *Vdx(i  ) * *Vdy(j  ) * *Vepr(i-1,j-1,k  ))/DXDY;
                    double SGM  =(*Vdx(i-1) * *Vdy(j-1) * *Vsgm(i  ,j  ,k  )
                                + *Vdx(i-1) * *Vdy(j  ) * *Vsgm(i  ,j-1,k  )
                                + *Vdx(i  ) * *Vdy(j-1) * *Vsgm(i-1,j  ,k  )
                                + *Vdx(i  ) * *Vdy(j  ) * *Vsgm(i-1,j-1,k  ))/DXDY;
                    double c0   = (2.*EP*ER-SGM*dt)/(2.*EP*ER+SGM*dt);
                    double c1   = 2*dt/(2.*EP*ER+SGM*dt)/DX;
                    double c2   = 2*dt/(2.*EP*ER+SGM*dt)/DY;
                    *Vez(i,j,k) =
                          c0 *  *Vez(i,j,k)
                        + c1 * (*Vhy(i  ,j  ,k  ) - *Vhy(i-1,j  ,k  ))
                        - c2 * (*Vhx(i  ,j  ,k  ) - *Vhx(i  ,j-1,k  ));
                }
        """
        ex, ey, ez = yee.e
        hx, hy, hz = yee.h
        dx, dy, dz = yee.axes
        dt         = yee.dt.get()
        epr        = yee.epr
        sgm        = yee.sgm
        scipy.weave.inline(code,['hx','hy','ez','dx','dy','dt','epr','sgm'])

class RectilinearPML:
    class Dataset:
        def __init__(self,delta,sigm,dt=0.1):
            self.dt    = dt
            dx, dy, dz = delta
            shape = (len(dx),len(dy),len(dz))
            self.alloc(shape)
            self.dx,   self.dy,   self.dz   = delta
            self.sgmx, self.sgmy, self.sgmz = sigm
        def alloc(self,shape):
            opts     = {'dtype':numpy.double}
            self.hxy = numpy.zeros([shape[0]+1,shape[1]  ,shape[2]  ],**opts)
            self.hxz = numpy.zeros([shape[0]+1,shape[1]  ,shape[2]  ],**opts)
            self.hyz = numpy.zeros([shape[0]  ,shape[1]+1,shape[2]  ],**opts)
            self.hyx = numpy.zeros([shape[0]  ,shape[1]+1,shape[2]  ],**opts)
            self.hzx = numpy.zeros([shape[0]  ,shape[1]  ,shape[2]+1],**opts)
            self.hzy = numpy.zeros([shape[0]  ,shape[1]  ,shape[2]+1],**opts)
            self.exy = numpy.zeros([shape[0]  ,shape[1]+1,shape[2]+1],**opts)
            self.exz = numpy.zeros([shape[0]  ,shape[1]+1,shape[2]+1],**opts)
            self.eyz = numpy.zeros([shape[0]+1,shape[1]  ,shape[2]+1],**opts)
            self.eyx = numpy.zeros([shape[0]+1,shape[1]  ,shape[2]+1],**opts)
            self.ezx = numpy.zeros([shape[0]+1,shape[1]+1,shape[2]  ],**opts)
            self.ezy = numpy.zeros([shape[0]+1,shape[1]+1,shape[2]  ],**opts)
        def clear(self):
            for m in [self.hxy, self.hxz, self.hyz, self.hyx, self.hzx, self.hzy,
                      self.exy, self.exz, self.eyz, self.eyx, self.ezx, self.ezy]: m[:,:,:] = 0.
    def __init__(self,yee):
        self.yee   = yee = yee.get()
        dx, dy, dz = yee.axes
        dt         = yee.dt.get()
        shape      = len(dx), len(dy), len(dz)
        zx, zy, zz = numpy.zeros(shape[0]), numpy.zeros(shape[1]), numpy.zeros(shape[2])
        sx, sy, sz = self.sigma(dx[0]), self.sigma(dy[0]), self.sigma(dz[0])
        ssx = numpy.zeros(2*len(sx)+2)
        ssx[1:len(sx)+1] = sx
        ssx[len(sx)+1:-1] = sx[::-1]
        ssy = numpy.zeros(2*len(sy)+2)
        ssy[1:len(sy)+1] = sy
        ssy[len(sy)+1:-1] = sy[::-1]
        ssz = numpy.zeros(2*len(sz)+2)
        ssz[1:len(sz)+1] = sz
        ssz[len(sz)+1:-1] = sz[::-1]
        self.x   = self.Dataset((dx[0]*numpy.ones(len(ssx)),dy,dz),(ssx,zy,zz),dt=dt)
        self.y   = self.Dataset((dx,dy[0]*numpy.ones(len(ssy)),dz),(zx,ssy,zz),dt=dt)
        self.z   = self.Dataset((dx,dy,dz[0]*numpy.ones(len(ssz))),(zx,zy,ssz),dt=dt)
        self.xy  = self.Dataset((dx[0]*numpy.ones(len(ssx)),dy[0]*numpy.ones(len(ssy)),dz),(ssx,ssy,zz),dt=dt)
        self.yz  = self.Dataset((dx,dy[0]*numpy.ones(len(ssy)),dz[0]*numpy.ones(len(ssz))),(zx,ssy,ssz),dt=dt)
        self.zx  = self.Dataset((dx[0]*numpy.ones(len(ssx)),dy,dz[0]*numpy.ones(len(ssz))),(ssx,zy,ssz),dt=dt)
        self.xyz = self.Dataset((dx[0]*numpy.ones(len(ssx)),dy[0]*numpy.ones(len(ssy)),dz[0]*numpy.ones(len(ssz))),(ssx,ssy,ssz),dt=dt)
    def clear(self):
        for m in [self.x, self.y, self.z, self.xy, self.yz, self.zx, self.xyz]: m.clear()
    def calce(self,yee):
        ex, ey, ez = yee.e
        self.calcepml(self.x)
        ey[-1,:,:] = self.x.eyz[ 1,:,:] + self.x.eyx[ 1,:,:]
        ez[-1,:,:] = self.x.ezx[ 1,:,:] + self.x.ezy[ 1,:,:]
        ey[ 0,:,:] = self.x.eyz[-2,:,:] + self.x.eyx[-2,:,:]
        ez[ 0,:,:] = self.x.ezx[-2,:,:] + self.x.ezy[-2,:,:]
        #
        self.calcepml(self.y)
        ez[:,-1,:] = self.y.ezx[:, 1,:] + self.y.ezy[:, 1,:]
        ex[:,-1,:] = self.y.exy[:, 1,:] + self.y.exz[:, 1,:]
        ez[:, 0,:] = self.y.ezx[:,-2,:] + self.y.ezy[:,-2,:]
        ex[:, 0,:] = self.y.exy[:,-2,:] + self.y.exz[:,-2,:]
        #
        self.calcepml(self.z)
        ex[:,:,-1] = self.z.exy[:,:, 1] + self.z.exz[:,:, 1]
        ey[:,:,-1] = self.z.eyz[:,:, 1] + self.z.eyx[:,:, 1]
        ex[:,:, 0] = self.z.exy[:,:,-2] + self.z.exz[:,:,-2]
        ey[:,:, 0] = self.z.eyz[:,:,-2] + self.z.eyx[:,:,-2]
        #
        self.calcepml(self.xy)
        self.x.ezx[:,-1,:], self.x.ezx[:, 0,:] = self.xy.ezx[:, 1,:], self.xy.ezx[:,-2,:]
        self.x.ezy[:,-1,:], self.x.ezy[:, 0,:] = self.xy.ezy[:, 1,:], self.xy.ezy[:,-2,:]
        self.x.exy[:,-1,:], self.x.exy[:, 0,:] = self.xy.exy[:, 1,:], self.xy.exy[:,-2,:]
        self.x.exz[:,-1,:], self.x.exz[:, 0,:] = self.xy.exz[:, 1,:], self.xy.exz[:,-2,:]
        self.y.eyz[-1,:,:], self.y.eyz[ 0,:,:] = self.xy.eyz[ 1,:,:], self.xy.eyz[-2,:,:]
        self.y.eyx[-1,:,:], self.y.eyx[ 0,:,:] = self.xy.eyx[ 1,:,:], self.xy.eyx[-2,:,:]
        self.y.ezx[-1,:,:], self.y.ezx[ 0,:,:] = self.xy.ezx[ 1,:,:], self.xy.ezx[-2,:,:]
        self.y.ezy[-1,:,:], self.y.ezy[ 0,:,:] = self.xy.ezy[ 1,:,:], self.xy.ezy[-2,:,:]
        #
        self.calcepml(self.yz)
        self.y.exy[:,:,-1], self.y.exy[:,:, 0] = self.yz.exy[:,:, 1], self.yz.exy[:,:,-2]
        self.y.exz[:,:,-1], self.y.exz[:,:, 0] = self.yz.exz[:,:, 1], self.yz.exz[:,:,-2]
        self.y.eyz[:,:,-1], self.y.eyz[:,:, 0] = self.yz.eyz[:,:, 1], self.yz.eyz[:,:,-2]
        self.y.eyx[:,:,-1], self.y.eyx[:,:, 0] = self.yz.eyx[:,:, 1], self.yz.eyx[:,:,-2]
        self.z.ezx[:,-1,:], self.z.ezx[:, 0,:] = self.yz.ezx[:, 1,:], self.yz.ezx[:,-2,:]
        self.z.ezy[:,-1,:], self.z.ezy[:, 0,:] = self.yz.ezy[:, 1,:], self.yz.ezy[:,-2,:]
        self.z.exy[:,-1,:], self.z.exy[:, 0,:] = self.yz.exy[:, 1,:], self.yz.exy[:,-2,:]
        self.z.exz[:,-1,:], self.z.exz[:, 0,:] = self.yz.exz[:, 1,:], self.yz.exz[:,-2,:]
        #
        self.calcepml(self.zx)
        self.z.eyz[-1,:,:], self.z.eyz[ 0,:,:] = self.zx.eyz[ 1,:,:], self.zx.eyz[-2,:,:]
        self.z.eyx[-1,:,:], self.z.eyx[ 0,:,:] = self.zx.eyx[ 1,:,:], self.zx.eyx[-2,:,:]
        self.z.ezx[-1,:,:], self.z.ezx[ 0,:,:] = self.zx.ezx[ 1,:,:], self.zx.ezx[-2,:,:]
        self.z.ezy[-1,:,:], self.z.ezy[ 0,:,:] = self.zx.ezy[ 1,:,:], self.zx.ezy[-2,:,:]
        self.x.exy[:,:,-1], self.x.exy[:,:, 0] = self.zx.exy[:,:, 1], self.zx.exy[:,:,-2]
        self.x.exz[:,:,-1], self.x.exz[:,:, 0] = self.zx.exz[:,:, 1], self.zx.exz[:,:,-2]
        self.x.eyz[:,:,-1], self.x.eyz[:,:, 0] = self.zx.eyz[:,:, 1], self.zx.eyz[:,:,-2]
        self.x.eyx[:,:,-1], self.x.eyx[:,:, 0] = self.zx.eyx[:,:, 1], self.zx.eyx[:,:,-2]
        #
        self.calcepml(self.xyz)
        self.yz.eyz[-1,:,:], self.yz.eyz[ 0,:,:] = self.xyz.eyz[ 1,:,:], self.xyz.eyz[-2,:,:]
        self.yz.eyx[-1,:,:], self.yz.eyx[ 0,:,:] = self.xyz.eyx[ 1,:,:], self.xyz.eyx[-2,:,:]
        self.yz.ezx[-1,:,:], self.yz.ezx[ 0,:,:] = self.xyz.ezx[ 1,:,:], self.xyz.ezx[-2,:,:]
        self.yz.ezy[-1,:,:], self.yz.ezy[ 0,:,:] = self.xyz.ezy[ 1,:,:], self.xyz.ezy[-2,:,:]
        self.zx.ezx[:,-1,:], self.zx.ezx[:, 0,:] = self.xyz.ezx[:, 1,:], self.xyz.ezx[:,-2,:]
        self.zx.ezy[:,-1,:], self.zx.ezy[:, 0,:] = self.xyz.ezy[:, 1,:], self.xyz.ezy[:,-2,:]
        self.zx.exy[:,-1,:], self.zx.exy[:, 0,:] = self.xyz.exy[:, 1,:], self.xyz.exy[:,-2,:]
        self.zx.exz[:,-1,:], self.zx.exz[:, 0,:] = self.xyz.exz[:, 1,:], self.xyz.exz[:,-2,:]
        self.xy.exy[:,:,-1], self.xy.exy[:,:, 0] = self.xyz.exy[:,:, 1], self.xyz.exy[:,:,-2]
        self.xy.exz[:,:,-1], self.xy.exz[:,:, 0] = self.xyz.exz[:,:, 1], self.xyz.exz[:,:,-2]
        self.xy.eyz[:,:,-1], self.xy.eyz[:,:, 0] = self.xyz.eyz[:,:, 1], self.xyz.eyz[:,:,-2]
        self.xy.eyx[:,:,-1], self.xy.eyx[:,:, 0] = self.xyz.eyx[:,:, 1], self.xyz.eyx[:,:,-2]
    def calcepml(self,pml):
        self.calcexy(pml)
        self.calcexz(pml)
        self.calceyz(pml)
        self.calceyx(pml)
        self.calcezx(pml)
        self.calcezy(pml)
    def calch(self,yee):
        hx, hy, hz = yee.h
        #
        self.calchpml(self.x)
        self.x.hyz[ 0,:,:], self.x.hyx[ 0,:,:] = hy[-1,:,:], 0.
        self.x.hzx[ 0,:,:], self.x.hzy[ 0,:,:] = hz[-1,:,:], 0.
        self.x.hyz[-1,:,:], self.x.hyx[-1,:,:] = hy[ 0,:,:], 0.
        self.x.hzx[-1,:,:], self.x.hzy[-1,:,:] = hz[ 0,:,:], 0.
        #
        self.calchpml(self.y)
        self.y.hxy[:, 0,:], self.y.hxz[:, 0,:] = hx[:,-1,:], 0.
        self.y.hzx[:, 0,:], self.y.hzy[:, 0,:] = hz[:,-1,:], 0.
        self.y.hxy[:,-1,:], self.y.hxz[:,-1,:] = hx[:, 0,:], 0.
        self.y.hzx[:,-1,:], self.y.hzy[:,-1,:] = hz[:, 0,:], 0.
        #
        self.calchpml(self.z)
        self.z.hxy[:,:, 0], self.z.hxz[:,:, 0] = hx[:,:,-1], 0.
        self.z.hyz[:,:, 0], self.z.hyx[:,:, 0] = hy[:,:,-1], 0.
        self.z.hxy[:,:,-1], self.z.hxz[:,:,-1] = hx[:,:, 0], 0.
        self.z.hyz[:,:,-1], self.z.hyx[:,:,-1] = hy[:,:, 0], 0.
        #
        self.calchpml(self.xy)
        self.xy.hxy[:, 0,:], self.xy.hxy[:,-1,:] = self.x.hxy[:,-1,:], self.x.hxy[:, 0,:]
        self.xy.hxz[:, 0,:], self.xy.hxz[:,-1,:] = self.x.hxz[:,-1,:], self.x.hxz[:, 0,:]
        self.xy.hzx[:, 0,:], self.xy.hzx[:,-1,:] = self.x.hzx[:,-1,:], self.x.hzx[:, 0,:]
        self.xy.hzy[:, 0,:], self.xy.hzy[:,-1,:] = self.x.hzy[:,-1,:], self.x.hzy[:, 0,:]
        self.xy.hyz[ 0,:,:], self.xy.hyz[-1,:,:] = self.y.hyz[-1,:,:], self.y.hyz[ 0,:,:]
        self.xy.hyx[ 0,:,:], self.xy.hyx[-1,:,:] = self.y.hyx[-1,:,:], self.y.hyx[ 0,:,:]
        self.xy.hzx[ 0,:,:], self.xy.hzx[-1,:,:] = self.y.hzx[-1,:,:], self.y.hzx[ 0,:,:]
        self.xy.hzy[ 0,:,:], self.xy.hzy[-1,:,:] = self.y.hzy[-1,:,:], self.y.hzy[ 0,:,:]
        #
        self.calchpml(self.yz)
        self.yz.hxy[:,:, 0], self.yz.hxy[:,:,-1] = self.y.hxy[:,:,-1], self.y.hxy[:,:, 0]
        self.yz.hxz[:,:, 0], self.yz.hxz[:,:,-1] = self.y.hxz[:,:,-1], self.y.hxz[:,:, 0]
        self.yz.hyz[:,:, 0], self.yz.hyz[:,:,-1] = self.y.hyz[:,:,-1], self.y.hyz[:,:, 0]
        self.yz.hyx[:,:, 0], self.yz.hyx[:,:,-1] = self.y.hyx[:,:,-1], self.y.hyx[:,:, 0]
        self.yz.hxy[:, 0,:], self.yz.hxy[:,-1,:] = self.z.hxy[:,-1,:], self.z.hxy[:, 0,:]
        self.yz.hxz[:, 0,:], self.yz.hxz[:,-1,:] = self.z.hxz[:,-1,:], self.z.hxz[:, 0,:]
        self.yz.hzx[:, 0,:], self.yz.hzx[:,-1,:] = self.z.hzx[:,-1,:], self.z.hzx[:, 0,:]
        self.yz.hzy[:, 0,:], self.yz.hzy[:,-1,:] = self.z.hzy[:,-1,:], self.z.hzy[:, 0,:]
        #
        self.calchpml(self.zx)
        self.zx.hyz[ 0,:,:], self.zx.hyz[-1,:,:] = self.z.hyz[-1,:,:], self.z.hyz[ 0,:,:]
        self.zx.hyx[ 0,:,:], self.zx.hyx[-1,:,:] = self.z.hyx[-1,:,:], self.z.hyx[ 0,:,:]
        self.zx.hzx[ 0,:,:], self.zx.hzx[-1,:,:] = self.z.hzx[-1,:,:], self.z.hzx[ 0,:,:]
        self.zx.hzy[ 0,:,:], self.zx.hzy[-1,:,:] = self.z.hzy[-1,:,:], self.z.hzy[ 0,:,:]
        self.zx.hxy[:,:, 0], self.zx.hxy[:,:,-1] = self.x.hxy[:,:,-1], self.x.hxy[:,:, 0]
        self.zx.hxz[:,:, 0], self.zx.hxz[:,:,-1] = self.x.hxz[:,:,-1], self.x.hxz[:,:, 0]
        self.zx.hyz[:,:, 0], self.zx.hyz[:,:,-1] = self.x.hyz[:,:,-1], self.x.hyz[:,:, 0]
        self.zx.hyx[:,:, 0], self.zx.hyx[:,:,-1] = self.x.hyx[:,:,-1], self.x.hyx[:,:, 0]
        #
        self.calchpml(self.xyz)
        self.xyz.hyz[ 0,:,:], self.xyz.hyz[-1,:,:] = self.yz.hyz[-1,:,:], self.yz.hyz[ 0,:,:]
        self.xyz.hyx[ 0,:,:], self.xyz.hyx[-1,:,:] = self.yz.hyx[-1,:,:], self.yz.hyx[ 0,:,:]
        self.xyz.hzx[ 0,:,:], self.xyz.hzx[-1,:,:] = self.yz.hzx[-1,:,:], self.yz.hzx[ 0,:,:]
        self.xyz.hzy[ 0,:,:], self.xyz.hzy[-1,:,:] = self.yz.hzy[-1,:,:], self.yz.hzy[ 0,:,:]
        self.xyz.hxy[:, 0,:], self.xyz.hxy[:,-1,:] = self.zx.hxy[:,-1,:], self.zx.hxy[:, 0,:]
        self.xyz.hxz[:, 0,:], self.xyz.hxz[:,-1,:] = self.zx.hxz[:,-1,:], self.zx.hxz[:, 0,:]
        self.xyz.hzx[:, 0,:], self.xyz.hzx[:,-1,:] = self.zx.hzx[:,-1,:], self.zx.hzx[:, 0,:]
        self.xyz.hzy[:, 0,:], self.xyz.hzy[:,-1,:] = self.zx.hzy[:,-1,:], self.zx.hzy[:, 0,:]
        self.xyz.hxy[:,:, 0], self.xyz.hxy[:,:,-1] = self.xy.hxy[:,:,-1], self.xy.hxy[:,:, 0]
        self.xyz.hxz[:,:, 0], self.xyz.hxz[:,:,-1] = self.xy.hxz[:,:,-1], self.xy.hxz[:,:, 0]
        self.xyz.hyz[:,:, 0], self.xyz.hyz[:,:,-1] = self.xy.hyz[:,:,-1], self.xy.hyz[:,:, 0]
        self.xyz.hyx[:,:, 0], self.xyz.hyx[:,:,-1] = self.xy.hyx[:,:,-1], self.xy.hyx[:,:, 0]
    def calchpml(self,pml):
        self.calchxy(pml)
        self.calchxz(pml)
        self.calchyz(pml)
        self.calchyx(pml)
        self.calchzx(pml)
        self.calchzy(pml)
    @classmethod
    def sigma(cls,delta=.1,r0=1e-6,m=2,l=16):
        EP0 = 0.000008854187817
        MU0 = 1.256637061
        smax = -((m+1.)*EP0/numpy.sqrt(EP0*MU0))/(2.*l*delta)*numpy.log(r0)
        pos  = numpy.r_[:l*delta:delta]
        return (smax*((l*delta-pos)/(l*delta))**m)[::-1]
    @classmethod
    def calcec(cls,pml,sgm,delta):
        ep = 0.000008854187817
        sgm   = (delta[1:] * sgm[:-1] + delta[:-1] * sgm[1:]) / (delta[:-1] + delta[1:])
        delta = .5 * (delta[:-1] + delta[1:])
        c0    = 2. * pml.dt / delta / (2. * ep + sgm * pml.dt)
        c1    = (2. * ep - sgm * pml.dt) / (2. * ep + sgm * pml.dt)
        return c0, c1
    @classmethod
    def calchc(cls,pml,sgm,delta):
        ep = 0.000008854187817
        mu = 1.256637061
        c0    = 2. * pml.dt / delta / (2. * ep + sgm * pml.dt)
        c1    = (2. * ep - sgm * pml.dt) / (2. * ep + sgm * pml.dt)
        return ep / mu * c0, c1
    @classmethod
    def calcexy(cls,pml):
        emn, h = pml.exy, pml.hzy + pml.hzx
        c0, c1 = cls.calcec(pml,pml.sgmy,pml.dy)
        emn[:,1:-1,1:-1] =   c0.reshape(1,len(c0),1) * (h[:,1:,1:-1]-h[:,:-1,1:-1]) + c1.reshape(1,len(c1),1) * emn[:,1:-1,1:-1]
    @classmethod
    def calcexz(cls,pml):
        emn, h = pml.exz, pml.hyz + pml.hyx
        c0, c1 = cls.calcec(pml,pml.sgmz,pml.dz)
        emn[:,1:-1,1:-1] = - c0.reshape(1,1,len(c0)) * (h[:,1:-1,1:]-h[:,1:-1,:-1]) + c1.reshape(1,1,len(c1)) * emn[:,1:-1,1:-1]
    @classmethod
    def calceyz(cls,pml):
        emn, h = pml.eyz, pml.hxy + pml.hxz
        c0, c1 = cls.calcec(pml,pml.sgmz,pml.dz)
        emn[1:-1,:,1:-1] =   c0.reshape(1,1,len(c0)) * (h[1:-1,:,1:]-h[1:-1,:,:-1]) + c1.reshape(1,1,len(c1)) * emn[1:-1,:,1:-1]
    @classmethod
    def calceyx(cls,pml):
        emn, h = pml.eyx, pml.hzx + pml.hzy
        c0, c1 = cls.calcec(pml,pml.sgmx,pml.dx)
        emn[1:-1,:,1:-1] = - c0.reshape(len(c0),1,1) * (h[1:,:,1:-1]-h[:-1,:,1:-1]) + c1.reshape(len(c1),1,1) * emn[1:-1,:,1:-1]
    @classmethod
    def calcezx(cls,pml):
        emn, h = pml.ezx, pml.hyz + pml.hyx
        c0, c1 = cls.calcec(pml,pml.sgmx,pml.dx)
        emn[1:-1,1:-1,:] =   c0.reshape(len(c0),1,1) * (h[1:,1:-1,:]-h[:-1,1:-1,:]) + c1.reshape(len(c1),1,1) * emn[1:-1,1:-1,:]
    @classmethod
    def calcezy(cls,pml):
        emn, h = pml.ezy, pml.hxy + pml.hxz
        c0, c1 = cls.calcec(pml,pml.sgmy,pml.dy)
        emn[1:-1,1:-1,:] = - c0.reshape(1,len(c0),1) * (h[1:-1,1:,:]-h[1:-1,:-1,:]) + c1.reshape(1,len(c1),1) * emn[1:-1,1:-1,:]
    @classmethod
    def calchxy(cls,pml):
        h, e = pml.hxy, pml.ezx + pml.ezy
        c0, c1 = cls.calchc(pml,pml.sgmy,pml.dy)
        h[:,:,:] = - c0.reshape(1,len(c0),1) * (e[:,1:,:]-e[:,:-1,:]) + c1.reshape(1,len(c1),1) * h[:,:,:]
    @classmethod
    def calchxz(cls,pml):
        h, e = pml.hxz, pml.eyz + pml.eyx
        c0, c1 = cls.calchc(pml,pml.sgmz,pml.dz)
        h[:,:,:] =   c0.reshape(1,1,len(c0)) * (e[:,:,1:]-e[:,:,:-1]) + c1.reshape(1,1,len(c1)) * h[:,:,:]
    @classmethod
    def calchyz(cls,pml):
        h, e = pml.hyz, pml.exy + pml.exz
        c0, c1 = cls.calchc(pml,pml.sgmz,pml.dz)
        h[:,:,:] = - c0.reshape(1,1,len(c0)) * (e[:,:,1:]-e[:,:,:-1]) + c1.reshape(1,1,len(c1)) * h[:,:,:]
    @classmethod
    def calchyx(cls,pml):
        h, e = pml.hyx, pml.ezx + pml.ezy
        c0, c1 = cls.calchc(pml,pml.sgmx,pml.dx)
        h[:,:,:] =   c0.reshape(len(c0),1,1) * (e[1:,:,:]-e[:-1,:,:]) + c1.reshape(len(c1),1,1) * h[:,:,:]
    @classmethod
    def calchzx(cls,pml):
        h, e = pml.hzx, pml.eyz + pml.eyx
        c0, c1 = cls.calchc(pml,pml.sgmx,pml.dx)
        h[:,:,:] = - c0.reshape(len(c0),1,1) * (e[1:,:,:]-e[:-1,:,:]) + c1.reshape(len(c1),1,1) * h[:,:,:]
    @classmethod
    def calchzy(cls,pml):
        h, e = pml.hzy, pml.exy + pml.exz
        c0, c1 = cls.calchc(pml,pml.sgmy,pml.dy)
        h[:,:,:] =   c0.reshape(1,len(c0),1) * (e[:,1:,:]-e[:,:-1,:]) + c1.reshape(1,len(c1),1) * h[:,:,:]
        