// waveout stuff based on buzz code

#include "stdafx.h"
#include "WaveOut.h"
#include "resource.h"
#include "WaveOutDialog.h"
#include "Registry.h"
#include "Configuration.h"
#include "MidiInput.h"
#include <process.h>

#define BYTES_PER_SAMPLE 4	// 2 * 16bits

AudioDriverInfo WaveOut::_info = { "Windows WaveOut MME" };
AudioDriverEvent WaveOut::_event;
CCriticalSection WaveOut::_lock;

WaveOut::WaveOut()
{
	_initialized = false;
	_configured = false;
	_running = false;
	_pCallback = NULL;
}


WaveOut::~WaveOut()
{
	if (_initialized) Reset();
}

void WaveOut::Error(char const *msg)
{
	MessageBox(NULL, msg, "WAVE oײް", MB_OK);
}

bool WaveOut::Start()
{
	CSingleLock lock(&_lock, TRUE);
	if (_running)
	{
		return true;
	}

	if (_pCallback == NULL)
	{
		return false;
	}

	WAVEFORMATEX format;
	format.wFormatTag = WAVE_FORMAT_PCM;
	format.wBitsPerSample = 16;//_bitDepth;
	format.nSamplesPerSec = _samplesPerSec;
	format.cbSize = 0;
	format.nChannels = 2;

	format.nBlockAlign = format.nChannels * format.wBitsPerSample / 8;
	format.nAvgBytesPerSec = format.nSamplesPerSec * format.nBlockAlign;

	if (waveOutOpen(&_handle, _deviceID, &format, NULL, 0, CALLBACK_NULL) != MMSYSERR_NOERROR)
	{
		Error("waveOutOpen() : Wave ޲̃I[vɎs܂");
		return false;
	}

	_currentBlock = 0;
	_writePos = 0;

	// allocate blocks
	for (CBlock *pBlock = _blocks; pBlock < _blocks + _numBlocks; pBlock++)
	{
		pBlock->Handle = GlobalAlloc(GMEM_MOVEABLE | GMEM_SHARE, _blockSize);
		pBlock->pData = (byte *)GlobalLock(pBlock->Handle);
	}

	// allocate block headers
	for (pBlock = _blocks; pBlock < _blocks + _numBlocks; pBlock++)
	{
		pBlock->HeaderHandle = GlobalAlloc(GMEM_MOVEABLE | GMEM_SHARE, sizeof(WAVEHDR));
		pBlock->pHeader = (WAVEHDR *)GlobalLock(pBlock->HeaderHandle);

		WAVEHDR *ph = pBlock->pHeader;
		ph->lpData = (char *)pBlock->pData;
		ph->dwBufferLength = _blockSize;
		ph->dwFlags = WHDR_DONE;
		ph->dwLoops = 0;

		pBlock->Prepared = false;
	}

	_stopPolling = false;
	_event.ResetEvent();
	_beginthread(PollerThread, 0, this);
	_running = true;
	CMidiInput::Instance()->ReSync();	// MIDI IMPLEMENTATION
	return true;
}

bool WaveOut::Stop()
{
	CSingleLock lock(&_lock, TRUE);
	if (!_running)
	{
		return true;
	}

	_stopPolling = true;
	CSingleLock event(&_event, TRUE);

	if (waveOutReset(_handle) != MMSYSERR_NOERROR)
	{
		Error("waveOutReset() : Đ~Ɏs܂");
		return false;
	}

	while (1)
	{
		bool alldone = true;

		for (CBlock *pBlock = _blocks; pBlock < _blocks + _numBlocks; pBlock++)
		{
			if ((pBlock->pHeader->dwFlags & WHDR_DONE) == 0)
			{
				alldone = false;
			}
		}
		if (alldone)
		{
			break;
		}
		Sleep(20);
	}
	for (CBlock *pBlock = _blocks; pBlock < _blocks + _numBlocks; pBlock++)
	{
		if (pBlock->Prepared)
		{
			if (waveOutUnprepareHeader(_handle, pBlock->pHeader, sizeof(WAVEHDR)) != MMSYSERR_NOERROR)
			{
				Error("waveOutUnprepareHeader() : ޯ̧p؂̊JɎs܂");
			}
		}
	}
	if (waveOutClose(_handle) != MMSYSERR_NOERROR)
	{
		Error("waveOutClose() : Wave ޲̸۰ނɎs܂");
		return false;
	}
	for (pBlock = _blocks; pBlock < _blocks + _numBlocks; pBlock++)
	{
		GlobalUnlock(pBlock->Handle);
		GlobalFree(pBlock->Handle);
		GlobalUnlock(pBlock->HeaderHandle);
		GlobalFree(pBlock->HeaderHandle);
	}
	_running = false;
	return true;
}

void
WaveOut::PollerThread(
	void *pWaveOut)
{
	WaveOut* pThis = (WaveOut*)pWaveOut;
	SetThreadPriority(GetCurrentThread(), THREAD_PRIORITY_ABOVE_NORMAL);

	while (!pThis->_stopPolling)
	{
		pThis->DoBlocks();
		::Sleep(pThis->_pollSleep);
	}
	_event.SetEvent();
	_endthread();
}

void WaveOut::DoBlocks()
{
	CBlock *pb = _blocks + _currentBlock;

	while(pb->pHeader->dwFlags & WHDR_DONE)
	{
		if (pb->Prepared)
		{
			if (waveOutUnprepareHeader(_handle, pb->pHeader, sizeof(WAVEHDR)) != MMSYSERR_NOERROR)
			{
				Error("waveOutUnprepareHeader() : ޯ̧p؂̊JɎs܂");
			}
			pb->Prepared = false;
		}
		int *pOut = (int *)pb->pData;
		int bs = _blockSize / BYTES_PER_SAMPLE;

		do
		{
			int n = bs;
			float *pBuf = _pCallback(_callbackContext, n);
			if (_dither)
			{
				QuantizeWithDither(pBuf, pOut, n);
			}
			else
			{
				Quantize(pBuf, pOut, n);
			}
			pOut += n;
			bs -= n;
		}
		while (bs > 0);

		_writePos += _blockSize/BYTES_PER_SAMPLE;

		pb->pHeader->dwFlags = 0;
		pb->pHeader->lpData = (char *)pb->pData;
		pb->pHeader->dwBufferLength = _blockSize;
		pb->pHeader->dwFlags = 0;
		pb->pHeader->dwLoops = 0;

		if (waveOutPrepareHeader(_handle, pb->pHeader, sizeof(WAVEHDR)) != MMSYSERR_NOERROR)
		{
			Error("waveOutPrepareHeader() : ޯ̧p؂̊mۂɎs܂");
		}
		pb->Prepared = true;

		if (waveOutWrite(_handle, pb->pHeader, sizeof(WAVEHDR)) != MMSYSERR_NOERROR)
		{
			Error("waveOutWrite() : Wave ޲ް]̂Ɏs܂");
		}
		pb++;
		if (pb == _blocks + _numBlocks)
		{
			pb = _blocks;
		}
	}
	_currentBlock = pb - _blocks;
}

void WaveOut::ReadConfig()
{
	bool configured;
	DWORD type;
	DWORD numData;
	Registry reg;

	// Default configuration
	//
	_samplesPerSec=44100;
	_deviceID=0;
	_numBlocks = 7;
	_blockSize = 4096;
	_pollSleep = 20;
	_dither = 0;
	_channelmode = 3;
	_bitDepth = 16;

	if (reg.OpenRootKey(HKEY_CURRENT_USER, CONFIG_ROOT_KEY) != ERROR_SUCCESS)
	{
		return;
	}
	if (reg.OpenKey("WaveOut") != ERROR_SUCCESS)
	{
		return;
	}
	configured = true;
	numData = sizeof(_numBlocks);
	configured &= (reg.QueryValue("NumBlocks", &type, (BYTE*)&_numBlocks, &numData) == ERROR_SUCCESS);
	numData = sizeof(_blockSize);
	configured &= (reg.QueryValue("BlockSize", &type, (BYTE*)&_blockSize, &numData) == ERROR_SUCCESS);
	numData = sizeof(_deviceID);
	configured &= (reg.QueryValue("DeviceID", &type, (BYTE*)&_deviceID, &numData) == ERROR_SUCCESS);
	numData = sizeof(_pollSleep);
	configured &= (reg.QueryValue("PollSleep", &type, (BYTE*)&_pollSleep, &numData) == ERROR_SUCCESS);
	numData = sizeof(_dither);
	configured &= (reg.QueryValue("Dither", &type, (BYTE*)&_dither, &numData) == ERROR_SUCCESS);
	numData = sizeof(_samplesPerSec);
	configured &= (reg.QueryValue("SamplesPerSec", &type, (BYTE*)&_samplesPerSec, &numData) == ERROR_SUCCESS);
//	numData = sizeof(_bitDepth);
//	(reg.QueryValue("BitDepth", &type, (BYTE*)&_bitDepth, &numData) == ERROR_SUCCESS);
	reg.CloseKey();
	reg.CloseRootKey();
	_configured = configured;
}

void WaveOut::WriteConfig()
{
	Registry reg;
	if (reg.OpenRootKey(HKEY_CURRENT_USER, CONFIG_ROOT_KEY) != ERROR_SUCCESS)
	{
		Error("ڼ޽؂ɐݒނƂo܂");
		return;
	}
	if (reg.OpenKey("WaveOut") != ERROR_SUCCESS)
	{
		if (reg.CreateKey("WaveOut") != ERROR_SUCCESS)
		{
			Error("ڼ޽؂ɐݒނƂo܂");
			return;
		}
	}
	reg.SetValue("NumBlocks", REG_DWORD, (BYTE*)&_numBlocks, sizeof(_numBlocks));
	reg.SetValue("BlockSize", REG_DWORD, (BYTE*)&_blockSize, sizeof(_blockSize));
	reg.SetValue("DeviceID", REG_DWORD, (BYTE*)&_deviceID, sizeof(_deviceID));
	reg.SetValue("PollSleep", REG_DWORD, (BYTE*)&_pollSleep, sizeof(_pollSleep));
	reg.SetValue("Dither", REG_DWORD, (BYTE*)&_dither, sizeof(_dither));
	reg.SetValue("SamplesPerSec", REG_DWORD, (BYTE*)&_samplesPerSec, sizeof(_samplesPerSec));
//	reg.SetValue("BitDepth", REG_DWORD, (BYTE*)&_bitDepth, sizeof(_bitDepth));
	reg.CloseKey();
	reg.CloseRootKey();
}


void WaveOut::Initialize(
	HWND hwnd,
	AUDIODRIVERWORKFN pCallback,
	void* context)
{
	_callbackContext = context;
	_pCallback = pCallback;
	_running = false;
	ReadConfig();
	_initialized = true;
}

void WaveOut::Reset()
{
	if (_running) Stop();
}

void WaveOut::Configure()
{
	ReadConfig();

	CWaveOutDialog dlg;
	dlg.m_BufNum = _numBlocks;
	dlg.m_BufSize = _blockSize;
	dlg.m_Device = _deviceID;
	dlg.m_Dither = _dither;
	dlg.m_SampleRate = _samplesPerSec;

	if (dlg.DoModal() != IDOK)
	{
		return;
	}
	
	int oldnb = _numBlocks;
	int oldbs = _blockSize;
	int olddid = _deviceID;
	int olddither = _dither;
	int oldsps = _samplesPerSec;

	if (_initialized)
	{
		Stop();
	}

	_numBlocks = dlg.m_BufNum;
	_blockSize = dlg.m_BufSize;
	_deviceID = dlg.m_Device;
	_dither = dlg.m_Dither;
	_samplesPerSec = dlg.m_SampleRate;

	_configured = true;

	if (_initialized)
	{
		if (Start())
		{
			WriteConfig();
		}
		else
		{
			_numBlocks = oldnb;
			_blockSize = oldbs;
			_deviceID = olddid;
			_dither = olddither;
			_samplesPerSec = oldsps;

			Start();
		}
	}
	else
	{
		WriteConfig();
	}
	
}

int WaveOut::GetPlayPos()
{
	if (!_running)
	{
		return 0;
	}
	MMTIME time;
	time.wType = TIME_SAMPLES;

	if (waveOutGetPosition(_handle, &time, sizeof(MMTIME)) != MMSYSERR_NOERROR)
	{
		Error("waveOutGetPosition() : Wave ޲^錻݂̍ĐʒǔɎs܂");
	}
	if (time.wType != TIME_SAMPLES)
	{
		Error("TIME_SAMPLES T|[gĂȂ̂ŁAwaveOutGetPosition() so܂");
	}
	return (time.u.sample & ((1 << 23) - 1));
}

int WaveOut::GetWritePos()
{
	if (!_running)
	{
		return 0;
	}
	return (_writePos & ((1 << 23) - 1));
}

bool WaveOut::Enable(bool e)
{
	return e ? Start() : Stop();
}

