<?php
/*! @file
 *@brief
 * PSGraph
 *  折れ線グラフと、棒グラフ
 *
 *@author hirohito
 *@note
 */
/* This file is distributed under BSD licence. Please read the COPYRIGHT file. */


include_once 'psgraph_base.php';



//================================================
/*!@brief
 * 座標軸スーパークラス
 *
 */
abstract class PSGraph_axis extends PSGraph_base
{
    protected $ScaleMax = null;			//!< 目盛り最大値
    protected $ScaleMin = null;			//!< 目盛り最小値
    protected $ScaleInterval = null;		//!< 目盛り幅
    protected $ScaleMaxUser = null;		//!< ユーザ設定目盛り最大値
    protected $ScaleMinUser = null;		//!< ユーザ設定目盛り最小値
    protected $ScaleIntervalUser = null;	//!< ユーザ設定目盛り幅
    protected $ScaleMaxMinWidth = null;		//!< 最大値-最小値のキャッシュ
    protected $FlagReverse = false;		//!< 目盛り逆方向（反転）フラグ

    public $AtScaleLine = array( 'Stroke'=>'black', 'StrokeWidth'=>1 );		//!< 軸アトリビュート
    public $AtIntervalMarks = array( 'Length'=>8,
				     'Stroke'=>'#999999', 'StrokeWidth'=>1 );	//!< 目盛アトリビュート
    public $AtLabels = array( 'FontSize'=>10 );					//!< 目盛ラベルアトリビュート
    public $Labels = null;							//!< 目盛ラベル(配列)

    abstract function do_scaling();
    abstract function calc_pixcel_position( $v );


    //================================================================
    // setter, getter and others.
    //================================================================
    //! 目盛り最大値のユーザ指定
    function setMax( $v ) { $this->ScaleMaxUser = $v; }

    //! 最大値取得
    function getMax() { return $this->ScaleMax; }
    
    //! 最小値取得
    function getMin() { return $this->ScaleMin; }
    
    //! 目盛り最小値のユーザ指定
    function setMin( $v ) { $this->ScaleMinUser = $v; }

    //! 目盛り幅のユーザ指定
    function setInterval( $v ) { $this->ScaleIntervalUser = $v; }

    //! 目盛り逆方向（反転）指示
    function setReverse( $f = true ) { $this->FlagReverse = $f; }

    //! グリッド線の付与
    function addGrid() { $this->AtIntervalMarks['Grid'] = true; }

    //! グリッド線の消去
    function clearGrid() { $this->AtIntervalMarks['Grid'] = false; }

    //! 軸線の消去
    function clearScaleLine() {	$this->AtScaleLine = null; }

    //! 間隔マークの消去
    function clearIntervalMarks() { $this->AtIntervalMarks = null; }

    //! 軸ラベルの設定
    function setLabels( array $labels ) { $this->Labels = $labels; }

    //! 軸ラベルの消去
    function clearLabels() { $this->AtLabels = null; }

}



//================================================
/*!@brief
 * X軸クラス
 *
 */
class PSGraph_xaxis extends PSGraph_axis
{
    protected $ModeDrawPosition = "LEFT";	//!< 軸描画モード {LEFT|CENTER}


    //================================================================
    /*! constructor
     *
     *@param	width	幅
     *@param	height	高さ
     *@param	output	出力制御オブジェクト
     */
    function __construct( $width, $height, PSGraph_output &$output )
    {
	parent::__construct( $width, $height, $output );
	$this->AtIntervalMarks['Grid'] = false;
    }

    
    //================================================================
    /*! 軸描画モード変更
     *
     *@param	mode	モード {LEFT|CENTER}
     */
    function changeMode( $mode ) {
	switch( $mode ) {
	case 'LEFT':
	case 'CENTER':
	    $this->ModeDrawPosition = $mode;
	    break;
	}
    }


    //================================================================
    // setter, getter and others.
    //================================================================
    //! 目盛り最大値のユーザ指定
    function setMax( $v )	// override.
    {
	$this->ScaleMaxUser = $v - 1;
    }

    //! 目盛り最小値のユーザ指定
    function setMin( $v )	// override.
    {
	$this->ScaleMinUser = $v - 1;
    }


    //================================================================
    /*! 目盛りスケーリング
     *
     *@retval	bool	成功 or 不成功
     *@note
     * あらかじめ与えられているデータ系列情報などを元に、
     * オートスケール処理など、内部データの整合性をとる。
     */
    function do_scaling()
    {
	$this->ScaleMin = 0;
	$this->ScaleMax = null;
	$this->ScaleInterval = 1;

	/*
	 * maximum quantity amang registered datas.
	 */
	foreach( $this->DataSeries as $ds ) {
	    $num = count( $ds->YData ) - 1;
	    if( $this->ScaleMax === null || $this->ScaleMax < $num ) $this->ScaleMax = $num;
	}
	if( $this->ScaleMax !== null && $this->ScaleMax < 1 ) $this->ScaleMax = 1;

	/*
	 * refrect user settings
	 */
	if( $this->ScaleMaxUser !== null ) $this->ScaleMax = $this->ScaleMaxUser;
	if( $this->ScaleMinUser !== null ) $this->ScaleMin = $this->ScaleMinUser;
	if( $this->ScaleIntervalUser !== null ) $this->ScaleInterval = $this->ScaleIntervalUser;

	if( $this->ScaleMax === null ) return false;
	$this->ScaleMaxMinWidth = $this->ScaleMax - $this->ScaleMin;
	if( $this->ScaleMaxMinWidth == 0 ) $this->ScaleMaxMinWidth = 1;

	/*
	 * check scale intarval value, and adjust.
	 */
	if( $this->Width > 0 && (($this->ScaleMax - $this->ScaleMin) / $this->ScaleInterval) > $this->Width ) {
	    $this->ScaleInterval = ($this->ScaleMax - $this->ScaleMin) / ($this->Width / 2);
	}

	return true;
    }


    //================================================================
    /*! 軸上のピクセル位置を計算する。
     *
     *@param	v	実数
     *@retval	int	ピクセル位置
     *@note
     * 引数が軸上にない場合、返り値も軸上にはないピクセル位置が返る。
     */
    function calc_pixcel_position( $v )
    {
	switch( $this->ModeDrawPosition ) {
	case 'LEFT':
	    return (int)( $this->Width * ($v - $this->ScaleMin) / $this->ScaleMaxMinWidth );
	case 'CENTER':
	    return (int)(( 2 * $this->Width * ($v - $this->ScaleMin) + $this->Width ) /
			 ($this->ScaleMaxMinWidth+1) / 2);
	}
	return 0;
    }

    
    //================================================================
    /*! 描画（ダミー）
     *
     *@note
     * スケール単独で描画することはないので、エラーとなる。
     * 実際の描画は、上下関係(z軸)の関係から、２パスに分ける。
     */
    function draw()
    {
	trigger_error( "This function was replaced by draw_z1,draw_z2.", E_USER_ERROR );
    }


    //================================================================
    /*! 描画　1st pass
     *
     *@note
     * スケール描画　パス１。
     */
    function draw_z1()
    {
	/*
	 * draw interval marks
	 */
	if( $this->AtIntervalMarks ) {
	    $this->Output->printf( "\n<!-- draw X-axis pass 1 -->\n" );
	    $this->Output->printf( "<g %s>\n", $this->make_common_attribute_string( $this->AtIntervalMarks ) );
	    $y1 = $this->Height;
	    $y2 = $this->Height;
	    if( $this->AtIntervalMarks['Length'] < 0 ) {
		$y1 = $this->Height - $this->AtIntervalMarks['Length'];
	    } else {
		$y2 = $this->Height - $this->AtIntervalMarks['Length'];
	    }
	    if( $this->AtIntervalMarks['Grid'] ) {
		$y2 = 0;
	    }

	    switch( $this->ModeDrawPosition ) {
	    case 'LEFT':
		for( $i = $this->ScaleMin; $i <= $this->ScaleMax; $i += $this->ScaleInterval ) {
		    $x = $this->calc_pixcel_position( $i );
		    $this->Output->printf( "  <line x1=\"%d\" y1=\"%d\" x2=\"%d\" y2=\"%d\" />\n", $x, $y1, $x, $y2 );
		}
		break;

	    case 'CENTER':
		$max_loops = ($this->ScaleMax - $this->ScaleMin) / $this->ScaleInterval + 1;
		for( $i = 0; $i <= $max_loops; $i++ ) {
		    $x = (int)($this->Width * $i / $max_loops);
		    $this->Output->printf( "  <line x1=\"%d\" y1=\"%d\" x2=\"%d\" y2=\"%d\" />\n", $x, $y1, $x, $y2 );
		}
		break;
	    }
	    $this->Output->printf( "</g>\n" );
	}
    }

    
    //================================================================
    /*! 描画　2nd pass
     *
     *@note
     * スケール描画　パス２。
     */
    function draw_z2()
    {
	$this->Output->printf( "\n<!-- draw X-axis pass 2 -->\n" );

	/*
	 * draw scale line
	 */
	if( $this->AtScaleLine ) {
	    $this->Output->printf( "<line x1=\"%d\" y1=\"%d\" x2=\"%d\" y2=\"%d\" %s/>\n",
				   0, $this->Height, $this->Width, $this->Height,
				   $this->make_common_attribute_string( $this->AtScaleLine ) );
	}

	/*
	 * draw labels
	 */
	if( $this->AtLabels ) {

	    // テキストアンカー位置の決定
	    if( !isset( $this->AtLabels['TextAnchor'] ) ) {
		$this->AtLabels['TextAnchor'] = 'middle';

		// 回転させる時は、回転角に応じて自動調整
		if( isset( $this->AtLabels['Rotate'] ) ) {
		    if( 0 < $this->AtLabels['Rotate'] && $this->AtLabels['Rotate'] < 180 ) {
			$this->AtLabels['TextAnchor'] = 'start';
		    } elseif( -180 < $this->AtLabels['Rotate'] && $this->AtLabels['Rotate'] < 0 ) {
			$this->AtLabels['TextAnchor'] = 'end';
		    }
		}
	    }

	    $this->Output->printf( "<g %s>\n", $this->make_common_attribute_string( $this->AtLabels ) );

	    // ラベル出力
	    for( $i = 0; ($v = $this->ScaleMin + $this->ScaleInterval * $i) <= $this->ScaleMax; $i++ ) {
		$x = $this->calc_pixcel_position( $v );
		$y = $this->Height + $this->AtLabels['FontSize'] + 5;
		if( ! isset( $this->AtLabels['Rotate'] ) ) {
		    $this->Output->printf( '  <text x="%d" y="%d" >', $x, $y );
		} else {
		    $this->Output->printf( '  <text x="%d" y="%d" transform="rotate(%d,%d,%d)" >', $x, $y,
			    $this->AtLabels['Rotate'], $x, $y - $this->AtLabels['FontSize'] / 2 );
		}

		if( $this->Labels !== null ) {
		    $v = isset( $this->Labels[$v] )? htmlspecialchars( $this->Labels[$v] ): "";
		} else {
		    $v++;
		}
		$this->Output->printf( "$v</text>\n" );
	    }
	    $this->Output->printf( "</g>\n" );
	}
    }
}



//================================================
/*!@brief
 * Y軸クラス
 *
 */
class PSGraph_yaxis extends PSGraph_axis
{
    //================================================================
    /*! constructor
     *
     *@param	width	幅
     *@param	height	高さ
     *@param	output	出力制御オブジェクト
     */
    function __construct( $width, $height, PSGraph_output &$output )
    {
	parent::__construct( $width, $height, $output );
	$this->AtLabels['TextAnchor'] = 'end';
	$this->AtIntervalMarks['Grid'] = true;
    }
    

    //================================================================
    /*! 目盛りスケーリング
     *
     *@retval	bool	成功 or 不成功
     *@note
     * あらかじめ与えられているデータ系列情報などを元に、
     * オートスケール処理など、内部データの整合性をとる。
     */
    function do_scaling()
    {
	$max = null;
	$min = null;
	$interval = null;

	/*
	 * search max and min value.
	 */
	foreach( $this->DataSeries as $ds ) {
	    if( $max === null || $max < $ds->YDataMax ) $max = $ds->YDataMax;
	    if( $min === null || $min > $ds->YDataMin ) $min = $ds->YDataMin;
	}
	if( $max == $min ) {
	    $max += abs($max) * 0.5;
	    $min -= abs($min) * 0.5;
	}

	/*
	 * max point adjustment.
	 */
	$diff = $max - $min;
	$max += $diff * 0.1;

	/*
	 * zero point adjustment.
	 */
	$diff = $max - $min;
	if( $min > 0 && $diff*2 > $min ) {
	    $min = 0;
	} elseif( $max < 0 && $diff*2 > -$max ) {
	    $max = 0;
	}

	/*
	 * refrect user settings.
	 */
	if( $this->ScaleMaxUser !== null) $max = $this->ScaleMaxUser;
	if( $this->ScaleMinUser !== null) $min = $this->ScaleMinUser;
	if( $this->ScaleIntervalUser !== null ) $interval = $this->ScaleIntervalUser;

	/*
	 * auto scaling.
	 */
	$diff = $max - $min;
	if( $diff > 0 ) {

	    // calc interval.
	    if( $this->ScaleIntervalUser === null ) {
		$interval = pow( 10, floor( log10( $diff ) ) - 1 );
		$tick = $diff / $interval;
		if( $tick > 50 )
		    $interval *= 10;
		elseif( $tick > 20 )
		    $interval *= 5;
		elseif( $tick > 10 )
		    $interval *= 2;
	    }

	    // max and min point adjustment.
	    if( $this->ScaleMaxUser === null ) $max = ceil( $max / $interval ) * $interval;
	    if( $this->ScaleMinUser === null ) $min = floor( $min / $interval ) * $interval;
	}

	if( $interval === null ) return false;

	$this->ScaleMax = $max;
	$this->ScaleMin = $min;
	$this->ScaleInterval = $interval;
	$this->ScaleMaxMinWidth = $this->ScaleMax - $this->ScaleMin;
	if( $this->ScaleMaxMinWidth == 0 ) $this->ScaleMaxMinWidth = 1;

	/*
	 * check scale intarval value, and fource adjust.
	 */
	if( $this->Width > 0 && (($max - $min) / $interval) > $this->Width ) {
	    $this->ScaleInterval = ($max - $min) / ($this->Width / 2);
	}
	
	return true;
    }


    //================================================================
    /*! 軸上のピクセル位置を計算する。
     *
     *@param	v	実数
     *@retval	int	ピクセル位置
     *@note
     * 引数が軸上にない場合、返り値も軸上にはないピクセル位置が返る。
     */
    function calc_pixcel_position( $v )
    {
	if( $this->FlagReverse )
	    return (int)( $this->Height * ($v - $this->ScaleMin) / $this->ScaleMaxMinWidth );
	else
	    return $this->Height - (int)( $this->Height * ($v - $this->ScaleMin) / $this->ScaleMaxMinWidth );
    }


    //================================================================
    /*! 描画（ダミー）
     *
     *@note
     * スケール単独で描画することはないので、エラーとなる。
     * 実際の描画は、上下関係(z軸)の関係から、２パスに分ける。
     */
    function draw()
    {
	trigger_error( "This function was replaced by draw_z1,draw_z2.", E_USER_ERROR );
    }

    
    //================================================================
    /*! 描画　1st pass
     *
     *@note
     * スケール描画　パス１。
     */
    function draw_z1()
    {
	/*
	 * draw interval marks
	 */
	if( $this->AtIntervalMarks ) {
	    $this->Output->printf( "\n<!-- draw Y-axis pass 1 -->\n" );
	    
	    $x1 = 0;
	    $x2 = 0;
	    if( $this->AtIntervalMarks['Length'] < 0 ) {
		$x1 = $this->AtIntervalMarks['Length'];
	    } else {
		$x2 = $this->AtIntervalMarks['Length'];
	    }
	    if( $this->AtIntervalMarks['Grid'] ) {
		$x2 = $this->Width;
	    }

	    $this->Output->printf( "<g %s>\n", $this->make_common_attribute_string( $this->AtIntervalMarks ) );
	    for( $i = 0; ($v = $this->ScaleMin + $this->ScaleInterval * $i) <= $this->ScaleMax; $i++ ) {
		$y = $this->calc_pixcel_position( $v );
		$this->Output->printf( "  <line x1=\"%d\" y1=\"%d\" x2=\"%d\" y2=\"%d\" />\n", $x1, $y, $x2, $y );
	    }
	    $this->Output->printf( "</g>\n" );
	}
    }


    //================================================================
    /*! 描画　2nd pass
     *
     *@note
     * スケール描画　パス２。
     */
    function draw_z2()
    {
	$this->Output->printf( "\n<!-- draw Y-axis pass 2 -->\n" );

	/*
	 * draw scale line
	 */
	if( $this->AtScaleLine ) {
	    $this->Output->printf( "<line x1=\"%d\" y1=\"%d\" x2=\"%d\" y2=\"%d\" %s/>\n", 0, 0, 0, $this->Height,
				   $this->make_common_attribute_string( $this->AtScaleLine ) );
	}

	/*
	 * draw labels
	 */
	if( $this->AtLabels ) {
	    $this->Output->printf( "<g %s>\n", $this->make_common_attribute_string( $this->AtLabels ) );

	    for( $i = 0; ($v = $this->ScaleMin + $this->ScaleInterval * $i) <= $this->ScaleMax; $i++ ) {
		$this->Output->printf( '  <text x="%d" y="%d">',
			-5, $this->calc_pixcel_position( $v ) + $this->AtLabels['FontSize'] / 2 );

		if( $this->Labels !== null ) {
		    $v = isset( $this->Labels[$i] )? htmlspecialchars( $this->Labels[$i] ): "";
		}

		if( isset( $this->AtLabels['Format'] ) ) {
		    $this->Output->printf( "{$this->AtLabels['Format']}</text>\n", $v );
		} else {
		    $this->Output->printf( "$v</text>\n" );
		}
	    }
	    $this->Output->printf( "</g>\n" );
	}
    }
}



//================================================
/*!@brief
 * 第２Ｙ軸クラス
 *
 */
class PSGraph_y2axis extends PSGraph_yaxis
{
    //================================================================
    /*! constructor
     *
     *@param	width	幅
     *@param	height	高さ
     *@param	output	出力制御オブジェクト
     */
    function __construct( $width, $height, PSGraph_output &$output )
    {
	parent::__construct( $width, $height, $output );
	$this->AtLabels['TextAnchor'] = 'start';
	$this->AtIntervalMarks['Grid'] = false;
    }


    //================================================================
    /*! 描画　1st pass
     *
     *@note
     * スケール描画　パス１。
     */
    function draw_z1()
    {
	/*
	 * draw interval marks
	 */
	if( $this->AtIntervalMarks ) {
	    $this->Output->printf( "\n<!-- draw Y2-axis pass 1 -->\n" );

	    $x1 = $this->Width;
	    $x2 = $this->Width;
	    if( $this->AtIntervalMarks['Length'] < 0 ) {
		$x1 = $this->Width - $this->AtIntervalMarks['Length'];
	    } else {
		$x2 = $this->Width - $this->AtIntervalMarks['Length'];
	    }
	    if( $this->AtIntervalMarks['Grid'] ) {
		$x2 = 0;
	    }
	    
	    $this->Output->printf( "<g %s\n>", $this->make_common_attribute_string( $this->AtIntervalMarks ) );
	    for( $i = 0; ($v = $this->ScaleMin + $this->ScaleInterval * $i) <= $this->ScaleMax; $i++ ) {
		$y = $this->calc_pixcel_position( $v );
		$this->Output->printf( "  <line x1=\"%d\" y1=\"%d\" x2=\"%d\" y2=\"%d\" />\n", $x1, $y, $x2, $y );
	    }
	    $this->Output->printf( "</g>\n" );
	}
    }


    //================================================================
    /*! 描画　2nd pass
     *
     *@note
     * スケール描画　パス２。
     */
    function draw_z2()
    {
	$this->Output->printf( "\n<!-- draw Y2-axis pass 2 -->\n" );

	/*
	 * draw scale line
	 */
	if( $this->AtScaleLine ) {
	    $this->Output->printf( "<line x1=\"%d\" y1=\"%d\" x2=\"%d\" y2=\"%d\" %s/>\n",
				   $this->Width, 0, $this->Width, $this->Height,
				   $this->make_common_attribute_string( $this->AtScaleLine ) );
	}

	/*
	 * draw labels
	 */
	if( $this->AtLabels ) {
	    $this->Output->printf( "<g %s>\n", $this->make_common_attribute_string( $this->AtLabels ) );

	    for( $i = 0; ($v = $this->ScaleMin + $this->ScaleInterval * $i) <= $this->ScaleMax; $i++ ) {
		$this->Output->printf( '  <text x="%d" y="%d">', $this->Width + 5,
			$this->calc_pixcel_position( $v ) + $this->AtLabels['FontSize'] / 2 );

		if( $this->Labels !== null ) {
		    $v = isset( $this->Labels[$i] )? htmlspecialchars( $this->Labels[$i] ): "";
		}

		if( isset( $this->AtLabels['Format'] ) ) {
		    $this->Output->printf( "{$this->AtLabels['Format']}</text>\n", $v );
		} else {
		    $this->Output->printf( "$v</text>\n" );
		}
	    }
	    $this->Output->printf( "</g>\n" );
	}
    }
}



//================================================
/*!@brief
 * 折れ線と棒グラフ用クラス
 *
 *@note
 * ユーザがnewして利用するクラス。
 */
class PSGraph extends PSGraph_util
{
    //! マーカ形状リスト（折れ線グラフのみ）
    static protected $SHAPELIST = array( 'cock', 'circle', 'rectangle', 'diamond', 'triangle' );

    //! プロットエリアアトリビュート
    public $AtPlotArea = array( 'X'=>40, 'Y'=>10, 'Fill'=>'#eee' );
    
    public $AtXAxisTitle = null;	//!< Ｘ軸タイトルアトリビュート
    public $AtYAxisTitle = null;	//!< Ｙ軸タイトルアトリビュート
    public $XAxis;			//!< Ｘ軸オブジェクト
    public $YAxis;			//!< Ｙ軸オブジェクト
    public $Y2Axis = null;		//!< Ｙ２軸オブジェクト（もしあれば）
    public $LinePlot = null;		//!< 折れ線グラフオブジェクト
    public $BarPlot = null;		//!< 棒グラフオブジェクト
    /* (note)
       Y2Axisオブジェクト、折れ線グラフオブジェクト、棒グラフオブジェクトは、必要に応じて作られる。
       つまり、必要なければ null なので、注意する。
    */


    //================================================================
    /*! constructor
     *
     *@param	width	幅
     *@param	height	高さ
     */
    function __construct( $width = 320, $height = 240 )
    {
	parent::__construct( $width, $height );

	/*
	 * make default
	 */
	$this->AtPlotArea['Width'] = $this->Width - 50;
	if( $this->AtPlotArea['Width'] < 0 ) $this->AtPlotArea['Width'] = 0;
	$this->AtPlotArea['Height'] = $this->Height - 30;
	if( $this->AtPlotArea['Height'] < 0 ) $this->AtPlotArea['Height'] = 0;
	$this->XAxis = new PSGraph_xaxis( $this->AtPlotArea['Width'],
					  $this->AtPlotArea['Height'], $this->Output );
	$this->YAxis = new PSGraph_yaxis( $this->AtPlotArea['Width'],
					  $this->AtPlotArea['Height'], $this->Output );
    }


    //================================================================
    /*! 折れ線の追加
     *
     *@param	ydata	データの配列
     *@param	legend	データの名前（凡例）
     */
    function addDataLine( array $ydata, $legend = null )
    {
	$data_obj = $this->add_data_line_common( $ydata, $legend );
	$data_obj->set_attribute( $this->XAxis, $this->YAxis, $this->LinePlot,
		self::$COLORLIST[ count( $this->DataSeries ) % count( self::$COLORLIST ) ],
		self::$SHAPELIST[ count( $this->DataSeries ) % count( self::$SHAPELIST ) ] );
	$this->XAxis->add_data_series( $data_obj );
	$this->YAxis->add_data_series( $data_obj );

	return $data_obj;
    }


    //================================================================
    /*! 折れ線の第２Ｙ軸上への追加
     *
     *@param	ydata	データの配列
     *@param	legend	データの名前（凡例）
     */
    function addDataLineY2( array $ydata, $legend = null )
    {
	$data_obj = $this->add_data_line_common( $ydata, $legend );

	if( $this->Y2Axis === null ) {
	    $this->Y2Axis = new PSGraph_y2axis( $this->AtPlotArea['Width'],
						$this->AtPlotArea['Height'], $this->Output );
	    $right = $this->Width - $this->AtPlotArea['Width'] - $this->AtPlotArea['X'] + 20;
	    $this->setMargin( null, $right, null, null );
	}

	$data_obj->set_attribute( $this->XAxis, $this->Y2Axis, $this->LinePlot,
		self::$COLORLIST[ count( $this->DataSeries ) % count( self::$COLORLIST ) ],
		self::$SHAPELIST[ count( $this->DataSeries ) % count( self::$SHAPELIST ) ] );
	$this->XAxis->add_data_series( $data_obj );
	$this->Y2Axis->add_data_series( $data_obj );

	return $data_obj;
    }


    //================================================================
    /*! 折れ線の追加　内部処理
     *
     *@param	ydata	データの配列
     *@param	legend	データの名前（凡例）
     */
    protected function add_data_line_common( $ydata, $legend )
    {
	$data_obj = new PSGraph_container_line( $ydata, $legend );
	$this->DataSeries[] = $data_obj;

	if( $this->LinePlot === null ) {
	    $this->LinePlot = new PSGraph_line_plot( $this->Output );
	}
	$this->LinePlot->add_data_series( $data_obj );

	if( ! $this->AtLegend && $data_obj->Legend ) {
	    $this->addLegend();
	}

	return $data_obj;
    }


    //================================================================
    /*! 棒グラフの追加
     *
     *@param	ydata	データの配列
     *@param	legend	データの名前（凡例）
     *@param	base_bar 積み重ねする場合、ベースになるデータコンテナ
     */
    function addDataBar( array $ydata, $legend = null, $base_bar = null )
    {
	$data_obj = $this->add_data_bar_common( $ydata, $legend, $base_bar );
	$data_obj->set_attribute( $this->XAxis, $this->YAxis, $this->BarPlot,
		self::$COLORLIST[ count( $this->DataSeries ) % count( self::$COLORLIST ) ] );

	$this->XAxis->add_data_series( $data_obj );
	$this->YAxis->add_data_series( $data_obj );

	return $data_obj;
    }


    //================================================================
    /*! 棒グラフの第２Ｙ軸上への追加
     *
     *@param	ydata	データの配列
     *@param	legend	データの名前（凡例）
     *@param	base_bar 積み重ねする場合、ベースになるデータコンテナ
     */
    function addDataBarY2( array $ydata, $legend = null, $base_bar = null )
    {
	$data_obj = $this->add_data_bar_common( $ydata, $legend, $base_bar );

	if( $this->Y2Axis === null ) {
	    $this->Y2Axis = new PSGraph_y2axis( $this->AtPlotArea['Width'],
						$this->AtPlotArea['Height'], $this->Output );
	    $right = $this->Width - $this->AtPlotArea['Width'] - $this->AtPlotArea['X'] + 20;
	    $this->setMargin( null, $right, null, null );
	}

	$data_obj->set_attribute( $this->XAxis, $this->Y2Axis, $this->BarPlot,
		self::$COLORLIST[ count( $this->DataSeries ) % count( self::$COLORLIST ) ] );
	$this->XAxis->add_data_series( $data_obj );
	$this->Y2Axis->add_data_series( $data_obj );

	return $data_obj;
    }


    //================================================================
    /*! 棒グラフの追加　内部処理
     *
     *@param	ydata	データの配列
     *@param	legend	データの名前（凡例）
     *@param	base_bar 積み重ねする場合、ベースになるデータコンテナ
     */
    protected function add_data_bar_common( $ydata, $legend, $base_bar )
    {
	/*
	 * 積み重ねの場合、Y値を調整。
	 */
	if( $base_bar ) {
	    for( $i = 0; $i < count($ydata); $i++ ) {
		$ydata[$i] += $base_bar->YData[$i];
	    }
	}

	/*
	 * コンテナオブジェクトの生成
	 */
	$data_obj = new PSGraph_container_bar( $ydata, $legend );

	/*
	 * コンテナを配列に保存
	 */
	if( $base_bar ) {
	    $data_obj->set_stack( $base_bar );
	    $i = 0;
	    foreach( $this->DataSeries as $ds ) {
		if( $ds == $base_bar ) {
		    array_splice( $this->DataSeries, $i, 0, array($data_obj) );
		    break;
		}
		$i++;
	    }
	} else {
	    $this->DataSeries[] = $data_obj;
	}

	/*
	 * その他必要なオブジェクトを生成
	 */
	if( $this->BarPlot === null ) {
	    $this->BarPlot = new PSGraph_bar_plot( $this->Output );
	}
	$this->BarPlot->add_data_series( $data_obj, $base_bar );
	$this->XAxis->changeMode( 'CENTER' );

	if( ! $this->AtLegend && $data_obj->Legend ) {
	    $this->addLegend();
	}

	return $data_obj;
    }


    //================================================================
    /*! 描画
     *
     *@note
     * 管理下のオブジェクトを次々とcallして、全体を描画する。
     * （ユーザが個々の内部オブジェクトのdrawメソッドを使うことは無い）
     */
    function draw()
    {
	/*
	 * scaling.
	 */
	if( ! $this->XAxis->do_scaling() ||
	    ! $this->YAxis->do_scaling() ) {
	    trigger_error( "Wrong data. Can't auto scaling.", E_USER_WARNING );
	    return;
	}
	if( $this->Y2Axis && ! $this->Y2Axis->do_scaling() ) {
	    trigger_error( "Wrong data. Can't auto scaling.", E_USER_WARNING );
	    return;
	}

	/*
	 * draw base items.
	 */
	$this->draw_common1();

	/*
	 * output plot area's clipping path.
	 */
	$this->Output->printf( "<clipPath id=\"plotarea\">\n" );
	$this->Output->printf( "  <rect x=\"%d\" y=\"%d\" width=\"%d\" height=\"%d\" />\n",
			       -5, -5, $this->AtPlotArea['Width'] + 10, $this->AtPlotArea['Height'] + 10 );
	$this->Output->printf( "</clipPath>\n" );

	/*
	 * grouping in plot items.
	 */
	$this->Output->printf( "<g transform=\"translate(%d,%d)\">\n",
		$this->AtPlotArea['X'], $this->AtPlotArea['Y'] );

	/*
	 * draw X,Y axis
	 */
	$this->XAxis->draw_z1();
	$this->YAxis->draw_z1();
	if( $this->Y2Axis ) $this->Y2Axis->draw_z1();

	$this->XAxis->draw_z2();
	$this->YAxis->draw_z2();
	if( $this->Y2Axis ) $this->Y2Axis->draw_z2();

	/*
	 * draw data series.
	 */
	$this->Output->printf( "\n<!-- draw lines and bars in clipping path -->\n" );
	$this->Output->printf( "<g clip-path=\"url(#plotarea)\">\n" );

	if( $this->BarPlot ) $this->BarPlot->draw();
	if( $this->LinePlot ) $this->LinePlot->draw();

	$this->Output->printf( "</g><!-- end of clip -->\n" );

	/*
	 * end of group
	 */
	$this->Output->printf( "</g><!-- end of plot area -->\n\n" );
	
	/*
	 * draw legend
	 */
	if( $this->AtLegend ) {
	    if( !isset( $this->AtLegend['Y'] ) ) {
		$this->AtLegend['Y'] = ($this->Height - count( $this->DataSeries )
					* ($this->AtLegend['FontSize'] + 4)) / 2;
		if( $this->AtLegend['Y'] < 0 ) $this->AtLegend['Y'] = 0;
	    }

	    $attr = $this->AtLegend;
	    $attr['X'] += 10;
	    $attr['Y'] += $attr['FontSize'];

	    foreach( $this->DataSeries as $ds ) {
		$this->Output->printf( "<text %s>%s</text>\n",
			$this->make_common_attribute_string( $attr ), htmlspecialchars( $ds->Legend ) );
		$ds->Plot->draw_legend_marker( $attr['X']-10, $attr['Y']-$attr['FontSize']/3, $ds );
		$attr['Y'] += $this->AtLegend['FontSize'] + 4;
	    }
	}

	/*
	 * draw x-axis title
	 */
	if( $this->AtXAxisTitle ) {
	    if( ! isset($this->AtXAxisTitle['X']) ) {
		$this->AtXAxisTitle['X'] = $this->AtPlotArea['X']
		    + $this->AtPlotArea['Width'] / 2;
	    }
	    if( ! isset( $this->AtXAxisTitle['Y'] ) ) {
		$this->AtXAxisTitle['Y'] = $this->Height - 5;
	    }
	    $this->Output->printf( "<text %s>%s</text>\n",
		    $this->make_common_attribute_string( $this->AtXAxisTitle ),
		    htmlspecialchars( $this->AtXAxisTitle['Title'] ) );
	}

	/*
	 * draw y-axis title
	 */
	if( $this->AtYAxisTitle ) {
	    if( ! isset( $this->AtYAxisTitle['X'] ) ) {
		$this->AtYAxisTitle['X'] = $this->AtYAxisTitle['FontSize'] + 5;
	    }
	    if( ! isset( $this->AtYAxisTitle['Y'] ) ) {
		$this->AtYAxisTitle['Y'] = $this->AtPlotArea['Y']
		    + $this->AtPlotArea['Height'] / 2;
	    }
	    $this->Output->printf( "<text %s transform=\"rotate(-90,%d,%d)\">%s</text>\n",
		    $this->make_common_attribute_string( $this->AtYAxisTitle ),
		    $this->AtYAxisTitle['X'], $this->AtYAxisTitle['Y'],
		    htmlspecialchars( $this->AtYAxisTitle['Title'] ) );
	}

	$this->draw_common2();
    }


    //================================================================
    /*! プロットエリアのマージン設定
     *
     *@param	top	上マージン
     *@param	right	右マージン
     *@param	bottom	下マージン
     *@param	left	左マージン
     *@note
     * 上下左右個別に設定できる。
     * 設定値を変えない場合は、そのパラメータをnullにしてcallする。
     */
    function setMargin( $top, $right, $bottom, $left )
    {
	parent::setMargin( $top, $right, $bottom, $left );

	// set axis objects parameters
	$this->XAxis->Width = $this->AtPlotArea['Width'];
	$this->XAxis->Height = $this->AtPlotArea['Height'];
	$this->YAxis->Width = $this->AtPlotArea['Width'];
	$this->YAxis->Height = $this->AtPlotArea['Height'];
	if( $this->Y2Axis ) {
	    $this->Y2Axis->Width = $this->AtPlotArea['Width'];
	    $this->Y2Axis->Height = $this->AtPlotArea['Height'];
	}
    }


    //================================================================
    /*! Ｘ軸タイトルの追加
     *
     *@param	title_string	タイトル文字列
     */
    function addXAxisTitle( $title_string )
    {
	$this->setMargin( null, null, 35, null );
	$this->AtXAxisTitle = array( 'Title'=>$title_string, 'FontSize'=>12,
				     'TextAnchor'=>'middle' );
    }


    //================================================================
    /*! Ｙ軸タイトルの追加
     *
     *@param	title_string	タイトル文字列
     */
    function addYAxisTitle( $title_string )
    {
	$this->setMargin( null, null, null, 50 );
	$this->AtYAxisTitle = array( 'Title'=>$title_string, 'FontSize'=>12,
				     'TextAnchor'=>'middle' );
    }

}



//================================================
/*!@brief
 * 折れ線グラフプロットクラス
 *
 */
class PSGraph_line_plot extends PSGraph_base
{
    //================================================================
    /*! constructor
     *
     *@param	output	出力制御オブジェクト
     */
    function __construct( PSGraph_output &$output )
    {
	$this->Output = &$output;
    }


    //================================================================
    /*! 描画
     *
     *@note
     * 管理下のデータコンテナすべてについて、実際に描画を行う。
     */
    function draw()
    {
	foreach( $this->DataSeries as $ds ) {
	    $this->Output->printf( "\n<!-- draw line-plot '$ds->Legend' -->\n" );

	    /*
	     * Plot poly line.
	     */
	    if( $ds->AtPlotLine ) {
		$this->Output->printf( '<polyline points="' );
		$i = -1;
		foreach( $ds->YData as $yd ) {
		    $i++;
		    if( $i < $ds->XAxis->getMin() ) continue;
		    if( $i > $ds->XAxis->getMax() ) break;
		    if( $yd === null ) continue;
		    
		    $x = $ds->XAxis->calc_pixcel_position( $i );
		    $y = $ds->YAxis->calc_pixcel_position( $yd );
		    $this->Output->printf( "%d,%d ", $x, $y );
		}

		$this->Output->printf( "\" %s/>\n", $this->make_common_attribute_string( $ds->AtPlotLine ) );
	    }

	    /*
	     * Plot markers.
	     */
	    if( $ds->AtMarker ) {
		$this->Output->printf( "<g %s>\n", $this->make_common_attribute_string( $ds->AtMarker ) );
		$i = -1;
		foreach( $ds->YData as $yd ) {
		    $i++;
		    if( $i < $ds->XAxis->getMin() ) continue;
		    if( $i > $ds->XAxis->getMax() ) break;
		    if( $yd === null ) continue;
		    
		    $x = $ds->XAxis->calc_pixcel_position( $i );
		    $y = $ds->YAxis->calc_pixcel_position( $yd );
		    $this->draw_marker( $x, $y, $ds->AtMarker['Shape'], "", $yd );
		}
		$this->Output->printf( "</g>\n" );
	    }

	    /*
	     * Data labels.
	     */
	    if( $ds->AtDataLabels ) {
		$this->Output->printf( "<g %s>\n", $this->make_common_attribute_string( $ds->AtDataLabels ) );
		$i = -1;
		foreach( $ds->YData as $yd ) {
		    $i++;
		    if( $i < $ds->XAxis->getMin() ) continue;
		    if( $i > $ds->XAxis->getMax() ) break;
		    if( $yd === null ) continue;
		    
		    $x = $ds->XAxis->calc_pixcel_position( $i );
		    $y = $ds->YAxis->calc_pixcel_position( $yd );
		    switch( $ds->AtDataLabels['Position'] ) {
		    case 'ABOVE': $y -= 6; break;
		    case 'BELOW': $y += $ds->AtDataLabels['FontSize'] + 6; break;
		    case 'LEFT':  $x -= 6; $y += $ds->AtDataLabels['FontSize'] / 2; break;
		    case 'RIGHT': $x += 6; $y += $ds->AtDataLabels['FontSize'] / 2; break;
		    case 'CENTER': $y += $ds->AtDataLabels['FontSize'] / 2; break;
		    }
		    
		    $this->Output->printf( '  <text x="%d" y="%d" >', $x, $y );
		    if( isset( $ds->AtDataLabels['Format'] ) ) {
			$this->Output->printf( "{$ds->AtDataLabels['Format']}</text>", $yd );
		    } else {
			$this->Output->printf( "$yd</text>\n" );
		    }
		}
		$this->Output->printf( "</g>\n" );
	    }
	}
    }


    //================================================================
    /*! マーカーを描画する
     *
     *@param	x	Ｘ値
     *@param	y	Ｙ値
     *@param	kind	マーカ種類
     *@param	attrstr	付加アトリビュート文字列
     *@param	title	付加タイトル
     */
    function draw_marker( $x, $y, $kind, $attrstr, $title )
    {
	switch( $kind ) {
	case 'circle':
	    $this->Output->printf( "  <circle cx=\"%d\" cy=\"%d\" r=\"4\" %s title=\"%s\" />\n",
				   $x, $y, $attrstr, htmlspecialchars( $title ) );
	    break;
	    
	case 'rectangle':
	    $this->Output->printf( "  <rect x=\"%d\" y=\"%d\" width=\"7\" height=\"7\" %s title=\"%s\" />\n",
				   $x-3, $y-3, $attrstr, htmlspecialchars( $title ) );
	    break;

	case 'diamond':
	    $this->Output->printf( "  <polygon points=\"%d,%d %d,%d %d,%d %d,%d\" %s title=\"%s\" />\n",
				   $x-5, $y, $x, $y-5, $x+5, $y, $x, $y+5,
				   $attrstr, htmlspecialchars( $title ) );
	    break;

	case 'triangle':
	    $this->Output->printf( "  <polygon points=\"%d,%d %d,%d %d,%d\" %s title=\"%s\" />\n",
				   $x, $y-5, $x-5, $y+5, $x+5, $y+5,
				   $attrstr, htmlspecialchars( $title ) );
	    break;

	case 'cock':
	    $this->Output->printf( "  <polygon points=\"%d,%d %d,%d %d,%d %d,%d\" %s title=\"%s\" />\n",
				   $x-4, $y-4, $x+4, $y+4, $x+4, $y-4, $x-4, $y+4,
				   $attrstr, htmlspecialchars( $title ) );
	}
    }


    //================================================================
    /*! 凡例部マーカー描画
     *
     *@param	x	Ｘ値
     *@param	y	Ｙ値
     *@retval	data_obj	データコンテナオブジェクト
     */
    function draw_legend_marker( $x, $y, $data_obj )
    {
	if( $data_obj->AtPlotLine ) {
	    $this->Output->printf( "<line x1=\"%d\" y1=\"%d\" x2=\"%d\" y2=\"%d\" %s />\n",
				   $x - 9, $y, $x + 9, $y,
				   $this->make_common_attribute_string( $data_obj->AtPlotLine ) );
	}
	if( $data_obj->AtMarker ) {
	    $this->draw_marker( $x, $y, $data_obj->AtMarker['Shape'],
				$this->make_common_attribute_string( $data_obj->AtMarker ), "" );
	}
    }
}



//================================================
/*!@brief
 * バーグラフプロットクラス
 *
 */
class PSGraph_bar_plot extends PSGraph_base
{
    protected $Overlap = 0;		//!< 棒のオーバーラップ率（％）
    protected $Spacing = 100;		//!< 棒の間隔率（％：100%で軸とスペースが同じ幅）


    //================================================================
    /*! constructor
     *
     *@param	output	出力制御オブジェクト
     */
    function __construct( PSGraph_output &$output )
    {
	$this->Output = &$output;
    }


    //================================================================
    /*! データコンテナ追加
     *
     *@param	data_obj	データコンテナオブジェクト
     *@param	base_bar	積み重ねする場合、ベースになるデータコンテナ
     */
    function add_data_series( $data_obj, $base_bar )
    {
	if( $base_bar ) {
	    $i = 0;
	    foreach( $this->DataSeries as $ds ) {
		if( $ds == $base_bar ) {
		    array_splice( $this->DataSeries, $i, 0, array($data_obj) );
		    break;
		}
		$i++;
	    }
	} else {
	    $this->DataSeries[] = $data_obj;
	}
    }


    //================================================================
    /*! 棒どうしのオーバーラップ率指定
     *
     *@param	v	オーバーラップ率 (%)
     *@note
     * 0から100を指定する。
     */
    function setOverlap( $v )
    {
	$this->Overlap = $v;
    }


    //================================================================
    /*! 棒どうしの間隔指定
     *
     *@param	v	間隔率 (%)
     *@note
     * 0から100を指定する。
     */
    function setSpacing( $v )
    {
	$this->Spacing = $v;
    }


    //================================================================
    /*! 描画
     *
     *@note
     * 管理下のデータコンテナすべてについて、実際に描画を行う。
     */
    function draw()
    {
	$num = count( $this->DataSeries );
	foreach( $this->DataSeries as $ds ) {
	    if( $ds->BaseContainer ) $num--;
	}
	
	$ov = 1 - $this->Overlap / 100;				// 棒のオーバーラップ率
	$sp = $this->Spacing / 100;				// 棒幅に対する棒間の率
	$w_all = $this->DataSeries[0]->XAxis->calc_pixcel_position( 1 )
	       - $this->DataSeries[0]->XAxis->calc_pixcel_position( 0 );	// 全幅 (px)
	$w_b = $w_all / ( 1 + $ov * ($num - 1) + $sp );				// 棒幅 (px)
	$w_s = $w_b * $sp / 2;							// 間隔幅 (px)

	$n = 0;
	foreach( $this->DataSeries as $ds ) {
	    $this->Output->printf( "\n<!-- draw bar-plot '$ds->Legend' -->\n" );

	    /*
	     * Draw bar (1 series)
	     */
	    $this->Output->printf( "<g %s>\n", $this->make_common_attribute_string( $ds->AtBar ) );
	    $i = -1;
	    foreach( $ds->YData as $yd ) {
		$i++;
		if( $i < $ds->XAxis->getMin() ) continue;
		if( $i > $ds->XAxis->getMax() ) break;
		if( $yd === null ) continue;
		
		$x = $ds->XAxis->calc_pixcel_position( $i ) - ($w_all / 2);
		$x1 = $x + $w_s + $n * $w_b * $ov;
		$x2 = $x1 + $w_b;
		if( ! $ds->BaseContainer ) {
		    $y1 = $ds->YAxis->calc_pixcel_position( 0 );
		} else {
		    $y = $ds->BaseContainer->YData[$i];
		    $y1 = $ds->YAxis->calc_pixcel_position( $y );
		}
		$y2 = $ds->YAxis->calc_pixcel_position( $yd );
		if( $y1 != $y2 ) {
		    $this->Output->printf( "  <polyline points=\"%.2f,%.2f %.2f,%.2f %.2f,%.2f %.2f,%.2f\" title=\"%s\"/>\n",
			    $x1,$y1, $x1,$y2, $x2,$y2, $x2,$y1, htmlspecialchars($yd) );
		}
	    }
	    $this->Output->printf( "</g>\n" );

	    /*
	     * Data labels.
	     */
	    if( $ds->AtDataLabels ) {
		$this->Output->printf( "<g %s>\n", $this->make_common_attribute_string( $ds->AtDataLabels ) );
		$i = -1;
		foreach( $ds->YData as $yd ) {
		    $i++;
		    if( $i < $ds->XAxis->getMin() ) continue;
		    if( $i > $ds->XAxis->getMax() ) break;
		    if( $yd === null ) continue;
		    
		    $x = $ds->XAxis->calc_pixcel_position( $i ) - ($w_all / 2) + $w_s + $n * $w_b * $ov;
		    $y = $ds->YAxis->calc_pixcel_position( $yd );
		    switch( $ds->AtDataLabels['Position'] ) {
		    case 'ABOVE': $x += $w_b/2; $y -= 6; break;
		    case 'BELOW': $x += $w_b/2; $y += $ds->AtDataLabels['FontSize'] + 6; break;
		    case 'LEFT':  $x -= 3;      $y += $ds->AtDataLabels['FontSize'] / 2; break;
		    case 'RIGHT': $x += $w_b+3; $y += $ds->AtDataLabels['FontSize'] / 2; break;
		    case 'CENTER':$x += $w_b/2; $y += $ds->AtDataLabels['FontSize'] / 2; break;
		    }
		    
		    $this->Output->printf( '  <text x="%d" y="%d" >', $x, $y );
		    if( isset( $ds->AtDataLabels['Format'] ) ) {
			$this->Output->printf( "{$ds->AtDataLabels['Format']}</text>", $yd );
		    } else {
			$this->Output->printf( "$yd</text>\n" );
		    }
		}
		$this->Output->printf( "</g>\n" );
	    }

	    if( ! $ds->BaseContainer ) $n++;
	}
    }


    //================================================================
    /*! 凡例部マーカー描画
     *
     *@param	x	Ｘ値
     *@param	y	Ｙ値
     *@retval	data_obj	データコンテナオブジェクト
     */
    function draw_legend_marker( $x, $y, $data_obj )
    {
	$this->Output->printf( "  <rect x=\"%d\" y=\"%d\" width=\"8\" height=\"8\" %s />\n",
		$x-3, $y-3, $this->make_common_attribute_string( $data_obj->AtBar ) );
    }
}



//================================================
/*!@brief
 * データコンテナ　スーパークラス
 *
 */
abstract class PSGraph_data_container
{
    public $YData;		//!< Y値データ配列
    public $YDataMax;		//!< Y値最大値
    public $YDataMin;		//!< Y値最小値
    public $AtDataLabels = null; //!< データラベルアトリビュート
    public $Legend;		//!< 凡例文字列

    public $XAxis;		//!< 使用するＸ軸オブジェクト
    public $YAxis;		//!< 使用するＹ軸オブジェクト
    public $Plot;		//!< 使用するプロットオブジェクト


    //================================================================
    /*! constructor
     *
     *@param	ydata	Ｙ値データ配列
     *@param	legend	凡例文字列
     *@note
     */
    function __construct( array $ydata, $legend = null )
    {
	$this->YData = $ydata;
	$this->YDataMax = null;
	$this->YDataMin = null;
	foreach( $ydata as $y ) {
	    if( $y === null ) continue;
	    if( $this->YDataMax === null || $this->YDataMax < $y ) $this->YDataMax = $y;
	    if( $this->YDataMin === null || $this->YDataMin > $y ) $this->YDataMin = $y;
	}
	$this->Legend = $legend;
    }


    //================================================================
    /*! 値ラベルを表示
     *
     *@param	pos	位置 {ABOVE|BELOW|LEFT|RIGHT|CENTER}
     *@note
     * 位置以外は、デフォルト値で表示するよう設定。
     */
    function addDataLabels( $pos = 'ABOVE' )
    {
	switch( $pos ) {
	case 'ABOVE':
	case 'BELOW':
	case 'CENTER':
	    $this->AtDataLabels = array( 'Position'=>$pos, 'FontSize'=>9, 'TextAnchor'=>'middle' );
	    break;
	    
	case 'LEFT':
	    $this->AtDataLabels = array( 'Position'=>$pos, 'FontSize'=>9, 'TextAnchor'=>'end' );
	    break;
	    
	case 'RIGHT':
	    $this->AtDataLabels = array( 'Position'=>$pos, 'FontSize'=>9, 'TextAnchor'=>'start' );
	    break;
	}	
    }


    abstract function setColor( $color );
}



//================================================
/*!@brief
 * 折れ線グラフ用データコンテナ
 *
 *@note
 * 線を消してマーカのみの表示にすることもできる。
 */
class PSGraph_container_line extends PSGraph_data_container
{
    public $AtPlotLine = array( 'StrokeWidth'=>2, 'Fill'=>'none' );
				//!< 線の描画アトリビュート
    public $AtMarker = array( 'Format'=>null, 'Stroke'=>'black', 'StrokeWidth'=>2 );
				//!< マーカーの描画アトリビュート


    //================================================================
    /*! 内部利用アトリビュート類の設定
     *
     *@param	xaxis	Ｘ軸オブジェクト
     *@param	yaxis	Ｙ軸オブジェクト
     *@param	plot	プロットオブジェクト
     *@param	color	色
     *@param	shape	マーカー形状
     */
    function set_attribute( $xaxis, $yaxis, $plot, $color, $shape )
    {
	$this->XAxis = $xaxis;
	$this->YAxis = $yaxis;
	$this->Plot = $plot;

	if( $this->AtPlotLine && !isset( $this->AtPlotLine['Stroke'] ) ) {
	    $this->AtPlotLine['Stroke'] = $color;
	}
	
	if( $this->AtMarker && !isset( $this->AtMarker['Fill'] ) ) {
	    $this->AtMarker['Fill'] = $color;
	}

	if( $this->AtMarker && !isset( $this->AtMarker['Shape'] ) ) {
	    $this->AtMarker['Shape'] = $shape;
	}
    }


    //================================================================
    // setter, getter and others.
    //================================================================
    //! 線を表示しない
    function clearLine() { $this->AtPlotLine = null; }

    //! マーカーを表示しない
    function clearMarker() { $this->AtMarker = null; }


    //================================================================
    /*! 色の指定
     *
     *@param	color	色
     *@note
     * 折れ線の場合、線とマーカーの両方の色変えなければならないので、
     * アトリビュートを2ヶ所変更するよりも簡単にするために作成。
     */
    function setColor( $color )
    {
	if( $this->AtPlotLine ) $this->AtPlotLine['Stroke'] = $color;
	if( $this->AtMarker ) $this->AtMarker['Fill'] = $color;
    }
}



//================================================
/*!@brief
 * バーグラフ用データコンテナ
 *
 */
class PSGraph_container_bar extends PSGraph_data_container
{
    public $AtBar = array( 'StrokeWidth'=>1, 'Stroke'=>'black' );
					//!< バーの描画アトリビュート
    public $BaseContainer = null;	//!< 積み重ねグラフの時、下になるコンテナオブジェクト


    //================================================================
    /*! 積み重ね設定
     *
     *@param	base	ベースになるデータコンテナ
     *@note
     */
    function set_stack( $base )
    {
	$this->BaseContainer = $base;
    }


    //================================================================
    /*! 内部利用アトリビュート類の設定
     *
     *@param	xaxis	Ｘ軸オブジェクト
     *@param	yaxis	Ｙ軸オブジェクト
     *@param	plot	プロットオブジェクト
     *@param	color	色
     */
    function set_attribute( $xaxis, $yaxis, $plot, $color )
    {
	$this->XAxis = $xaxis;
	$this->YAxis = $yaxis;
	$this->Plot = $plot;

	if( $this->AtBar && !isset( $this->AtBar['Fill'] ) ) {
	    $this->AtBar['Fill'] = $color;
	}
    }


    //================================================================
    /*! 色の指定
     *
     *@param	color	色
     *@note
     * PSGraph_container_line::setColor() との対称性のため定義。
     */
    function setColor( $color )
    {
	if( $this->AtBar ) $this->AtBar['Fill'] = $color;
    }
}
