/****************************************************************
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License or any later
 * version.  This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details. You should have received a copy
 * of the GNU Lesser General Public License along with this program; if not,
 * write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307 USA.
 ****************************************************************
 * copyright (c) 2005,2006 TOMITA, Ryuzo(r-tomita01@sda.nagoya-cu.ac.jp)
 * $Id: rast.c,v 1.12 2006/02/03 18:54:46 ryuzo Exp $ 
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "apr_pools.h"
#include <apr_strings.h>

#undef PACKAGE_NAME
#undef PACKAGE_TARNAME
#undef PACKAGE_VERSION
#undef PACKAGE_STRING

#include "rast/config.h"
#include "rast/text_index.h"
#include "rast/local_db.h"
#include "rast/query.h"
#include "rast/merger.h"

#include "php.h"
#include "php_ini.h"
#include "ext/standard/info.h"
#include "php_rast.h"

/* True global resources - no need for thread safety here */
static int le_rast_db;
static int le_rast_result;

/* {{{ structures */
typedef struct {
	apr_pool_t *pool;
	rast_db_t *db;
	int closed;
} php_rast_data;

typedef struct {
	rast_result_t *result;
	int num_properties;
	const char** properties;
} php_rast_result;
/* }}} */

/* {{{ rast_functions[]
 *
 * Every user visible function must have an entry in rast_functions[].
 */
function_entry rast_functions[] = {
	PHP_FE(rast_db_create,NULL)
	PHP_FE(rast_open,NULL)
	PHP_FE(rast_merger_open,NULL)
	PHP_FE(rast_close,NULL)
	PHP_FE(rast_register,NULL)
	PHP_FE(rast_update,NULL)
	PHP_FE(rast_delete,NULL)
	PHP_FE(rast_search,NULL)
	PHP_FE(rast_get_encoding,NULL)
	PHP_FE(rast_get_properties,NULL)
	PHP_FE(rast_result_get_terms,NULL)
	PHP_FE(rast_result_get_hit_count,NULL)
	PHP_FE(rast_result_get_items,NULL)
	{NULL, NULL, NULL}	/* Must be the last line in rast_functions[] */
};
/* }}} */

/* {{{ rast_module_entry
 */
zend_module_entry rast_module_entry = {
#if ZEND_MODULE_API_NO >= 20010901
	STANDARD_MODULE_HEADER,
#endif
	"rast",
	rast_functions,
	PHP_MINIT(rast),
	PHP_MSHUTDOWN(rast),
	NULL,		/* Replace with NULL if there's nothing to do at request start */
	NULL,	/* Replace with NULL if there's nothing to do at request end */
	PHP_MINFO(rast),
#if ZEND_MODULE_API_NO >= 20010901
	"0.1", /* Replace with version number for your extension */
#endif
	STANDARD_MODULE_PROPERTIES
};
/* }}} */

#ifdef COMPILE_DL_RAST
ZEND_GET_MODULE(rast)
#endif

/* {{{ helper functions */
static int apr_memory_error(int retcode)
{
	zend_error(E_ERROR,"apr memory error");
	return -1;
}

void php_rast_free_db(zend_rsrc_list_entry *rsrc TSRMLS_DC)
{
	php_rast_data *data = (php_rast_data *) rsrc->ptr;
	if(!data->closed){
		rast_db_close(data->db);
	}
	apr_pool_destroy(data->pool);
	efree(data);
}

void php_rast_free_result(zend_rsrc_list_entry *rsrc TSRMLS_DC)
{
	php_rast_result *result = (php_rast_result *) rsrc->ptr;
	efree(result);
}

static rast_value_t * get_property_values (zval *zproperties, php_rast_data *data)
{
	const rast_property_t *properties;
	zval **zproperty;
	rast_value_t *property_values;
	int i, num_properties;

	properties = rast_db_properties(data->db, &num_properties);
	property_values = (rast_value_t *) apr_palloc(data->pool, sizeof(rast_value_t) * num_properties);;
	for (i = 0; i < num_properties; i++) {
		const rast_property_t *property = properties + i;
		if(zend_hash_find(Z_ARRVAL_P(zproperties), property->name, strlen(property->name)+1,
				(void **) &zproperty) == SUCCESS) {
			switch (property->type){
				case RAST_TYPE_STRING:
				case RAST_TYPE_DATE:
					convert_to_string_ex(zproperty);
					rast_value_set_string(property_values + i, Z_STRVAL_PP(zproperty));
					break;
				case RAST_TYPE_UINT:
					convert_to_long_ex(zproperty);
					rast_value_set_uint(property_values + i, Z_LVAL_PP(zproperty));
					break;
				default:
					zend_error(E_ERROR, "unknown property type");
			}
		}
	}
	return property_values;
}

static rast_property_t *
get_properties(apr_pool_t *pool, zval **zproperties, int *num_properties)
{
	zval **zpdata, **zpdata2;
    rast_property_t *properties;
	int i, num_elements;
	HashTable *pht;
	rast_property_t *property;

	convert_to_array_ex(zproperties);
	pht = HASH_OF(*zproperties);
	num_elements = zend_hash_num_elements(pht);
    properties = (rast_property_t *) apr_palloc(pool,
                                                sizeof(rast_property_t) *
                                                num_elements);

	for (i = 0; i < num_elements; i++) {
		zend_hash_get_current_data(pht, (void **) &zpdata);
		convert_to_array_ex(zpdata);

		property = properties + i;

		if(zend_hash_find(Z_ARRVAL_PP(zpdata), "type", sizeof("type"), (void **) &zpdata2) == SUCCESS) {
			convert_to_long_ex(zpdata2);
			property->type = Z_LVAL_PP(zpdata2);
		} else {
			zend_error(E_WARNING, "property has no type");
		}
		if(zend_hash_find(Z_ARRVAL_PP(zpdata), "flags", sizeof("flags"), (void **) &zpdata2) == SUCCESS) {
			convert_to_long_ex(zpdata2);
			property->flags = Z_LVAL_PP(zpdata2);
		} else {
			zend_error(E_WARNING, "property has no flags");
		}

		if(zend_hash_find(Z_ARRVAL_PP(zpdata), "name", sizeof("name"), (void **) &zpdata2) == SUCCESS) {
			convert_to_string_ex(zpdata2);
			property->name = apr_pstrdup(pool, Z_STRVAL_PP(zpdata2));
		} else {
			zend_error(E_WARNING, "property has no name");
		}
		zend_hash_move_forward(pht);	
	}
	*num_properties = num_elements;
	return properties;
}
/* }}} */

/* {{{ PHP_INI
 */
/* Remove comments and fill if you need to have entries in php.ini
PHP_INI_BEGIN()
    STD_PHP_INI_ENTRY("rast.global_value",      "42", PHP_INI_ALL, OnUpdateInt, global_value, zend_rast_globals, rast_globals)
    STD_PHP_INI_ENTRY("rast.global_string", "foobar", PHP_INI_ALL, OnUpdateString, global_string, zend_rast_globals, rast_globals)
PHP_INI_END()
*/
/* }}} */

/* {{{ php_rast_init_globals
 */
/* Uncomment this function if you have INI entries
static void php_rast_init_globals(zend_rast_globals *rast_globals)
{
	rast_globals->global_value = 0;
	rast_globals->global_string = NULL;
}
*/
/* }}} */

/* {{{ PHP_MINIT_FUNCTION
 */
PHP_MINIT_FUNCTION(rast)
{

	rast_error_t *error;

	le_rast_db = zend_register_list_destructors_ex(php_rast_free_db, NULL, "Rast DB", module_number);
	le_rast_result = zend_register_list_destructors_ex(php_rast_free_result, 
			NULL, "Rast Search Result", module_number);
	REGISTER_LONG_CONSTANT("RAST_LITTLE_ENDIAN", RAST_LITTLE_ENDIAN, CONST_CS|CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("RAST_BIG_ENDIAN", RAST_BIG_ENDIAN, CONST_CS|CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("RAST_NATIVE_ENDIAN", RAST_NATIVE_ENDIAN, CONST_CS|CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("RAST_DB_RDWR", RAST_DB_RDWR, CONST_CS|CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("RAST_DB_RDONLY", RAST_DB_RDONLY, CONST_CS|CONST_PERSISTENT);

	/* obsolete */
	REGISTER_LONG_CONSTANT("RAST_RDWR", RAST_DB_RDWR, CONST_CS|CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("RAST_RDONLY", RAST_DB_RDONLY, CONST_CS|CONST_PERSISTENT);

	REGISTER_LONG_CONSTANT("RAST_SORT_METHOD_SCORE", RAST_SORT_METHOD_SCORE, CONST_CS|CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("RAST_SORT_METHOD_PROPERTY", 
			RAST_SORT_METHOD_PROPERTY, CONST_CS|CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("RAST_RESULT_ALL_ITEMS", RAST_RESULT_ALL_ITEMS, CONST_CS|CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("RAST_SORT_ORDER_DEFAULT", RAST_SORT_ORDER_DEFAULT, CONST_CS|CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("RAST_SORT_ORDER_ASCENDING", 
			RAST_SORT_ORDER_ASCENDING, CONST_CS|CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("RAST_SORT_ORDER_DESCENDING", 
			RAST_SORT_ORDER_DESCENDING, CONST_CS|CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("RAST_SCORE_METHOD_TFIDF", RAST_SCORE_METHOD_TFIDF, CONST_CS|CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("RAST_SCORE_METHOD_NONE", RAST_SCORE_METHOD_NONE, CONST_CS|CONST_PERSISTENT);

	REGISTER_LONG_CONSTANT("RAST_PROPERTY_FLAG_OMIT",
			RAST_PROPERTY_FLAG_OMIT, CONST_CS|CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("RAST_PROPERTY_FLAG_SEARCH", 
			RAST_PROPERTY_FLAG_SEARCH, CONST_CS|CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("RAST_PROPERTY_FLAG_TEXT_SEARCH", 
			RAST_PROPERTY_FLAG_TEXT_SEARCH, CONST_CS|CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("RAST_PROPERTY_FLAG_FULL_TEXT_SEARCH", 
			RAST_PROPERTY_FLAG_FULL_TEXT_SEARCH, CONST_CS|CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("RAST_PROPERTY_FLAG_UNIQUE", 
			RAST_PROPERTY_FLAG_UNIQUE, CONST_CS|CONST_PERSISTENT);

	REGISTER_LONG_CONSTANT("RAST_TYPE_STRING", RAST_TYPE_STRING, CONST_CS|CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("RAST_TYPE_DATE", RAST_TYPE_DATE, CONST_CS|CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("RAST_TYPE_UINT", RAST_TYPE_UINT, CONST_CS|CONST_PERSISTENT);

	apr_initialize();
	error = rast_initialize();
	if (error == RAST_OK) {
		return SUCCESS;
	} else {
		zend_error(E_WARNING,error->message);
		rast_error_destry(error);
		return FAILURE;
	}
}
/* }}} */

/* {{{ PHP_MSHUTDOWN_FUNCTION
 */
PHP_MSHUTDOWN_FUNCTION(rast)
{
	/* uncomment this line if you have INI entries
	UNREGISTER_INI_ENTRIES();
	*/
	rast_finalize();
	apr_terminate();

	return SUCCESS;
}
/* }}} */

/* {{{ PHP_RINIT_FUNCTION
 */
PHP_RINIT_FUNCTION(rast)
{
	return SUCCESS;
}
/* }}} */

/* {{{ PHP_RSHUTDOWN_FUNCTION
 */
PHP_RSHUTDOWN_FUNCTION(rast)
{
	return SUCCESS;
}
/* }}} */

/*  {{{ PHP_MINFO_FUNCTION
 */
PHP_MINFO_FUNCTION(rast)
{
	php_info_print_table_start();
	php_info_print_table_row(2, "rast support", "enabled");
	php_info_print_table_row(2, "Rast version", RAST_VERSION);
	php_info_print_table_end();

	/* Remove comments if you have entries in php.ini
	DISPLAY_INI_ENTRIES();
	*/
}
/* }}} */

/* {{{ proto rast_db_create(string dbpath, array options) */
PHP_FUNCTION(rast_db_create)
{
	zval *zoptions;
	zval **zoptvalue;
	char *dbpath;
	int dbpath_len;
	rast_db_create_option_t *options;
	apr_pool_t *pool;
	rast_error_t *rast_error;

	if (FAILURE == zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "sa", 
				&dbpath, &dbpath_len, &zoptions)) {
		return;
	}

	apr_pool_create_ex(&pool, NULL, apr_memory_error, NULL);
	options = rast_db_create_option_create(pool);

	if (zend_hash_find(Z_ARRVAL_P(zoptions), "byte_order", sizeof("byte_order"),
				(void **) &zoptvalue) == SUCCESS) {
		convert_to_long_ex(zoptvalue);
		options->byte_order = Z_LVAL_PP(zoptvalue);
	}

	if (zend_hash_find(Z_ARRVAL_P(zoptions), "pos_block_size", sizeof("pos_block_size"),
				(void **) &zoptvalue) == SUCCESS) {
		convert_to_long_ex(zoptvalue);
		options->pos_block_size = Z_LVAL_PP(zoptvalue);
	}

	if (zend_hash_find(Z_ARRVAL_P(zoptions), "encoding", sizeof("encoding"),
				(void **) &zoptvalue) == SUCCESS) {
		convert_to_string_ex(zoptvalue);
		options->encoding = Z_STRVAL_PP(zoptvalue);
	}

	if (zend_hash_find(Z_ARRVAL_P(zoptions), "preserve_text", sizeof("preserve_text"),
				(void **) &zoptvalue) == SUCCESS) {
		convert_to_long_ex(zoptvalue);
		options->preserve_text = Z_LVAL_PP(zoptvalue);
	}

	if (zend_hash_find(Z_ARRVAL_P(zoptions), "properties", sizeof("properties"),
				(void **) &zoptvalue) == SUCCESS) {
		options->properties = get_properties(pool, zoptvalue, &options->num_properties);
	}

	rast_error = rast_db_create(dbpath, options, pool);
	if (rast_error != RAST_OK) {
		zend_error(E_WARNING, rast_error->message);
		rast_error_destroy(rast_error);
		RETURN_FALSE;
	}
	RETURN_TRUE;
}
/* }}} */

/* {{{ proto resource rast_open(string dbpath, const mode)
	open rast database */
PHP_FUNCTION(rast_open)
{
	char *dbpath;
	int dbpath_len;
    int flag = RAST_RDONLY;
	rast_error_t *rast_error;
	rast_db_t *db;
	php_rast_data *data;
	apr_pool_t *pool;

	if (FAILURE == zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "sl",
				&dbpath, &dbpath_len, &flag)) {
		return;
	}

	apr_pool_create_ex(&pool, NULL, apr_memory_error, NULL);
	rast_error = rast_db_open(&db, dbpath, flag, NULL, pool);

	if (rast_error != RAST_OK) {
		zend_error(E_WARNING, rast_error->message);
		apr_pool_destroy(pool);
		rast_error_destroy(rast_error);
		RETURN_FALSE;
	}

	data = (php_rast_data *) ecalloc(1,sizeof(php_rast_data));
	data->db = db;
	data->pool = pool;
	data->closed = 0;
	ZEND_REGISTER_RESOURCE(return_value, data, le_rast_db);
}
/* }}} */

/* {{{ proto resource rast_merger_open(array resources) */
PHP_FUNCTION(rast_merger_open)
{
	zval *zdbary;
	zval *zdb;
	php_rast_data *data, *mdata;
	apr_pool_t *pool;
	rast_db_t *db;
    rast_db_t **dbs;
	rast_error_t *rast_error;
	HashTable *pht;
    int num_dbs, i;

	if (FAILURE == zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "a",
				&zdbary)) {
		return;
	}

	apr_pool_create_ex(&pool, NULL, apr_memory_error, NULL);
	convert_to_array_ex(&zdbary);
	pht = HASH_OF(zdbary);
	num_dbs = zend_hash_num_elements(pht);
	dbs = (rast_db_t **) apr_palloc(pool, sizeof(rast_db_t *) * num_dbs);
	zend_hash_internal_pointer_reset(pht);
	for (i=0; i<zend_hash_num_elements(pht); i++){
		zend_hash_get_current_data(pht, (void *) &zdb);
		ZEND_FETCH_RESOURCE(data, php_rast_data *, &zdb, -1, "Rast DB Handle", le_rast_db);
		dbs[i] = data->db;
	}

	rast_error = rast_merger_open(&db, dbs, num_dbs, pool);
    if (rast_error != RAST_OK) {
		zend_error(E_WARNING, rast_error->message);
        apr_pool_destroy(pool);
		rast_error_destroy(rast_error);
    }
	mdata = (php_rast_data *) ecalloc(1, sizeof(php_rast_data));
	mdata->db = db;
	mdata->pool = pool;
	mdata->closed = 0;
	ZEND_REGISTER_RESOURCE(return_value, mdata, le_rast_db);
}
/* }}} */

/* {{{ proto bool rast_close(resource db)
	close an open rast database */
PHP_FUNCTION(rast_close)
{
	zval *zres;
	php_rast_data *data;
	if (FAILURE == zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "r", &zres)) {
		return;
	}
	ZEND_FETCH_RESOURCE(data, php_rast_data *, &zres, -1, "Rast DB Handle", le_rast_db);

	if(!data->closed){
		rast_db_close(data->db);
		data->closed = 1;
	}
	zend_list_delete(Z_RESVAL_P(zres));
	RETURN_TRUE;
}
/* }}} */

/* {{{ proto int rast_register(resource db, string text, array properties)
	returns doc id or false
*/
PHP_FUNCTION(rast_register)
{
	zval *zres, *zproperties;
	char *text;
	int text_len;
    rast_doc_id_t doc_id;
	rast_value_t *property_values;
	rast_error_t *rast_error;
	php_rast_data *data;

	if (FAILURE == zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "rsa", 
				&zres, &text, &text_len, &zproperties)) {
		return;
	}
	ZEND_FETCH_RESOURCE(data, php_rast_data *, &zres, -1, "Rast DB Handle", le_rast_db);
	
	property_values = get_property_values(zproperties, data);
	rast_error = rast_db_register(data->db, text, text_len, property_values, &doc_id);
	if (rast_error != RAST_OK){
		zend_error(E_WARNING, rast_error->message);
		rast_error_destroy(rast_error);
		RETURN_FALSE;
	}
	RETURN_LONG(doc_id);
}
/* }}} */

/* {{{ proto int rast_update(resource db, int docid, string text,  array properties)
*/
PHP_FUNCTION(rast_update)
{
	zval *zres, *zproperties;
	char *text;
	int text_len;
   	rast_doc_id_t old_doc_id, new_doc_id;
	rast_value_t *property_values;
	rast_error_t *rast_error;
	php_rast_data *data;

	if (FAILURE == zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "rlsa", 
				&zres, &old_doc_id, &text, &text_len, &zproperties)) {
		return;
	}
	ZEND_FETCH_RESOURCE(data, php_rast_data *, &zres, -1, "Rast DB Handle", le_rast_db);
	property_values = get_property_values(zproperties, data);
	rast_error = rast_db_update(data->db, old_doc_id, text, text_len, property_values, &new_doc_id);
	if (rast_error != RAST_OK){
		zend_error(E_WARNING, rast_error->message);
		rast_error_destroy(rast_error);
		RETURN_FALSE;
	}
	RETURN_LONG(new_doc_id);
}
/* }}} */

/* {{{ proto int rast_delete(resource db, int docid)
*/
PHP_FUNCTION(rast_delete)
{
	zval *zres;
	int doc_id;
	php_rast_data *data;
	rast_error_t *rast_error;

	if (FAILURE == zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "rl", 
				&zres, &doc_id)) {
		return;
	}
	ZEND_FETCH_RESOURCE(data, php_rast_data *, &zres, -1, "Rast DB Handle", le_rast_db);
	rast_error = rast_db_delete(data->db, doc_id);
	if (rast_error != RAST_OK){
		zend_error(E_WARNING, rast_error->message);
		rast_error_destroy(rast_error);
		RETURN_FALSE;
	}
	RETURN_FALSE;
}
/* }}} */

/* {{{ proto bool rast_search(resource db, string query, array option)
*/
PHP_FUNCTION(rast_search)
{
	zval *zres, *zopt;
	zval **zoptvalue, **zpdata;
	HashTable *pht;
	int q_len, psize, i;
	char *query;
	php_rast_data *data;
	php_rast_result *result_rsrc;
	rast_search_option_t *options;
	rast_result_t *result;
	rast_error_t *rast_error;

	if (FAILURE == zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "rsa", &zres, &query, &q_len, &zopt)) {
		return;
	}

	ZEND_FETCH_RESOURCE(data, php_rast_data *, &zres, -1, "Rast DB Handle", le_rast_db);
	options = rast_search_option_create(data->pool);

	if (zend_hash_find(Z_ARRVAL_P(zopt), "start_no", sizeof("start_no"), 
				(void **) &zoptvalue) == SUCCESS) {
		convert_to_long_ex(zoptvalue);
		options->start_no = Z_LVAL_PP(zoptvalue);
	}
	if (zend_hash_find(Z_ARRVAL_P(zopt), "num_items", sizeof("num_items"), 
				(void **) &zoptvalue) == SUCCESS) {
		convert_to_long_ex(zoptvalue);
		options->num_items = Z_LVAL_PP(zoptvalue);
	}
	if (zend_hash_find(Z_ARRVAL_P(zopt), "need_summary", sizeof("need_summary"), 
				(void **) &zoptvalue) == SUCCESS) {
		convert_to_long_ex(zoptvalue);
		options->need_summary = Z_LVAL_PP(zoptvalue);
	}
	if (zend_hash_find(Z_ARRVAL_P(zopt), "summary_nchars", sizeof("summary_nchars"), 
				(void **) &zoptvalue) == SUCCESS) {
		convert_to_long_ex(zoptvalue);
		options->summary_nchars = Z_LVAL_PP(zoptvalue);
	}
	if (zend_hash_find(Z_ARRVAL_P(zopt), "sort_method", sizeof("sort_method"), 
				(void **) &zoptvalue) == SUCCESS) {
		convert_to_long_ex(zoptvalue);
		options->sort_method = Z_LVAL_PP(zoptvalue);
	}
	if (zend_hash_find(Z_ARRVAL_P(zopt), "sort_property", sizeof("sort_property"), 
				(void **) &zoptvalue) == SUCCESS) {
		convert_to_string_ex(zoptvalue);
		options->sort_property = apr_pstrdup(data->pool, Z_STRVAL_PP(zoptvalue));
	}
	if (zend_hash_find(Z_ARRVAL_P(zopt), "sort_order", sizeof("sort_order"), 
				(void **) &zoptvalue) == SUCCESS) {
		convert_to_long_ex(zoptvalue);
		options->sort_order = Z_LVAL_PP(zoptvalue);
	}

	if (zend_hash_find(Z_ARRVAL_P(zopt), "properties", sizeof("properties"),
				(void **) &zoptvalue) == SUCCESS) {

		convert_to_array_ex(zoptvalue);
		pht = HASH_OF(*zoptvalue);
		options->num_properties = zend_hash_num_elements(pht);

		psize = (zend_hash_num_elements(pht) + 1) * sizeof(char *);
		options->properties = (const char **) apr_palloc(data->pool, psize);

		zend_hash_internal_pointer_reset(pht);
		for (i=0;i<zend_hash_num_elements(pht); i++) {
			zend_hash_get_current_data(pht, (void **) &zpdata);
			convert_to_string_ex(zpdata);
			options->properties[i] = apr_pstrdup(data->pool, Z_STRVAL_PP(zpdata));
			zend_hash_move_forward(pht);
		}
	}

	rast_error = rast_db_search(data->db, query, options, &result, data->pool);
	if (rast_error != RAST_OK){
		zend_error(E_WARNING, rast_error->message);
		rast_error_destroy(rast_error);
		RETURN_FALSE
	}
	result_rsrc = (php_rast_result *) ecalloc(1,sizeof(php_rast_result));
	result_rsrc->result = result;
	result_rsrc->num_properties = options->num_properties;
	result_rsrc->properties = options->properties;

	ZEND_REGISTER_RESOURCE(return_value,result_rsrc,le_rast_result);
}
/* }}} */

/* {{{ proto array rast_get_properties(resource db) 
*/
PHP_FUNCTION(rast_get_properties)
{
	zval *zres, *zproperty;
	const rast_property_t *property;
	const rast_property_t *properties;
	int num_properties, i;
	php_rast_data *data;

	if (FAILURE == zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "r",&zres)) {
		return;
	}
	ZEND_FETCH_RESOURCE(data, php_rast_data *, &zres, -1, "Rast DB Handle", le_rast_db);

	array_init(return_value);
	properties = rast_db_properties(data->db, &num_properties);

	for (i = 0; i < num_properties; i++) {
		MAKE_STD_ZVAL(zproperty);
		array_init(zproperty);
		property = properties + i;

		add_assoc_string(zproperty,"name",property->name,1);
		add_assoc_long(zproperty,"type",property->type);
		add_assoc_long(zproperty,"flags",property->flags);

		add_index_zval(return_value, i, zproperty);
	}
}
/* }}} */

/* {{{ proto string rast_get_encoding(resource db)
*/
PHP_FUNCTION(rast_get_encoding)
{
	zval *zres;
	php_rast_data *data;
	const char *encoding;

	if (FAILURE == zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "r",&zres)) {
		return;
	}
	ZEND_FETCH_RESOURCE(data, php_rast_data *, &zres, -1, "Rast DB Handle", le_rast_db);
	encoding = rast_db_encoding(data->db);
	RETURN_STRING((char *) encoding , 1);
}
/* }}} */

/* {{{ proto array rast_result_get_terms(resource result) 
*/
PHP_FUNCTION(rast_result_get_terms)
{
	zval *zres, *zterm;
	int i, num_terms;
	php_rast_result *result;
	rast_result_term_t *terms;

	if (FAILURE == zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "r", &zres)) {
		return;
	}
	ZEND_FETCH_RESOURCE(result, php_rast_result *, &zres, -1, "Rast query result", le_rast_result);

	array_init(return_value);
	terms = result->result->terms;
	num_terms = result->result->num_terms;

	for (i = 0; i < num_terms; i++) {
		const rast_result_term_t *term = terms + i;
		MAKE_STD_ZVAL(zterm);
		array_init(zterm);
		add_assoc_string(zterm, "term", term->term, 1);
		add_assoc_long(zterm, "doc_count", term->doc_count);

		add_index_zval(return_value, i, zterm);
	}
}
/* }}} */

/* {{{ proto int rast_result_get_hit_count(resource result)
*/
PHP_FUNCTION(rast_result_get_hit_count)
{
	zval *zres;
	php_rast_result *result;
	if (FAILURE == zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "r", &zres)) {
		return;
	}
	ZEND_FETCH_RESOURCE(result, php_rast_result *, &zres, -1, "Rast query result", le_rast_result);
	RETURN_LONG(result->result->hit_count);
}
/* }}} */

/* {{{ proto array rast_result_get_items(resource result)
*/
PHP_FUNCTION(rast_result_get_items)
{
	zval *zres, *zary, *zproperties;
	int i, j;
	php_rast_result *result;
	rast_result_item_t *item;

	if (FAILURE == zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "r", &zres)) {
		return;
	}
	ZEND_FETCH_RESOURCE(result, php_rast_result *, &zres, -1, "Rast query result", le_rast_result);

	array_init(return_value);
	for (i = 0; i < result->result->num_items; i++) {
		MAKE_STD_ZVAL(zary);
		array_init(zary);
		item = result->result->items[i];
		add_assoc_long(zary, "doc_id", item->doc_id);
		add_assoc_long(zary, "score", item->score);
		add_assoc_stringl(zary, "summary", item->summary, item->summary_nbytes, 1);

		MAKE_STD_ZVAL(zproperties);
		array_init(zproperties);
		for (j = 0; j < result->num_properties; j++) {
			rast_value_t *value = item->properties + j;
			switch (value->type){
				case RAST_TYPE_STRING:
				case RAST_TYPE_DATE:
					add_assoc_string(zproperties, (char *) result->properties[j], rast_value_string(value), 1);
					break;
				case RAST_TYPE_UINT:
					add_assoc_long(zproperties, (char *) result->properties[j], rast_value_uint(value));
					break;
			}
		}
		add_assoc_zval(zary, "properties", zproperties);
		add_index_zval(return_value, i, zary);
	}
}
/* }}} */

/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * End:
 * vim600: noet sw=4 ts=4 fdm=marker cin
 * vim<600: noet sw=4 ts=4
 */
