package Win32::Dokan::Mounter;

use strict;
use warnings;

use Carp;

use base qw(Win32::Dokan);
use Win32::Dokan qw(:all);

use Win32::Dokan::DokanFileInfo;
use Win32::Dokan::DokanOptions;
use Win32::Dokan::FileInfo;
use Win32::Dokan::FindDataW;

use Fcntl qw(:DEFAULT);
use Errno;

our $VERSION = '0.01_1';

# if context is not used, mark opened file.
our $Dummy_file_context;

sub cb_create_file {
    my $self = shift;
    my ($fileName, $desiredAccess, $shareMode, $createDisposition, $flagsAndAttributes, $DFileInfo) = @_;

    my $ret;
    my $mode = 0;

    if ($shareMode & FILE_SHARE_DELETE) {
	# is this needed?
	$DFileInfo->delete_on_close(1);
    }

    if ($DFileInfo->is_directory) {
	return 0;
    }

    # "/*" style access ?
    $fileName = $1 if ($fileName =~ /(.*)\/\*?$/);

    my $is_dir = $self->{_fs}->opendir($fileName);
    if (defined($is_dir) && $is_dir >= 0) {
	$DFileInfo->is_directory(1);

	if ($createDisposition == CREATE_ALWAYS
	    || $createDisposition == OPEN_ALWAYS) {
	    return ERROR_ALREADY_EXISTS;
	}
	return 0;
    }

    if ($shareMode & FILE_SHARE_READ && $shareMode & FILE_SHARE_WRITE) {
	$mode = O_RDWR;
    }
    elsif ($shareMode & FILE_SHARE_READ) {
	$mode = O_RDONLY;
    }
    elsif ($shareMode & FILE_SHARE_WRITE) {
	$mode = O_WRONLY;
    }
    else {
	$mode = O_RDONLY;
    }

    if ($createDisposition == CREATE_NEW) {
	$mode |= (O_CREAT | O_EXCL);

	my $stat = $self->{_fs}->stat($fileName, $DFileInfo);
	if (ref($stat) && ref($stat) eq 'ARRAY') {
	    $ret = -1*ERROR_ALREADY_EXISTS;
	}
	else {
	    $ret = $self->{_fs}->create($fileName, $mode, $DFileInfo);
	}

	$ret = 0 if (defined($ret) && $ret >= 0);
    }
    elsif ($createDisposition == CREATE_ALWAYS
	   || $createDisposition == OPEN_ALWAYS) {

	my $stat = $self->{_fs}->stat($fileName, $DFileInfo);
	if (ref($stat) && ref($stat) eq 'ARRAY') {
	    if ($createDisposition == CREATE_ALWAYS) {
		$ret = $self->{_fs}->create($fileName, $mode, $DFileInfo);
	    }
	    else {
		$ret = $self->{_fs}->open($fileName, $mode, $DFileInfo);
	    }
	    $ret = 0 if (defined($ret) && $ret >= 0);
	}
	else {
	    $mode |= O_WRONLY unless ($mode & O_RDWR || $mode & O_WRONLY);
	    $mode |= O_CREAT;
	    $ret = $self->{_fs}->create($fileName, $mode, $DFileInfo);
	    $ret = 0 if (defined($ret) && $ret >= 0);
	}
	$ret = ERROR_ALREADY_EXISTS if (defined($ret) && $ret >= 0);
    }
    elsif ($createDisposition == OPEN_EXISTING) {
	$ret = $self->{_fs}->open($fileName, $mode, $DFileInfo);
	$ret = 0 if (defined($ret) && $ret >= 0);
    }
    elsif ($createDisposition == TRUNCATE_EXISTING) {
	my $stat = $self->{_fs}->stat("$fileName", $DFileInfo);
	if (ref($stat) && ref($stat) eq 'ARRAY') {
	    $mode |= O_WRONLY unless ($mode & O_RDWR || $mode & O_WRONLY);
	    $mode |= O_TRUNC;
	    $ret = $self->{_fs}->create($fileName, $mode, $DFileInfo);
	    $ret = 0 if (defined($ret) && $ret >= 0);
	}
	else {
	    $ret = -2;
	}
    }

    $ret = -2 unless (defined($ret));

    if ($ret >= 0 && !defined($DFileInfo->context)) {
	$DFileInfo->context(\$Dummy_file_context);
    }

    return $ret;
}

sub cb_open_directory {
    my $self = shift;
    my ($pathName, $DFileInfo) = @_;

    my $ret = $self->{_fs}->opendir($pathName, $DFileInfo);
    return -1 unless (defined($ret));

    return $ret >= 0 ? 0 : $ret;
}

sub cb_create_directory {
    my $self = shift;
    my ($pathName, $DFileInfo) = @_;
 
    my $ret = $self->{_fs}->mkdir($pathName, $DFileInfo);
    return -1 unless (defined($ret));

    return $ret >= 0 ? 0 : $ret;
}

sub cb_cleanup {
    my $self = shift;
    my ($fileName, $DFileInfo) = @_;

    my $ret = $self->{_fs}->close($fileName, $DFileInfo);
    return -1 unless (defined($ret));
    return $ret if ($ret < 0);

    if ($DFileInfo->delete_on_close) {
	if ($DFileInfo->is_directory) {
	    $ret = $self->{_fs}->rmdir($fileName, $DFileInfo);
	}
	else {
	    $ret = $self->{_fs}->remove($fileName, $DFileInfo);
	}
    }

    return -1 unless (defined($ret));

    return $ret >= 0 ? 0 : $ret;
}

sub cb_close_file {
    my $self = shift;
    my ($fileName, $DFileInfo) = @_;

    return 0;
}

sub cb_read_file {
    my $self = shift;
    my ($fileName, $dummy, $offset, $length, $DFileInfo) = @_;

    my $local_open;
    unless (defined($DFileInfo->context)) {
	my $ret = $self->{_fs}->open($fileName, O_RDONLY, $DFileInfo);
	return -1 unless (defined($ret));
	return $ret if ($ret < 0);

	$local_open = 1;
    }
    my ($data, $err);
    eval {
	($data, $err) = $self->{_fs}->read($fileName, $offset, $length, $DFileInfo);
    };
    if ($local_open) {
	eval {
	    $self->{_fs}->close($fileName, $DFileInfo);
	};
	$DFileInfo->context(undef);
    }

    unless (defined($data)) {
	return $err if (defined($err) && $err < 0);
	return -1;
    }

    $_[1] = $data;

    return 0;
}

sub cb_write_file {
    my $self = shift;
    my ($fileName, $data, $offset, $dummy, $DFileInfo) = @_;
 
    my $local_open;
    unless (defined($DFileInfo->context)) {
	my $ret = $self->{_fs}->open($fileName, O_WRONLY, $DFileInfo);
	return -1 unless (defined($ret));
	return $ret if ($ret < 0);

	$local_open = 1;
    }
    my ($len, $err);
    eval {
	($len, $err) = $self->{_fs}->write($fileName, $offset, $data, $DFileInfo);
	$_[3] = defined($len) ? $len : 0;
    };
    if ($local_open) {
	eval {
	    $self->{_fs}->close($fileName, $DFileInfo);
	};
	$DFileInfo->context(undef);
    }

    return $err if (defined($err) && $err < 0);
    return defined($len) ? 0 : -1;
}

sub cb_flush_file_buffers {
    my $self = shift;
    my ($fileName, $DFileInfo) = @_;

    return $self->{_fs}->flush($fileName, $DFileInfo);
}

sub cb_get_file_information {
    my $self = shift;
    my ($fileName, $fileInfo, $DFileInfo) = @_;

    # "/*" style access ?
    $fileName = $1 if ($fileName =~ /(.*)\/\*?$/);
    my $stat = $self->{_fs}->stat($fileName, $DFileInfo);
    if (ref($stat) && ref($stat) eq 'ARRAY') {
	$fileInfo->file_size(int($stat->[0] || 0));
	$fileInfo->file_attributes(int($stat->[1]));
	$fileInfo->creation_time($stat->[2]);
	$fileInfo->last_access_time($stat->[3]);
	$fileInfo->last_write_time($stat->[4]);

	return 0;
    }

    return (defined($stat) && $stat < 0) ? $stat : -1;
}

sub cb_find_files {
    my $self = shift;
    my ($pathName, $DFileInfo) = @_;

    $pathName = $1 if ($pathName =~ /(.*)\/\*?$/);

    my $elements = $self->{_fs}->readdir($pathName, $DFileInfo);
    unless (ref($elements) && ref($elements) eq 'ARRAY') {
	return (defined($$elements) && $elements < 0) ? $elements : -1;
    }

    my @ret;
    for my $e (@{$elements}) {
	my $find_data = Win32::Dokan::FindDataW->new();

	my $stat = $self->{_fs}->stat("$pathName/$e", $DFileInfo);
	if (ref($stat) && ref($stat) eq 'ARRAY') {
	    $find_data->file_name($e);
	    $find_data->file_size(int($stat->[0] || 0));
	    $find_data->file_attributes($stat->[1]);
	    $find_data->creation_time($stat->[2]);
	    $find_data->last_access_time($stat->[3]);
	    $find_data->last_write_time($stat->[4]);
	    $find_data->number_of_links(1);

	    push(@ret, $find_data);
	}
    }

    return (@ret, 0);
}

sub cb_delete_file {
    my $self = shift;
    my ($fileName, $DFileInfo) = @_;

    my $stat = $self->{_fs}->stat($fileName, $DFileInfo);
    if (ref($stat) && ref($stat) eq 'ARRAY') {
	return 0;
    }
    else {
	return -2 unless (defined($stat));
    }

    return $stat >= 0 ? 0 : $stat;
}

sub cb_delete_directory {
    my $self = shift;
    my ($pathName, $DFileInfo) = @_;

    my $stat = $self->{_fs}->stat("$pathName", $DFileInfo);
    if (ref($stat) && ref($stat) eq 'ARRAY') {
	if ($stat->[1] & Win32::Dokan::FILE_ATTRIBUTE_DIRECTORY()) {
	    if (grep { $_ ne '.' && $_ ne '..' } @{$self->{_fs}->readdir($pathName)}) {
		return -ERROR_DIR_NOT_EMPTY();
	    }
	    return 0;
	}
	return 0;
    }
    else {
	return -2 unless (defined($stat));
    }

    return $stat >= 0 ? 0 : $stat;
}

sub cb_set_file_attributes {
    my $self = shift;
    my ($pathName, $attributes, $DFileInfo) = @_;

    my $ret = $self->{_fs}->setattr($pathName, $attributes, $DFileInfo);
    return -1 unless (defined($ret));

    return $ret >= 0 ? 0 : $ret;
}

sub cb_set_file_time {
    my $self = shift;
    my ($pathName, $ctime, $atime, $mtime, $DFileInfo) = @_;

    my $ret = $self->{_fs}->utime($pathName, $ctime, $atime, $mtime, $DFileInfo);
    return -1 unless (defined($ret));

    return $ret >= 0 ? 0 : $ret;
}

sub cb_move_file {
    my $self = shift;
    my ($existingName, $newFileName, $repaceExisting, $DFileInfo) = @_;

    my $ret = $self->{_fs}->rename($existingName, $newFileName, $DFileInfo);
    return -1 unless (defined($ret));

    return $ret >= 0 ? 0 : $ret;
}

sub cb_set_end_of_file {
    my $self = shift;
    my ($fileName, $length, $DFileInfo) = @_;

    my $ret = $self->{_fs}->truncate($fileName, $length, $DFileInfo);
    return -1 unless (defined($ret));

    return $ret >= 0 ? 0 : $ret;
}

sub cb_lock_file {
    my $self = shift;
    my ($fileName, $offset, $length, $DFileInfo) = @_;

    my $ret = $self->{_fs}->lock($fileName, $offset, $length, $DFileInfo);
    return -1 unless (defined($ret));

    return $ret >= 0 ? 0 : $ret;
}

sub cb_unlock_file {
    my $self = shift;
    my ($fileName, $offset, $length, $DFileInfo) = @_;

    my $ret = $self->{_fs}->lock($fileName, $offset, $length, $DFileInfo);
    return -1 unless (defined($ret));

    return $ret >= 0 ? 0 : $ret;
}

sub cb_get_disk_free_space {
    my $self = shift;
    my ($DFileInfo) = @_;

    # avail (for current user), total (used bytes), free (of disk)
    return (123.0*1E15, 456.0*1E15, 123.0*1E15);
}

sub cb_get_volume_information {
    my $self = shift;
    my ($DFileInfo) = @_;
    my ($volume_name, $serial, $component_length, $file_system_flags, $file_system_name);

    $volume_name = "test";
    $serial = 0x123;
    $component_length = 255;
    $file_system_flags = 0;

    return ($volume_name, $serial, $component_length, $file_system_flags, $file_system_name);
}

sub cb_unmount {
    my $self = shift;
    my ($DFileInfo) = @_;

    return 0;
}

sub mount {
    my $self = shift;
    my ($drive, $fs) = @_;

    $self->fs_encoding($fs->encoding);
    $self->{_fs} = $fs;

    $self->main_loop($drive);
} 

sub new {
    my $class = shift;
    my $opt = shift;

    $class->SUPER::new($opt);
}


1;
__END__
# Below is stub documentation for your module. You'd better edit it!

=head1 NAME

Win32::Dokan::Mounter - Inteface for Dokan library (user mode file system for windows)

=head1 SYNOPSIS

  ###############################################
  # in your filesystem
  #
  package Your::File::System;
  use Win32::Dokan::FS;

  use base qw(Win32::Dokan::FS);
  # override methods below...


  ###############################################
  # in your main script
  #
  use Win32::Dokan::Mounter;
  use Your::File::System;

  my $fs = Your::File::System->new();
  my $mounter = Win32::Dokan::Mounter->new({debug_mode => 1,
					   use_std_err => 1});
  $mounter->mount('W', $fs);


=head1 DESCRIPTION

Win32::Dokan itself is a very low level inteface for Dokan.dll.
Don't use this module directly.

See Win32::Dokan::FS to implement filesystem.

=head2 EXPORT

None by default.

=head2 Exportable constants

=head1 SEE ALSO

Win32::Dokan::FS

Dokan related documents.

=head1 AUTHOR

Toshimitsu FUJIWARA, C<< <tttfjw at gmail.com> >>

=head1 BUGS

Threading is not supported.


=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Win32::Dokan::FS

=head1 ACKNOWLEDGEMENTS


=head1 COPYRIGHT & LICENSE

Copyright 2009 Toshimitsu FUJIWARA, all rights reserved.

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.
