package Win32::Dokan;

use strict;
use warnings;

use threads;

use Carp;

use Win32::Dokan::DokanFileInfo;
use Win32::Dokan::DokanOptions;
use Win32::Dokan::FileInfo;
use Win32::Dokan::FindDataW;

use Encode;
use Fcntl qw(:DEFAULT);
use Errno;

require Exporter;
# use AutoLoader;

our @ISA = qw(Exporter);

# Items to export into callers namespace by default. Note: do not export
# names by default without a very good reason. Use EXPORT_OK instead.
# Do not simply export all your public functions/methods/constants.

# This allows declaration	use Win32::Dokan ':all';
# If you do not need this, moving things directly into @EXPORT or @EXPORT_OK
# will save memory.
our %EXPORT_TAGS = ( 'all' => [ qw(
	DOKAN_DRIVER_INSTALL_ERROR
	DOKAN_DRIVE_LETTER_ERROR
	DOKAN_ERROR
	DOKAN_MOUNT_ERROR
	DOKAN_START_ERROR
	DOKAN_SUCCESS
	FILE_ATTRIBUTE_READONLY
	FILE_ATTRIBUTE_HIDDEN
	FILE_ATTRIBUTE_SYSTEM
	FILE_ATTRIBUTE_DIRECTORY
	FILE_ATTRIBUTE_ARCHIVE
	FILE_ATTRIBUTE_DEVICE
	FILE_ATTRIBUTE_NORMAL
	FILE_ATTRIBUTE_TEMPORARY
	FILE_ATTRIBUTE_SPARSE_FILE
	FILE_ATTRIBUTE_REPARSE_POINT
	FILE_ATTRIBUTE_COMPRESSED
	FILE_ATTRIBUTE_OFFLINE
	FILE_ATTRIBUTE_NOT_CONTENT_INDEXED
	FS_CASE_IS_PRESERVED
	FS_CASE_SENSITIVE
	FS_UNICODE_STORED_ON_DISK
	FS_PERSISTENT_ACLS
	FS_FILE_COMPRESSION
	FS_VOL_IS_COMPRESSED
	FILE_NAMED_STREAMS
	FILE_SUPPORTS_ENCRYPTION
	FILE_SUPPORTS_OBJECT_IDS
	FILE_SUPPORTS_REPARSE_POINTS
	FILE_SUPPORTS_SPARSE_FILES
	FILE_VOLUME_QUOTAS

	GENERIC_READ
	GENERIC_WRITE
	GENERIC_EXECUTE

	CREATE_NEW
	CREATE_ALWAYS
	OPEN_EXISTING
	OPEN_ALWAYS
	TRUNCATE_EXISTING

	FILE_SHARE_READ
	FILE_SHARE_WRITE
	FILE_SHARE_DELETE

	ERROR_DIR_NOT_EMPTY
	ERROR_ALREADY_EXISTS
) ] );


our @EXPORT_OK = qw(
	DOKAN_DRIVER_INSTALL_ERROR
	DOKAN_DRIVE_LETTER_ERROR
	DOKAN_ERROR
	DOKAN_MOUNT_ERROR
	DOKAN_START_ERROR
	DOKAN_SUCCESS
	FILE_ATTRIBUTE_READONLY
	FILE_ATTRIBUTE_HIDDEN
	FILE_ATTRIBUTE_SYSTEM
	FILE_ATTRIBUTE_DIRECTORY
	FILE_ATTRIBUTE_ARCHIVE
	FILE_ATTRIBUTE_DEVICE
	FILE_ATTRIBUTE_NORMAL
	FILE_ATTRIBUTE_TEMPORARY
	FILE_ATTRIBUTE_SPARSE_FILE
	FILE_ATTRIBUTE_REPARSE_POINT
	FILE_ATTRIBUTE_COMPRESSED
	FILE_ATTRIBUTE_OFFLINE
	FILE_ATTRIBUTE_NOT_CONTENT_INDEXED
	FS_CASE_IS_PRESERVED
	FS_CASE_SENSITIVE
	FS_UNICODE_STORED_ON_DISK
	FS_PERSISTENT_ACLS
	FS_FILE_COMPRESSION
	FS_VOL_IS_COMPRESSED
	FILE_NAMED_STREAMS
	FILE_SUPPORTS_ENCRYPTION
	FILE_SUPPORTS_OBJECT_IDS
	FILE_SUPPORTS_REPARSE_POINTS
	FILE_SUPPORTS_SPARSE_FILES
	FILE_VOLUME_QUOTAS

	GENERIC_READ
	GENERIC_WRITE
	GENERIC_EXECUTE

	CREATE_NEW
	CREATE_ALWAYS
	OPEN_EXISTING
	OPEN_ALWAYS
	TRUNCATE_EXISTING

	FILE_SHARE_READ
	FILE_SHARE_WRITE
	FILE_SHARE_DELETE

	ERROR_DIR_NOT_EMPTY
	ERROR_ALREADY_EXISTS
);

our $VERSION = '0.02_1';

sub AUTOLOAD {
    # This AUTOLOAD is used to 'autoload' constants from the constant()
    # XS function.

    my $constname;
    our $AUTOLOAD;
    ($constname = $AUTOLOAD) =~ s/.*:://;
    croak "&Win32::Dokan::constant not defined" if $constname eq 'constant';
    my ($error, $val) = constant($constname);
    if ($error) { croak $error; }
    {
	no strict 'refs';
	# Fixed between 5.005_53 and 5.005_61
#XXX	if ($] >= 5.00561) {
#XXX	    *$AUTOLOAD = sub () { $val };
#XXX	}
#XXX	else {
	    *$AUTOLOAD = sub { $val };
#XXX	}
    }
    goto &$AUTOLOAD;
}

require XSLoader;
XSLoader::load('Win32::Dokan', $VERSION);

# CreationDisposition
use constant {
    CREATE_NEW => 1,
    CREATE_ALWAYS => 2,
    OPEN_EXISTING => 3,
    OPEN_ALWAYS => 4,
    TRUNCATE_EXISTING => 5};

# DesiredAccess
use constant {
    GENERIC_READ => 0x80000000,
    GENERIC_WRITE => 0x40000000,
    GENERIC_EXECUTE => 0x20000000};

# ShareMode
use constant {
    FILE_SHARE_READ => 0x00000001,
    FILE_SHARE_WRITE => 0x00000002,
    FILE_SHARE_DELETE => 0x00000004};

# open error
use constant {
    ERROR_DIR_NOT_EMPTY => 145,
    ERROR_ALREADY_EXISTS => 183,
};

sub _fsname_to_unicode {
    my $self = shift;
    my $name = shift;

    if ($self->{_use_mbcs}) {
	return $self->convert_native_to_unicode($name);
    }

    my $ret = $self->{_fs_enc} ? $self->{_fs_enc}->decode($name) : $name;

    return $self->{_uni_enc}->encode($ret);
}

sub _unicode_to_fsname {
    my $self = shift;
    my $name = shift;

    if ($self->{_use_mbcs}) {
	my $ret;
	if ($self->{_convert_path}) {
	    $ret = $self->{_uni_enc}->decode($name);
	    $ret =~ s/\\/\//g if ($self->{_convert_path});
	    $ret = $self->{_uni_enc}->encode($ret);
	}
	else {
	    $ret = $name;
	}
	return $self->convert_unicode_to_native($ret);
    }

    my $ret = $self->{_uni_enc}->decode($name);
    $ret =~ s/\\/\//g if ($self->{_convert_path});
    
    if ($self->{_fs_enc}) {
	return $self->{_fs_enc}->encode($ret);
    }

    return $ret;
}

sub _enflat_context {
    my $self = shift;
    my $dfi = shift;
    my $old_context_value = shift;

    my $context = $dfi->context;
    my $new_context_value;

    if (defined($context)) {
	if (ref($context)) {
	    $new_context_value = int(0+$context);
	    $self->{_files}->{$new_context_value} = [$context];
	}
	else {
	    my $new_ref = [$context];
	    $new_context_value = int(0+$new_ref);
	    $self->{_files}->{$new_context_value} = $new_ref;
	}
	$dfi->context($new_context_value);
    }

    if (defined($old_context_value)) {
	if (defined($new_context_value)) {
	    if ($new_context_value != $old_context_value) {
		delete $self->{_files}->{$old_context_value};
	    }
	}
	else {
	    delete $self->{_files}->{$old_context_value};
	}
    }
}

sub _populate_context {
    my $self = shift;
    my $dfi = shift;

    my $context = $self->{_files}->{$dfi->context};

    if (ref($context) eq 'ARRAY') {
	$dfi->context($context->[0])
    }
    else {
	$dfi->context(undef);
    }
}

sub _call_cb_simple {
    my $self = shift;
    my $method = shift;
    my $fileName = shift;
    # rest is in @_

    my $dfi;
    my $org_context_value;

    if (ref($_[$#_]) eq 'Win32::Dokan::DokanFileInfo'
	&& defined($_[$#_]->context)) {

	$dfi = $_[$#_];
	$org_context_value = $dfi->context;
	$self->_populate_context($dfi);
    }

    $fileName = $self->_unicode_to_fsname($fileName);

    if (wantarray()) {
	my @ret;
	eval {
	    @ret = $self->$method($fileName, @_);
	    if ($dfi) {
		$self->_enflat_context($dfi, $org_context_value);
	    };
	};
	my $err = $@;
	carp "************** exception: $@" if ($@);
	return $err ? -1 : @ret;
    }
    else {
	my $ret;
	eval {
	    $ret = $self->$method($fileName, @_);
	    if ($dfi) {
		$self->_enflat_context($dfi, $org_context_value);
	    };
	};
	my $err = $@;
	carp "************** exception: $@" if ($@);
	return $err ? -1 : $ret;
    }
}

sub _cb_create_file {
    shift->_call_cb_simple("cb_create_file", @_);
}

sub cb_create_file {
    my $self = shift;
    my ($fileName, $desiredAccess, $shareMode, $createDisposition, $flagsAndAttributes, $DFileInfo) = @_;

    # carp "cb_create_file";
    return -2;
}

sub _cb_open_directory {
    shift->_call_cb_simple("cb_open_directory", @_);
}

sub cb_open_directory {
    my $self = shift;
    my ($pathName, $DFileInfo) = @_;

    carp "cb_open_directory";
    return -2;
}

sub _cb_create_directory {
    shift->_call_cb_simple("cb_create_directory", @_);
}

sub cb_create_directory {
    my $self = shift;
    my ($pathName, $DFileInfo) = @_;
 
    carp "cb_create_directory";
    return -2;
}

sub _cb_cleanup {
    shift->_call_cb_simple("_cb_cleanup2", @_);
}

sub _cb_cleanup2 {
    my $self = shift;
    my ($fileName, $DFileInfo) = @_;
    my $ret = $self->cb_cleanup($fileName, $DFileInfo);

    # free stored context
    $DFileInfo->context(undef);

    return $ret;
}

sub cb_cleanup {
    my $self = shift;
    my ($fileName, $DFileInfo) = @_;

    carp "cb_cleanup";
    return 0;
}

sub _cb_close_file {
    shift->_call_cb_simple("cb_close_file", @_);
}

sub cb_close_file {
    my $self = shift;
    my ($fileName, $DFileInfo) = @_;

    carp "cb_close_file";
    return 0;
}

sub _cb_read_file {
    shift->_call_cb_simple("cb_read_file", @_);
}

sub cb_read_file {
    my $self = shift;
    my ($fileName, $dummy, $offset, $length, $DFileInfo) = @_;

    # read data in $_[1] !!!

    carp "cb_read_file";
    return;
}

sub _cb_write_file {
    shift->_call_cb_simple("cb_write_file", @_);
}

sub cb_write_file {
    my $self = shift;
    my ($fileName, $data, $offset, $dummy, $DFileInfo) = @_;
 
    # store len in $_[3] !!!

    carp "cb_write_file";
    return 0;
}

sub _cb_flush_file_buffers {
    shift->_call_cb_simple("cb_flush_file_buffers", @_);
}

sub cb_flush_file_buffers {
    my $self = shift;
    my ($fileName, $DFileInfo) = @_;

    carp "cb_flush_file_buffers";
    return 0;
}

sub _cb_get_file_information {
    shift->_call_cb_simple("cb_get_file_information", @_);
}

sub cb_get_file_information {
    my $self = shift;
    my ($fileName, $fileInfo, $DFileInfo) = @_;

    # store info into $fileInfo
    carp "cb_get_file_information";
    return -1;
}

sub _cb_find_files {
    my $self = shift;
    $self->_call_cb_simple("cb_find_files", @_);
}

sub cb_find_files {
    my $self = shift;
    my ($pathName, $DFileInfo) = @_;

    # return array of Win32::Dokan::FindDataW, and errno
    carp "cb_find_files";
    return (0);
}

sub _cb_delete_file {
    shift->_call_cb_simple("cb_delete_file", @_);
}

sub cb_delete_file {
    my $self = shift;
    my ($fileName, $DFileInfo) = @_;

    carp "cb_delete_file";
    return -2;
}

sub _cb_delete_directory {
    shift->_call_cb_simple("cb_delete_directory", @_);
}

sub cb_delete_directory {
    my $self = shift;
    my ($pathName, $DFileInfo) = @_;

    carp "cb_delete_directory";
    return -2;
}

sub _cb_set_file_attributes {
    shift->_call_cb_simple("cb_set_file_attributes", @_);
}

sub cb_set_file_attributes {
    my $self = shift;
    my ($pathName, $attributes, $DFileInfo) = @_;

    carp "cb_set_file_attributes";
    return -2;
}

sub _cb_set_file_time {
    shift->_call_cb_simple("cb_set_file_time", @_);
}

sub cb_set_file_time {
    my $self = shift;
    my ($pathName, $ctime, $atime, $mtime, $DFileInfo) = @_;

    carp "cb_set_file_time";
    return -2;
}

sub _cb_move_file {
    my $self = shift;
    my $name1 = shift;
    my $name2 = shift;
    # rest is in @_

    $name1 = $self->_unicode_to_fsname($name1);
    $name2 = $self->_unicode_to_fsname($name2);

    my $ret;
    eval {
	$ret = $self->cb_move_file($name1, $name2, @_);
    };
    my $err = $@;
    carp "************** exception: $@" if ($@);
    return $err ? -1 : $ret;
}

sub cb_move_file {
    my $self = shift;
    my ($existingName, $newFileName, $repaceExisting, $DFileInfo) = @_;

    carp "cb_move_file";
    return -2;
}

sub _cb_set_end_of_file {
    shift->_call_cb_simple("cb_set_end_of_file", @_);
}

sub cb_set_end_of_file {
    my $self = shift;
    my ($fileName, $length, $DFileInfo) = @_;

    carp "cb_set_end_of_file";
    return -2;
}

sub _cb_lock_file {
    shift->_call_cb_simple("cb_lock_file", @_);
}

sub cb_lock_file {
    my $self = shift;
    my ($fileName, $offset, $length, $DFileInfo) = @_;

    carp "cb_lock_file";
    return -2;
}

sub _cb_unlock_file {
    shift->_call_cb_simple("cb_unlock_file", @_);
}

sub cb_unlock_file {
    my $self = shift;
    my ($fileName, $offset, $length, $DFileInfo) = @_;

    carp "cb_unlock_file";
    return -2;
}

sub _call_cb_noname {
    my $self = shift;
    my $method = shift;

    # rest is in @_
    if (wantarray()) {
	my @ret;
	eval {
	    @ret = $self->$method(@_);
	};
	my $err = $@;
	carp "************** exception: $@" if ($@);
	return $err ? -1 : @ret;
    }
    else {
	my $ret;
	eval {
	    $ret = $self->$method(@_);
	};
	my $err = $@;
	carp "************** exception: $@" if ($@);
	return $err ? -1 : $ret;
    }
}

sub _cb_get_disk_free_space {
    my $self = shift;
    my ($avail, $total, $free)
	= $self->_call_cb_noname("cb_get_disk_free_space", @_);

    return (0.0+$avail, 0.0+$total, 0.0+$free);
}

sub cb_get_disk_free_space {
    my $self = shift;
    my ($DFileInfo) = @_;

    carp "cb_get_disk_free_space";

    # avail (for current user), total (used bytes), free (of disk)
    return (0, 0, 0);
}

sub _cb_get_volume_information {
    my $self = shift;
  
    my ($volume_name,
	$serial,
	$component_length,
	$file_system_flags,
	$file_system_name);

    ($volume_name,
     $serial,
     $component_length,
     $file_system_flags,
     $file_system_name)
	= $self->_call_cb_noname("cb_get_volume_information", @_);

    if (defined($volume_name)) {
	$volume_name = $self->_fsname_to_unicode($volume_name);
    }
    else {
	$volume_name = "Win32-Dokan";
    }

    if (defined($file_system_name)) {
	$file_system_name = $self->_fsname_to_unicode($file_system_name);
    }
    else {
	$file_system_name = "Win32-Dokan";
    }

    $serial = 0 unless (defined($serial));
    $component_length = 255 unless (defined($component_length));
    $file_system_flags = 0 unless (defined($file_system_flags));


    return ($volume_name, $serial, $component_length, $file_system_flags, $file_system_name);
}

sub cb_get_volume_information {
    my $self = shift;
    my ($DFileInfo) = @_;

    my ($volume_name, $serial, $component_length, $file_system_flags, $file_system_name);

    carp "cb_get_volume_information";
    return ($volume_name, $serial, $component_length, $file_system_flags, $file_system_name);
}

sub _cb_unmount {
    shift->_call_cb_noname("cb_unmount", @_);
}

sub cb_unmount {
    my $self = shift;
    my ($DFileInfo) = @_;

    carp "cb_unmount";

    return 0;
}

our $_DLL = Win32::Dokan::_DLL->new;

sub prepare_main {
    my $self = shift;
    my $opt = shift;

    $self->start_main_thread($_DLL->{handle}, $opt)
	|| croak("cannot start Dokan main thread");

    # refcnt++
    $self->{_dll} = $_DLL;

    # need unmount
    $self->{_running} = 1;
}

sub main_loop {
    my $self = shift;
    my ($drive, $opt) = @_;

    if (defined($opt)) {
	if (ref($opt) eq 'HASH') {
	    $opt = $self->{options} = Win32::Dokan::DokanOptions->new($opt);
	}
        elsif (ref($opt) eq 'Win32::Dokan::DokanOptions') {
	    $opt = $self->{options} = $opt;
	}
	else {
	    croak "unknown type option: " . ref($opt);
	}
    }
    unless ($opt) {
	$opt = $self->{options};
    }

    croak("Bad drive letter: $drive") unless ($drive =~ /^[A-Z]$/i);

    $opt->drive_letter(uc $drive);
    $self->prepare_main($opt);

    local $Win32::Dokan::FindDataW::Unicode_func = sub {
	my $name = shift;
	return $self->_fsname_to_unicode($name);
    };

    local $SIG{INT} = sub {
	$self->unmount($_DLL->{handle}, $self->{options}->drive_letter);
	$self->{_running} = 0;
    };
    local $SIG{TERM} = $SIG{INT};
    local $SIG{HUP} = $SIG{INT};

    $self->main;

    $self->{_running} = 0;
} 

sub fs_encoding {
    my $self = shift;
    return $self->{_fs_encoding} unless (@_);

    my $encoding = shift;
    if (defined($encoding)) {
	if ($encoding =~ /^X-MBCS$/i) {
	    $self->{_use_mbcs} = 1;
	}
	else {
	    my $enc = Encode::find_encoding($encoding)
		or croak "unknown encoding: $encoding";
	    $self->{_fs_enc} = $enc;
	}
    }
    else {
	$self->{_fs_enc} = undef;
    }

    return $encoding;
}

sub new {
    my $class = shift;
    my $opt = shift;

    my $dokan_options;
    my $enc;

    my $self = bless {
    }, $class;

    if (defined($opt) && ref($opt) eq 'HASH') {
	$self->fs_encoding($opt->{fs_encoding}) if ($opt->{fs_encoding});
	$self->{_convert_path} = $opt->{convert_path};

	$dokan_options = Win32::Dokan::DokanOptions->new($opt);
    }
    else {
	$dokan_options = Win32::Dokan::DokanOptions->new();
    }

    $self->{options} = $dokan_options;
    $self->{_files} = {};
    $self->{_uni_enc} = Encode::find_encoding('UTF-16LE');

    return $self;
}

sub DESTROY {
    my $self = shift;
    if ($self->{_running}
	&& defined($self->{options})
	&& defined($self->{options}->drive_letter)) {

	$self->unmount($_DLL->{handle}, $self->{options}->drive_letter);
    }
}


package Win32::Dokan::_DLL;

use Carp;

sub new {
    my $class = shift;

    my @path = split(';', $ENV{PATH});
    unshift(@path, "$ENV{windir}\\System32");

    my $handle;

    for my $p (@path) {
	my $dll_path = "$p\\Dokan.dll";
	if (-f $dll_path) {
	    last if ($handle = Win32::Dokan->load_library($dll_path));
	}
    }
    croak "Cannot load Dokan.dll" unless ($handle);

    # print STDERR "dll loaded\n";

    bless {
	handle => $handle
    }, $class;
}

sub DESTROY {
    my $self = shift;

    # print STDERR "dll unloaded\n";

    Win32::Dokan->free_library($self->{handle}) if ($self->{handle});
}

1;
__END__
# Below is stub documentation for your module. You'd better edit it!

=head1 NAME

Win32::Dokan - Inteface for Dokan library (user mode file system for windows)

=head1 SYNOPSIS

  ###############################################
  # in your filesystem
  #
  package Your::File::System;
  use Win32::Dokan::FS;

  use base qw(Win32::Dokan::FS);
  # override methods below...


  ###############################################
  # in your main script
  #
  use Win32::Dokan::Mounter;
  use Your::File::System;

  my $fs = Your::File::System->new();
  my $mounter = Win32::Dokan::Mounter->new({debug_mode => 1,
					   use_std_err => 1});
  $mounter->mount('W', $fs);


=head1 DESCRIPTION

Win32::Dokan itself is a very low level inteface for Dokan.dll.
Don't use this module directly.

See Win32::Dokan::FS to implement filesystem.

=head2 EXPORT

None by default.

=head2 Exportable constants

  DOKAN_DRIVER_INSTALL_ERROR
  DOKAN_DRIVE_LETTER_ERROR
  DOKAN_ERROR
  DOKAN_MOUNT_ERROR
  DOKAN_START_ERROR
  DOKAN_SUCCESS

  FILE_ATTRIBUTE_READONLY
  FILE_ATTRIBUTE_HIDDEN
  FILE_ATTRIBUTE_SYSTEM
  FILE_ATTRIBUTE_DIRECTORY
  FILE_ATTRIBUTE_ARCHIVE
  FILE_ATTRIBUTE_DEVICE
  FILE_ATTRIBUTE_NORMAL
  FILE_ATTRIBUTE_TEMPORARY
  FILE_ATTRIBUTE_SPARSE_FILE
  FILE_ATTRIBUTE_REPARSE_POINT
  FILE_ATTRIBUTE_COMPRESSED
  FILE_ATTRIBUTE_OFFLINE
  FILE_ATTRIBUTE_NOT_CONTENT_INDEXED

  FS_CASE_IS_PRESERVED
  FS_CASE_SENSITIVE
  FS_UNICODE_STORED_ON_DISK
  FS_PERSISTENT_ACLS
  FS_FILE_COMPRESSION
  FS_VOL_IS_COMPRESSED

  FILE_NAMED_STREAMS
  FILE_SUPPORTS_ENCRYPTION
  FILE_SUPPORTS_OBJECT_IDS
  FILE_SUPPORTS_REPARSE_POINTS
  FILE_SUPPORTS_SPARSE_FILES
  FILE_VOLUME_QUOTAS

=head1 SEE ALSO

Win32::Dokan::FS

Dokan related documents.

=head1 AUTHOR

Toshimitsu FUJIWARA, C<< <tttfjw at gmail.com> >>

=head1 BUGS

Threading is not supported.


=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Win32::Dokan::FS

=head1 ACKNOWLEDGEMENTS


=head1 COPYRIGHT & LICENSE

Copyright 2009 Toshimitsu FUJIWARA, all rights reserved.

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.
