# testresponse.rb: testcases for additional methods on WEBrick::HTTPResponse

require 'test/unit'
load 'blazer-ja-proxy'

require 'nkf'
require 'uri'

class TestBlazerModule < Test::Unit::TestCase

	def test_charset
		{
			'iso-8859-1' => NKF::ASCII,
			'EUC-JP' => NKF::EUC,
			'utf-8' => NKF::UTF8,
			'ISO-2022-JP' => NKF::JIS,
			'x-sjis' => NKF::SJIS,
			'shift-jis' => NKF::SJIS,
			nil => nil,
		}.each_pair do |charset_string, constant|
			assert_equal(constant, BlazerProxy::charset_to_nkf(charset_string))
		end
		{
			NKF::ASCII => 'ISO-8859-1',
			NKF::EUC => 'EUC-JP',
			NKF::JIS => 'ISO-2022-JP',
			NKF::SJIS => 'Shift-JIS',
			NKF::UTF8 => 'UTF-8',
			NKF::UTF16 => 'UTF-16',
			NKF::BINARY => nil,
			nil => nil,
		}.each_pair do |constant, str|
			assert_equal(str, BlazerProxy::nkf_to_charset(constant))
		end
	end

	def test_content_type
		{
			'text/html; charset=iso-8859-1' => NKF::ASCII,
			'text/html; charset=EUC-JP' => NKF::EUC,
			'text/html; charset=utf-8' => NKF::UTF8,
			'text/plain; charset=utf-8' => NKF::UTF8,
			'text/html' => nil,
			'text/plain' => nil,
			'text/css' => nil,
			'image/gif' => NKF::BINARY,
			nil => nil,
		}.each_pair do |content_type, charset|
			assert_equal(charset, BlazerProxy::content_type_to_nkf(content_type), "Content-type: " + content_type.inspect)
		end
	end

	def test_registry
		input = {
			'http://A.B.C/' => NKF::ASCII,
			'http://b.c.d/' => NKF::EUC,
			'http://b.c.d/EFG#moge' => NKF::UTF8,
		}
		reg = {
			'http://a.b.c/' => NKF::ASCII,
			'http://b.c.d/' => NKF::EUC,
			'http://b.c.d/EFG' => NKF::UTF8,
		}

		t = BlazerProxy::CharsetRegistry.new
		input.each_pair do |url, const|
			t[URI::parse(url)] = const
		end
		reg.each_pair do |url, const|
			assert_equal(const, t[URI::parse(url)])
		end
	end

	def test_registry_max
		input = [
			['http://A.B.C/', NKF::ASCII],
			['http://b.c.d/', NKF::EUC],
			['http://b.c.d/EFG#moge', NKF::UTF8],
			['http://b.c.d/', NKF::SJIS],
		]
		reg = {
			'http://b.c.d/' => NKF::SJIS,
		}

		t = BlazerProxy::CharsetRegistry.new(1)
		input.each do |e|
			t[URI::parse(e[0])] = e[1]
		end
		reg.each_pair do |url, const|
			assert_equal(const, t[URI::parse(url)])
		end
	end

	def test_convert_form_data
		src = NKF.nkf("-e -Lw", <<'_END')
--------------------------------117511377109921378543360
Content-Disposition: form-data; name="file"; filename="test"
Content-Type: application/octet-stream


--------------------------------117511377109921378543360
Content-Disposition: form-data; name="ja"

ܸʸϤǤ

--------------------------------117511377109921378543360
Content-Disposition: form-data; name="en"

English text

--------------------------------117511377109921378543360--
_END
		dst = NKF.nkf("-s -E", src)
		converted = BlazerProxy::convert_form_data(src, '------------------------------117511377109921378543360', NKF::SJIS)
		assert_equal(dst, converted)
	end

	def test_multibyte
		src = 'testǤȡdesuto'
		%w( -e -j -s -w ).each do |code|
			assert(BlazerProxy::have_multibyte?(NKF::nkf(code, src)), code)
		end
		#assert(!BlazerProxy::have_multibyte?("Gr\xfc\xdf Gott"), "Gruess Got")
		# Some ISO-8859-1 8-bit stream seem like EUC Kanji, unfortunately.
	end

end
