/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.pal.admin.logic;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import jp.sf.pal.admin.PALAdminException;
import jp.sf.pal.admin.pager.RepositoryPager;
import jp.sf.pal.admin.rpad.RPADException;
import jp.sf.pal.admin.rpad.repository.Repository;
import jp.sf.pal.admin.rpad.repository.RepositoryManager;
import jp.sf.pal.admin.rpad.repository.impl.SimpleRepository;

import org.seasar.framework.log.Logger;

public class RepositoryManagementLogic implements Serializable
{

    /**
     * 
     */
    private static final long serialVersionUID = 4720684230441168283L;

    /**
     * Logger for this class
     */
    private static final Logger logger = Logger
            .getLogger(RepositoryManagementLogic.class);

    protected RepositoryManager getRepositoryManager()
    {
        return RepositoryManager.getInstance();
    }

    public List<Repository> getRepositories(RepositoryPager pager)
    {
        List<Repository> repositoryList = new ArrayList<Repository>();
        List<Repository> repositories = getRepositoryManager()
                .getRepositories();

        // set count
        pager.setCount(repositories.size());

        // create list
        int offset = pager.getOffset();
        int limit = pager.getLimit();
        int size = repositories.size();
        if (logger.isDebugEnabled())
        {
            logger
                    .debug("getPortletApplications(PortletApplicationPager) - offset="
                            + offset + ", limit=" + limit + ", size=" + size);
        }
        for (int i = offset; i < offset + limit && i < size; i++)
        {
            repositoryList.add(repositories.get(i));
        }

        return repositoryList;
    }

    public List<Repository> getRepositories()
    {
        RepositoryManager repositoryManager = RepositoryManager.getInstance();
        return repositoryManager.getRepositories();
    }

    //  For Simple Repository
    public void create(String name, String configPath) throws PALAdminException
    {
        Repository r = getRepositoryManager().getRepository(name);
        if (r != null)
        {
            throw new PALAdminException("The repository already exists. name="
                    + name);
        }

        SimpleRepository repository = new SimpleRepository();
        repository.setName(name);
        repository.setConfigPath(configPath);
        try
        {
            getRepositoryManager().addRepository(name, repository);
        }
        catch (RPADException e)
        {
            logger.error("Could not add the repository. repository="
                    + repository, e);
            throw new PALAdminException(
                    "Could not add the repository. repository=" + repository, e);
        }
    }

    //  For Simple Repository
    public void update(String name, String configPath) throws PALAdminException
    {
        Repository repository = getRepositoryManager().getRepository(name);
        if (repository == null)
        {
            throw new PALAdminException("Could not find the repository. name="
                    + name);
        }

        if (repository instanceof SimpleRepository)
        {
            SimpleRepository sr = (SimpleRepository) repository;
            sr.setName(name);
            sr.setConfigPath(configPath);
            try
            {
                getRepositoryManager().store();
            }
            catch (RPADException e)
            {
                logger.error("Could not update the repository. repository="
                        + repository, e);
                throw new PALAdminException(
                        "Could not update the repository. repository="
                                + repository, e);
            }
        }
        else
        {
            throw new PALAdminException("The repository is not supported.");
        }
    }

    //  For Simple Repository
    public void delete(String name) throws PALAdminException
    {
        Repository repository = getRepositoryManager().getRepository(name);
        if (repository == null)
        {
            throw new PALAdminException("Could not find the repository. name="
                    + name);
        }

        try
        {
            getRepositoryManager().removeRepository(repository.getName());
        }
        catch (RPADException e)
        {
            logger.error("Could not delete the repository. repository="
                    + repository, e);
            throw new PALAdminException(
                    "Could not delete the repository. repository=" + repository,
                    e);
        }

    }

    public void reload() throws PALAdminException
    {
        try
        {
            getRepositoryManager().reload();
        }
        catch (RPADException e)
        {
            logger.error("Could not reload repositories.", e);
            throw new PALAdminException("Could not reload repositories.", e);
        }
    }

    public Repository getRepository(String name)
    {
        return getRepositoryManager().getRepository(name);
    }
}
