/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.pal.admin.filter;

import java.io.IOException;

import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;
import javax.portlet.PortletConfig;
import javax.portlet.PortletContext;
import javax.portlet.PortletException;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;

import jp.sf.pal.admin.PALAdminConstants;
import jp.sf.pal.admin.rpad.RPADException;
import jp.sf.pal.admin.rpad.repository.RepositoryManager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.jetspeed.CommonPortletServices;
import org.apache.jetspeed.components.portletregistry.PortletRegistry;
import org.apache.jetspeed.deployment.DeploymentManager;
import org.apache.jetspeed.factory.PortletFactory;
import org.apache.jetspeed.tools.pamanager.servletcontainer.ApplicationServerManager;
import org.apache.portals.bridges.portletfilter.PortletFilter;
import org.apache.portals.bridges.portletfilter.PortletFilterChain;
import org.apache.portals.bridges.portletfilter.PortletFilterConfig;

public class PortletManagementFilter implements PortletFilter
{
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory
            .getLog(PortletManagementFilter.class);

    protected final static String RPAD_CONFIG_FILE = "config-file";

    protected final static String DEFAULT_RPAD_CONFIG_FILE = "/WEB-INF/rpad-config.xml";

    protected final static String WEBAPP_ROOT_PREFIX = "${webapp}";

    private ApplicationServerManager applicationServerManager;

    private PortletRegistry portletRegistry;

    private PortletFactory portletFactory;

    private DeploymentManager deploymentManager;

    public void destroy()
    {
        applicationServerManager = null;
        portletFactory = null;
        portletRegistry = null;
        try
        {
            RepositoryManager.getInstance().store();
        }
        catch (RPADException e)
        {
            log.error("Could not store the configuration.", e);
        }
    }

    public void init(PortletFilterConfig filterConfig) throws PortletException
    {
        PortletConfig portletConfig = filterConfig.getPortletConfig();
        PortletContext portletContext = portletConfig.getPortletContext();

        portletRegistry = (PortletRegistry) portletContext
                .getAttribute(CommonPortletServices.CPS_REGISTRY_COMPONENT);
        portletFactory = (PortletFactory) portletContext
                .getAttribute(CommonPortletServices.CPS_PORTLET_FACTORY_COMPONENT);
        applicationServerManager = (ApplicationServerManager) portletContext
                .getAttribute(CommonPortletServices.CPS_APPLICATION_SERVER_MANAGER_COMPONENT);
        deploymentManager = (DeploymentManager) portletContext
                .getAttribute(CommonPortletServices.CPS_DEPLOYMENT_MANAGER_COMPONENT);

        //RPAD
        initRRPAD(portletConfig);
    }

    protected void initRRPAD(PortletConfig portletConfig)
            throws PortletException
    {
        PortletContext portletContext = portletConfig.getPortletContext();

        String rpadConfigFile = portletConfig
                .getInitParameter(RPAD_CONFIG_FILE);
        if (rpadConfigFile == null)
        {
            rpadConfigFile = portletContext
                    .getRealPath(DEFAULT_RPAD_CONFIG_FILE);
        }
        else if (rpadConfigFile.startsWith(WEBAPP_ROOT_PREFIX))
        {
            rpadConfigFile = portletContext.getRealPath(rpadConfigFile
                    .substring(WEBAPP_ROOT_PREFIX.length()));
        }

        if (log.isDebugEnabled())
        {
            log.debug("init(PortletConfig) - configFile=" + rpadConfigFile);
        }

        // Create RepositoryManager
        try
        {
            RepositoryManager.init(rpadConfigFile);
        }
        catch (Exception e)
        {
            throw new PortletException(
                    "Could not create RepositoryManager. The config file is "
                            + rpadConfigFile, e);
        }
    }

    public void processActionFilter(ActionRequest request,
            ActionResponse response, PortletFilterChain chain)
            throws PortletException, IOException
    {
        request.setAttribute(
                PALAdminConstants.APPLICATION_SERVER_MANAGER_COMPONENT,
                applicationServerManager);
        request.setAttribute(PALAdminConstants.PORTLET_FACTORY_COMPONENT,
                portletFactory);
        request.setAttribute(PALAdminConstants.PORTLET_REGISTRY_COMPONENT,
                portletRegistry);
        request.setAttribute(PALAdminConstants.DEPLOYMENT_MANAGER_COMPONENT,
                deploymentManager);

        chain.processActionFilter(request, response);
    }

    public void renderFilter(RenderRequest request, RenderResponse response,
            PortletFilterChain chain) throws PortletException, IOException
    {
        request.setAttribute(
                PALAdminConstants.APPLICATION_SERVER_MANAGER_COMPONENT,
                applicationServerManager);
        request.setAttribute(PALAdminConstants.PORTLET_FACTORY_COMPONENT,
                portletFactory);
        request.setAttribute(PALAdminConstants.PORTLET_REGISTRY_COMPONENT,
                portletRegistry);
        request.setAttribute(PALAdminConstants.DEPLOYMENT_MANAGER_COMPONENT,
                deploymentManager);

        chain.renderFilter(request, response);
    }

}
