/*
 * Copyright 2000-2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.pal.wcm.portlet;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.StringTokenizer;

import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;
import javax.portlet.GenericPortlet;
import javax.portlet.PortletConfig;
import javax.portlet.PortletException;
import javax.portlet.PortletMode;
import javax.portlet.PortletRequest;
import javax.portlet.PortletRequestDispatcher;
import javax.portlet.PortletSession;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;

import jp.sf.pal.wcm.PALWcmConstants;

import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.jetspeed.CommonPortletServices;
import org.apache.jetspeed.PortalReservedParameters;
import org.apache.jetspeed.om.page.ContentFragment;
import org.apache.jetspeed.om.page.ContentPage;
import org.apache.jetspeed.om.page.Fragment;
import org.apache.jetspeed.om.preference.FragmentPreference;
import org.apache.jetspeed.page.PageManager;
import org.apache.jetspeed.page.PageNotUpdatedException;
import org.apache.jetspeed.page.document.NodeException;
import org.apache.jetspeed.request.RequestContext;

/**
 * HelloWorldPortlet is a portlet to say "Hello!"
 * 
 * @author <a href="mailto:shinsuke@yahoo.co.jp">Shinsuke Sugaya</a>
 */
public class PALWcmPortlet extends GenericPortlet {
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory.getLog(PALWcmPortlet.class);

    private PageManager pageManager;

    /* (non-Javadoc)
     * @see javax.portlet.Portlet#init(javax.portlet.PortletConfig)
     */
    public void init(PortletConfig config) throws PortletException {
        super.init(config);

        // pageManager
        pageManager = (PageManager) getPortletContext().getAttribute(
                CommonPortletServices.CPS_PAGE_MANAGER_COMPONENT);
        if (null == pageManager) {
            throw new PortletException(
                    "Failed to find the Page Manager on portlet initialization");
        }
    }

    /* (non-Javadoc)
     * @see javax.portlet.GenericPortlet#doDispatch(javax.portlet.RenderRequest, javax.portlet.RenderResponse)
     */
    @Override
    protected void doDispatch(RenderRequest request, RenderResponse response)
            throws PortletException, IOException {
        super.doDispatch(request, response);
        PortletMode mode = request.getPortletMode();
        if (mode.equals(PortletMode.VIEW)) {
            String title = (String) request.getAttribute(PALWcmConstants.TITLE);
            if (title != null) {
                response.setTitle(title);
            } else {
                Fragment fragment = getCurrentFragment(request);

                Locale currentLocale = request.getLocale();
                List localeList = getLocaleList(fragment);
                String localeName = "";
                String language = currentLocale.getLanguage();
                String country = currentLocale.getCountry();
                String variant = currentLocale.getVariant();
                if (localeList.contains(language + "_" + country + "_"
                        + variant)) {
                    localeName = PALWcmConstants.LOCALE_SEPARETER + language
                            + "_" + country + "_" + variant;
                } else if (localeList.contains(language + "_" + country)) {
                    localeName = PALWcmConstants.LOCALE_SEPARETER + language
                            + "_" + country;
                } else if (localeList.contains(language)) {
                    localeName = PALWcmConstants.LOCALE_SEPARETER + language;
                }

                response.setTitle(getFragmentPreferenceValue(fragment,
                        PALWcmConstants.TITLE + localeName, ""));
            }
        }
        //TODO doHelp
    }

    /* (non-Javadoc)
     * @see javax.portlet.GenericPortlet#doView(javax.portlet.RenderRequest, javax.portlet.RenderResponse)
     */
    protected void doView(RenderRequest request, RenderResponse response)
            throws PortletException, IOException {
        response.setContentType("text/html");

        Fragment fragment = getCurrentFragment(request);

        Locale currentLocale = request.getLocale();
        List localeList = getLocaleList(fragment);
        String localeName = "";
        String language = currentLocale.getLanguage();
        String country = currentLocale.getCountry();
        String variant = currentLocale.getVariant();
        if (localeList.contains(language + "_" + country + "_" + variant)) {
            localeName = PALWcmConstants.LOCALE_SEPARETER + language + "_"
                    + country + "_" + variant;
        } else if (localeList.contains(language + "_" + country)) {
            localeName = PALWcmConstants.LOCALE_SEPARETER + language + "_"
                    + country;
        } else if (localeList.contains(language)) {
            localeName = PALWcmConstants.LOCALE_SEPARETER + language;
        }

        String title = getFragmentPreferenceValue(fragment,
                PALWcmConstants.TITLE + localeName, "");
        String content = getFragmentPreferenceValue(fragment,
                PALWcmConstants.CONTENT + localeName, "");

        request.setAttribute(PALWcmConstants.TITLE, title);
        request.setAttribute(PALWcmConstants.CONTENT, content);

        PortletRequestDispatcher rd = getPortletContext().getRequestDispatcher(
                "/view/wcm/wcmView.jsp");
        rd.include(request, response);
    }

    /* (non-Javadoc)
     * @see javax.portlet.GenericPortlet#doEdit(javax.portlet.RenderRequest, javax.portlet.RenderResponse)
     */
    protected void doEdit(RenderRequest request, RenderResponse response)
            throws PortletException, IOException {
        response.setContentType("text/html");

        String msg = request.getParameter(PALWcmConstants.MESSAGE);
        if (msg != null) {
            request.setAttribute(PALWcmConstants.MESSAGE, msg);
        } else {
            request.setAttribute(PALWcmConstants.MESSAGE, "");
        }

        storeLoginStatus(request);

        String editType = request.getParameter(PALWcmConstants.EDIT_TYPE);
        if (PALWcmConstants.CONTENT_EDIT_TYPE.equals(editType)) {
            doContentEdit(request, response);
        } else if (PALWcmConstants.PREFERENCES_EDIT_TYPE.equals(editType)) {
            doPreferencesEdit(request, response);
        } else {
            doContentEdit(request, response);
        }
    }

    protected void doContentEdit(RenderRequest request, RenderResponse response)
            throws PortletException, IOException {
        String localeName = getEditLocaleName(request);
        String selectedLocale = localeName;
        if (!localeName.equals("")) {
            localeName = PALWcmConstants.LOCALE_SEPARETER + localeName;
        }

        Fragment fragment = getCurrentFragment(request);
        String title = getFragmentPreferenceValue(fragment,
                PALWcmConstants.TITLE + localeName, "");
        String content = getFragmentPreferenceValue(fragment,
                PALWcmConstants.CONTENT + localeName, "");
        String localeOptions = getLocaleOptionString(fragment, selectedLocale);
        String fragmentId = fragment.getId();

        response.setTitle(title);
        request.setAttribute(PALWcmConstants.TITLE, title);
        request.setAttribute(PALWcmConstants.CONTENT, StringEscapeUtils
                .escapeJavaScript(content));
        request.setAttribute(PALWcmConstants.LOCALE_LIST, localeOptions);
        request.setAttribute(PALWcmConstants.FRAGMENT_ID, fragmentId);
        request.setAttribute(PALWcmConstants.PORTAL_CONTEXT_PATH,
                getPortalContextPath(request));

        PortletRequestDispatcher rd = getPortletContext().getRequestDispatcher(
                "/view/wcm/wcmEdit.jsp");
        rd.include(request, response);

    }

    protected String getEditLocaleName(PortletRequest request) {
        PortletSession portletSession = request.getPortletSession();
        Object locale = portletSession
                .getAttribute(PALWcmConstants.EDIT_LOCALE);
        if (locale != null) {
            return locale.toString();
        }
        return "";
    }

    protected void setEditLocaleName(PortletRequest request, String localeName) {
        PortletSession portletSession = request.getPortletSession();
        portletSession.setAttribute(PALWcmConstants.EDIT_LOCALE, localeName);
    }

    protected void doPreferencesEdit(RenderRequest request,
            RenderResponse response) throws PortletException, IOException {
        Fragment fragment = getCurrentFragment(request);
        String fragmentId = fragment.getId();

        request.setAttribute(PALWcmConstants.FRAGMENT_ID, fragmentId);
        request.setAttribute(PALWcmConstants.LOCALE_LIST,
                getLocaleOptionString(fragment, null));

        PortletRequestDispatcher rd = getPortletContext().getRequestDispatcher(
                "/view/wcm/wcmPreferences.jsp");
        rd.include(request, response);

    }

    private String getLocaleOptionString(Fragment fragment, String localeName) {
        List locales = getLocaleList(fragment);
        StringBuffer localesStr = new StringBuffer();
        for (Iterator ite = locales.iterator(); ite.hasNext();) {
            String lc = ite.next().toString();
            localesStr.append("<option value=\"").append(lc).append("\"");
            if (lc.equals(localeName)) {
                localesStr.append(" selected");
            }
            localesStr.append(">").append(lc).append("</option>");
        }
        return localesStr.toString();
    }

    private List getLocaleList(Fragment fragment) {
        List locales = new ArrayList();
        //        List preferences = fragment.getPreferences();
        //        for (Iterator ite = preferences.iterator(); ite.hasNext();)
        //        {
        //            FragmentPreference fp = (FragmentPreference) ite.next();
        //            if (fp.getName() != null
        //                    && fp.getName().startsWith(PALWcmConstants.TITLE))
        //            {
        //                locales.add(fp.getName().substring(
        //                        PALWcmConstants.TITLE.length() + 1));
        //            }
        //        }

        String localeNames = getFragmentPreferenceValue(fragment,
                PALWcmConstants.LOCALE_LIST, null);
        if (localeNames != null) {
            StringTokenizer st = new StringTokenizer(localeNames, ",");
            while (st.hasMoreTokens()) {
                locales.add(st.nextToken());
            }
        }
        return locales;
    }

    /* (non-Javadoc)
     * @see javax.portlet.Portlet#processAction(javax.portlet.ActionRequest, javax.portlet.ActionResponse)
     */
    public void processAction(ActionRequest request, ActionResponse response)
            throws PortletException, IOException {
        if (PortletMode.VIEW.equals(request.getPortletMode())) {
            processView(request, response);
        } else if (PortletMode.EDIT.equals(request.getPortletMode())) {
            processEdit(request, response);
        } else {
            log.warn("Invalid action prcess.");
        }
    }

    protected void processView(ActionRequest request, ActionResponse response)
            throws PortletException, IOException {
        //nothing
    }

    protected void processEdit(ActionRequest request, ActionResponse response)
            throws PortletException, IOException {
        String editType = request.getParameter(PALWcmConstants.EDIT_TYPE);
        if (PALWcmConstants.CONTENT_EDIT_TYPE.equals(editType)) {
            processContentEdit(request, response);
            response.setRenderParameter(PALWcmConstants.EDIT_TYPE,
                    PALWcmConstants.CONTENT_EDIT_TYPE);
        } else if (PALWcmConstants.PREFERENCES_EDIT_TYPE.equals(editType)) {
            processPreferencesEdit(request, response);
            response.setRenderParameter(PALWcmConstants.EDIT_TYPE,
                    PALWcmConstants.PREFERENCES_EDIT_TYPE);
        } else {
            //TODO print error
            processContentEdit(request, response);
            response.setRenderParameter(PALWcmConstants.EDIT_TYPE,
                    PALWcmConstants.CONTENT_EDIT_TYPE);
        }
    }

    protected void processContentEdit(ActionRequest request,
            ActionResponse response) throws PortletException, IOException {
        String localeName = request.getParameter(PALWcmConstants.EDIT_LOCALE);
        setEditLocaleName(request, localeName);
        String actionType = request.getParameter(PALWcmConstants.ACTION_TYPE);
        if (PALWcmConstants.CHANGE_LOCALE.equals(actionType)) {
            // no save 
            return;
        }

        String title = request.getParameter(PALWcmConstants.TITLE);
        String content = request.getParameter(PALWcmConstants.CONTENT);
        String fragmentId = request.getParameter(PALWcmConstants.FRAGMENT_ID);

        if (localeName == null) {
            localeName = "";
        } else if (!localeName.equals("")) {
            localeName = PALWcmConstants.LOCALE_SEPARETER + localeName;
        }

        // TODO check above

        ContentPage page = getCurrentPage(request);
        Fragment fragment = page.getFragmentById(fragmentId);

        setFragmentPreferenceValue(fragment,
                PALWcmConstants.TITLE + localeName, title);
        setFragmentPreferenceValue(fragment, PALWcmConstants.CONTENT
                + localeName, content);

        try {
            pageManager.updatePage(page);
            request.setAttribute(PALWcmConstants.MESSAGE,
                    "Failed to update data. ");
        } catch (PageNotUpdatedException e) {
            log.error("Failed to update data. ", e);
            //TODO i18n
            request.setAttribute(PALWcmConstants.MESSAGE,
                    "Failed to update data. ");
        } catch (NodeException e) {
            log.error("Failed to update data. ", e);
            //TODO i18n
            request.setAttribute(PALWcmConstants.MESSAGE,
                    "Failed to update data. ");
        }
    }

    protected void processPreferencesEdit(ActionRequest request,
            ActionResponse response) throws PortletException, IOException {
        String actionType = request.getParameter(PALWcmConstants.ACTION_TYPE);
        if (PALWcmConstants.ADD_LOCALE.equals(actionType)) {
            // add locale
            String fragmentId = request
                    .getParameter(PALWcmConstants.FRAGMENT_ID);
            ContentPage page = getCurrentPage(request);
            Fragment fragment = page.getFragmentById(fragmentId);

            // TODO check above

            String targetLocale = request
                    .getParameter(PALWcmConstants.TARGET_LOCALE);
            if (targetLocale == null || targetLocale.equals("")) {
                //TODO i18n
                request
                        .setAttribute(PALWcmConstants.MESSAGE,
                                "Failed to add a locale. The specified locale is empty. ");
                return;
            }

            List locales = getLocaleList(fragment);
            if (locales.contains(targetLocale)) {
                //TODO i18n
                request
                        .setAttribute(PALWcmConstants.MESSAGE,
                                "Failed to add the locale. The specified locale has been already added. ");
                return;
            }

            locales.add(targetLocale);
            Object[] objs = locales.toArray();
            Arrays.sort(objs);
            StringBuffer localeNames = new StringBuffer();
            for (int i = 0; i < objs.length; i++) {
                if (i != 0) {
                    localeNames.append(",");
                }
                localeNames.append(objs[i].toString());
            }
            setFragmentPreferenceValue(fragment, PALWcmConstants.LOCALE_LIST,
                    localeNames.toString());

            try {
                pageManager.updatePage(page);
                //TODO i18n
                request.setAttribute(PALWcmConstants.MESSAGE,
                        "Added the locale. ");
            } catch (PageNotUpdatedException e) {
                log.error("Failed to add the locale. ", e);
                //TODO i18n
                request.setAttribute(PALWcmConstants.MESSAGE,
                        "Failed to add the locale. ");
            } catch (NodeException e) {
                log.error("Failed to add the locale. ", e);
                //TODO i18n
                request.setAttribute(PALWcmConstants.MESSAGE,
                        "Failed to add the locale. ");
            }
        } else if (PALWcmConstants.DELETE_LOCALE.equals(actionType)) {
            // delete locale
            String fragmentId = request
                    .getParameter(PALWcmConstants.FRAGMENT_ID);
            ContentPage page = getCurrentPage(request);
            Fragment fragment = page.getFragmentById(fragmentId);

            // TODO check above

            String selectedLocale = request
                    .getParameter(PALWcmConstants.SELECTED_LOCALE);
            if (selectedLocale == null || selectedLocale.equals("")) {
                //TODO i18n
                request
                        .setAttribute(PALWcmConstants.MESSAGE,
                                "Failed to delete a locale. The specified locale is empty. ");
                return;
            }

            List locales = getLocaleList(fragment);
            if (!locales.contains(selectedLocale)) {
                //TODO i18n
                request
                        .setAttribute(PALWcmConstants.MESSAGE,
                                "Failed to delete the locale. The specified locale is not included. ");
                return;
            }

            locales.remove(selectedLocale);
            Object[] objs = locales.toArray();
            StringBuffer localeNames = new StringBuffer();
            for (int i = 0; i < objs.length; i++) {
                if (i != 0) {
                    localeNames.append(",");
                }
                localeNames.append(objs[i].toString());
            }
            setFragmentPreferenceValue(fragment, PALWcmConstants.LOCALE_LIST,
                    localeNames.toString());
            removeFragmentPreferenceValue(fragment, PALWcmConstants.TITLE
                    + PALWcmConstants.LOCALE_SEPARETER + selectedLocale);
            removeFragmentPreferenceValue(fragment, PALWcmConstants.CONTENT
                    + PALWcmConstants.LOCALE_SEPARETER + selectedLocale);

            try {
                pageManager.updatePage(page);
                //TODO i18n
                request.setAttribute(PALWcmConstants.MESSAGE,
                        "Deleted the locale. ");
            } catch (PageNotUpdatedException e) {
                log.error("Failed to delete the locale. ", e);
                //TODO i18n
                request.setAttribute(PALWcmConstants.MESSAGE,
                        "Failed to delete the locale. ");
            } catch (NodeException e) {
                log.error("Failed to delete the locale. ", e);
                //TODO i18n
                request.setAttribute(PALWcmConstants.MESSAGE,
                        "Failed to delete the locale. ");
            }
        }
    }

    private String getPortalContextPath(PortletRequest request) {
        RequestContext requestContext = (RequestContext) request
                .getAttribute(PortalReservedParameters.REQUEST_CONTEXT_ATTRIBUTE);
        return requestContext.getRequest().getContextPath();
    }

    private ContentPage getCurrentPage(PortletRequest request) {
        RequestContext requestContext = (RequestContext) request
                .getAttribute(PortalReservedParameters.REQUEST_CONTEXT_ATTRIBUTE);
        return requestContext.getPage();
    }

    private ContentFragment getCurrentFragment(RenderRequest request) {
        return (ContentFragment) request
                .getAttribute(PortalReservedParameters.FRAGMENT_ATTRIBUTE);
    }

    private FragmentPreference getFragmentPreferenceByName(Fragment fragment,
            String name) {
        if (name == null) {
            return null;
        }
        List preferences = fragment.getPreferences();
        for (Iterator ite = preferences.iterator(); ite.hasNext();) {
            FragmentPreference fp = (FragmentPreference) ite.next();
            if (fp.getName() != null && fp.getName().equals(name)) {
                return fp;
            }
        }
        return null;
    }

    private void setFragmentPreferenceValue(Fragment fragment, String name,
            String value) {
        FragmentPreference fp = getFragmentPreferenceByName(fragment, name);
        if (fp == null) {
            fp = pageManager.newFragmentPreference();
            fp.setName(name);
            fp.setReadOnly(false);
            List list = new ArrayList();
            list.add(StringEscapeUtils.escapeXml(value));
            fp.setValueList(list);
            fragment.getPreferences().add(fp);
        } else {
            fp.getValueList().set(0, StringEscapeUtils.escapeXml(value));
        }
    }

    private void removeFragmentPreferenceValue(Fragment fragment, String name) {
        FragmentPreference fp = getFragmentPreferenceByName(fragment, name);
        fragment.getPreferences().remove(fp);
    }

    private String getFragmentPreferenceValue(Fragment fragment, String name,
            String defaultValue) {
        FragmentPreference fp = getFragmentPreferenceByName(fragment, name);
        if (fp != null) {
            if (fp.getValueList().size() > 0) {
                return StringEscapeUtils.unescapeXml((String) fp.getValueList()
                        .get(0));
            }
        }
        return defaultValue;
    }

    protected void storeLoginStatus(PortletRequest request) {
        PortletSession portletSession = request.getPortletSession();
        if (request.getRemoteUser() != null) {
            portletSession.setAttribute(PALWcmConstants.LOGIN_USER_NAME,
                    request.getRemoteUser(), PortletSession.APPLICATION_SCOPE);
        } else {
            portletSession.removeAttribute(PALWcmConstants.LOGIN_USER_NAME,
                    PortletSession.APPLICATION_SCOPE);
        }
    }

}
