/*
 * Copyright 2005-2006 Portal Application Laboratory project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.pal.jsf.custom.fileupload;

import java.io.IOException;
import java.util.Map;

import javax.faces.component.UIComponent;
import javax.faces.context.ExternalContext;
import javax.faces.context.FacesContext;
import javax.faces.context.ResponseWriter;
import javax.faces.convert.ConverterException;
import javax.faces.internal.IgnoreAttribute;
import javax.servlet.ServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;

import jp.sf.pal.jsf.JsfConstants;
import jp.sf.pal.jsf.multipart.MultipartRequestWrapper;

import org.apache.commons.fileupload.FileItem;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.seasar.teeda.core.render.AbstractInputRenderer;
import org.seasar.teeda.core.util.RendererUtil;
import org.seasar.teeda.core.util.ValueHolderUtil;

public class HtmlFileUploadRenderer extends AbstractInputRenderer {
    private static final Log log = LogFactory
            .getLog(HtmlFileUploadRenderer.class);

    private final IgnoreAttribute ignoreComponent = new IgnoreAttribute();
    {
        ignoreComponent.addAttributeName(JsfConstants.ID_ATTR);
        ignoreComponent.addAttributeName(JsfConstants.NAME_ATTR);
        ignoreComponent.addAttributeName(JsfConstants.TYPE_ATTR);
        ignoreComponent.addAttributeName(JsfConstants.VALUE_ATTR);
        ignoreComponent.addAttributeName(JsfConstants.STYLE_CLASS_ATTR);
    }

    public void encodeEnd(FacesContext context, UIComponent component)
            throws IOException {
        assertNotNull(context, component);
        if (!component.isRendered()) {
            return;
        }
        encodeHtmlInputFileUploadEnd(context, (HtmlInputFileUpload) component);
    }

    protected void encodeHtmlInputFileUploadEnd(FacesContext context,
            HtmlInputFileUpload htmlInputFileUpload) throws IOException {
        ResponseWriter writer = context.getResponseWriter();
        writer.startElement(JsfConstants.INPUT_ELEM, htmlInputFileUpload);
        RendererUtil.renderAttribute(writer, JsfConstants.TYPE_ATTR,
                JsfConstants.FILE_ATTR);
        RendererUtil.renderIdAttributeIfNecessary(writer, htmlInputFileUpload,
                getIdForRender(context, htmlInputFileUpload));
        RendererUtil.renderAttribute(writer, JsfConstants.NAME_ATTR,
                htmlInputFileUpload.getClientId(context));
        UploadedFile value = (UploadedFile) ValueHolderUtil
                .getValue(htmlInputFileUpload);
        if (value != null) {
            if (value.getName() != null) {
                writer.writeAttribute(JsfConstants.VALUE_ATTR, value.getName(),
                        null);
            }
        }
        RendererUtil.renderAttribute(writer, JsfConstants.AUTOCOMPLETE_ATTR,
                htmlInputFileUpload.getAutocomplete());
        renderStyleClass(context, htmlInputFileUpload, writer);
        renderRemainAttributes(htmlInputFileUpload, writer, ignoreComponent);
        writer.endElement(JsfConstants.INPUT_ELEM);
    }

    protected void renderStyleClass(FacesContext context,
            HtmlInputFileUpload htmlInputFileUpload, ResponseWriter writer)
            throws IOException {
        final String styleClass = htmlInputFileUpload.getStyleClass();
        RendererUtil.renderAttribute(writer, JsfConstants.STYLE_CLASS_ATTR,
                styleClass);
    }

    public void decode(FacesContext context, UIComponent component) {
        assertNotNull(context, component);
        
        Object request = context.getExternalContext().getRequest();
        if (!(request instanceof ServletRequest)) {
            ExternalContext externalContext = context.getExternalContext();
            Map fileItems = (Map) externalContext.getRequestMap().get(
                    MultipartRequestWrapper.UPLOADED_FILES_ATTRIBUTE);
            FileItem fileItem = null;
            if (fileItems != null) {
                String paramName = component.getClientId(context);
                fileItem = (FileItem) fileItems.get(paramName);
            }
            if (fileItem != null) {
                try {
                    UploadedFile upFile;
                    String implementation = ((HtmlInputFileUpload) component)
                            .getStorage();
                    if (implementation == null
                            || ("memory").equals(implementation))
                        upFile = new UploadedFileDefaultMemoryImpl(fileItem);
                    else
                        upFile = new UploadedFileDefaultFileImpl(fileItem);
                    ((HtmlInputFileUpload) component)
                            .setSubmittedValue(upFile);
                    ((HtmlInputFileUpload) component).setValid(true);
                } catch (IOException ioe) {
                    log.error(ioe);
                }
            }
            return;
        }
        if (context.getExternalContext().getRequest() instanceof ServletRequest) {
            ServletRequest multipartRequest = (ServletRequest) context
                    .getExternalContext().getRequest();
            while (multipartRequest != null
                    && !(multipartRequest instanceof MultipartRequestWrapper)) {
                if (multipartRequest instanceof HttpServletRequestWrapper) {
                    multipartRequest = ((HttpServletRequestWrapper) multipartRequest)
                            .getRequest();
                } else {
                    multipartRequest = null;
                }
            }

            if (multipartRequest != null) {
                MultipartRequestWrapper mpReq = (MultipartRequestWrapper) multipartRequest;

                String paramName = component.getClientId(context);
                FileItem fileItem = mpReq.getFileItem(paramName);
                if (fileItem != null) {
                    try {
                        UploadedFile upFile;
                        String implementation = ((HtmlInputFileUpload) component)
                                .getStorage();
                        if (implementation == null
                                || ("memory").equals(implementation))
                            upFile = new UploadedFileDefaultMemoryImpl(fileItem);
                        else
                            upFile = new UploadedFileDefaultFileImpl(fileItem);
                        ((HtmlInputFileUpload) component)
                                .setSubmittedValue(upFile);
                        ((HtmlInputFileUpload) component).setValid(true);
                    } catch (IOException ioe) {
                        log.error(ioe);
                    }
                }
            }
        }
    }

    public Object getConvertedValue(FacesContext context,
            UIComponent component, Object submittedValue)
            throws ConverterException {
        if (submittedValue instanceof UploadedFile) {
            UploadedFile file = (UploadedFile) submittedValue;

            if (file.getName() != null && file.getName().length() > 0) {
                return file;
            }
        }

        return null;
    }
}
