/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.pal.admin.service;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import jp.sf.pal.admin.PALAdminConstants;
import jp.sf.pal.admin.PALAdminException;
import jp.sf.pal.admin.dxo.UserDxo;
import jp.sf.pal.admin.entity.Group;
import jp.sf.pal.admin.entity.PrincipalRule;
import jp.sf.pal.admin.entity.Role;
import jp.sf.pal.admin.entity.User;
import jp.sf.pal.admin.logic.SecurityProviderLogic;
import jp.sf.pal.admin.pager.UserPager;
import jp.sf.pal.admin.util.PALAdminUtil;
import jp.sf.pal.admin.util.PagerUtil;
import jp.sf.pal.admin.web.user.AbstractUserPage;
import jp.sf.pal.admin.web.user.UserEditPage;
import jp.sf.pal.admin.web.user.UserGroupEditPage;
import jp.sf.pal.admin.web.user.UserListPage;
import jp.sf.pal.admin.web.user.UserProfilingRuleEditPage;
import jp.sf.pal.admin.web.user.UserRoleEditPage;

import org.seasar.teeda.extension.util.LabelHelper;

public class UserManagementService implements Serializable
{
    /**
     * 
     */
    private static final long serialVersionUID = -2335697469833425547L;

    private UserPager userPager;

    private SecurityProviderLogic securityProviderLogic;

    private UserDxo userDxo;

    private LabelHelper labelHelper;

    /**
     * @return the securityProviderLogic
     */
    public SecurityProviderLogic getSecurityProviderLogic()
    {
        return securityProviderLogic;
    }

    /**
     * @param securityProviderLogic the securityProviderLogic to set
     */
    public void setSecurityProviderLogic(
            SecurityProviderLogic securityProviderLogic)
    {
        this.securityProviderLogic = securityProviderLogic;
    }

    /**
     * @return the userDxo
     */
    public UserDxo getUserDxo()
    {
        return userDxo;
    }

    /**
     * @param userDxo the userDxo to set
     */
    public void setUserDxo(UserDxo userDxo)
    {
        this.userDxo = userDxo;
    }

    /**
     * @return the userListPager
     */
    public UserPager getUserPager()
    {
        return userPager;
    }

    /**
     * @param userListPager the userListPager to set
     */
    public void setUserPager(UserPager userListPager)
    {
        this.userPager = userListPager;
    }

    /**
     * @return the labelHelper
     */
    public LabelHelper getLabelHelper()
    {
        return labelHelper;
    }

    /**
     * @param labelHelper the labelHelper to set
     */
    public void setLabelHelper(LabelHelper labelHelper)
    {
        this.labelHelper = labelHelper;
    }

    public void loadPage(UserListPage page) throws PALAdminException
    {
        UserPager pager = getUserPager();

        // check page action
        PagerUtil.updatePageNumber(pager,
                PALAdminConstants.PREVIOUS_PAGE_NUMBER,
                PALAdminConstants.NEXT_PAGE_NUMBER);

        // set user list
        List<Map<String, Object>> list = new ArrayList<Map<String, Object>>();
        userDxo.convert(securityProviderLogic.getUsers(pager), list);
        PagerUtil.updatePagerPage(page, pager);
        page.setUserItems(list);

    }

    public void insert(AbstractUserPage page) throws PALAdminException
    {
        User entity = new User();
        userDxo.convert(page, entity);
        securityProviderLogic.insertUser(entity);
    }

    public void delete(AbstractUserPage page) throws PALAdminException
    {
        User entity = new User();
        userDxo.convert(page, entity);
        securityProviderLogic.deleteUser(entity);
    }

    public List<Map<String, String>> getProfilingRuleNames()
    {
        return PALAdminUtil.createItems(securityProviderLogic
                .getProfilingRuleNames().iterator());
    }

    public boolean isUserExists(String username)
    {
        if (username == null || username.equals(""))
        {
            return false;
        }
        User user = getSecurityProviderLogic().getUser(username);
        if (user != null)
        {
            return true;
        }
        return false;
    }

    public void loadPage(UserEditPage page)
    {
        User user = getSecurityProviderLogic().getUser(page.getName());
        getUserDxo().convert(user, page);

        // set pull-down
        LabelHelper labelHelper = getLabelHelper();
        List<Map<String, String>> list = new ArrayList<Map<String, String>>();
        list.add(PALAdminUtil.createItem(labelHelper.getLabelValue("Active"),
                PALAdminConstants.USER_CREDENTIAL_ACTIVE));
        list.add(PALAdminUtil.createItem(labelHelper.getLabelValue("Expired"),
                PALAdminConstants.USER_CREDENTIAL_EXPIRED));
        list.add(PALAdminUtil.createItem(labelHelper.getLabelValue("Extend"),
                PALAdminConstants.USER_CREDENTIAL_EXTEND));
        list.add(PALAdminUtil.createItem(labelHelper
                .getLabelValue("ExtendUnlimited"),
                PALAdminConstants.USER_CREDENTIAL_EXTEND_UNLIMITED));
        page.setExpiresItems(list);
    }

    public void update(UserEditPage page) throws PALAdminException
    {
        User entity = new User();
        getUserDxo().convert(page, entity);
        getSecurityProviderLogic().updateUser(entity);
    }

    public void loadPage(UserGroupEditPage page)
    {
        page.setNewGroupItems(getGroupNames());
        page.setGroupsItems(getGroupNamesForUsername(page.getName()));
    }

    public List<Map<String, String>> getGroupNames()
    {
        return PALAdminUtil.createItems(securityProviderLogic.getGroupNames()
                .iterator());
    }

    public List<Map<String, String>> getGroupNamesForUsername(String username)
    {
        return PALAdminUtil.createItems(securityProviderLogic
                .getGroupNamesForUsername(username).iterator());
    }

    public void addGroupToUser(UserGroupEditPage page) throws PALAdminException
    {
        User user = new User();
        user.setName(page.getName());
        Group group = new Group();
        group.setName(page.getNewGroup());
        getSecurityProviderLogic().addUserToGroup(user, group);
    }

    public void removeGroupsFromUser(UserGroupEditPage page)
            throws PALAdminException
    {
        List<String> failedGroups = new ArrayList<String>();
        User user = new User();
        user.setName(page.getName());
        String[] groupnames = page.getGroups();
        for (int i = 0; i < groupnames.length; i++)
        {
            Group group = new Group();
            group.setName(groupnames[i]);
            try
            {
                getSecurityProviderLogic().removeGroupFromUser(user, group);
            }
            catch (PALAdminException e)
            {
                failedGroups.add(groupnames[i]);
            }
        }
        if (failedGroups.size() > 0)
        {
            throw new PALAdminException("Could not remove groups: "
                    + failedGroups);
        }
    }

    public boolean isGroupExists(String groupname)
    {
        Group group = new Group();
        group.setName(groupname);
        return getSecurityProviderLogic().exists(group);
    }

    public boolean isGroupExistsInUser(String username, String groupname)
    {
        if (username == null || groupname == null)
        {
            return false;
        }
        Iterator<String> ite = getSecurityProviderLogic()
                .getGroupNamesForUsername(username).iterator();
        while (ite.hasNext())
        {
            if (groupname.equals(ite.next()))
            {
                return true;
            }
        }

        return false;
    }

    public void loadPage(UserRoleEditPage page)
    {
        page.setNewRoleItems(getRoleNames());
        page.setRolesItems(getRoleNamesForUsername(page.getName()));
    }

    public void updateQuery(UserListPage page)
    {
        if (page.getFilter() != null && !page.getFilter().equals(""))
        {
            getUserPager().setFilter(page.getFilter());
        }
        else
        {
            getUserPager().setFilter("");
        }
    }

    public List<Map<String, String>> getRoleNames()
    {
        return PALAdminUtil.createItems(securityProviderLogic.getRoleNames()
                .iterator());
    }

    public List<Map<String, String>> getRoleNamesForUsername(String username)
    {
        return PALAdminUtil.createItems(securityProviderLogic
                .getRoleNamesForUsername(username).iterator());
    }

    public void addRoleToUser(UserRoleEditPage page) throws PALAdminException
    {
        User user = new User();
        user.setName(page.getName());
        Role role = new Role();
        role.setName(page.getNewRole());
        getSecurityProviderLogic().addUserToRole(user, role);
    }

    public void removeRolesFromUser(UserRoleEditPage page)
            throws PALAdminException
    {
        List<String> failedRoles = new ArrayList<String>();
        User user = new User();
        user.setName(page.getName());
        String[] rolenames = page.getRoles();
        for (int i = 0; i < rolenames.length; i++)
        {
            Role role = new Role();
            role.setName(rolenames[i]);
            try
            {
                getSecurityProviderLogic().removeRoleFromUser(user, role);
            }
            catch (PALAdminException e)
            {
                failedRoles.add(rolenames[i]);
            }
        }
        if (failedRoles.size() > 0)
        {
            throw new PALAdminException("Could not remove roles: "
                    + failedRoles);
        }
    }

    public boolean isRoleExists(String rolename)
    {
        Role role = new Role();
        role.setName(rolename);
        return getSecurityProviderLogic().exists(role);
    }

    public boolean isRoleExistsInUser(String username, String rolename)
    {
        if (username == null || rolename == null)
        {
            return false;
        }
        Iterator<String> ite = getSecurityProviderLogic()
                .getRoleNamesForUsername(username).iterator();
        while (ite.hasNext())
        {
            if (rolename.equals(ite.next()))
            {
                return true;
            }
        }

        return false;
    }

    public void loadPage(UserProfilingRuleEditPage page)
    {
        List<Map<String, String>> locatorList = new ArrayList<Map<String, String>>();
        locatorList.add(PALAdminUtil.createItem(PALAdminConstants.LOCATOR_PAGE,
                PALAdminConstants.LOCATOR_PAGE));
        locatorList.add(PALAdminUtil.createItem(PALAdminConstants.LOCATOR_MENU,
                PALAdminConstants.LOCATOR_MENU));
        page.setNewLocatorNameItems(locatorList);

        page.setNewProfilingRuleNameItems(getProfilingRuleNames());

        User user = new User();
        user.setName(page.getName());
        page.setPrincipalRulesItems(getSecurityProviderLogic()
                .getPrincipalRuleForUser(user));
    }

    public void addPrincipalRuleToUser(UserProfilingRuleEditPage page)
            throws PALAdminException
    {
        User user = new User();
        user.setName(page.getName());
        PrincipalRule principalRule = new PrincipalRule();
        principalRule.setLocatorName(page.getNewLocatorName());
        principalRule.setProfilingRuleName(page.getNewProfilingRuleName());
        getSecurityProviderLogic().addPrincipalRuleToUser(user, principalRule);
    }

    public void removePrincipalRulesFromUser(UserProfilingRuleEditPage page)
            throws PALAdminException
    {
        List<String> failedPrincipalRules = new ArrayList<String>();
        User user = new User();
        user.setName(page.getName());
        String[] principalRules = page.getPrincipalRules();
        for (int i = 0; i < principalRules.length; i++)
        {
            try
            {
                PrincipalRule principalRule = PrincipalRule
                        .getInstance(principalRules[i]);
                getSecurityProviderLogic().removePrincipalRuleFromUser(user,
                        principalRule);
            }
            catch (PALAdminException e)
            {
                failedPrincipalRules.add(principalRules[i]);
            }
        }
        if (failedPrincipalRules.size() > 0)
        {
            throw new PALAdminException("Could not remove principalRules: "
                    + failedPrincipalRules);
        }
    }
}
