/*
 * ttf2owata  --  Convert embedded bitmap data of TrueType font to owata font
 * version 1.0
 */
 
/*************************************************************
Comment and Copyright of original program source
*************************************************************/
/*
 * sbitget  --  extract bitmap-data from a TrueType font
 * version 0.1
 * Mon Jan 30 2002
 * Itou Hiroki
 */

/*
 * Copyright (c) 2002 ITOU Hiroki
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#define _CRT_SECURE_NO_DEPRECATE

#define PROGNAME "ttf2owata"
#define PROGVERSION "1.0"

#include <windows.h>
#include <stdio.h>
#include <sys/stat.h> /* stat() */

#define uchar unsigned char
#define ulong unsigned long
#define ushort unsigned short
#define BUFSIZE 64
#define GLYPHBUFSIZE 1000
#define MAXGLYPHWIDTH 128
#define TMPFILE "sbit.tmp"
#define MAXFILENAMECHAR 256
#define MAXSTRINGINBDF 1000
#define LEVELCOPYRIGHTSTR 4
#define LEVELFONTNAMESTR 8
#define STRUNKNOWN "???"

//info of tables
typedef struct linkedlist_tag{
    struct linkedlist_tag *next; //next element's on-memory location
    ulong offset; //(byte) offset from top of TrueTypeFile to top of this table
    //    ulong len; //(byte) length of this table
    char tag[5];     //name of this table (4 characters + '\0')
} tableinfo;

//info of glyph-metric (metric means size)
typedef struct {
    uchar width;
    uchar height;
    ulong off;  //offset from top of EBDT to top of glyph data
    uchar advance; // same as DWIDTH(BDF)
    int offsetx;   //offset of boundingbox from origin to point of
                   //most right-under.
                   // that is, the axis of most right-under
    int offsety;
    uchar ppem; //pixel-size in this strike
    uchar *ebdtL; //on-memory location: top of EBDT
    int imageFormat;
    ushort id; //glyph ID number (index number)
} metricinfo;

typedef struct {
    uchar *subtableL; //location top of indexSubTable (in EBLC)
    ushort first; //firstGlyphIndex
    ushort last; //lastGlyphIndex
    int indexFormat;
    ulong off; //imageDataOffset from EBDTtop to locationOfGlyphsData
} indexSubTable_info;

typedef struct {
    uchar *strL;
    ushort slen;
} str_info;

typedef struct {
    ushort endCount;
    ushort startCount;
    ushort idDelta;
    ushort idRangeOffset;
} mapSegment;

/* func prototype */
int main(int argc, char **argv);
uchar* see_cmap(uchar *cmapL, int* segCount);
int getGlyphId(mapSegment *map, int segCount, int hi, int lo);
void see_indexSubTable(indexSubTable_info *st, uchar *ebdtL, FILE *outfp, int glyphId, metricinfo bbox);
void getTableInfo(uchar *p, tableinfo *t);
void validiateTTF(uchar *p);
void see_bitmapSizeTable(uchar *p, int *numElem, ulong *arrayOffset, metricinfo *bbox);
uchar *see_sbitLineMetrics(uchar *p, metricinfo *bbox, int direction);
void see_indexSubTableArray(uchar *elemL, uchar *arrayL, indexSubTable_info *st);
int see_indexSubHeader(indexSubTable_info *st);
void putNoGlyph(FILE *outfp, metricinfo bbox);
void putglyph(metricinfo *glyph, int size, FILE *outfp, metricinfo bbox);
void setGlyphHead(metricinfo *g, FILE *outfp);
void setGlyphBody_byte(uchar *p, const uchar *end, metricinfo *g, FILE *outfp, metricinfo bbox);
void setGlyphBody_bit(uchar *p, const uchar *end, metricinfo *g, FILE *outfp, metricinfo bbox);
void errexit(char *fmt, ...);
uchar *see_glyphMetrics(uchar *p, metricinfo *met, int big);
void see_name(uchar *nameL, char *copyright, char *fontname);
void copystr(char *dst, uchar *src, ushort len, int forFilename);


#define readShort(V) \
    (V)=(uchar)*p++;\
    (V)=(V)<<8 | (uchar)*p++;

#define readLong(V) \
    (V)=(uchar)*p++;\
    (V)=(V)<<8 | (uchar)*p++;\
    (V)=(V)<<8 | (uchar)*p++;\
    (V)=(V)<<8 | (uchar)*p++;

/*
 * reading a TrueType font
 *   calling functions for 'name', 'EBLC', 'EBDT' tables
 */
int main(int argc, char **argv){
    uchar *ttfL; //on memory location: top of TrueTypeFile
    char copyright[MAXSTRINGINBDF] = STRUNKNOWN;
    char fontname[MAXSTRINGINBDF] = STRUNKNOWN;

    if(argc!=2){
        //argv[1] = "D:\\Project\\win\\ttf2owata\\debug\\FONT01.ttf";
        //argv[1] = "D:\\Project\\win\\ttf2owata\\debug\\ipamona.ttf";
        fprintf(stderr, PROGNAME " version " PROGVERSION " - Convert embedded bitmap data of TrueType font to owata font\n");
        fprintf(stderr, "usage:  " PROGNAME " file.ttf\n");
        exit(1);
    }
    /*
     * reading TrueTypeFile to Memory
     */
    {
        FILE *fp;
        size_t ttfsize;
        struct stat info;

        if((fp=fopen(argv[1],"rb"))==NULL)
            errexit("cannot open '%s'", argv[1]);
        if(stat(argv[1], &info) != 0)
            errexit("stat");
        ttfsize = info.st_size;
        if((ttfL=malloc(ttfsize))==NULL)
            errexit("malloc");
        if(fread(ttfL, 1, ttfsize, fp)!=ttfsize)
            errexit("fread");
        fclose(fp);
    }
    //ckeck this file is TrueType? or not
    validiateTTF(ttfL);
    {
        int numSize; //number of bitmapSizeTable
        int segCount;
        int i, j, hi, lo, glyphId;
        tableinfo table; //store the first table
                          //dynamically allocate second and after tables
        tableinfo *t; //loop
        uchar *p;
        uchar *eblcL = NULL; //on memory location: top of EBLC table
        uchar *ebdtL = NULL; //on memory location: top of EBDT table
        uchar *cmapL = NULL; //on memory location: top of cmap table
        mapSegment *map = NULL;
        indexSubTable_info *st = NULL;

        //get locations of tables
        getTableInfo(ttfL, &table);

        /*
         * reading name table
         *   get strings of copyright, fontname
         */
        for(t=table.next; t->next!=NULL; t=t->next){
            if(strcmp("name", t->tag)==0){
                see_name(ttfL + t->offset, copyright, fontname);
                break;
            }
        }
        //EBDT table
        for(t=table.next; t->next!=NULL; t=t->next){
            if(strcmp("EBDT", t->tag)==0 || strcmp("bdat", t->tag)==0)
                ebdtL = ttfL + t->offset;
        }
        if(ebdtL == NULL){
            errexit("This font has no bitmap-data.");
        }
        // EBLC table
        for(t=table.next; t->next!=NULL; t=t->next){
            if(strcmp("EBLC", t->tag)==0 || strcmp("bloc", t->tag)==0)
                eblcL = ttfL + t->offset;
        }
        if(eblcL == NULL){
            errexit("This font has no bitmap-data.");
        }
        // cmap table
        for(t=table.next; t->next!=NULL; t=t->next){
            if(strcmp("cmap", t->tag)==0)
                cmapL = ttfL + t->offset;
        }
        if(cmapL == NULL){
            errexit("This font has no cmap-data.");
        }
        // get segment number of cmap
        p=see_cmap(cmapL, &segCount);
        map = (mapSegment*)malloc(sizeof(mapSegment)*segCount);
        if (map == NULL) {
            errexit("malloc map");
        }
        for (i = 0; i < segCount; i++) {
            readShort(map[i].endCount);
        }
        p += 2;
        for (i = 0; i < segCount; i++) {
            readShort(map[i].startCount);
        }
        for (i = 0; i < segCount; i++) {
            readShort(map[i].idDelta);
            map[i].idDelta %= 0x10000;
        }
        for (i = 0; i < segCount; i++) {
            readShort(map[i].idRangeOffset);
        }
        // get number of bitmapSizeTable
        p = eblcL;
        // skip version
        p += 4;
        readLong(numSize);
        for(i=0; i<numSize; i++){
            uchar *arrayL; //on-memory location: top of indexSubTableArray
            int numElem; //number of indexSubTableArray-elements
            metricinfo bbox; //strike's bounding box: metric info for strike
            FILE *outfp;
            char fname[MAXFILENAMECHAR];

            /*
             * reading a bitmapSizeTable
             *    get number of indexSubTableArrays
             *    get location of a first indexSubTableArray
             *    get info of BoundingBox
             */
            {
                ulong offset; //from EBLCtop to a first indexSubTableArray
                //8 = size of EBLCheader,  48 = size of one bitmapSizeTable
                see_bitmapSizeTable(eblcL+8+(48*i), &numElem, &offset, &bbox);
                arrayL = eblcL + offset;
            }
            if (bbox.height > 15) {
                continue;
            }
            /*
             * reading indexSubTableArrays and indexSubTables
             */
            free(st);
            st = (indexSubTable_info*)malloc(sizeof(indexSubTable_info)*numElem);
            for(j=0; j<numElem; j++){
                /*
                 * reading an indexSubTableArray
                 *   get firstGlyphIndex, lastGlyphIndex
                 *   get location of indexSubTable
                 */
                //8 = size of one indexSubTableArray
                see_indexSubTableArray(arrayL+(j*8), arrayL, &st[j]);
            }
            /*
             * single byte font
             */
            {
                if(strcmp(fontname,STRUNKNOWN)==0) {
                    sprintf(fname, "sbit%02dx%2dA.bin", bbox.width, bbox.height);
                }
                else {
                    sprintf(fname, "%s%02dx%2dA.bin", fontname, bbox.width, bbox.height);
                }
                if((outfp=fopen(fname,"wb"))==NULL) {
                    errexit("fopen");
                }
                printf("Writing %s\n", fname);
                // ASCII
                for (lo=0x20; lo<0x80; lo++) {
                    glyphId = getGlyphId(map, segCount, 0, lo);
                    for(j=0; j<numElem; j++){
                        if (glyphId >= st[j].first && glyphId <= st[j].last) {
                            break;
                        }
                    }
                    if (j >= numElem) {
                        putNoGlyph(outfp, bbox);
                    }
                    else {
                        see_indexSubTable(&st[j], ebdtL, outfp, glyphId, bbox);
                    }
                }
                // hankaku kana
                for (lo=0xA1; lo<0xE0; lo++) {
                    glyphId = getGlyphId(map, segCount, 0, lo);
                    for(j=0; j<numElem; j++){
                        if (glyphId >= st[j].first && glyphId <= st[j].last) {
                            break;
                        }
                    }
                    if (j >= numElem) {
                            putNoGlyph(outfp, bbox);
                    }
                    else {
                        see_indexSubTable(&st[j], ebdtL, outfp, glyphId, bbox);
                    }
                }
                fclose(outfp);
            }
            /*
             * multi bytes font
             */
            {
                if(strcmp(fontname,STRUNKNOWN)==0) {
                    sprintf(fname, "sbit%02dx%2dJ.bin", bbox.width, bbox.height);
                }
                else {
                    sprintf(fname, "%s%02dx%2dJ.bin", fontname, bbox.width, bbox.height);
                }
                if((outfp=fopen(fname,"wb"))==NULL) {
                    errexit("fopen");
                }
                printf("Writing %s\n", fname);
                for (hi=0x21; hi<0x7F; hi++) {
                    for (lo=0x21; lo<0x7F; lo++) {
                        glyphId = getGlyphId(map, segCount, hi, lo);
                        for(j=0; j<numElem; j++){
                            if (glyphId >= st[j].first && glyphId <= st[j].last) {
                                break;
                            }
                        }
                        if (j >= numElem) {
                            putNoGlyph(outfp, bbox);
                        }
                        else {
                            see_indexSubTable(&st[j], ebdtL, outfp, glyphId, bbox);
                        }
                    }
                }
                fclose(outfp);
            }
        }
    }

    return(EXIT_SUCCESS);
}

/*
 * get cmap segCount
 * in:  on-memory location to read
 *      (for out) int pointer to segCount
 * out: on-memory location to read next
 */
uchar* see_cmap(uchar *cmapL, int *segCount) {
    int i;
    int numEncTables, platformID, encodingID, format;
    ulong encOffset;
    uchar *p = cmapL;

    //skip version number
    p += 2;

    //number of encodingtables
    readShort(numEncTables);
    for (i=0; i<numEncTables; i++) {
        readShort(platformID);
        readShort(encodingID);
        readLong(encOffset);
        // Microsoft && Unicode
        if (platformID == 3 && encodingID == 1) {
            break;
        }
        encOffset=0;
    }
    if (encOffset == 0) {
        errexit("cmap ID error.");
    }
    p = cmapL+encOffset;
    readShort(format);
    // Format 4: Segment mapping to delta values
    if (format != 4) {
        errexit("format error.");
    }
    // skip length and version
    p += 4;
    readShort(*segCount);
    *segCount >>= 1;
    // skip searchRange, entrySelector  and rangeShift
    p += 6;

    return p;
}

/*
 * get Glyph index
 * in:  array of cmap segment
 *      number of segments
 *      JIS code
 * out: Glyph index
 */
int getGlyphId(mapSegment *map, int segCount, int hi, int lo)
{
    int i, ret, glyphId;
    CHAR bufa[4];
    WCHAR bufw[4];

    // jis to sjis
    if (hi) {
        hi -= 0x21;
        if (hi & 1) {
            lo += 0x7E;
        }
        else {
            lo += 0x1F;
            if (lo>= 0x7F) {
                lo++;
            }
        }
        hi >>= 1;
        hi += (hi <= 0x1E) ? 0x81 : 0xC1;
        bufa[0]=hi;
        bufa[1]=lo;
        bufa[2]='\0';
    }
    else {
        bufa[0] = lo;
        bufa[1] ='\0';
    }
    ret=MultiByteToWideChar(932, 0, bufa, 4, bufw, 4);
    for (i = 0; i < segCount; i++) {
        if (bufw[0] <= map[i].endCount) {
            break;
        }
    }
    if (map[i].startCount > bufw[0]) {
        //printf("jis=%02X%02X c=%04X startCount=%04X endCount=%04X missingGlyph\n", hi, lo, bufw[0], map[i].startCount, map[i].endCount);
        return -1;
    }
    if (map[i].idRangeOffset == 0) {
        glyphId = bufw[0] + map[i].idDelta;
    }
    else {
        glyphId = *(&map[i].idRangeOffset + map[i].idRangeOffset/2 + (bufw[0] - map[i].startCount));
    }
    glyphId %= 0x10000;
    //printf("sjis=%02X%02X c=%04X idDelta=%X glyphId=%04X\n", hi, lo, bufw[0], map[i].idDelta, glyphId);
    return glyphId;
}

/*
 * reading indexSubTable
 * in:  info of indexSubTable
 * out: number of glyphs contained in this indexSubTable
 */
void see_indexSubTable(indexSubTable_info *st, uchar *ebdtL, FILE *outfp, int glyphId, metricinfo bbox){
    metricinfo glyph;
    int i;
    uchar *p;

    /*
     * reading the header of indexSubTable
     *   get indexFormat (order of glyphs in EBDT)
     *   get imageFormat (type of image format of glyphs in EBDT)
     *   get offset from EBDTtop to locationOfGlyphsData
     */
    glyph.imageFormat = see_indexSubHeader(st);
    p = st->subtableL + 8; //8 = size of indexSubHeader
    glyph.ebdtL = ebdtL;

    /*
     * reading the body of indexSubTable
     */
    switch (st->indexFormat){
    case 1: // proportional with 4byte offset
        {
            ulong nextoff, curoff; //next glyph's offset, current offset
            // to know last glyph's length, +1 is needed
            p += (glyphId - st->first) * 4;
            readLong(curoff);
            readLong(nextoff);
            if(nextoff-curoff>0){
                glyph.off =  st->off + curoff;
                glyph.id = glyphId;
                putglyph(&glyph, nextoff-curoff, outfp, bbox);
            }
        }
        break;
    case 2: //monospaced with close codes
        {
            ulong imageSize;
            readLong(imageSize);
            see_glyphMetrics(p, &glyph, 1);
            glyph.off = st->off + imageSize * (glyphId - st->first);
            glyph.id = glyphId;
            putglyph(&glyph, imageSize, outfp, bbox);
        }
        break;
    case 3: //proportional with 2byte offset
        {
            ushort nextoff, curoff;
            readShort(curoff);
            readShort(nextoff);
            if(nextoff-curoff>0){
                glyph.off =  st->off + curoff;
                glyph.id = glyphId;
                putglyph(&glyph, nextoff-curoff, outfp, bbox);
            }
        }
        break;
    case 4: // proportional with sparse codes
        {
            ulong nGlyphs;
            ushort nextoff, curoff=0xFFFF, nextid, curid=0xFFFF; //current ID, nextID
            readLong(nGlyphs);
            for(i=0; (ulong)i<nGlyphs+1; i++){
                readShort(nextid);
                readShort(nextoff);
                if(curid==glyphId && nextoff-curoff>0){
                    glyph.off = st->off + curoff;
                    glyph.id = curid;
                    putglyph(&glyph, nextoff-curoff, outfp, bbox);
                    break;
                }
                curoff = nextoff;
                curid = nextid;
            }
        }
        break;
    case 5:     //monospaced with sparse codes
        {
            ulong imageSize, nGlyphs;
            ushort curid;
            readLong(imageSize);
            p = see_glyphMetrics(p, &glyph, 1);
            readLong(nGlyphs);
            for(i=0; (ulong)i<nGlyphs; i++){
                readShort(curid);
                if (curid == glyphId) {
                    glyph.id = curid;
                    glyph.off = st->off + imageSize * i;
                    putglyph(&glyph, imageSize, outfp, bbox);
                    break;
                }
            }
        }
        break;
    default:
        errexit("indexFormat %d is not supported.", st->indexFormat);
        break;
    }
}

/*
 * reading TrueTypefont header
 * in:  on-memory top of truetype font
 *      (for out) info of each table
 * out: nothing
 */
void getTableInfo(uchar *p, tableinfo *t){
    int i;
    ushort numTable;

    p += sizeof(ulong); //version

    readShort(numTable);//number of tables

    p += sizeof(ushort); //searchRange
    p += sizeof(ushort); //entrySelector
    p += sizeof(ushort); //rangeShift

    /*
     * assign info of table
     */
    for(i=0; i<numTable; i++){
        //allocate for a tableinfo (free() don't exist)
        if( (t->next=malloc(sizeof(tableinfo))) == NULL)
            errexit("malloc");
        t = t->next;

        //a name of table (tag)
        memset(t->tag, 0x00, 5); // 0 clear
        memcpy(t->tag, p, 4);
        p += sizeof(ulong);

        p += sizeof(ulong); //checksum (not use now)

        //offset
        readLong(t->offset); //number of tables

        p += sizeof(ulong); //length (not use now)

    }
    t->next = NULL;
}

/*
 * checking TrueTypefont or not
 * in:  on-memory location: top of truetype font
 * out: nothing
 *
 *  If errors, exit program.
 */
void validiateTTF(uchar *p){
    ulong version;
    
    readLong(version); //version
    if(version == 0x00010000){
        printf("  Microsoft TrueType/OpenType\n");
    }else if(version == 0x74746366){
        printf("  Microsoft TrueTypeCollection\n");
        errexit("TTC file is not supported yet.");
    }else if(version == 0x74727565){
        printf("  Apple TrueType\n");
    }else{
        errexit("This file is not a TrueTypeFont.");
    }
}

/*
 * reading bitmapSizeTable(defines info of strike) in EBLC
 * in:  on-memory location: top of a bitmapSizeTable
 *      (for out) number of indexSubTableArray-elements
 *      (for out) on-memory location: top of indexSubTableArray
 *      (for out) info of BoundingBox
 * out: nothing
 */
void see_bitmapSizeTable(uchar *p, int *numElem, ulong *arrayOffset, metricinfo *bbox){

    //offset from EBLCtop to indexSubTableArray
    readLong(*arrayOffset);
    //indexSubTable size (byte) (not use)
    p += sizeof(ulong);
    //number of indexSubTableArray-elements (= number of indexSubTables)
    readLong(*numElem);
    //colorRef (always 0)
    p += sizeof(ulong);
    //sbitLineMetrics: hori
    //info of metric about all glyphs in a strike
    p = see_sbitLineMetrics(p, bbox, 1); // 1 means direction==horizontal
    //sbitLineMetrics: vert  (not use)
    p = see_sbitLineMetrics(p, NULL, 0); // 0 means direction==vertical
    //start glyphIndex for this size
    p += sizeof(ushort);
    //end glyphIndex for this size
    p += sizeof(ushort);
    //ppemX (the strike's boundingbox width (pixel))
    bbox->ppem = *p++;
    //ppemY
    p += sizeof(uchar);
    //bitDepth (always 1)
    p += sizeof(uchar);
    //flags (1=horizontal, 2=vertical)
    p += sizeof(uchar);
}

/*
 * reading info of metric
 * in:  on-memory location:  top of metric-info
 *      directionHori: 1=horizontal,  0=vertical
 * out: on-memory location:  just after metric-info
 */
uchar *see_sbitLineMetrics(uchar *p, metricinfo *bbox, int directionHori){
    char widthMax, maxBeforeBL, minAfterBL, minOriginSB;

    //ascender: distance from baseline to upper-line (px)
    p++;
    //descender: distance from baseline to bottom-line (px)
    p++;
    //widthMax
    widthMax = *p++;
    //caratSlopeNumerator
    p++;
    //caratSlopeDenominator
    p++;
    //caratOffset
    p++;
    //minOriginSB
    minOriginSB = *p++;
    //minAdvanceSB
    p++;
    //maxBeforeBL
    maxBeforeBL = *p++;
    //minAfterBL
    minAfterBL = *p++;
    //pad1
    p++;
    //pad2
    p++;
    //if direction==hori, assign metric-info
    //elseif direction==vert, don't assign
    if(directionHori){
        bbox->width = widthMax;
        bbox->height = maxBeforeBL - minAfterBL;
        bbox->offsetx = minOriginSB;
        bbox->offsety = minAfterBL;
    }
    return p;
}

/*
 * reading an indexSubTableArray
 * in:  on-memery location: top of indexSubTableArray-elements
 *      on-memery location: top of indexSubTableArray
 *      (for out) info of indexSubTable
 * out: nothing
 */
void see_indexSubTableArray(uchar *elemL, uchar *arrayL, indexSubTable_info *st){
    uchar *p = elemL;
    ulong loc;

    //firstGlyphIndex
    readShort(st->first);

    //lastGlyphIndex
    readShort(st->last);

    //location of indexSubTable
    readLong(loc);
    st->subtableL = arrayL + loc;
}

/*
 * raeding header of an indexSubTable
 * in:  (for in and out) info of indexSubTable
 * out: imageFormat
 */
int see_indexSubHeader(indexSubTable_info *st){
    char *p = st->subtableL;
    int imageFormat;

    //indexFormat
    readShort(st->indexFormat);

    //imageFormat
    readShort(imageFormat);

    //imageDataOffset
    readLong(st->off);

    return imageFormat;
}

/*
 * write blank font data
 * in:  writing-file pointer
 *      bounding box data
 * out: nothing
 */
void putNoGlyph(FILE *outfp, metricinfo bbox){
    ushort c[] = {bbox.width, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
    fwrite(c, sizeof(ushort), 16, outfp);
}

/*
 * read one glyph, and put to file
 * in:  info of a glyph (type of EBDT-imageFormat,... )
 *      size of bitmapdata
 *      writing-file pointer
 * out: nothing
 */
void putglyph(metricinfo *glyph, int size, FILE *outfp, metricinfo bbox){
    uchar *glyphL = glyph->ebdtL + glyph->off;
    uchar *p = glyphL;
    char s[GLYPHBUFSIZE];
    memset(s, 0x00, GLYPHBUFSIZE);

    switch (glyph->imageFormat){
    case 1:
        //EBDT format 1: byte-aligned, small-metric
        p = see_glyphMetrics(p, glyph, 0);
        setGlyphHead(glyph, outfp);
        setGlyphBody_byte(p, glyphL+size, glyph, outfp, bbox);
        break;
    case 2:
        //EBDT format 2: bit-aligned, small-metric
        p = see_glyphMetrics(p, glyph, 0);
        setGlyphHead(glyph, outfp);
        setGlyphBody_bit(p, glyphL+size, glyph, outfp, bbox);
        break;
    case 5:
        //EBDT format 5: bit-aligned, EBLC-metric
        setGlyphHead(glyph, outfp);
        setGlyphBody_bit(p, glyphL+size, glyph, outfp, bbox);
        break;
    case 6:
        //EBDT format 6: byte-aligned, big-metric
        p = see_glyphMetrics(p, glyph, 1);
        setGlyphHead(glyph, outfp);
        setGlyphBody_byte(p, glyphL+size, glyph, outfp, bbox);
        break;
    case 7:
        //EBDT format 7: bit-aligned, big-metric
        p = see_glyphMetrics(p, glyph, 1);
        setGlyphHead(glyph, outfp);
        setGlyphBody_bit(p, glyphL+size, glyph, outfp, bbox);
        break;
    default:
        errexit("imageFormat %d is not supported.", glyph->imageFormat);
        break;
    }
}

/*
 * set header of a BDF glyph to string
 * in:  info of glyph(glyph's boudingbox)
 *      (for out) string pointer
 * out: nothing
 */
void setGlyphHead(metricinfo *g, FILE *outfp){
    /*
    printf(
            "STARTCHAR glyphID:%04x\n"
            "ENCODING %d\n"
            "DWIDTH %d\n"
            "BBX %d %d %d %d\n"
            "BITMAP\n",
            g->id, -1, g->advance,
            g->width, g->height, g->offsetx, g->offsety
            );
            */
    char c = 0;
    fwrite(&g->advance, 1, 1, outfp);
    fwrite(&c, 1, 1, outfp);
}

/*
 * set bitmap of a BDF glyph: byte-aligned
 * in:  on-memory location: beginning of bitmapdata in EBDT
 *      on-memory location: end of bitmapdata in EBDT
 *      info of glyph (width)
 *      (for out) string pointer
 * out: nothing
 */
void setGlyphBody_byte(uchar *p, const uchar *end, metricinfo *g, FILE *outfp, metricinfo bbox){
    int cnt = 0; //counter: print newline or not
    char str[GLYPHBUFSIZE];
    memset(str, 0x00, GLYPHBUFSIZE);

    while(p < end){
        //read one byte, change to 2 characters of hexadecimal
        printf("%x%x", (*p)>>4, (*p)&0x0f);
        //not print '\n' if cnt < (glyph.width-(int)1)/8
        if( (g->width-(int)1)/8 <= cnt){
            printf("\n");
            cnt = 0;
        }else{
            cnt++;
        }
        p++;
    }
}

/*
 * set bitmap of a BDF glyph: bit-aligned
 * in:  on-memory location: beginning of bitmapdata in EBDT
 *      on-memory location: end of bitmapdata in EBDT
 *      info of glyph (width, height)
 *      (for out) string pointer
 * out: nothing
 */
void setGlyphBody_bit(uchar *p, const uchar *end, metricinfo *g, FILE *outfp, metricinfo bbox){
    uchar bin;
    ushort val;
    int row = bbox.height;
    ushort paddata = 0;
    int padRow = 15;
    int i, j, bit, pad;

    g->offsety -= bbox.offsety;
    row -= g->height;
    row -= g->offsety;
    while(row--) {
        fwrite(&paddata, 2, 1, outfp);
        padRow--;
    }
    pad = ((g->width+7)/8*8) - g->width;
    for (i = 0, bit = 0, bin = 0; i < g->height; i++) {
        val = 0;
        if (g->width > 8) { 
            for (j = 0; j < 8; j++) {
                bit--;
                if (bit < 0) {
                    bit = 7;
                    bin = *p++;
                }
                val <<= 1;
                if (bin & 0x80) {
                    val |= 1;
                }
                bin <<= 1;
            }
        }
        for (j = 0; j < 8 - pad; j++) {
            bit--;
            if (bit < 0) {
                bit = 7;
                bin = *p++;
            }
            val <<= 1;
            if (bin & 0x80) {
                val |= 1;
            }
            bin <<= 1;
        }
        if (g->width <= 8) { 
            val <<= 8;
        }
        val <<= pad;
        val >>= g->offsetx;
        fwrite(&val, 2, 1, outfp);
        padRow--;
    }
    while(padRow--) {
        fwrite(&paddata, 2, 1, outfp);
    }
}

/*
 * display error messages, and exit this program
 * in:  variable arguments to print
 * out: nothing
 */
void errexit(char *fmt, ...){
        va_list ap;
        va_start(ap, fmt);
        //        fprintf(stderr, "%s: ", PROGNAME);
        fprintf(stderr, "  Error: ");
        vfprintf(stderr, fmt, ap);
        fprintf(stderr, "\n");
        va_end(ap);

        exit(EXIT_FAILURE);
}

/*
 * reading BigGlyphMetrics/SmallGlyphMetrics in EBLC
 * in:  on-memory location: top of Big/Small GlyphMetrics
 *      (for out) info of glyph
 *      Big or Small (1==big, 0==small)
 * out: on-memory location: just after Big/Small GlyphMetrics
 */
uchar *see_glyphMetrics(uchar *p, metricinfo *met, int big){

    //height
    met->height = *p++;
    //width
    met->width = *p++;
    //horiBearingX
    met->offsetx = (char)*p++;
    //horiBearingY
    met->offsety = (char)*p++ -  met->height;
    //horiAdvance
    met->advance = *p++;
    if(big){
        //vertBearingX
        p++;
        //vertBearingY
        p++;
        //vertAdvance
        p++;
    }
    return p;
}

uchar *mread(uchar *p, int size, char *s){
    int i;
    memset(s, 0x00, BUFSIZE); //zero clear
    sprintf(s, "0x"); //add '0x' for make indentified as hexadecimal

    for(i=0; i<size; i++){
        sprintf(s+(i*2)+2, "%02x", *p);
        p++;
    }
    return p;
}
/*
 * reading 'name' table
 *   in:  on-memory location: top of 'name'
 *        (for out) strings of copyright
 *        (for out) strings of fontname
 *   out: nothing
 */
void see_name(uchar *nameL, char *copyright, char *fontname){
    char *p = nameL;
    ushort numRecord, storageoff;
    int i, j;
    //make copyright/fontname string have priority(level).
    str_info sright[LEVELCOPYRIGHTSTR];
    str_info sname[LEVELFONTNAMESTR];
    for(i=0; i<LEVELCOPYRIGHTSTR; i++)
        sright[i].slen = 0;
    for(i=0; i<LEVELFONTNAMESTR; i++)
        sname[i].slen = 0;
    /*
     * reading header of 'name' table
     */
    p += sizeof(ushort); //format selector
    //number of name records
    readShort(numRecord);
    //offset from 'name'top to string storage
    readShort(storageoff);
    /*
     * reading nameRecords
     */
    for(i=0; i<numRecord; i++){
        ushort platformid, specificid, langid, nameid, slen, soff;
        //PlatformID
        readShort(platformid);
        //Platform-specificID
        readShort(specificid);
        //LanguageID
        readShort(langid);
        //NameID
        readShort(nameid);
        //String Length
        readShort(slen);
        //string storage offset from start of storage area
        readShort(soff);
        if(platformid==1 && langid==0){
            //Macintosh, ???(0==Roman), English
            if(nameid==0){
                //Copyright
                //  priority 0 (highest)
                sright[0].strL = nameL + storageoff + soff;
                sright[0].slen = slen;
            }else if(nameid==6){
                //fontname (postscript)
                sname[0].strL = nameL + storageoff + soff;
                sname[0].slen = slen;
            }else if(nameid==4){
                //fontname (full fontname)
                sname[1].strL = nameL + storageoff + soff;
                sname[1].slen = slen;
            }
        }else if(platformid==1){
            //Macintosh, ???(0==Roman), ???(0x0411=Japanese)
            if(nameid==0){
                sright[1].strL = nameL + storageoff + soff;
                sright[1].slen = slen;
            }else if(nameid==6){
                sname[2].strL = nameL + storageoff + soff;
                sname[2].slen = slen;
            }else if(nameid==4){
                sname[3].strL = nameL + storageoff + soff;
                sname[3].slen = slen;
            }
        }else if(platformid==3 && langid==0x0409){
            //Microsoft, ???, English
            if(nameid==0){
                sright[2].strL = nameL + storageoff + soff;
                sright[2].slen = slen;
            }else if(nameid==6){
                sname[4].strL = nameL + storageoff + soff;
                sname[4].slen = slen;
            }else if(nameid==4){
                sname[5].strL = nameL + storageoff + soff;
                sname[5].slen = slen;
            }
        }else if(platformid==3){
            //Microsoft, ???, ???(0x0411==Japanese)
            if(nameid==0){
                sright[3].strL = nameL + storageoff + soff;
                sright[3].slen = slen;
            }else if(nameid==6){
                sname[6].strL = nameL + storageoff + soff;
                sname[6].slen = slen;
            }else if(nameid==4){
                sname[7].strL = nameL + storageoff + soff;
                sname[7].slen = slen;
            }
        }       
        /*
         * assign copyright string
         */
        for(j=0; j<LEVELCOPYRIGHTSTR; j++){
            if(sright[j].slen && strcmp(copyright,STRUNKNOWN)==0)
                copystr(copyright, sright[j].strL, sright[j].slen, 0);
        }
        /*
         * assign fontname string
         */
        for(j=0; j<LEVELFONTNAMESTR; j++){
            if(sname[j].slen && strcmp(fontname,STRUNKNOWN)==0)
                copystr(fontname, sname[j].strL, sname[j].slen, 1);
        }
    }
}

/*
 * assign string with convert non-ascii character to ascii-character
 * in:  destination pointer
 *      on-memory location: top of string
 *      length of string
 *      this string is for filename or not? (1==yes, 0==no)
 * out: nothing
 */
void copystr(char *dst, uchar *src, ushort len, int forFilename){
    int i;

    memset(dst, 0x00, MAXSTRINGINBDF);
    for(i=0; i<len; i++){
        uchar s = *(src+i);

        if(s==0xa9){
            memcpy(dst, "(c)", 3);
            dst+=3;
        }else if(s==0xae){
            memcpy(dst, "(r)", 3);
            dst+=3;
        }else if(forFilename && (s=='?' || s=='*' || s=='/' || s=='\\' ||
                                 s==':' || s==';' || s=='|' || s=='<' ||
                                 s=='>' || s==0x22)){
            ; //ignore (unable to use as filename in Microsoft Windows)
        }else if(s<0x20 || s>0x7e){
            ; //ignore (probably, Unicode first byte)
        }else{
            *dst = s;
            dst++;
        }
    }
    *dst='\0';
}
//end of file
