using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using System.Resources;
using System.Text.RegularExpressions;
using System.Reflection;

namespace Oratorio
{
    public static class ProgramUtility
    {
		public static void ShowError(string sMessage)
		{
			MessageBox.Show(sMessage, Application.ProductName);
		}
		public static void ShowError(Exception ex)
		{
			string message = ex.Message;
			message += "\n----------------------------------------\n";
			if (ex.InnerException != null)
			{
				message += ex.InnerException.Message;
				message += "\n----------------------------------------\n";
			}
			message += ex.ToString();
			message += "\n----------------------------------------\n";
            WriteLine(message);
			ShowError(message);
		}

        public static void ShowError(Form form, string message,  Exception exception)
        {
            if (message == null)
            {
                message = exception.Message;
            }
            else if (exception != null)
            {
                message += "\n----------------------------------------\n";
                message += exception.Message;
            }
            if (exception != null)
            {
                message += "\n----------------------------------------\n";
                if (exception.InnerException != null)
                {
                    message += exception.InnerException.Message;
                    message += "\n----------------------------------------\n";
                }
                message += exception.ToString();
                message += "\n----------------------------------------\n";
            }
            WriteLine(message);
            MessageBox.Show(form, message, Application.ProductName, MessageBoxButtons.OK, MessageBoxIcon.Error);
        }

        public static void WriteLine(object message)
        {
            if (message == null)
            {
                message = "";
                return;
            }
            System.Diagnostics.Debug.WriteLine(DateTime.Now.ToString() + "; " + message);
        }
        public static void EnableStripMenuItem(ToolStrip menuStrip, ToolStripItem item, bool flag)
        {
            List<ToolStripItemCollection> itemCollList = new List<ToolStripItemCollection>();
            foreach (ToolStripItem item2 in menuStrip.Items)
            {
                if (item2 == item)
                {
                    item2.Enabled = flag;
                    return;
                }
                if (item2 is ToolStripMenuItem)
                {
                    ToolStripMenuItem item3 = (ToolStripMenuItem)item2;
                    itemCollList.Add(item3.DropDownItems);
                }
            }

            foreach (ToolStripItemCollection itemCol in itemCollList)
            {
                foreach (ToolStripItem item2 in itemCol)
                {
                    if (item2 == item)
                    {
                        item2.Enabled = flag;
                        return;
                    }
                }
            }
        }

		public enum CallEditActionModes
		{
			COPY,
			CUT,
			PASTE,
			SELECT_ALL,
            DELETE
		}
		public static void CallEditAction(Control target, CallEditActionModes mode)
		{
			if (target == null)
			{
				return;
			}
			if (target is TextBoxBase)
			{
				TextBoxBase textBox = (TextBoxBase)target;
				switch(mode)
				{
					case CallEditActionModes.COPY:
						textBox.Copy();
						break;
					case CallEditActionModes.CUT:
						textBox.Cut();
						break;
					case CallEditActionModes.PASTE:
						textBox.Paste();
						break;
                    case CallEditActionModes.SELECT_ALL:
                        textBox.SelectAll();
                        break;
                    case CallEditActionModes.DELETE:
                        {
                            int selectionStart = textBox.SelectionStart;
                            if (textBox.SelectionLength > 0)
                            {
                                textBox.Text = textBox.Text.Substring(0, selectionStart) + textBox.Text.Substring(selectionStart + textBox.SelectionLength);
                            }
                            else if (selectionStart < textBox.Text.Length)
                            {
                                textBox.Text = textBox.Text.Substring(0, selectionStart) + textBox.Text.Substring(selectionStart + 1);
                            }
                            textBox.SelectionStart = selectionStart;
                            break;
                        }
                }
			}
			if (target is DataGridView)
            {
                DataGridView view = (DataGridView)target;
                switch (mode)
                {
                    case CallEditActionModes.COPY:
                        string text = GetSelectedDataGridViewText(view);
                        if (text.Length > 0)
                        {
                            Clipboard.SetText(text);
                        }
                        else
                        {
                            Clipboard.Clear();
                        }
                        break;
                }
            }
		}

        public class DataGdidViewCellComparer : IComparer<DataGridViewCell>
        {
            public DataGdidViewCellComparer()
            {
            }

            public int Compare(DataGridViewCell x, DataGridViewCell y)
            {
                if (x.RowIndex != y.RowIndex)
                {
                    return ((x.RowIndex > y.RowIndex) ? 1 : (((x.RowIndex < y.RowIndex) ? -1 : 0)));
                }
                else if (x.ColumnIndex != y.ColumnIndex)
                {
                    return ((x.ColumnIndex > y.ColumnIndex) ? 1 : (((x.ColumnIndex < y.ColumnIndex) ? -1 : 0)));
                }
                return 0;
            }
        }

        public static string GetSelectedDataGridViewText(DataGridView gridView)
        {
            StringBuilder result = new StringBuilder("");
            if (gridView.SelectedRows.Count > 0)
            {
                List<int> rowIndexList = new List<int>();
                foreach (DataGridViewRow row in gridView.SelectedRows)
                {
                    if (row.IsNewRow)
                    {
                        continue;
                    }
                    rowIndexList.Add(row.Index);
                }
                rowIndexList.Sort();
                foreach (int rowNum in rowIndexList)
                {
                    DataGridViewRow row = gridView.Rows[rowNum];
                    bool firstCell = true;
                    foreach (DataGridViewCell cell in row.Cells)
                    {
                        if (firstCell)
                        {
                            firstCell = false;
                        }
                        else
                        {
                            result.Append("\t");
                        }
                        object value = cell.Value;
                        string text = (value == null ? "" : value.ToString());
                        result.Append(text);
                    }
                    result.Append("\r\n");
                }
            }
            else if (gridView.SelectedCells.Count > 0)
            {
                List<DataGridViewCell> cellList = new List<DataGridViewCell>();
                foreach (DataGridViewCell cell in gridView.SelectedCells)
                {
                    cellList.Add(cell);
                }
                cellList.Sort(new DataGdidViewCellComparer());
                int beforeRowIndex = -1;
                foreach (DataGridViewCell cell in cellList)
                {
                    if (cell.RowIndex != beforeRowIndex)
                    {
                        if (beforeRowIndex != -1)
                        {
                            result.Append("\r\n");
                        }
                        beforeRowIndex = cell.RowIndex;
                    }
                    else
                    {
                        result.Append("\t");
                    }
                    object value = cell.Value;
                    string text = (value == null ? "" : value.ToString());
                    result.Append(text);
                }
                result.Append("\r\n");
            }
            return result.ToString();
        }

        public static Control FindFocus(Form form)
		{
			Control activeControl = form.ActiveControl;
			List<Control> controls = new List<Control>();
			controls.Add(activeControl);
			while (controls.Count > 0)
			{
				Control control = controls[0];
				controls.RemoveAt(0);
				if (control.Focused)
				{
					ProgramUtility.WriteLine("FindForcus=" + control.GetType().Name);
					return control;
				}
				else if (control.ContainsFocus)
				{
					foreach (Control temp in control.Controls)
					{
						if (temp.ContainsFocus)
						{
							controls.Add(temp);
						}
					}
				}
			}
			return null;
		}

        public static void LoadControlStringsFromResource(Control.ControlCollection controls, string resourceName)
        {
            ResourceManager resourceManager = new ResourceManager(resourceName, typeof(Program).Assembly);

            foreach (Control item in controls)
            {
                string menuStr = resourceManager.GetString(item.Name);
                if (menuStr == null)
                {
                    continue;
                }
                Match altMatchNew = Regex.Match(menuStr, "\\(\\&[A-Z]\\)");
                if (!altMatchNew.Success)
                {
                    Match altMatchOld = Regex.Match(item.Text, "\\(\\&[A-Z]\\)");
                    if (altMatchOld.Success)
                    {
                        menuStr += item.Text.Substring(altMatchOld.Index);
                    }
                }
                item.Text = menuStr;
            }
        }

        public static void LoadMenuStringsFromResource(ToolStripItemCollection items, string resourceName)
        {
            ResourceManager resourceManager = new ResourceManager(resourceName, typeof(Program).Assembly);

            List<ToolStripItem> menuItemList = new List<ToolStripItem>();
            List<ToolStripItemCollection> itemColList = new List<ToolStripItemCollection>();
            itemColList.Add(items);
            while (itemColList.Count > 0)
            {
                ToolStripItemCollection itemCol = itemColList[0];
                itemColList.RemoveAt(0);
                foreach (ToolStripItem item in itemCol)
                {
                    menuItemList.Add(item);
                    if (item is ToolStripMenuItem)
                    {
                        itemColList.Add(((ToolStripMenuItem)item).DropDownItems);
                    }
                }
            }

            foreach (ToolStripItem item in menuItemList)
            {
                string menuStr = resourceManager.GetString(item.Name);
                if (menuStr == null)
                {
                    continue;
                }
                Match altMatchNew = Regex.Match(menuStr, "\\(\\&[A-Z]\\)");
				if (!altMatchNew.Success)
				{
                    Match altMatchOld = Regex.Match(item.Text, "\\(\\&[A-Z]\\)");
					if (altMatchOld.Success)
					{
                        menuStr += item.Text.Substring(altMatchOld.Index);
                    }
                }
                item.Text = menuStr;
            }
        }

        public static string LoadStringsFromResource(Type type, string itemName, string defaultValue)
        {
            ResourceManager resourceManager = new ResourceManager(type.FullName, type.Assembly);
            string value = resourceManager.GetString(itemName);
            if (value == null)
            {
                return defaultValue;
            }
            return value;
        }
        public static string GetAssemblyAttribute(Type attrType)
		{
			object[] attributes = Assembly.GetExecutingAssembly().GetCustomAttributes(attrType, false);
			if (attributes.Length == 0)
			{
				return null;
			}
			AssemblyTitleAttribute titleAttribute = (AssemblyTitleAttribute)attributes[0];
			return titleAttribute.Title;
		}

    }
}
