using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using System.Data.Common;
using System.IO;

namespace Oratorio
{
	public partial class ObjectTreeControl : UserControl
	{
		private DatabaseContext _database;

		public DatabaseContext Database
		{
			get { return _database; }
			set { _database = value; }
		}
	

		public TreeView TreeView
		{
			get { return this._treeView; }
		}
	
		public ObjectTreeControl()
		{
			InitializeComponent();
		}

        public MainForm MainForm
        {
            get { return (MainForm)Parent; }
        }

        private void ObjectTreeControl_Load(object sender, EventArgs e)
        {
            TreeView treeView = _treeView;
            treeView.ImageList = new ImageList();
            treeView.ImageList.Images.Add(Properties.Resources.ObjectUnknown);
            treeView.ImageList.Images.Add(Properties.Resources.ObjectDatabase);
            treeView.ImageList.Images.Add(Properties.Resources.ObjectFolder);
            treeView.ImageList.Images.Add(Properties.Resources.ObjectFolder);
            treeView.ImageList.Images.Add(Properties.Resources.ObjectFolder);
            treeView.ImageList.Images.Add(Properties.Resources.ObjectFolder);
            treeView.ImageList.Images.Add(Properties.Resources.ObjectFolder);

            ProgramUtility.LoadMenuStringsFromResource(_contextMenuStrip.Items, GetType().FullName);
        }

        public void RefreshNode()
        {
            TreeView treeView = this.TreeView;

            MainForm.OutputScriptLogLine("-- #Update object tree...", Color.Blue);
            treeView.Nodes.Clear();

            TreeNode databaseNode = new TreeNode("Database [" + _database.CurrentCatalogName + "]");
            databaseNode.SelectedImageIndex = databaseNode.ImageIndex = (int)ObjectTreeData.NodeTypes.DATABASE;
            databaseNode.Tag = new ObjectTreeData(ObjectTreeData.NodeTypes.DATABASE);

            treeView.Nodes.Add(databaseNode);

            TreeNode currentTablesNode = new TreeNode("Tables");
            ObjectTreeData currentTablesData = new ObjectTreeData(ObjectTreeData.NodeTypes.TABLE_LIST);
            currentTablesData.Catalog = _database.CurrentCatalogName;
            currentTablesData.Schema = _database.CurrentSchemaName;
            currentTablesNode.Tag = currentTablesData;
            currentTablesNode.SelectedImageIndex = currentTablesNode.ImageIndex = (int)ObjectTreeData.NodeTypes.TABLE_LIST;
            currentTablesNode.Nodes.Add(ObjectTreeData.CreateReloadNode());

            databaseNode.Nodes.Add(currentTablesNode);

            TreeNode currentViewsNode = new TreeNode("Views");
            ObjectTreeData currentViewsData = new ObjectTreeData(ObjectTreeData.NodeTypes.VIEW_LIST);
            currentViewsData.Catalog = _database.CurrentCatalogName;
            currentViewsData.Schema = _database.CurrentSchemaName;
            currentViewsNode.Tag = currentViewsData;
            currentViewsNode.SelectedImageIndex = currentViewsNode.ImageIndex = (int)ObjectTreeData.NodeTypes.VIEW_LIST;
            currentViewsNode.Nodes.Add(ObjectTreeData.CreateReloadNode());

            databaseNode.Nodes.Add(currentViewsNode);

            TreeNode schemaListNode = new TreeNode("Schemas");
            ObjectTreeData schemaListData = new ObjectTreeData(ObjectTreeData.NodeTypes.SCHEMA_LIST);
            schemaListNode.Tag = schemaListData;
            schemaListNode.Nodes.Add(ObjectTreeData.CreateReloadNode());
            schemaListNode.SelectedImageIndex = schemaListNode.ImageIndex = (int)ObjectTreeData.NodeTypes.TABLE_LIST;
            databaseNode.Nodes.Add(schemaListNode);

            databaseNode.Expand();
            currentTablesNode.Expand();
            treeView.SelectedNode = databaseNode;

            MainForm.OutputScriptLogLine("-- #Update object tree success.", Color.Blue);
        }

        private void _treeView_BeforeExpand(object sender, TreeViewCancelEventArgs e)
		{
			TreeNode node = e.Node;
			ObjectTreeData data = (ObjectTreeData)node.Tag;
			if (data.Type == ObjectTreeData.NodeTypes.DATABASE)
			{
				return;
			}
            else if (data.Type == ObjectTreeData.NodeTypes.TABLE_LIST)
			{
				TreeNodeCollection targetNodes = node.Nodes;
				if (targetNodes.Count == 1 && ObjectTreeData.IsReloadNode(targetNodes[0]))
				{
					string sTargetCatalog = data.Catalog;
					string sTargetSchema = data.Schema;
					string[] aTableList = _database.GetTableList(sTargetCatalog, sTargetSchema);
					targetNodes.Clear();
					foreach (string sTableName in aTableList)
					{
						TreeNode tableNode = new TreeNode(sTableName);
						ObjectTreeData tableData = new ObjectTreeData(ObjectTreeData.NodeTypes.TABLE);
						tableData.Catalog = sTargetCatalog;
						tableData.Schema = data.Schema;
                        tableData.Name = sTableName;
						tableNode.Tag = tableData;
						targetNodes.Add(tableNode);
					}
				}
			}
            else if (data.Type == ObjectTreeData.NodeTypes.VIEW_LIST)
            {
                TreeNodeCollection targetNodes = node.Nodes;
                if (targetNodes.Count == 1 && ObjectTreeData.IsReloadNode(targetNodes[0]))
                {
                    string sTargetCatalog = data.Catalog;
                    string sTargetSchema = data.Schema;
                    string[] aTableList = _database.GetViewList(sTargetCatalog, sTargetSchema);
                    targetNodes.Clear();
                    foreach (string sTableName in aTableList)
                    {
                        TreeNode tableNode = new TreeNode(sTableName);
                        ObjectTreeData tableData = new ObjectTreeData(ObjectTreeData.NodeTypes.VIEW);
                        tableData.Catalog = sTargetCatalog;
                        tableData.Schema = data.Schema;
                        tableData.Name = sTableName;
                        tableNode.Tag = tableData;
                        targetNodes.Add(tableNode);
                    }
                }
            }
            else if (data.Type == ObjectTreeData.NodeTypes.SCHEMA_LIST)
			{
				TreeNodeCollection targetNodes = node.Nodes;
				if (targetNodes.Count == 1 && ObjectTreeData.IsReloadNode(targetNodes[0]))
				{
					string sTargetDatabase = _database.CurrentCatalogName;
					string[] aSchemaList = null;
					try
					{
						aSchemaList = _database.GetSchemaList();
					}
					catch (DatabaseContextException dce)
					{
						ProgramUtility.ShowError(dce);
						e.Cancel = true;
						return;
					}
					catch (NotSupportedException nse)
					{
						ProgramUtility.ShowError(nse);
						e.Cancel = true;
						return;
					}
					targetNodes.Clear();
					foreach (string schemaName in aSchemaList)
					{
						TreeNode schemaNode = new TreeNode(schemaName);
						ObjectTreeData tableData = new ObjectTreeData(ObjectTreeData.NodeTypes.SCHEMA);
						tableData.Catalog = _database.CurrentCatalogName;
						tableData.Schema = schemaName;
						schemaNode.Tag = tableData;
                        schemaNode.ImageIndex = (int)ObjectTreeData.NodeTypes.SCHEMA_LIST;
                        schemaNode.SelectedImageIndex = schemaNode.ImageIndex;
                        targetNodes.Add(schemaNode);

                        TreeNode tableListNode = new TreeNode("Tables");
						ObjectTreeData tableListData = new ObjectTreeData(ObjectTreeData.NodeTypes.TABLE_LIST);
						tableListData.Catalog = _database.CurrentCatalogName;
						tableListData.Schema = schemaName;
                        tableListNode.ImageIndex = (int)ObjectTreeData.NodeTypes.TABLE_LIST;
                        tableListNode.SelectedImageIndex = tableListNode.ImageIndex;
                        tableListNode.Tag = tableListData;
						tableListNode.Nodes.Add(ObjectTreeData.CreateReloadNode());
						schemaNode.Nodes.Add(tableListNode);

                        TreeNode viewListNode = new TreeNode("Views");
                        ObjectTreeData viewListData = new ObjectTreeData(ObjectTreeData.NodeTypes.VIEW_LIST);
                        viewListData.Catalog = _database.CurrentCatalogName;
                        viewListData.Schema = schemaName;
                        viewListNode.ImageIndex = (int)ObjectTreeData.NodeTypes.VIEW_LIST;
                        viewListNode.SelectedImageIndex = viewListNode.ImageIndex;
                        viewListNode.Tag = viewListData;
                        viewListNode.Nodes.Add(ObjectTreeData.CreateReloadNode());
                        schemaNode.Nodes.Add(viewListNode);
                    }
				}
			}
		}

		private void _treeView_NodeMouseDoubleClick(object sender, TreeNodeMouseClickEventArgs e)
		{
			MainForm mainForm = (MainForm)Parent;
			TreeNode node = e.Node;
			ObjectTreeData data = (ObjectTreeData)node.Tag;
            if (data.Type == ObjectTreeData.NodeTypes.TABLE || data.Type == ObjectTreeData.NodeTypes.VIEW)
			{
				string sTableName = node.Text;
				ScriptForm form = mainForm.CreateGridForm(data.Catalog, data.Schema, sTableName, false);
                if (form != null)
                {
                    form.GridCtrl.Focus();
                }
			}
		}

        private void _treeView_NodeMouseClick(object sender, TreeNodeMouseClickEventArgs e)
        {
            if (e.Button == MouseButtons.Right)
            {
                TreeNode node = e.Node;
                _treeView.SelectedNode = node;
                //ObjectTreeData data = (ObjectTreeData)node.Tag;
                //if (data.Type == ObjectTreeData.NodeTypes.TABLE)
                //{
                //    _contextMenuStrip.Show(_treeView, e.X, e.Y);
                //}
                ObjectTreeData data = GetNodeData(node);
                switch(data.Type)
                {
                    case ObjectTreeData.NodeTypes.TABLE:
                        _contextMenuStrip.Show(_treeView, e.X, e.Y);
                        break;
                    case ObjectTreeData.NodeTypes.VIEW:
                        _contextMenuStrip.Show(_treeView, e.X, e.Y);
                        break;
                }
            }
        }

        private void _openMenuItem_Click(object sender, EventArgs e)
        {
            TreeNode node = _treeView.SelectedNode;
            ObjectTreeData data = (ObjectTreeData)node.Tag;
            string sTableName = node.Text;
            MainForm mainForm = (MainForm)Parent;
            ScriptForm form = mainForm.CreateGridForm(data.Catalog, data.Schema, sTableName, false);
            if (form != null)
            {
                form.GridCtrl.Focus();
            }
        }

        private void _selectMenuItem_Click(object sender, EventArgs e)
        {
            TreeNode node = _treeView.SelectedNode;
            ObjectTreeData data = (ObjectTreeData)node.Tag;
            string script = "";
            try
            {
                script = _database.ScriptGenerator.GetSelectScript(data.Catalog, data.Schema, data.Name);
            }
            catch (DatabaseContextException de)
            {
                ProgramUtility.ShowError(de);
                return;
            }

            ScriptForm form = MainForm.CreateEditForm();
            form.EditControl.TextBox.Text = script;
            form.EditControl.TextBox.Focus();

        }

        private void _createSubMenu_Click(object sender, EventArgs e)
        {
            TreeNode node = _treeView.SelectedNode;
            ObjectTreeData data = (ObjectTreeData)node.Tag;
			string script = "";
			try
			{
				script = _database.ScriptGenerator.GetCreateTableScript(data.Catalog, data.Schema, data.Name);
			}
			catch (DatabaseContextException de)
			{
				ProgramUtility.ShowError(de);
				return;
			}

            ScriptForm form = MainForm.CreateEditForm();
			form.EditControl.TextBox.Text = script;
            form.EditControl.TextBox.Focus();
        }

        private void _copyNameMenuItem_Click(object sender, EventArgs e)
        {
            TreeNode node = _treeView.SelectedNode;
            Clipboard.SetText(node.Text);
        }

        private void _selectAsteriskMenuItem_Click(object sender, EventArgs e)
        {
            TreeNode node = _treeView.SelectedNode;
            ObjectTreeData data = (ObjectTreeData)node.Tag;
            string script = "";
            try
            {
                script = _database.ScriptGenerator.GetSelectAsteriskScript(data.Catalog, data.Schema, data.Name);
            }
            catch (DatabaseContextException de)
            {
                ProgramUtility.ShowError(de);
                return;
            }

            ScriptForm form = MainForm.CreateEditForm();
            form.EditControl.TextBox.Text = script;
            form.EditControl.TextBox.Focus();
        }

        private void _selectWhereMenuItem_Click(object sender, EventArgs e)
        {
            TreeNode node = _treeView.SelectedNode;
            ObjectTreeData data = (ObjectTreeData)node.Tag;
            string script = "";
            try
            {
                script = _database.ScriptGenerator.GetSelectWhereScript(data.Catalog, data.Schema, data.Name);
            }
            catch (DatabaseContextException de)
            {
                ProgramUtility.ShowError(de);
                return;
            }

            ScriptForm form = MainForm.CreateEditForm();
            form.EditControl.TextBox.Text = script;
            form.EditControl.TextBox.Focus();
        }

        private void _newRecordMenuItem_Click(object sender, EventArgs e)
        {
            TreeNode node = _treeView.SelectedNode;
            ObjectTreeData data = (ObjectTreeData)node.Tag;
            string sTableName = node.Text;
            MainForm mainForm = (MainForm)Parent;
            ScriptForm form = mainForm.CreateGridForm(data.Catalog, data.Schema, sTableName, true);
            if (form != null)
            {
                form.GridCtrl.Focus();
            }
        }

        private void _updateScriptMenuItem_Click(object sender, EventArgs e)
        {
            TreeNode node = _treeView.SelectedNode;
            ObjectTreeData data = (ObjectTreeData)node.Tag;
            string script = "";
            try
            {
                script = _database.ScriptGenerator.GetUpdateScript(data.Catalog, data.Schema, data.Name);
            }
            catch (DatabaseContextException de)
            {
                ProgramUtility.ShowError(de);
                return;
            }

            ScriptForm form = MainForm.CreateEditForm();
            form.EditControl.TextBox.Text = script;
            form.EditControl.TextBox.Focus();
        }

        private void _deleteScriptMenuItem_Click(object sender, EventArgs e)
        {
            TreeNode node = _treeView.SelectedNode;
            ObjectTreeData data = (ObjectTreeData)node.Tag;
            string script = "";
            try
            {
                script = _database.ScriptGenerator.GetDeleteScript(data.Catalog, data.Schema, data.Name);
            }
            catch (DatabaseContextException de)
            {
                ProgramUtility.ShowError(de);
                return;
            }

            ScriptForm form = MainForm.CreateEditForm();
            form.EditControl.TextBox.Text = script;
            form.EditControl.TextBox.Focus();
        }

        protected string GetDataReaderToCsv(IDataReader reader, string separator)
        {
            StringBuilder headBuffer = new StringBuilder("", 1024 * 1024);
            int columnCount = reader.FieldCount;
            for (int column = 0; column < columnCount; column++)
            {
                string name = reader.GetName(column);
                if (headBuffer.Length > 0)
                {
                    headBuffer.Append(separator);
                }
                if (name.IndexOf('\"') >= 0)
                {
                    name = name.Replace("\"", "\"\"");
                }
                headBuffer.Append('\"');
                headBuffer.Append('[');
                headBuffer.Append(name);
                headBuffer.Append(']');
                headBuffer.Append('\"');
            }
            headBuffer.Append("\r\n");
            StringBuilder buffer = new StringBuilder("", 1024 * 1024);
            while (reader.Read())
            {
                StringBuilder lineBuffer = new StringBuilder();
                for (int column = 0; column < columnCount; column++)
                {
                    object value = reader.GetValue(column);
                    if (column > 0)
                    {
                        lineBuffer.Append(separator);
                    }
                    if(value == null)
                    {
                        value = "";
                    }
                    string text = value.ToString();
                    if (text.IndexOf('\"') >= 0)
                    {
                        text = text.Replace("\"", "\"\"");
                    }
                    text = "\"" + text + "\"";
                    lineBuffer.Append(text);
                }
                lineBuffer.Append("\r\n");
                buffer.Append(lineBuffer);
            }
            String result = headBuffer.Append(buffer).ToString();
            return result;
        }

        protected string GetDataReaderToSql(IDataReader reader, string tableName)
        {
            StringBuilder scriptHead = new StringBuilder("");
            scriptHead.Append("INSERT INTO ");
            scriptHead.Append(tableName);
            scriptHead.Append("\r\n");
            scriptHead.Append("(");

            int columnCount = reader.FieldCount;
            for (int column = 0; column < columnCount; column++)
            {
                string name = reader.GetName(column);
                if (column > 0)
                {
                    scriptHead.Append(",");
                }
                scriptHead.Append("\r\n\t");
                scriptHead.Append(name);
            }
            scriptHead.Append("\r\n");
            scriptHead.Append(")\r\n");
            scriptHead.Append("VALUES\r\n");
            scriptHead.Append("(");

            StringBuilder buffer = new StringBuilder("", 1024 * 1024);
            while (reader.Read())
            {
                StringBuilder lineBuffer = new StringBuilder();
                for (int column = 0; column < columnCount; column++)
                {
                    object value = reader.GetValue(column);
                    if (column > 0)
                    {
                        lineBuffer.Append(",");
                    }
                    lineBuffer.Append("\r\n\t");
                    string text = value.ToString();
                    if (value == null || value == DBNull.Value)
                    {
                        text = "NULL";
                    }
                    else
                    {
                        if (text.IndexOf('\'') >= 0)
                        {
                            text = text.Replace("\'", "\'\'");
                        }
                        text = "\'" + text + "\'";
                    }
                    lineBuffer.Append(text);
                }
                lineBuffer.Append("\r\n");
                buffer.Append(scriptHead);
                buffer.Append(lineBuffer);
                buffer.Append(")\r\n;\r\n\r\n");
            }
            String result = buffer.ToString();
            return result;
        }


        private void _exportMenuItem_Click(object sender, EventArgs e)
        {
            TreeNode node = _treeView.SelectedNode;
            ObjectTreeData data = (ObjectTreeData)node.Tag;

            string tableName = data.Name;

            ExportForm form = new ExportForm();
            if (form.ShowDialog(this) != DialogResult.OK)
                return;

            string folder = form.Folder;
            string where = form.Filter;
            string path = Path.Combine(folder, tableName);
            ExportForm.Formats format = (ExportForm.Formats)form.Format;
            switch(format)
            {
                case ExportForm.Formats.CSV:
                    path += ".csv";
                    break;
                case ExportForm.Formats.TSV:
                    path += ".tsv";
                    break;
                case ExportForm.Formats.SQL:
                    path += ".sql";
                    break;
                default:
                    throw new NotSupportedException();
            }
            if (File.Exists(path))
            {
                if (MessageBox.Show(this, "Over write file?\r\n" + Path.GetFileName(path), Application.ProductName, MessageBoxButtons.OKCancel) != DialogResult.OK)
                {
                    return;
                }
            }

            where = where.Trim();
            if(where.Length == 0 || where.Equals("WHERE", StringComparison.OrdinalIgnoreCase))
            {
                where = "";
            }
            string query = "SELECT * FROM " + tableName;
            if(where.Length > 0)
            {
                query += " " + where;
            }
            DatabaseContext database = _database;
            try
            {
                Cursor.Current = Cursors.WaitCursor;

                string exportText = "";
                using (DbCommand command = _database.CreateCommand(query))
                {
                    using (DbDataReader reader = command.ExecuteReader())
                    {
                        command.Dispose();
                        switch (format)
                        {
                            case ExportForm.Formats.CSV:
                                exportText = GetDataReaderToCsv(reader, ",");
                                break;
                            case ExportForm.Formats.TSV:
                                exportText = GetDataReaderToCsv(reader, "\t");
                                break;
                            case ExportForm.Formats.SQL:
                                exportText = GetDataReaderToSql(reader, tableName);
                                break;
                            default:
                                throw new NotSupportedException();
                        }
                    }
                }
                Encoding encoding = Encoding.Default;
                byte[] byteBuffer = encoding.GetBytes(exportText);
                using (FileStream stream = new FileStream(path, FileMode.Create, FileAccess.Write))
                {
                    stream.Write(byteBuffer, 0, byteBuffer.Length);
                }
            }
            catch (DbException dbe)
            {
                ProgramUtility.ShowError(MainForm, "Export failed.", dbe);
            }
            catch (DatabaseContextException de)
            {
                ProgramUtility.ShowError(MainForm, "Export failed.", de);
            }
            catch (IOException ioe)
            {
                ProgramUtility.ShowError(MainForm, "Export failed.", ioe);
            }
            finally
            {
                Cursor.Current = Cursors.Default;
            }

        }

        ObjectTreeData GetNodeData(TreeNode node)
        {
            if (node == null)
            {
                System.Diagnostics.Debug.Assert(false);
                throw new ArgumentNullException("node");
            }
            ObjectTreeData data = node.Tag as ObjectTreeData;
            if (data == null)
            {
                System.Diagnostics.Debug.Assert(false);
                throw new ArgumentNullException("data");
            }
            return data;
        }

        private void _propertyMenuItem_Click(object sender, EventArgs e)
        {
            TreeNode node = _treeView.SelectedNode;
            ObjectTreeData data = GetNodeData(node);
            try
            {
                MainForm.CreateTableDefineForm(data.Catalog, data.Schema, data.Name);
            }
            catch (ProgramCancelException)
            {
            }
        }

        private void _insertScriptMenuItem_Click(object sender, EventArgs e)
        {
            ObjectTreeData data = GetNodeData(_treeView.SelectedNode);
            string script = "";
            try
            {
                script = _database.ScriptGenerator.GetInsertScript(data.Catalog, data.Schema, data.Name);
            }
            catch (DatabaseContextException de)
            {
                ProgramUtility.ShowError(de);
                return;
            }

            ScriptForm form = MainForm.CreateEditForm();
            form.EditControl.TextBox.Text = script;
        }

        private void _dropScriptMenuItem_Click(object sender, EventArgs e)
        {
            ObjectTreeData data = GetNodeData(_treeView.SelectedNode);
            string script = "";
            try
            {
                script = _database.ScriptGenerator.GetDropScript(data.Catalog, data.Schema, data.Name);
            }
            catch (DatabaseContextException de)
            {
                ProgramUtility.ShowError(de);
                return;
            }

            ScriptForm form = MainForm.CreateEditForm();
            form.EditControl.TextBox.Text = script;
        }

    }
}
