/*
 * Copyright 2003-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.velocity.tools.view.tools;

import org.apache.velocity.context.Context;
import org.apache.velocity.tools.generic.NullTool;

/**
 * Tool for working with <code>null</code> in Velocity templates.
 * It provides a method to set a VTL reference back to <code>null</code>,
 * automatically using the current context.
 * Also provides methods to check if a VTL reference is <code>null</code> or not.
 * <p>
 *   NOTE: <code>set</code> and <code>setNull</code> does not clear attributes in servlet objects.
 *   To remove attributes from <code>request</code>, <code>session</code> and <code>application</code>,
 *   use <code>$request.removeAttribute()</code>, <code>$session.removeAttribute()</code> and
 *   <code>$application.removeAttribute()</code> respectively.
 * </p>
 * <p><pre>
 * Example uses:
 *  #set($foo = "bar")
 *  $foo                              -> bar
 *  $null.isNull($foo)                -> false
 *  $null.isNotNull($foo)             -> true
 *
 *  $null.setNull("foo")
 *  $foo                              -> $foo (null)
 *  $null.isNull($foo)                -> true
 *  $null.isNotNull($foo)             -> false
 *
 *  $null.set($foo, "hoge")
 *  $foo                              -> hoge
 *  $null.set($foo, $null.null)
 *  $foo                              -> $foo (null)
 *
 *
 * Toolbox configuration:
 * &lt;tool&gt;
 *   &lt;key&gt;null&lt;/key&gt;
 *   &lt;scope&gt;request&lt;/scope&gt;
 *   &lt;class&gt;org.apache.velocity.tools.view.tools.ViewNullTool&lt;/class&gt;
 * &lt;/tool&gt;
 * </pre></p>
 *
 * <p>This tool is NOT meant to be used in either application or
 * session scopes of a servlet environment.</p>
 *
 * @author <a href="mailto:shinobu@ieee.org">Shinobu Kawai</a>
 * @version $Id: ViewNullTool.java,v 1.1 2007/07/19 04:26:59 itf_yshibayama Exp $
 */
public class ViewNullTool extends NullTool implements ViewTool
{

    /** The Context. */
    private Context context = null;

    /**
     * Default constructor.
     */
    public ViewNullTool()
    {
    }

    /**
     * Initializes this instance.
     *
     * @param object the current Context
     */
    public void init(Object object)
    {
        if (!(object instanceof Context))
        {
            throw new IllegalArgumentException("Tool can only be initialized with a Context");
        }

        this.context = (Context) object;
    }

    /**
     * Sets the given VTL reference back to <code>null</code> using the current context.
     * @param key the VTL reference to set back to <code>null</code>.
     */
    public void setNull(String key)
    {
        this.setNull(this.context, key);
    }

    /**
     * Sets the given VTL reference to the given value using the current context.
     * If the value is <code>null</code>,
     * the VTL reference is set to <code>null</code>.
     * @param key the VTL reference to set.
     * @param value the value to set the VTL reference to.
     */
    public void set(String key, Object value)
    {
        this.set(this.context, key, value);
    }

}

