/*
 * Copyright (c) 2003-2004 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: response_editor.c,v 1.12.2.1 2004/05/03 06:58:34 fuyu Exp $
 */

#include "config.h"

#include "ochusha.h"
#include "ochusha_private.h"

#include "ochusha_ui.h"
#include "response_editor.h"

#include "marshal.h"

#include "utils.h"

#include <gdk/gdkkeysyms.h>
#include <glib.h>
#include <gtk/gtk.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


static void response_editor_class_init(ResponseEditorClass *klass);
static void response_editor_init(ResponseEditor *editor);
static void response_editor_finalize(GObject *object);
static void response_editor_destroy(GtkObject *object);


GType
response_editor_get_type(void)
{
  static GType response_editor_type = 0;

  if (response_editor_type == 0)
    {
      static const GTypeInfo response_editor_info =
	{
	  sizeof(ResponseEditorClass),
	  NULL,	/* base_init */
	  NULL,	/* base_finalize */
	  (GClassInitFunc)response_editor_class_init,
	  NULL,	/* class_finalize */
	  NULL,	/* class_data */
	  sizeof(ResponseEditor),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)response_editor_init,
	};

      response_editor_type = g_type_register_static(GTK_TYPE_VBOX,
						    "ResponseEditor",
						    &response_editor_info, 0);
    }

  return response_editor_type;
}


enum {
  NOP_SIGNAL,
  LAST_SIGNAL
};


static GtkVBoxClass *parent_class;
static int response_editor_signals[LAST_SIGNAL] = { 0 };


static void
response_editor_class_init(ResponseEditorClass *klass)
{
  GObjectClass *o_class = (GObjectClass *)klass;
  GtkObjectClass *object_class = (GtkObjectClass *)klass;
#if 0
  GtkBindingSet *binding_set = gtk_binding_set_by_class(klass);
#endif

  parent_class = g_type_class_peek_parent(klass);

  /* GObject signals */
  o_class->finalize = response_editor_finalize;

  /* GtkObject signals */
  object_class->destroy = response_editor_destroy;

  response_editor_signals[NOP_SIGNAL] =
    g_signal_new("nop",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
		 G_STRUCT_OFFSET(ResponseEditorClass, nop),
		 NULL, NULL,
		 ochusha_marshal_VOID__VOID,
		 G_TYPE_NONE, 0);

#if 0
  gtk_binding_entry_add_signal(binding_set, GDK_Escape, 0,
			       "nop", 0);
#endif
}


static void
response_editor_init(ResponseEditor *editor)
{
  GtkWidget *hbox;
  GtkWidget *name_label;
  GtkWidget *name_combo;
  GtkWidget *mail_label;
  GtkWidget *mail_combo;
  GtkWidget *scrolled_window;
  GtkWidget *text_view;

  /* hboxν */
  hbox = gtk_hbox_new(FALSE, 5);
  editor->hbox = GTK_BOX(hbox);
  gtk_widget_show(hbox);
  gtk_box_pack_start(GTK_BOX(&editor->container), hbox, FALSE, FALSE, 0);

  name_label = gtk_label_new(_("Name:"));
  gtk_widget_show(name_label);
  gtk_box_pack_start(editor->hbox, name_label, FALSE, FALSE, 0);

#if GTK_MINOR_VERSION <= 2
  name_combo = gtk_combo_new();
  editor->name_combo = GTK_COMBO(name_combo);
#else
  name_combo = gtk_combo_box_entry_new_text();
  editor->name_combo = GTK_COMBO_BOX(name_combo);
#endif
  gtk_widget_show(name_combo);
  gtk_box_pack_start(editor->hbox, name_combo, TRUE, TRUE, 0);

  mail_label = gtk_label_new(_("E-mail:"));
  gtk_widget_show(mail_label);
  gtk_box_pack_start(editor->hbox, mail_label, FALSE, FALSE, 0);

#if GTK_MINOR_VERSION <= 2
  mail_combo = gtk_combo_new();
  editor->mail_combo = GTK_COMBO(mail_combo);
#else
  mail_combo = gtk_combo_box_entry_new_text();
  editor->mail_combo = GTK_COMBO_BOX(mail_combo);
#endif
  gtk_widget_show(mail_combo);
  gtk_box_pack_start(editor->hbox, mail_combo, TRUE, TRUE, 0);

  /* GtkTextEditorν */
  scrolled_window = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window),
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_widget_show(scrolled_window);
  gtk_box_pack_start(GTK_BOX(&editor->container),
		     scrolled_window, TRUE, TRUE, 0);

  text_view = gtk_text_view_new();
  editor->text_view = GTK_TEXT_VIEW(text_view);
  gtk_text_view_set_wrap_mode(editor->text_view, GTK_WRAP_CHAR);
  gtk_widget_show(text_view);
  gtk_container_add(GTK_CONTAINER(scrolled_window), text_view);
}


static void
response_editor_finalize(GObject *object)
{
  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


static void
response_editor_destroy(GtkObject *object)
{
  ResponseEditor *editor = RESPONSE_EDITOR(object);

  if (editor->name_list != NULL)
    {
      g_list_foreach(editor->name_list, (GFunc)g_free, NULL);
      g_list_free(editor->name_list);
      editor->name_list = NULL;
    }

  if (editor->mail_list != NULL)
    {
      g_list_foreach(editor->mail_list, (GFunc)g_free, NULL);
      g_list_free(editor->mail_list);
      editor->mail_list = NULL;
    }

  if (GTK_OBJECT_CLASS(parent_class)->destroy)
    (*GTK_OBJECT_CLASS(parent_class)->destroy)(object);
}


GtkWidget *
response_editor_new(void)
{
  ResponseEditor *editor = RESPONSE_EDITOR(g_object_new(RESPONSE_EDITOR_TYPE,
							NULL));
  return GTK_WIDGET(editor);
}


const gchar *
response_editor_get_name(ResponseEditor *editor)
{
  g_return_val_if_fail(IS_RESPONSE_EDITOR(editor), NULL);

#if GTK_MINOR_VERSION <= 2
  return gtk_entry_get_text(GTK_ENTRY(editor->name_combo->entry));
#else
  return gtk_entry_get_text(GTK_ENTRY(GTK_BIN(editor->name_combo)->child));
#endif
}


void
response_editor_set_name(ResponseEditor *editor, const gchar *name)
{
  g_return_if_fail(IS_RESPONSE_EDITOR(editor));

#if GTK_MINOR_VERSION <= 2
  gtk_entry_set_text(GTK_ENTRY(editor->name_combo->entry), name);
#else
  gtk_entry_set_text(GTK_ENTRY(GTK_BIN(editor->name_combo)->child), name);
#endif
}


void
response_editor_add_name(ResponseEditor *editor, const gchar *name)
{
  g_return_if_fail(IS_RESPONSE_EDITOR(editor));

  if (name == NULL)
    return;

  if (g_list_find_custom(editor->name_list, name, (GCompareFunc)strcmp)
      != NULL)
    return;

  editor->name_list = g_list_append(editor->name_list, g_strdup(name));

#if GTK_MINOR_VERSION <= 2
  gtk_combo_set_popdown_strings(editor->name_combo, editor->name_list);
#else
  gtk_combo_box_append_text(editor->name_combo, name);
#endif
}


const gchar *
response_editor_get_mail(ResponseEditor *editor)
{
  g_return_val_if_fail(IS_RESPONSE_EDITOR(editor), NULL);

#if GTK_MINOR_VERSION <= 2
  return gtk_entry_get_text(GTK_ENTRY(editor->mail_combo->entry));
#else
  return gtk_entry_get_text(GTK_ENTRY(GTK_BIN(editor->mail_combo)->child));
#endif
}


void
response_editor_set_mail(ResponseEditor *editor, const gchar *mail)
{
  g_return_if_fail(IS_RESPONSE_EDITOR(editor));

#if GTK_MINOR_VERSION <= 2
  gtk_entry_set_text(GTK_ENTRY(editor->mail_combo->entry), mail);
#else
  gtk_entry_set_text(GTK_ENTRY(GTK_BIN(editor->mail_combo)->child), mail);
#endif
}


void
response_editor_add_mail(ResponseEditor *editor, const gchar *mail)
{
  g_return_if_fail(IS_RESPONSE_EDITOR(editor));

  if (mail == NULL)
    return;

  if (g_list_find_custom(editor->mail_list, mail, (GCompareFunc)strcmp)
      != NULL)
    return;

  editor->mail_list = g_list_append(editor->mail_list, g_strdup(mail));

#if GTK_MINOR_VERSION <= 2
  gtk_combo_set_popdown_strings(editor->mail_combo, editor->mail_list);
#else
  gtk_combo_box_append_text(editor->mail_combo, mail);
#endif
}


void
response_editor_clear_response(ResponseEditor *editor)
{
  GtkTextIter start_iter;
  GtkTextIter end_iter;
  GtkTextBuffer *buffer;

  g_return_if_fail(IS_RESPONSE_EDITOR(editor));

  buffer = gtk_text_view_get_buffer(editor->text_view);
  gtk_text_buffer_get_start_iter(buffer, &start_iter);
  gtk_text_buffer_get_end_iter(buffer, &end_iter);
  gtk_text_buffer_delete(buffer, &start_iter, &end_iter);
}


gchar *
response_editor_get_response(ResponseEditor *editor)
{
  GtkTextIter start_iter;
  GtkTextIter end_iter;
  GtkTextBuffer *buffer;

  g_return_val_if_fail(IS_RESPONSE_EDITOR(editor), NULL);

  buffer = gtk_text_view_get_buffer(editor->text_view);
  gtk_text_buffer_get_start_iter(buffer, &start_iter);
  gtk_text_buffer_get_end_iter(buffer, &end_iter);
  return gtk_text_buffer_get_text(buffer, &start_iter, &end_iter, FALSE);
}


void
response_editor_append_response(ResponseEditor *editor, const gchar *response)
{
  GtkTextIter end_iter;
  GtkTextBuffer *buffer;

  g_return_if_fail(IS_RESPONSE_EDITOR(editor));
  g_return_if_fail(response != NULL);

  buffer = gtk_text_view_get_buffer(editor->text_view);
  gtk_text_buffer_get_end_iter(buffer, &end_iter);
  if (gtk_text_buffer_get_char_count(buffer) > 0)
    gtk_text_buffer_insert(buffer, &end_iter, "\n", 1);
  gtk_text_buffer_insert(buffer, &end_iter, response, -1);
}


#if 0
void
response_editor_set_response(ResponseEditor *editor, const gchar *response)
{
  g_return_if_fail(IS_RESPONSE_EDITOR(editor));
}
#endif
