/*
 * Copyright (c) 2003-2004 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: main.c,v 1.28 2004/02/27 22:19:59 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "htmlutils.h"
#include "worker.h"
#include "utils.h"

#define _OCHUSHA_MAIN_C_
#include "ochusha_ui.h"
#include "session.h"
#include "xsmp.h"

#if HAVE_ONIG_ONIGPOSIX_H
# include <onig/onigposix.h>
#else
# include <onigposix.h>
#endif

#include <glib.h>
#include <gtk/gtk.h>
#include <libxml/parser.h>

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>


#define INITIAL_NUM_THREADS		2
#define MAXIMUM_NUM_THREADS		32
#define MAXIMUM_NUM_MODEST_THREADS	4


#if DEBUG_GDK_LOCK
pthread_t volatile last_lock_owner = NULL;
volatile const char *last_file;
volatile int last_line;
#endif

#if USE_THREAD_FOR_MAIN || DEBUG_GDK_LOCK
pthread_t main_thread = NULL;
#endif

#if USE_THREAD_FOR_MAIN
static void
real_main(OchushaApplication *application)
{
  OCHU_THREADS_ENTER();

  ENTER_MAIN_EVENT_LOOP();
  gtk_main();

#if TRACE_MEMORY_USAGE
  if (application->top_level != NULL)
    gtk_widget_destroy(GTK_WIDGET(application->top_level));
#endif

#if 1
  MAIN_THREAD_EVENT_CALLBACK();	/* ä */
  OCHU_THREADS_LEAVE();
#else
  gdk_threads_leave();
#endif
}
#endif


int
main(int argc, char *argv[])
{
  int real_argc;
  char **real_argv;
  char *resource_file;
  char *session_id = NULL;
  OchushaApplication *application;
  int i;
  char buf[PATH_MAX];
  
#if TRACE_MEMORY_USAGE
  init_memory_trace();
#endif

#if TRACE_REFCOUNT
  init_refcount_trace();
#endif

#ifdef REG_POSIX_ENCODING_UTF8
  reg_set_encoding(REG_POSIX_ENCODING_UTF8);
#else	/* old style */
  reg_set_encoding(REG_ENCODING_UTF8);
#endif

  application = G_NEW0(OchushaApplication, 1);
  ochusha_config_prepare_home(&application->config);

#if 0
  {
    int fd = ochusha_config_open_file(&application->config, "log.err",
				      NULL, O_RDWR | O_CREAT);
    application->log_file = fdopen(fd, "a+");

    fprintf(application->log_file, "====== Ochusha invoked ======\n");
    for (i = 0; i < argc; i++)
      {
	fprintf(application->log_file, "argv[%d]=\"%s\"\n", i, argv[i]);
      }
    fflush(application->log_file);
  }
#else
  application->log_file = stderr;
#endif

  real_argc = argc;
  for (i = 1; i < argc; i++)
    {
      if (strcmp("--restart", argv[i]) == 0
	  || strncmp("--restart=", argv[i], 10) == 0)
	{
	  session_id = argv[i] + 9;
	  argv[i] = NULL;
	  real_argc--;
	  if (*session_id == '=')
	    session_id++;
	  else if (i + 1 < argc)
	    {
	      i++;
	      session_id = argv[i];
	      argv[i] = NULL;
	      real_argc--;
	    }
#if 0
	  fprintf(application->log_file, "restart session with ID:\"%s\"\n",
		  session_id);
	  fflush(application->log_file);
#endif
	}
      else if (strcmp("--session", argv[i]) == 0)
	{
	  argv[i] = NULL;
	  real_argc--;
#if 0
	  fprintf(application->log_file, "start session\"\n");
	  fflush(application->log_file);
#endif
	}
      else if (strcmp("--discard", argv[i]) == 0
	       || strncmp("--discard=", argv[i], 10) == 0)
	{
	  session_id = argv[i] + 9;
	  argv[i] = NULL;
	  real_argc--;
	  if (*session_id == '=')
	    session_id++;
	  else if (i + 1 < argc)
	    {
	      i++;
	      session_id = argv[i];
	      argv[i] = NULL;
	      real_argc--;
	    }
#if 0
	  fprintf(application->log_file, "discard session ID:\"%s\"\n",
		  session_id);
	  fflush(application->log_file);
#endif

	  ochusha_session_discard_states(application, session_id);
	  return 0;
	}
    }

  /* Remove session related arguments */
  real_argv = g_malloc(sizeof(char *) * real_argc);	/* ĤϳȤ */
  real_argc = 0;
  for (i = 0; i < argc; i++)
    {
      if (argv[i] != NULL)
	real_argv[real_argc++] = g_strdup(argv[i]);	/* ĤϳȤ */
    }

  application->argc = real_argc;
  application->argv = real_argv;

  gtk_rc_parse_string("gtk-toolbar-style = icons\n"
		      "gtk-toolbar-icon-size = small-toolbar\n");
  gtk_rc_add_default_file(PKGDATADIR "/" OCHUSHA_GTKRC);

  g_thread_init(NULL);
  gdk_threads_init();
  gtk_set_locale();
  gtk_init(&real_argc, &real_argv);

  bindtextdomain(PACKAGE_NAME, LOCALEDIR);
  textdomain(PACKAGE_NAME);

  bind_textdomain_codeset(PACKAGE_NAME, "UTF-8");

  xmlInitParser();

  initialize_default_entity_dictionary();
  initialize_common_converters();

  initialize_worker(INITIAL_NUM_THREADS, MAXIMUM_NUM_THREADS,
		    MAXIMUM_NUM_MODEST_THREADS);

  resource_file = ochusha_config_find_file(&application->config,
					   OCHUSHA_GTKRC, NULL);
  if (resource_file != NULL)
    {
      gtk_rc_parse(resource_file);
      gtk_rc_add_default_file(resource_file);
      G_FREE(resource_file);
    }

  resource_file = ochusha_config_find_file(&application->config,
					   OCHUSHA_PREFS_GTKRC, NULL);
  if (resource_file != NULL)
    {
      gtk_rc_parse(resource_file);
      gtk_rc_add_default_file(resource_file);
      G_FREE(resource_file);
    }

  if (session_id != NULL)
    application->sm_client_id = G_STRDUP(session_id);

  initialize_xsmp_handlers(application);

  if (application->sm_client_id != NULL)
    application->session_id = G_STRDUP(application->sm_client_id);
  else
    application->session_id = G_STRDUP("ochusha_self_managed_session");

  if (application->session_id != NULL
      && snprintf(buf, PATH_MAX, ".sm/%s", application->session_id) < PATH_MAX)
    application->session_subdir = buf;
  else
    application->session_subdir = NULL;

  ochusha_init_application(application);

#if ENABLE_STRICT_CHECK	/* Ѥθå */
  g_log_set_always_fatal(G_LOG_LEVEL_WARNING
			 | G_LOG_LEVEL_CRITICAL
			 | G_LOG_LEVEL_ERROR);
#endif

#if USE_THREAD_FOR_MAIN
  if (pthread_create(&main_thread, NULL,
		     (void *(*)(void *))real_main, application) != 0)
    {
      fprintf(stderr, "Couldn't start main thread.\n");
      abort();
    }

  if (pthread_join(main_thread, NULL) != 0)
    {
      fprintf(stderr, "Couldn't join to main thread.\n");
      abort();
    }
#else
# if DEBUG_GDK_LOCK
  main_thread = pthread_self();
# endif

  OCHU_THREADS_ENTER();

  ENTER_MAIN_EVENT_LOOP();
  gtk_main();

# if TRACE_MEMORY_USAGE
  if (application->top_level != NULL)
    gtk_widget_destroy(GTK_WIDGET(application->top_level));
# endif

# if 1
  MAIN_THREAD_EVENT_CALLBACK();	/* ä */
  OCHU_THREADS_LEAVE();
# else
  gdk_threads_leave();
# endif
#endif

  ochusha_network_broker_terminate(application->broker);
  terminate_workers();
  join_workers();

  xmlCleanupParser();

  if (application->log_file != stderr)
    fclose(application->log_file);

#if TRACE_MEMORY_USAGE
  OCHU_OBJECT_UNREF(G_OBJECT(application->table));
  application->table = NULL;
  OCHU_OBJECT_UNREF(G_OBJECT(application->broker));
  application->broker = NULL;

  G_FREE(application->threadlist_view_contents);
  application->threadlist_view_contents = NULL;
  if (application->thread_view_font_name != NULL)
    {
      G_FREE(application->thread_view_font_name);
      application->thread_view_font_name = NULL;
    }
  G_FREE(application->user_default_filter.rule);
  application->user_default_filter.rule = NULL;
  G_FREE(application->web_browser_template);
  application->web_browser_template = NULL;
  G_FREE(application->config.home);
  application->config.home = NULL;
  G_FREE(application->config.bbsmenu_url);
  application->config.bbsmenu_url = NULL;
  G_FREE(application->config.proxy_url);
  application->config.proxy_url = NULL;
  G_FREE(application->config.proxy_user);
  application->config.proxy_user = NULL;
  G_FREE(application->config.proxy_password);
  application->config.proxy_password = NULL;

  if (application->bookmark_text != NULL)
    {
      G_FREE(application->bookmark_text);
      application->bookmark_text = NULL;
    }

  if (application->a_bone_by_name_pattern != NULL)
    {
      G_FREE(application->a_bone_by_name_pattern);
      application->a_bone_by_name_pattern = NULL;
    }

  if (application->a_bone_by_id_pattern != NULL)
    {
      G_FREE(application->a_bone_by_id_pattern);
      application->a_bone_by_id_pattern = NULL;
    }

  if (application->a_bone_by_content_pattern != NULL)
    {
      G_FREE(application->a_bone_by_content_pattern);
      application->a_bone_by_content_pattern = NULL;
    }

  if (application->last_name != NULL)
    {
      G_FREE(application->last_name);
      application->last_name = NULL;
    }

  if (application->last_mail != NULL)
    {
      G_FREE(application->last_mail);
      application->last_mail = NULL;
    }

  if (application->session_id != NULL)
    {
      G_FREE(application->session_id);
      application->session_id = NULL;
    }

  G_FREE(application);

  dump_pointers();
#endif

#if TRACE_REFCOUNT
  dump_objects();
#endif

  return 0;
}


#if DEBUG_GDK_LOCK
void
ochu_threads_enter(void)
{
  OCHU_THREADS_ENTER();
}


void
ochu_threads_leave(void)
{
  OCHU_THREADS_LEAVE();
}
#endif
