/*
 * Copyright (c) 2003-2004 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: image_ui.c,v 1.4 2004/02/28 23:45:13 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "ochusha_async_buffer.h"
#include "ochusha_network_broker.h"
#include "utils.h"

#include "worker.h"

#include "ochusha_ui.h"
#include "image_ui.h"

#include <glib.h>
#include <gtk/gtk.h>

#include <fcntl.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


static GQuark download_image_job_args_id;

static pthread_mutex_t image_ui_lock;

static GHashTable *image_table = NULL;
static GHashTable *image_view_table = NULL;
static GHashTable *a_bone_url_table = NULL;


#define IMAGE_UI_LOCK					\
  if (pthread_mutex_lock(&image_ui_lock) != 0)		\
    {							\
      fprintf(stderr, "Couldn't lock a mutex.\n");	\
      abort();						\
    }

#define IMAGE_UI_UNLOCK					\
  if (pthread_mutex_unlock(&image_ui_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't unlock a mutex.\n");	\
      abort();						\
    }


void
initialize_image_ui(OchushaApplication *application)
{
  int fd;

  if (pthread_mutex_init(&image_ui_lock, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a mutex.\n");
      abort();
    }

  image_table = g_hash_table_new_full(g_str_hash, g_str_equal,
				      (GDestroyNotify)g_free, NULL);
  image_view_table = g_hash_table_new_full(g_str_hash, g_str_equal,
					   (GDestroyNotify)g_free, NULL);
  a_bone_url_table = g_hash_table_new_full(g_str_hash, g_str_equal,
					   (GDestroyNotify)g_free, NULL);

  download_image_job_args_id
    = g_quark_from_static_string("DownloadUI::DownloadImageJobArgs");

  /* ܡURLꥹȤե뤫ɤ */
  fd = ochusha_config_open_file(&application->config, OCHUSHA_A_BONE_IMAGE_TXT,
				NULL, O_RDONLY);
  if (fd >= 0)
    {
      FILE *file = fdopen(fd, "r");
      if (file != NULL)
	{
	  char url[PATH_MAX];
	  url[PATH_MAX - 1] = '\0';
	  while (!feof(file))
	    {
	      char *tmp_str = fgets(url, PATH_MAX - 1, file);
	      if (tmp_str != NULL)
		{
		  char *key;
		  tmp_str = strpbrk(url, "\r\n");
		  if (tmp_str != NULL)
		    *tmp_str = '\0';
		  key = g_strdup(url);
		  g_hash_table_insert(a_bone_url_table, key, key);
		}
	    }
	  fclose(file);
	}
      else
	close(fd);
    }
}


static gboolean
output_a_bone_url(gpointer key, gpointer value, gpointer user_data)
{
  FILE *file = (FILE *)user_data;

  fprintf(file, "%s\n", (const char *)key);

  return TRUE;
}


void
finalize_image_ui(OchushaApplication *application)
{
  int fd = ochusha_config_open_file(&application->config,
				    OCHUSHA_A_BONE_IMAGE_TXT, NULL,
				    O_WRONLY | O_TRUNC | O_CREAT);
  if (fd >= 0)
    {
      FILE *file = fdopen(fd, "w");
      /* ܡURLꥹȤե˽񤭽Ф */
      if (file != NULL)
	{
	  g_hash_table_foreach(a_bone_url_table,
			       (GHFunc)output_a_bone_url, file);
	  fclose(file);
	}
      else
	close(fd);
    }
}


static void
a_bone_image_shown(GtkImage *image, gpointer unused)
{
  g_return_if_fail(GTK_IS_IMAGE(image));
  gtk_image_set_from_stock(image, GTK_STOCK_DIALOG_WARNING,
			   GTK_ICON_SIZE_DIALOG);
}


void
ochusha_a_bone_image(OchushaApplication *application, const char *url)
{
  /* gdk_threads_enter()ĶƤ֤ */
  GSList *image_list;
  char *key;

  if (url == NULL || *url == '\0')
    return;

  key = g_strdup(url);

  g_hash_table_insert(a_bone_url_table, key, key);

  image_list = g_hash_table_lookup(image_view_table, key);
  g_slist_foreach(image_list, (GFunc)a_bone_image_shown, NULL);

  ochusha_config_cache_unlink_file(&application->config, url);
}


typedef struct _SetPixbufArgs
{
  GtkWidget *image;
  GdkPixbufAnimation *animation;
  GdkPixbufAnimationIter *anim_iter;
  GdkPixbuf *pixbuf;
  char *url;
} SetPixbufArgs;


static gboolean
set_pixbuf_on_idle(SetPixbufArgs *args)
{
  GtkWidget *image;
  OCHU_THREADS_ENTER();

  image = args->image;

  if (image != NULL)
    {
      GSList *image_list = g_hash_table_lookup(image_view_table, args->url);
      if (g_slist_find(image_list, image) == NULL)
	image = NULL;
    }

  if (image != NULL && GTK_WIDGET_VISIBLE(image))
    {
      if (g_hash_table_lookup(a_bone_url_table, args->url) == NULL)
	{
	  if (args->animation != NULL)
	    {
	      if (gdk_pixbuf_animation_is_static_image(args->animation)
		  || args->anim_iter == NULL)
		gtk_image_set_from_pixbuf(GTK_IMAGE(image),
					  gdk_pixbuf_animation_get_static_image(args->animation));
	      else
		gtk_image_set_from_pixbuf(GTK_IMAGE(image),
					  gdk_pixbuf_animation_iter_get_pixbuf(args->anim_iter));
	    }
	  else
	    gtk_image_set_from_pixbuf(GTK_IMAGE(image), args->pixbuf);
	}
      else
	gtk_image_set_from_stock(GTK_IMAGE(image),
				 GTK_STOCK_DIALOG_WARNING,
				 GTK_ICON_SIZE_DIALOG);
    }

  if (args->animation != NULL)
    OCHU_OBJECT_UNREF(args->animation);
  if (args->pixbuf != NULL)
    OCHU_OBJECT_UNREF(args->pixbuf);

  OCHU_THREADS_LEAVE();

  G_FREE(args->url);

  return FALSE;
}


typedef struct _DownloadImageJobArgs
{
  OchushaAsyncBuffer *buffer;
  GtkWidget *image;
  int width;
  int height;
  char *url;
  gboolean owner;
  gboolean pixbuf_shown;

  GdkPixbufAnimationIter *anim_iter;
  guint anim_timeout_id;
} DownloadImageJobArgs;


static gboolean
anim_timeout_cb(gpointer user_data)
{
  DownloadImageJobArgs *job_args = (DownloadImageJobArgs *)user_data;
  GdkPixbufAnimationIter *anim_iter = job_args->anim_iter;

#if 0
  fprintf(stderr, "anim_timeout_cb(): job_args->image=%p, anim_iter=%p\n",
	  job_args->image, anim_iter);
#endif

  OCHU_THREADS_ENTER();

  job_args->anim_timeout_id = 0;

  if (job_args->image != NULL && anim_iter != NULL)
    {
      int delay;
      if (gdk_pixbuf_animation_iter_advance(anim_iter, NULL))
	{
	  gtk_image_set_from_pixbuf(GTK_IMAGE(job_args->image),
				    gdk_pixbuf_animation_iter_get_pixbuf(anim_iter));
	}
      delay = gdk_pixbuf_animation_iter_get_delay_time(anim_iter);
      job_args->anim_timeout_id
	= g_timeout_add(delay, (GSourceFunc)anim_timeout_cb, job_args);
#if 0
      fprintf(stderr, "(2)timeout_add: delay=%d\n", delay);
#endif
     }

  OCHU_THREADS_LEAVE();

  return FALSE;
}


static void
set_pixbuf(GdkPixbufLoader *loader, DownloadImageJobArgs *job_args)
{
  SetPixbufArgs *args;
  GdkPixbufAnimation *animation;
  GdkPixbuf *pixbuf;
  GtkWidget *image;

#if 0
  if (job_args->pixbuf_shown)
    return;
#endif

  if (g_hash_table_lookup(a_bone_url_table, job_args->url) != NULL)
    return;

  animation = gdk_pixbuf_loader_get_animation(loader);
  if (animation != NULL)
    pixbuf = NULL;
  else
    {
      pixbuf = gdk_pixbuf_loader_get_pixbuf(loader);
      if (pixbuf == NULL)
	return;
    }

  IMAGE_UI_LOCK
  {
    image = job_args->image;
#if 0
    if (image != NULL)
      OCHU_OBJECT_REF(image);
#endif
  }
  IMAGE_UI_UNLOCK;

  if (image == NULL)
    return;

  if (animation != NULL)
    OCHU_OBJECT_REF(animation);

  if (pixbuf != NULL)
    OCHU_OBJECT_REF(pixbuf);

  if (animation != NULL
      && !gdk_pixbuf_animation_is_static_image(animation))
    {
      int delay;
      job_args->anim_iter = gdk_pixbuf_animation_get_iter(animation, NULL);
      delay = gdk_pixbuf_animation_iter_get_delay_time(job_args->anim_iter);
      if (job_args->anim_timeout_id != 0)
	g_source_remove(job_args->anim_timeout_id);
      job_args->anim_timeout_id
	= g_timeout_add(delay, (GSourceFunc)anim_timeout_cb, job_args);
#if 0
      fprintf(stderr, "(1)timeout_add: delay=%d\n", delay);
#endif
    }

  args = g_new0(SetPixbufArgs, 1);
  args->image = job_args->image;
  args->animation = animation;
  args->anim_iter = job_args->anim_iter;
  args->pixbuf = pixbuf;
  args->url = G_STRDUP(job_args->url);

  job_args->pixbuf_shown = TRUE;

  g_idle_add_full(G_PRIORITY_HIGH_IDLE + 13,
		  (GSourceFunc)set_pixbuf_on_idle,
		  args, (GDestroyNotify)g_free);

}


static void
size_prepared_cb(GdkPixbufLoader *loader, int width, int height,
		 GdkPixbufLoader *args)
{
  DownloadImageJobArgs *job_args;
  double width_request;
  double height_request;
  double width_scale;
  double height_scale;

  g_return_if_fail(GDK_IS_PIXBUF_LOADER(loader));
  job_args = g_object_get_qdata(G_OBJECT(loader), download_image_job_args_id);
  g_return_if_fail(job_args != NULL);

  if (g_hash_table_lookup(a_bone_url_table, job_args->url) != NULL)
    return;

  width_request = job_args->width;
  height_request = job_args->height;

  if (width <= width_request && height <= height_request)
    return;	/* Ϥʤ */

  width_scale = width_request / width;
  height_scale = height_request / height;
  if (width_scale > height_scale)
    {
      width *= height_scale;
      height *= height_scale;
    }
  else
    {
      width *= width_scale;
      height *= width_scale;
    }
  if (width < 0 || height < 0)
    return;

  gdk_pixbuf_loader_set_size(loader, width, height);

  set_pixbuf(loader, job_args);
}


static void
area_prepared_cb(GdkPixbufLoader *loader, GdkPixbufLoader *args)
{
  DownloadImageJobArgs *job_args;
  g_return_if_fail(GDK_IS_PIXBUF_LOADER(loader));

  job_args = g_object_get_qdata(G_OBJECT(loader), download_image_job_args_id);
  g_return_if_fail(job_args != NULL);

  set_pixbuf(loader, job_args);
}


static void
area_updated_cb(GdkPixbufLoader *loader, int x, int y, int width, int height,
		GdkPixbufLoader *args)
{
  DownloadImageJobArgs *job_args;
  g_return_if_fail(GDK_IS_PIXBUF_LOADER(loader));

  job_args = g_object_get_qdata(G_OBJECT(loader), download_image_job_args_id);
  g_return_if_fail(job_args != NULL);

  set_pixbuf(loader, job_args);
}


static void
download_image(WorkerThread *employee, gpointer args)
{
  GdkPixbufLoader *loader = GDK_PIXBUF_LOADER(args);
  DownloadImageJobArgs *job_args
    = g_object_get_qdata(G_OBJECT(loader), download_image_job_args_id);
  OchushaAsyncBuffer *buffer = job_args->buffer;
  GtkWidget *image;
  gboolean image_shown = FALSE;

  OCHU_THREADS_ENTER();
  IMAGE_UI_LOCK
  {
    image = job_args->image;
  }
  IMAGE_UI_UNLOCK;
  OCHU_THREADS_LEAVE();

  if (job_args->width >= 0 || job_args->height >= 0)
    g_signal_connect(G_OBJECT(loader), "size_prepared",
		     G_CALLBACK(size_prepared_cb), loader);

  g_signal_connect(G_OBJECT(loader), "area_prepared",
		   G_CALLBACK(area_prepared_cb), loader);

  g_signal_connect(G_OBJECT(loader), "area_updated",
		   G_CALLBACK(area_updated_cb), loader);

  if (ochusha_async_buffer_active_ref(buffer))
    {
      size_t offset = 0;

      ochusha_async_buffer_lock(buffer);
      {
	while (TRUE)
	  {
	    size_t length = buffer->length;
	    if (length > 0 && length > offset)
	      {
		size_t updated_len = length - offset;
		OCHU_THREADS_ENTER();
		image_shown
		  |= gdk_pixbuf_loader_write(loader,
					     ((const guchar *)buffer->buffer
					      + offset), updated_len, NULL);
		OCHU_THREADS_LEAVE();
	      }
	    if (buffer->fixed)
	      goto terminated;
	    offset = length;
	    do
	      {
		if (!ochusha_async_buffer_wait(buffer))
		  goto terminated;
	      } while (length >= buffer->length && !buffer->fixed);
	  }
      }
    terminated:
      ochusha_async_buffer_unlock(buffer);
      ochusha_async_buffer_active_unref(buffer);
    }

  if (!image_shown)
    {
      OCHU_THREADS_ENTER();
      gtk_image_set_from_stock(GTK_IMAGE(image),
			       GTK_STOCK_MISSING_IMAGE,
			       GTK_ICON_SIZE_DIALOG);
      OCHU_THREADS_LEAVE();
    }

  gdk_pixbuf_loader_close(loader, NULL);

  IMAGE_UI_LOCK
  {
    if (job_args->owner)
      {
	g_hash_table_remove(image_table, job_args->url);
      }
  }
  IMAGE_UI_UNLOCK;

  OCHU_OBJECT_UNREF(buffer);

  OCHU_OBJECT_UNREF(loader);
}


static void
image_destroyed(GtkWidget *image, GdkPixbufLoader *loader)
{
  DownloadImageJobArgs *job_args
    = g_object_get_qdata(G_OBJECT(loader), download_image_job_args_id);
  GSList *image_list;
#if 0
  fprintf(stderr, "image_destroyed(%p, %p)\n", image, job_args);
#endif

  if (job_args != NULL)
    {
      image_list = g_hash_table_lookup(image_view_table, job_args->url);
      image_list = g_slist_remove(image_list, image);
      if (image_list != NULL)
	g_hash_table_insert(image_view_table,
			    g_strdup(job_args->url), image_list);
      else
	g_hash_table_remove(image_view_table, job_args->url);
      if (job_args->anim_timeout_id != 0)
	{
	  g_source_remove(job_args->anim_timeout_id);
	  job_args->anim_timeout_id = 0;
	}
      if (job_args->anim_iter != NULL)
	{
	  g_object_unref(job_args->anim_iter);
	  job_args->anim_iter = NULL;
	}

      IMAGE_UI_LOCK
      {
	job_args->image = NULL;
      }
      IMAGE_UI_UNLOCK;
    }

  OCHU_OBJECT_UNREF(loader);
}


static void
free_job_args(DownloadImageJobArgs *job_args)
{
  if (job_args->url != NULL)
    G_FREE(job_args->url);
  G_FREE(job_args);
}


GtkWidget *
ochusha_download_image(OchushaApplication *application, const char *url,
		       int width, int height)
{
  OchushaAsyncBuffer *buffer;
  GtkWidget *image;
  WorkerJob *job = NULL;
  GdkPixbufLoader *loader;
  DownloadImageJobArgs *job_args = NULL;
  char *scheme;
  gboolean owner = FALSE;
  GSList *image_list;

  g_return_val_if_fail(url != NULL, NULL);

  if (*url == '\0')
    return NULL;

  if (g_hash_table_lookup(a_bone_url_table, url) != NULL)
    return NULL;

  scheme = ochusha_utils_url_extract_scheme(url);
  if (scheme == NULL)
    return NULL;
  if (strcmp(scheme, "http") != 0)
    {
      G_FREE(scheme);
      return NULL;
    }
  G_FREE(scheme);

  IMAGE_UI_LOCK
  {
    buffer = (OchushaAsyncBuffer *)g_hash_table_lookup(image_table, url);
    if (buffer == NULL)
      {
	buffer = ochusha_network_broker_read_from_url(application->broker,
					NULL, url, NULL,
					OCHUSHA_NETWORK_BROKER_CACHE_AS_IS,
					TRUE,
					application->image_chunksize);
	if (buffer != NULL)
	  {
	    g_hash_table_insert(image_table, g_strdup(url), buffer);
	    owner = TRUE;
	  }
      }
    else
      OCHU_OBJECT_REF(buffer);
  }
  IMAGE_UI_UNLOCK;

  if (buffer == NULL)
    return NULL;

  loader = gdk_pixbuf_loader_new();
  OCHU_OBJECT_REF(loader);

  job_args = G_NEW0(DownloadImageJobArgs, 1);
  job_args->buffer = buffer;
  image = gtk_image_new_from_stock(OCHUSHA_STOCK_MISSING_IMAGE,
				   GTK_ICON_SIZE_DIALOG);

  image_list = g_hash_table_lookup(image_view_table, url);
  image_list = g_slist_append(image_list, image);
  g_hash_table_insert(image_view_table, g_strdup(url), image_list);

#if 0
  OCHU_OBJECT_REF(image);
  gtk_object_sink(GTK_OBJECT(image));
#endif
  job_args->image = image;
  job_args->width = width;
  job_args->height = height;
  job_args->url = G_STRDUP(url);
  job_args->owner = owner;

  g_object_set_qdata_full(G_OBJECT(loader), download_image_job_args_id,
			  job_args, (GDestroyNotify)free_job_args);
  g_signal_connect(G_OBJECT(image), "destroy",
		   G_CALLBACK(image_destroyed), loader);

  job = G_NEW0(WorkerJob, 1);
  job->canceled = FALSE;
  job->job = download_image;
  job->args = loader;

  commit_modest_job(job);

  return image;
}
