/*
 * Copyright (c) 2003-2004 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: session.c,v 1.12 2004/01/15 02:18:05 fuyu Exp $
 */

#include "config.h"

#include "ochusha.h"
#include "ochusha_private.h"

#include "ochusha_ui.h"
#include "boardlist_ui.h"
#include "session.h"

#include <glib.h>
#include <gtk/gtk.h>

#include <libxml/SAX.h>
#include <libxml/parser.h>

#include <fcntl.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <unistd.h>

#include <zlib.h>


static void ochusha_session_write_session_xml(OchushaApplication *application,
					      const char *session_id);


void
ochusha_session_save_states(OchushaApplication *application)
{
  GtkWidget *widget;
  GtkAllocation *allocation;
  int x_pos;
  int y_pos;
  int sx;
  int sy;

#if 0
  fprintf(stderr, "ochusha_session_save_states.\n");
#endif

  g_return_if_fail(GTK_IS_WIDGET(application->top_level));
  widget = GTK_WIDGET(application->top_level);

  g_return_if_fail(GDK_IS_WINDOW(widget->window));

  sx = gdk_screen_width();
  sy = gdk_screen_height();
  gdk_window_get_root_origin(widget->window, &x_pos, &y_pos);

  x_pos %= sx;
  y_pos %= sy;

  application->x_pos = (x_pos >= 0 ? x_pos : 0);
  application->y_pos = (y_pos >= 0 ? y_pos : 0);

  allocation = &widget->allocation;
  application->width = allocation->width;
  application->height = allocation->height;

  allocation = &application->boardlist_view->allocation;
  application->boardlist_width = allocation->width;

  finalize_boardlist(application);

  application->shutdown_succeeded = TRUE;
  ochusha_session_write_session_xml(application, application->session_subdir);

  ochusha_write_preferences(application);
  application->states_saved = TRUE;
}


void
ochusha_session_discard_states(OchushaApplication *application,
			       const char *session_id)
{
  char pathname[PATH_MAX];

  g_return_if_fail(session_id != NULL);

  if (snprintf(pathname, PATH_MAX, ".sm/%s", session_id)
      >= PATH_MAX)
    return;

#if 0
  fprintf(application->log_file,
	  "Removing session directory: %s\n", pathname);
  fflush(application->log_file);
#endif

  ochusha_config_unlink_directory(&application->config, pathname);
}


#define OUTPUT_SESSION_ATTRIBUTE_INT(gzfile, application, attribute)	\
  do									\
    {									\
      if ((application)->attribute)					\
	gzprintf(gzfile,						\
		"    <attribute name=\"" #attribute	"\">\n"		\
		"      <int val=\"%d\"/>\n"				\
		"    </attribute>\n", (application)->attribute);	\
    } while (0)

#define OUTPUT_SESSION_ATTRIBUTE_BOOLEAN(gzfile, config, attribute)	\
  gzprintf(gzfile,							\
	   "    <attribute name=\"" #attribute	"\">\n"			\
	   "      <boolean val=\"%s\"/>\n"				\
	   "    </attribute>\n",					\
	   (application)->attribute ? "true" : "false" )

#define OUTPUT_SESSION_ATTRIBUTE_STRING(gzfile, config, attribute)	\
  do									\
    {									\
      if ((config)->attribute != NULL)					\
	{								\
	  gchar *text = g_markup_escape_text((config)->attribute, -1);	\
	  gzprintf(gzfile,						\
		   "    <attribute name=\"" #attribute	"\">\n"		\
		   "      <string>%s</string>\n"			\
		   "    </attribute>\n", text);				\
	  g_free(text);							\
	}								\
    } while (0)


static void
ochusha_session_write_session_xml(OchushaApplication *application,
				  const char *session_subdir)
{
  int fd;
  gzFile *session_xml;

  fd = ochusha_config_open_file(&application->config, OCHUSHA_SESSION_XML,
				session_subdir, O_WRONLY | O_TRUNC | O_CREAT);
  if (fd < 0)
    {
      fprintf(application->log_file, "Couldn't open session.xml -- <2>.\n");
      fflush(application->log_file);
      return;
    }

  session_xml = gzdopen(fd, "w");
  if (session_xml == NULL)
    {
      close(fd);
      fprintf(application->log_file, "Couldn't open session.xml -- <3>.\n");
      fflush(application->log_file);
      return;
    }

  gzprintf(session_xml, "<?xml version=\"1.0\"?>\n");
  gzprintf(session_xml, "<ochusha>\n");
  gzprintf(session_xml, "  <session>\n");

  OUTPUT_SESSION_ATTRIBUTE_INT(session_xml, application, x_pos);
  OUTPUT_SESSION_ATTRIBUTE_INT(session_xml, application, y_pos);
  OUTPUT_SESSION_ATTRIBUTE_INT(session_xml, application, width);
  OUTPUT_SESSION_ATTRIBUTE_INT(session_xml, application, height);
  OUTPUT_SESSION_ATTRIBUTE_INT(session_xml, application, boardlist_width);
  OUTPUT_SESSION_ATTRIBUTE_INT(session_xml, application, threadlist_height);

  OUTPUT_SESSION_ATTRIBUTE_INT(session_xml, application,
			       threadlist_search_direction);
  OUTPUT_SESSION_ATTRIBUTE_BOOLEAN(session_xml, application,
				  threadlist_search_enable_wrap);
  OUTPUT_SESSION_ATTRIBUTE_BOOLEAN(session_xml, application,
				  threadlist_search_match_case);
  OUTPUT_SESSION_ATTRIBUTE_BOOLEAN(session_xml, application,
				  threadlist_search_use_regexp);

  OUTPUT_SESSION_ATTRIBUTE_INT(session_xml, application,
			       thread_search_direction);
  OUTPUT_SESSION_ATTRIBUTE_BOOLEAN(session_xml, application,
				  thread_search_enable_wrap);
  OUTPUT_SESSION_ATTRIBUTE_BOOLEAN(session_xml, application,
				  thread_search_match_case);
  OUTPUT_SESSION_ATTRIBUTE_BOOLEAN(session_xml, application,
				  thread_search_use_regexp);

  gzprintf(session_xml, "  </session>\n");
  gzprintf(session_xml, "</ochusha>\n");

  gzclose(session_xml);
}


typedef enum
{
  SAX_INITIAL,
  SAX_OCHUSHA,
  SAX_SESSION,
  SAX_SESSION_ATTRIBUTE,
  SAX_SESSION_ATTRIBUTE_BOOLEAN,
  SAX_SESSION_ATTRIBUTE_INT,
  SAX_SESSION_ATTRIBUTE_STRING,
  SAX_ACCEPTED,
  SAX_ERROR
} SAXState;


typedef struct _SAXContext
{
  SAXState state;

  char *current_attr_name;
  char *current_attr_val;

  GHashTable *attributes;
} SAXContext;


static void
cleanup_sax_context(SAXContext *context)
{
  if (context->current_attr_name != NULL)
    {
      G_FREE(context->current_attr_name);
      context->current_attr_name = NULL;
    }

  if (context->current_attr_val != NULL)
    {
      G_FREE(context->current_attr_val);
      context->current_attr_val = NULL;
    }

  if (context->attributes != NULL)
    {
      g_hash_table_destroy(context->attributes);
      context->attributes = NULL;
    }
}


#if TRACE_MEMORY_USAGE
static void
trace_free(gpointer pointer)
{
  G_FREE(pointer);
}
#define TRACE_FREE	trace_free
#else
#define TRACE_FREE	g_free
#endif


static void
startElementHandler(void *context, const xmlChar *name, const xmlChar **atts)
{
  SAXContext *sax_context = (SAXContext *)context;

  switch (sax_context->state)
    {
    case SAX_INITIAL:
      if (strcmp(name, "ochusha") == 0)
	{ sax_context->state = SAX_OCHUSHA; return; }
      break;	/* 顼 */

    case SAX_OCHUSHA:
      if (strcmp(name, "session") == 0)
	{
	  sax_context->attributes
	    = g_hash_table_new_full(g_str_hash, g_str_equal,
				    TRACE_FREE, TRACE_FREE);
	  sax_context->state = SAX_SESSION;
	  return;
	}
      break;	/* 顼 */

    case SAX_SESSION:
      if (strcmp(name, "attribute") == 0
	  && atts != NULL && strcmp(atts[0], "name") == 0)
	{
	  sax_context->state = SAX_SESSION_ATTRIBUTE;

	  if (sax_context->current_attr_val != NULL)
	    {
#if DEBUG_SAX_HANDLER
	      fprintf(stderr, "unexpected attribute found: %s=%s\n",
		      sax_context->current_attr_name,
		      sax_context->current_attr_val);
#endif
	      G_FREE(sax_context->current_attr_name);
	      G_FREE(sax_context->current_attr_val);
	      sax_context->current_attr_name = NULL;
	      sax_context->current_attr_val = NULL;
	      break;	/* 顼 */
	    }
	  sax_context->current_attr_name = G_STRDUP(atts[1]);
	  return;
	}
      break;	/* 顼 */

    case SAX_SESSION_ATTRIBUTE:
      if (atts != NULL && strcmp(atts[0], "val") == 0)
	{
	  /* int/booleanβǽ */
	  if (strcmp(name, "int") == 0)
	    {
	      sax_context->state = SAX_SESSION_ATTRIBUTE_INT;
	    }
	  else if (strcmp(name, "boolean") == 0)
	    {
	      sax_context->state = SAX_SESSION_ATTRIBUTE_BOOLEAN;
	    }
	  else
	    {
#if DEBUG_SAX_HANDLER
	      fprintf(stderr, "element unexpected in state(%d) found: %s\n",
		      sax_context->state, name);
#endif
	      break;	/* 顼 */
	    }

	  if (sax_context->current_attr_val != NULL)
	    {
	      /* 顼ǤɤΤġ̯ */
#if DEBUG_SAX_HANDLER
	      fprintf(stderr,
		      "attribute %s=\"%s\" is overwritten by \"%s\"!\n",
		      sax_context->current_attr_name,
		      sax_context->current_attr_val, atts[1]);
#endif
	      G_FREE(sax_context->current_attr_val);
	    }

	  sax_context->current_attr_val = G_STRDUP(atts[1]);
	  return;
	}
      else if (strcmp(name, "string") == 0)
	{
	  sax_context->state = SAX_SESSION_ATTRIBUTE_STRING;
	  return;
	}
      break;	/* 顼 */

    case SAX_SESSION_ATTRIBUTE_STRING:
    case SAX_SESSION_ATTRIBUTE_BOOLEAN:
    case SAX_SESSION_ATTRIBUTE_INT:
    case SAX_ACCEPTED:
    case SAX_ERROR:
      break;	/* 顼 */

    default:
      fprintf(stderr, "startHandler is called in unknown state: %d\n",
	      sax_context->state);
    }
  sax_context->state = SAX_ERROR;
}


static void
endElementHandler(void *context, const xmlChar *name)
{
  SAXContext *sax_context = (SAXContext *)context;

  switch (sax_context->state)
    {
    case SAX_OCHUSHA:
      if (strcmp(name, "ochusha") == 0)
	{ sax_context->state = SAX_ACCEPTED; return; }
      break;	/* 顼 */

    case SAX_SESSION:
      if (strcmp(name, "session") == 0)
	{ sax_context->state = SAX_OCHUSHA; return; }
      break;	/* 顼 */

    case SAX_SESSION_ATTRIBUTE:
      if (strcmp(name, "attribute") == 0)
	{
	  GHashTable *hash_table = sax_context->attributes;
	  sax_context->state = SAX_SESSION;
#if DEBUG_SAX_HANDLER_NEW
	  fprintf(stderr, "%s = \"%s\"\n", sax_context->current_attr_name,
		  sax_context->current_attr_val);
#endif
	  g_hash_table_insert(hash_table,
			      sax_context->current_attr_name,
			      sax_context->current_attr_val);
	  sax_context->current_attr_name = NULL;
	  sax_context->current_attr_val = NULL;
	  return;
	}
      break;	/* 顼 */

    case SAX_SESSION_ATTRIBUTE_STRING:
      if (strcmp(name, "string") == 0)
	{
	  sax_context->state = SAX_SESSION_ATTRIBUTE;

	  if (sax_context->current_attr_val == NULL)
	    sax_context->current_attr_val = G_STRDUP("");
	  return;
	}
      break;	/* 顼 */

    case SAX_SESSION_ATTRIBUTE_BOOLEAN:
      if (strcmp(name, "boolean") == 0)
	{ sax_context->state = SAX_SESSION_ATTRIBUTE; return; }
      break;	/* 顼 */

    case SAX_SESSION_ATTRIBUTE_INT:
      if (strcmp(name, "int") == 0)
	{ sax_context->state = SAX_SESSION_ATTRIBUTE; return; }
      break;	/* 顼 */

    case SAX_INITIAL:
    case SAX_ACCEPTED:
    case SAX_ERROR:
      break;	/* 顼 */

    default:
#if DEBUG_SAX_HANDLER
      fprintf(stderr, "endHandler called in unknown state: %d.\n",
	      sax_context->state);
#endif
      break;	/* 顼 */
    }
#if DEBUG_SAX_HANDLER
  fprintf(stderr, "Invalid document: </%s> appeared in state=%d\n",
	  name, sax_context->state);
#endif
  sax_context->state = SAX_ERROR;
}


static void
charactersHandler(void *context, const xmlChar *ch, int len)
{
  SAXContext *sax_context = (SAXContext *)context;

  if (sax_context->state == SAX_SESSION_ATTRIBUTE_STRING)
    {
      if (sax_context->current_attr_val == NULL)
	sax_context->current_attr_val = G_STRNDUP(ch, len);
      else
	{
	  int old_len = strlen(sax_context->current_attr_val);
	  sax_context->current_attr_val
	    = G_REALLOC(sax_context->current_attr_val, old_len + len + 1);
	  strncat(sax_context->current_attr_val + old_len, ch, len);
	}
    }
}


static void
nopHandler(void *context)
{
}


static xmlEntityPtr
getEntityHandler(void *context, const xmlChar *name)
{
  return NULL;
}


void
ochusha_session_restore_states(OchushaApplication *application)
{
  SAXContext context =
    {
      SAX_INITIAL,	/* state */
      NULL, NULL,	/* current_attr_name, current_attr_val */
      NULL		/* attributes */
    };
  xmlSAXHandler sax_handler;
  char *pathname;

  pathname = ochusha_config_find_file(&application->config,
				      OCHUSHA_SESSION_XML,
				      application->session_subdir);

  if (pathname == NULL)
    return;

  memset(&sax_handler, 0, sizeof(xmlSAXHandler));

#if LIBXML_VERSION >= 20600
  xmlSAX2InitDefaultSAXHandler(&sax_handler, TRUE);
#else
  initxmlDefaultSAXHandler(&sax_handler, TRUE);
#endif

  sax_handler.getEntity = getEntityHandler;
  sax_handler.startDocument = nopHandler;
  sax_handler.endDocument = nopHandler;
  sax_handler.startElement = startElementHandler;
  sax_handler.endElement = endElementHandler;
#if LIBXML_VERSION >= 20600
  sax_handler.startElementNs = NULL;
  sax_handler.endElementNs = NULL;
#endif
  sax_handler.characters = charactersHandler;

  xmlSAXUserParseFile(&sax_handler, &context, pathname);

  /* xmlCleanupParser(); */

  if (context.state == SAX_ACCEPTED)
    {
      GHashTable *attr = context.attributes;

      application->x_pos = ochusha_utils_get_attribute_int(attr, "x_pos");
      application->y_pos = ochusha_utils_get_attribute_int(attr, "y_pos");
      application->width = ochusha_utils_get_attribute_int(attr, "width");
      application->height = ochusha_utils_get_attribute_int(attr, "height");
      application->boardlist_width
	= ochusha_utils_get_attribute_int(attr, "boardlist_width");
      application->threadlist_height
	= ochusha_utils_get_attribute_int(attr, "threadlist_height");

      application->threadlist_search_direction
	= ochusha_utils_get_attribute_int(attr, "threadlist_search_direction");
      application->threadlist_search_enable_wrap
	= ochusha_utils_get_attribute_boolean(attr,
					      "threadlist_search_enable_wrap");
      application->threadlist_search_match_case
	= ochusha_utils_get_attribute_boolean(attr,
					      "threadlist_search_match_case");
      application->threadlist_search_use_regexp
	= ochusha_utils_get_attribute_boolean(attr,
					      "threadlist_search_use_regexp");

      application->thread_search_direction
	= ochusha_utils_get_attribute_int(attr, "thread_search_direction");
      application->thread_search_enable_wrap
	= ochusha_utils_get_attribute_boolean(attr,
					      "thread_search_enable_wrap");
      application->thread_search_match_case
	= ochusha_utils_get_attribute_boolean(attr,
					      "thread_search_match_case");
      application->thread_search_use_regexp
	= ochusha_utils_get_attribute_boolean(attr,
					      "thread_search_use_regexp");
    }
  else
    fprintf(stderr, "%s is unacceptable as an ochusha's session.xml.\n",
	    pathname);

  G_FREE(pathname);

  cleanup_sax_context(&context);

  return;
}
