/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ochusha_board_category.c,v 1.10 2003/12/30 21:53:17 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha_bulletin_board.h"
#include "ochusha_board_category.h"

#include "utils.h"

#include "marshal.h"

#include <glib-object.h>
#include <glib.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <zlib.h>


static void ochusha_board_category_class_init(OchushaBoardCategoryClass *klass);
static void ochusha_board_category_init(OchushaBoardCategory *category);
static void ochusha_board_category_finalize(GObject *object);

static void ochusha_board_category_set_property(GObject *object, guint prop_id,
						const GValue *value,
						GParamSpec *pspec);
static void ochusha_board_category_get_property(GObject *object, guint prop_id,
						GValue *value,
						GParamSpec *pspec);

static void ochusha_board_category_read_boardlist_element(
						OchushaBoardCategory *category,
						GHashTable *cat_attributes);
static void ochusha_board_category_write_boardlist_element(
						OchushaBoardCategory *category,
						gzFile *boardlist_xml);


GType
ochusha_board_category_get_type(void)
{
  static GType board_category_type = 0;

  if (board_category_type == 0)
    {
      static const GTypeInfo board_category_info =
	{
	  sizeof(OchushaBoardCategoryClass),
	  NULL, /* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc)ochusha_board_category_class_init,
	  NULL, /* class_finalize */
	  NULL, /* class_data */
	  sizeof(OchushaBoardCategory),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)ochusha_board_category_init,
	};

      board_category_type = g_type_register_static(G_TYPE_OBJECT,
						   "OchushaBoardCategory",
						   &board_category_info, 0);
    }

  return board_category_type;
}


enum {
  PROP_0,
  PROP_NAME
};


enum {
  READ_BOARDLIST_ELEMENT_SIGNAL,
  WRITE_BOARDLIST_ELEMENT_SIGNAL,
  LAST_SIGNAL
};


static GObjectClass *parent_class = NULL;
static int board_category_signals[LAST_SIGNAL] = { 0, 0 };


static void
ochusha_board_category_class_init(OchushaBoardCategoryClass *klass)
{
  GObjectClass *o_class = G_OBJECT_CLASS(klass);

  parent_class = g_type_class_peek_parent(klass);

  o_class->set_property = ochusha_board_category_set_property;
  o_class->get_property = ochusha_board_category_get_property;
  o_class->finalize = ochusha_board_category_finalize;

  g_object_class_install_property(o_class,
				  PROP_NAME,
				  g_param_spec_string("name",
						      _("Name"),
						      _("The name of the category"),
						      _("Unknown category"),
						      G_PARAM_READWRITE));

  board_category_signals[READ_BOARDLIST_ELEMENT_SIGNAL] =
    g_signal_new("read_boardlist_element",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(OchushaBoardCategoryClass,
				 read_boardlist_element),
		 NULL, NULL,
		 libochusha_marshal_VOID__POINTER,
		 G_TYPE_NONE, 1,
		 G_TYPE_POINTER);
  board_category_signals[WRITE_BOARDLIST_ELEMENT_SIGNAL] =
    g_signal_new("write_boardlist_element",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(OchushaBoardCategoryClass,
				 write_boardlist_element),
		 NULL, NULL,
		 libochusha_marshal_VOID__POINTER,
		 G_TYPE_NONE, 1,
		 G_TYPE_POINTER);

  klass->read_boardlist_element
    = ochusha_board_category_read_boardlist_element;
  klass->write_boardlist_element
    = ochusha_board_category_write_boardlist_element;
}


static void
ochusha_board_category_init(OchushaBoardCategory *category)
{
  category->name = NULL;
  category->board_list = NULL;
}


static void
ochusha_board_category_finalize(GObject *object)
{
  OchushaBoardCategory *category = OCHUSHA_BOARD_CATEGORY(object);

  if (category->name != NULL)
    {
      G_FREE(category->name);
      category->name = NULL;
    }

  if (category->board_list != NULL)
    {
      g_slist_foreach(category->board_list,
		      (GFunc)OCHU_OBJECT_UNREF_FUNC, NULL);
      g_slist_free(category->board_list);
      category->board_list = NULL;
    }

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


static void
ochusha_board_category_set_property(GObject *object, guint prop_id,
				    const GValue *value, GParamSpec *pspec)
{
  OchushaBoardCategory *category = OCHUSHA_BOARD_CATEGORY(object);

  switch (prop_id)
    {
    case PROP_NAME:
      ochusha_board_category_set_name(category, g_value_get_string(value));
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
      break;
    }
}


static void
ochusha_board_category_get_property(GObject *object, guint prop_id,
				    GValue *value, GParamSpec *pspec)
{
  OchushaBoardCategory *category = OCHUSHA_BOARD_CATEGORY(object);

  switch (prop_id)
    {
    case PROP_NAME:
      g_value_set_string(value, category->name);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
      break;
    }
}


OchushaBoardCategory *
ochusha_board_category_new(const gchar *name)
{
  g_assert(name != NULL);

  return OCHUSHA_BOARD_CATEGORY(g_object_new(OCHUSHA_TYPE_BOARD_CATEGORY,
					     "name", name,
					     NULL));
}


void
ochusha_board_category_set_name(OchushaBoardCategory *category,
				const gchar *name)
{
  g_return_if_fail(OCHUSHA_IS_BOARD_CATEGORY(category) && name != NULL);

  if (category->name != NULL)
    G_FREE(category->name);

  category->name = wipe_string(name);
}


const gchar *
ochusha_board_category_get_name(OchushaBoardCategory *category)
{
  g_return_val_if_fail(OCHUSHA_IS_BOARD_CATEGORY(category), NULL);

  return category->name;
}


void
ochusha_board_category_add_board(OchushaBoardCategory *category,
				 OchushaBulletinBoard *board)
{
  g_return_if_fail(OCHUSHA_IS_BOARD_CATEGORY(category)
		   && OCHUSHA_IS_BULLETIN_BOARD(board));

  if (g_slist_find(category->board_list, board) == NULL)
    {
      OCHU_OBJECT_REF(G_OBJECT(board));
      category->board_list = g_slist_append(category->board_list, board);
    }
}


void
ochusha_board_category_remove_board(OchushaBoardCategory *category,
				    OchushaBulletinBoard *board)
{
  g_return_if_fail(OCHUSHA_IS_BOARD_CATEGORY(category)
		   && OCHUSHA_IS_BULLETIN_BOARD(board));

  if (g_slist_find(category->board_list, board) == NULL)
    return;

  category->board_list = g_slist_remove(category->board_list, board);
  OCHU_OBJECT_UNREF(G_OBJECT(board));
}


static void
append_boards_helper(gpointer data, gpointer user_data)
{
  g_return_if_fail(OCHUSHA_IS_BULLETIN_BOARD(data));

  ochusha_board_category_add_board(OCHUSHA_BOARD_CATEGORY(user_data),
				   OCHUSHA_BULLETIN_BOARD(data));
}


void
ochusha_board_category_append_boards(OchushaBoardCategory *category,
				     GSList *board_list)
{
  g_return_if_fail(OCHUSHA_IS_BOARD_CATEGORY(category));

  g_slist_foreach(board_list, append_boards_helper, category);
}


static int
compare_url(gconstpointer list_data, gconstpointer url)
{
  return strcmp(((OchushaBulletinBoard *)list_data)->base_url, url);
}


OchushaBulletinBoard *
ochusha_board_category_lookup_board_by_url(OchushaBoardCategory *category,
					   const gchar *url)
{
  GSList *entry;

  g_return_val_if_fail(OCHUSHA_IS_BOARD_CATEGORY(category) && url != NULL,
		       NULL);

  entry = g_slist_find_custom(category->board_list, url, compare_url);

  return (OchushaBulletinBoard *)(entry != NULL ? entry->data : NULL);
}


static int
compare_name(gconstpointer list_data, gconstpointer name)
{
  return strcmp(((OchushaBulletinBoard *)list_data)->name, name);
}


OchushaBulletinBoard *
ochusha_board_category_lookup_board_by_name(OchushaBoardCategory *category,
					    const gchar *name)
{
  GSList *entry;

  g_return_val_if_fail(OCHUSHA_IS_BOARD_CATEGORY(category) && name != NULL,
		       NULL);

  entry = g_slist_find_custom(category->board_list, name, compare_name);

  return (OchushaBulletinBoard *)(entry != NULL ? entry->data : NULL);
}


#define OUTPUT_CATEGORY_ATTRIBUTE_BOOLEAN(gzfile, category, attribute)	\
  do									\
    {									\
      if ((category)->attribute)					\
	gzprintf(gzfile,						\
		 "      <attribute name=\"" #attribute	"\">\n"		\
		 "        <boolean val=\"%s\"/>\n"			\
		 "      </attribute>\n",				\
		 (category)->attribute ? "true" : "false");		\
    } while (0)

#define OUTPUT_CATEGORY_ATTRIBUTE_STRING(gzfile, category, attribute)	\
  do									\
    {									\
      if ((category)->attribute)					\
	{								\
	  gchar *text = g_markup_escape_text((category)->attribute, -1); \
	  gzprintf(gzfile,						\
		   "      <attribute name=\"" #attribute	"\">\n"	\
		   "        <string>%s</string>\n"			\
		   "      </attribute>\n", text);			\
	  g_free(text);							\
	}								\
    } while (0)

static void
ochusha_board_category_read_boardlist_element(OchushaBoardCategory *category,
					      GHashTable *cat_attributes)
{
  category->hidden = ochusha_utils_get_attribute_boolean(cat_attributes,
							 "hidden");
}


static void
ochusha_board_category_write_boardlist_element(OchushaBoardCategory *category,
					       gzFile *boardlist_xml)
{
  OUTPUT_CATEGORY_ATTRIBUTE_STRING(boardlist_xml, category, name);
  OUTPUT_CATEGORY_ATTRIBUTE_BOOLEAN(boardlist_xml, category, hidden);
}

