/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: threadlist_view.c,v 1.41 2004/01/04 06:46:09 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "ochusha_bbs_thread.h"

#include "ochusha_ui.h"
#include "threadlist_view.h"

#include "marshal.h"

#include <gdk/gdkkeysyms.h>
#include <glib.h>
#include <gtk/gtk.h>

#ifndef HAVE_LOCALTIME_R
#include <pthread.h>
#endif

#if DEBUG_WIDGET || DEBUG_WIDGET_MOST
#include <stdio.h>
#endif
#include <stdlib.h>
#include <string.h>

#include <time.h>


static void threadlist_view_class_init(ThreadlistViewClass *klass);
static void popup_menu_callback(gpointer data, guint action,
				GtkWidget *widget);
static void popup_menu_callback2(gpointer data, guint action,
				 GtkWidget *widget);
static void popup_menu_destructed_cb(gpointer data);
static void threadlist_view_init(ThreadlistView *view);

static void threadlist_view_finalize(GObject *object);
static void threadlist_view_destroy(GtkObject *object);

static gboolean threadlist_view_title_not_equals(GtkTreeModel *model,
						 int column,
						 const gchar *key,
						 GtkTreeIter *iter,
						 gpointer search_data);

static gboolean threadlist_view_move_cursor(GtkTreeView *tree_view,
					    GtkMovementStep step,
					    int count);

static void threadlist_view_do_on_selection(ThreadlistView *view,
					    int what_to_do);
static void threadlist_view_do_on_thread_at_cursor(ThreadlistView *view,
						   int what_to_do);

static gboolean threadlist_view_motion_notify(GtkWidget *widget,
					      GdkEventMotion *event);
static gboolean threadlist_view_leave_notify(GtkWidget *widget,
					     GdkEventCrossing *event);
static gboolean threadlist_view_button_press(GtkWidget *widget,
					     GdkEventButton *event);
static gboolean threadlist_view_button_release(GtkWidget *widget,
					       GdkEventButton *event);

static void decorate_threadlist_entry(GtkTreeViewColumn *tree_column,
				      GtkCellRenderer *renderer,
				      GtkTreeModel *tree_model,
				      GtkTreeIter *iter, gpointer view);
static void text_set_num_of_responses_verbose(GtkTreeViewColumn *tree_column,
					      GtkCellRenderer *renderer,
					      GtkTreeModel *tree_model,
					      GtkTreeIter *iter,
					      gpointer view);
static void text_set_cell_data_rank(GtkTreeViewColumn *tree_column,
				    GtkCellRenderer *renderer,
				    GtkTreeModel *tree_model,
				    GtkTreeIter *iter, gpointer view);
static void active_set_cell_data_flags(GtkTreeViewColumn *tree_column,
				       GtkCellRenderer *renderer,
				       GtkTreeModel *tree_model,
				       GtkTreeIter *iter,
				       gpointer unused);
static void text_set_cell_data_rank_diff(GtkTreeViewColumn *tree_column,
					 GtkCellRenderer *renderer,
					 GtkTreeModel *tree_model,
					 GtkTreeIter *iter,
					 gpointer unused);
static void text_set_cell_data_last_modified(GtkTreeViewColumn *tree_column,
					     GtkCellRenderer *renderer,
					     GtkTreeModel *tree_model,
					     GtkTreeIter *iter,
					     gpointer unused);


GType
threadlist_view_get_type(void)
{
  static GType tlv_type = 0;

  if (tlv_type == 0)
    {
      static const GTypeInfo tlv_info =
	{
	  sizeof(ThreadlistViewClass),
	  NULL, /* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc)threadlist_view_class_init,
	  NULL, /* class_finalize */
	  NULL, /* class_data */
	  sizeof(ThreadlistView),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)threadlist_view_init,
	};

      tlv_type = g_type_register_static(GTK_TYPE_TREE_VIEW,
					"ThreadlistView", &tlv_info, 0);
    }

  return tlv_type;
}


enum {
  OPEN_THREAD_VIEW_SIGNAL,

  THREAD_MOUSE_OVER_SIGNAL,
  THREAD_MOUSE_OUT_SIGNAL,

  TOGGLE_MARK_SIGNAL,
  TOGGLE_HIDE_SIGNAL,
  MARK_THREAD_SIGNAL,
  RESET_THREAD_SIGNAL,
  COPY_THREAD_URL_SIGNAL,
  ADVANCE_VIEW_SIGNAL,
  BACK_VIEW_SIGNAL,
  WRITE_RESPONSE_SIGNAL,

  CLOSE_THREADLIST_SIGNAL,

  INTERACTIVE_SEARCH_SIGNAL,

  DO_ON_SELECTION_SIGNAL,
  DO_ON_THREAD_AT_CURSOR_SIGNAL,

  LAST_SIGNAL
};


enum {
  MARK_THREAD,
  UNMARK_THREAD,
  TOGGLE_HIDDEN_STATE,
  ADVANCE_VIEW,
  BACK_VIEW,
  WRITE_RESPONSE,
};


enum {
  POPUP_MENU_OPEN_SELECTION = 1,
  POPUP_MENU_OPEN_SELECTION_IN_TAB,
  POPUP_MENU_OPEN_SELECTION_WITH_BROWSER,
  POPUP_MENU_MARK_SELECTION,
  POPUP_MENU_UNMARK_SELECTION,
  POPUP_MENU_TOGGLE_HIDE_SELECTION,
  POPUP_MENU_RESET_SELECTION,
  POPUP_MENU_COPY_THREAD_URL
};


static GtkTreeViewClass *parent_class = NULL;
static int threadlist_view_signals[LAST_SIGNAL] =
  { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
static GtkItemFactory *popup_menu_item_factory = NULL;
static GtkItemFactory *popup_menu_item_factory2 = NULL;
static GQuark list_entry_info_list_id;


static const char * const
default_foreground_color[NUMBER_OF_LIST_ENTRY_FOREGROUND_COLORS] =
  { NULL, "red", "red", "gray75" };


static const char * const
default_background_color[NUMBER_OF_LIST_ENTRY_BACKGROUND_COLORS] =
  { NULL, "lavender", "#ffe0e0", NULL };


static char *
list_entry_foreground_color[NUMBER_OF_LIST_ENTRY_FOREGROUND_COLORS] =
  { NULL, NULL, NULL, NULL };


static char *
list_entry_background_color[NUMBER_OF_LIST_ENTRY_BACKGROUND_COLORS] =
  { NULL, NULL, NULL, NULL };


void
threadlist_entry_set_foreground_color(ThreadlistEntryForeground color,
				      const char *color_name)
{
  g_return_if_fail(color < NUMBER_OF_LIST_ENTRY_FOREGROUND_COLORS);

  if (color_name == NULL)
    color_name = default_foreground_color[color];

  if (list_entry_foreground_color[color] != NULL)
    G_FREE(list_entry_foreground_color[color]);

  if (color_name == NULL)
    list_entry_foreground_color[color] = NULL;
  else
    list_entry_foreground_color[color] = G_STRDUP(color_name);
}


void
threadlist_entry_set_background_color(ThreadlistEntryBackground color,
				      const char *color_name)
{
  g_return_if_fail(color < NUMBER_OF_LIST_ENTRY_BACKGROUND_COLORS);

  if (color_name == NULL)
    color_name = default_background_color[color];

  if (list_entry_background_color[color] != NULL)
    G_FREE(list_entry_background_color[color]);

  if (color_name == NULL)
    list_entry_background_color[color] = NULL;
  else
    list_entry_background_color[color] = G_STRDUP(color_name);
}


#ifndef HAVE_LOCALTIME_R
static pthread_mutex_t localtime_lock;

#define LOCALTIME_LOCK					\
  if (pthread_mutex_lock(&localtime_lock) != 0)		\
    {							\
      fprintf(stderr, "Couldn't lock a mutex.\n");	\
      abort();						\
    }

#define LOCALTIME_UNLOCK				\
  if (pthread_mutex_unlock(&localtime_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't unlock a mutex.\n");	\
      abort();						\
    }

#endif


static void
threadlist_view_class_init(ThreadlistViewClass *klass)
{
  GObjectClass *o_class = (GObjectClass *)klass;
  GtkObjectClass *object_class = (GtkObjectClass *)klass;
  GtkWidgetClass *widget_class = (GtkWidgetClass *)klass;
  GtkTreeViewClass *tree_view_class = (GtkTreeViewClass *)klass;
  GtkBindingSet *binding_set;

  /* ݥåץåץ˥塼ν */
  GtkItemFactoryEntry menu_items[] =
    {
      {
	_("/_Open"),				/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback,			/* callback_func */
	POPUP_MENU_OPEN_SELECTION,		/* act */
	NULL					/* type */
      },
      {
	_("/Open in _Tab"),			/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback,			/* callback_func */
	POPUP_MENU_OPEN_SELECTION_IN_TAB,	/* act */
	NULL					/* type */
      },
      {
	_("/Open with _Web Browser"),		/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback,			/* callback_func */
	POPUP_MENU_OPEN_SELECTION_WITH_BROWSER,	/* act */
	NULL					/* type */
      },
      {
	_("/-------"),
	NULL,
	NULL,
	0,
	(char *)"<Separator>"
      },
      {
	_("/Mark(_*)"),				/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback,			/* callback_func */
	POPUP_MENU_MARK_SELECTION,		/* act */
	NULL					/* type */
      },
      {
	_("/_Unmark"),				/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback,			/* callback_func */
	POPUP_MENU_UNMARK_SELECTION,		/* act */
	NULL					/* type */
      },
      {
	_("/Toggle Hide(_D)"),			/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback,			/* callback_func */
	POPUP_MENU_TOGGLE_HIDE_SELECTION,	/* act */
	NULL					/* type */
      },
      {
	_("/-------"),
	NULL,
	NULL,
	0,
	(char *)"<Separator>"
      },
      {
	_("/Reset(_R)"),			/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback,			/* callback_func */
	POPUP_MENU_RESET_SELECTION,		/* act */
	NULL					/* type */
      },
      {
	_("/-------"),
	NULL,
	NULL,
	0,
	(char *)"<Separator>"
      },
      {
	_("/Copy Thread's URL"),		/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback,			/* callback_func */
	POPUP_MENU_COPY_THREAD_URL,		/* act */
	NULL					/* type */
      },
    };
  int nmenu_items = sizeof(menu_items) / sizeof(menu_items[0]);

  GtkItemFactoryEntry menu_items2[] =
    {
      {
	_("/_Open"),				/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback2,			/* callback_func */
	POPUP_MENU_OPEN_SELECTION,		/* act */
	NULL					/* type */
      },
      {
	_("/Open in _Tab"),			/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback2,			/* callback_func */
	POPUP_MENU_OPEN_SELECTION_IN_TAB,	/* act */
	NULL					/* type */
      },
      {
	_("/Open with _Web Browser"),		/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback2,			/* callback_func */
	POPUP_MENU_OPEN_SELECTION_WITH_BROWSER,	/* act */
	NULL					/* type */
      },
      {
	_("/-------"),
	NULL,
	NULL,
	0,
	(char *)"<Separator>"
      },
      {
	_("/Mark(_*)"),				/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback2,			/* callback_func */
	POPUP_MENU_MARK_SELECTION,		/* act */
	NULL					/* type */
      },
      {
	_("/_Unmark"),				/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback2,			/* callback_func */
	POPUP_MENU_UNMARK_SELECTION,		/* act */
	NULL					/* type */
      },
      {
	_("/Hide(_D)"),				/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback2,			/* callback_func */
	POPUP_MENU_TOGGLE_HIDE_SELECTION,	/* act */
	NULL					/* type */
      },
      {
	_("/-------"),
	NULL,
	NULL,
	0,
	(char *)"<Separator>"
      },
      {
	_("/Reset(_R)"),			/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback2,			/* callback_func */
	POPUP_MENU_RESET_SELECTION,		/* act */
	NULL					/* type */
      },
      {
	_("/-------"),
	NULL,
	NULL,
	0,
	(char *)"<Separator>"
      },
      {
	_("/Copy Thread's URL"),		/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback2,			/* callback_func */
	POPUP_MENU_COPY_THREAD_URL,		/* act */
	NULL					/* type */
      },
    };
  int nmenu_items2 = sizeof(menu_items2) / sizeof(menu_items2[0]);

  popup_menu_item_factory = gtk_item_factory_new(GTK_TYPE_MENU,
						 "<ThreadlistViewPopup>",
						 NULL);
  gtk_item_factory_create_items(popup_menu_item_factory,
				nmenu_items, menu_items, NULL);

  popup_menu_item_factory2 = gtk_item_factory_new(GTK_TYPE_MENU,
						  "<ThreadlistViewPopup2>",
						  NULL);
  gtk_item_factory_create_items(popup_menu_item_factory2,
				nmenu_items2, menu_items2, NULL);

  parent_class = g_type_class_peek_parent(klass);
  binding_set = gtk_binding_set_by_class(klass);


  /* GObject signals */
  o_class->finalize = threadlist_view_finalize;

  /* GtkObject signals */
  object_class->destroy = threadlist_view_destroy;

  gtk_widget_class_install_style_property(widget_class,
					  g_param_spec_string("fg_normal",
							      _("FG normal"),
							      _("Foreground color for ordinal list entry"),
							      NULL,
							      G_PARAM_READABLE));
  gtk_widget_class_install_style_property(widget_class,
					  g_param_spec_string("fg_emph",
							      _("FG emph"),
							      _("Foreground color for list entry to be emphasized"),
							      "red",
							      G_PARAM_READABLE));
  gtk_widget_class_install_style_property(widget_class,
					  g_param_spec_string("fg_strong",
							      _("FG strong"),
							      _("Foreground color for list entry to be shown strongly"),
							      NULL,
							      G_PARAM_READABLE));
  gtk_widget_class_install_style_property(widget_class,
					  g_param_spec_string("fg_hidden",
							      _("FG hidden"),
							      _("Foreground color for hidden list entry"),
							      "gray75",
							      G_PARAM_READABLE));
  gtk_widget_class_install_style_property(widget_class,
					  g_param_spec_string("bg_normal",
							      _("BG normal"),
							      _("Background color for ordinal list entry"),
							      NULL,
							      G_PARAM_READABLE));
  gtk_widget_class_install_style_property(widget_class,
					  g_param_spec_string("bg_emph",
							      _("BG emph"),
							      _("Boreground color for list entry to be emphasized"),
							      "lavender",
							      G_PARAM_READABLE));
  gtk_widget_class_install_style_property(widget_class,
					  g_param_spec_string("bg_strong",
							      _("BG strong"),
							      _("Background color for list entry to be shown strongly"),
							      "#ffe0e0",
							      G_PARAM_READABLE));
  gtk_widget_class_install_style_property(widget_class,
					  g_param_spec_string("bg_hidden",
							      _("BG hidden"),
							      _("Background color for hidden list entry"),
							      NULL,
							      G_PARAM_READABLE));


  /* GtkTreeView signals */
  tree_view_class->move_cursor = threadlist_view_move_cursor;

  threadlist_view_signals[OPEN_THREAD_VIEW_SIGNAL] =
    g_signal_new("open_thread_view",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistViewClass, open_thread_view),
		 NULL, NULL,
		 ochusha_marshal_VOID__OBJECT_BOOLEAN_BOOLEAN,
		 G_TYPE_NONE, 3,
		 OCHUSHA_TYPE_BBS_THREAD,
		 G_TYPE_BOOLEAN,
		 G_TYPE_BOOLEAN);

  threadlist_view_signals[THREAD_MOUSE_OVER_SIGNAL] =
    g_signal_new("thread_mouse_over",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistViewClass, thread_mouse_over),
		 NULL, NULL,
		 ochusha_marshal_VOID__OBJECT,
		 G_TYPE_NONE, 1,
		 OCHUSHA_TYPE_BBS_THREAD);
  threadlist_view_signals[THREAD_MOUSE_OUT_SIGNAL] =
    g_signal_new("thread_mouse_out",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistViewClass, thread_mouse_out),
		 NULL, NULL,
		 ochusha_marshal_VOID__OBJECT,
		 G_TYPE_NONE, 1,
		 OCHUSHA_TYPE_BBS_THREAD);

  threadlist_view_signals[TOGGLE_MARK_SIGNAL] =
    g_signal_new("toggle_mark",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistViewClass, toggle_mark),
		 NULL, NULL,
		 ochusha_marshal_VOID__OBJECT,
		 G_TYPE_NONE, 1,
		 OCHUSHA_TYPE_BBS_THREAD);
  threadlist_view_signals[TOGGLE_HIDE_SIGNAL] =
    g_signal_new("toggle_hide",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistViewClass, toggle_hide),
		 NULL, NULL,
		 ochusha_marshal_VOID__OBJECT,
		 G_TYPE_NONE, 1,
		 OCHUSHA_TYPE_BBS_THREAD);
  threadlist_view_signals[MARK_THREAD_SIGNAL] =
    g_signal_new("mark_thread",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistViewClass, mark_thread),
		 NULL, NULL,
		 ochusha_marshal_VOID__OBJECT_BOOLEAN,
		 G_TYPE_NONE, 2,
		 OCHUSHA_TYPE_BBS_THREAD,
		 G_TYPE_BOOLEAN);
  /* ɥۥå᤮*/
  threadlist_view_signals[RESET_THREAD_SIGNAL] =
    g_signal_new("reset_thread",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistViewClass, reset_thread),
		 NULL, NULL,
		 ochusha_marshal_VOID__OBJECT,
		 G_TYPE_NONE, 1,
		 OCHUSHA_TYPE_BBS_THREAD);
  threadlist_view_signals[COPY_THREAD_URL_SIGNAL] =
    g_signal_new("copy_thread_url",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistViewClass, copy_thread_url),
		 NULL, NULL,
		 ochusha_marshal_VOID__OBJECT,
		 G_TYPE_NONE, 1,
		 OCHUSHA_TYPE_BBS_THREAD);
  threadlist_view_signals[ADVANCE_VIEW_SIGNAL] =
    g_signal_new("advance_view",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistViewClass, advance_view),
		 NULL, NULL,
		 ochusha_marshal_BOOLEAN__OBJECT,
		 G_TYPE_BOOLEAN, 1,
		 OCHUSHA_TYPE_BBS_THREAD);
  threadlist_view_signals[BACK_VIEW_SIGNAL] =
    g_signal_new("back_view",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistViewClass, back_view),
		 NULL, NULL,
		 ochusha_marshal_BOOLEAN__OBJECT,
		 G_TYPE_BOOLEAN, 1,
		 OCHUSHA_TYPE_BBS_THREAD);
  threadlist_view_signals[WRITE_RESPONSE_SIGNAL] =
    g_signal_new("write_response",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistViewClass, write_response),
		 NULL, NULL,
		 ochusha_marshal_VOID__OBJECT,
		 G_TYPE_NONE, 1,
		 OCHUSHA_TYPE_BBS_THREAD);

  threadlist_view_signals[CLOSE_THREADLIST_SIGNAL] =
    g_signal_new("close_threadlist",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
		 G_STRUCT_OFFSET(ThreadlistViewClass, close_threadlist),
		 NULL, NULL,
		 ochusha_marshal_VOID__VOID,
		 G_TYPE_NONE, 0);

  threadlist_view_signals[INTERACTIVE_SEARCH_SIGNAL] =
    g_signal_new("interactive_search",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
		 G_STRUCT_OFFSET(ThreadlistViewClass, interactive_search),
		 NULL, NULL,
		 ochusha_marshal_VOID__INT,
		 G_TYPE_NONE, 1,
		 G_TYPE_INT);

  threadlist_view_signals[DO_ON_SELECTION_SIGNAL] =
    g_signal_new("do_on_selection",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
		 G_STRUCT_OFFSET(ThreadlistViewClass, do_on_selection),
		 NULL, NULL,
		 ochusha_marshal_VOID__INT,
		 G_TYPE_NONE, 1,
		 G_TYPE_INT);
  threadlist_view_signals[DO_ON_THREAD_AT_CURSOR_SIGNAL] =
    g_signal_new("do_on_thread_at_cursor",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
		 G_STRUCT_OFFSET(ThreadlistViewClass, do_on_thread_at_cursor),
		 NULL, NULL,
		 ochusha_marshal_VOID__INT,
		 G_TYPE_NONE, 1,
		 G_TYPE_INT);


  /* Key bindings */
  gtk_binding_entry_add_signal(binding_set, GDK_o, 0, "do_on_selection", 1,
			       G_TYPE_INT, POPUP_MENU_OPEN_SELECTION);
  gtk_binding_entry_add_signal(binding_set, GDK_t, 0, "do_on_selection", 1,
			       G_TYPE_INT, POPUP_MENU_OPEN_SELECTION_IN_TAB);
  gtk_binding_entry_add_signal(binding_set, GDK_r, 0, "do_on_selection", 1,
			       G_TYPE_INT, POPUP_MENU_RESET_SELECTION);
  gtk_binding_entry_add_signal(binding_set, GDK_asterisk, 0,
			       "do_on_thread_at_cursor", 1,
			       G_TYPE_INT, MARK_THREAD);
  gtk_binding_entry_add_signal(binding_set, GDK_u, 0,
			       "do_on_thread_at_cursor", 1,
			       G_TYPE_INT, UNMARK_THREAD);
  gtk_binding_entry_add_signal(binding_set, GDK_d, 0,
			       "do_on_thread_at_cursor", 1,
			       G_TYPE_INT, TOGGLE_HIDDEN_STATE);
  gtk_binding_entry_add_signal(binding_set, GDK_space, 0,
			       "do_on_thread_at_cursor", 1,
			       G_TYPE_INT, ADVANCE_VIEW);
  gtk_binding_entry_add_signal(binding_set, GDK_BackSpace, 0,
			       "do_on_thread_at_cursor", 1,
			       G_TYPE_INT, BACK_VIEW);
  gtk_binding_entry_add_signal(binding_set, GDK_a, 0,
			       "do_on_thread_at_cursor", 1,
			       G_TYPE_INT, WRITE_RESPONSE);

  gtk_binding_entry_add_signal(binding_set, GDK_Up, 0,
			       "move_cursor", 2,
			       G_TYPE_ENUM, GTK_MOVEMENT_DISPLAY_LINES,
			       G_TYPE_INT, -1);
  gtk_binding_entry_add_signal(binding_set, GDK_Down, 0,
			       "move_cursor", 2,
			       G_TYPE_ENUM, GTK_MOVEMENT_DISPLAY_LINES,
			       G_TYPE_INT, 1);

  gtk_binding_entry_add_signal(binding_set, GDK_p, GDK_CONTROL_MASK,
			       "move_cursor", 2,
			       G_TYPE_ENUM, GTK_MOVEMENT_DISPLAY_LINES,
			       G_TYPE_INT, -1);
  gtk_binding_entry_add_signal(binding_set, GDK_n, GDK_CONTROL_MASK,
			       "move_cursor", 2,
			       G_TYPE_ENUM, GTK_MOVEMENT_DISPLAY_LINES,
			       G_TYPE_INT, 1);

  gtk_binding_entry_add_signal(binding_set, GDK_v, GDK_CONTROL_MASK,
			       "move_cursor", 2,
			       G_TYPE_ENUM, GTK_MOVEMENT_PAGES,
			       G_TYPE_INT, 1);
  gtk_binding_entry_add_signal(binding_set, GDK_v, GDK_MOD1_MASK,
			       "move_cursor", 2,
			       G_TYPE_ENUM, GTK_MOVEMENT_PAGES,
			       G_TYPE_INT, -1);
  gtk_binding_entry_add_signal(binding_set, GDK_v, GDK_MOD4_MASK,
			       "move_cursor", 2,
			       G_TYPE_ENUM, GTK_MOVEMENT_PAGES,
			       G_TYPE_INT, -1);

  gtk_binding_entry_add_signal(binding_set, GDK_f, GDK_CONTROL_MASK,
			       "interactive_search", 1,
			       G_TYPE_INT, THREADLIST_SEARCH_ACTION_FORWARD);

  gtk_binding_entry_add_signal(binding_set, GDK_s, GDK_CONTROL_MASK,
			       "interactive_search", 1,
			       G_TYPE_INT, THREADLIST_SEARCH_ACTION_FORWARD);
  gtk_binding_entry_add_signal(binding_set, GDK_r, GDK_CONTROL_MASK,
			       "interactive_search", 1,
			       G_TYPE_INT, THREADLIST_SEARCH_ACTION_BACKWARD);
  gtk_binding_entry_add_signal(binding_set, GDK_S,
			       GDK_CONTROL_MASK | GDK_SHIFT_MASK,
			       "interactive_search", 1,
			       G_TYPE_INT,
			       THREADLIST_SEARCH_ACTION_REGEXP_FORWARD);
  gtk_binding_entry_add_signal(binding_set, GDK_R,
			       GDK_CONTROL_MASK | GDK_SHIFT_MASK,
			       "interactive_search", 1,
			       G_TYPE_INT,
			       THREADLIST_SEARCH_ACTION_REGEXP_BACKWARD);

  gtk_binding_entry_add_signal(binding_set, GDK_W, GDK_CONTROL_MASK,
			       "close_threadlist", 0);

  gtk_binding_entry_add_signal(binding_set, GDK_Escape, 0,
			       "interactive_search", 1,
			       G_TYPE_INT,
			       THREADLIST_SEARCH_ACTION_TERMINATE);
  gtk_binding_entry_add_signal(binding_set, GDK_g, GDK_CONTROL_MASK,
			       "interactive_search", 1,
			       G_TYPE_INT,
			       THREADLIST_SEARCH_ACTION_TERMINATE);

  widget_class->leave_notify_event = threadlist_view_leave_notify;
  widget_class->motion_notify_event = threadlist_view_motion_notify;
  widget_class->button_press_event = threadlist_view_button_press;
  widget_class->button_release_event = threadlist_view_button_release;

  klass->open_thread_view = NULL;

  klass->thread_mouse_over = NULL;
  klass->thread_mouse_out = NULL;

  klass->toggle_mark = NULL;
  klass->toggle_hide = NULL;
  klass->mark_thread = NULL;
  klass->advance_view = NULL;
  klass->back_view = NULL;
  klass->write_response = NULL;

  klass->close_threadlist = NULL;
  klass->interactive_search = NULL;

  klass->do_on_selection = threadlist_view_do_on_selection;
  klass->do_on_thread_at_cursor = threadlist_view_do_on_thread_at_cursor;

  list_entry_info_list_id
    = g_quark_from_static_string("ThreadlistView::ListEntryInfoList");

#ifndef HAVE_LOCALTIME_R
  if (pthread_mutex_init(&localtime_lock, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a mutex.\n");
      abort();
    }
#endif
}


typedef enum
{
  THREADLIST_COUNT = 0,
  THREADLIST_RANK,
  THREADLIST_VISUAL_PARAMETER,
  THREADLIST_TITLE,
  THREADLIST_N_RESPONSES,
  THREADLIST_N_RESPONSES_GOT,
  THREADLIST_N_RESPONSES_UNGOT,
  THREADLIST_N_RESPONSES_SHOWN,
  THREADLIST_N_RESPONSES_UNREAD,
  THREADLIST_RANK_DIFFERENCE,
  THREADLIST_LAST_MODIFIED,
  THREADLIST_WEIGHT,
  THREADLIST_DATA,
  THREADLIST_N_COLUMNS
} OchushaThreadlistItem;


enum
{
  TARGET_URI_LIST,
  TARGET_UTF8_STRING,
  TARGET_STRING,
  TARGET_TEXT,
  TARGET_COMPOUND_TEXT
};


static void
collect_uri_list(GtkTreeModel *model, GtkTreePath *path,
		 GtkTreeIter *iter, gchar **uri_list_p)
{
  gpointer data = NULL;
  gtk_tree_model_get(model, iter, THREADLIST_DATA, &data, -1);

  if (OCHUSHA_IS_BBS_THREAD(data))
    {
      if (*uri_list_p == NULL)
	*uri_list_p = g_strdup_printf("%s\r\n", ochusha_bbs_thread_get_url(OCHUSHA_BBS_THREAD(data)));
      else
	{
	  char *prev_list = *uri_list_p;
	  *uri_list_p = g_strdup_printf("%s%s\r\n", prev_list, ochusha_bbs_thread_get_url(OCHUSHA_BBS_THREAD(data)));
	  g_free(prev_list);
	}
    }
}


static void
collect_utf8_text(GtkTreeModel *model, GtkTreePath *path,
		  GtkTreeIter *iter, gchar **utf8_text_p)
{
  gpointer data = NULL;
  gtk_tree_model_get(model, iter, THREADLIST_DATA, &data, -1);

  if (OCHUSHA_IS_BBS_THREAD(data))
    {
      OchushaBBSThread *thread = OCHUSHA_BBS_THREAD(data);
      OchushaBulletinBoard *board = thread->board;
      if (*utf8_text_p == NULL)
	*utf8_text_p = g_strdup_printf(_("%s @ %s board\n%s\n"),
				       ochusha_bbs_thread_get_title(thread),
				       ochusha_bulletin_board_get_name(board),
				       ochusha_bbs_thread_get_url(thread));
      else
	{
	  char *prev_text = *utf8_text_p;
	  *utf8_text_p = g_strdup_printf(_("%s%s @ %s board\n%s\n"),
					 prev_text,
					 ochusha_bbs_thread_get_title(thread),
					 ochusha_bulletin_board_get_name(board),
					 ochusha_bbs_thread_get_url(thread));
	  g_free(prev_text);
	}
    }
}


static void
drag_data_get_cb(GtkWidget *widget, GdkDragContext *context,
		 GtkSelectionData *selection_data, guint info, guint time,
		 ThreadlistView *view)
{
  switch (info)
    {
    case TARGET_URI_LIST:
      {
	char *uri_list = NULL;
	gtk_tree_selection_selected_foreach(view->selection,
				(GtkTreeSelectionForeachFunc)collect_uri_list,
				&uri_list);
	if (uri_list != NULL)
	  {
	    gtk_selection_data_set(selection_data, selection_data->target,
				   8, uri_list, strlen(uri_list));
	    g_free(uri_list);
	  }
	break;
      }

    case TARGET_UTF8_STRING:
    case TARGET_STRING:
    case TARGET_TEXT:
    case TARGET_COMPOUND_TEXT:
      {
	char *utf8_text = NULL;
	gtk_tree_selection_selected_foreach(view->selection,
				(GtkTreeSelectionForeachFunc)collect_utf8_text,
				&utf8_text);
	if (utf8_text != NULL)
	  {
	    gtk_selection_data_set_text(selection_data, utf8_text, -1);
	    g_free(utf8_text);
	  }
	break;
      }
    }

  if (view->last_event_path != NULL)
    {
      gtk_tree_path_free(view->last_event_path);
      view->last_event_path = NULL;
    }
}


static void
threadlist_view_init(ThreadlistView *view)
{
  static const GtkTargetEntry drag_types[] =
    {
      { "text/uri-list", 0, TARGET_URI_LIST },
      { "UTF8_STRING", 0, TARGET_UTF8_STRING },
      { "STRING", 0, 0 },
      { "TEXT", 0, 0 },
      { "COMPOUND_TEXT", 0, 0 }
    };
  static int n_drag_types = sizeof(drag_types)/sizeof(drag_types[0]);

  GTK_WIDGET_SET_FLAGS(view, GTK_CAN_FOCUS | GTK_RECEIVES_DEFAULT);

  view->selection = gtk_tree_view_get_selection(&view->view);
  gtk_tree_selection_set_mode(view->selection, GTK_SELECTION_MULTIPLE);

  view->model = NULL;
  view->model_is_opened = FALSE;

  gtk_tree_view_set_search_column(&view->view, THREADLIST_TITLE);
  gtk_tree_view_set_search_equal_func(&view->view,
				      threadlist_view_title_not_equals,
				      NULL, NULL);
  gtk_tree_view_set_enable_search(&view->view, TRUE);
  view->use_native_search = TRUE;
  view->being_searched = FALSE;

  /* D&Dμ¸ */
  gtk_tree_view_enable_model_drag_source(&view->view,
					 GDK_BUTTON1_MASK,
					 drag_types, n_drag_types,
					 GDK_ACTION_COPY);
  g_signal_connect(view, "drag_data_get",
		   G_CALLBACK(drag_data_get_cb), view);
}


static void
threadlist_view_finalize(GObject *object)
{
  ThreadlistView *view = (ThreadlistView *)object;

#if DEBUG_WIDGET_MOST
  fprintf(stderr, "threadlist_view_finalize: view->ref_count=%d\n",
	  object->ref_count);
  fprintf(stderr, "threadlist_view_finalize: model->ref_count=%d\n",
	  G_OBJECT(view->model)->ref_count);
#endif
  if (view->model != NULL)
    {
      GObject *model = G_OBJECT(view->model);
      view->model = NULL;
      g_object_unref(model);
    }

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


static void
threadlist_view_destroy(GtkObject *object)
{
  ThreadlistView *view = THREADLIST_VIEW(object);
#if DEBUG_WIDGET_MOST
  fprintf(stderr, "threadlist_view_destroy: object->ref_count=%d\n",
	  G_OBJECT(object)->ref_count);
#endif

  if (view->tail_row != NULL)
    {
      gtk_tree_row_reference_free(view->tail_row);
      view->tail_row = NULL;
    }

  if (GTK_OBJECT_CLASS(parent_class)->destroy)
    (*GTK_OBJECT_CLASS(parent_class)->destroy)(object);
}


typedef struct _ThreadlistInfo
{
  GtkTreeRowReference *reference;
  ThreadlistView *view;
} ThreadlistInfo;


static void
threadlist_info_free(gpointer pointer)
{
  ThreadlistInfo *info = (ThreadlistInfo *)pointer;
  gtk_tree_row_reference_free(info->reference);
  info->reference = NULL;
  G_FREE(info);
}


static int
compare_info(gconstpointer list_data, gconstpointer user_data)
{
  return ((ThreadlistInfo *)list_data)->view != user_data;
}


static void
view_destroy_cb(ThreadlistView *view, OchushaBBSThread *thread)
{
  GSList *info_list;
  GSList *list_entry;

  info_list = g_object_get_qdata(G_OBJECT(thread), list_entry_info_list_id);
  list_entry = g_slist_find_custom(info_list, view, compare_info);
  if (list_entry != NULL)
    {
      info_list = g_slist_remove_link(info_list, list_entry);
      g_object_set_qdata(G_OBJECT(thread), list_entry_info_list_id, info_list);
      threadlist_info_free(list_entry->data);
      g_slist_free_1(list_entry);
    }
}


static ThreadlistInfo *
ensure_threadlist_info0(OchushaBBSThread *thread, ThreadlistView *view)
{
  GSList *info_list = g_object_get_qdata(G_OBJECT(thread),
					 list_entry_info_list_id);
  GSList *list_entry = g_slist_find_custom(info_list, view, compare_info);
  ThreadlistInfo *info;

  if (list_entry == NULL)
    {
      info = G_NEW0(ThreadlistInfo, 1);
      info->view = view;
      info_list = g_slist_append(info_list, info);
      g_object_set_qdata(G_OBJECT(thread), list_entry_info_list_id, info_list);
      g_signal_connect(G_OBJECT(view), "destroy",
		       G_CALLBACK(view_destroy_cb), thread);
    }
  else
    info = list_entry->data;

  return info;
}


static ThreadlistInfo *
get_threadlist_info(OchushaBBSThread *thread, ThreadlistView *view)
{
  GSList *info_list = g_object_get_qdata(G_OBJECT(thread),
					 list_entry_info_list_id);
  GSList *list_entry = g_slist_find_custom(info_list, view, compare_info);

  if (list_entry != NULL)
    return list_entry->data;

  return NULL;
}


static gboolean
threadlist_view_title_not_equals(GtkTreeModel *model, int column,
				 const gchar *key, GtkTreeIter *iter,
				 gpointer search_data)
{
  const gchar *title;
  gchar *normalized_key;
  gchar *case_normalized_key;
  gchar *normalized_title;
  gchar *case_normalized_title;
  gboolean result;
#if DEBUG_SEARCH
  char *native_string;
#endif

#if DEBUG_SEARCH
  if (key != NULL)
    {
      native_string = convert_string(utf8_to_native, key, -1);
      fprintf(stderr,
	      "threadlist_view_title_not_equals(): column=%d, key=\"%s\":",
	      column, key);
      G_FREE(native_string);
    }
#endif
  if (key == NULL)
    return FALSE;

  gtk_tree_model_get(model, iter, THREADLIST_TITLE, &title, -1);
#if DEBUG_SEARCH
  if (title != NULL)
    {
      native_string = convert_string(utf8_to_native, title, -1);
      fprintf(stderr, " title at iter=\"%s\"\n", native_string);
      G_FREE(native_string);
    }
  else
    fprintf(stderr, " title at iter=NULL\n");
#endif
  if (title == NULL)
    return FALSE;

  normalized_key = g_utf8_normalize(key, -1, G_NORMALIZE_ALL);
  case_normalized_key = g_utf8_casefold(normalized_key, -1);

  normalized_title = g_utf8_normalize(title, -1, G_NORMALIZE_ALL);
  case_normalized_title = g_utf8_casefold(normalized_title, -1);

  result = (strstr(case_normalized_title, case_normalized_key) == NULL);

  g_free(normalized_key);
  g_free(case_normalized_key);
  g_free(normalized_title);
  g_free(case_normalized_title);

  return result;
}


static gboolean
threadlist_view_move_cursor(GtkTreeView *tree_view, GtkMovementStep step,
			    int count)
{
  ThreadlistView *view = THREADLIST_VIEW(tree_view);
  gboolean result = FALSE;

  MAIN_THREAD_EVENT_CALLBACK();

  /*
   * use_native_searchǤʤ硢ġʤ饫ưѤ
   * ʥФ
   */
#if DEBUG_WIDGET_MOST
  fprintf(stderr, "view->being_searched=%d\n", view->being_searched);
#endif
  if (!view->use_native_search && view->being_searched
      && step == GTK_MOVEMENT_DISPLAY_LINES)
    {
      g_signal_emit(G_OBJECT(view),
		    threadlist_view_signals[INTERACTIVE_SEARCH_SIGNAL],
		    0,
		    (count == 1
		     ? THREADLIST_SEARCH_ACTION_FORWARD
		     : THREADLIST_SEARCH_ACTION_BACKWARD),
		    &result);
      ENTER_MAIN_EVENT_LOOP();
      return TRUE;
    }

  ENTER_MAIN_EVENT_LOOP();
  return (*parent_class->move_cursor)(&view->view, step, count);
}


static void
find_initial_path_helper(GtkTreeModel *model, GtkTreePath *path,
			 GtkTreeIter *iter, gpointer data)
{
  GtkTreePath **initial_path = (GtkTreePath **)data;
  if (*initial_path != NULL)
    return;

  *initial_path = gtk_tree_path_copy(path);
  return;
}


static GtkTreePath *
threadlist_view_get_current_path(ThreadlistView *view)
{
  GtkTreePath *initial_path = NULL;
  GtkTreeIter iter;

  gtk_tree_view_get_cursor(&view->view, &initial_path, NULL);
  if (initial_path != NULL)
    return initial_path;

  gtk_widget_grab_focus(GTK_WIDGET(view));
  gtk_tree_view_get_cursor(&view->view, &initial_path, NULL);
  if (initial_path != NULL)
    return initial_path;

  gtk_tree_selection_selected_foreach(view->selection,
				      find_initial_path_helper, &initial_path);
  if (initial_path != NULL)
    return initial_path;

  if (!gtk_tree_model_get_iter_first(view->model, &iter))
    return NULL;

  return gtk_tree_model_get_path(view->model, &iter);
}


void
threadlist_view_set_being_searched(ThreadlistView *view, gboolean being)
{
  g_return_if_fail(IS_THREADLIST_VIEW(view));

  view->being_searched = (being ? 1 : 0);
}


gboolean
threadlist_view_find_next(ThreadlistView *view,
			  ThreadlistSearchDirection direction,
			  gboolean enable_wrap, gboolean include_current_pos,
			  gpointer search_data)
{
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreePath *initial_path;
  GtkTreePath *path = NULL;
  gboolean advance_iter = FALSE;

  g_return_val_if_fail(IS_THREADLIST_VIEW(view), FALSE);
  g_return_val_if_fail(search_data != NULL, FALSE);
  g_return_val_if_fail(view->title_match_func != NULL
		       && !view->use_native_search, FALSE);

  view->being_searched = TRUE;

  selection = view->selection;
  initial_path = threadlist_view_get_current_path(view);
  if (initial_path == NULL)
    return FALSE;

  gtk_tree_selection_unselect_all(selection);

  if (!gtk_tree_model_get_iter(view->model, &iter, initial_path))
    {
#if DEBUG_WIDGET_MOST
      fprintf(stderr, "Couldn't get iter.\n");
#endif
      goto no_match_found;
    }

  if (!include_current_pos)
    advance_iter = TRUE;

  while (TRUE)
    {
      const gchar *title;
      if (advance_iter)
	{
	  if (direction == THREADLIST_SEARCH_DIRECTION_FORWARD)
	    {
	      if (!gtk_tree_model_iter_next(view->model, &iter))
		{
		  if (view->last_search_position == 1 && enable_wrap)
		    {
		      view->last_search_position = 0;
		      if (!gtk_tree_model_get_iter_first(view->model, &iter))
			{
#if DEBUG_WIDGET_MOST
			  fprintf(stderr, "Couldn't get first iter.\n");
#endif
			  goto no_match_found;
			}
		    }
		  else
		    {
		      {
			view->last_search_position = 1;
			goto no_match_found;
		      }
		    }
		}
	    }
	  else
	    {
	      if (path == NULL)
		path = gtk_tree_path_copy(initial_path);

	      if (path == NULL)
		goto no_match_found;

	      if (!gtk_tree_path_prev(path))
		{
		  if (view->last_search_position == -1 && enable_wrap)
		    {
		      view->last_search_position = 0;
		      if (view->tail_row == NULL)
			{
#if DEBUG_WIDGET_MOST
			  fprintf(stderr, "Couldn't get last iter.\n");
#endif
			  goto no_match_found;
			}
		      if (path != NULL)
			gtk_tree_path_free(path);
		      path = gtk_tree_row_reference_get_path(view->tail_row);
		    }
		  else
		    {
		      view->last_search_position = -1;
		      goto no_match_found;
		    }
		}

	      if (!gtk_tree_model_get_iter(view->model, &iter, path))
		{
#if DEBUG_WIDGET_MOST
		  fprintf(stderr, "Couldn't get prev iter\n");
#endif
		  goto no_match_found;
		}
	    }
	}

      gtk_tree_model_get(view->model, &iter, THREADLIST_TITLE, &title, -1);
      if (title == NULL)
	{
#if DEBUG_WIDGET_MOST
	  fprintf(stderr, "title is NULL\n");
#endif
	  goto no_match_found;
	}

      if ((*view->title_match_func)(title, search_data))
	{
	  /* found */
	  gtk_tree_path_free(initial_path);
	  if (path != NULL)
	    gtk_tree_path_free(path);
	  path = gtk_tree_model_get_path(view->model, &iter);
	  gtk_tree_view_scroll_to_cell(&view->view, path, NULL,
				       TRUE, 0.5, 0.0);
	  gtk_tree_selection_select_iter(selection, &iter);
	  gtk_tree_view_set_cursor(&view->view, path, NULL, FALSE);
	  gtk_tree_path_free(path);

	  return TRUE;
	}

      advance_iter = TRUE;
    }

 no_match_found:
  if (path != NULL)
    gtk_tree_path_free(path);

  gtk_tree_view_set_cursor(&view->view, initial_path, NULL, FALSE);
  gtk_tree_path_free(initial_path);

  return FALSE;
}


void
threadlist_view_set_enable_native_search(ThreadlistView *view,
					 gboolean enable)
{
  g_return_if_fail(IS_THREADLIST_VIEW(view));

  gtk_tree_view_set_enable_search(&view->view, enable);
  view->use_native_search = enable;
}


gboolean
threadlist_view_get_enable_native_search(ThreadlistView *view)
{
  g_return_val_if_fail(IS_THREADLIST_VIEW(view), FALSE);

  return gtk_tree_view_get_enable_search(&view->view);
}


static gboolean
threadlist_view_interactive_search(ThreadlistView *view,
				   ThreadlistSearchAction search_action)
{
  gboolean result;

  MAIN_THREAD_EVENT_CALLBACK();

  if (view->use_native_search)
    {
      gtk_widget_grab_focus(GTK_WIDGET(view));

      ENTER_MAIN_EVENT_LOOP();
      return (*parent_class->start_interactive_search)(&view->view);
    }
  else
    {
      view->being_searched = TRUE;
      g_signal_emit(G_OBJECT(view),
		    threadlist_view_signals[INTERACTIVE_SEARCH_SIGNAL],
		    0,
		    search_action,
		    &result);
    }

  ENTER_MAIN_EVENT_LOOP();
  return result;
}


gboolean
threadlist_view_start_interactive_search(ThreadlistView *view)
{
  g_return_val_if_fail(IS_THREADLIST_VIEW(view), FALSE);

  return threadlist_view_interactive_search(view,
					THREADLIST_SEARCH_DIRECTION_FORWARD);
}


void
threadlist_view_set_title_match_func(ThreadlistView *view,
				     ThreadlistViewTitleMatchFunc *func)
{
  g_return_if_fail(IS_THREADLIST_VIEW(view));
  view->title_match_func = func;
}


ThreadlistViewTitleMatchFunc *
threadlist_view_get_title_match_func(ThreadlistView *view)
{
  g_return_val_if_fail(IS_THREADLIST_VIEW(view), NULL);
  return view->title_match_func;
}


typedef struct _DoOnSelectionArgs
{
  ThreadlistView *view;
  int what_to_do;
  GSList *hidden_threads;
} DoOnSelectionArgs;


static void
do_on_selection_helper(GtkTreeModel *model, GtkTreePath *path,
		       GtkTreeIter *iter, DoOnSelectionArgs *args)
{
  OchushaBBSThread *thread;
  int what_to_do = args->what_to_do;

  gtk_tree_model_get(model, iter,
		     THREADLIST_DATA, &thread, -1);
  if (thread == NULL)
    return;	/* ͭʤǰΤ*/

  switch (what_to_do)
    {
    case POPUP_MENU_OPEN_SELECTION:
      /* 2ܰʹߤ϶Ū˥֤ǳ */
      args->what_to_do = POPUP_MENU_OPEN_SELECTION_IN_TAB;

      /* FALL THROUGH */
    case POPUP_MENU_OPEN_SELECTION_IN_TAB:
      g_signal_emit(G_OBJECT(args->view),
		    threadlist_view_signals[OPEN_THREAD_VIEW_SIGNAL],
		    0,
		    thread,
		    what_to_do == POPUP_MENU_OPEN_SELECTION_IN_TAB,
		    FALSE);
      break;

    case POPUP_MENU_OPEN_SELECTION_WITH_BROWSER:
      g_signal_emit(G_OBJECT(args->view),
		    threadlist_view_signals[OPEN_THREAD_VIEW_SIGNAL],
		    0,
		    thread,
		    FALSE,
		    TRUE);
      break;

    case POPUP_MENU_MARK_SELECTION:
      g_signal_emit(G_OBJECT(args->view),
		    threadlist_view_signals[MARK_THREAD_SIGNAL],
		    0,
		    thread,
		    TRUE);
	  break;

    case POPUP_MENU_UNMARK_SELECTION:
      g_signal_emit(G_OBJECT(args->view),
		    threadlist_view_signals[MARK_THREAD_SIGNAL],
		    0,
		    thread,
		    FALSE);
	  break;

    case POPUP_MENU_TOGGLE_HIDE_SELECTION:
      g_signal_emit(G_OBJECT(args->view),
		    threadlist_view_signals[TOGGLE_HIDE_SIGNAL],
		    0,
		    thread);
      if (args->view->remove_hidden_thread)
	{
	  args->hidden_threads
	    = g_slist_append(args->hidden_threads,
			     gtk_tree_row_reference_new(model, path));
	}
      break;

    case POPUP_MENU_RESET_SELECTION:
      g_signal_emit(G_OBJECT(args->view),
		    threadlist_view_signals[RESET_THREAD_SIGNAL],
		    0,
		    thread);
      break;

    case POPUP_MENU_COPY_THREAD_URL:
      g_signal_emit(G_OBJECT(args->view),
		    threadlist_view_signals[COPY_THREAD_URL_SIGNAL],
		    0,
		    thread);
      break;
    }
}


static void
remove_row(gpointer list_data, gpointer user_data)
{
  GtkTreeRowReference *reference = (GtkTreeRowReference *)list_data;
  GtkListStore *store = (GtkListStore *)user_data;
  GtkTreePath *path = gtk_tree_row_reference_get_path(reference);
  GtkTreeIter iter;

  if (!gtk_tree_model_get_iter(GTK_TREE_MODEL(store), &iter, path))
    goto done;

  gtk_list_store_remove(store, &iter);

 done:
  if (path != NULL)
    gtk_tree_path_free(path);
  gtk_tree_row_reference_free(reference);
}


static void
threadlist_view_do_on_selection(ThreadlistView *view, int what_to_do)
{
  DoOnSelectionArgs args = { view, what_to_do, NULL };

  MAIN_THREAD_EVENT_CALLBACK();

  gtk_tree_selection_selected_foreach(view->selection,
			(GtkTreeSelectionForeachFunc)do_on_selection_helper,
			&args);

  if (args.hidden_threads != NULL)
    {
      g_slist_foreach(args.hidden_threads, remove_row, view->model);
      g_slist_free(args.hidden_threads);
    }

  ENTER_MAIN_EVENT_LOOP();
}


static void
threadlist_view_do_on_thread_at_cursor(ThreadlistView *view, int what_to_do)
{
  GtkTreeIter iter;
  GtkTreePath *path = NULL;
  gpointer data = NULL;

  gtk_tree_view_get_cursor(&view->view, &path, NULL);
  if (path == NULL)
    return;

  if (!gtk_tree_model_get_iter(view->model, &iter, path))
    goto done;

  MAIN_THREAD_EVENT_CALLBACK();

  gtk_tree_model_get(view->model, &iter, THREADLIST_DATA, &data, -1);

  if (data != NULL)
    {
      gboolean result = FALSE;
      gtk_tree_selection_unselect_all(view->selection);
      (*parent_class->select_cursor_row)(&view->view, FALSE);

      switch (what_to_do)
	{
	case MARK_THREAD:
	  g_signal_emit(G_OBJECT(view),
			threadlist_view_signals[MARK_THREAD_SIGNAL],
			0,
			data,
			TRUE);
	  break;

	case UNMARK_THREAD:
	  g_signal_emit(G_OBJECT(view),
			threadlist_view_signals[MARK_THREAD_SIGNAL],
			0,
			data,
			FALSE);
	  break;

	case TOGGLE_HIDDEN_STATE:
	  g_signal_emit(G_OBJECT(view),
			threadlist_view_signals[TOGGLE_HIDE_SIGNAL],
			0,
			data);
	  break;

	case ADVANCE_VIEW:
	  g_signal_emit(G_OBJECT(view),
			threadlist_view_signals[ADVANCE_VIEW_SIGNAL],
			0,
			data,
			&result);
	  break;

	case BACK_VIEW:
	  g_signal_emit(G_OBJECT(view),
			threadlist_view_signals[BACK_VIEW_SIGNAL],
			0,
			data,
			&result);
	  break;

	case WRITE_RESPONSE:
	  g_signal_emit(G_OBJECT(view),
			threadlist_view_signals[WRITE_RESPONSE_SIGNAL],
			0,
			data);
	  break;
	}
    }
  ENTER_MAIN_EVENT_LOOP();

 done:
  gtk_tree_path_free(path);
}


static void
popup_menu_callback(gpointer data, guint action, GtkWidget *widget)
{
  ThreadlistView *view = gtk_item_factory_popup_data(popup_menu_item_factory);
#if DEBUG_GUI_MOST
  fprintf(stderr, "popup_menu_callback: view=%p, action=%d\n",
	  view, action);
#endif
  /* MAIN_THREAD_EVENT_CALLBACK(); */
  /*
   * threadlist_view_do_on_selection()ϥ٥ȤľܤƤФΤǡ
   * äб
   */
  threadlist_view_do_on_selection(view, action);

  /* ENTER_MAIN_EVENT_LOOP(); */
}


static void
popup_menu_callback2(gpointer data, guint action, GtkWidget *widget)
{
  ThreadlistView *view = gtk_item_factory_popup_data(popup_menu_item_factory2);
#if DEBUG_GUI_MOST
  fprintf(stderr, "popup_menu_callback: view=%p, action=%d\n",
	  view, action);
#endif
  /* MAIN_THREAD_EVENT_CALLBACK(); */
  /*
   * threadlist_view_do_on_selection()ϥ٥ȤľܤƤФΤǡ
   * äб
   */
  threadlist_view_do_on_selection(view, action);

  /* ENTER_MAIN_EVENT_LOOP(); */
}


static void
popup_menu_destructed_cb(gpointer data)
{
#if DEBUG_GUI_MOST
  ThreadlistView *view = THREADLIST_VIEW(data);
  fprintf(stderr, "popup_menu_destructed_cb: data=%p\n", view);
#endif
}


GtkWidget *
threadlist_view_new(void)
{
  return GTK_WIDGET(g_object_new(THREADLIST_VIEW_TYPE, NULL));
}


gboolean
threadlist_view_get_remove_hidden_thread(ThreadlistView *view)
{
  g_return_val_if_fail(IS_THREADLIST_VIEW(view), FALSE);
  return view->remove_hidden_thread;
}


void
threadlist_view_set_remove_hidden_thread(ThreadlistView *view, gboolean remove)
{
  g_return_if_fail(IS_THREADLIST_VIEW(view));
  view->remove_hidden_thread = remove;
}


gboolean
threadlist_view_get_default_open_in_tab(ThreadlistView *view)
{
  g_return_val_if_fail(IS_THREADLIST_VIEW(view), FALSE);
  return view->default_open_in_tab;
}


void
threadlist_view_set_default_open_in_tab(ThreadlistView *view,
					gboolean default_open_in_tab)
{
  g_return_if_fail(IS_THREADLIST_VIEW(view));
  view->default_open_in_tab = default_open_in_tab;
}


static gboolean
threadlist_view_motion_notify(GtkWidget *widget, GdkEventMotion *event)
{
  ThreadlistView *view = THREADLIST_VIEW(widget);
  GtkTreeIter iter;
  GtkTreePath *path = NULL;
  GtkTreeViewColumn *column;
  gpointer data = NULL;
  gboolean result = FALSE;

  if (!gtk_tree_view_get_path_at_pos(&view->view, event->x, event->y, &path,
				     &column, NULL, NULL))
    goto done;

  if (!gtk_tree_model_get_iter(view->model, &iter, path))
    goto done;

  MAIN_THREAD_EVENT_CALLBACK();

  gtk_tree_model_get(view->model, &iter, THREADLIST_DATA, &data, -1);

  if (view->recently_pointed_thread != data)
    {
      OchushaBBSThread *thread = (OchushaBBSThread *)data;

      if (view->recently_pointed_thread != NULL)
	g_signal_emit(G_OBJECT(view),
		      threadlist_view_signals[THREAD_MOUSE_OUT_SIGNAL],
		      0,
		      view->recently_pointed_thread);

      view->recently_pointed_thread = thread;

      if (thread != NULL)
	g_signal_emit(G_OBJECT(view),
		      threadlist_view_signals[THREAD_MOUSE_OVER_SIGNAL],
		      0,
		      thread);
    }

 done:
  if (path != NULL)
    gtk_tree_path_free(path);

  if (GTK_WIDGET_CLASS(parent_class)->motion_notify_event != NULL)
    result = (*GTK_WIDGET_CLASS(parent_class)->motion_notify_event)(widget,
								    event);

  ENTER_MAIN_EVENT_LOOP();

  return result;
}


static gboolean
threadlist_view_leave_notify(GtkWidget *widget, GdkEventCrossing *event)
{
  ThreadlistView *view = THREADLIST_VIEW(widget);
  gboolean result = FALSE;

  MAIN_THREAD_EVENT_CALLBACK();

  if (view->recently_pointed_thread != NULL)
    {
      g_signal_emit(G_OBJECT(view),
		    threadlist_view_signals[THREAD_MOUSE_OUT_SIGNAL],
		    0,
		    view->recently_pointed_thread);

      view->recently_pointed_thread = NULL;
    }

  if (GTK_WIDGET_CLASS(parent_class)->leave_notify_event != NULL)
    result = (*GTK_WIDGET_CLASS(parent_class)->leave_notify_event)(widget,
								   event);

  ENTER_MAIN_EVENT_LOOP();

  return result;
}


static gboolean
threadlist_view_button_press(GtkWidget *widget, GdkEventButton *event)
{
  ThreadlistView *view = THREADLIST_VIEW(widget);
  GtkTreeIter iter;
  GtkTreePath *path = NULL;
  GtkTreeViewColumn *column;
  gpointer data = NULL;
  gboolean result = FALSE;

  MAIN_THREAD_EVENT_CALLBACK();

  if (view->last_event_path != NULL)
    {
      gtk_tree_path_free(view->last_event_path);
      view->last_event_path = NULL;
    }

  if (!gtk_tree_view_get_path_at_pos(&view->view, event->x, event->y, &path,
				     &column, NULL, NULL))
    goto done;

  if (!gtk_tree_model_get_iter(view->model, &iter, path))
    goto done;

  gtk_tree_model_get(view->model, &iter, THREADLIST_DATA, &data, -1);

  if (data != NULL)
    {
      if (event->button == 3)
	{
	  if (view->remove_hidden_thread)
	    gtk_item_factory_popup_with_data(popup_menu_item_factory2,
					     view, popup_menu_destructed_cb,
					     event->x_root, event->y_root,
					     event->button,
					     gtk_get_current_event_time());
	  else
	    gtk_item_factory_popup_with_data(popup_menu_item_factory,
					     view, popup_menu_destructed_cb,
					     event->x_root, event->y_root,
					     event->button,
					     gtk_get_current_event_time());
	  if (!gtk_tree_selection_iter_is_selected(view->selection, &iter))
	    {
	      /* XXX: ȡårow򤵤줿֤ˤʤ롣
	       *      CTRLSHIFT򲡤ʤξˤϰ㤦񤤤
	       *      ٤ȿˡĤʤΤŶ
	       *      DnDݡΥСǤϡå줿row
	       *      ƤʤäˤϤΥʥϥɥ餬FALSE
	       *      ֤ȤˤꡢGtkTreeView˽Ǥä
	       *      DnDݡȻƱˡȡܥ󥯥åǤ
	       *      DnDǧƤʤϤʤΤDnD⡼ɤäƤޤ
	       *      ݥåץåץ˥塼̥ɥʤΤǡ̯ʸ
	       *      롣
	       */
	      gtk_tree_view_set_cursor(&view->view, path, NULL, FALSE);
	    }
	  result = TRUE;
	}
      else if ((event->state & (GDK_CONTROL_MASK | GDK_SHIFT_MASK)) == 0)
	{
	  if (column == view->mark_column)
	    {
	      g_signal_emit(G_OBJECT(view),
			    threadlist_view_signals[TOGGLE_MARK_SIGNAL],
			    0,
			    data);
	    }
	  else
	    {
#if 0
	      g_signal_emit(G_OBJECT(view),
			    threadlist_view_signals[OPEN_THREAD_VIEW_SIGNAL],
			    0,
			    data,
			    view->default_open_in_tab
			    ? event->button == 1 : event->button == 2,
			    FALSE);
#else
	      view->last_event_path = path;
	      path = NULL;
#endif
	    }
	}
    }

 done:
  if (path != NULL)
    gtk_tree_path_free(path);

  if (!result && GTK_WIDGET_CLASS(parent_class)->button_press_event != NULL)
    result = (*GTK_WIDGET_CLASS(parent_class)->button_press_event)(widget,
								   event);

  ENTER_MAIN_EVENT_LOOP();

  return result;
}


static gboolean
threadlist_view_button_release(GtkWidget *widget, GdkEventButton *event)
{
  ThreadlistView *view = (ThreadlistView *)widget;
  GtkTreePath *path = NULL;
  gboolean result = FALSE;

  g_return_val_if_fail(IS_THREADLIST_VIEW(view), FALSE);
  g_return_val_if_fail(event != NULL, FALSE);

  MAIN_THREAD_EVENT_CALLBACK();

  if (event->button != 3 && view->last_event_path != NULL)
    {
      GtkTreeIter iter;
      GtkTreeViewColumn *column;
      gpointer data = NULL;

      if (!gtk_tree_view_get_path_at_pos(&view->view, event->x, event->y,
					 &path,
					 &column, NULL, NULL))
	goto done;

      if (gtk_tree_path_compare(path, view->last_event_path) != 0)
	goto done;

      if (!gtk_tree_model_get_iter(view->model, &iter, path))
	goto done;

      gtk_tree_model_get(view->model, &iter, THREADLIST_DATA, &data, -1);

      if (data != NULL)
	g_signal_emit(G_OBJECT(view),
		      threadlist_view_signals[OPEN_THREAD_VIEW_SIGNAL],
		      0,
		      data,
		      view->default_open_in_tab
		      ? event->button == 1 : event->button == 2,
		      FALSE);
    }

 done:
  if (path != NULL)
    gtk_tree_path_free(path);

  if (view->last_event_path != NULL)
    {
      gtk_tree_path_free(view->last_event_path);
      view->last_event_path = NULL;
    }

  if (!result && GTK_WIDGET_CLASS(parent_class)->button_release_event != NULL)
    result = (*GTK_WIDGET_CLASS(parent_class)->button_release_event)(widget,
								     event);

  ENTER_MAIN_EVENT_LOOP();

  return FALSE;
}


#define RANK_UNSPECIFIED		-1
#define RANK_DAT_DROPPED		100000
#define RANK_KAKO_THREAD		200000

#define RANK_NOT_CHANGED		0
#define RANK_NEW_COMER_MAGIC		0x7fffffff
#define RANK_SAME_RANK_DOWN_MAGIC	0x7ffffffe


static void
set_thread_values(ThreadlistView *view, GtkListStore *store,
		  int num_shown, int num_unread,
		  int last_rank, int new_rank, int weight,
		  OchushaBBSThread *thread)
{
  ThreadlistInfo *info = get_threadlist_info(thread, view);
  int num_ungot;
  GtkTreePath *path = NULL;
  GtkTreeIter iter;

  if (info == NULL)
    return;	/* Ϥɽ˴طʤ */

  path = gtk_tree_row_reference_get_path(info->reference);
  if (path == NULL)
    return;

  if (!gtk_tree_model_get_iter(view->model, &iter, path))
    goto done;

  num_ungot = (thread->number_of_responses_on_server
	       - thread->number_of_responses_read);
  num_ungot = (num_ungot >= 0 ? num_ungot : 0);

  if (new_rank != RANK_UNSPECIFIED)
    {
      int rank_diff;
      if (new_rank == RANK_DAT_DROPPED || new_rank == RANK_KAKO_THREAD)
	rank_diff = RANK_NOT_CHANGED;
      else
	rank_diff = (last_rank == 0
		     ? RANK_NEW_COMER_MAGIC : last_rank - new_rank);
      if (view->last_rank_diff != RANK_NEW_COMER_MAGIC
	  && view->last_rank_diff != RANK_NOT_CHANGED
	  && view->last_rank_diff == rank_diff
	  && rank_diff < 0)
	rank_diff = RANK_SAME_RANK_DOWN_MAGIC;
      else
	view->last_rank_diff = rank_diff;

      gtk_list_store_set(store, &iter,
		THREADLIST_COUNT, view->count++,
		THREADLIST_RANK, new_rank,
		THREADLIST_VISUAL_PARAMETER, 0,
		THREADLIST_TITLE, thread->title,
		THREADLIST_N_RESPONSES, thread->number_of_responses_on_server,
		THREADLIST_N_RESPONSES_GOT, thread->number_of_responses_read,
		THREADLIST_N_RESPONSES_UNGOT, num_ungot,
		THREADLIST_N_RESPONSES_SHOWN, num_shown,
		THREADLIST_N_RESPONSES_UNREAD, num_unread,
		THREADLIST_RANK_DIFFERENCE, rank_diff,
		THREADLIST_LAST_MODIFIED, ochusha_bbs_thread_get_last_modified_utc(thread),
		THREADLIST_WEIGHT, weight,
		THREADLIST_DATA, thread,
		-1);
    }
  else
    {
      gtk_list_store_set(store, &iter,
		THREADLIST_TITLE, thread->title,
		THREADLIST_N_RESPONSES, thread->number_of_responses_on_server,
		THREADLIST_N_RESPONSES_GOT, thread->number_of_responses_read,
		THREADLIST_N_RESPONSES_UNGOT, num_ungot,
		THREADLIST_N_RESPONSES_SHOWN, num_shown,
		THREADLIST_N_RESPONSES_UNREAD, num_unread,
		THREADLIST_LAST_MODIFIED, ochusha_bbs_thread_get_last_modified_utc(thread),
		THREADLIST_WEIGHT, weight,
		THREADLIST_DATA, thread,
		-1);
    }

 done:
  if (path != NULL)
    gtk_tree_path_free(path);
}


static guint
threadlist_view_encode_visual_params(ThreadlistView *view,
				     ThreadlistEntryVisualParameter *parameter)
{
  guint result = 0;

  result |= (parameter->flags & 3);
  result |= (parameter->foreground_color & 0xf)
    << (THREADLIST_ENTRY_VISUAL_FLAGS_WIDTH);
  result |= (parameter->background_color & 0xf)
    << (THREADLIST_ENTRY_VISUAL_FLAGS_WIDTH
	+ THREADLIST_ENTRY_VISUAL_FOREGROUND_COLOR_WIDTH);
#if DEBUG_WIDGET_MOST
  fprintf(stderr, "parameter{ flags=%d, foreground_color=%d, background_color=%d}, result=%d\n", parameter->flags, parameter->foreground_color, parameter->background_color, result);
#endif

  return result;
}


static guint
threadlist_view_decode_visual_param_flags(ThreadlistView *view,
					  guint parameter)
{
  return parameter & 3;
}


static const char *threadlist_view_decode_visual_param_foreground(ThreadlistView *view, guint parameter);
static const char *threadlist_view_decode_visual_param_background(ThreadlistView *view, guint parameter);


static const char *
find_replacement_color(ThreadlistView *view, const char *color)
{
  const char *result_color;
  GdkColor *gdk_color;
  char name_buf[8];
  g_return_val_if_fail(color != NULL && *color == '<', NULL);

  if (strncmp(color + 1, "fg_", 3) == 0)
    {
      if (strcmp(color + 4, "normal>") == 0)
	result_color = threadlist_view_decode_visual_param_foreground(view,
				(LIST_ENTRY_FOREGROUND_NORMAL
				 << THREADLIST_ENTRY_VISUAL_FLAGS_WIDTH));
      else if (strcmp(color + 4, "emph>") == 0)
	result_color = threadlist_view_decode_visual_param_foreground(view,
				(LIST_ENTRY_FOREGROUND_EMPH
				 << THREADLIST_ENTRY_VISUAL_FLAGS_WIDTH));
      else if (strcmp(color + 4, "strong>") == 0)
	result_color = threadlist_view_decode_visual_param_foreground(view,
				(LIST_ENTRY_FOREGROUND_STRONG
				 << THREADLIST_ENTRY_VISUAL_FLAGS_WIDTH));
      else if (strcmp(color + 4, "hidden>") == 0)
	result_color = threadlist_view_decode_visual_param_foreground(view,
				(LIST_ENTRY_FOREGROUND_HIDDEN
				 << THREADLIST_ENTRY_VISUAL_FLAGS_WIDTH));
      else
	return NULL;
      if (result_color != NULL)
	return result_color;

      gdk_color = &GTK_WIDGET(view)->style->text[GTK_STATE_NORMAL];
      snprintf(name_buf, 8, "#%02x%02x%02x",
	       gdk_color->red / 256, gdk_color->green / 256,
	       gdk_color->blue / 256);
      return g_strdup(name_buf);	/* Ĥϥ꡼뤱̵ */
    }
  else if (strncmp(color + 1, "bg_", 3) == 0)
    {
      if (strcmp(color + 4, "normal>") == 0)
	result_color = threadlist_view_decode_visual_param_background(view,
		(LIST_ENTRY_BACKGROUND_NORMAL
		 << (THREADLIST_ENTRY_VISUAL_FLAGS_WIDTH
		     + THREADLIST_ENTRY_VISUAL_FOREGROUND_COLOR_WIDTH)));
      else if (strcmp(color + 4, "emph>") == 0)
	result_color = threadlist_view_decode_visual_param_background(view,
		(LIST_ENTRY_BACKGROUND_EMPH
		 << (THREADLIST_ENTRY_VISUAL_FLAGS_WIDTH
		     + THREADLIST_ENTRY_VISUAL_FOREGROUND_COLOR_WIDTH)));
      else if (strcmp(color + 4, "strong>") == 0)
	result_color = threadlist_view_decode_visual_param_background(view,
		(LIST_ENTRY_BACKGROUND_STRONG
		 << (THREADLIST_ENTRY_VISUAL_FLAGS_WIDTH
		     + THREADLIST_ENTRY_VISUAL_FOREGROUND_COLOR_WIDTH)));
      else if (strcmp(color + 4, "hidden>") == 0)
	result_color = threadlist_view_decode_visual_param_background(view,
		(LIST_ENTRY_BACKGROUND_HIDDEN
		 << (THREADLIST_ENTRY_VISUAL_FLAGS_WIDTH
		     + THREADLIST_ENTRY_VISUAL_FOREGROUND_COLOR_WIDTH)));
      else
	return NULL;
      if (result_color != NULL)
	return result_color;

      gdk_color = &GTK_WIDGET(view)->style->base[GTK_STATE_NORMAL];
      snprintf(name_buf, 8, "#%02x%02x%02x",
	       gdk_color->red / 256, gdk_color->green / 256,
	       gdk_color->blue / 256);
      return g_strdup(name_buf);	/* Ĥϥ꡼뤱̵ */
    }

  return NULL;
}


static const char *
threadlist_view_decode_visual_param_foreground(ThreadlistView *view,
					       guint parameter)
{
  guint color_code = (parameter >> THREADLIST_ENTRY_VISUAL_FLAGS_WIDTH) & 0xf;
  const char *color;
  char *tmp_color;

  g_return_val_if_fail(color_code < NUMBER_OF_LIST_ENTRY_FOREGROUND_COLORS,
		       NULL);

  if (list_entry_foreground_color[color_code] != NULL)
    {
      color = list_entry_foreground_color[color_code];
      if (color != NULL && *color == '\0')
	return NULL;
      return color;
    }

  switch (color_code)
    {
    case LIST_ENTRY_FOREGROUND_NORMAL:
      gtk_widget_style_get(GTK_WIDGET(view), "fg_normal", &color, NULL);
      break;
    case LIST_ENTRY_FOREGROUND_EMPH:
      gtk_widget_style_get(GTK_WIDGET(view), "fg_emph", &color, NULL);
      break;
    case LIST_ENTRY_FOREGROUND_STRONG:
      gtk_widget_style_get(GTK_WIDGET(view), "fg_strong", &color, NULL);
      break;
    case LIST_ENTRY_FOREGROUND_HIDDEN:
      gtk_widget_style_get(GTK_WIDGET(view), "fg_hidden", &color, NULL);
      break;
    default:
      color = NULL;
    }

  if (color == NULL)
    return color;

  if (*color == '\0')
    return NULL;

  if (*color != '<')
    return color;

  list_entry_foreground_color[color_code] = G_STRDUP("");

  color = find_replacement_color(view, color);

  if (color == NULL)
    return NULL;

  tmp_color = G_STRDUP(color);
  G_FREE(list_entry_foreground_color[color_code]);

  list_entry_foreground_color[color_code] = tmp_color;
  if (*tmp_color == '\0')
    return NULL;
  return tmp_color;
}


static const char *
threadlist_view_decode_visual_param_background(ThreadlistView *view,
					       guint parameter)
{
  guint color_code
    = (parameter
       >> (THREADLIST_ENTRY_VISUAL_FLAGS_WIDTH
	   + THREADLIST_ENTRY_VISUAL_FOREGROUND_COLOR_WIDTH)) & 0xf;
  const char *color;
  char *tmp_color;

  g_return_val_if_fail(color_code < NUMBER_OF_LIST_ENTRY_BACKGROUND_COLORS,
		       NULL);

  if (list_entry_background_color[color_code] != NULL)
    {
      color = list_entry_background_color[color_code];
      if (color != NULL && *color == '\0')
	return NULL;
      return color;
    }

  switch (color_code)
    {
    case LIST_ENTRY_BACKGROUND_NORMAL:
      gtk_widget_style_get(GTK_WIDGET(view), "bg_normal", &color, NULL);
      break;
    case LIST_ENTRY_BACKGROUND_EMPH:
      gtk_widget_style_get(GTK_WIDGET(view), "bg_emph", &color, NULL);
      break;
    case LIST_ENTRY_BACKGROUND_STRONG:
      gtk_widget_style_get(GTK_WIDGET(view), "bg_strong", &color, NULL);
      break;
    case LIST_ENTRY_BACKGROUND_HIDDEN:
      gtk_widget_style_get(GTK_WIDGET(view), "bg_hidden", &color, NULL);
      break;
    default:
      color = NULL;
    }

  if (color == NULL)
    return color;

  if (*color == '\0')
    return NULL;

  if (*color != '<')
    return color;

  list_entry_background_color[color_code] = G_STRDUP("");

  color = find_replacement_color(view, color);

  if (color == NULL)
    return NULL;

  tmp_color = G_STRDUP(color);
  G_FREE(list_entry_background_color[color_code]);

  list_entry_background_color[color_code] = tmp_color;
  if (*tmp_color == '\0')
    return NULL;
  return tmp_color;
}


void
threadlist_view_update_thread_visual(ThreadlistView *view,
				     OchushaBBSThread *thread,
				     ThreadlistEntryVisualParameter *parameter)
{
  gpointer data = NULL;
  ThreadlistInfo *info;
  GtkTreePath *path = NULL;
  GtkTreeIter iter;

  g_return_if_fail(IS_THREADLIST_VIEW(view) && thread != NULL);

  info = get_threadlist_info(thread, view);

  if (info == NULL)
    return;	/* Ϥɽ˴طʤ */

  path = gtk_tree_row_reference_get_path(info->reference);
  if (path == NULL)
    return;

  if (!gtk_tree_model_get_iter(view->model, &iter, path))
    goto done;

  gtk_tree_model_get(view->model, &iter,
		     THREADLIST_DATA, &data, -1);
  g_return_if_fail(thread == data);

  gtk_list_store_set(GTK_LIST_STORE(view->model), &iter,
		     THREADLIST_VISUAL_PARAMETER,
		     threadlist_view_encode_visual_params(view, parameter),
		     -1);
 done:
  gtk_tree_path_free(path);
}


static void
decorate_threadlist_entry(GtkTreeViewColumn *tree_column,
			  GtkCellRenderer *renderer, GtkTreeModel *tree_model,
			  GtkTreeIter *iter, gpointer view)
{
  guint flags;
  const char *color;
  GValue fg = { 0, };
  GValue bg = { 0, };

  g_return_if_fail(gtk_list_store_iter_is_valid(GTK_LIST_STORE(tree_model),
						iter));
  gtk_tree_model_get(tree_model, iter,
		     THREADLIST_VISUAL_PARAMETER, &flags, -1);

  if (!GTK_IS_CELL_RENDERER_TOGGLE(renderer))
    {
      color = threadlist_view_decode_visual_param_foreground(view, flags);
      g_value_init(&fg, G_TYPE_STRING);
      g_value_set_static_string(&fg, color);
      g_object_set_property((GObject *)renderer, "foreground", &fg);
    }

  color = threadlist_view_decode_visual_param_background(view, flags);
  g_value_init(&bg, G_TYPE_STRING);
  g_value_set_static_string(&bg, color);
  g_object_set_property((GObject *)renderer, "cell_background", &bg);

  if (!GTK_IS_CELL_RENDERER_TEXT(renderer))
    return;

  if (threadlist_view_decode_visual_param_flags(view, flags)
      & THREADLIST_ENTRY_FLAG_STRIKE)
    {
      GValue st = { 0, };
      g_value_init(&st, G_TYPE_BOOLEAN);
      g_value_set_boolean(&st, TRUE);
      g_object_set_property((GObject *)renderer, "strikethrough", &st);
      g_object_set_property((GObject *)renderer, "strikethrough_set", &st);
    }
  else
    {
      GValue st = { 0, };
      g_value_init(&st, G_TYPE_BOOLEAN);
      g_value_set_boolean(&st, FALSE);
      g_object_set_property((GObject *)renderer, "strikethrough", &st);
      g_object_set_property((GObject *)renderer, "strikethrough_set", &st);
    }
}


static void
text_set_num_of_responses_verbose(GtkTreeViewColumn *tree_column,
				  GtkCellRenderer *renderer,
				  GtkTreeModel *tree_model, GtkTreeIter *iter,
				  gpointer view)
{
  GValue value = { 0, };
  int num_responses;
  int num_responses_got;
  int num_responses_available;
  gchar verbose_text[64];

  g_return_if_fail(gtk_list_store_iter_is_valid(GTK_LIST_STORE(tree_model),
						iter));

  gtk_tree_model_get(tree_model, iter,
		     THREADLIST_N_RESPONSES, &num_responses,
		     THREADLIST_N_RESPONSES_GOT, &num_responses_got,
		     -1);

  num_responses_available = MAX(num_responses, num_responses_got);
  if (num_responses_got == 0)
    snprintf(verbose_text, 64, "%d", num_responses_available);
  else if (num_responses_got == num_responses_available)
    snprintf(verbose_text, 64, "%d (0)", num_responses_available);
  else
    snprintf(verbose_text, 64, "%d (+%d)", num_responses_available,
	     num_responses_available - num_responses_got);

  g_value_init(&value, G_TYPE_STRING);
  g_value_set_string(&value, verbose_text);

  g_object_set_property((GObject *)renderer, "text", &value);
  decorate_threadlist_entry(tree_column, renderer, tree_model, iter, view);
}


static void
text_set_cell_data_rank(GtkTreeViewColumn *tree_column,
			GtkCellRenderer *renderer,
			GtkTreeModel *tree_model, GtkTreeIter *iter,
			gpointer view)
{
  GValue src = { 0, };
  GValue value = { 0, };
  int rank;

  g_return_if_fail(gtk_list_store_iter_is_valid(GTK_LIST_STORE(tree_model),
						iter));

  gtk_tree_model_get_value(tree_model, iter, THREADLIST_RANK, &src);
  rank = g_value_get_int(&src);

  g_value_init(&value, G_TYPE_STRING);
  if (rank >= RANK_DAT_DROPPED)
    {
      if (rank == RANK_KAKO_THREAD)
	g_value_set_static_string(&value, _("Past"));
      else
	g_value_set_static_string(&value, _("Dead"));
    }
  else
    {
      /* ̾ */
      gchar buffer[512];
      snprintf(buffer, 512, "%d", rank);
      g_value_set_string(&value, buffer);
    }

  g_object_set_property((GObject *)renderer, "text", &value);
  decorate_threadlist_entry(tree_column, renderer, tree_model, iter, view);
}


static void
active_set_cell_data_flags(GtkTreeViewColumn *tree_column,
			   GtkCellRenderer *renderer, GtkTreeModel *tree_model,
			   GtkTreeIter *iter, gpointer view)
{
  guint flags;
  GtkCellRendererToggle *toggle_renderer = GTK_CELL_RENDERER_TOGGLE(renderer);
  g_return_if_fail(gtk_list_store_iter_is_valid(GTK_LIST_STORE(tree_model),
						iter));

  gtk_tree_model_get(tree_model, iter,
		     THREADLIST_VISUAL_PARAMETER, &flags, -1);

  flags = threadlist_view_decode_visual_param_flags(view, flags);

  gtk_cell_renderer_toggle_set_active(toggle_renderer,
				      flags & THREADLIST_ENTRY_FLAG_MARK);
  decorate_threadlist_entry(tree_column, renderer, tree_model, iter, view);
}


static void
text_set_cell_data_rank_diff(GtkTreeViewColumn *tree_column,
			     GtkCellRenderer *renderer,
			     GtkTreeModel *tree_model, GtkTreeIter *iter,
			     gpointer view)
{
  GValue src = { 0, };
  GValue value = { 0, };
  int diff;
  gchar buffer[512];

  g_return_if_fail(gtk_list_store_iter_is_valid(GTK_LIST_STORE(tree_model),
						iter));

  gtk_tree_model_get_value(tree_model, iter, THREADLIST_RANK_DIFFERENCE, &src);
  diff = g_value_get_int(&src);

  g_value_init(&value, G_TYPE_STRING);
  if (diff == RANK_NOT_CHANGED)
    {
      g_value_set_static_string(&value, "");
    }
  else if (diff == RANK_NEW_COMER_MAGIC)
    {
      /* оξ */
      g_value_set_static_string(&value, _("New Face!"));
    }
  else if (diff == RANK_SAME_RANK_DOWN_MAGIC)
    {
      /* ľΥȥƱ̥ξ */
      g_value_set_static_string(&value, _("down"));
    }
  else if (diff > 0)
    {
      /* ̥åפξ */
      snprintf(buffer, 512, _("%d Rank Up"), diff);
      g_value_set_string(&value, buffer);
    }
  else
    {
      /* ̥ξ */
      snprintf(buffer, 512, _("%d Rank Down"), -diff);
      g_value_set_string(&value, buffer);
    }

  g_object_set_property((GObject *)renderer, "text", &value);
  decorate_threadlist_entry(tree_column, renderer, tree_model, iter, view);
}


static void
text_set_cell_data_last_modified(GtkTreeViewColumn *tree_column,
				 GtkCellRenderer *renderer,
				 GtkTreeModel *tree_model, GtkTreeIter *iter,
				 gpointer view)
{
  GValue src = { 0, };
  GValue value = { 0, };
  time_t utc;

  g_return_if_fail(gtk_list_store_iter_is_valid(GTK_LIST_STORE(tree_model),
						iter));

  gtk_tree_model_get_value(tree_model, iter, THREADLIST_LAST_MODIFIED, &src);
  utc = g_value_get_long(&src);

  g_value_init(&value, G_TYPE_STRING);
  if (utc != 0)
    {
      gchar buffer[512];

#ifdef HAVE_LOCALTIME_R
      struct tm tm;
      size_t result;
      localtime_r(&utc, &tm);
      result = strftime(buffer, 512, "%Y-%m-%d %T", &tm);
      if (result == 0)
	buffer[0] = '\0';
#else
      LOCALTIME_LOCK
      {
	struct tm *tm;
	size_t result;
	tm = localtime(&utc);
	result = strftime(buffer, 512, "%Y-%m-%d %T", tm);
	if (result == 0)
	  buffer[0] = '\0';
      }
      LOCALTIME_UNLOCK;
#endif
      g_value_set_string(&value, buffer);
    }
  else
    g_value_set_static_string(&value, "");

  g_object_set_property((GObject *)renderer, "text", &value);
  decorate_threadlist_entry(tree_column, renderer, tree_model, iter, view);
}


static int
compare_by_rank(GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b,
		gpointer unused)
{
  int a_count;
  int a_rank;
  int b_count;
  int b_rank;

  gtk_tree_model_get(model, a,
		     THREADLIST_COUNT, &a_count, THREADLIST_RANK, &a_rank, -1);
  gtk_tree_model_get(model, b,
		     THREADLIST_COUNT, &b_count, THREADLIST_RANK, &b_rank, -1);
  if (a_rank != b_rank)
    return a_rank - b_rank;

  return a_count - b_count;
}


static void
rank_column_clicked_cb(GtkTreeViewColumn *column, ThreadlistView *view)
{
  gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(view->model),
				       THREADLIST_SORT_BY_RANK,
				       GTK_SORT_ASCENDING);
}


void
threadlist_view_append_rank(ThreadlistView *view)
{
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;
  GValue xalign = { 0, };

  g_return_if_fail(IS_THREADLIST_VIEW(view) && view->rank_column == NULL);

  renderer = gtk_cell_renderer_text_new();
  g_value_init(&xalign, G_TYPE_FLOAT);
  g_value_set_float(&xalign, 0.8);
  g_object_set_property((GObject *)renderer, "xalign", &xalign);
  column = gtk_tree_view_column_new_with_attributes(_("Rank"), renderer,
						    "text", THREADLIST_RANK,
						    NULL);
  gtk_tree_view_column_set_resizable(column, TRUE);
  gtk_tree_view_column_set_clickable(column, TRUE);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  text_set_cell_data_rank,
					  view, NULL);
  g_signal_connect(G_OBJECT(column), "clicked",
		   G_CALLBACK(rank_column_clicked_cb), view);
  gtk_tree_view_append_column(&view->view, column);
  view->rank_column = column;

  view->view_is_initialized = TRUE;
}


static int
compare_by_mark(GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b,
		gpointer unused)
{
  int a_count;
  int a_flags;
  int b_count;
  int b_flags;

  gtk_tree_model_get(model, a,
		     THREADLIST_COUNT, &a_count,
		     THREADLIST_VISUAL_PARAMETER, &a_flags, -1);
  gtk_tree_model_get(model, b,
		     THREADLIST_COUNT, &b_count,
		     THREADLIST_VISUAL_PARAMETER, &b_flags, -1);

  a_flags &= THREADLIST_ENTRY_FLAG_MARK;
  b_flags &= THREADLIST_ENTRY_FLAG_MARK;

  if (a_flags != b_flags)
    return b_flags - a_flags;
  return a_count - b_count;
}


static void
mark_column_clicked_cb(GtkTreeViewColumn *column, ThreadlistView *view)
{
  gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(view->model),
				       THREADLIST_SORT_BY_MARK,
				       GTK_SORT_ASCENDING);
}


void
threadlist_view_append_mark(ThreadlistView *view)
{
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;

  g_return_if_fail(IS_THREADLIST_VIEW(view) && view->mark_column == NULL);

  renderer = gtk_cell_renderer_toggle_new();
  gtk_cell_renderer_toggle_set_radio(GTK_CELL_RENDERER_TOGGLE(renderer),
				     FALSE);
  column = gtk_tree_view_column_new_with_attributes(_("Mark"), renderer,
					"active", THREADLIST_VISUAL_PARAMETER,
					NULL);
  gtk_tree_view_column_set_resizable(column, TRUE);
  gtk_tree_view_column_set_clickable(column, TRUE);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  active_set_cell_data_flags,
					  view, NULL);
  g_signal_connect(G_OBJECT(column), "clicked",
		   G_CALLBACK(mark_column_clicked_cb), view);
  gtk_tree_view_append_column(&view->view, column);
  view->mark_column = column;

  view->view_is_initialized = TRUE;
}


static int
compare_by_title(GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b,
		 gpointer unused)
{
  int a_count;
  gchar *a_title;
  int b_count;
  gchar *b_title;
  int result = 0;

  gtk_tree_model_get(model, a,
		     THREADLIST_COUNT, &a_count,
		     THREADLIST_TITLE, &a_title, -1);
  gtk_tree_model_get(model, b,
		     THREADLIST_COUNT, &b_count,
		     THREADLIST_TITLE, &b_title, -1);

  if (a_title != NULL && b_title != NULL)
    result = strcmp(a_title, b_title);

  if (result != 0)
    return result;

  return a_count - b_count;
}


static void
title_column_clicked_cb(GtkTreeViewColumn *column, ThreadlistView *view)
{
  gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(view->model),
				       THREADLIST_SORT_BY_TITLE,
				       GTK_SORT_ASCENDING);
}


void
threadlist_view_append_title(ThreadlistView *view)
{
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;

  g_return_if_fail(IS_THREADLIST_VIEW(view) && view->title_column == NULL);

  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(_("Thread Title"),
						    renderer, "text",
						    THREADLIST_TITLE, NULL);
  gtk_tree_view_column_set_resizable(column, TRUE);
  gtk_tree_view_column_set_clickable(column, TRUE);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  decorate_threadlist_entry,
					  view, NULL);
  g_signal_connect(G_OBJECT(column), "clicked",
		   G_CALLBACK(title_column_clicked_cb), view);
  gtk_tree_view_append_column(&view->view, column);
  view->title_column = column;

  view->view_is_initialized = TRUE;
}


static int
compare_by_n_res(GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b,
		 gpointer unused)
{
  int a_count;
  int a_n_res;
  int b_count;
  int b_n_res;

  gtk_tree_model_get(model, a,
		     THREADLIST_COUNT, &a_count,
		     THREADLIST_N_RESPONSES, &a_n_res, -1);
  gtk_tree_model_get(model, b,
		     THREADLIST_COUNT, &b_count,
		     THREADLIST_N_RESPONSES, &b_n_res, -1);
  if (a_n_res != b_n_res)
    return b_n_res - a_n_res;

  return a_count - b_count;
}


static void
n_res_column_clicked_cb(GtkTreeViewColumn *column, ThreadlistView *view)
{
  gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(view->model),
				       THREADLIST_SORT_BY_N_RESPONSES,
				       GTK_SORT_ASCENDING);
}


void
threadlist_view_append_number_of_responses(ThreadlistView *view)
{
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;
  GValue xalign = { 0, };

  g_return_if_fail(IS_THREADLIST_VIEW(view) && view->n_res_column == NULL);

  renderer = gtk_cell_renderer_text_new();
  g_value_init(&xalign, G_TYPE_FLOAT);
  g_value_set_float(&xalign, 0.8);
  g_object_set_property((GObject *)renderer, "xalign", &xalign);
  column = gtk_tree_view_column_new_with_attributes(_("#Res"),
						    renderer, "text",
						    THREADLIST_N_RESPONSES,
						    NULL);
  gtk_tree_view_column_set_resizable(column, TRUE);
  gtk_tree_view_column_set_clickable(column, TRUE);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  decorate_threadlist_entry,
					  view, NULL);
  g_signal_connect(G_OBJECT(column), "clicked",
		   G_CALLBACK(n_res_column_clicked_cb), view);
  gtk_tree_view_append_column(&view->view, column);
  view->n_res_column = column;

  view->view_is_initialized = TRUE;
}


static int
compare_by_n_got(GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b,
		 gpointer unused)
{
  int a_count;
  int a_n_got;
  int b_count;
  int b_n_got;

  gtk_tree_model_get(model, a,
		     THREADLIST_COUNT, &a_count,
		     THREADLIST_N_RESPONSES_GOT, &a_n_got, -1);
  gtk_tree_model_get(model, b,
		     THREADLIST_COUNT, &b_count,
		     THREADLIST_N_RESPONSES_GOT, &b_n_got, -1);
  if (a_n_got != b_n_got)
    return b_n_got - a_n_got;

  return a_count - b_count;
}


static void
n_got_column_clicked_cb(GtkTreeViewColumn *column, ThreadlistView *view)
{
  gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(view->model),
				       THREADLIST_SORT_BY_N_RESPONSES_GOT,
				       GTK_SORT_ASCENDING);
}


void
threadlist_view_append_number_of_responses_got(ThreadlistView *view)
{
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;
  GValue xalign = { 0, };

  g_return_if_fail(IS_THREADLIST_VIEW(view) && view->n_got_column == NULL);

  renderer = gtk_cell_renderer_text_new();
  g_value_init(&xalign, G_TYPE_FLOAT);
  g_value_set_float(&xalign, 0.8);
  g_object_set_property((GObject *)renderer, "xalign", &xalign);
  column = gtk_tree_view_column_new_with_attributes(_("#Got"),
						renderer, "text",
						THREADLIST_N_RESPONSES_GOT,
						NULL);
  gtk_tree_view_column_set_resizable(column, TRUE);
  gtk_tree_view_column_set_clickable(column, TRUE);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  decorate_threadlist_entry,
					  view, NULL);
  g_signal_connect(G_OBJECT(column), "clicked",
		   G_CALLBACK(n_got_column_clicked_cb), view);
  gtk_tree_view_append_column(&view->view, column);
  view->n_got_column = column;

  view->view_is_initialized = TRUE;
}


static int
compare_by_n_ungot(GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b,
		   gpointer unused)
{
  int a_count;
  int a_n_ungot;
  int b_count;
  int b_n_ungot;

  gtk_tree_model_get(model, a,
		     THREADLIST_COUNT, &a_count,
		     THREADLIST_N_RESPONSES_UNGOT, &a_n_ungot, -1);
  gtk_tree_model_get(model, b,
		     THREADLIST_COUNT, &b_count,
		     THREADLIST_N_RESPONSES_UNGOT, &b_n_ungot, -1);
  if (a_n_ungot != b_n_ungot)
    return b_n_ungot - a_n_ungot;

  return a_count - b_count;
}


static void
n_ungot_column_clicked_cb(GtkTreeViewColumn *column, ThreadlistView *view)
{
  gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(view->model),
				       THREADLIST_SORT_BY_N_RESPONSES_UNGOT,
				       GTK_SORT_ASCENDING);
}


void
threadlist_view_append_number_of_responses_ungot(ThreadlistView *view)
{
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;
  GValue xalign = { 0, };

  g_return_if_fail(IS_THREADLIST_VIEW(view) && view->n_ungot_column == NULL);

  renderer = gtk_cell_renderer_text_new();
  g_value_init(&xalign, G_TYPE_FLOAT);
  g_value_set_float(&xalign, 0.8);
  g_object_set_property((GObject *)renderer, "xalign", &xalign);
  column = gtk_tree_view_column_new_with_attributes(_("#Ungot"),
						renderer, "text",
						THREADLIST_N_RESPONSES_UNGOT,
						NULL);
  gtk_tree_view_column_set_resizable(column, TRUE);
  gtk_tree_view_column_set_clickable(column, TRUE);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  decorate_threadlist_entry,
					  view, NULL);
  g_signal_connect(G_OBJECT(column), "clicked",
		   G_CALLBACK(n_ungot_column_clicked_cb), view);
  gtk_tree_view_append_column(&view->view, column);
  view->n_ungot_column = column;

  view->view_is_initialized = TRUE;
}


void
threadlist_view_append_number_of_responses_shown(ThreadlistView *view)
{
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;
  GValue xalign = { 0, };

  g_return_if_fail(IS_THREADLIST_VIEW(view) && view->n_shown_column == NULL);

  renderer = gtk_cell_renderer_text_new();
  g_value_init(&xalign, G_TYPE_FLOAT);
  g_value_set_float(&xalign, 0.8);
  g_object_set_property((GObject *)renderer, "xalign", &xalign);
  column = gtk_tree_view_column_new_with_attributes(_("#Shown"),
						renderer, "text",
						THREADLIST_N_RESPONSES_SHOWN,
						NULL);
  gtk_tree_view_column_set_resizable(column, TRUE);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  decorate_threadlist_entry,
					  view, NULL);
  gtk_tree_view_append_column(&view->view, column);
  view->n_shown_column = column;

  view->view_is_initialized = TRUE;
}


static int
compare_by_n_unread(GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b,
		    gpointer unused)
{
  int a_count;
  int a_n_unread;
  int b_count;
  int b_n_unread;

  gtk_tree_model_get(model, a,
		     THREADLIST_COUNT, &a_count,
		     THREADLIST_N_RESPONSES_UNREAD, &a_n_unread, -1);
  gtk_tree_model_get(model, b,
		     THREADLIST_COUNT, &b_count,
		     THREADLIST_N_RESPONSES_UNREAD, &b_n_unread, -1);
  if (a_n_unread != b_n_unread)
    return b_n_unread - a_n_unread;

  return a_count - b_count;
}


static void
n_unread_column_clicked_cb(GtkTreeViewColumn *column, ThreadlistView *view)
{
  gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(view->model),
				       THREADLIST_SORT_BY_N_RESPONSES_UNREAD,
				       GTK_SORT_ASCENDING);
}


void
threadlist_view_append_number_of_responses_unread(ThreadlistView *view)
{
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;
  GValue xalign = { 0, };

  g_return_if_fail(IS_THREADLIST_VIEW(view) && view->n_unread_column == NULL);

  renderer = gtk_cell_renderer_text_new();
  g_value_init(&xalign, G_TYPE_FLOAT);
  g_value_set_float(&xalign, 0.8);
  g_object_set_property((GObject *)renderer, "xalign", &xalign);
  column = gtk_tree_view_column_new_with_attributes(_("#Unread"),
						renderer, "text",
						THREADLIST_N_RESPONSES_UNREAD,
						NULL);
  gtk_tree_view_column_set_resizable(column, TRUE);
  gtk_tree_view_column_set_clickable(column, TRUE);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  decorate_threadlist_entry,
					  view, NULL);
  g_signal_connect(G_OBJECT(column), "clicked",
		   G_CALLBACK(n_unread_column_clicked_cb), view);
  gtk_tree_view_append_column(&view->view, column);
  view->n_unread_column = column;

  view->view_is_initialized = TRUE;
}


static int
compare_by_n_res_verb(GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b,
		      gpointer unused)
{
  int a_count;
  int a_n_got;
  int a_n_ungot;
  int b_count;
  int b_n_got;
  int b_n_ungot;

  gtk_tree_model_get(model, a,
		     THREADLIST_COUNT, &a_count,
		     THREADLIST_N_RESPONSES_GOT, &a_n_got,
		     THREADLIST_N_RESPONSES_UNGOT, &a_n_ungot, -1);
  gtk_tree_model_get(model, b,
		     THREADLIST_COUNT, &b_count,
		     THREADLIST_N_RESPONSES_GOT, &b_n_got,
		     THREADLIST_N_RESPONSES_UNGOT, &b_n_ungot, -1);

  if ((a_n_got > 0 || b_n_got > 0) && (a_n_got == 0 || b_n_got == 0))
    return b_n_got - a_n_got;

  if (a_n_ungot != b_n_ungot)
    return b_n_ungot - a_n_ungot;

  return a_count - b_count;
}


static void
n_res_verbose_column_clicked_cb(GtkTreeViewColumn *column,
				ThreadlistView *view)
{
  gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(view->model),
				       THREADLIST_SORT_BY_N_RESPONSES_VERBOSE,
				       GTK_SORT_ASCENDING);
}


void
threadlist_view_append_number_of_responses_verbose(ThreadlistView *view)
{
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;
  GValue xalign = { 0, };

  g_return_if_fail(IS_THREADLIST_VIEW(view)
		   && view->n_res_verbose_column == NULL);

  renderer = gtk_cell_renderer_text_new();
  g_value_init(&xalign, G_TYPE_FLOAT);
  g_value_set_float(&xalign, 0.8);
  g_object_set_property((GObject *)renderer, "xalign", &xalign);
  column = gtk_tree_view_column_new_with_attributes(_("#Res"),
						    renderer, "text",
						    THREADLIST_N_RESPONSES,
						    NULL);
  gtk_tree_view_column_set_resizable(column, TRUE);
  gtk_tree_view_column_set_clickable(column, TRUE);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  text_set_num_of_responses_verbose,
					  view, NULL);
  g_signal_connect(G_OBJECT(column), "clicked",
		   G_CALLBACK(n_res_verbose_column_clicked_cb), view);
  gtk_tree_view_append_column(&view->view, column);
  view->n_res_verbose_column = column;

  view->view_is_initialized = TRUE;
}


void
threadlist_view_append_rank_difference(ThreadlistView *view)
{
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;

  g_return_if_fail(IS_THREADLIST_VIEW(view) && view->rank_diff_column == NULL);

  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(_("Rank Difference"),
						    renderer, "text",
						    THREADLIST_RANK_DIFFERENCE,
						    NULL);
  gtk_tree_view_column_set_resizable(column, TRUE);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  text_set_cell_data_rank_diff,
					  view, NULL);
  gtk_tree_view_append_column(&view->view, column);
  view->rank_diff_column = column;

  view->view_is_initialized = TRUE;
}


static int
compare_by_last_modified(GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b,
			 gpointer unused)
{
  int a_count;
  long a_utc;
  int b_count;
  long b_utc;

  gtk_tree_model_get(model, a,
		     THREADLIST_COUNT, &a_count,
		     THREADLIST_LAST_MODIFIED, &a_utc, -1);
  gtk_tree_model_get(model, b,
		     THREADLIST_COUNT, &b_count,
		     THREADLIST_LAST_MODIFIED, &b_utc, -1);
  if (a_utc != b_utc)
    return b_utc - a_utc;

  return a_count - b_count;
}


static void
last_modified_column_clicked_cb(GtkTreeViewColumn *column,
				ThreadlistView *view)
{
  gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(view->model),
				       THREADLIST_SORT_BY_LAST_MODIFIED,
				       GTK_SORT_ASCENDING);
}


void
threadlist_view_append_last_modified(ThreadlistView *view)
{
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;

  g_return_if_fail(IS_THREADLIST_VIEW(view)
		   && view->last_modified_column == NULL);

  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(_("Last Modified"),
						    renderer, "text",
						    THREADLIST_LAST_MODIFIED,
						    NULL);
  gtk_tree_view_column_set_resizable(column, TRUE);
  gtk_tree_view_column_set_clickable(column, TRUE);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  text_set_cell_data_last_modified,
					  view, NULL);
  g_signal_connect(G_OBJECT(column), "clicked",
		   G_CALLBACK(last_modified_column_clicked_cb), view);
  gtk_tree_view_append_column(&view->view, column);
  view->last_modified_column = column;

  view->view_is_initialized = TRUE;
}


static int
compare_by_weight(GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b,
		  gpointer unused)
{
  int a_count;
  int a_weight;
  int b_count;
  int b_weight;

  gtk_tree_model_get(model, a,
		     THREADLIST_COUNT, &a_count,
		     THREADLIST_WEIGHT, &a_weight, -1);
  gtk_tree_model_get(model, b,
		     THREADLIST_COUNT, &b_count,
		     THREADLIST_WEIGHT, &b_weight, -1);
  if (a_weight != b_weight)
    return a_weight - b_weight;

  return a_count - b_count;
}


static void
weight_column_clicked_cb(GtkTreeViewColumn *column, ThreadlistView *view)
{
  gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(view->model),
				       THREADLIST_SORT_BY_WEIGHT,
				       GTK_SORT_ASCENDING);
}


void
threadlist_view_append_weight(ThreadlistView *view)
{
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;
  GValue xalign = { 0, };

  g_return_if_fail(IS_THREADLIST_VIEW(view) && view->weight_column == NULL);

  renderer = gtk_cell_renderer_text_new();
  g_value_init(&xalign, G_TYPE_FLOAT);
  g_value_set_float(&xalign, 0.8);
  g_object_set_property((GObject *)renderer, "xalign", &xalign);
  column = gtk_tree_view_column_new_with_attributes(_("Weight"),
						renderer, "text",
						THREADLIST_WEIGHT,
						NULL);
  gtk_tree_view_column_set_resizable(column, TRUE);
  gtk_tree_view_column_set_clickable(column, TRUE);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  decorate_threadlist_entry,
					  view, NULL);
  g_signal_connect(G_OBJECT(column), "clicked",
		   G_CALLBACK(weight_column_clicked_cb), view);
  gtk_tree_view_append_column(&view->view, column);
  view->weight_column = column;

  view->view_is_initialized = TRUE;
}


void
threadlist_view_initialize_default_view(ThreadlistView *view)
{
  g_return_if_fail(IS_THREADLIST_VIEW(view) && !view->view_is_initialized);

  threadlist_view_append_rank(view);
  threadlist_view_append_mark(view);
  threadlist_view_append_title(view);
  threadlist_view_append_number_of_responses(view);
  threadlist_view_append_number_of_responses_got(view);
  threadlist_view_append_number_of_responses_ungot(view);
  threadlist_view_append_rank_difference(view);
}


gboolean
threadlist_view_open(ThreadlistView *view)
{
  GtkListStore *store;
  GtkTreeSortable *sortable;

  g_return_val_if_fail(IS_THREADLIST_VIEW(view)
		       && !view->model_is_opened, FALSE);

  if (!view->view_is_initialized)
    threadlist_view_initialize_default_view(view);

  if (view->model != NULL)
    g_object_unref(G_OBJECT(view->model));

  store = gtk_list_store_new(THREADLIST_N_COLUMNS,
			     G_TYPE_INT,	/* count */
			     G_TYPE_INT,	/* rank */
			     G_TYPE_INT,	/* flags */
			     G_TYPE_STRING,	/* thread title */
			     G_TYPE_INT,	/* # of responses on server */
			     G_TYPE_INT,	/* # of responses got */
			     G_TYPE_INT,	/* # of responses ungot*/
			     G_TYPE_INT,	/* # of responses shown */
			     G_TYPE_INT,	/* # of responses unread */
			     G_TYPE_INT,	/* rank difference */
			     G_TYPE_LONG,	/* last modified */
			     G_TYPE_INT,	/* weight */
			     OCHUSHA_TYPE_BBS_THREAD);	/* data */

  gtk_tree_view_set_model(&view->view, GTK_TREE_MODEL(store));
  view->model = GTK_TREE_MODEL(store);

  sortable = GTK_TREE_SORTABLE(store);
  gtk_tree_sortable_set_sort_func(sortable,
				  THREADLIST_SORT_BY_RANK,
				  compare_by_rank, NULL, NULL);
  gtk_tree_sortable_set_sort_func(sortable,
				  THREADLIST_SORT_BY_MARK,
				  compare_by_mark, NULL, NULL);
  gtk_tree_sortable_set_sort_func(sortable,
				  THREADLIST_SORT_BY_TITLE,
				  compare_by_title, NULL, NULL);
  gtk_tree_sortable_set_sort_func(sortable,
				  THREADLIST_SORT_BY_N_RESPONSES,
				  compare_by_n_res, NULL, NULL);
  gtk_tree_sortable_set_sort_func(sortable,
				  THREADLIST_SORT_BY_N_RESPONSES_GOT,
				  compare_by_n_got, NULL, NULL);
  gtk_tree_sortable_set_sort_func(sortable,
				  THREADLIST_SORT_BY_N_RESPONSES_UNGOT,
				  compare_by_n_ungot, NULL, NULL);
  gtk_tree_sortable_set_sort_func(sortable,
				  THREADLIST_SORT_BY_N_RESPONSES_UNREAD,
				  compare_by_n_unread, NULL, NULL);
  gtk_tree_sortable_set_sort_func(sortable,
				  THREADLIST_SORT_BY_N_RESPONSES_VERBOSE,
				  compare_by_n_res_verb, NULL, NULL);
  gtk_tree_sortable_set_sort_func(sortable,
				  THREADLIST_SORT_BY_LAST_MODIFIED,
				  compare_by_last_modified, NULL, NULL);
  gtk_tree_sortable_set_sort_func(sortable,
				  THREADLIST_SORT_BY_WEIGHT,
				  compare_by_weight, NULL, NULL);

  view->count = 0;
  view->last_rank_diff = 0;

  return TRUE;
}


void
threadlist_view_set_sort_mode(ThreadlistView *view,
			      ThreadlistViewSortMode mode)
{
  g_return_if_fail(IS_THREADLIST_VIEW(view));
  g_return_if_fail(GTK_TREE_SORTABLE(view->model));

  if (mode == THREADLIST_SORT_BY_NONE)
    {
      if (!gtk_tree_sortable_get_sort_column_id(GTK_TREE_SORTABLE(view->model),
						NULL, NULL))
	return;
      mode = THREADLIST_SORT_BY_RANK;
    }

  gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(view->model),
				       mode, GTK_SORT_ASCENDING);
}


void
threadlist_view_close(ThreadlistView *view, OchushaBBSThread *last_thread)
{
  ThreadlistInfo *info;
  g_return_if_fail(IS_THREADLIST_VIEW(view));

  view->model_is_opened = FALSE;

  if (last_thread == NULL)
    return;

  info = get_threadlist_info(last_thread, view);
  if (info == NULL)
    return;	/* åȤ˻Ǥ */

  if (view->tail_row != NULL)
    gtk_tree_row_reference_free(view->tail_row);
  view->tail_row = gtk_tree_row_reference_copy(info->reference);
}


void
threadlist_view_append_thread(ThreadlistView *view, OchushaBBSThread *thread,
			      int num_shown, int num_unread,
			      int last_rank, int new_rank, int weight)
{
  ThreadlistInfo *info;
  int rank;
  GtkTreePath *path;
  GtkTreeIter iter;

  g_return_if_fail(IS_THREADLIST_VIEW(view));
  g_return_if_fail(OCHUSHA_IS_BBS_THREAD(thread));

  g_assert(view->model != NULL);

  info = ensure_threadlist_info0(thread, view);
  gtk_list_store_append(GTK_LIST_STORE(view->model), &iter);

  path = gtk_tree_model_get_path(view->model, &iter);
  info->reference = gtk_tree_row_reference_new(view->model, path);
  gtk_tree_path_free(path);

  if (thread->flags & OCHUSHA_BBS_THREAD_KAKO)
    rank = RANK_KAKO_THREAD;
  else if (thread->flags & OCHUSHA_BBS_THREAD_DAT_DROPPED)
    rank = RANK_DAT_DROPPED;
  else
    rank = new_rank;

  CHECK_THREADS_ENTERED();
  set_thread_values(view, GTK_LIST_STORE(view->model),
		    num_shown, num_unread, last_rank, rank, weight, thread);
}


void
threadlist_view_update_thread(ThreadlistView *view, OchushaBBSThread *thread,
			      int num_shown, int num_unread, int weight)
{
  gpointer data = NULL;
  ThreadlistInfo *info;
  GtkTreePath *path;
  GtkTreeIter iter;

  g_return_if_fail(IS_THREADLIST_VIEW(view));
  g_return_if_fail(OCHUSHA_IS_BBS_THREAD(thread));

  info = get_threadlist_info(thread, view);

  if (info == NULL)
    return;

  path = gtk_tree_row_reference_get_path(info->reference);
  if (path == NULL)
    return;

  if (!gtk_tree_model_get_iter(view->model, &iter, path))
    goto done;

  gtk_tree_model_get(view->model, &iter, THREADLIST_DATA, &data, -1);

#if DEBUG_WIDGET
  if (thread != data)
    fprintf(stderr, "inconsistent ThreadlistView?\n");
#endif
  g_return_if_fail(thread == data);

  CHECK_THREADS_ENTERED();
  set_thread_values(view, GTK_LIST_STORE(view->model),
		    num_shown, num_unread, 0, RANK_UNSPECIFIED, weight,
		    thread);

 done:
  gtk_tree_path_free(path);
}


void
threadlist_view_scroll_to_thread(ThreadlistView *view,
				 OchushaBBSThread *thread)
{
  GtkTreePath *path;
  ThreadlistInfo *info;

  g_return_if_fail(IS_THREADLIST_VIEW(view));
  g_return_if_fail(OCHUSHA_IS_BBS_THREAD(thread));

  info = get_threadlist_info(thread, view);

  if (info == NULL)
    return;	/* threadϴ˥ɽƤʤ */

  path = gtk_tree_row_reference_get_path(info->reference);
  if (path == NULL)
    return;

  gtk_tree_view_scroll_to_cell(&view->view, path, NULL, FALSE, 0.0f, 0.0f);
  gtk_tree_view_set_cursor_on_cell(&view->view, path, NULL, NULL, FALSE);
  gtk_tree_path_free(path);
}

