/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ochusha_async_buffer.h,v 1.7 2003/12/04 07:30:42 fuyu Exp $
 */

#ifndef _OCHUSHA_ASYNCBUF_H_
#define _OCHUSHA_ASYNCBUF_H_

#include <glib-object.h>
#include <glib.h>

#ifdef __cplusplus
extern "C" {
#endif


#define DEBUG_ASYNC_BUFFER	0
#define DEBUG_ASYNC_BUFFER_MOST	0


#define OCHUSHA_TYPE_ASYNC_BUFFER		(ochusha_async_buffer_get_type())
#define OCHUSHA_ASYNC_BUFFER(obj)		(G_TYPE_CHECK_INSTANCE_CAST((obj), OCHUSHA_TYPE_ASYNC_BUFFER, OchushaAsyncBuffer))
#define OCHUSHA_ASYNC_BUFFER_CLASS(klass)	(G_TYPE_CHECK_CLASS_CAST((klass), OCHUSHA_TYPE_ASYNC_BUFFER, OchushaAsyncBufferClass))
#define OCHUSHA_IS_ASYNC_BUFFER(obj)		(G_TYPE_CHECK_INSTANCE_TYPE((obj), OCHUSHA_TYPE_ASYNC_BUFFER))
#define OCHUSHA_IS_ASYNC_BUFFER_CLASS(klass)	(G_TYPE_CHECK_CLASS_TYPE((klass), OCHUSHA_TYPE_ASYNC_BUFFER))
#define OCHUSHA_ASYNC_BUFFER_GET_CLASS(obj)	(G_TYPE_INSTANCE_GET_CLASS((obj), OCHUSHA_TYPE_ASYNC_BUFFER, OchushaAsyncBufferClass))


typedef struct _OchushaAsyncBuffer OchushaAsyncBuffer;
typedef struct _OchushaAsyncBufferClass OchushaAsyncBufferClass;

typedef struct _BufferSyncObject BufferSyncObject;
typedef void SyncFunc(BufferSyncObject *sync_object);
typedef void DestructFunc(OchushaAsyncBuffer *buffer);

typedef enum
{
  OCHUSHA_ASYNC_BUFFER_OK,
  OCHUSHA_ASYNC_BUFFER_SUSPENDED,
  OCHUSHA_ASYNC_BUFFER_TERMINATED,
} OchushaAsyncBufferStatus;


struct _OchushaAsyncBuffer
{
  GObject parent_object;

  volatile gboolean fixed;
  char volatile *buffer;
  volatile int length;		/* bufferͭʥǡĹ */

  /* ʲprivate */
  int buffer_length;
  volatile int number_of_active_users;
  volatile int number_of_suspended_users;
  volatile OchushaAsyncBufferStatus state;

  BufferSyncObject *sync_object;
  DestructFunc *destructor;

  volatile unsigned int number_of_lock_waiters;
};


struct _OchushaAsyncBufferClass
{
  GObjectClass parent_class;

  void (*wakeup_now)(OchushaAsyncBuffer *buffer);
};


GType ochusha_async_buffer_get_type(void);

OchushaAsyncBuffer *ochusha_async_buffer_new(char *buffer, int length,
					     DestructFunc *destructor);

gboolean ochusha_async_buffer_active_ref(OchushaAsyncBuffer *buffer,
					 const char *user);
void ochusha_async_buffer_active_unref(OchushaAsyncBuffer *buffer,
				       const char *user);

void ochusha_async_buffer_suspend_all(void);
void ochusha_async_buffer_resume_all(void);

void ochusha_async_buffer_suspend(OchushaAsyncBuffer *buffer);
void ochusha_async_buffer_resume(OchushaAsyncBuffer *buffer);
void ochusha_async_buffer_terminate(OchushaAsyncBuffer *buffer);


gboolean ochusha_async_buffer_fix(OchushaAsyncBuffer *buffer,
				  const char *user);
gboolean ochusha_async_buffer_update_length(OchushaAsyncBuffer *buffer,
					    int length,
					    const char *user);
gboolean ochusha_async_buffer_resize(OchushaAsyncBuffer *buffer, int length,
				     const char *user);
gboolean ochusha_async_buffer_ensure_free_space(OchushaAsyncBuffer *buffer,
						int length,
						const char *user);

gboolean ochusha_async_buffer_append_data(OchushaAsyncBuffer *buffer,
					  const char *data,
					  int length, const char *user);

void ochusha_async_buffer_lock(OchushaAsyncBuffer *buffer);
void ochusha_async_buffer_unlock(OchushaAsyncBuffer *buffer);

gboolean ochusha_async_buffer_check_active(OchushaAsyncBuffer *buffer,
					   const char *user);
gboolean ochusha_async_buffer_signal(OchushaAsyncBuffer *buffer,
				     const char *user);
gboolean ochusha_async_buffer_broadcast(OchushaAsyncBuffer *buffer,
					const char *user);
gboolean ochusha_async_buffer_wait(OchushaAsyncBuffer *buffer,
				   const char *user);
gboolean ochusha_async_buffer_is_busy(OchushaAsyncBuffer *buffer,
				      const char *user);

void ochusha_async_buffer_free_when_finished(OchushaAsyncBuffer *buffer);


#ifdef __cplusplus
} // extern "C"
#endif

#endif	/* _OCHUSHA_ASYNCBUF_H_ */
