/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ochusha_utils_jbbs.c,v 1.11 2003/11/24 22:24:57 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "ochusha_utils_jbbs.h"
#include "ochusha_network_broker.h"

#include "utils.h"
#include "worker.h"

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>


#if ENABLE_REGEXP
# if HAVE_ONIGURUMA
#  if HAVE_ONIG_ONIGURUMA_H
#    include <onig/oniguruma.h>
#    define ENABLE_HTML_TO_DAT_CONVERTER	1
#  elif HAVE_ONIGURUMA_H
#    include <oniguruma.h>
#    define ENABLE_HTML_TO_DAT_CONVERTER	1
#  else
#    error "Use of oniguruma without oniguruma.h header isn't considered."
#  endif
# else
#    define ENABLE_HTML_TO_DAT_CONVERTER	0
# endif	/* HAVE_ONIGURUMA */
#endif	/* HAVE_REGEXP */


#if ENABLE_HTML_TO_DAT_CONVERTER
#define DEFAULT_LINE_BUFFER_SIZE		8192
#define DEBUG_HTML_TO_DAT_CONVERTER		0
#define DEBUG_HTML_TO_DAT_CONVERTER_MOST	0


typedef struct _HTML2DATConverterJobArgs
{
  OchushaAsyncBuffer *html_buffer;
  OchushaAsyncBuffer *dat_buffer;
  OchushaNetworkBroker *broker;
  OchushaThread2ch *thread_2ch;
  int incoming_offset;
  int number_of_cached_responses;
} HTML2DATConverterJobArgs;


static char *title_pattern_string = NULL;
static char *title_pattern_end = NULL;
static char *jbbs_shitaraba_response_pattern_string = NULL;
static char *jbbs_shitaraba_response_pattern_end = NULL;
static char *machibbs_response_pattern_string = NULL;
static char *machibbs_response_pattern_end = NULL;
static char *mitinoku_response_pattern_string = NULL;
static char *mitinoku_response_pattern_end = NULL;


static void
initialize_default_regexp_patterns(void)
{
  title_pattern_string = "<title>(.*)</title>";
  title_pattern_end
    = title_pattern_string + strlen(title_pattern_string);

  /* "\314\276\301\260\241\247" == "̾" in EUC-JP */
  /* "\305\352\271\306\306\374\241\247" == "" in EUC-JP */
#if 0
  jbbs_shitaraba_response_pattern_string = "<[dD][tT]>([1-9][0-9]*) \314\276\301\260\241\247[^<]*(<[fF][oO][nN][tT][^>]*|<[aA][^>]*)>(.*)\305\352\271\306\306\374\241\247 ([^<]*)<[bB][rR]><[dD][dD]>(.*)<[bB][rR]><[bB][rR]>$";
#else
  /* ʤ"̾"ѥˤȥޥåʤġġ*/
  jbbs_shitaraba_response_pattern_string = "<[dD][tT]>([1-9][0-9]*) [^<]*(<[fF][oO][nN][tT][^>]*|<[aA][^>]*)>(.*)\305\352\271\306\306\374\241\247 ([^<]*)<[bB][rR]><[dD][dD]>(.*)<[bB][rR]><[bB][rR]>$";
#endif
  jbbs_shitaraba_response_pattern_end
    = (jbbs_shitaraba_response_pattern_string
       + strlen(jbbs_shitaraba_response_pattern_string));

  /* "\226\274\221\117\201\106" == "̾" in Shift_JIS */
  /* "\223\212\215\145\223\372\201\106" == "" in Shift_JIS */
#if 0
  machibbs_response_pattern_string = "<[dD][tT]>([1-9][0-9]*) \226\274\221\117\201\106[^<]*(<[fF][oO][nN][tT][^>]*|<[aA][^>]*)>(.*)\223\212\215\145\223\372\201\106 (.*)<[bB][rR]><[dD][dD]>(.*)<[bB][rR]><[bB][rR]>$";
#else
  machibbs_response_pattern_string = "<[dD][tT]>([1-9][0-9]*) [^<]*(<[fF][oO][nN][tT][^>]*|<[aA][^>]*)>(.*)\223\212\215\145\223\372\201\106 (.*)<[bB][rR]><[dD][dD]>(.*)<[bB][rR]><[bB][rR]>$";
#endif
  machibbs_response_pattern_end
    = (machibbs_response_pattern_string
       + strlen(machibbs_response_pattern_string));

  /* "\314\276\301\260\241\247" == "̾" in EUC-JP */
  /* "\305\352\271\306\306\374\241\247" == "" in EUC-JP */
#if 0
  mitinoku_response_pattern_string = "<[dD][tT]>([1-9][0-9]*) \314\276\301\260\241\247[^<]*(<[fF][oO][nN][tT][^>]*|<[aA][^>]*)>(.*)\305\352\271\306\306\374\241\247(.*)<[dD][dD]>(.*)<[bB][rR]><[bB][rR]>$";
#else
  mitinoku_response_pattern_string = "<[dD][tT]>([1-9][0-9]*) [^<]*(<[fF][oO][nN][tT][^>]*|<[aA][^>]*)>(.*)\305\352\271\306\306\374\241\247(.*)<[dD][dD]>(.*)<[bB][rR]><[bB][rR]>$";
#endif
  mitinoku_response_pattern_end
    = (mitinoku_response_pattern_string
       + strlen(mitinoku_response_pattern_string));
}


static void
convert_html_to_dat(WorkerThread *employee, HTML2DATConverterJobArgs *args)
{
  OchushaAsyncBuffer *html_buffer = args->html_buffer;
  OchushaAsyncBuffer *dat_buffer = args->dat_buffer;
  OchushaThread2ch *thread_2ch = args->thread_2ch;
  OchushaBBSThread *thread = OCHUSHA_BBS_THREAD(thread_2ch);
  int incoming_offset = args->incoming_offset;
  int number_of_cached_responses = args->number_of_cached_responses;
  gboolean signal_result;
  unsigned int html_offset;
  unsigned int response_number;
  char *title = NULL;
  gboolean dump_html = FALSE;
  regex_t *title_pattern = NULL;
  regex_t *response_pattern = NULL;
  RegRegion *match_region = NULL;

  if (title_pattern_string == NULL)
    initialize_default_regexp_patterns();

  switch (thread->board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
      if (regex_new(&title_pattern, title_pattern_string, title_pattern_end,
		    REG_OPTION_EXTEND, REGCODE_EUCJP,
		    REG_SYNTAX_POSIX_EXTENDED, NULL) != REG_NORMAL)
	{
	  fprintf(stderr, "invalid regular expression\n");
	  goto terminated;
	}
      if (regex_new(&response_pattern, jbbs_shitaraba_response_pattern_string,
		    jbbs_shitaraba_response_pattern_end,
		    REG_OPTION_EXTEND, REGCODE_EUCJP,
		    REG_SYNTAX_POSIX_EXTENDED, NULL) != REG_NORMAL)
	{
	  fprintf(stderr, "invalid regular expression\n");
	  goto terminated;
	}
#if DEBUG_HTML_TO_DAT_CONVERTER_MOST
      fprintf(stderr, "title_pattern: \"%s\"\n", title_pattern_string);
      fprintf(stderr, "response_pattern: \"%s\"\n",
	      jbbs_shitaraba_response_pattern_string);
#endif
      break;

    case OCHUSHA_BBS_TYPE_MACHIBBS:
      if (regex_new(&title_pattern, title_pattern_string, title_pattern_end,
		    REG_OPTION_EXTEND, REGCODE_SJIS,
		    REG_SYNTAX_POSIX_EXTENDED, NULL) != REG_NORMAL)
	{
	  fprintf(stderr, "invalid regular expression\n");
	  goto terminated;
	}
        if (regex_new(&response_pattern, machibbs_response_pattern_string,
		      machibbs_response_pattern_end,
		      REG_OPTION_EXTEND, REGCODE_SJIS,
		      REG_SYNTAX_POSIX_EXTENDED, NULL) != REG_NORMAL)
	{
	  fprintf(stderr, "invalid regular expression\n");
	  goto terminated;
	}
#if DEBUG_HTML_TO_DAT_CONVERTER_MOST
      fprintf(stderr, "title_pattern: \"%s\"\n", title_pattern_string);
      fprintf(stderr, "response_pattern: \"%s\"\n",
	      machibbs_response_pattern_string);
#endif
      break;

    case OCHUSHA_BBS_TYPE_MITINOKU:
      if (regex_new(&title_pattern, title_pattern_string, title_pattern_end,
		    REG_OPTION_EXTEND, REGCODE_EUCJP,
		    REG_SYNTAX_POSIX_EXTENDED, NULL) != REG_NORMAL)
	{
	  fprintf(stderr, "invalid regular expression\n");
	  goto terminated;
	}
      if (regex_new(&response_pattern, mitinoku_response_pattern_string,
		    mitinoku_response_pattern_end,
		    REG_OPTION_EXTEND, REGCODE_EUCJP,
		    REG_SYNTAX_POSIX_EXTENDED, NULL) != REG_NORMAL)
	{
	  fprintf(stderr, "invalid regular expression\n");
	  goto terminated;
	}
#if DEBUG_HTML_TO_DAT_CONVERTER_MOST
      fprintf(stderr, "title_pattern: \"%s\"\n", title_pattern_string);
      fprintf(stderr, "response_pattern: \"%s\"\n",
	      mitinoku_response_pattern_string);
#endif
      break;

    case OCHUSHA_BBS_TYPE_JBBS:
      dump_html = TRUE;
      break;

    default:
      goto terminated;
    }

#if DEBUG_HTML_TO_DAT_CONVERTER_MOST
  fprintf(stderr, "convert_html_to_dat called\n");
#endif

  if (!ochusha_async_buffer_active_ref(html_buffer,
				"ochusha_utils_jbbs.c: convert_html_to_dat()"))
    {
      g_signal_emit_by_name(G_OBJECT(args->broker),
			    "access_terminated",
			    dat_buffer,
			    &signal_result);
      goto terminated;
    }

  if (!ochusha_async_buffer_active_ref(dat_buffer,
				"ochusha_utils_jbbs.c: convert_html_to_dat()"))
    {
      g_signal_emit_by_name(G_OBJECT(args->broker),
			    "access_terminated",
			    dat_buffer,
			    &signal_result);
      ochusha_async_buffer_active_unref(html_buffer,
				"ochusha_utils_jbbs.c: convert_html_to_dat()");
      goto terminated;
    }

  html_offset = 0;
  response_number = number_of_cached_responses + 1;

  g_assert(title_pattern != NULL);
  g_assert(response_pattern != NULL);
  match_region = regex_region_new();

  ochusha_async_buffer_lock(html_buffer);
  while (TRUE)
    {
      char *html_buffer_top = (char *)html_buffer->buffer;
      char *html_buffer_pos = html_buffer_top + html_offset;
      int rest_of_data = html_buffer->length - html_offset;
      char *eol_pos = NULL;
      char *previous_line = NULL;

#if DEBUG_HTML_TO_DAT_CONVERTER_MOST
      fprintf(stderr,
	      "html_buffer_top=%p, html_buffer_pos=%p, rest_of_data=%d\n",
	      html_buffer_top, html_buffer_pos, rest_of_data);
#endif

      while (rest_of_data > 0 && (eol_pos = memchr(html_buffer_pos, '\n',
						   rest_of_data)) != NULL)
	{
	  int line_size = eol_pos - html_buffer_pos;
	  char *html_line_buffer = G_STRNDUP(html_buffer_pos, line_size);
	  char *html_line_buffer_tail = html_line_buffer + line_size;
	  line_size++;

	  if (previous_line != NULL)
	    {
	      char *tmp_line = g_strconcat(previous_line, html_line_buffer,
					   NULL);
	      g_free(previous_line);
	      previous_line = html_line_buffer;
	      html_line_buffer_tail = tmp_line + strlen(tmp_line);
	      html_line_buffer = tmp_line;
	    }
#if DEBUG_HTML_TO_DAT_CONVERTER_MOST
	  fprintf(stderr, "html_buffer_pos=%p, eol_pos=%p, line_size=%d\n",
		  html_buffer_pos, eol_pos, line_size);
	  fprintf(stderr, "html_line: \"%s\"\n", html_line_buffer);
#endif

	  regex_region_clear(match_region);
	  if (response_number == 1 && title == NULL
	      && regex_search(title_pattern,
			      html_line_buffer, html_line_buffer_tail,
			      html_line_buffer, html_line_buffer_tail,
			      match_region, 0) >= 0)
	    {
	      char *title_head = html_line_buffer + match_region->beg[1];
	      int title_len = match_region->end[1] - match_region->beg[1];
	      title = G_STRNDUP(title_head, title_len);
#if DEBUG_HTML_TO_DAT_CONVERTER_MOST
	      fprintf(stderr, "title: %s\n", title);
#endif
	    }
	  else if (dump_html)
	    {
	      fprintf(stderr, "html_line: \"%s\"\n", html_line_buffer);
	    }
	  else if (regex_search(response_pattern,
				html_line_buffer, html_line_buffer_tail,
				html_line_buffer, html_line_buffer_tail,
				match_region, 0) >= 0)
	    {
	      int res_num;
	      char *tag_head = html_line_buffer + match_region->beg[2];
	      int tag_len = match_region->end[2] - match_region->beg[2];

	      char *name_head = html_line_buffer + match_region->beg[3];
	      int name_len = match_region->end[3] - match_region->beg[3];

	      char *date_head = html_line_buffer + match_region->beg[4];
	      int date_len = match_region->end[4] - match_region->beg[4];

	      char *body_head = html_line_buffer + match_region->beg[5];
	      int body_len = match_region->end[5] - match_region->beg[5];

	      char *mailto_head = NULL;
	      int mailto_len = 0;

	      if (tag_len > 2 && tag_head[1] == 'a')
		{
		  int tag_rest = tag_len - 2;
		  mailto_head = g_strstr_len(tag_head + 2, tag_rest,
					     "href=\"mailto:");
		  if (mailto_head != NULL)
		    {
		      mailto_head += 13;	/* "href=\"mailto:"Ф */
		      tag_rest -= 13;
		      if (tag_rest > 0)
			{
			  char *mailto_tail
			    = memchr(mailto_head, '"', tag_rest);
			  if (mailto_tail != NULL)
			    mailto_len = mailto_tail - mailto_head;
			}
		    }
		}

	      sscanf(html_line_buffer + match_region->beg[1], "%d", &res_num);

#if DEBUG_HTML_TO_DAT_CONVERTER_MOST
	      fprintf(stderr, "res_num=%d : response_number=%d\n",
		      res_num, response_number);
#endif

	      while (res_num > response_number)
		{
		  /* ŪDATƤ */
#if DEBUG_HTML_TO_DAT_CONVERTER_MOST
		  fprintf(stderr, "breaking DAT for: \"%s\"\n",
			  html_line_buffer);
#endif
		  ochusha_async_buffer_append_data(dat_buffer, "\n", 1,
				"ochusha_utils_jbbs.c: convert_html_to_dat()");
		  response_number++;
		}

	      if (res_num == response_number)
		{
		  char dat_line_buffer[DEFAULT_LINE_BUFFER_SIZE];
		  char *dat_line_buffer_pos = dat_line_buffer;
		  int dat_line_buffer_rest = DEFAULT_LINE_BUFFER_SIZE;

#define APPEND_TEXT_TO_DAT_LINE_BUFFER(text, text_len)			\
  do									\
    {									\
      if ((text_len) > 0)						\
	{								\
	  if (dat_line_buffer_rest > (text_len))			\
	    {								\
	      memcpy(dat_line_buffer_pos, (text), (text_len));		\
	      dat_line_buffer_rest -= (text_len);			\
	      dat_line_buffer_pos += (text_len);			\
	    }								\
	  else								\
	    dat_line_buffer_rest = -1;					\
	}								\
      if (dat_line_buffer_rest > 2)					\
	{								\
	  dat_line_buffer_pos[0] = '<';					\
	  dat_line_buffer_pos[1] = '>';					\
	  dat_line_buffer_rest -= 2;					\
	  dat_line_buffer_pos += 2;					\
	}								\
      else								\
	dat_line_buffer_rest = -1;					\
    } while (0)

#if 1	/* ɥۥå᤮ */
		  /*
		   * ̾礤Ȥ줤ˤ
		   */
		  /* 1: ̾ζ */
		  while (name_len > 0 && name_head[name_len - 1] == ' ')
		    name_len--;

		  /* 2: ᥤ󤬶Ǥʤϡ</a>
		   *    ξϡ</font>ĤƤΤ
		   *    
		   */
		  if (mailto_len > 0)
		    {
		      /* </a>ĤƤ */
		      if (name_len > 4
			  && g_ascii_strncasecmp(name_head + name_len - 4,
						 "</a>", 4) == 0)
			name_len -= 4;
		    }
		  else
		    {
		      /* </font>ĤƤ */
		      if (name_len > 7
			  && g_ascii_strncasecmp(name_head + name_len - 7,
						 "</font>", 7) == 0)
			name_len -= 7;
		    }

		  /* 3: ᥤ<b></b>ǰϤޤƤɤ */
		  if (name_len > 4
		      && g_ascii_strncasecmp(name_head + name_len - 4,
					      "</b>", 4) == 0)
		    {
		      name_len -= 4;
		      if (name_len > 3
			  && g_ascii_strncasecmp(name_head, "<b>", 3) == 0)
			{
			  name_len -= 3;
			  name_head += 3;
			}
		    }

		  /* 4: ̾Ƭζ */
		  while (name_len > 0 && *name_head == ' ')
		    {
		      name_len--;
		      name_head++;
		    }

		  /* 5: ̾ζ */
		  while (name_len > 0 && name_head[name_len - 1] == ' ')
		    name_len--;
#endif

		  APPEND_TEXT_TO_DAT_LINE_BUFFER(name_head, name_len);
		  APPEND_TEXT_TO_DAT_LINE_BUFFER(mailto_head, mailto_len);
		  APPEND_TEXT_TO_DAT_LINE_BUFFER(date_head, date_len);
		  APPEND_TEXT_TO_DAT_LINE_BUFFER(body_head, body_len);
		  if (response_number == 1 && title != NULL)
		    {
		      int title_len = strlen(title);
		      if (dat_line_buffer_rest > title_len)
			{
			  memcpy(dat_line_buffer_pos, title, title_len);
			  dat_line_buffer_rest -= title_len;
			  dat_line_buffer_pos += title_len;
			}
		      else
			dat_line_buffer_rest = -1;
		    }
		  if (dat_line_buffer_rest > 1)
		    {
		      dat_line_buffer_pos[0] = '\n';
		      dat_line_buffer_rest -= 1;
		      dat_line_buffer_pos += 1;
		    }
		  else
		    dat_line_buffer_rest = -1;
#undef APPEND_TEXT_TO_DAT_LINE_BUFFER
		  if (dat_line_buffer_rest > 0)
		    {
		      ochusha_async_buffer_append_data(dat_buffer,
				dat_line_buffer,
				dat_line_buffer_pos - dat_line_buffer,
				"ochusha_utils_jbbs.c: convert_html_to_dat()");
#if DEBUG_HTML_TO_DAT_CONVERTER_MOST
		      {
			dat_line_buffer_pos[0] = '\0';
			fprintf(stderr, "DAT: %s\n", dat_line_buffer);
		      }
#endif
		    }
		  else
		    {
		      /* Ĺ */
		      fprintf(stderr, "Implement this!\n");
		    }

		  response_number++;
		}

	      if (title != NULL)
		{
		  G_FREE(title);
		  title = NULL;
		}
	    }
	  else
	    {
	      if (previous_line != NULL)
		{
#if DEBUG_HTML_TO_DAT_CONVERTER
		  fprintf(stderr, "IRREGULAR html_line: \"%s\"\n",
			  html_line_buffer);
#endif
		  G_FREE(html_line_buffer);
		  html_line_buffer = previous_line;
		  previous_line = NULL;
		}

	      if (g_ascii_strncasecmp("<dt>", html_line_buffer, 4) == 0)
		{
		  int len = strlen(html_line_buffer) - 1;
		  while (len > 0 &&
			 (html_line_buffer[len] == '\n'
			  || html_line_buffer[len] == '\r'))
		    {
		      html_line_buffer[len--] = '\0';
		    }
		  previous_line = html_line_buffer;
#if DEBUG_HTML_TO_DAT_CONVERTER
		  fprintf(stderr, "IRREGULAR html_line: \"%s\"\n",
			  html_line_buffer);
#endif
		  html_line_buffer = NULL;
		}
	    }

	  if (html_line_buffer != NULL)
	    G_FREE(html_line_buffer);

	  html_buffer_pos = eol_pos + 1;
	  rest_of_data -= line_size;
	  html_offset += line_size;
	}

      if (html_buffer->fixed)
	{
#if DEBUG_HTML_TO_DAT_CONVERTER_MOST
	  fprintf(stderr, "conversion done.\n");
#endif
	  g_signal_emit_by_name(G_OBJECT(args->broker),
				"access_completed",
				dat_buffer,
				&signal_result);

	  if (previous_line != NULL)
	    {
	      G_FREE(previous_line);
	      previous_line = NULL;
	    }
	  break;
	}

      if (!ochusha_async_buffer_wait(html_buffer,
				"ochusha_utils_jbbs.c: convert_html_to_dat()"))
	{
#if DEBUG_HTML_TO_DAT_CONVERTER_MOST
	  fprintf(stderr, "wait terminated.\n");
#endif
	  g_signal_emit_by_name(G_OBJECT(args->broker),
				"access_terminated",
				dat_buffer,
				&signal_result);
	  if (previous_line != NULL)
	    {
	      G_FREE(previous_line);
	      previous_line = NULL;
	    }
	  break;
	}
#if DEBUG_HTML_TO_DAT_CONVERTER_MOST
      else
	fprintf(stderr, "wait signaled.\n");
#endif
    }
  ochusha_async_buffer_unlock(html_buffer);

  ochusha_async_buffer_active_unref(html_buffer,
				"ochusha_utils_jbbs.c: convert_html_to_dat()");
  ochusha_async_buffer_active_unref(dat_buffer,
				"ochusha_utils_jbbs.c: convert_html_to_dat()");

  /* ɬפʤХåեƤ򥭥å˽񤭹 */
  if (dat_buffer->length > incoming_offset)
    {
      const char *fake_dat_url = ochusha_thread_2ch_get_dat_url(thread_2ch);
      int fd = ochusha_config_cache_open_file(args->broker->config,
					      fake_dat_url,
					      O_WRONLY | O_TRUNC | O_CREAT);
      if (fd >= 0)
	{
	  ssize_t len = write(fd, (void *)dat_buffer->buffer,
			      dat_buffer->length);
	  close(fd);
	  if (len != dat_buffer->length)
	    {
	      ochusha_config_cache_unlink_file(args->broker->config,
					       fake_dat_url);
	      fd = -1;
	    }
	}
#if DEBUG_NETWORK
      if (fd < 0)
	fprintf(stderr, "Couldn't update cache file for %s: %s (%d)\n",
		fake_dat_url, strerror(errno), errno);
#endif	      
    }

  ochusha_async_buffer_fix(dat_buffer,
			   "ochusha_utils_jbbs.c: convert_html_to_dat()");

 terminated:
  if (title_pattern != NULL)
    regex_free(title_pattern);
  if (response_pattern != NULL)
    regex_free(response_pattern);
  if (match_region != NULL)
    regex_region_free(match_region, 1);
  g_object_unref(G_OBJECT(dat_buffer));
  g_object_unref(G_OBJECT(args->broker));
}


static void
unref_html_buffer(OchushaAsyncBuffer *html_buffer)
{
  /* dat_buffer̤Ȥ˸ƤФ롣*/
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr, "unref_html_buffer: %p\n", html_buffer);
#endif
  g_object_unref(html_buffer);
}
#endif	/* ENABLE_HTML_TO_DAT_CONVERTER */


/*
 * ochusha_utils_jbbs_fake_responses_source
 *
 * cgiͳhtmlʥɤ߹ߡ2chDATѴХåե֤
 * cgiͳǤɹϥͥåȥ򤹤륹åɤǹԤ2chDAT
 * ѴΤѤΥåɤǹԤ
 *
 * δؿ֤Τϡ̥åɤˤäDATѴ줿̤Ǽ
 * ХåեǤꡢͥåȥľܻȤХåե̤˺롣
 */
OchushaAsyncBuffer *
ochusha_utils_jbbs_fake_responses_source(OchushaThread2ch *thread_2ch,
					 OchushaNetworkBroker *broker,
					 OchushaNetworkBrokerCacheMode mode)
{
#if ENABLE_HTML_TO_DAT_CONVERTER
  OchushaAsyncBuffer *dat_buffer = NULL;
  OchushaAsyncBuffer *html_buffer;
  OchushaBBSThread *thread = OCHUSHA_BBS_THREAD(thread_2ch);
  OchushaBoard2ch *board_2ch = OCHUSHA_BOARD_2CH(thread->board);
  const char *fake_dat_url = ochusha_thread_2ch_get_dat_url(thread_2ch);
  WorkerJob *job;
  HTML2DATConverterJobArgs *args;
  int incoming_offset = 0;
  int number_of_cached_responses = 0;
  char cgi_url[PATH_MAX];

  switch (thread->board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_JBBS:
    case OCHUSHA_BBS_TYPE_MACHIBBS:
    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
    case OCHUSHA_BBS_TYPE_MITINOKU:
      break;

    default:
      return NULL;	/* ̤ */
    }

  if (mode != OCHUSHA_NETWORK_BROKER_CACHE_IGNORE)
    {
      int fd = ochusha_config_cache_open_file(broker->config, fake_dat_url,
					      O_RDONLY);
      if (fd >= 0)
	{
	  /* å夢ꡣ*/
	  dat_buffer = ochusha_async_buffer_new(NULL, 0, NULL);
	  incoming_offset = lseek(fd, 0, SEEK_END);
	  lseek(fd, 0, SEEK_SET);
	  if (ochusha_async_buffer_resize(dat_buffer, incoming_offset,
					  "ochusha_utils_jbbs.c: ochusha_utils_jbbs_fake_responses_source()"))
	    {
	      char *tmp_pos;
	      char *tail_pos;
	      incoming_offset = read(fd, (void *)dat_buffer->buffer,
				     incoming_offset);
	      if (!ochusha_async_buffer_update_length(dat_buffer,
						      incoming_offset,
						      "ochusha_utils_jbbs.c: ochusha_utils_jbbs_fake_responses_source()"))
		{
#if DEBUG_ASYNC_BUFFER
		  fprintf(stderr, "ochusha_utils_jbbs_fake_responses_source(): buffer may have been terminated.\n");
#endif
		  /* ξϤɤ褦ʤ */
		}

	      /* ¸Υ쥹 */
	      tmp_pos = (char *)dat_buffer->buffer;
	      tail_pos = tmp_pos + incoming_offset;
	      while ((tmp_pos = memchr(tmp_pos, '\n', tail_pos - tmp_pos))
		     != NULL)
		{
		  number_of_cached_responses++;
		  tmp_pos++;
		}
	    }
	  close(fd);
	}
    }

  /* λǼεDATե뤬ѲǽʾΤdat_bufferNULL */

  switch (thread->board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_JBBS:
    case OCHUSHA_BBS_TYPE_MACHIBBS:
    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
      if (snprintf(cgi_url, PATH_MAX, "%s?BBS=%s&KEY=%s&START=%d%s",
		   ochusha_board_2ch_get_read_cgi_url(board_2ch),
		   thread->board->id, thread->id,
		   number_of_cached_responses + 1,
		   number_of_cached_responses == 0 ? "" : "&NOFIRST=TRUE")
	  >= PATH_MAX)
	cgi_url[0] = '\0';
      break;

    case OCHUSHA_BBS_TYPE_MITINOKU:
      if (thread->number_of_responses_on_server <= number_of_cached_responses)
	return dat_buffer;

      if (snprintf(cgi_url, PATH_MAX, "%s?bbs=%s&key=%s&st=%d&to=%d%s",
		   ochusha_board_2ch_get_read_cgi_url(board_2ch),
		   thread->board->id, thread->id,
		   number_of_cached_responses + 1,
		   thread->number_of_responses_on_server,
		   number_of_cached_responses == 0 ? "" : "&nofirst=true")
	  >= PATH_MAX)
	cgi_url[0] = '\0';
      break;

    default:	/* ̤ݡ */
      cgi_url[0] = '\0';
      break;
    }

#if DEBUG_NETWORK
  fprintf(stderr, "cgi_url=\"%s\"\n", cgi_url);
#endif

  if (cgi_url[0] != '\0')
    html_buffer = ochusha_network_broker_read_from_cgi(broker, cgi_url);
  else
    html_buffer = NULL;

  if (html_buffer == NULL)
    {
      if (dat_buffer != NULL)
	{
	  gboolean signal_result;
	  g_signal_emit_by_name(G_OBJECT(broker),
				"access_failed",
				dat_buffer,
				OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
				_("Access failed: unknown reason"),
				&signal_result);
	  ochusha_async_buffer_fix(dat_buffer,
				   "ochusha_utils_jbbs.c: ochusha_utils_jbbs_fake_responses_source()");
	}
      return dat_buffer;
    }

  if (dat_buffer == NULL)
    dat_buffer = ochusha_async_buffer_new(NULL, 0, NULL);

  args = G_NEW0(HTML2DATConverterJobArgs, 1);
  args->html_buffer = html_buffer;
  args->dat_buffer = dat_buffer;
  args->broker = broker;
  args->thread_2ch = thread_2ch;
  args->incoming_offset = incoming_offset;
  args->number_of_cached_responses = number_of_cached_responses;

  g_object_set_data(G_OBJECT(dat_buffer), "OchushaNetworkBroker::BufferStatus",
		    g_object_get_data(G_OBJECT(html_buffer),
				      "OchushaNetworkBroker::BufferStatus"));
  g_object_set_data_full(G_OBJECT(dat_buffer), "OchushaUtilsJBBS::HTMLBuffer",
			 html_buffer, (GDestroyNotify)unref_html_buffer);

  job = G_NEW0(WorkerJob, 1);
  job->canceled = FALSE;
  job->job = (JobFunc *)convert_html_to_dat;
  job->args = args;

  /* åɤΤ˳ */
  g_object_ref(G_OBJECT(dat_buffer));
  g_object_ref(G_OBJECT(broker));

  commit_job(job);

  return dat_buffer;
#else
  return NULL;
#endif	/* ENABLE_HTML_TO_DAT_CONVERTER */
}
