/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id$
 */

#include "ochusha.h"
#include "ochusha_private.h"

#include "htmlutils.h"

#include <glib.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define USE_AS_IS_TEXT	0


static void unmark_alive(gpointer data, gpointer user_data);
static void collect_dropped_thread(gpointer data, gpointer user_data);


/*
 * ochusha_analyze_subject_txt
 *
 * Ϳ줿ХåեƤ2chĤsubject.txtȸʤƲϤͿ줿
 * BulletinBoard¤Τthread_listthread_tableʤɤ򹹿롣å
 * ˻ꤵ줿ХåؿƤ֡
 *
 * converterNULLξ硢Ѥƥåɥȥʸencoding
 * Ѵ롣λsimple_string_canonѤΤǥåɥȥ
 * &lt;Τ褦entity referenceʸѴ롣&#xxxx;entity
 * UTF-8ѴΤǡUTF-8ؤconverterʳϹθƤʤȤա
 *
 * ХåؿFALSE֤硢λǲϤλ롣
 *
 * ϤˤTRUE֤ԤFALSE֤
 *
 * MEMO: ϽλˡDATåɤбBBSThreadۤäfree
 *       뤬Υåɤɽȥ˵դΤǡ
 *       Ŭн褹٤
 *       褺DATåɤfree̤ΥꥹȤˤĤʤǼäƤ
 *         Ȥˤ¾ˤȤƤϡlibochushaγƹ¤Τ
 *         GObjectΥ֥饹ˤƤޤref_countȤȤ⤢뤬
 *         ޤǤͤϤʤȻפʤGCΤǽ񤯤衣
 *         ġĤȤref_countϻȤΤݽ
 */
gboolean
ochusha_analyze_subject_txt(BulletinBoard *board, AsyncBuffer *buffer,
			    iconv_t converter,
			    EachThreadCallback *each_thread_cb,
			    gpointer user_data)
{
  char default_buffer[PATH_MAX];
  char scan_buffer[5];
  gboolean result;
  GSList *thread_list = NULL;
  GHashTable *thread_table = board->thread_table;
  GSList *old_thread_list = board->thread_list;

  if (!async_buffer_active_ref(buffer, "threadlist.c: ochusha_analyze_subject_txt"))
    {
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "buffer has been terminated.\n");
#endif
      return FALSE;
    }

  g_slist_foreach(old_thread_list, unmark_alive, NULL);

  async_buffer_lock(buffer);
  {
    unsigned int offset = 0;

    while (TRUE)
      {
	char *buffer_top = (char *)buffer->buffer;
	char *cur_pos = buffer_top + offset;
	unsigned int length = buffer->length;
	unsigned int rest_of_data = length - offset;
	char *eol_pos;

	while (rest_of_data > 0
	       && (eol_pos = memchr(cur_pos, '\n', rest_of_data)) != NULL)
	  {
	    char *dat_file;
	    gchar *thread_title;
	    BBSThread *thread;
	    char *title_pos;
	    int n_responses_on_server = 0;
	    char *tmp_pos = g_strstr_len(cur_pos, eol_pos - cur_pos, "<>");
	    int title_len;
	    if (tmp_pos == NULL)
	      {
#if 0	/* debug */
		fprintf(stderr, "Unknown format found in subject.txt.\n");
#endif
		goto prepare_next_line;
	      }
	    if ((tmp_pos - cur_pos) < PATH_MAX)
	      {
		dat_file = memcpy(default_buffer, cur_pos, tmp_pos - cur_pos);
		dat_file[tmp_pos - cur_pos] = '\0';
	      }
	    else
	      dat_file = g_strndup(cur_pos, tmp_pos - cur_pos);
	    title_pos = tmp_pos + 2;	/* skip "<>" */
	    title_len = eol_pos - title_pos;

	    thread = g_hash_table_lookup(thread_table, dat_file);

	    for (tmp_pos = eol_pos - 1; tmp_pos > title_pos; tmp_pos--)
	      if (*tmp_pos == '(')
		break;

	    if (*tmp_pos == '(')
	      {
		char *close_paren = memchr(tmp_pos + 1, ')',
					   eol_pos - tmp_pos - 1);
		if (close_paren != NULL)
		  {
		    int tmp_len = close_paren - tmp_pos - 1;
		    if (tmp_len > 0 && tmp_len <= 4)
		      {
			memcpy(scan_buffer, tmp_pos + 1, tmp_len);
			scan_buffer[tmp_len] = '\0';
			sscanf(scan_buffer, "%d", &n_responses_on_server);
			title_len -= (tmp_len + 2);
		      }
		  }
	      }

	    if (thread == NULL)
	      {
		if (converter != NULL)
		  thread_title = simple_string_canon(title_pos, title_len,
						     converter);
		else
		  thread_title = g_strndup(title_pos, title_len);

		thread = ochusha_bbs_thread_new(board, g_strdup(dat_file),
						thread_title);
		if (thread == NULL)
		  {
		    if (dat_file != NULL && dat_file != default_buffer)
		      free(dat_file);
		    if (thread_title != NULL)
		      free(thread_title);
		    result = FALSE;
		    break;
		  }
		g_hash_table_insert(thread_table,
				    thread->dat_filename, thread);
	      }
	    else if (dat_file != default_buffer)
	      free(dat_file);

	    thread->alive = TRUE;

	    thread->number_of_responses_on_server = n_responses_on_server;

	    thread_list = g_slist_append(thread_list, thread);

	    if (each_thread_cb != NULL
		&& !(*each_thread_cb)(thread, user_data))
	      {
		result = FALSE;
		break;
	      }

	  prepare_next_line:
	    rest_of_data -= (eol_pos - cur_pos + 1);
	    cur_pos = eol_pos + 1;
	    offset = cur_pos - buffer_top;
	  }

	if (buffer->fixed)
	  {
	    result = TRUE;
	    break;
	  }

	if (!async_buffer_wait(buffer, "threadlist.c: ochusha_analyze_subject_txt"))
	  {
#if DEBUG_ASYNC_BUFFER_MOST
	    fprintf(stderr, "buffer has been terminated.\n");
#endif
	    result = FALSE;
	    break;
	  }

	if (((OchushaNetworkStatus *)buffer->user_data)->state
	    == OCHUSHA_CACHE_IS_DIRTY)
	  {
	    result = FALSE;
	    break;
	  }
      }
  }
  async_buffer_unlock(buffer);

  async_buffer_active_unref(buffer, "threadlist.c: ochusha_analyze_subject_txt");

  g_slist_foreach(old_thread_list, collect_dropped_thread, NULL);
  g_slist_free(old_thread_list);

  board->thread_list = thread_list;

  return result;
}


static void
unmark_alive(gpointer data, gpointer user_data)
{
  BBSThread *thread = (BBSThread *)data;
  thread->alive = FALSE;
}


static void
collect_dropped_thread(gpointer data, gpointer user_data)
{
  BBSThread *thread = (BBSThread *)data;
  if (!thread->alive)
    {
      /* threadDATäݤ*/
      thread->board->dropped_list = g_slist_append(thread->board->dropped_list,
						   thread);
    }
}
