/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id$
 */

#include "ochusha.h"
#include "ochusha_private.h"

#include <glib.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


#define DEBUG_DAT	0
#define DEBUG_DAT_MOST	0

#define ANALYSIS_INTERVAL	20

static char *extract_response(char *start_pos, char *eol_pos,
			      BBSResponse *response);


/*
 * ochusha_bbs_thread_new
 *
 * BBSThread¤Τ򿷤˳Ƥ롣­ξˤ
 * NULL֤
 */
BBSThread *
ochusha_bbs_thread_new(BulletinBoard *board, gchar *dat_filename, gchar *title)
{
  BBSThread *new_thread;
  gchar pseudo_path[PATH_MAX];
  gchar *tmp_pos;

  if (board == NULL || dat_filename == NULL || title == NULL)
    return NULL;

  if (strlen(dat_filename) >= PATH_MAX)
    return NULL;

  new_thread = (BBSThread *)calloc(1, sizeof(BBSThread));
  if (new_thread == NULL)
    return NULL;	/* out of memory */

  if (snprintf(pseudo_path, PATH_MAX, "test/read.cgi/%s%s",
	       board->base_path, dat_filename) >= PATH_MAX)
    goto free_and_exit;

  tmp_pos = strstr(pseudo_path, ".dat");

  if (tmp_pos == NULL)
    goto free_and_exit;

  tmp_pos[0] = '/';
  tmp_pos[1] = '\0';

  new_thread->pseudo_path = strdup(pseudo_path);
  if (new_thread->pseudo_path == NULL)
    goto free_and_exit;

  new_thread->board = board;
  new_thread->dat_filename = dat_filename;
  new_thread->title = title;

#if 0
  fprintf(stderr, "bbs_thread_new: new_thread=0x%x\n", (int)new_thread);
  fprintf(stderr, "  new_thread->board=0x%x\n", (int)board);
  fprintf(stderr, "  new_thread->dat_filename=\"%s\"\n", dat_filename);
  fprintf(stderr, "  new_thread->title=\"%s\"\n", title);
#endif
  return new_thread;

 free_and_exit:
  free(new_thread);
  return NULL;
}


/*
 * ochusha_bbs_thread_free
 *
 * ƤBBSThread¤Τ롣
 */
void
ochusha_bbs_thread_free(BBSThread *thread)
{
  if (thread->title != NULL)
    free(thread->title);
  if (thread->dat_filename != NULL)
    free(thread->dat_filename);
  if (thread->dat_url != NULL)
    free(thread->dat_url);
  if (thread->responses != NULL)
    free(thread->responses);
  free(thread);
}


/*
 * Ϳ줿ФDATեURL֤
 */
const char *
ochusha_bbs_thread_get_dat_url(BBSThread *thread)
{
  char url[PATH_MAX];
  if (thread->dat_url != NULL)
    return thread->dat_url;

  g_return_val_if_fail(thread->board != NULL, NULL);

  if (snprintf(url, PATH_MAX, "%sdat/%s",
	       thread->board->base_url, thread->dat_filename) >= PATH_MAX)
    return NULL;

  thread->dat_url = strdup(url);
  return thread->dat_url;
}


/*
 * extract_response
 *
 * cur_posʸ2chDATǡ1쥹ʬȸʤƲϤ
 * 1쥹ɽBBSResponse¤Τ˳Ǽ롣
 * δؿϡߤΰ(cur_pos)龯ʤȤ1ʬDATǡ
 * ¸ߤꤹΤǸƤӽФ¦դɬסʾ֤ǸƤӽФȡ
 * Хåեۤǡ˥뤳Ȥˤʤ롣
 *
 * 1쥹ʬΥǡ줿硢쥹ľΥǥߥ"<>"
 * ľʸؤݥ󥿤֤ԻˤNULL֤
 */
static char *
extract_response(char *start_pos, char *eol_pos, BBSResponse *response)
{
  /*
   * DATեγƹԤ
   * ̾<>ᥤ<><>쥹<>[åɥȥ]
   * ȤʤäƤ롣
   */
  char *clone_buffer = NULL;
  char *cur_pos = start_pos;
  char *tmp_pos;
  gboolean comma_delimiter = FALSE;
  int i = 0;

  /* DATǡƤ뤳Ȥ⤢Τǥå */
  tmp_pos = memchr(start_pos, '\0', eol_pos - cur_pos);
  if (tmp_pos != NULL)
    {
      int rest;
      /* 줿DATǡ */
      clone_buffer = (char *)malloc(eol_pos - cur_pos);
      if (clone_buffer == NULL)
	goto error_exit;

      memcpy(clone_buffer, start_pos, eol_pos - cur_pos);
      /*
       * XXX: ԡƤ'\0'ŬʸѤΤȡ
       *      '\0'ФʤͤƥԡΤΤɤ餬ɤΤġġ
       *      ҤȤޤԡơƬ'\0'Ф
       */
      rest = eol_pos - cur_pos;

      cur_pos = clone_buffer;
      eol_pos = clone_buffer + rest;
      while (*cur_pos == '\0')
	cur_pos++;

      rest -= (cur_pos - clone_buffer);

      tmp_pos = cur_pos + 1;
      while (--rest > 0)
	if (*tmp_pos == '\0')
	  *tmp_pos++ = '*';

#if DEBUG_DAT
      {
	char *broken_line = g_strndup(cur_pos, eol_pos - cur_pos);
	fprintf(stderr, "broken line: %s\n", broken_line);
	free(broken_line);
      }
#endif
    }


  tmp_pos = g_strstr_len(cur_pos, eol_pos - cur_pos, "<>");
  if (tmp_pos == NULL)
    goto error_exit;

  while (tmp_pos != NULL)
    {
      i++;
      tmp_pos = g_strstr_len(tmp_pos + 2, eol_pos - tmp_pos - 2, "<>");
    }

  if (i == 0)
    goto error_exit;

  if (i == 1)
    {
      i = 0;
      tmp_pos = memchr(start_pos, ',', eol_pos - start_pos);
      while (tmp_pos != NULL)
	{
	  i++;
	  tmp_pos = memchr(tmp_pos + 1, ',', eol_pos - tmp_pos - 1);
	}
      if (i != 3)
	goto error_exit;	/* ȤΤʤ */
      comma_delimiter = TRUE;
    }

  tmp_pos = comma_delimiter ? memchr(cur_pos, ',', eol_pos - cur_pos)
                            : g_strstr_len(cur_pos, eol_pos - cur_pos, "<>");
  response->name = g_strndup(cur_pos, tmp_pos - cur_pos);
  if (response->name == NULL)
    goto error_exit;

  cur_pos = tmp_pos + (comma_delimiter ? 1 : 2);
  /* DATǡƤ뤳Ȥ⤢ */
  while (*cur_pos == '\0')
    cur_pos++;

  tmp_pos = comma_delimiter ? memchr(cur_pos, ',', eol_pos - cur_pos)
                            : g_strstr_len(cur_pos, eol_pos - cur_pos, "<>");
  if (tmp_pos - cur_pos > 0)
    {
      response->mailto = g_strndup(cur_pos, tmp_pos - cur_pos);
      if (response->mailto == NULL)
	goto error_exit;
    }
  else
    response->mailto = NULL;

  cur_pos = tmp_pos + (comma_delimiter ? 1 : 2);
  /* DATǡƤ뤳Ȥ⤢ */
  while (*cur_pos == '\0')
    cur_pos++;

  tmp_pos = comma_delimiter ? memchr(cur_pos, ',', eol_pos - cur_pos)
                            : g_strstr_len(cur_pos, eol_pos - cur_pos, "<>");
  if (tmp_pos == NULL)
    goto error_exit;

  response->date_id = g_strndup(cur_pos, tmp_pos - cur_pos);
  if (response->date_id == NULL)
    goto error_exit;

  cur_pos = tmp_pos + (comma_delimiter ? 1 : 2);
  /* DATǡƤ뤳Ȥ⤢ */
  while (*cur_pos == 0)
    cur_pos++;

  tmp_pos = g_strstr_len(cur_pos, eol_pos - cur_pos, "<>");
  if (tmp_pos == NULL)
    goto error_exit;

  response->content = g_strndup(cur_pos, tmp_pos - cur_pos);
  if (response->content == NULL)
    goto error_exit;

  if (clone_buffer != NULL)
    {
      free(clone_buffer);
      return start_pos + (tmp_pos - clone_buffer) + 2;
    }
  else
    return tmp_pos + 2;

 error_exit:
#if DEBUG_DAT_MOST
  tmp_pos = g_strndup(start_pos, eol_pos - start_pos);
  fprintf(stderr, "DAT Parsing failed: \"%s\"", tmp_pos);
  free(tmp_pos);
#endif
  if (response->name != NULL)
    {
#if DEBUG_DAT
      fprintf(stderr, "name=%s\n", response->name);
#endif
      free(response->name);
      response->name = NULL;
    }

  if (response->mailto != NULL)
    {
#if DEBUG_DAT
      fprintf(stderr, "mailto=%s\n", response->mailto);
#endif
      free(response->mailto);
      response->mailto = NULL;
    }

  if (response->date_id != NULL)
    {
#if DEBUG_DAT
      fprintf(stderr, "date_id=%s\n", response->date_id);
#endif
      free(response->date_id);
      response->date_id = NULL;
    }

  if (response->content != NULL)
    {
#if DEBUG_DAT
      fprintf(stderr, "content=%s\n", response->content);
#endif
      free(response->content);
      response->content = NULL;
    }

  if (clone_buffer != NULL)
    free(clone_buffer);

  return NULL;
}


/*
 * ochusha_analyze_dat
 *
 * bufferƤåɤDATǡǤȸʤƥåɤ
 * ɽBBSThread¤Τˤ륹åɤʬۤ롣
 *
 * åɤγϻåɥȥˤstart_thread_cb
 * ġΥ쥹Фeach_response_cbåΤβϽλˤ
 * end_thread_cb򤽤줾ƤӽФΥХåFALSE֤顢
 * λǲϤŪˤ롽褦ˤͽꡣ
 *
 * start+1ܤnumberʬΥ쥹ϤġĤȤؿˤͽꡣ
 * åã硢number˴ؤ餺Ϥλ롣
 *
 * number == -1ʤ顢start饹åޤǡ
 * start == -1ʤ顢ånumberʬΥ쥹
 *
 * Ĥޤꡢstart=0,number=-1ǡ쥹ϡ
 *         start=-1,number=100Ǻǿ100쥹Ȥä
 * 褺start=-1ξ̤Τޤ֡
 */
gboolean
ochusha_analyze_dat(BBSThread *thread, AsyncBuffer *buffer,
		    int start, int number,
		    StartThreadCallback *start_thread_cb,
		    EachResponseCallback *each_response_cb,
		    EndThreadCallback *end_thread_cb,
		    gpointer user_data)
{
  unsigned int *responses = thread->responses;
  gboolean result = TRUE;

  if (!async_buffer_active_ref(buffer, "bbs_thread.c: ochusha_analyze_dat"))
    {
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "buffer has been terminated.\n");
#endif
      return FALSE;
    }

  if (responses == NULL)
    {
      /*
       * 1000쥹ʬǤ⡹4000ХȤʤΤǸĹ϶Ȥޤ衣
       */
      responses = (unsigned int *)calloc(MAX_RESPONSE, sizeof(unsigned int));
      if (responses == NULL)
	{
	  fprintf(stderr, "Out of memory during analysis of DAT.\n");
	  async_buffer_active_unref(buffer, "bbs_thread.c: ochusha_analyze_dat");
	  return FALSE;
	}
      thread->responses = responses;
    }

  if (number == -1 || number > MAX_RESPONSE)
    number = MAX_RESPONSE;

  async_buffer_lock(buffer);
  {
    int rest_of_responses;
    unsigned int offset;
    int i;
    gboolean buffer_fixed = FALSE;

  retry_analysis:
    rest_of_responses = number;
    offset = 0;
    i = 0;
    if (((OchushaNetworkStatus *)buffer->user_data)->state
	!= OCHUSHA_CACHE_IS_DIRTY)
      {
	if (start > 1 && start < MAX_RESPONSE)
	  {
	    offset = responses[start];
	    if (offset != 0)
	      i = start;
	    else
	      {
		/*
		 * nʬΥ쥹ɤ߹n+1ܰʹߤɤ
		 * ġĤȤΤ̰ƤХʥꥵ
		 * ȤơǤᤤϺʬõȤǤ뤬
		 * ޤǤɬפʤ
		 */
		offset = responses[start - 1];
		if (offset != 0)
		  i = start - 1;
		else
		  offset = 0;
	      }
	  }
      }

    while (TRUE)
      {
	char *buffer_top = (char *)buffer->buffer;
	char *cur_pos = buffer_top + offset;
	int length = buffer->length;
	int rest_of_data = length - offset;
	char *eol_pos;
	int interval = ANALYSIS_INTERVAL;

#if DEBUG_DAT_MOST
	fprintf(stderr, "1: bbs_thread.c: length=%d\n", length);
#endif
	while (result && rest_of_data > 0 && rest_of_responses > 0
	       && (buffer_fixed || interval-- > 0)
	       && (eol_pos = memchr(cur_pos, '\n', rest_of_data)) != NULL)
	  {
	    BBSResponse response = { NULL, NULL, NULL, NULL };
	    char *title_pos;

#if DEBUG_DAT_MOST
	    fprintf(stderr, "i=%d, offset=%d, rest_of_data=%d, cur_pos=0x%x\n",
		    i, offset, rest_of_data, (int)cur_pos);
#endif

	    title_pos = extract_response(cur_pos, eol_pos, &response);

#if DEBUG_DAT_MOST
	    fprintf(stderr, "title_pos=0x%x\n", (int)title_pos);
#endif

	    if (i == 0 && start == 0 && title_pos != NULL)
	      {
		char *title = g_strndup(title_pos, eol_pos - title_pos);
		if (start_thread_cb != NULL)
		  result = (*start_thread_cb)(thread, title, user_data);
		free(title);
		if (!result)
		  break;
	      }
	    responses[i] = offset;
	    cur_pos = eol_pos + 1;
	    offset = cur_pos - buffer_top;
	    rest_of_data = length - offset;

	    if (title_pos != NULL)
	      {
		i++;
		if (i > start)
		  {
		    if (each_response_cb != NULL)
		      result = (*each_response_cb)(thread, i, &response,
						   user_data);
		    rest_of_responses--;
		  }
		if (!result)
		  break;
	      }
#if DEBUG_DAT
	    else
	      {
		char *line;
		fprintf(stderr, "response #%d: Couldn't analyze DAT file...ignoring parts.\n", i);
		line = g_strndup(buffer_top + responses[i], offset - responses[i]);
		fprintf(stderr, "failed: \"%s\"\n", line);
		free(line);
		fprintf(stderr, "offset=%d\n", responses[i]);
	      }
#endif
	    if (response.name != NULL)
	      free(response.name);
	    if (response.mailto != NULL)
	      free(response.mailto);
	    if (response.date_id != NULL)
	      free(response.date_id);
	    if (response.content != NULL)
	      free(response.content);
	  }

	if (!buffer_fixed && buffer->fixed)
	  {
	    buffer_fixed = TRUE;
	    continue;
	  }
	if (buffer_fixed)
	  break;

	if (!async_buffer_wait(buffer, "bbs_thread.c: ochusha_analyze_dat"))
	  {
	    /* bufferϤ޲٤callerˤޤ */
#if DEBUG_ASYNC_BUFFER_MOST
	    fprintf(stderr, "ochusha_analyze_dat: buffer has been terminated.\n");
#endif
	    break;
	  }

	if (((OchushaNetworkStatus *)buffer->user_data)->state
	    == OCHUSHA_CACHE_IS_DIRTY)
	  {
	    if (i > start && end_thread_cb != NULL)
	      result = (*end_thread_cb)(thread, FALSE, user_data);
	    /* å夬äƤ뤳ȤϤ⤦狼ä */
	    ((OchushaNetworkStatus *)buffer->user_data)->state
	      = OCHUSHA_DIRECT_ACCESS;
	    goto retry_analysis;
	  }
      }

    if (number == MAX_RESPONSE)
      thread->number_of_responses_read = i;
  }
  async_buffer_unlock(buffer);

  async_buffer_active_unref(buffer, "bbs_thread.c: ochusha_analyze_dat");

  if (end_thread_cb != NULL)
    result = (*end_thread_cb)(thread, TRUE, user_data);

  return TRUE;
}
