/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id$
 */

#include "asyncbuf.h"

#include <glib.h>
#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>


struct _PThreadSyncObject
{
  pthread_mutex_t mutex;
  pthread_cond_t cond;
};


static GObjectClass *parent_class = NULL;

static pthread_mutex_t async_buffer_global_lock;
static pthread_cond_t async_buffer_global_condition;

/*
 * pthreadϢΥ֥ȤλȤΤƤƧߤʤΤǡѤ
 * ݤ롣
 */
static GSList *sync_object_list = NULL;
/*
 * 饤֤(newƤfinalizeޤǤδ֤)ƤAsyncBufferݻ
 */
static GSList *async_buffer_list = NULL;


#define LOCK_GLOBAL_LOCK					\
  if (pthread_mutex_lock(&async_buffer_global_lock) != 0)	\
    {								\
      fprintf(stderr, "Couldn't lock a mutex.\n");		\
      abort();							\
    }

#define UNLOCK_GLOBAL_LOCK					\
  if (pthread_mutex_unlock(&async_buffer_global_lock) != 0)	\
    {								\
      fprintf(stderr, "Couldn't unlock a mutex.\n");		\
      abort();							\
    }

#define WAIT_GLOBAL_CONDITION					\
  if (pthread_cond_wait(&async_buffer_global_condition,		\
			&async_buffer_global_lock) != 0)	\
    {								\
      fprintf(stderr, "Couldn't wait a condition.\n");		\
      abort();							\
    }

#define SIGNAL_GLOBAL_CONDITION					\
  if (pthread_cond_signal(&async_buffer_global_condition) != 0)	\
    {								\
      fprintf(stderr, "Couldn't unlock a mutex.\n");		\
      abort();							\
    }

#define LOCK_BUFFER_LOCK(buffer)				\
  if (pthread_mutex_lock(&buffer->sync_object->mutex) != 0)	\
    {								\
      fprintf(stderr, "Couldn't lock a mutex.\n");		\
      abort();							\
    }

#define UNLOCK_BUFFER_LOCK(buffer)				\
  if (pthread_mutex_unlock(&buffer->sync_object->mutex) != 0)	\
    {								\
      fprintf(stderr, "Couldn't unlock a mutex.\n");		\
      abort();							\
    }

#define WAIT_BUFFER_CONDITION(buffer)				\
  if (pthread_cond_wait(&buffer->sync_object->cond,		\
			&buffer->sync_object->mutex) != 0)	\
    {								\
      fprintf(stderr, "Couldn't wait a condition.\n");		\
      abort();							\
    }

#define SIGNAL_BUFFER_CONDITION(buffer)				\
  if (pthread_cond_signal(&buffer->sync_object->cond) != 0)	\
    {								\
      fprintf(stderr, "Couldn't siganl a condition.\n");	\
      abort();							\
    }

#define BROADCAST_BUFFER_CONDITION(buffer)			\
  if (pthread_cond_broadcast(&buffer->sync_object->cond) != 0)	\
    {								\
      fprintf(stderr, "Couldn't siganl a condition.\n");	\
      abort();							\
    }


static void async_buffer_class_init(AsyncBufferClass *klass);
static void async_buffer_init(AsyncBuffer *buffer);
static void async_buffer_finalize(GObject *object);

static PThreadSyncObject *get_sync_object(void);


GType
async_buffer_get_type(void)
{
  static GType ab_type = 0;

  if (ab_type == 0)
    {
      static const GTypeInfo ab_info =
	{
	  sizeof(AsyncBufferClass),
	  NULL, /* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc)async_buffer_class_init,
	  NULL, /* class_finalize */
	  NULL, /* class_data */
	  sizeof(AsyncBuffer),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)async_buffer_init,
	};

      ab_type = g_type_register_static(G_TYPE_OBJECT,
				       "AsyncBuffer", &ab_info, 0);
    }

  return ab_type;
}


static void
async_buffer_class_init(AsyncBufferClass *klass)
{
  GObjectClass *o_class = (GObjectClass *)klass;

  if (pthread_mutex_init(&async_buffer_global_lock, NULL) != 0)
    {
      fprintf(stderr, "Couldn't initialize a mutex.\n");
      abort();
    }

  if (pthread_cond_init(&async_buffer_global_condition, NULL) != 0)
    {
      fprintf(stderr, "Couldn't initialize a condvar.\n");
      abort();
    }

  parent_class = g_type_class_peek_parent(klass);
  o_class->finalize = async_buffer_finalize;
}


static void
async_buffer_init(AsyncBuffer *buffer)
{
  buffer->fixed = FALSE;
}


static void
async_buffer_finalize(GObject *object)
{
  AsyncBuffer *buffer = ASYNC_BUFFER(object);
#if DEBUG_ASYNC_BUFFER
  fprintf(stderr, "async_buffer_finalize: AsyncBuffer(0x%x)\n", (int)object);
#endif
  if (buffer->destructor != NULL)
    (*buffer->destructor)(buffer, buffer->user_data);

  LOCK_GLOBAL_LOCK;
  if (buffer->sync_object != NULL)
    sync_object_list = g_slist_append(sync_object_list, buffer->sync_object);
  async_buffer_list = g_slist_remove(async_buffer_list, buffer);
  UNLOCK_GLOBAL_LOCK;

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


static PThreadSyncObject *
get_sync_object(void)
{
  GSList *recycled_sync_object;
  PThreadSyncObject *sync_object;

  recycled_sync_object = sync_object_list;
  sync_object_list = g_slist_remove_link(sync_object_list,
					 recycled_sync_object);
  if (recycled_sync_object != NULL)
    {
      sync_object = (PThreadSyncObject *)recycled_sync_object->data;
      g_slist_free_1(recycled_sync_object);
      return sync_object;
    }

  sync_object = (PThreadSyncObject *)g_malloc(sizeof(PThreadSyncObject));
  if (pthread_mutex_init(&sync_object->mutex, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a mutex.\n");
      abort();
    }
  if (pthread_cond_init(&sync_object->cond, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a condition variable.\n");
      abort();
    }
  return sync_object;
}


AsyncBuffer *
async_buffer_new(char *buffer, int length,
		 gpointer user_data, DestructFunc *destructor)
{
  PThreadSyncObject *sync_object;
  AsyncBuffer *buf = ASYNC_BUFFER(g_object_new(ASYNC_BUFFER_TYPE, NULL));

  LOCK_GLOBAL_LOCK
    {
      sync_object = get_sync_object();
      async_buffer_list = g_slist_append(async_buffer_list, buf);
    }
  UNLOCK_GLOBAL_LOCK

  if (buffer == NULL && destructor == NULL)
    {
      if (length != 0)
	buffer = (char *)g_malloc(length);
      destructor = free_when_finished;
      buf->length = 0;
    }
  else
    buf->length = length;
  
  buf->fixed = FALSE;
  buf->buffer = buffer;

  buf->buffer_length = length;	/* XXX: Ǥ餦٤ġġ*/

  buf->sync_object = sync_object;
  buf->user_data = user_data;
  buf->destructor = destructor;

  return buf;
}


/* ƱߴϢ */
/*
 * MEMO: AsyncBufferȤäƤƤΥåɤ⤷ΥåɤФ
 *       ᥤ󥹥åɤƱŪ˳ߤ򤫤ȻפäΤǡ
 *       ʲAPIѰդۤɰΤΤǤʤġġ
 *       pthreadUNIXʥ뤬ꤷƻȤФȤ⤢Τ
 *       pthreadUNIXʥȹ碌ϵʤΤ򤱤롣
 *
 * active user: AsyncBufferproducer⤷consumeråɡ
 *              ХåեݻƤΥƥȤϴޤޤʤ
 *              active userϥХåե˿ˡasync_buffer_active_ref()
 *              ƤӽФ⤦Хåե˿ʤȤ
 *              async_buffer_active_unref()ƤӽФȤˤ롣
 *              δؿg_object_{ref,unref}Ƥǡ
 *              number_of_active_usersĴ롣
 *              active userϻsuspend/resume/terminate׵᤬Ƥ
 *              ʤɤǧ뤳Ȥ롣
 */
gboolean
async_buffer_active_ref(AsyncBuffer *buffer, const char *user)
{
  gboolean result;

#if DEBUG_ASYNC_BUFFER_MOST
  fprintf(stderr, "AsyncBuffer(0x%x) is actively refered by %s\n",
	  (int)buffer, user);
#endif
  g_object_ref(G_OBJECT(buffer));

  LOCK_BUFFER_LOCK(buffer);

  buffer->number_of_active_users++;
  while (buffer->state == ASYNC_BUFFER_SUSPENDED)
    {
      buffer->number_of_suspended_users++;

      LOCK_GLOBAL_LOCK;
      SIGNAL_GLOBAL_CONDITION;
      UNLOCK_GLOBAL_LOCK;

      WAIT_BUFFER_CONDITION(buffer);
    }

  result = buffer->state != ASYNC_BUFFER_TERMINATED;
  if (!result)
    {
      buffer->number_of_active_users--;
      g_object_unref(G_OBJECT(buffer));
    }

  UNLOCK_BUFFER_LOCK(buffer);

  return result;
}


void
async_buffer_active_unref(AsyncBuffer *buffer, const char *user)
{
#if DEBUG_ASYNC_BUFFER_MOST
  fprintf(stderr, "AsyncBuffer(0x%x) actively refered by %s is unrefered\n",
	  (int)buffer, user);
#endif

  LOCK_BUFFER_LOCK(buffer);

  buffer->number_of_active_users--;

  if (buffer->state != ASYNC_BUFFER_OK)
    {
      LOCK_GLOBAL_LOCK;
      SIGNAL_GLOBAL_CONDITION;
      UNLOCK_GLOBAL_LOCK;
    }

  UNLOCK_BUFFER_LOCK(buffer);

  g_object_unref(G_OBJECT(buffer));
}


void
async_buffer_suspend_all(void)
{
  g_slist_foreach(async_buffer_list, (GFunc)async_buffer_suspend, NULL);
}


void
async_buffer_resume_all(void)
{
  g_slist_foreach(async_buffer_list, (GFunc)async_buffer_resume, NULL);
}


void
async_buffer_suspend(AsyncBuffer *buffer)
{
#if DEBUG_ASYNC_BUFFER_MOST
  fprintf(stderr, "suspend threads refering AsyncBuffer(0x%x)\n", (int)buffer);
#endif

  LOCK_BUFFER_LOCK(buffer);
  buffer->number_of_suspended_users = 0;
  buffer->state = ASYNC_BUFFER_SUSPENDED;

  LOCK_GLOBAL_LOCK;

  BROADCAST_BUFFER_CONDITION(buffer);
  UNLOCK_BUFFER_LOCK(buffer);		/* νǤʤȤʤʤ
					 * UNLOCK_BUFFER_LOCKִ֤
					 * ΥåԤäƤåɤ
					 * ưϤ롣åɤξ
					 * åSUSPENDED֤Ǥ
					 * ȤΤΤǡSUSPENDED֤
					 * ֲ򤱤ΤԤġwait롣
					 * BROADCASTϿSUSPENDED֤
					 * ʤäȤΤʤåɤ
					 * 뤿ΤΤǤ
					 * ա
					 */
  while (buffer->number_of_suspended_users < buffer->number_of_active_users)
    {
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "async_buffer_suspend: suspended=%d, active_users=%d\n",
	      buffer->number_of_suspended_users,
	      buffer->number_of_active_users);
#endif
      WAIT_GLOBAL_CONDITION;
    }
#if DEBUG_ASYNC_BUFFER_MOST
  fprintf(stderr, "async_buffer_suspend: suspended=%d, active_users=%d\n",
	  buffer->number_of_suspended_users,
	  buffer->number_of_active_users);
#endif

  UNLOCK_GLOBAL_LOCK;
}


void
async_buffer_resume(AsyncBuffer *buffer)
{
  LOCK_BUFFER_LOCK(buffer);
  buffer->state = ASYNC_BUFFER_OK;
  BROADCAST_BUFFER_CONDITION(buffer);
  UNLOCK_BUFFER_LOCK(buffer);
}


void
async_buffer_terminate(AsyncBuffer *buffer)
{
#if DEBUG_ASYNC_BUFFER_MOST
  fprintf(stderr, "async_buffer_terminate: buffer=0x%x\n", (int)buffer);
#endif

  LOCK_BUFFER_LOCK(buffer);
  buffer->state = ASYNC_BUFFER_TERMINATED;
  BROADCAST_BUFFER_CONDITION(buffer);

#if DEBUG_ASYNC_BUFFER_MOST
  fprintf(stderr, "async_buffer_terminate: buffer condition broadcast done\n");
#endif

  LOCK_GLOBAL_LOCK;

#if DEBUG_ASYNC_BUFFER_MOST
  fprintf(stderr, "async_buffer_terminate: global lock locked\n");
#endif

  UNLOCK_BUFFER_LOCK(buffer);

#if DEBUG_ASYNC_BUFFER_MOST
  fprintf(stderr, "async_buffer_terminate: buffer lock unlocked\n");
#endif

  while (buffer->number_of_active_users > 0)
    {
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "async_buffer_terminate: active_users=%d\n",
	      buffer->number_of_active_users);
#endif
      WAIT_GLOBAL_CONDITION;
    }
#if DEBUG_ASYNC_BUFFER_MOST
  fprintf(stderr, "async_buffer_terminate: active_users=%d\n",
	  buffer->number_of_active_users);
#endif

  UNLOCK_GLOBAL_LOCK;
}


/*
 * ХåեTERMINATED֤ξFALSE֤
 */
gboolean
async_buffer_update_length(AsyncBuffer *buffer, int length, const char *user)
{
#if DEBUG_ASYNC_BUFFER_MOST
  fprintf(stderr, "\"%s\" called async_buffer_update_length\n", user);
#endif
  LOCK_BUFFER_LOCK(buffer);
  buffer->length = length;
  UNLOCK_BUFFER_LOCK(buffer);

  return async_buffer_broadcast(buffer, user);
}


/*
 * ХåեTERMINATED֤ξFALSE֤
 */
gboolean
async_buffer_fix(AsyncBuffer *buffer, const char *user)
{
#if DEBUG_ASYNC_BUFFER_MOST
  fprintf(stderr, "\"%s\" called async_buffer_fix\n", user);
#endif

  LOCK_BUFFER_LOCK(buffer);
  buffer->fixed = TRUE;
  UNLOCK_BUFFER_LOCK(buffer);

  return async_buffer_broadcast(buffer, user);
}


/*
 * Out of memory⤷ϥХåեTERMINATED֤ξFALSE֤
 */
gboolean
async_buffer_resize(AsyncBuffer *buffer, int length, const char *user)
{
  char *new_buf;
  gboolean result;

#if DEBUG_ASYNC_BUFFER_MOST
  fprintf(stderr, "\"%s\" called async_buffer_resize\n", user);
#endif

  LOCK_BUFFER_LOCK(buffer);

  if (buffer->fixed)
    {
      fprintf(stderr, "Invalid use of async_buffer: Fixed buffer isn't resizable.\n");
      abort();
    }

  new_buf = (char *)realloc((void *)buffer->buffer, length);
  if (new_buf == NULL)
    result = FALSE;
  else
    {
      buffer->buffer = new_buf;
      buffer->buffer_length = length;
      result = TRUE;
    }

  UNLOCK_BUFFER_LOCK(buffer);

  return async_buffer_broadcast(buffer, user) && result;
}


/*
 * Out of memory⤷ϥХåեTERMINATED֤ξFALSE֤
 */
gboolean
async_buffer_ensure_free_space(AsyncBuffer *buffer, int length,
			       const char *user)
{
  char *new_buf;
  gboolean result = TRUE;

#if DEBUG_ASYNC_BUFFER_MOST
  fprintf(stderr, "async_buffer_ensure_free_space(0x%x, %d, \"%s\") called.\n",
	  (int)buffer, length, user);
#endif

  LOCK_BUFFER_LOCK(buffer);
  if (buffer->fixed)
    {
      fprintf(stderr, "Invalid use of async_buffer: Fixed buffer couldn't be enlarged.\n");
      abort();
    }

  while (buffer->state == ASYNC_BUFFER_SUSPENDED)
    {
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "\"%s\" is suspended at async_buffer_free_space.\n",
	      user);
#endif
      buffer->number_of_suspended_users++;

      LOCK_GLOBAL_LOCK;
      SIGNAL_GLOBAL_CONDITION;
      UNLOCK_GLOBAL_LOCK;

      WAIT_BUFFER_CONDITION(buffer);
      /* λbuffer->state != ASYNC_BUFFER_SUSPENDEDʤ
       * ʥsuspend򤯤˥ᥤ󥹥åɤäƤΡ
       *
       * buffer->state == ASYNC_BUFFER_SUSPENDEDʤ顢waitƤ֤
       * ⤦ٿSUSPENDED֤ˤʤäȤȡ
       */
    }

  while (buffer->buffer_length - buffer->length < length)
    {
      size_t new_buf_len
	= (buffer->buffer_length > 0)
	? (buffer->buffer_length * 2) : INITIAL_ASYNC_BUFFER_SIZE;
      new_buf = (char *)realloc((void *)buffer->buffer, new_buf_len);
      if (new_buf != NULL)
	{
	  buffer->buffer = new_buf;
	  buffer->buffer_length = new_buf_len;
	}
      else
	{
	  result = FALSE;
	  break;
	}
    }
  result = (buffer->state != ASYNC_BUFFER_TERMINATED) && result;
  UNLOCK_BUFFER_LOCK(buffer);

  return result;
}


gboolean
async_buffer_append_data(AsyncBuffer *buffer, const char *data, int length,
			 const char *user)
{
  gboolean result = TRUE;

#if DEBUG_ASYNC_BUFFER_MOST
  fprintf(stderr, "\"%s\" called async_buffer_append_data\n", user);
#endif

  if (async_buffer_ensure_free_space(buffer, length, user))
    {
      memcpy((char *)buffer->buffer + buffer->length, data, length);
      buffer->length += length;
    }
  else
    result = FALSE;

  return async_buffer_broadcast(buffer, user) && result;
}


void
async_buffer_lock(AsyncBuffer *buffer)
{
  LOCK_BUFFER_LOCK(buffer);
}


void
async_buffer_unlock(AsyncBuffer *buffer)
{
  UNLOCK_BUFFER_LOCK(buffer);
}


gboolean
async_buffer_check_active(AsyncBuffer *buffer, const char *user)
{
  gboolean result;

#if DEBUG_ASYNC_BUFFER_MOST
  fprintf(stderr, "\"%s\" calls async_buffer_check_active for AsyncBuffer(0x%x)\n",
	  user, (int)buffer);
#endif

  LOCK_BUFFER_LOCK(buffer);

  while (buffer->state == ASYNC_BUFFER_SUSPENDED)
    {
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "\"%s\" is suspended at async_buffer_check_signal.\n",
	      user);
#endif
      buffer->number_of_suspended_users++;

      LOCK_GLOBAL_LOCK;
      SIGNAL_GLOBAL_CONDITION;
      UNLOCK_GLOBAL_LOCK;

      WAIT_BUFFER_CONDITION(buffer);	/* ԤäƤ륷ʥ
					 * ᥤ󥹥åɤΤΡ
					 */
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "\"%s\" resumed.\n", user);
#endif
    }
  /* ХåեmutexunlockǤstateѤäƤǽΤ
   * λξ֤ǺοȤο롣Ĥޤꡢλ
   * AYNC_BUFFER_TERMINATEDǤʤä顢ĴޤǤϾư
   * ޤλǤASYNC_BUFFER_OKʤΤǡbuffercondvarԤäƤ
   * ΤconsumerΤߤǤ뤳ȤݾڤƤ롣
   */
  result = buffer->state != ASYNC_BUFFER_TERMINATED;

  UNLOCK_BUFFER_LOCK(buffer);

  return result;
}


gboolean
async_buffer_signal(AsyncBuffer *buffer, const char *user)
{
  gboolean result;

#if DEBUG_ASYNC_BUFFER_MOST
  fprintf(stderr, "\"%s\" calls async_buffer_signal for AsyncBuffer(0x%x)\n",
	  user, (int)buffer);
#endif

  LOCK_BUFFER_LOCK(buffer);

  while (buffer->state == ASYNC_BUFFER_SUSPENDED)
    {
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "\"%s\" is suspended at async_buffer_signal.\n", user);
#endif
      buffer->number_of_suspended_users++;

      LOCK_GLOBAL_LOCK;
      SIGNAL_GLOBAL_CONDITION;
      UNLOCK_GLOBAL_LOCK;

      WAIT_BUFFER_CONDITION(buffer);	/* ԤäƤ륷ʥ
					 * ᥤ󥹥åɤΤΡ
					 */
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "\"%s\" resumed.\n", user);
#endif
    }
  /* ХåեmutexunlockǤstateѤäƤǽΤ
   * λξ֤ǺοȤο롣Ĥޤꡢλ
   * AYNC_BUFFER_TERMINATEDǤʤä顢ĴޤǤϾư
   * ޤλǤASYNC_BUFFER_OKʤΤǡbuffercondvarԤäƤ
   * ΤconsumerΤߤǤ뤳ȤݾڤƤ롣
   */
  result = buffer->state != ASYNC_BUFFER_TERMINATED;
  if (result)
    SIGNAL_BUFFER_CONDITION(buffer);	/* λǡbufferwaitƤ
					 * åɤԤäƤ륷ʥ
					 * ᥤ󥹥åɤΤΤǤϤʤ
					 */
  UNLOCK_BUFFER_LOCK(buffer);

  return result;
}


gboolean
async_buffer_broadcast(AsyncBuffer *buffer, const char *user)
{
  gboolean result;

#if DEBUG_ASYNC_BUFFER_MOST
  fprintf(stderr, "\"%s\" calls async_buffer_broadcast for AsyncBuffer(0x%x)\n",
	  user, (int)buffer);
#endif

  LOCK_BUFFER_LOCK(buffer);

  while (buffer->state == ASYNC_BUFFER_SUSPENDED)
    {
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "\"%s\" is suspended at async_buffer_broadcast.\n",
	      user);
#endif
      buffer->number_of_suspended_users++;

      LOCK_GLOBAL_LOCK;
      SIGNAL_GLOBAL_CONDITION;
      UNLOCK_GLOBAL_LOCK;

      WAIT_BUFFER_CONDITION(buffer);	/* ԤäƤ륷ʥ
					 * ᥤ󥹥åɤΤΡ
					 */
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "\"%s\" resumed.\n", user);
#endif
    }
  /* ХåեmutexunlockǤstateѤäƤǽΤ
   * λξ֤ǺοȤο롣Ĥޤꡢλ
   * AYNC_BUFFER_TERMINATEDǤʤä顢ĴޤǤϾư
   */
  result = buffer->state != ASYNC_BUFFER_TERMINATED;
  if (result)
      BROADCAST_BUFFER_CONDITION(buffer);/* λǡbufferwaitƤ
					  * åɤԤäƤ륷ʥ
					  * ᥤ󥹥åɤΤΤǤϤʤ
					  */
  UNLOCK_BUFFER_LOCK(buffer);

  return result;
}


gboolean
async_buffer_wait(AsyncBuffer *buffer, const char *user)
{
#if DEBUG_ASYNC_BUFFER_MOST
  fprintf(stderr, "\"%s\" called async_buffer_wait\n", user);
#endif

  while (buffer->state == ASYNC_BUFFER_SUSPENDED)
    {
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "\"%s\" is suspended at async_buffer_wait.\n", user);
#endif
      buffer->number_of_suspended_users++;

      LOCK_GLOBAL_LOCK;
      SIGNAL_GLOBAL_CONDITION;
      UNLOCK_GLOBAL_LOCK;

      WAIT_BUFFER_CONDITION(buffer);
      /* λbuffer->state != ASYNC_BUFFER_SUSPENDEDʤ
       * ʥsuspend򤯤˥ᥤ󥹥åɤäƤΡ
       */
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "\"%s\" resumed.\n", user);
#endif
    }

  if (buffer->state != ASYNC_BUFFER_TERMINATED)
    WAIT_BUFFER_CONDITION(buffer);

  /* λASYNC_BUFFER_SUSPENDEDʾ⤢뤬ξsuspend
   * Ȥˤ롣
   */

  return buffer->state != ASYNC_BUFFER_TERMINATED;
}


void
free_when_finished(AsyncBuffer *buffer, gpointer unused)
{
  if (buffer->buffer != NULL)
    free((void *)buffer->buffer);
}
