/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id$
 */

#include "ochusha.h"
#include "ochusha_ui.h"

#include "bbs_thread_ui.h"
#include "bulletin_board_ui.h"

#include "marshal.h"

#include <glib.h>
#include <gtk/gtk.h>

#include <stdlib.h>
#include <string.h>

static void threadlist_filter_setting_class_init(ThreadlistFilterSettingClass *klass);

static void append_rule(ThreadlistFilterSetting *setting, gint rule);
static void remove_rule(ThreadlistFilterSetting *setting, gint rule);
static void append_rule_helper(GtkTreeModel *model, GtkTreePath *path,
			       GtkTreeIter *iter,
			       ThreadlistFilterSetting *setting);
static void append_rule_button_cb(GtkWidget *widget,
				  ThreadlistFilterSetting *setting);
static void remove_rule_helper(GtkTreeRowReference *ref,
			       ThreadlistFilterSetting *setting);
static void remove_rule_button_cb(GtkWidget *widget,
				  ThreadlistFilterSetting *setting);
static void goto_top_button_cb(GtkWidget *widget,
			       ThreadlistFilterSetting *setting);
static void go_up_button_cb(GtkWidget *widget,
			    ThreadlistFilterSetting *setting);
static void go_down_button_cb(GtkWidget *widget,
			    ThreadlistFilterSetting *setting);
static void goto_bottom_button_cb(GtkWidget *widget,
				  ThreadlistFilterSetting *setting);
static void current_rules_row_changed_cb(GtkTreeModel *model,
					 GtkTreePath *unused,
					 GtkTreeIter *iter,
					 ThreadlistFilterSetting *setting);
static void update_current_rule_string(ThreadlistFilterSetting *setting);
static gboolean all_rules_selection_filter(GtkTreeSelection *selection,
					   GtkTreeModel *model,
					   GtkTreePath *path,
					   gboolean currently_selected,
					   ThreadlistFilterSetting *setting);
static void setup_models(ThreadlistFilterSetting *setting);
static RuleModelRelation *get_relation_for_rule(ThreadlistFilterSetting *setting, gint rule);
static void update_models(ThreadlistFilterSetting *setting);
static void radio_button_toggled_cb(GtkToggleButton *unused,
				    ThreadlistFilterSetting *setting);
static void spin_button_value_changed_cb(GtkSpinButton *spin_button,
					 ThreadlistFilterSetting *setting);
static void threadlist_filter_setting_restore_default(ThreadlistFilterSetting *setting);
static void threadlist_filter_setting_changed(ThreadlistFilterSetting *setting,
					      ThreadlistFilter *unused);

static void threadlist_filter_setting_init(ThreadlistFilterSetting *setting);

static void threadlist_filter_setting_finalize(GObject *object);
static void threadlist_filter_setting_destroy(GtkObject *object);

static const gchar *get_threadlist_filter_rule_string(int rule);
static void text_set_cell_data_available_rule(GtkTreeViewColumn *column,
					      GtkCellRenderer *renderer,
					      GtkTreeModel *model,
					      GtkTreeIter *iter,
					      ThreadlistFilterSetting *data);
static void text_set_cell_data_applied_rule(GtkTreeViewColumn *column,
					    GtkCellRenderer *renderer,
					    GtkTreeModel *model,
					    GtkTreeIter *iter,
					    ThreadlistFilterSetting *data);

static const gchar *threadlist_filter_rule_ochusha_default = "FrhSNsa";
static gchar *threadlist_filter_rule_default = NULL;
static int threadlist_filter_rule_default_ignore_threshold;


static const gchar *show_all;
static const gchar *hide_all;
static const gchar *show_favorite;
static const gchar *hide_non_favorite;
static const gchar *show_seen;
static const gchar *hide_unseen;
static const gchar *show_seen_offline;
static const gchar *hide_unseen_offline;
static const gchar *show_new_comer;
static const gchar *show_new_res;
static const gchar *hide_without_new_res;
static const gchar *show_dat_dropped;
static const gchar *hide_ignored;
static const gchar *hide_hidden;
static const gchar *invalid_rule;


GType
threadlist_filter_setting_get_type(void)
{
  static GType tlfs_type = 0;

  if (tlfs_type == 0)
    {
      static const GTypeInfo tlfs_info =
	{
	  sizeof(ThreadlistFilterSettingClass),
	  NULL,	/* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc)threadlist_filter_setting_class_init,
	  NULL,	/* class_finalize */
	  NULL, /* class_data */
	  sizeof(ThreadlistFilterSetting),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)threadlist_filter_setting_init,
	};

      tlfs_type = g_type_register_static(GTK_TYPE_VBOX,
					 "ThreadlistFilterSetting",
					 &tlfs_info, 0);
    }

  return tlfs_type;
}


enum {
  RESTORE_DEFAULT_SIGNAL,
  CHANGED_SIGNAL,
  LAST_SIGNAL
};


static GtkVBoxClass *parent_class = NULL;
static gint threadlist_filter_setting_signals[LAST_SIGNAL] = { 0, 0 };


static void
threadlist_filter_setting_class_init(ThreadlistFilterSettingClass *klass)
{
  GObjectClass *o_class = (GObjectClass *)klass;
  GtkObjectClass *object_class = (GtkObjectClass *)klass;

  parent_class = g_type_class_peek_parent(klass);

  /* GObject signals */
  o_class->finalize = threadlist_filter_setting_finalize;

  /* GtkObject signals */
  object_class->destroy = threadlist_filter_setting_destroy;

  threadlist_filter_setting_signals[RESTORE_DEFAULT_SIGNAL] =
    g_signal_new("restore_default",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistFilterSettingClass,
				 restore_default),
		 NULL, NULL,
		 ochusha_marshal_VOID__VOID,
		 G_TYPE_NONE, 0);
  threadlist_filter_setting_signals[CHANGED_SIGNAL] =
    g_signal_new("changed",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistFilterSettingClass, changed),
		 NULL, NULL,
		 ochusha_marshal_VOID__POINTER,
		 G_TYPE_NONE, 1,
		 G_TYPE_POINTER);

  klass->restore_default = threadlist_filter_setting_restore_default;
  klass->changed = threadlist_filter_setting_changed;

  show_all = _("Show all threads");
  hide_all = _("Hide all threads");
  show_favorite = _("Show favorite threads");
  hide_non_favorite = _("Hide non favorite threads");
  show_seen = _("Show threads ever seen");
  hide_unseen = _("Hide threads never seen");
  show_seen_offline = _("Show threads ever seen during offline");
  hide_unseen_offline	= _("Hide threads never seen during offline");
  show_new_comer = _("Show new comer threads");
  show_new_res = _("Show threads having new responses");
  hide_without_new_res = _("Hide threads having no unread responses");
  show_dat_dropped = _("Show threads whose DAT was dropped");
  hide_ignored = _("Hide threads too much times ignored");
  hide_hidden = _("Hide hidden threads");
  invalid_rule = _("Invalid rule");
}


enum {
  FILTER_RULE = 0,
  FILTER_RULE_PRIORITY,
  FILTER_RULE_N_COLUMNS
};


static void
append_rule(ThreadlistFilterSetting *setting, gint rule)
{
  GtkListStore *store;
  RuleModelRelation *relation = get_relation_for_rule(setting, rule);

  if (relation == NULL)
    {
#if DEBUG_WIDGET
      fprintf(stderr, "Invalid rule('%c'=%d) specified.\n", rule, rule);
#endif
      return;
    }

  if (relation->priority != -1)
    {
#if DEBUG_WIDGET
      fprintf(stderr, "Inconsistent state: rule('%c'=%d) might have been included at priority=%d\n", rule, rule, relation->priority);
#endif
      return;
    }

  store = GTK_LIST_STORE(setting->current_rules_model);
  relation->priority = setting->number_of_rules_applied++;
  gtk_list_store_append(store, &relation->current_rules_iter);
  gtk_list_store_set(store, &relation->current_rules_iter,
		     FILTER_RULE, rule,
		     FILTER_RULE_PRIORITY, relation->priority,
		     -1);
}


static void
remove_rule(ThreadlistFilterSetting *setting, gint rule)
{
  GtkListStore *store;
  RuleModelRelation *relation = get_relation_for_rule(setting, rule);

  if (relation == NULL)
    {
#if DEBUG_WIDGET
      fprintf(stderr, "Invalid rule('%c'=%d) specified.\n", rule, rule);
#endif
      return;
    }

  if (relation->priority == -1)
    {
#if DEBUG_WIDGET
      fprintf(stderr, "Inconsistent state: rule('%c'=%d) might not have been included at priority=%d\n", rule, rule, relation->priority);
#endif
      return;
    }

  store = GTK_LIST_STORE(setting->current_rules_model);
  relation->priority = setting->number_of_rules_applied--;

  /* äˡġ*/
  gtk_list_store_set(store, &relation->current_rules_iter,
		     FILTER_RULE_PRIORITY, -1,
		     -1);
  gtk_list_store_remove(store, &relation->current_rules_iter);
}


static void
append_rule_helper(GtkTreeModel *model, GtkTreePath *path,
		   GtkTreeIter *iter, ThreadlistFilterSetting *setting)
{
  gint rule;
  gint priority;

  gtk_tree_model_get(model, iter,
		     FILTER_RULE, &rule,
		     FILTER_RULE_PRIORITY, &priority, -1);

  if (priority == -1)
    append_rule(setting, rule);
#if DEBUG_WIDGET
  else
    fprintf(stderr,
	    "rule('%c') might have already been included (priority = %d).\n",
	    rule, priority);
#endif
}


static void
append_rule_button_cb(GtkWidget *widget, ThreadlistFilterSetting *setting)
{
  gint last_num_rules = setting->number_of_rules_applied;

  gtk_tree_selection_selected_foreach(setting->all_rules_selection,
			(GtkTreeSelectionForeachFunc)append_rule_helper,
			setting);
  setting->unselecting = TRUE;
  gtk_tree_selection_unselect_all(setting->all_rules_selection);
  setting->unselecting = FALSE;

  update_current_rule_string(setting);

  if (setting->number_of_rules_applied > last_num_rules)
    g_signal_emit(G_OBJECT(setting),
		  threadlist_filter_setting_signals[CHANGED_SIGNAL],
		  0,
		  &setting->filter);
}


static void
remove_rule_helper(GtkTreeRowReference *ref, ThreadlistFilterSetting *setting)
{
  gint rule;
  gint priority;
  GtkTreeIter iter;
  GtkTreePath *path;

  if (ref == NULL)
    return;

  path = gtk_tree_row_reference_get_path(ref);

  if (path == NULL)
    return;

  if (!gtk_tree_model_get_iter(setting->current_rules_model, &iter, path))
    return;

  gtk_tree_model_get(setting->current_rules_model, &iter,
		     FILTER_RULE, &rule,
		     FILTER_RULE_PRIORITY, &priority, -1);

  if (priority != -1)
    {
      GtkTreeModel *model = setting->current_rules_model;
      GtkListStore *store = GTK_LIST_STORE(model);
      while (gtk_tree_model_iter_next(model, &iter))
	{
#if DEBUG_WIDGET_MOST
	  fprintf(stderr, "updating priority: %d\n", priority);
#endif
	  gtk_list_store_set(store, &iter,
			     FILTER_RULE_PRIORITY, priority++, -1);
	}
      remove_rule(setting, rule);
    }
#if DEBUG_WIDGET
  else
    fprintf(stderr,
	    "rule('%c') might not have been included (priority = %d).\n",
	    rule, priority);
#endif

  gtk_tree_row_reference_free(ref);
}


/*
 * ˽ʤΤǼ谷ա
 */
static void
convert_path_list_to_ref_list(GtkTreeModel *model, GList *list)
{
  GList *current_list_item = list;
  while (current_list_item != NULL)
    {
      GtkTreePath *path;

      path = (GtkTreePath *)current_list_item->data;
      current_list_item->data = gtk_tree_row_reference_new(model, path);
      gtk_tree_path_free(path);

      current_list_item = current_list_item->next;
    }
}


static void
remove_rule_button_cb(GtkWidget *widget, ThreadlistFilterSetting *setting)
{
  GList *selected_rules;

  selected_rules
    = gtk_tree_selection_get_selected_rows(setting->current_rules_selection,
					   NULL);
  if (selected_rules == NULL)
    return;

  convert_path_list_to_ref_list(setting->current_rules_model, selected_rules);

  g_list_foreach(selected_rules, (GFunc)remove_rule_helper, setting);
  g_list_free(selected_rules);

  gtk_tree_selection_unselect_all(setting->current_rules_selection);
  update_current_rule_string(setting);

  g_signal_emit(G_OBJECT(setting),
		threadlist_filter_setting_signals[CHANGED_SIGNAL],
		0,
		&setting->filter);
}


/* XXX: ־ΤĤäΤľ٤*/
static void
goto_top_button_cb(GtkWidget *widget, ThreadlistFilterSetting *setting)
{
  gchar tmp_rule_string[MAXIMUM_LENGTH_OF_RULE_STRING];
  GList *selected_rules;
  GList *current_rule;
  GtkTreeModel *model;
  gint new_priority = 0;
  gchar *tmp_pos;
  gboolean changed = FALSE;

  selected_rules
    = gtk_tree_selection_get_selected_rows(setting->current_rules_selection,
					   NULL);
  if (selected_rules == NULL)
    return;

  model = setting->current_rules_model;

  strncpy(tmp_rule_string, setting->current_rule_string,
	  MAXIMUM_LENGTH_OF_RULE_STRING);

  current_rule = selected_rules;
  while (current_rule != NULL)
    {
      gint rule;
      GtkTreeIter tmp_iter;
      RuleModelRelation *relation;
      GtkTreePath *path = (GtkTreePath *)current_rule->data;

      if (path == NULL)
	continue;

      if (!gtk_tree_model_get_iter(model, &tmp_iter, path))
	continue;

      gtk_tree_model_get(model, &tmp_iter,
			 FILTER_RULE, &rule,
			 -1);
      relation = get_relation_for_rule(setting, rule);
      if (relation == NULL)
	continue;

      if (setting->current_rule_string[new_priority] != rule)
	{
	  setting->current_rule_string[new_priority] = rule;
	  changed = TRUE;
	}

      new_priority++;
#if DEBUG_WIDGET
      if (tmp_rule_string[relation->priority] != rule)
	{
	  fprintf(stderr, "Inconsistent state: '%c' should be '%c'\n",
		  tmp_rule_string[relation->priority], rule);
	}
#endif
      tmp_rule_string[relation->priority] = THREADLIST_FILTER_RULE_NOP;

      current_rule = current_rule->next;
    }

  g_list_foreach(selected_rules, (GFunc)gtk_tree_path_free, NULL);
  g_list_free(selected_rules);

  tmp_pos = tmp_rule_string;
  while (*tmp_pos != THREADLIST_FILTER_RULE_END)
    {
      if (*tmp_pos != THREADLIST_FILTER_RULE_NOP)
	setting->current_rule_string[new_priority++] = *tmp_pos;
      tmp_pos++;
    }

  update_models(setting);
  if (changed)
    g_signal_emit(G_OBJECT(setting),
		  threadlist_filter_setting_signals[CHANGED_SIGNAL],
		  0,
		  &setting->filter);
}


static void
go_up_button_cb(GtkWidget *widget, ThreadlistFilterSetting *setting)
{
  gchar tmp_rule_string[MAXIMUM_LENGTH_OF_RULE_STRING];
  gchar new_rule_string[MAXIMUM_LENGTH_OF_RULE_STRING];
  GList *selected_rules;
  GList *current_rule;
  GtkTreeModel *model;
  gint new_priority = 0;
  int i;
  gchar *tmp_pos;

  selected_rules
    = gtk_tree_selection_get_selected_rows(setting->current_rules_selection,
					   NULL);
  if (selected_rules == NULL)
    return;

  model = setting->current_rules_model;
  strncpy(tmp_rule_string, setting->current_rule_string,
	  MAXIMUM_LENGTH_OF_RULE_STRING);
  memset(new_rule_string, 0, MAXIMUM_LENGTH_OF_RULE_STRING);

  current_rule = selected_rules;
  while (current_rule != NULL)
    {
      gint rule;
      GtkTreeIter tmp_iter;
      RuleModelRelation *relation;
      GtkTreePath *path = (GtkTreePath *)current_rule->data;

      if (path == NULL)
	continue;

      if (!gtk_tree_model_get_iter(model, &tmp_iter, path))
	{
	  gtk_tree_path_free(path);
	  current_rule->data = NULL;
	  continue;
	}

      gtk_tree_model_get(model, &tmp_iter,
			 FILTER_RULE, &rule,
			 -1);
      relation = get_relation_for_rule(setting, rule);
      if (relation == NULL)
	{
	  gtk_tree_path_free(path);
	  current_rule->data = NULL;
	  continue;
	}

      tmp_rule_string[relation->priority] = THREADLIST_FILTER_RULE_NOP;

      new_priority = relation->priority - 1;
      if (new_priority < 0)
	new_priority = relation->priority;

      while (new_rule_string[new_priority] != '\0')
	new_priority++;
      new_rule_string[new_priority++] = rule;

      gtk_tree_path_free(path);
      current_rule->data = relation;
      current_rule = current_rule->next;
    }

  tmp_pos = tmp_rule_string;
  for (i = 0; i < setting->number_of_rules_applied; i++)
    {
      if (new_rule_string[i] == '\0')
	{
	  while (*tmp_pos == THREADLIST_FILTER_RULE_NOP)
	    tmp_pos++;
	  new_rule_string[i] = *tmp_pos;
	  tmp_pos++;
	}
    }
  new_rule_string[setting->number_of_rules_applied]
    = tmp_rule_string[setting->number_of_rules_applied];
  new_rule_string[setting->number_of_rules_applied + 1] = '\0';

  if (strcmp(setting->current_rule_string, new_rule_string) == 0)
    {
      g_list_free(selected_rules);
      return;
    }

  strncpy(setting->current_rule_string, new_rule_string,
	  MAXIMUM_LENGTH_OF_RULE_STRING);

  update_models(setting);

  current_rule = selected_rules;
  while (current_rule != NULL)
    {
      RuleModelRelation *relation = (RuleModelRelation *)current_rule->data;
      if (relation == NULL)
	continue;

      gtk_tree_selection_select_iter(setting->current_rules_selection,
				     &relation->current_rules_iter);

      current_rule = current_rule->next;
    }

  g_list_free(selected_rules);

  g_signal_emit(G_OBJECT(setting),
		threadlist_filter_setting_signals[CHANGED_SIGNAL],
		0,
		&setting->filter);
}


static void
go_down_button_cb(GtkWidget *widget, ThreadlistFilterSetting *setting)
{
  gchar tmp_rule_string[MAXIMUM_LENGTH_OF_RULE_STRING];
  gchar new_rule_string[MAXIMUM_LENGTH_OF_RULE_STRING];
  GList *selected_rules;
  GList *current_rule;
  GtkTreeModel *model;
  gint new_priority = 0;
  int i;
  gchar *tmp_pos;

  selected_rules
    = gtk_tree_selection_get_selected_rows(setting->current_rules_selection,
					   NULL);
  if (selected_rules == NULL)
    return;

  model = setting->current_rules_model;
  strncpy(tmp_rule_string, setting->current_rule_string,
	  MAXIMUM_LENGTH_OF_RULE_STRING);
  memset(new_rule_string, 0, MAXIMUM_LENGTH_OF_RULE_STRING);

  current_rule = selected_rules;
  while (current_rule != NULL)
    {
      gint rule;
      GtkTreeIter tmp_iter;
      RuleModelRelation *relation;
      GtkTreePath *path = (GtkTreePath *)current_rule->data;
      gchar tmp_rule;

      if (path == NULL)
	continue;

      if (!gtk_tree_model_get_iter(model, &tmp_iter, path))
	{
	  gtk_tree_path_free(path);
	  current_rule->data = NULL;
	  continue;
	}

      gtk_tree_model_get(model, &tmp_iter,
			 FILTER_RULE, &rule,
			 -1);
      relation = get_relation_for_rule(setting, rule);
      if (relation == NULL)
	{
	  gtk_tree_path_free(path);
	  current_rule->data = NULL;
	  continue;
	}

      tmp_rule_string[relation->priority] = THREADLIST_FILTER_RULE_NOP;

      new_priority = relation->priority + 1;
      if (new_priority >= setting->number_of_rules_applied)
	new_priority = relation->priority;

      do
	{
	  tmp_rule = new_rule_string[new_priority];
	  new_rule_string[new_priority--] = rule;
	  rule = tmp_rule;
	}
      while (tmp_rule != '\0' && new_priority >= 0);

      gtk_tree_path_free(path);
      current_rule->data = relation;
      current_rule = current_rule->next;
    }

  tmp_pos = tmp_rule_string;
  for (i = 0; i < setting->number_of_rules_applied; i++)
    {
      if (new_rule_string[i] == '\0')
	{
	  while (*tmp_pos == THREADLIST_FILTER_RULE_NOP)
	    tmp_pos++;
	  new_rule_string[i] = *tmp_pos;
	  tmp_pos++;
	}
    }
  new_rule_string[setting->number_of_rules_applied]
    = tmp_rule_string[setting->number_of_rules_applied];
  new_rule_string[setting->number_of_rules_applied + 1] = '\0';

  if (strcmp(setting->current_rule_string, new_rule_string) == 0)
    {
      g_list_free(selected_rules);
      return;
    }

  strncpy(setting->current_rule_string, new_rule_string,
	  MAXIMUM_LENGTH_OF_RULE_STRING);

  update_models(setting);

  current_rule = selected_rules;
  while (current_rule != NULL)
    {
      RuleModelRelation *relation = (RuleModelRelation *)current_rule->data;
      if (relation == NULL)
	continue;

      gtk_tree_selection_select_iter(setting->current_rules_selection,
				     &relation->current_rules_iter);

      current_rule = current_rule->next;
    }

  g_list_free(selected_rules);

  g_signal_emit(G_OBJECT(setting),
		threadlist_filter_setting_signals[CHANGED_SIGNAL],
		0,
		&setting->filter);
}


static void
goto_bottom_button_cb(GtkWidget *widget, ThreadlistFilterSetting *setting)
{
  gchar tmp_rule_string[MAXIMUM_LENGTH_OF_RULE_STRING];
  gchar last_rule_string[MAXIMUM_LENGTH_OF_RULE_STRING];
  GList *selected_rules;
  GList *current_rule;
  GtkTreeModel *model;
  gint new_priority = 0;
  gint i;
  gchar *tmp_pos;

  selected_rules
    = gtk_tree_selection_get_selected_rows(setting->current_rules_selection,
					   NULL);
  if (selected_rules == NULL)
    return;

  model = setting->current_rules_model;
  strncpy(last_rule_string, setting->current_rule_string,
	  MAXIMUM_LENGTH_OF_RULE_STRING);

  current_rule = selected_rules;
  while (current_rule != NULL)
    {
      gint rule;
      GtkTreeIter tmp_iter;
      RuleModelRelation *relation;
      GtkTreePath *path = (GtkTreePath *)current_rule->data;

      if (path == NULL)
	continue;

      if (!gtk_tree_model_get_iter(model, &tmp_iter, path))
	continue;

      gtk_tree_model_get(model, &tmp_iter,
			 FILTER_RULE, &rule,
			 -1);
      relation = get_relation_for_rule(setting, rule);
      if (relation == NULL)
	continue;

      tmp_rule_string[new_priority++] = rule;
      setting->current_rule_string[relation->priority]
	= THREADLIST_FILTER_RULE_NOP;

      current_rule = current_rule->next;
    }

  g_list_foreach(selected_rules, (GFunc)gtk_tree_path_free, NULL);
  g_list_free(selected_rules);

  tmp_pos = setting->current_rule_string;
  for (i = 0; i < setting->number_of_rules_applied; i++)
    {
      if (setting->current_rule_string[i] != THREADLIST_FILTER_RULE_NOP)
	{
	  *tmp_pos = setting->current_rule_string[i];
	  tmp_pos++;
	}
    }

  for (i = 0; i < new_priority; i++)
    {
      *tmp_pos = tmp_rule_string[i];
      tmp_pos++;
    }

  update_models(setting);
  if (strcmp(last_rule_string, setting->current_rule_string) != 0)
    g_signal_emit(G_OBJECT(setting),
		  threadlist_filter_setting_signals[CHANGED_SIGNAL],
		  0,
		  &setting->filter);
}


static void
current_rules_row_changed_cb(GtkTreeModel *model,
			     GtkTreePath *unused, GtkTreeIter *iter,
			     ThreadlistFilterSetting *setting)
{
  gint rule;
  gint priority;
  RuleModelRelation *relation;

  gtk_tree_model_get(model, iter,
		     FILTER_RULE, &rule, FILTER_RULE_PRIORITY, &priority, -1);
  relation = get_relation_for_rule(setting, rule);
  if (relation == NULL)
    {
#if DEBUG_WIDGET_MOST
      fprintf(stderr, "Invalid rule=%d, priority=%d\n", rule, priority);
#endif
      return;
    }
  gtk_list_store_set(GTK_LIST_STORE(setting->all_rules_model),
		     &relation->all_rules_iter,
		     FILTER_RULE_PRIORITY, priority,
		     -1);
  relation->priority = priority;
}


static void
update_current_rule_string(ThreadlistFilterSetting *setting)
{
  gint largest_priority = -1;
  const gchar *normal_rules = THREADLIST_FILTER_NORMAL_RULES;
  const gchar *rule = normal_rules;

  while (*rule != THREADLIST_FILTER_RULE_END)
    {
      RuleModelRelation *relation = get_relation_for_rule(setting, *rule);
      if (relation->priority != -1)
	{
	  if (setting->current_rule_string[relation->priority] != *rule)
	    setting->current_rule_string[relation->priority] = *rule;
	  if (relation->priority > largest_priority)
	    largest_priority = relation->priority;
	}
      rule++;
    }

  if (gtk_toggle_button_get_active(setting->default_show_all_button))
    setting->current_rule_string[largest_priority + 1]
      = THREADLIST_FILTER_RULE_SHOW_ALL;
  else
    setting->current_rule_string[largest_priority + 1]
      = THREADLIST_FILTER_RULE_HIDE_ALL;

  setting->current_rule_string[largest_priority + 2]
    = THREADLIST_FILTER_RULE_END;
}


static gboolean
all_rules_selection_filter(GtkTreeSelection *selection, GtkTreeModel *model,
			   GtkTreePath *path, gboolean currently_selected,
			   ThreadlistFilterSetting *setting)
{
  GtkTreeIter iter;
  gint priority;

  if (setting->unselecting)
    {
      return TRUE;
    }

  if (!gtk_tree_model_get_iter(model, &iter, path))
    return FALSE;

  gtk_tree_model_get(model, &iter, FILTER_RULE_PRIORITY, &priority, -1);

  return (priority == -1);
}


static void
setup_models(ThreadlistFilterSetting *setting)
{
  GtkListStore *store = gtk_list_store_new(FILTER_RULE_N_COLUMNS,
					   G_TYPE_INT, G_TYPE_INT);
  setting->all_rules_model = GTK_TREE_MODEL(store);

  gtk_list_store_append(store, &setting->show_favorite.all_rules_iter);
  gtk_list_store_set(store, &setting->show_favorite.all_rules_iter,
	FILTER_RULE, THREADLIST_FILTER_RULE_SHOW_FAVORITE,
	FILTER_RULE_PRIORITY, -1,
	-1);
  setting->show_favorite.priority = -1;

  gtk_list_store_append(store, &setting->hide_non_favorite.all_rules_iter);
  gtk_list_store_set(store, &setting->hide_non_favorite.all_rules_iter,
	FILTER_RULE, THREADLIST_FILTER_RULE_HIDE_NON_FAVORITE,
	FILTER_RULE_PRIORITY, -1,
	-1);
  setting->hide_non_favorite.priority = -1;

  gtk_list_store_append(store, &setting->show_seen.all_rules_iter);
  gtk_list_store_set(store, &setting->show_seen.all_rules_iter,
	FILTER_RULE, THREADLIST_FILTER_RULE_SHOW_THREAD_SEEN,
	FILTER_RULE_PRIORITY, -1,
	-1);
  setting->show_seen.priority = -1;

  gtk_list_store_append(store, &setting->hide_never_seen.all_rules_iter);
  gtk_list_store_set(store, &setting->hide_never_seen.all_rules_iter,
	FILTER_RULE, THREADLIST_FILTER_RULE_HIDE_THREAD_NEVER_SEEN,
	FILTER_RULE_PRIORITY, -1,
	-1);
  setting->hide_never_seen.priority = -1;

  gtk_list_store_append(store, &setting->show_seen_offline.all_rules_iter);
  gtk_list_store_set(store, &setting->show_seen_offline.all_rules_iter,
	FILTER_RULE, THREADLIST_FILTER_RULE_SHOW_THREAD_SEEN_OFFLINE,
	FILTER_RULE_PRIORITY, -1,
	-1);
  setting->show_seen_offline.priority = -1;

  gtk_list_store_append(store,
			&setting->hide_never_seen_offline.all_rules_iter);
  gtk_list_store_set(store, &setting->hide_never_seen_offline.all_rules_iter,
	FILTER_RULE, THREADLIST_FILTER_RULE_HIDE_THREAD_NEVER_SEEN_OFFLINE,
	FILTER_RULE_PRIORITY, -1,
	-1);
  setting->hide_never_seen_offline.priority = -1;

  gtk_list_store_append(store, &setting->show_new_comer.all_rules_iter);
  gtk_list_store_set(store, &setting->show_new_comer.all_rules_iter,
	FILTER_RULE, THREADLIST_FILTER_RULE_SHOW_NEW_COMER_THREAD,
	FILTER_RULE_PRIORITY, -1,
	-1);
  setting->show_new_comer.priority = -1;

  gtk_list_store_append(store,
			&setting->show_with_new_responses.all_rules_iter);
  gtk_list_store_set(store, &setting->show_with_new_responses.all_rules_iter,
	FILTER_RULE, THREADLIST_FILTER_RULE_SHOW_THREAD_WITH_NEW_RESPONSES,
	FILTER_RULE_PRIORITY, -1,
	-1);
  setting->show_with_new_responses.priority = -1;

  gtk_list_store_append(store,
			&setting->hide_without_new_responses.all_rules_iter);
  gtk_list_store_set(store,
		     &setting->hide_without_new_responses.all_rules_iter,
	FILTER_RULE, THREADLIST_FILTER_RULE_HIDE_THREAD_WITHOUT_NEW_RESPONSES,
	FILTER_RULE_PRIORITY, -1,
	-1);
  setting->hide_without_new_responses.priority = -1;

  gtk_list_store_append(store, &setting->show_dat_dropped.all_rules_iter);
  gtk_list_store_set(store, &setting->show_dat_dropped.all_rules_iter,
	FILTER_RULE, THREADLIST_FILTER_RULE_SHOW_DAT_DROPPED_THREAD,
	FILTER_RULE_PRIORITY, -1,
	-1);
  setting->show_dat_dropped.priority = -1;

  gtk_list_store_append(store, &setting->hide_much_ignored.all_rules_iter);
  gtk_list_store_set(store, &setting->hide_much_ignored.all_rules_iter,
	FILTER_RULE, THREADLIST_FILTER_RULE_HIDE_MUCH_IGNORED_THREAD,
	FILTER_RULE_PRIORITY, -1,
	-1);
  setting->hide_much_ignored.priority = -1;

  gtk_list_store_append(store, &setting->hide_hidden.all_rules_iter);
  gtk_list_store_set(store, &setting->hide_hidden.all_rules_iter,
	FILTER_RULE, THREADLIST_FILTER_RULE_HIDE_HIDDEN_THREAD,
	FILTER_RULE_PRIORITY, -1,
	-1);
  setting->hide_hidden.priority = -1;

  store = gtk_list_store_new(FILTER_RULE_N_COLUMNS, G_TYPE_INT, G_TYPE_INT);
  setting->current_rules_model = GTK_TREE_MODEL(store);

  g_signal_connect(G_OBJECT(store), "row_changed",
		   G_CALLBACK(current_rules_row_changed_cb), setting);
}


static void
update_models(ThreadlistFilterSetting *setting)
{
  const gchar *normal_rules = THREADLIST_FILTER_NORMAL_RULES;
  const gchar *rule = normal_rules;

  if (setting->filter.rule == NULL)
    return;

  /* ǽƤξ֤ꥻå */
  gtk_list_store_clear(GTK_LIST_STORE(setting->current_rules_model));
  while (*rule != THREADLIST_FILTER_RULE_END)
    {
      RuleModelRelation *relation = get_relation_for_rule(setting, *rule);
      gtk_list_store_set((GtkListStore *)setting->all_rules_model,
			 &relation->all_rules_iter,
			 FILTER_RULE_PRIORITY, -1,
			 -1);
      relation->priority = -1;
      rule++;
    }
  setting->number_of_rules_applied = 0;
  
  rule = setting->filter.rule;
  while (*rule != THREADLIST_FILTER_RULE_END)
    {
      if (*rule == THREADLIST_FILTER_RULE_SHOW_ALL)
	{
	  gtk_toggle_button_set_active(setting->default_show_all_button, TRUE);
	  break;
	}
      else if (*rule == THREADLIST_FILTER_RULE_HIDE_ALL)
	{
	  gtk_toggle_button_set_active(setting->default_hide_all_button, TRUE);
	  break;
	}

      append_rule(setting, *rule);
      rule++;
    }

  update_current_rule_string(setting);	/* rule(߾õ) */
}


static void
radio_button_toggled_cb(GtkToggleButton *unused,
			ThreadlistFilterSetting *setting)
{
  update_current_rule_string(setting);
  g_signal_emit(G_OBJECT(setting),
		threadlist_filter_setting_signals[CHANGED_SIGNAL],
		0,
		&setting->filter);
}  


static void
spin_button_value_changed_cb(GtkSpinButton *spin_button,
			     ThreadlistFilterSetting *setting)
{
  setting->filter.ignore_threshold = gtk_spin_button_get_value(spin_button);
  g_signal_emit(G_OBJECT(setting),
		threadlist_filter_setting_signals[CHANGED_SIGNAL],
		0,
		&setting->filter);
}


static void
restore_default_button_cb(GtkButton *unused,
			  ThreadlistFilterSetting *setting)
{
  g_signal_emit(G_OBJECT(setting),
		threadlist_filter_setting_signals[RESTORE_DEFAULT_SIGNAL],
		0);
		
}


static void
threadlist_filter_setting_restore_default(ThreadlistFilterSetting *setting)
{
  gtk_widget_set_sensitive(setting->restore_default_button, FALSE);
}


static void
threadlist_filter_setting_changed(ThreadlistFilterSetting *setting,
				  ThreadlistFilter *unused)
{
  gtk_widget_set_sensitive(setting->restore_default_button, TRUE);
}


static void
threadlist_filter_setting_init(ThreadlistFilterSetting *setting)
{
  GtkWidget *tmp_container;
  GtkWidget *tmp_widget;
  GtkWidget *table = gtk_table_new(3, 4, FALSE);

  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;

  setup_models(setting);

  tmp_widget = gtk_tree_view_new();
  setting->all_rules_view = GTK_TREE_VIEW(tmp_widget);
  setting->all_rules_selection
    = gtk_tree_view_get_selection(setting->all_rules_view);
  gtk_tree_selection_set_mode(setting->all_rules_selection,
			      GTK_SELECTION_MULTIPLE);
  gtk_tree_selection_set_select_function(setting->all_rules_selection,
			(GtkTreeSelectionFunc)all_rules_selection_filter,
			setting, NULL);
  gtk_tree_view_set_model(setting->all_rules_view, setting->all_rules_model);

  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(_("Available Rules"),
						    renderer,
						    "text", 0,
						    NULL);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
			(GtkTreeCellDataFunc)text_set_cell_data_available_rule,
			setting, NULL);
  gtk_tree_view_append_column(setting->all_rules_view, column);
  gtk_widget_show(tmp_widget);

  gtk_table_attach_defaults(GTK_TABLE(table), tmp_widget, 0, 1, 0, 1);



  tmp_container = gtk_vbutton_box_new();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(tmp_container),
			    GTK_BUTTONBOX_SPREAD);

  tmp_widget = gtk_button_new_from_stock(GTK_STOCK_ADD);
  g_signal_connect(G_OBJECT(tmp_widget), "clicked",
		   G_CALLBACK(append_rule_button_cb), setting);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(tmp_container), tmp_widget, FALSE, FALSE, 0);

  tmp_widget = gtk_button_new_from_stock(GTK_STOCK_REMOVE);
  g_signal_connect(G_OBJECT(tmp_widget), "clicked",
		   G_CALLBACK(remove_rule_button_cb), setting);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(tmp_container), tmp_widget, FALSE, FALSE, 0);

  gtk_widget_show(tmp_container);
  gtk_table_attach_defaults(GTK_TABLE(table), tmp_container, 1, 2, 0, 1);

  tmp_widget = gtk_tree_view_new();
  setting->current_rules_view = GTK_TREE_VIEW(tmp_widget);
  setting->current_rules_selection
    = gtk_tree_view_get_selection(setting->current_rules_view);
  gtk_tree_selection_set_mode(setting->current_rules_selection,
			      GTK_SELECTION_MULTIPLE);
  gtk_tree_view_set_model(setting->current_rules_view,
			  setting->current_rules_model);

  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(_("Filtering Rules"),
						    renderer,
						    "text", 0,
						    NULL);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
			(GtkTreeCellDataFunc)text_set_cell_data_applied_rule,
			setting, NULL);
  gtk_tree_view_append_column(setting->current_rules_view, column);

  gtk_widget_show(tmp_widget);

  gtk_table_attach_defaults(GTK_TABLE(table), tmp_widget, 2, 3, 0, 1);



  tmp_container = gtk_vbutton_box_new();

  tmp_widget = gtk_button_new_from_stock(GTK_STOCK_GOTO_TOP);
  g_signal_connect(G_OBJECT(tmp_widget), "clicked",
		   G_CALLBACK(goto_top_button_cb), setting);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(tmp_container), tmp_widget, FALSE, FALSE, 0);

  tmp_widget = gtk_button_new_from_stock(GTK_STOCK_GO_UP);
  g_signal_connect(G_OBJECT(tmp_widget), "clicked",
		   G_CALLBACK(go_up_button_cb), setting);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(tmp_container), tmp_widget, FALSE, FALSE, 0);


  tmp_widget = gtk_button_new_from_stock(GTK_STOCK_GO_DOWN);
  g_signal_connect(G_OBJECT(tmp_widget), "clicked",
		   G_CALLBACK(go_down_button_cb), setting);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(tmp_container), tmp_widget, FALSE, FALSE, 0);

  tmp_widget = gtk_button_new_from_stock(GTK_STOCK_GOTO_BOTTOM);
  g_signal_connect(G_OBJECT(tmp_widget), "clicked",
		   G_CALLBACK(goto_bottom_button_cb), setting);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(tmp_container), tmp_widget, FALSE, FALSE, 0);

  gtk_widget_show(tmp_container);
  gtk_table_attach_defaults(GTK_TABLE(table), tmp_container, 3, 4, 0, 1);



  tmp_container = gtk_hbox_new(FALSE, 0);

  tmp_widget = gtk_label_new(_("Threads not filtered:"));
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(tmp_container), tmp_widget, FALSE, FALSE, 0);

  tmp_widget = gtk_radio_button_new_with_label(NULL, _("Show all"));
  g_signal_connect(G_OBJECT(tmp_widget), "toggled",
		   G_CALLBACK(radio_button_toggled_cb), setting);

  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(tmp_container), tmp_widget, FALSE, FALSE, 0);
  setting->default_show_all_button = GTK_TOGGLE_BUTTON(tmp_widget);

  tmp_widget
    = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(tmp_widget),
						  _("Hide all"));
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(tmp_container), tmp_widget, FALSE, FALSE, 0);
  setting->default_hide_all_button = GTK_TOGGLE_BUTTON(tmp_widget);

  gtk_widget_show(tmp_container);
  gtk_table_attach_defaults(GTK_TABLE(table), tmp_container, 2, 4, 1, 2);


  tmp_container = gtk_hbox_new(FALSE, 0);

  tmp_widget
    = gtk_spin_button_new(GTK_ADJUSTMENT(gtk_adjustment_new(setting->filter.ignore_threshold,
							    0, 10000,
							    1, 10, 0)),
			  1.0, 0);
  setting->spin_button = GTK_SPIN_BUTTON(tmp_widget);
  g_signal_connect(G_OBJECT(tmp_widget), "value_changed",
		   G_CALLBACK(spin_button_value_changed_cb), setting);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_end(GTK_BOX(tmp_container), tmp_widget, FALSE, FALSE, 0);

  tmp_widget = gtk_label_new(_("Ignored Threshold: "));
  gtk_widget_show(tmp_widget);
  gtk_box_pack_end(GTK_BOX(tmp_container), tmp_widget, FALSE, FALSE, 0);

  gtk_widget_show(tmp_container);
  gtk_table_attach_defaults(GTK_TABLE(table), tmp_container, 0, 1, 1, 3);


  tmp_container = gtk_vbutton_box_new();

  tmp_widget = gtk_button_new_with_label(_("Restore the default settings"));
  setting->restore_default_button = tmp_widget;
  g_signal_connect(G_OBJECT(tmp_widget), "clicked",
		   G_CALLBACK(restore_default_button_cb), setting);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_end(GTK_BOX(tmp_container), tmp_widget, FALSE, FALSE, 0);

  gtk_widget_show(tmp_container);
  gtk_table_attach_defaults(GTK_TABLE(table), tmp_container, 2, 4, 2, 3);



  gtk_widget_show(table);
  gtk_box_pack_start(GTK_BOX(&setting->container), table, FALSE, FALSE, 3);

  setting->filter.rule = setting->current_rule_string;
}


static void
threadlist_filter_setting_finalize(GObject *object)
{
  ThreadlistFilterSetting *setting = THREADLIST_FILTER_SETTING(object);

  if (setting->all_rules_model != NULL)
    {
      g_object_unref(setting->all_rules_model);
      setting->all_rules_model = NULL;
    }
  if (setting->current_rules_model != NULL)
    {
      g_object_unref(setting->current_rules_model);
      setting->current_rules_model = NULL;
    }

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


static void
threadlist_filter_setting_destroy(GtkObject *object)
{
  if (GTK_OBJECT_CLASS(parent_class)->destroy)
    (*GTK_OBJECT_CLASS(parent_class)->destroy)(object);
}


static const gchar *
get_threadlist_filter_rule_string(int rule)
{
  switch (rule)
    {
    case THREADLIST_FILTER_RULE_SHOW_ALL:
      return show_all;

    case THREADLIST_FILTER_RULE_HIDE_ALL:
      return hide_all;

    case THREADLIST_FILTER_RULE_SHOW_FAVORITE:
      return show_favorite;

    case THREADLIST_FILTER_RULE_HIDE_NON_FAVORITE:
      return hide_non_favorite;
      
    case THREADLIST_FILTER_RULE_SHOW_THREAD_SEEN:
      return show_seen;

    case THREADLIST_FILTER_RULE_HIDE_THREAD_NEVER_SEEN:
      return hide_unseen;

    case THREADLIST_FILTER_RULE_SHOW_THREAD_SEEN_OFFLINE:
      return show_seen_offline;

    case THREADLIST_FILTER_RULE_HIDE_THREAD_NEVER_SEEN_OFFLINE:
      return hide_unseen_offline;

    case THREADLIST_FILTER_RULE_SHOW_NEW_COMER_THREAD:
      return show_new_comer;

    case THREADLIST_FILTER_RULE_SHOW_THREAD_WITH_NEW_RESPONSES:
      return show_new_res;

    case THREADLIST_FILTER_RULE_HIDE_THREAD_WITHOUT_NEW_RESPONSES:
      return hide_without_new_res;

    case THREADLIST_FILTER_RULE_SHOW_DAT_DROPPED_THREAD:
      return show_dat_dropped;

    case THREADLIST_FILTER_RULE_HIDE_MUCH_IGNORED_THREAD:
      return hide_ignored;

    case THREADLIST_FILTER_RULE_HIDE_HIDDEN_THREAD:
      return hide_hidden;

    default:
      return invalid_rule;
    }
}


GtkWidget *threadlist_filter_setting_new(void)
{
  return GTK_WIDGET(g_object_new(THREADLIST_FILTER_SETTING_TYPE, NULL));
}


GtkWidget *
threadlist_filter_setting_new_with_initial_setting(const ThreadlistFilter *filter)
{
  ThreadlistFilterSetting *setting
    = THREADLIST_FILTER_SETTING(g_object_new(THREADLIST_FILTER_SETTING_TYPE,
					     NULL));
  threadlist_filter_setting_set_filter(setting, filter);

  return GTK_WIDGET(setting);
}


static RuleModelRelation *
get_relation_for_rule(ThreadlistFilterSetting *setting, gint rule)
{
  switch (rule)
    {
    case THREADLIST_FILTER_RULE_SHOW_FAVORITE:
      return &setting->show_favorite;
      
    case THREADLIST_FILTER_RULE_HIDE_NON_FAVORITE:
      return &setting->hide_non_favorite;
      
    case THREADLIST_FILTER_RULE_SHOW_THREAD_SEEN:
      return &setting->show_seen;

    case THREADLIST_FILTER_RULE_HIDE_THREAD_NEVER_SEEN:
      return &setting->hide_never_seen;

    case THREADLIST_FILTER_RULE_SHOW_THREAD_SEEN_OFFLINE:
      return &setting->show_seen_offline;

    case THREADLIST_FILTER_RULE_HIDE_THREAD_NEVER_SEEN_OFFLINE:
      return &setting->hide_never_seen_offline;

    case THREADLIST_FILTER_RULE_SHOW_NEW_COMER_THREAD:
      return &setting->show_new_comer;

    case THREADLIST_FILTER_RULE_SHOW_THREAD_WITH_NEW_RESPONSES:
      return &setting->show_with_new_responses;

    case THREADLIST_FILTER_RULE_HIDE_THREAD_WITHOUT_NEW_RESPONSES:
      return &setting->hide_without_new_responses;

    case THREADLIST_FILTER_RULE_SHOW_DAT_DROPPED_THREAD:
      return &setting->show_dat_dropped;

    case THREADLIST_FILTER_RULE_HIDE_MUCH_IGNORED_THREAD:
      return &setting->hide_much_ignored;

    case THREADLIST_FILTER_RULE_HIDE_HIDDEN_THREAD:
      return &setting->hide_hidden;

    default:
      break;
    }
#if DEBUG_WIDGET
  fprintf(stderr, "Invalid rule: '%c'=%d\n", rule, rule);
#endif
  return NULL;
}


static void
text_set_cell_data_available_rule(GtkTreeViewColumn *column,
				  GtkCellRenderer *renderer,
				  GtkTreeModel *model,
				  GtkTreeIter *iter,
				  ThreadlistFilterSetting *setting)
{
  /*  ThreadlistFilter *filter = &setting->filter; */
  GValue src = { 0, };
  GValue psrc = { 0, };
  GValue text = { 0, };
  GValue visible = { 0, };
  gint rule;
  gint priority;

  gtk_tree_model_get_value(model, iter, FILTER_RULE, &src);
  rule = g_value_get_int(&src);
  gtk_tree_model_get_value(model, iter, FILTER_RULE_PRIORITY, &psrc);
  priority = g_value_get_int(&psrc);

  g_value_init(&text, G_TYPE_STRING);
  g_value_set_static_string(&text, get_threadlist_filter_rule_string(rule));
  g_object_set_property((GObject *)renderer, "text", &text);

#if 0
  g_value_init(&visible, G_TYPE_BOOLEAN);
  g_value_set_boolean(&visible, priority == -1);
  g_object_set_property((GObject *)renderer, "visible", &visible);
#else	/* XXX: ġɤ꤬ɤʤʡ */
  g_value_init(&visible, G_TYPE_STRING);
  if (priority != -1)
    g_value_set_static_string(&visible, "gray70");
  else
    g_value_set_static_string(&visible, "black");
  g_object_set_property((GObject *)renderer, "foreground", &visible);
#endif
}


static void
text_set_cell_data_applied_rule(GtkTreeViewColumn *column,
				GtkCellRenderer *renderer,
				GtkTreeModel *model,
				GtkTreeIter *iter,
				ThreadlistFilterSetting *setting)
{
  /*  ThreadlistFilter *filter = &setting->filter; */
  GValue src = { 0, };
  GValue text = { 0, };
  gint rule;

  gtk_tree_model_get_value(model, iter, FILTER_RULE, &src);
  rule = g_value_get_int(&src);

  g_value_init(&text, G_TYPE_STRING);
  g_value_set_static_string(&text, get_threadlist_filter_rule_string(rule));
  g_object_set_property((GObject *)renderer, "text", &text);
}


void
threadlist_filter_setting_set_filter(ThreadlistFilterSetting *setting,
				     const ThreadlistFilter *filter)
{
  strncpy(setting->current_rule_string, filter->rule,
	  MAXIMUM_LENGTH_OF_RULE_STRING);
  setting->filter.ignore_threshold = filter->ignore_threshold;

  gtk_spin_button_set_value(setting->spin_button, filter->ignore_threshold);

  update_models(setting);
}


const ThreadlistFilter *
threadlist_filter_setting_get_filter(ThreadlistFilterSetting *setting)
{
  return &setting->filter;
}


void
threadlist_filter_setting_set_restore_default_button_sensitive(ThreadlistFilterSetting *setting,
							       gboolean state)
{
  gtk_widget_set_sensitive(setting->restore_default_button, state);
}


void
setup_default_threadlist_filter(OchushaApplication *application)
{
  if (threadlist_filter_rule_default != NULL)
    free(threadlist_filter_rule_default);

  if (application->user_default_filter.rule == NULL)
    {
      threadlist_filter_rule_default
	= g_strdup(threadlist_filter_rule_ochusha_default);
      application->user_default_filter.rule
	= g_strdup(threadlist_filter_rule_default);
    }
  else
    {
      threadlist_filter_rule_default =
	g_strdup(application->user_default_filter.rule);
    }

  threadlist_filter_rule_default_ignore_threshold
    = application->user_default_filter.ignore_threshold;
}


gboolean
threadlist_filter_check_thread(ThreadlistFilter *filter,
			       OchushaApplication *application,
			       BBSThread *thread)
{
  BBSThreadGUIInfo *thread_info = (BBSThreadGUIInfo *)thread->user_data;
  const gchar *rules;
  int threshold;
  const gchar *cur_rule;

  if (filter->rule != NULL)
    {
      rules = filter->rule;
      threshold = filter->ignore_threshold;
    }
  else
    {
      rules = threadlist_filter_rule_default;
      threshold = threadlist_filter_rule_default_ignore_threshold;
    }
  cur_rule = rules;

  while (*cur_rule != THREADLIST_FILTER_RULE_END)
    {
      switch (*cur_rule)
	{
	case THREADLIST_FILTER_RULE_SHOW_ALL:
#if DEBUG_GUI_MOST
	  fprintf(stderr, "TRUE: show all\n");
#endif
	  return TRUE;

	case THREADLIST_FILTER_RULE_HIDE_ALL:
#if DEBUG_GUI_MOST
	  fprintf(stderr, "FALSE: hide all\n");
#endif
	  return FALSE;

	case THREADLIST_FILTER_RULE_SHOW_FAVORITE:
	  if (thread_info != NULL
	      && thread_info->info.view_flags & BBS_THREAD_FAVORITE)
	    {
#if DEBUG_GUI_MOST
	      fprintf(stderr, "TRUE: show favorite\n");
#endif
	      return TRUE;
	    }
	  break;
      
	case THREADLIST_FILTER_RULE_HIDE_NON_FAVORITE:
	  if (thread_info != NULL
	      && (thread_info->info.view_flags & BBS_THREAD_FAVORITE) == 0)
	    {
#if DEBUG_GUI_MOST
	      fprintf(stderr, "FALSE: hide non favorite\n");
#endif
	      return FALSE;
	    }
	  break;
      
	case THREADLIST_FILTER_RULE_SHOW_THREAD_SEEN:
	  if (thread->number_of_responses_read > 0)
	    {
#if DEBUG_GUI_MOST
	      fprintf(stderr, "TRUE: show thread seen\n");
#endif
	      return TRUE;
	    }
	  break;

	case THREADLIST_FILTER_RULE_HIDE_THREAD_NEVER_SEEN:
	  if (thread->number_of_responses_read == 0)
	    {
#if DEBUG_GUI_MOST
	      fprintf(stderr, "FALSE: hide thread never seen\n");
#endif
	      return FALSE;
	    }
	  break;

	case THREADLIST_FILTER_RULE_SHOW_THREAD_SEEN_OFFLINE:
	  if (application->conf.offline
	      && thread->number_of_responses_read > 0)
	    {
#if DEBUG_GUI_MOST
	      fprintf(stderr, "TRUE: show thread seen (offline)\n");
#endif
	      return TRUE;
	    }
	  break;

	case THREADLIST_FILTER_RULE_HIDE_THREAD_NEVER_SEEN_OFFLINE:
	  if (application->conf.offline
	      && thread->number_of_responses_read == 0)
	    {
#if DEBUG_GUI_MOST
	      fprintf(stderr, "FALSE: hide thread never seen (offline)\n");
#endif
	      return FALSE;
	    }
	  break;

	case THREADLIST_FILTER_RULE_SHOW_NEW_COMER_THREAD:
	  if (thread_info != NULL
	      && thread_info->info.view_rank == 0)
	    {
#if DEBUG_GUI_MOST
	      fprintf(stderr, "TRUE: show new comer thread.\n");
#endif
	      return TRUE;
	    }
	  break;

	case THREADLIST_FILTER_RULE_SHOW_THREAD_WITH_NEW_RESPONSES:
	  if (thread->number_of_responses_on_server
	      > thread->number_of_responses_read)
	    {
#if DEBUG_GUI_MOST
	      fprintf(stderr, "TRUE: show thread with new responses.\n");
#endif
	      return TRUE;
	    }
	  break;

	case THREADLIST_FILTER_RULE_HIDE_THREAD_WITHOUT_NEW_RESPONSES:
	  if (thread->number_of_responses_on_server
	      <= thread->number_of_responses_read)
	    {
#if DEBUG_GUI_MOST
	      fprintf(stderr, "FALSE: hide thread without new responses.\n");
#endif
	      return FALSE;
	    }
	  break;

	case THREADLIST_FILTER_RULE_SHOW_DAT_DROPPED_THREAD:
	  if (thread->flags & BBS_THREAD_DAT_DROPPED)
	    {
#if DEBUG_GUI_MOST
	      fprintf(stderr, "TRUE: show thread whose dat was dropped.\n");
#endif
	      return TRUE;
	    }
	  break;

	case THREADLIST_FILTER_RULE_HIDE_MUCH_IGNORED_THREAD:
	  if (threshold != 0
	      && thread_info != NULL && thread_info->view_ignored > threshold)
	    {
#if DEBUG_GUI_MOST
	      fprintf(stderr, "FALSE: hide thread that is too many times ignored.\n");
#endif
	      return FALSE;
	    }
	  break;

	case THREADLIST_FILTER_RULE_HIDE_HIDDEN_THREAD:
	  if (thread_info != NULL
	      && thread_info->info.view_flags & BBS_THREAD_HIDDEN)
	    {
#if DEBUG_GUI_MOST
	      fprintf(stderr, "FALSE: hide thread marked as hidden.\n");
#endif
	      return FALSE;
	    }
	  break;

#if DEBUG_GUI_MOST
	default:
	  fprintf(stderr, "threadlist_filter: no match rule for '%c'\n",
		  *cur_rule);
#endif
	}
      cur_rule++;
    }

  /* 롼뤬ȤƤ餳ˤʤͽġ */
  return TRUE;
}
