/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id$
 */

#include "icon_label.h"
#include "ochusha_ui.h"

#include "marshal.h"

#include <gtk/gtk.h>

#include <stdlib.h>


static void icon_label_class_init(IconLabelClass *klass);
static void icon_label_init(IconLabel *icon_label);
static void icon_label_finalize(GObject *object);
static void icon_label_destroy(GtkObject *object);
static void real_alternate_icon(IconLabel *icon_label, int icon_number);


GType
icon_label_get_type(void)
{
  static GType tl_type = 0;

  if (tl_type == 0)
    {
      static const GTypeInfo tl_info =
	{
	  sizeof(IconLabelClass),
	  NULL,	/* base_init */
	  NULL,	/* base_finalize */
	  (GClassInitFunc)icon_label_class_init,
	  NULL,	/* class_finalize */
	  NULL,	/* class_data */
	  sizeof(IconLabel),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)icon_label_init,
	};

      tl_type = g_type_register_static(GTK_TYPE_HBOX,
				       "IconLabel", &tl_info, 0);
    }

  return tl_type;
}


enum {
  ICON_LABEL_ALTERNATE_ICON_SIGNAL,
  LAST_SIGNAL
};


static GtkHBoxClass *parent_class = NULL;
static gint icon_label_signals[LAST_SIGNAL] = { 0 };


static void
icon_label_class_init(IconLabelClass *klass)
{
  GObjectClass *o_class = (GObjectClass *)klass;
  GtkObjectClass *object_class = (GtkObjectClass *)klass;

  parent_class = g_type_class_peek_parent(klass);

  /* GObject signals */
  o_class->finalize = icon_label_finalize;

  /* GtkObject signals */
  object_class->destroy = icon_label_destroy;

  icon_label_signals[ICON_LABEL_ALTERNATE_ICON_SIGNAL] =
    g_signal_new("alternate_icon",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(IconLabelClass, alternate_icon),
		 NULL, NULL,
		 ochusha_marshal_VOID__INT,
		 G_TYPE_NONE, 1,
		 G_TYPE_INT);
  klass->alternate_icon = real_alternate_icon;
}


static void
icon_label_init(IconLabel *icon_label)
{
  icon_label->current_icon = -1;
  icon_label->number_of_icons = 0;
  icon_label->icon_list = NULL;
  icon_label->counter = 0;
}


static void
icon_label_finalize(GObject *object)
{
  IconLabel *icon_label = ICON_LABEL(object);
  int i;
#if DEBUG_WIDGET_MOST
  fprintf(stderr, "icon_label_finalize\n");
#endif

  for (i = 0; i < icon_label->number_of_icons; i++)
    {
      if (icon_label->icon_list[i] != NULL)
	{
#if DEBUG_WIDGET_MOST
	  fprintf(stderr, "icon_list[%d]->ref_count=%d\n", i,
		  G_OBJECT(icon_label->icon_list[i])->ref_count);
#endif
	  g_object_unref(G_OBJECT(icon_label->icon_list[i]));
	  icon_label->icon_list[i] = NULL;
	}
    }
  
  icon_label->current_icon = -1;
  icon_label->number_of_icons = 0;
  free(icon_label->icon_list);
  icon_label->icon_list = NULL;

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


static void
icon_label_destroy(GtkObject *object)
{

#if DEBUG_WIDGET_MOST
  fprintf(stderr, "icon_label_destroy\n");
#endif

  if (GTK_OBJECT_CLASS(parent_class)->destroy)
    (*GTK_OBJECT_CLASS(parent_class)->destroy)(object);
}


static void
real_alternate_icon(IconLabel *icon_label, int icon_number)
{
  if (icon_label->number_of_icons <= icon_number && icon_number != -1)
    return;

  if (icon_label->current_icon != -1)
    {
#if 0
      g_object_ref(G_OBJECT(icon_label->icon_list[icon_label->current_icon]));
#endif
      gtk_container_remove(GTK_CONTAINER(&icon_label->container),
			   icon_label->icon_list[icon_label->current_icon]);
#if DEBUG_WIDGET_MOST
      fprintf(stderr, "icon_list[%d]->ref_count=%d\n", icon_label->current_icon,
	      G_OBJECT(icon_label->icon_list[icon_label->current_icon])->ref_count);
#endif
    }

  icon_label->current_icon = icon_number;

  if (icon_number == -1)
    return;

  gtk_widget_show(icon_label->icon_list[icon_number]);
  gtk_box_pack_start(GTK_BOX(&icon_label->container),
		     icon_label->icon_list[icon_number], FALSE, FALSE, 0);
  gtk_box_reorder_child(GTK_BOX(&icon_label->container),
			icon_label->icon_list[icon_number], 0);
#if DEBUG_WIDGET_MOST
  fprintf(stderr, "icon_list[%d]->ref_count=%d\n", icon_number,
	  G_OBJECT(icon_label->icon_list[icon_number])->ref_count);
#endif
}


GtkWidget *
icon_label_new(const gchar *label_text, GtkWidget *icon_list[],
	      int number_of_icons, int initial_icon)
{
  GtkWidget *widget = g_object_new(ICON_LABEL_TYPE, NULL);
  IconLabel *icon_label = ICON_LABEL(widget);

  GtkWidget *label = gtk_label_new(label_text);
  icon_label->label = GTK_LABEL(label);
  gtk_widget_show(label);
  gtk_box_pack_start(GTK_BOX(&icon_label->container), label, FALSE, FALSE, 0);

  icon_label->icon_list = icon_list;
  icon_label->number_of_icons = number_of_icons;

  if (icon_list != NULL)
    {
      int i;
      for (i = 0; i < number_of_icons; i++)
	{
	  g_object_ref(G_OBJECT(icon_list[i]));
	  gtk_object_sink(GTK_OBJECT(icon_list[i]));
	}
    }

  if (number_of_icons > 0 && initial_icon != -1)
    real_alternate_icon(icon_label, initial_icon);

  return widget;
}


void
icon_label_set_label_text(IconLabel *icon_label, const gchar *label_text)
{
  gtk_label_set_text(icon_label->label, label_text);
}


void
icon_label_alternate_icon(IconLabel *icon_label, int icon_number)
{
  real_alternate_icon(icon_label, icon_number);
}

