/*
 * Copyright (c) 2003-2004 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: utils.c,v 1.30.2.6 2004/11/25 08:55:51 fuyu Exp $
 */

#define OCHUSHA_UTILS_IMPLEMENTATION	1

#include "config.h"

#include "ochusha_private.h"
#include "utils.h"

#include <errno.h>
#include <iconv.h>
#if HAVE_LANGINFO_CODESET
#include <langinfo.h>
#endif

#if HAVE_LOCALE_H
#include <locale.h>
#endif

#include <fcntl.h>
#include <limits.h>

#if !defined(HAVE_TIMEGM) || TRACE_MEMORY_USAGE || TRACE_REFCOUNT
#include <pthread.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>


iconv_t utf8_to_native = 0;	/* ǥХå */

#define DEFAULT_BUFFER_SIZE	4096
#define MAX_HOSTNAME_LEN	256



#if 0
static void validate_eucjp_to_utf8_tables(void);
#endif


/*
 * initialize_common_convertes
 *
 * ɤȤencodingconverter롣
 *
 * MEMO: converterϥåɥդʤȤա
 */
void
initialize_common_converters(void)
{
  if (utf8_to_native == 0)
    {
#if HAVE_LANGINFO_CODESET
      char codeset[256];
      int len = snprintf(codeset, 256, "%s//IGNORE", nl_langinfo(CODESET));
      if (len < 256)
	{
	  utf8_to_native = iconv_open(codeset, "UTF-8");
	  if (utf8_to_native == (iconv_t)-1)
	    {
	      fprintf(stderr, "Couldn't iconv_open(\"%s\", \"UTF-8\")\n",
		      codeset);
	      utf8_to_native = 0;	/* try hardcoded one */
	    }
	}

      if (utf8_to_native == 0)
	utf8_to_native = iconv_open("EUC-JP//IGNORE", "UTF-8");
#else
      utf8_to_native = iconv_open("EUC-JP//IGNORE", "UTF-8");
#endif
      if (utf8_to_native == (iconv_t)-1)
	{
	  fprintf(stderr, "iconv_open() failed.\n");
	  exit(1);
	}
    }

#if 0
  validate_eucjp_to_utf8_tables();
#endif
}


/*
 * convert_string
 *
 * iconvȤäsrcʸencodingѴѴ줿ʸ֤
 * Ѵ줿ʸheapmalloc줿ΰ˳ǼΤǡ桼
 * פˤʤ꼡freeǤ롣
 * Ϳ줿length-1ξ硢srcNULüCʸȤƼ갷졢
 * strlen(src)η̤ʸĹȤƼ갷롣ʳξ硢
 * Ϳ줿lengthĹʬѴ롣
 * NULLhelperͿ줿硢ǤiconvѴ˼Ԥ֤
 * helperѴ³ꤹ롣
 */
char *
convert_string(iconv_t converter, iconv_helper *helper,
	       const char *src, int length)
{
  char default_buffer[DEFAULT_BUFFER_SIZE];
  size_t buffer_size = DEFAULT_BUFFER_SIZE;
  char *buffer = default_buffer;
  const char *inbuf;
  char *outbuf;
  size_t outbytesleft;
  size_t inbytesleft;

  if (length < 0)
    length = strlen(src);

  while (1)
    {
      size_t size;

      inbuf = src;
      outbuf = buffer;
      outbytesleft = buffer_size;
      inbytesleft = length;

#if ENABLE_MS_IZONMOJI
      do
	{
	  size = iconv(converter,
		       (ICONV_CONST char **)&inbuf, &inbytesleft,
		       &outbuf, &outbytesleft);
	  if (size == -1 && errno == EILSEQ && helper != NULL)
	    size = (*helper)(&inbuf, &inbytesleft, &outbuf, &outbytesleft);
	  else
	    break;
	} while (size != -1);

      if (size != -1)
	break;	/* ｪλ */

      if (errno == E2BIG)
	{
	  /* allocate 2 times larger buffer in the heap. */
	  buffer_size *= 2;
	  if (buffer != default_buffer)
	    buffer = (char *)G_REALLOC(buffer, buffer_size);
	  else
	    buffer = (char *)G_MALLOC(buffer_size);
	}
      else
	{
#if DEBUG_ICONV
	  fprintf(stderr, "iconv failed(errno = %d).\n", errno);
	  fprintf(stderr, "character code = 0x%02x\n", (int)*inbuf);
#endif
	  if (buffer != default_buffer)
	    G_FREE(buffer);
	  return NULL;
	}
#else
      size = iconv(converter, (ICONV_CONST char **)&inbuf, &inbytesleft,
		   &outbuf, &outbytesleft);
      if (size == -1)
	{
	  if (errno == E2BIG)
	    {
	      /* allocate 2 times larger buffer in the heap. */
	      buffer_size *= 2;
	      if (buffer != default_buffer)
		buffer = (char *)G_REALLOC(buffer, buffer_size);
	      else
		buffer = (char *)G_MALLOC(buffer_size);
	    }
	  else
	    {
#if DEBUG_ICONV
	      fprintf(stderr, "iconv failed(errno = %d).\n", errno);
	      fprintf(stderr, "character code = 0x%02x\n", (int)*inbuf);
#endif
	      if (buffer != default_buffer)
		G_FREE(buffer);
	      return NULL;
	    }
	}
      else
	break;	/* ｪλ */
#endif
    }

  if (outbytesleft == 0)
    {
      if (buffer != default_buffer)
	buffer = (char *)G_REALLOC(buffer, buffer_size + 1);
      else
	{
	  buffer = (char *)G_MALLOC(buffer_size + 1);
	  memcpy(buffer, src, buffer_size);
	}
      buffer[buffer_size] = '\0';
      return buffer;
    }

  *outbuf = '\0';

  if (buffer == default_buffer)
    return G_STRDUP(buffer);

  return (char *)G_REALLOC(buffer, outbuf - buffer + 1);
}


/*
 * wipe_string:
 *   string˴ޤޤnon-printableʸȥĤζʸ
 *   ԡ֤
 */
gchar *
wipe_string(const gchar *string)
{
  gchar *result_string = g_strchomp(G_STRDUP(string));
  gchar *head = result_string;

  while (*head != '\0')
    {
      if (*head > '\0' && *head <= '\37')
	{	/* ȯˤĤʬФ */
	  int len;
	  gchar *tmp_pos;
	  for (tmp_pos = g_utf8_next_char(head);
	       *tmp_pos > '\0' && *tmp_pos <= '\37';
	       tmp_pos = g_utf8_next_char(tmp_pos));
	  len = strlen(tmp_pos);
	  memmove(head, tmp_pos, len + 1);
	}
      else
	head = g_utf8_next_char(head);
    }
  return result_string;
}


/*
 * ochusha_utils_shell_escape_text
 *
 * Ϳ줿ʸ򥷥Ϥޥɥ饤Ȥƥפɬ
 * ʸ򥨥פʸΥԡ֤
 * Ǥϡ#\#֤
 */
char *
ochusha_utils_shell_escape_text(const char *text)
{
  int count = 0;
  const char *cur_pos = text;
  char *result;
  char *tmp_pos;

  while (*cur_pos != '\0')
    {
      if (*cur_pos == '#')
	count++;
      cur_pos++;
    }

  if (count == 0)
    return G_STRDUP(text);

  result = G_MALLOC(strlen(text) + count + 1);
  cur_pos = text;
  tmp_pos = result;
  do
    {
      *tmp_pos++ = *cur_pos;
      if (*cur_pos == '#')
	{
	  tmp_pos[-1] = '\\';
	  *tmp_pos++ = '#';
	}
    } while (*cur_pos++ != '\0');

  return result;
}


/*
 * ochusha_utils_url_extract_scheme
 *
 * Ϳ줿ʸURLȤƲᤷschemeʬ֤
 * ˼ԤˤNULL֤
 */
char *
ochusha_utils_url_extract_scheme(const char *url)
{
  char *comma_pos;
  g_return_val_if_fail(url != NULL, NULL);

  comma_pos = strchr(url, ':');
  if (comma_pos == NULL)
    return NULL;

  /*  */
  return G_STRNDUP(url, comma_pos - url);
}


/*
 * ochusha_utils_url_extract_http_server
 *
 * Ϳ줿ʸhttpФURLǤȤƥ̾ʬ֤
 * ˼ԤˤNULL֤
 */
char *
ochusha_utils_url_extract_http_server(const char *url)
{
  /*
   * ݤʤΤǡurlhttp://server/.*ʷǤȷǤġ
   */
  char *cur_pos;
  char *hostpart_tail;
  int length;
  char buffer[MAX_HOSTNAME_LEN];

  cur_pos = strstr(url, "http://");
  if (cur_pos == NULL)
    {
      cur_pos = strstr(url, "https://");
      if (cur_pos == NULL)
	{
#if DEBUG_UTILS
	  fprintf(stderr, "Not a http URL: %s\n", url);
#endif
	  return NULL;
	}
      cur_pos += 8;	/* skip "https://" */
    }
  else
    cur_pos += 7;	/* skip "http://" */

  hostpart_tail = strchr(cur_pos, '/');
  if (hostpart_tail == NULL)
    {
      if (*cur_pos != '\n')
	return G_STRDUP(cur_pos);
      else
	return NULL;
    }

  length = hostpart_tail - cur_pos;
  if (length < 0 || length >= MAX_HOSTNAME_LEN)
    {
#if DEBUG_UTILS
      fprintf(stderr, "Couldn't parse URL: %s\n", url);
#endif
      return NULL;
    }

  memcpy(buffer, cur_pos, length);
  buffer[length] = '\0';
  return G_STRDUP(buffer);
}


/*
 * ochusha_utils_url_extract_http_absolute_path
 *
 * Ϳ줿ʸhttpФURLǤȤƲϤ
 * ɥȤΥФˤХѥ֤̾
 * ex) http://pc.2ch.net/unix/  /unix/
 * ˼ԤˤNULL֤
 */
char *
ochusha_utils_url_extract_http_absolute_path(const char *url)
{
  /*
   * ݤʤΤǡurlhttp://server/.*ʷǤȷǤġ
   */
  char *cur_pos = strstr(url, "http://");

  if (cur_pos == NULL)
    {
#if DEBUG_UTILS
      fprintf(stderr, "Couldn't parse URL: %s\n", url);
#endif
      return NULL;
    }
  cur_pos += 7; /* skip "http://" */
  cur_pos = strchr(cur_pos, '/');
  if (cur_pos == NULL)
    {
#if DEBUG_UTILS
      fprintf(stderr, "Couldn't get path parts: %s\n", url);
#endif
      return NULL;
    }

  return G_STRDUP(cur_pos);
}


char *
ochusha_utils_url_encode_string(const char *src)
{
  int src_len;
  int buffer_len;
  const char *src_pos;
  char *buf_pos;
  char *buf_tail;
  char *buffer;
  unsigned char c;
  gboolean expand_buf;

  g_return_val_if_fail(src != NULL, NULL);

  src_len = strlen(src);
  buffer_len = src_len * 3 + 1;
  buffer = (char *)G_MALLOC(buffer_len);
  buf_tail = buffer + buffer_len - 1;
  src_pos = src;
  buf_pos = buffer;
  expand_buf = FALSE;

#if DEBUG_UTILS
  fprintf(stderr, "url_encode: src=\"%s\"\n", src);
#endif

  while ((c = *src_pos) != '\0')
    {
      if (expand_buf)
	{
	  int result_len = buf_pos - buffer;
#if DEBUG_UTILS
	  fprintf(stderr, "expand_buf@url_encode: buffer_len %d => %d\n",
		  buffer_len, buffer_len * 2);
#endif
	  buffer_len *= 2;
	  buffer = (char *)G_REALLOC(buffer, buffer_len);
	  buf_tail = buffer + buffer_len - 1;
	  buf_pos = buffer + result_len;
	  expand_buf = FALSE;
	}

      if (g_ascii_isalnum(c)
	  || c == '*' || c == '-' || c == '.' || c == '@' || c == '_')
	{
	  if (buf_pos >= buf_tail)
	    {
	      expand_buf = TRUE;
	      continue;	/* bufferĥƤľ: ꤽˤʤʡ */
	    }
	  *buf_pos++ = c;
	}
      else if (c == ' ')
	{
	  if (buf_pos >= buf_tail)
	    {
	      expand_buf = TRUE;
	      continue;	/* bufferĥƤľ: ꤽˤʤʡ */
	    }
	  *buf_pos++ = '+';
	}
      else
	{
	  int left = buf_tail - buf_pos;
	  if (left < 4)
	    {
	      expand_buf = TRUE;
	      continue;	/* bufferĥƤľ: ꤽˤʤʡ */
	    }
	  buf_pos += g_snprintf(buf_pos, left, "%%%02X", c);
	}
      src_pos++;
    }

  g_assert(buf_pos <= buf_tail);
  *buf_pos = '\0';

#if DEBUG_UTILS
  fprintf(stderr, "url_encode: \"%s\"\n", buffer);
#endif

  return buffer;
}


char *
ochusha_utils_url_decode_string(const char *src)
{
  int src_len = strlen(src);
  char *buffer = (char *)G_MALLOC(src_len + 1);
  const char *cur_src = src;
  char *cur_dst = buffer;

  while (*cur_src != '\0')
    {
      if (*cur_src != '%')
	{
	  if (*cur_src == '+')
	    *cur_dst++ = ' ';
	  else
	    *cur_dst++ = *cur_src;
	  cur_src++;
	}
      else
	{
	  unsigned int code;
	  int result = sscanf(cur_src, "%%%02x", &code);
	  if (result == 1)
	    {
	      *cur_dst++ = (char)code;
	      cur_src += 3;
	    }
	  else
	    *cur_dst++ = *cur_src++;
	}
    }
  *cur_dst = '\0';
  return buffer;
}


/*
 * ochusha_utils_get_utc_time
 *
 * Ϳ줿ʸɽʸȤƲᤷUTC֤
 */
time_t
ochusha_utils_get_utc_time(const char *date_string)
{
  static char *lc_time = NULL;
  struct tm tm;
  char *parse_result;

  if (lc_time == NULL)
    {
#if HAVE_LOCALE_H
      lc_time = setlocale(LC_TIME, "C");
      tzset();
      g_return_val_if_fail(lc_time != NULL, -1);
#else
      return -1;
#endif
    }

  g_return_val_if_fail(date_string != NULL && *date_string != '\0', -1);

  memset(&tm, 0, sizeof(struct tm));
#if 0
  /*
   * GMTΤ褦timezoneλꤵ줿date_string顢
   * struct tmǤĤäΤ
   * LinuxǤtimezone̵꤬뤵Ƥ͡
   * SolarisǤ%Z"GMT"ȥ顼ˤʤäƤޤ
   */
  parse_result = strptime(date_string, "%a, %d %b %Y %T %Z", &tm);
  g_return_val_if_fail(parse_result != NULL, -1);

  return mktime(&tm);
#else
  /*
   * ȡdate_string"GMT"ʳ褿˻郎
   * 2chΥФ˴ؤ¤Linux/SolarisǤԤ褦
   * 郎֤äƤΤǡ괺Ƥ
   */
  parse_result = strptime(date_string, "%a, %d %b %Y %T", &tm);
  g_return_val_if_fail(parse_result != NULL, -1);

# ifdef HAVE_TIMEGM
  return timegm(&tm);
# else /* Ķad-hoc塼 */
  return mktime(&tm) + (strstr(date_string, "GMT") != NULL ? 32400 : 0);
# endif
#endif
}


/*
 * Ĺǽstrpbrkؿ
 * stringĹlenʸcharset˴ޤޤǽʸؤΥݥ󥿤֤
 * ʤNULL֤
 */
const char *
mempbrk(const char *string, const char *charset, size_t len)
{
  while (len > 0)
    {
      const char *c = charset;
      char s = *string;
      while (*c)
	{
	  if (*c == s)
	    return string;
	  c++;
	}
      string++;
      len--;
    }
  return NULL;
}


const char *
reverse_strpbrk(const char *string, const char *charset)
{
  const gchar *s = string;
  while (*s != '\0')
    {
      if (strchr(charset, *s) == NULL)
	return s;
      s++;
    }

  return NULL;
}


int
parse_int(const gchar *start, size_t len)
{
  int result = 0;
  const gchar *tmp_pos = start;
  const gchar *tail_pos = (len > 0 ? start + len : NULL);
  gunichar c;

  do
    {
      c = g_utf8_get_char(tmp_pos);
      if (c >= '0' && c <= '9')
	result = result * 10 + c - '0';
      else if (c >= 65296 && c <= 65305)
	result = result * 10 + c - 65296;
      else
	break;
      tmp_pos = g_utf8_find_next_char(tmp_pos, tail_pos);
    } while (tmp_pos != NULL);

  return result;
}


char *
ochusha_utils_find_cgi_query_key(const char *src, CGIQueryKeyTuple *result)
{
  const char *name_pos = src;
  char *tmp_pos;

  if (src == NULL)
    return NULL;

  while ((tmp_pos = strpbrk(name_pos, "&=")) != NULL)
    {
      char *term;
      if (*tmp_pos == '&')
	{
	  name_pos = tmp_pos + 1;
	  continue;
	}

      if (result != NULL)
	{
	  result->name = name_pos;
	  result->name_len = tmp_pos - name_pos;
	}

      term = strchr(tmp_pos + 1, '&');
      if (term == NULL)
	term = tmp_pos + strlen(tmp_pos);
      /* λǡ*value_tail'&''\0' */

      if (result != NULL)
	{
	  result->value = tmp_pos + 1;
	  result->value_len = term - tmp_pos - 1;
	}

      return term;
    }

  return NULL;
}


/*
 * mkdir_p
 *
 * mkdir -pޥɤΤ褦ˡpathǻꤵ줿ǥ쥯ȥŪ
 * 롣
 * ǥ쥯ȥ꤬¸ߤ뤫ǥ쥯ȥκˤ1
 * ʳξ0֤
 */
int
mkdir_p(const char *path)
{
  char buffer[PATH_MAX];
  char *cur_pos;

  if (path == NULL)
    return 0;

  strncpy(buffer, path, PATH_MAX);

  cur_pos = buffer;
  while (1)
    {
      int result;
      struct stat sb;
      char tmp_char;

      cur_pos = strchr(cur_pos, '/');
      if (cur_pos != NULL)
	{
	  tmp_char = *(++cur_pos);
	  *cur_pos = '\0';
	}
      else
	tmp_char = '\0';

      result = stat(buffer, &sb);

      if (result == -1)
	{
	  if (errno != ENOENT)
	    return 0;
	
	  result = mkdir(buffer, 0x1c0);
	  if (result != 0)
	    return 0;
	}
      else
	if (!S_ISDIR(sb.st_mode))
	  return 0;

      if (tmp_char == '\0')
	return 1;

      *cur_pos = tmp_char;
    }
}


/*
 * iconvϢ
 */
/* 0x87, 0x40-0x9c */
static const char *cp932_to_utf8_table_0x87[] = {
  /*====== 0x87, 0x40-0x7e ======*/
  /* 0x87 0x40 */  "\342\221\240",
  /* 0x87 0x41 */  "\342\221\241",
  /* 0x87 0x42 */  "\342\221\242",
  /* 0x87 0x43 */  "\342\221\243",
  /* 0x87 0x44 */  "\342\221\244",
  /* 0x87 0x45 */  "\342\221\245",
  /* 0x87 0x46 */  "\342\221\246",
  /* 0x87 0x47 */  "\342\221\247",
  /* 0x87 0x48 */  "\342\221\250",
  /* 0x87 0x49 */  "\342\221\251",
  /* 0x87 0x4a */  "\342\221\252",
  /* 0x87 0x4b */  "\342\221\253",
  /* 0x87 0x4c */  "\342\221\254",
  /* 0x87 0x4d */  "\342\221\255",
  /* 0x87 0x4e */  "\342\221\256",
  /* 0x87 0x4f */  "\342\221\257",
  /* 0x87 0x50 */  "\342\221\260",
  /* 0x87 0x51 */  "\342\221\261",
  /* 0x87 0x52 */  "\342\221\262",
  /* 0x87 0x53 */  "\342\221\263",
  /* 0x87 0x54 */  "\342\205\240",
  /* 0x87 0x55 */  "\342\205\241",
  /* 0x87 0x56 */  "\342\205\242",
  /* 0x87 0x57 */  "\342\205\243",
  /* 0x87 0x58 */  "\342\205\244",
  /* 0x87 0x59 */  "\342\205\245",
  /* 0x87 0x5a */  "\342\205\246",
  /* 0x87 0x5b */  "\342\205\247",
  /* 0x87 0x5c */  "\342\205\250",
  /* 0x87 0x5d */  "\342\205\251",
  /* 0x87 0x5e (invalid) */  NULL,
  /* 0x87 0x5f */  "\343\215\211",
  /* 0x87 0x60 */  "\343\214\224",
  /* 0x87 0x61 */  "\343\214\242",
  /* 0x87 0x62 */  "\343\215\215",
  /* 0x87 0x63 */  "\343\214\230",
  /* 0x87 0x64 */  "\343\214\247",
  /* 0x87 0x65 */  "\343\214\203",
  /* 0x87 0x66 */  "\343\214\266",
  /* 0x87 0x67 */  "\343\215\221",
  /* 0x87 0x68 */  "\343\215\227",
  /* 0x87 0x69 */  "\343\214\215",
  /* 0x87 0x6a */  "\343\214\246",
  /* 0x87 0x6b */  "\343\214\243",
  /* 0x87 0x6c */  "\343\214\253",
  /* 0x87 0x6d */  "\343\215\212",
  /* 0x87 0x6e */  "\343\214\273",
  /* 0x87 0x6f */  "\343\216\234",
  /* 0x87 0x70 */  "\343\216\235",
  /* 0x87 0x71 */  "\343\216\236",
  /* 0x87 0x72 */  "\343\216\216",
  /* 0x87 0x73 */  "\343\216\217",
  /* 0x87 0x74 */  "\343\217\204",
  /* 0x87 0x75 */  "\343\216\241",
  /* 0x87 0x76 (invalid) */  NULL,
  /* 0x87 0x77 (invalid) */  NULL,
  /* 0x87 0x78 (invalid) */  NULL,
  /* 0x87 0x79 (invalid) */  NULL,
  /* 0x87 0x7a (invalid) */  NULL,
  /* 0x87 0x7b (invalid) */  NULL,
  /* 0x87 0x7c (invalid) */  NULL,
  /* 0x87 0x7d (invalid) */  NULL,
  /* 0x87 0x7e */  "\343\215\273",
  /* 0x87 0x7f (invalid) */  NULL,

  /*====== 0x87, 0x80-0x9e ======*/
  /* 0x87 0x80 */  "\343\200\235",
  /* 0x87 0x81 */  "\343\200\237",
  /* 0x87 0x82 */  "\342\204\226",
  /* 0x87 0x83 */  "\343\217\215",
  /* 0x87 0x84 */  "\342\204\241",
  /* 0x87 0x85 */  "\343\212\244",
  /* 0x87 0x86 */  "\343\212\245",
  /* 0x87 0x87 */  "\343\212\246",
  /* 0x87 0x88 */  "\343\212\247",
  /* 0x87 0x89 */  "\343\212\250",
  /* 0x87 0x8a */  "\343\210\261",
  /* 0x87 0x8b */  "\343\210\262",
  /* 0x87 0x8c */  "\343\210\271",
  /* 0x87 0x8d */  "\343\215\276",
  /* 0x87 0x8e */  "\343\215\275",
  /* 0x87 0x8f */  "\343\215\274",
  /* 0x87 0x90 */  "\342\211\222",
  /* 0x87 0x91 */  "\342\211\241",
  /* 0x87 0x92 */  "\342\210\253",
  /* 0x87 0x93 */  "\342\210\256",
  /* 0x87 0x94 */  "\342\210\221",
  /* 0x87 0x95 */  "\342\210\232",
  /* 0x87 0x96 */  "\342\212\245",
  /* 0x87 0x97 */  "\342\210\240",
  /* 0x87 0x98 */  "\342\210\237",
  /* 0x87 0x99 */  "\342\212\277",
  /* 0x87 0x9a */  "\342\210\265",
  /* 0x87 0x9b */  "\342\210\251",
  /* 0x87 0x9c */  "\342\210\252",
  /* 0x87 0x9d (invalid) */  NULL,
  /* 0x87 0x9e (invalid) */  NULL,
};


/* 0xed, 0x40-0xfc */
static const char *cp932_to_utf8_table_0xed[] = {
  /*====== 0xed, 0x40-0x7e ======*/
  /* 0xed 0x40 */  "\347\272\212",
  /* 0xed 0x41 */  "\350\244\234",
  /* 0xed 0x42 */  "\351\215\210",
  /* 0xed 0x43 */  "\351\212\210",
  /* 0xed 0x44 */  "\350\223\234",
  /* 0xed 0x45 */  "\344\277\211",
  /* 0xed 0x46 */  "\347\202\273",
  /* 0xed 0x47 */  "\346\230\261",
  /* 0xed 0x48 */  "\346\243\210",
  /* 0xed 0x49 */  "\351\213\271",
  /* 0xed 0x4a */  "\346\233\273",
  /* 0xed 0x4b */  "\345\275\205",
  /* 0xed 0x4c */  "\344\270\250",
  /* 0xed 0x4d */  "\344\273\241",
  /* 0xed 0x4e */  "\344\273\274",
  /* 0xed 0x4f */  "\344\274\200",
  /* 0xed 0x50 */  "\344\274\203",
  /* 0xed 0x51 */  "\344\274\271",
  /* 0xed 0x52 */  "\344\275\226",
  /* 0xed 0x53 */  "\344\276\222",
  /* 0xed 0x54 */  "\344\276\212",
  /* 0xed 0x55 */  "\344\276\232",
  /* 0xed 0x56 */  "\344\276\224",
  /* 0xed 0x57 */  "\344\277\215",
  /* 0xed 0x58 */  "\345\201\200",
  /* 0xed 0x59 */  "\345\200\242",
  /* 0xed 0x5a */  "\344\277\277",
  /* 0xed 0x5b */  "\345\200\236",
  /* 0xed 0x5c */  "\345\201\206",
  /* 0xed 0x5d */  "\345\201\260",
  /* 0xed 0x5e */  "\345\201\202",
  /* 0xed 0x5f */  "\345\202\224",
  /* 0xed 0x60 */  "\345\203\264",
  /* 0xed 0x61 */  "\345\203\230",
  /* 0xed 0x62 */  "\345\205\212",
  /* 0xed 0x63 */  "\345\205\244",
  /* 0xed 0x64 */  "\345\206\235",
  /* 0xed 0x65 */  "\345\206\276",
  /* 0xed 0x66 */  "\345\207\254",
  /* 0xed 0x67 */  "\345\210\225",
  /* 0xed 0x68 */  "\345\212\234",
  /* 0xed 0x69 */  "\345\212\246",
  /* 0xed 0x6a */  "\345\213\200",
  /* 0xed 0x6b */  "\345\213\233",
  /* 0xed 0x6c */  "\345\214\200",
  /* 0xed 0x6d */  "\345\214\207",
  /* 0xed 0x6e */  "\345\214\244",
  /* 0xed 0x6f */  "\345\215\262",
  /* 0xed 0x70 */  "\345\216\223",
  /* 0xed 0x71 */  "\345\216\262",
  /* 0xed 0x72 */  "\345\217\235",
  /* 0xed 0x73 */  "\357\250\216",
  /* 0xed 0x74 */  "\345\222\234",
  /* 0xed 0x75 */  "\345\222\212",
  /* 0xed 0x76 */  "\345\222\251",
  /* 0xed 0x77 */  "\345\223\277",
  /* 0xed 0x78 */  "\345\226\206",
  /* 0xed 0x79 */  "\345\235\231",
  /* 0xed 0x7a */  "\345\235\245",
  /* 0xed 0x7b */  "\345\236\254",
  /* 0xed 0x7c */  "\345\237\210",
  /* 0xed 0x7d */  "\345\237\207",
  /* 0xed 0x7e */  "\357\250\217",
  /* 0xed 0x7f (invalid) */  NULL,

  /*====== 0xed, 0x80-0x9e ======*/
  /* 0xed 0x80 */  "\357\250\220",
  /* 0xed 0x81 */  "\345\242\236",
  /* 0xed 0x82 */  "\345\242\262",
  /* 0xed 0x83 */  "\345\244\213",
  /* 0xed 0x84 */  "\345\245\223",
  /* 0xed 0x85 */  "\345\245\233",
  /* 0xed 0x86 */  "\345\245\235",
  /* 0xed 0x87 */  "\345\245\243",
  /* 0xed 0x88 */  "\345\246\244",
  /* 0xed 0x89 */  "\345\246\272",
  /* 0xed 0x8a */  "\345\255\226",
  /* 0xed 0x8b */  "\345\257\200",
  /* 0xed 0x8c */  "\347\224\257",
  /* 0xed 0x8d */  "\345\257\230",
  /* 0xed 0x8e */  "\345\257\254",
  /* 0xed 0x8f */  "\345\260\236",
  /* 0xed 0x90 */  "\345\262\246",
  /* 0xed 0x91 */  "\345\262\272",
  /* 0xed 0x92 */  "\345\263\265",
  /* 0xed 0x93 */  "\345\264\247",
  /* 0xed 0x94 */  "\345\265\223",
  /* 0xed 0x95 */  "\357\250\221",
  /* 0xed 0x96 */  "\345\265\202",
  /* 0xed 0x97 */  "\345\265\255",
  /* 0xed 0x98 */  "\345\266\270",
  /* 0xed 0x99 */  "\345\266\271",
  /* 0xed 0x9a */  "\345\267\220",
  /* 0xed 0x9b */  "\345\274\241",
  /* 0xed 0x9c */  "\345\274\264",
  /* 0xed 0x9d */  "\345\275\247",
  /* 0xed 0x9e */  "\345\276\267",
  /* 0xed 0x9f (invalid) */  NULL,

  /*====== 0xed, 0x9f-0xfc ======*/
  /* 0xed 0x9f */  "\345\277\236",
  /* 0xed 0xa0 */  "\346\201\235",
  /* 0xed 0xa1 */  "\346\202\205",
  /* 0xed 0xa2 */  "\346\202\212",
  /* 0xed 0xa3 */  "\346\203\236",
  /* 0xed 0xa4 */  "\346\203\225",
  /* 0xed 0xa5 */  "\346\204\240",
  /* 0xed 0xa6 */  "\346\203\262",
  /* 0xed 0xa7 */  "\346\204\221",
  /* 0xed 0xa8 */  "\346\204\267",
  /* 0xed 0xa9 */  "\346\204\260",
  /* 0xed 0xaa */  "\346\206\230",
  /* 0xed 0xab */  "\346\210\223",
  /* 0xed 0xac */  "\346\212\246",
  /* 0xed 0xad */  "\346\217\265",
  /* 0xed 0xae */  "\346\221\240",
  /* 0xed 0xaf */  "\346\222\235",
  /* 0xed 0xb0 */  "\346\223\216",
  /* 0xed 0xb1 */  "\346\225\216",
  /* 0xed 0xb2 */  "\346\230\200",
  /* 0xed 0xb3 */  "\346\230\225",
  /* 0xed 0xb4 */  "\346\230\273",
  /* 0xed 0xb5 */  "\346\230\211",
  /* 0xed 0xb6 */  "\346\230\256",
  /* 0xed 0xb7 */  "\346\230\236",
  /* 0xed 0xb8 */  "\346\230\244",
  /* 0xed 0xb9 */  "\346\231\245",
  /* 0xed 0xba */  "\346\231\227",
  /* 0xed 0xbb */  "\346\231\231",
  /* 0xed 0xbc */  "\357\250\222",
  /* 0xed 0xbd */  "\346\231\263",
  /* 0xed 0xbe */  "\346\232\231",
  /* 0xed 0xbf */  "\346\232\240",
  /* 0xed 0xc0 */  "\346\232\262",
  /* 0xed 0xc1 */  "\346\232\277",
  /* 0xed 0xc2 */  "\346\233\272",
  /* 0xed 0xc3 */  "\346\234\216",
  /* 0xed 0xc4 */  "\357\244\251",
  /* 0xed 0xc5 */  "\346\235\246",
  /* 0xed 0xc6 */  "\346\236\273",
  /* 0xed 0xc7 */  "\346\241\222",
  /* 0xed 0xc8 */  "\346\237\200",
  /* 0xed 0xc9 */  "\346\240\201",
  /* 0xed 0xca */  "\346\241\204",
  /* 0xed 0xcb */  "\346\243\217",
  /* 0xed 0xcc */  "\357\250\223",
  /* 0xed 0xcd */  "\346\245\250",
  /* 0xed 0xce */  "\357\250\224",
  /* 0xed 0xcf */  "\346\246\230",
  /* 0xed 0xd0 */  "\346\247\242",
  /* 0xed 0xd1 */  "\346\250\260",
  /* 0xed 0xd2 */  "\346\251\253",
  /* 0xed 0xd3 */  "\346\251\206",
  /* 0xed 0xd4 */  "\346\251\263",
  /* 0xed 0xd5 */  "\346\251\276",
  /* 0xed 0xd6 */  "\346\253\242",
  /* 0xed 0xd7 */  "\346\253\244",
  /* 0xed 0xd8 */  "\346\257\226",
  /* 0xed 0xd9 */  "\346\260\277",
  /* 0xed 0xda */  "\346\261\234",
  /* 0xed 0xdb */  "\346\262\206",
  /* 0xed 0xdc */  "\346\261\257",
  /* 0xed 0xdd */  "\346\263\232",
  /* 0xed 0xde */  "\346\264\204",
  /* 0xed 0xdf */  "\346\266\207",
  /* 0xed 0xe0 */  "\346\265\257",
  /* 0xed 0xe1 */  "\346\266\226",
  /* 0xed 0xe2 */  "\346\266\254",
  /* 0xed 0xe3 */  "\346\267\217",
  /* 0xed 0xe4 */  "\346\267\270",
  /* 0xed 0xe5 */  "\346\267\262",
  /* 0xed 0xe6 */  "\346\267\274",
  /* 0xed 0xe7 */  "\346\270\271",
  /* 0xed 0xe8 */  "\346\271\234",
  /* 0xed 0xe9 */  "\346\270\247",
  /* 0xed 0xea */  "\346\270\274",
  /* 0xed 0xeb */  "\346\272\277",
  /* 0xed 0xec */  "\346\276\210",
  /* 0xed 0xed */  "\346\276\265",
  /* 0xed 0xee */  "\346\277\265",
  /* 0xed 0xef */  "\347\200\205",
  /* 0xed 0xf0 */  "\347\200\207",
  /* 0xed 0xf1 */  "\347\200\250",
  /* 0xed 0xf2 */  "\347\202\205",
  /* 0xed 0xf3 */  "\347\202\253",
  /* 0xed 0xf4 */  "\347\204\217",
  /* 0xed 0xf5 */  "\347\204\204",
  /* 0xed 0xf6 */  "\347\205\234",
  /* 0xed 0xf7 */  "\347\205\206",
  /* 0xed 0xf8 */  "\347\205\207",
  /* 0xed 0xf9 */  "\357\250\225",
  /* 0xed 0xfa */  "\347\207\201",
  /* 0xed 0xfb */  "\347\207\276",
  /* 0xed 0xfc */  "\347\212\261",
};


/* 0xee, 0x40-0xfc */
static const char *cp932_to_utf8_table_0xee[] = {
  /*====== 0xee, 0x40-0x7e ======*/
  /* 0xee 0x40 */  "\347\212\276",
  /* 0xee 0x41 */  "\347\214\244",
  /* 0xee 0x42 */  "\357\250\226",
  /* 0xee 0x43 */  "\347\215\267",
  /* 0xee 0x44 */  "\347\216\275",
  /* 0xee 0x45 */  "\347\217\211",
  /* 0xee 0x46 */  "\347\217\226",
  /* 0xee 0x47 */  "\347\217\243",
  /* 0xee 0x48 */  "\347\217\222",
  /* 0xee 0x49 */  "\347\220\207",
  /* 0xee 0x4a */  "\347\217\265",
  /* 0xee 0x4b */  "\347\220\246",
  /* 0xee 0x4c */  "\347\220\252",
  /* 0xee 0x4d */  "\347\220\251",
  /* 0xee 0x4e */  "\347\220\256",
  /* 0xee 0x4f */  "\347\221\242",
  /* 0xee 0x50 */  "\347\222\211",
  /* 0xee 0x51 */  "\347\222\237",
  /* 0xee 0x52 */  "\347\224\201",
  /* 0xee 0x53 */  "\347\225\257",
  /* 0xee 0x54 */  "\347\232\202",
  /* 0xee 0x55 */  "\347\232\234",
  /* 0xee 0x56 */  "\347\232\236",
  /* 0xee 0x57 */  "\347\232\233",
  /* 0xee 0x58 */  "\347\232\246",
  /* 0xee 0x59 */  "\357\250\227",
  /* 0xee 0x5a */  "\347\235\206",
  /* 0xee 0x5b */  "\345\212\257",
  /* 0xee 0x5c */  "\347\240\241",
  /* 0xee 0x5d */  "\347\241\216",
  /* 0xee 0x5e */  "\347\241\244",
  /* 0xee 0x5f */  "\347\241\272",
  /* 0xee 0x60 */  "\347\244\260",
  /* 0xee 0x61 */  "\357\250\230",
  /* 0xee 0x62 */  "\357\250\231",
  /* 0xee 0x63 */  "\357\250\232",
  /* 0xee 0x64 */  "\347\246\224",
  /* 0xee 0x65 */  "\357\250\233",
  /* 0xee 0x66 */  "\347\246\233",
  /* 0xee 0x67 */  "\347\253\221",
  /* 0xee 0x68 */  "\347\253\247",
  /* 0xee 0x69 */  "\357\250\234",
  /* 0xee 0x6a */  "\347\253\253",
  /* 0xee 0x6b */  "\347\256\236",
  /* 0xee 0x6c */  "\357\250\235",
  /* 0xee 0x6d */  "\347\265\210",
  /* 0xee 0x6e */  "\347\265\234",
  /* 0xee 0x6f */  "\347\266\267",
  /* 0xee 0x70 */  "\347\266\240",
  /* 0xee 0x71 */  "\347\267\226",
  /* 0xee 0x72 */  "\347\271\222",
  /* 0xee 0x73 */  "\347\275\207",
  /* 0xee 0x74 */  "\347\276\241",
  /* 0xee 0x75 */  "\357\250\236",
  /* 0xee 0x76 */  "\350\214\201",
  /* 0xee 0x77 */  "\350\215\242",
  /* 0xee 0x78 */  "\350\215\277",
  /* 0xee 0x79 */  "\350\217\207",
  /* 0xee 0x7a */  "\350\217\266",
  /* 0xee 0x7b */  "\350\221\210",
  /* 0xee 0x7c */  "\350\222\264",
  /* 0xee 0x7d */  "\350\225\223",
  /* 0xee 0x7e */  "\350\225\231",
  /* 0xee 0x7f (invalid) */  NULL,

  /*====== 0xee, 0x80-0x9e ======*/
  /* 0xee 0x80 */  "\350\225\253",
  /* 0xee 0x81 */  "\357\250\237",
  /* 0xee 0x82 */  "\350\226\260",
  /* 0xee 0x83 */  "\357\250\240",
  /* 0xee 0x84 */  "\357\250\241",
  /* 0xee 0x85 */  "\350\240\207",
  /* 0xee 0x86 */  "\350\243\265",
  /* 0xee 0x87 */  "\350\250\222",
  /* 0xee 0x88 */  "\350\250\267",
  /* 0xee 0x89 */  "\350\251\271",
  /* 0xee 0x8a */  "\350\252\247",
  /* 0xee 0x8b */  "\350\252\276",
  /* 0xee 0x8c */  "\350\253\237",
  /* 0xee 0x8d */  "\357\250\242",
  /* 0xee 0x8e */  "\350\253\266",
  /* 0xee 0x8f */  "\350\255\223",
  /* 0xee 0x90 */  "\350\255\277",
  /* 0xee 0x91 */  "\350\263\260",
  /* 0xee 0x92 */  "\350\263\264",
  /* 0xee 0x93 */  "\350\264\222",
  /* 0xee 0x94 */  "\350\265\266",
  /* 0xee 0x95 */  "\357\250\243",
  /* 0xee 0x96 */  "\350\273\217",
  /* 0xee 0x97 */  "\357\250\244",
  /* 0xee 0x98 */  "\357\250\245",
  /* 0xee 0x99 */  "\351\201\247",
  /* 0xee 0x9a */  "\351\203\236",
  /* 0xee 0x9b */  "\357\250\246",
  /* 0xee 0x9c */  "\351\204\225",
  /* 0xee 0x9d */  "\351\204\247",
  /* 0xee 0x9e */  "\351\207\232",

  /*====== 0xee, 0x9f-0xfc ======*/
  /* 0xee 0x9f */  "\351\207\227",
  /* 0xee 0xa0 */  "\351\207\236",
  /* 0xee 0xa1 */  "\351\207\255",
  /* 0xee 0xa2 */  "\351\207\256",
  /* 0xee 0xa3 */  "\351\207\244",
  /* 0xee 0xa4 */  "\351\207\245",
  /* 0xee 0xa5 */  "\351\210\206",
  /* 0xee 0xa6 */  "\351\210\220",
  /* 0xee 0xa7 */  "\351\210\212",
  /* 0xee 0xa8 */  "\351\210\272",
  /* 0xee 0xa9 */  "\351\211\200",
  /* 0xee 0xaa */  "\351\210\274",
  /* 0xee 0xab */  "\351\211\216",
  /* 0xee 0xac */  "\351\211\231",
  /* 0xee 0xad */  "\351\211\221",
  /* 0xee 0xae */  "\351\210\271",
  /* 0xee 0xaf */  "\351\211\247",
  /* 0xee 0xb0 */  "\351\212\247",
  /* 0xee 0xb1 */  "\351\211\267",
  /* 0xee 0xb2 */  "\351\211\270",
  /* 0xee 0xb3 */  "\351\213\247",
  /* 0xee 0xb4 */  "\351\213\227",
  /* 0xee 0xb5 */  "\351\213\231",
  /* 0xee 0xb6 */  "\351\213\220",
  /* 0xee 0xb7 */  "\357\250\247",
  /* 0xee 0xb8 */  "\351\213\225",
  /* 0xee 0xb9 */  "\351\213\240",
  /* 0xee 0xba */  "\351\213\223",
  /* 0xee 0xbb */  "\351\214\245",
  /* 0xee 0xbc */  "\351\214\241",
  /* 0xee 0xbd */  "\351\213\273",
  /* 0xee 0xbe */  "\357\250\250",
  /* 0xee 0xbf */  "\351\214\236",
  /* 0xee 0xc0 */  "\351\213\277",
  /* 0xee 0xc1 */  "\351\214\235",
  /* 0xee 0xc2 */  "\351\214\202",
  /* 0xee 0xc3 */  "\351\215\260",
  /* 0xee 0xc4 */  "\351\215\227",
  /* 0xee 0xc5 */  "\351\216\244",
  /* 0xee 0xc6 */  "\351\217\206",
  /* 0xee 0xc7 */  "\351\217\236",
  /* 0xee 0xc8 */  "\351\217\270",
  /* 0xee 0xc9 */  "\351\220\261",
  /* 0xee 0xca */  "\351\221\205",
  /* 0xee 0xcb */  "\351\221\210",
  /* 0xee 0xcc */  "\351\226\222",
  /* 0xee 0xcd */  "\357\247\234",
  /* 0xee 0xce */  "\357\250\251",
  /* 0xee 0xcf */  "\351\232\235",
  /* 0xee 0xd0 */  "\351\232\257",
  /* 0xee 0xd1 */  "\351\234\263",
  /* 0xee 0xd2 */  "\351\234\273",
  /* 0xee 0xd3 */  "\351\235\203",
  /* 0xee 0xd4 */  "\351\235\215",
  /* 0xee 0xd5 */  "\351\235\217",
  /* 0xee 0xd6 */  "\351\235\221",
  /* 0xee 0xd7 */  "\351\235\225",
  /* 0xee 0xd8 */  "\351\241\227",
  /* 0xee 0xd9 */  "\351\241\245",
  /* 0xee 0xda */  "\357\250\252",
  /* 0xee 0xdb */  "\357\250\253",
  /* 0xee 0xdc */  "\351\244\247",
  /* 0xee 0xdd */  "\357\250\254",
  /* 0xee 0xde */  "\351\246\236",
  /* 0xee 0xdf */  "\351\251\216",
  /* 0xee 0xe0 */  "\351\253\231",
  /* 0xee 0xe1 */  "\351\253\234",
  /* 0xee 0xe2 */  "\351\255\265",
  /* 0xee 0xe3 */  "\351\255\262",
  /* 0xee 0xe4 */  "\351\256\217",
  /* 0xee 0xe5 */  "\351\256\261",
  /* 0xee 0xe6 */  "\351\256\273",
  /* 0xee 0xe7 */  "\351\260\200",
  /* 0xee 0xe8 */  "\351\265\260",
  /* 0xee 0xe9 */  "\351\265\253",
  /* 0xee 0xea */  "\357\250\255",
  /* 0xee 0xeb */  "\351\270\231",
  /* 0xee 0xec */  "\351\273\221",
  /* 0xee 0xed (invalid) */  NULL,
  /* 0xee 0xee (invalid) */  NULL,
  /* 0xee 0xef */  "\342\205\260",
  /* 0xee 0xf0 */  "\342\205\261",
  /* 0xee 0xf1 */  "\342\205\262",
  /* 0xee 0xf2 */  "\342\205\263",
  /* 0xee 0xf3 */  "\342\205\264",
  /* 0xee 0xf4 */  "\342\205\265",
  /* 0xee 0xf5 */  "\342\205\266",
  /* 0xee 0xf6 */  "\342\205\267",
  /* 0xee 0xf7 */  "\342\205\270",
  /* 0xee 0xf8 */  "\342\205\271",
  /* 0xee 0xf9 */  "\357\277\242",
  /* 0xee 0xfa */  "\357\277\244",
  /* 0xee 0xfb */  "\357\274\207",
  /* 0xee 0xfc */  "\357\274\202",
};


/* 0xfa, 0x40-0xfc */
static const char *cp932_to_utf8_table_0xfa[] = {
  /*====== 0xfa, 0x40-0x7e ======*/
  /* 0xfa 0x40 */  "\342\205\260",
  /* 0xfa 0x41 */  "\342\205\261",
  /* 0xfa 0x42 */  "\342\205\262",
  /* 0xfa 0x43 */  "\342\205\263",
  /* 0xfa 0x44 */  "\342\205\264",
  /* 0xfa 0x45 */  "\342\205\265",
  /* 0xfa 0x46 */  "\342\205\266",
  /* 0xfa 0x47 */  "\342\205\267",
  /* 0xfa 0x48 */  "\342\205\270",
  /* 0xfa 0x49 */  "\342\205\271",
  /* 0xfa 0x4a */  "\342\205\240",
  /* 0xfa 0x4b */  "\342\205\241",
  /* 0xfa 0x4c */  "\342\205\242",
  /* 0xfa 0x4d */  "\342\205\243",
  /* 0xfa 0x4e */  "\342\205\244",
  /* 0xfa 0x4f */  "\342\205\245",
  /* 0xfa 0x50 */  "\342\205\246",
  /* 0xfa 0x51 */  "\342\205\247",
  /* 0xfa 0x52 */  "\342\205\250",
  /* 0xfa 0x53 */  "\342\205\251",
  /* 0xfa 0x54 */  "\357\277\242",
  /* 0xfa 0x55 */  "\357\277\244",
  /* 0xfa 0x56 */  "\357\274\207",
  /* 0xfa 0x57 */  "\357\274\202",
  /* 0xfa 0x58 */  "\343\210\261",
  /* 0xfa 0x59 */  "\342\204\226",
  /* 0xfa 0x5a */  "\342\204\241",
  /* 0xfa 0x5b */  "\342\210\265",
  /* 0xfa 0x5c */  "\347\272\212",
  /* 0xfa 0x5d */  "\350\244\234",
  /* 0xfa 0x5e */  "\351\215\210",
  /* 0xfa 0x5f */  "\351\212\210",
  /* 0xfa 0x60 */  "\350\223\234",
  /* 0xfa 0x61 */  "\344\277\211",
  /* 0xfa 0x62 */  "\347\202\273",
  /* 0xfa 0x63 */  "\346\230\261",
  /* 0xfa 0x64 */  "\346\243\210",
  /* 0xfa 0x65 */  "\351\213\271",
  /* 0xfa 0x66 */  "\346\233\273",
  /* 0xfa 0x67 */  "\345\275\205",
  /* 0xfa 0x68 */  "\344\270\250",
  /* 0xfa 0x69 */  "\344\273\241",
  /* 0xfa 0x6a */  "\344\273\274",
  /* 0xfa 0x6b */  "\344\274\200",
  /* 0xfa 0x6c */  "\344\274\203",
  /* 0xfa 0x6d */  "\344\274\271",
  /* 0xfa 0x6e */  "\344\275\226",
  /* 0xfa 0x6f */  "\344\276\222",
  /* 0xfa 0x70 */  "\344\276\212",
  /* 0xfa 0x71 */  "\344\276\232",
  /* 0xfa 0x72 */  "\344\276\224",
  /* 0xfa 0x73 */  "\344\277\215",
  /* 0xfa 0x74 */  "\345\201\200",
  /* 0xfa 0x75 */  "\345\200\242",
  /* 0xfa 0x76 */  "\344\277\277",
  /* 0xfa 0x77 */  "\345\200\236",
  /* 0xfa 0x78 */  "\345\201\206",
  /* 0xfa 0x79 */  "\345\201\260",
  /* 0xfa 0x7a */  "\345\201\202",
  /* 0xfa 0x7b */  "\345\202\224",
  /* 0xfa 0x7c */  "\345\203\264",
  /* 0xfa 0x7d */  "\345\203\230",
  /* 0xfa 0x7e */  "\345\205\212",
  /* 0xfa 0x7f (invalid) */  NULL,

  /*====== 0xfa, 0x80-0xfc ======*/
  /* 0xfa 0x80 */  "\345\205\244",
  /* 0xfa 0x81 */  "\345\206\235",
  /* 0xfa 0x82 */  "\345\206\276",
  /* 0xfa 0x83 */  "\345\207\254",
  /* 0xfa 0x84 */  "\345\210\225",
  /* 0xfa 0x85 */  "\345\212\234",
  /* 0xfa 0x86 */  "\345\212\246",
  /* 0xfa 0x87 */  "\345\213\200",
  /* 0xfa 0x88 */  "\345\213\233",
  /* 0xfa 0x89 */  "\345\214\200",
  /* 0xfa 0x8a */  "\345\214\207",
  /* 0xfa 0x8b */  "\345\214\244",
  /* 0xfa 0x8c */  "\345\215\262",
  /* 0xfa 0x8d */  "\345\216\223",
  /* 0xfa 0x8e */  "\345\216\262",
  /* 0xfa 0x8f */  "\345\217\235",
  /* 0xfa 0x90 */  "\357\250\216",
  /* 0xfa 0x91 */  "\345\222\234",
  /* 0xfa 0x92 */  "\345\222\212",
  /* 0xfa 0x93 */  "\345\222\251",
  /* 0xfa 0x94 */  "\345\223\277",
  /* 0xfa 0x95 */  "\345\226\206",
  /* 0xfa 0x96 */  "\345\235\231",
  /* 0xfa 0x97 */  "\345\235\245",
  /* 0xfa 0x98 */  "\345\236\254",
  /* 0xfa 0x99 */  "\345\237\210",
  /* 0xfa 0x9a */  "\345\237\207",
  /* 0xfa 0x9b */  "\357\250\217",
  /* 0xfa 0x9c */  "\357\250\220",
  /* 0xfa 0x9d */  "\345\242\236",
  /* 0xfa 0x9e */  "\345\242\262",
  /* 0xfa 0x9f */  "\345\244\213",
  /* 0xfa 0xa0 */  "\345\245\223",
  /* 0xfa 0xa1 */  "\345\245\233",
  /* 0xfa 0xa2 */  "\345\245\235",
  /* 0xfa 0xa3 */  "\345\245\243",
  /* 0xfa 0xa4 */  "\345\246\244",
  /* 0xfa 0xa5 */  "\345\246\272",
  /* 0xfa 0xa6 */  "\345\255\226",
  /* 0xfa 0xa7 */  "\345\257\200",
  /* 0xfa 0xa8 */  "\347\224\257",
  /* 0xfa 0xa9 */  "\345\257\230",
  /* 0xfa 0xaa */  "\345\257\254",
  /* 0xfa 0xab */  "\345\260\236",
  /* 0xfa 0xac */  "\345\262\246",
  /* 0xfa 0xad */  "\345\262\272",
  /* 0xfa 0xae */  "\345\263\265",
  /* 0xfa 0xaf */  "\345\264\247",
  /* 0xfa 0xb0 */  "\345\265\223",
  /* 0xfa 0xb1 */  "\357\250\221",
  /* 0xfa 0xb2 */  "\345\265\202",
  /* 0xfa 0xb3 */  "\345\265\255",
  /* 0xfa 0xb4 */  "\345\266\270",
  /* 0xfa 0xb5 */  "\345\266\271",
  /* 0xfa 0xb6 */  "\345\267\220",
  /* 0xfa 0xb7 */  "\345\274\241",
  /* 0xfa 0xb8 */  "\345\274\264",
  /* 0xfa 0xb9 */  "\345\275\247",
  /* 0xfa 0xba */  "\345\276\267",
  /* 0xfa 0xbb */  "\345\277\236",
  /* 0xfa 0xbc */  "\346\201\235",
  /* 0xfa 0xbd */  "\346\202\205",
  /* 0xfa 0xbe */  "\346\202\212",
  /* 0xfa 0xbf */  "\346\203\236",
  /* 0xfa 0xc0 */  "\346\203\225",
  /* 0xfa 0xc1 */  "\346\204\240",
  /* 0xfa 0xc2 */  "\346\203\262",
  /* 0xfa 0xc3 */  "\346\204\221",
  /* 0xfa 0xc4 */  "\346\204\267",
  /* 0xfa 0xc5 */  "\346\204\260",
  /* 0xfa 0xc6 */  "\346\206\230",
  /* 0xfa 0xc7 */  "\346\210\223",
  /* 0xfa 0xc8 */  "\346\212\246",
  /* 0xfa 0xc9 */  "\346\217\265",
  /* 0xfa 0xca */  "\346\221\240",
  /* 0xfa 0xcb */  "\346\222\235",
  /* 0xfa 0xcc */  "\346\223\216",
  /* 0xfa 0xcd */  "\346\225\216",
  /* 0xfa 0xce */  "\346\230\200",
  /* 0xfa 0xcf */  "\346\230\225",
  /* 0xfa 0xd0 */  "\346\230\273",
  /* 0xfa 0xd1 */  "\346\230\211",
  /* 0xfa 0xd2 */  "\346\230\256",
  /* 0xfa 0xd3 */  "\346\230\236",
  /* 0xfa 0xd4 */  "\346\230\244",
  /* 0xfa 0xd5 */  "\346\231\245",
  /* 0xfa 0xd6 */  "\346\231\227",
  /* 0xfa 0xd7 */  "\346\231\231",
  /* 0xfa 0xd8 */  "\357\250\222",
  /* 0xfa 0xd9 */  "\346\231\263",
  /* 0xfa 0xda */  "\346\232\231",
  /* 0xfa 0xdb */  "\346\232\240",
  /* 0xfa 0xdc */  "\346\232\262",
  /* 0xfa 0xdd */  "\346\232\277",
  /* 0xfa 0xde */  "\346\233\272",
  /* 0xfa 0xdf */  "\346\234\216",
  /* 0xfa 0xe0 */  "\357\244\251",
  /* 0xfa 0xe1 */  "\346\235\246",
  /* 0xfa 0xe2 */  "\346\236\273",
  /* 0xfa 0xe3 */  "\346\241\222",
  /* 0xfa 0xe4 */  "\346\237\200",
  /* 0xfa 0xe5 */  "\346\240\201",
  /* 0xfa 0xe6 */  "\346\241\204",
  /* 0xfa 0xe7 */  "\346\243\217",
  /* 0xfa 0xe8 */  "\357\250\223",
  /* 0xfa 0xe9 */  "\346\245\250",
  /* 0xfa 0xea */  "\357\250\224",
  /* 0xfa 0xeb */  "\346\246\230",
  /* 0xfa 0xec */  "\346\247\242",
  /* 0xfa 0xed */  "\346\250\260",
  /* 0xfa 0xee */  "\346\251\253",
  /* 0xfa 0xef */  "\346\251\206",
  /* 0xfa 0xf0 */  "\346\251\263",
  /* 0xfa 0xf1 */  "\346\251\276",
  /* 0xfa 0xf2 */  "\346\253\242",
  /* 0xfa 0xf3 */  "\346\253\244",
  /* 0xfa 0xf4 */  "\346\257\226",
  /* 0xfa 0xf5 */  "\346\260\277",
  /* 0xfa 0xf6 */  "\346\261\234",
  /* 0xfa 0xf7 */  "\346\262\206",
  /* 0xfa 0xf8 */  "\346\261\257",
  /* 0xfa 0xf9 */  "\346\263\232",
  /* 0xfa 0xfa */  "\346\264\204",
  /* 0xfa 0xfb */  "\346\266\207",
  /* 0xfa 0xfc */  "\346\265\257",
};


/* 0xfb, 0x40-0xfc */
static const char *cp932_to_utf8_table_0xfb[] = {
  /*====== 0xfb, 0x40-0x7e ======*/
  /* 0xfb 0x40 */  "\346\266\226",
  /* 0xfb 0x41 */  "\346\266\254",
  /* 0xfb 0x42 */  "\346\267\217",
  /* 0xfb 0x43 */  "\346\267\270",
  /* 0xfb 0x44 */  "\346\267\262",
  /* 0xfb 0x45 */  "\346\267\274",
  /* 0xfb 0x46 */  "\346\270\271",
  /* 0xfb 0x47 */  "\346\271\234",
  /* 0xfb 0x48 */  "\346\270\247",
  /* 0xfb 0x49 */  "\346\270\274",
  /* 0xfb 0x4a */  "\346\272\277",
  /* 0xfb 0x4b */  "\346\276\210",
  /* 0xfb 0x4c */  "\346\276\265",
  /* 0xfb 0x4d */  "\346\277\265",
  /* 0xfb 0x4e */  "\347\200\205",
  /* 0xfb 0x4f */  "\347\200\207",
  /* 0xfb 0x50 */  "\347\200\250",
  /* 0xfb 0x51 */  "\347\202\205",
  /* 0xfb 0x52 */  "\347\202\253",
  /* 0xfb 0x53 */  "\347\204\217",
  /* 0xfb 0x54 */  "\347\204\204",
  /* 0xfb 0x55 */  "\347\205\234",
  /* 0xfb 0x56 */  "\347\205\206",
  /* 0xfb 0x57 */  "\347\205\207",
  /* 0xfb 0x58 */  "\357\250\225",
  /* 0xfb 0x59 */  "\347\207\201",
  /* 0xfb 0x5a */  "\347\207\276",
  /* 0xfb 0x5b */  "\347\212\261",
  /* 0xfb 0x5c */  "\347\212\276",
  /* 0xfb 0x5d */  "\347\214\244",
  /* 0xfb 0x5e */  "\357\250\226",
  /* 0xfb 0x5f */  "\347\215\267",
  /* 0xfb 0x60 */  "\347\216\275",
  /* 0xfb 0x61 */  "\347\217\211",
  /* 0xfb 0x62 */  "\347\217\226",
  /* 0xfb 0x63 */  "\347\217\243",
  /* 0xfb 0x64 */  "\347\217\222",
  /* 0xfb 0x65 */  "\347\220\207",
  /* 0xfb 0x66 */  "\347\217\265",
  /* 0xfb 0x67 */  "\347\220\246",
  /* 0xfb 0x68 */  "\347\220\252",
  /* 0xfb 0x69 */  "\347\220\251",
  /* 0xfb 0x6a */  "\347\220\256",
  /* 0xfb 0x6b */  "\347\221\242",
  /* 0xfb 0x6c */  "\347\222\211",
  /* 0xfb 0x6d */  "\347\222\237",
  /* 0xfb 0x6e */  "\347\224\201",
  /* 0xfb 0x6f */  "\347\225\257",
  /* 0xfb 0x70 */  "\347\232\202",
  /* 0xfb 0x71 */  "\347\232\234",
  /* 0xfb 0x72 */  "\347\232\236",
  /* 0xfb 0x73 */  "\347\232\233",
  /* 0xfb 0x74 */  "\347\232\246",
  /* 0xfb 0x75 */  "\357\250\227",
  /* 0xfb 0x76 */  "\347\235\206",
  /* 0xfb 0x77 */  "\345\212\257",
  /* 0xfb 0x78 */  "\347\240\241",
  /* 0xfb 0x79 */  "\347\241\216",
  /* 0xfb 0x7a */  "\347\241\244",
  /* 0xfb 0x7b */  "\347\241\272",
  /* 0xfb 0x7c */  "\347\244\260",
  /* 0xfb 0x7d */  "\357\250\230",
  /* 0xfb 0x7e */  "\357\250\231",
  /* 0xfb 0x7f (invalid) */  NULL,

  /*====== 0xfb, 0x80-0xfc ======*/
  /* 0xfb 0x80 */  "\357\250\232",
  /* 0xfb 0x81 */  "\347\246\224",
  /* 0xfb 0x82 */  "\357\250\233",
  /* 0xfb 0x83 */  "\347\246\233",
  /* 0xfb 0x84 */  "\347\253\221",
  /* 0xfb 0x85 */  "\347\253\247",
  /* 0xfb 0x86 */  "\357\250\234",
  /* 0xfb 0x87 */  "\347\253\253",
  /* 0xfb 0x88 */  "\347\256\236",
  /* 0xfb 0x89 */  "\357\250\235",
  /* 0xfb 0x8a */  "\347\265\210",
  /* 0xfb 0x8b */  "\347\265\234",
  /* 0xfb 0x8c */  "\347\266\267",
  /* 0xfb 0x8d */  "\347\266\240",
  /* 0xfb 0x8e */  "\347\267\226",
  /* 0xfb 0x8f */  "\347\271\222",
  /* 0xfb 0x90 */  "\347\275\207",
  /* 0xfb 0x91 */  "\347\276\241",
  /* 0xfb 0x92 */  "\357\250\236",
  /* 0xfb 0x93 */  "\350\214\201",
  /* 0xfb 0x94 */  "\350\215\242",
  /* 0xfb 0x95 */  "\350\215\277",
  /* 0xfb 0x96 */  "\350\217\207",
  /* 0xfb 0x97 */  "\350\217\266",
  /* 0xfb 0x98 */  "\350\221\210",
  /* 0xfb 0x99 */  "\350\222\264",
  /* 0xfb 0x9a */  "\350\225\223",
  /* 0xfb 0x9b */  "\350\225\231",
  /* 0xfb 0x9c */  "\350\225\253",
  /* 0xfb 0x9d */  "\357\250\237",
  /* 0xfb 0x9e */  "\350\226\260",
  /* 0xfb 0x9f */  "\357\250\240",
  /* 0xfb 0xa0 */  "\357\250\241",
  /* 0xfb 0xa1 */  "\350\240\207",
  /* 0xfb 0xa2 */  "\350\243\265",
  /* 0xfb 0xa3 */  "\350\250\222",
  /* 0xfb 0xa4 */  "\350\250\267",
  /* 0xfb 0xa5 */  "\350\251\271",
  /* 0xfb 0xa6 */  "\350\252\247",
  /* 0xfb 0xa7 */  "\350\252\276",
  /* 0xfb 0xa8 */  "\350\253\237",
  /* 0xfb 0xa9 */  "\357\250\242",
  /* 0xfb 0xaa */  "\350\253\266",
  /* 0xfb 0xab */  "\350\255\223",
  /* 0xfb 0xac */  "\350\255\277",
  /* 0xfb 0xad */  "\350\263\260",
  /* 0xfb 0xae */  "\350\263\264",
  /* 0xfb 0xaf */  "\350\264\222",
  /* 0xfb 0xb0 */  "\350\265\266",
  /* 0xfb 0xb1 */  "\357\250\243",
  /* 0xfb 0xb2 */  "\350\273\217",
  /* 0xfb 0xb3 */  "\357\250\244",
  /* 0xfb 0xb4 */  "\357\250\245",
  /* 0xfb 0xb5 */  "\351\201\247",
  /* 0xfb 0xb6 */  "\351\203\236",
  /* 0xfb 0xb7 */  "\357\250\246",
  /* 0xfb 0xb8 */  "\351\204\225",
  /* 0xfb 0xb9 */  "\351\204\247",
  /* 0xfb 0xba */  "\351\207\232",
  /* 0xfb 0xbb */  "\351\207\227",
  /* 0xfb 0xbc */  "\351\207\236",
  /* 0xfb 0xbd */  "\351\207\255",
  /* 0xfb 0xbe */  "\351\207\256",
  /* 0xfb 0xbf */  "\351\207\244",
  /* 0xfb 0xc0 */  "\351\207\245",
  /* 0xfb 0xc1 */  "\351\210\206",
  /* 0xfb 0xc2 */  "\351\210\220",
  /* 0xfb 0xc3 */  "\351\210\212",
  /* 0xfb 0xc4 */  "\351\210\272",
  /* 0xfb 0xc5 */  "\351\211\200",
  /* 0xfb 0xc6 */  "\351\210\274",
  /* 0xfb 0xc7 */  "\351\211\216",
  /* 0xfb 0xc8 */  "\351\211\231",
  /* 0xfb 0xc9 */  "\351\211\221",
  /* 0xfb 0xca */  "\351\210\271",
  /* 0xfb 0xcb */  "\351\211\247",
  /* 0xfb 0xcc */  "\351\212\247",
  /* 0xfb 0xcd */  "\351\211\267",
  /* 0xfb 0xce */  "\351\211\270",
  /* 0xfb 0xcf */  "\351\213\247",
  /* 0xfb 0xd0 */  "\351\213\227",
  /* 0xfb 0xd1 */  "\351\213\231",
  /* 0xfb 0xd2 */  "\351\213\220",
  /* 0xfb 0xd3 */  "\357\250\247",
  /* 0xfb 0xd4 */  "\351\213\225",
  /* 0xfb 0xd5 */  "\351\213\240",
  /* 0xfb 0xd6 */  "\351\213\223",
  /* 0xfb 0xd7 */  "\351\214\245",
  /* 0xfb 0xd8 */  "\351\214\241",
  /* 0xfb 0xd9 */  "\351\213\273",
  /* 0xfb 0xda */  "\357\250\250",
  /* 0xfb 0xdb */  "\351\214\236",
  /* 0xfb 0xdc */  "\351\213\277",
  /* 0xfb 0xdd */  "\351\214\235",
  /* 0xfb 0xde */  "\351\214\202",
  /* 0xfb 0xdf */  "\351\215\260",
  /* 0xfb 0xe0 */  "\351\215\227",
  /* 0xfb 0xe1 */  "\351\216\244",
  /* 0xfb 0xe2 */  "\351\217\206",
  /* 0xfb 0xe3 */  "\351\217\236",
  /* 0xfb 0xe4 */  "\351\217\270",
  /* 0xfb 0xe5 */  "\351\220\261",
  /* 0xfb 0xe6 */  "\351\221\205",
  /* 0xfb 0xe7 */  "\351\221\210",
  /* 0xfb 0xe8 */  "\351\226\222",
  /* 0xfb 0xe9 */  "\357\247\234",
  /* 0xfb 0xea */  "\357\250\251",
  /* 0xfb 0xeb */  "\351\232\235",
  /* 0xfb 0xec */  "\351\232\257",
  /* 0xfb 0xed */  "\351\234\263",
  /* 0xfb 0xee */  "\351\234\273",
  /* 0xfb 0xef */  "\351\235\203",
  /* 0xfb 0xf0 */  "\351\235\215",
  /* 0xfb 0xf1 */  "\351\235\217",
  /* 0xfb 0xf2 */  "\351\235\221",
  /* 0xfb 0xf3 */  "\351\235\225",
  /* 0xfb 0xf4 */  "\351\241\227",
  /* 0xfb 0xf5 */  "\351\241\245",
  /* 0xfb 0xf6 */  "\357\250\252",
  /* 0xfb 0xf7 */  "\357\250\253",
  /* 0xfb 0xf8 */  "\351\244\247",
  /* 0xfb 0xf9 */  "\357\250\254",
  /* 0xfb 0xfa */  "\351\246\236",
  /* 0xfb 0xfb */  "\351\251\216",
  /* 0xfb 0xfc */  "\351\253\231",
};


/* 0xfc, 0x40-0x4b */
static const char *cp932_to_utf8_table_0xfc[] = {
  /*====== 0xfc, 0x40-0x4b ======*/
  /* 0xfc 0x40 */  "\351\253\234",
  /* 0xfc 0x41 */  "\351\255\265",
  /* 0xfc 0x42 */  "\351\255\262",
  /* 0xfc 0x43 */  "\351\256\217",
  /* 0xfc 0x44 */  "\351\256\261",
  /* 0xfc 0x45 */  "\351\256\273",
  /* 0xfc 0x46 */  "\351\260\200",
  /* 0xfc 0x47 */  "\351\265\260",
  /* 0xfc 0x48 */  "\351\265\253",
  /* 0xfc 0x49 */  "\357\250\255",
  /* 0xfc 0x4a */  "\351\270\231",
  /* 0xfc 0x4b */  "\351\273\221",
};


#define CP932_TO_UTF8_0X87_MIN	0x40
#define CP932_TO_UTF8_0X87_MAX	0x93
#define CP932_TO_UTF8_0XED_MIN	0x40
#define CP932_TO_UTF8_0XED_MAX	0xfc
#define CP932_TO_UTF8_0XEE_MIN	0x40
#define CP932_TO_UTF8_0XEE_MAX	0xfc
#define CP932_TO_UTF8_0XFA_MIN	0x40
#define CP932_TO_UTF8_0XFA_MAX	0xfc
#define CP932_TO_UTF8_0XFB_MIN	0x40
#define CP932_TO_UTF8_0XFB_MAX	0xfc
#define CP932_TO_UTF8_0XFC_MIN	0x40
#define CP932_TO_UTF8_0XFC_MAX	0x4b


size_t
cp932_to_utf8_helper(const char **inbuf, size_t *inbytesleft,
		     char **outbuf, size_t *outbytesleft)
{
  const char *utf8_string = NULL;
  unsigned char first;
  unsigned char second;

  if (*inbytesleft < 2)
    {
      errno = EINVAL;
      return -1;
    }

  first = (*inbuf)[0];
  second = (*inbuf)[1];

#if DEBUG_ICONV
  fprintf(stderr, "first=%02x second=%02x\n", first, second);
#endif

  switch (first)
    {
    case 0x87:
      if (second >= CP932_TO_UTF8_0X87_MIN && second <= CP932_TO_UTF8_0X87_MAX)
	utf8_string = cp932_to_utf8_table_0x87[second - CP932_TO_UTF8_0X87_MIN];
      break;

    case 0xed:
      if (second >= CP932_TO_UTF8_0XED_MIN && second <= CP932_TO_UTF8_0XED_MAX)
	utf8_string = cp932_to_utf8_table_0xed[second - CP932_TO_UTF8_0XED_MIN];
      break;

    case 0xee:
      if (second >= CP932_TO_UTF8_0XEE_MIN && second <= CP932_TO_UTF8_0XEE_MAX)
	utf8_string = cp932_to_utf8_table_0xee[second - CP932_TO_UTF8_0XEE_MIN];
      break;

    case 0xfa:
      if (second >= CP932_TO_UTF8_0XFA_MIN && second <= CP932_TO_UTF8_0XFA_MAX)
	utf8_string = cp932_to_utf8_table_0xfa[second - CP932_TO_UTF8_0XFA_MIN];
      break;

    case 0xfb:
      if (second >= CP932_TO_UTF8_0XFB_MIN && second <= CP932_TO_UTF8_0XFB_MAX)
	utf8_string = cp932_to_utf8_table_0xfb[second - CP932_TO_UTF8_0XFB_MIN];
      break;

    case 0xfc:
      if (second >= CP932_TO_UTF8_0XFC_MIN && second <= CP932_TO_UTF8_0XFC_MAX)
	utf8_string = cp932_to_utf8_table_0xfc[second - CP932_TO_UTF8_0XFC_MIN];
      break;

    }

  if (utf8_string == NULL)
    {
      /* Ʀ(={ 0xe2, 0x96, 0xa1 })ɤ*/
#if DEBUG_ICONV
      fprintf(stderr, "first=%02x second=%02x\n", first, second);
#endif
      utf8_string = "\342\226\241";
#if 0	/* 1 byteʤɤ */
      first = '\0';
#endif
    }

  if ((first & 0x80) != 0)
    {
      *inbuf += 2;
      *inbytesleft -= 2;
    }
  else
    {
      *inbuf += 1;
      *inbytesleft -= 1;
    }

  if ((utf8_string[0] & 0x80) == 0)
    {
      /* 1 byte */
      if (*outbytesleft < 1)
	{
	  errno = E2BIG;
	  return -1;
	}
      (*outbuf)[0] = utf8_string[0];
      *outbuf += 1;
      *outbytesleft -= 1;
    }
  else if ((utf8_string[0] & 0xe0) == 0xc0)
    {
      /* 2 byte */
      if (*outbytesleft < 2)
	{
	  errno = E2BIG;
	  return -1;
	}
      (*outbuf)[0] = utf8_string[0];
      (*outbuf)[1] = utf8_string[1];
      *outbuf += 2;
      *outbytesleft -= 2;
    }
  else
    {
      /* 3 byte */
      if (*outbytesleft < 3)
	{
	  errno = E2BIG;
	  return -1;
	}
      (*outbuf)[0] = utf8_string[0];
      (*outbuf)[1] = utf8_string[1];
      (*outbuf)[2] = utf8_string[2];
      *outbuf += 3;
      *outbytesleft -= 3;
    }

  return 0;
}


/* 0x8e, 0xa1-0xdf */
static const char *eucjp_to_utf8_table_0x8e[] = {
  /* 0x8e 0xa1 */  "\357\275\241",
  /* 0x8e 0xa2 */  "\357\275\242",
  /* 0x8e 0xa3 */  "\357\275\243",
  /* 0x8e 0xa4 */  "\357\275\244",
  /* 0x8e 0xa5 */  "\357\275\245",
  /* 0x8e 0xa6 */  "\357\275\246",
  /* 0x8e 0xa7 */  "\357\275\247",
  /* 0x8e 0xa8 */  "\357\275\250",
  /* 0x8e 0xa9 */  "\357\275\251",
  /* 0x8e 0xaa */  "\357\275\252",
  /* 0x8e 0xab */  "\357\275\253",
  /* 0x8e 0xac */  "\357\275\254",
  /* 0x8e 0xad */  "\357\275\255",
  /* 0x8e 0xae */  "\357\275\256",
  /* 0x8e 0xaf */  "\357\275\257",
  /* 0x8e 0xb0 */  "\357\275\260",
  /* 0x8e 0xb1 */  "\357\275\261",
  /* 0x8e 0xb2 */  "\357\275\262",
  /* 0x8e 0xb3 */  "\357\275\263",
  /* 0x8e 0xb4 */  "\357\275\264",
  /* 0x8e 0xb5 */  "\357\275\265",
  /* 0x8e 0xb6 */  "\357\275\266",
  /* 0x8e 0xb7 */  "\357\275\267",
  /* 0x8e 0xb8 */  "\357\275\270",
  /* 0x8e 0xb9 */  "\357\275\271",
  /* 0x8e 0xba */  "\357\275\272",
  /* 0x8e 0xbb */  "\357\275\273",
  /* 0x8e 0xbc */  "\357\275\274",
  /* 0x8e 0xbd */  "\357\275\275",
  /* 0x8e 0xbe */  "\357\275\276",
  /* 0x8e 0xbf */  "\357\275\277",
  /* 0x8e 0xc0 */  "\357\276\200",
  /* 0x8e 0xc1 */  "\357\276\201",
  /* 0x8e 0xc2 */  "\357\276\202",
  /* 0x8e 0xc3 */  "\357\276\203",
  /* 0x8e 0xc4 */  "\357\276\204",
  /* 0x8e 0xc5 */  "\357\276\205",
  /* 0x8e 0xc6 */  "\357\276\206",
  /* 0x8e 0xc7 */  "\357\276\207",
  /* 0x8e 0xc8 */  "\357\276\210",
  /* 0x8e 0xc9 */  "\357\276\211",
  /* 0x8e 0xca */  "\357\276\212",
  /* 0x8e 0xcb */  "\357\276\213",
  /* 0x8e 0xcc */  "\357\276\214",
  /* 0x8e 0xcd */  "\357\276\215",
  /* 0x8e 0xce */  "\357\276\216",
  /* 0x8e 0xcf */  "\357\276\217",
  /* 0x8e 0xd0 */  "\357\276\220",
  /* 0x8e 0xd1 */  "\357\276\221",
  /* 0x8e 0xd2 */  "\357\276\222",
  /* 0x8e 0xd3 */  "\357\276\223",
  /* 0x8e 0xd4 */  "\357\276\224",
  /* 0x8e 0xd5 */  "\357\276\225",
  /* 0x8e 0xd6 */  "\357\276\226",
  /* 0x8e 0xd7 */  "\357\276\227",
  /* 0x8e 0xd8 */  "\357\276\230",
  /* 0x8e 0xd9 */  "\357\276\231",
  /* 0x8e 0xda */  "\357\276\232",
  /* 0x8e 0xdb */  "\357\276\233",
  /* 0x8e 0xdc */  "\357\276\234",
  /* 0x8e 0xdd */  "\357\276\235",
  /* 0x8e 0xde */  "\357\276\236",
  /* 0x8e 0xdf */  "\357\276\237",
};


/* 0xa2, 0xaf-0xf1 */
static const char *eucjp_to_utf8_table_0xa2[] = {
  /* 0xa2 0xaf */  "\313\230",
  /* 0xa2 0xb0 */  "\313\207",
  /* 0xa2 0xb1 */  "\302\270",
  /* 0xa2 0xb2 */  "\313\231",
  /* 0xa2 0xb3 */  "\313\235",
  /* 0xa2 0xb4 */  "\302\257",
  /* 0xa2 0xb5 */  "\313\233",
  /* 0xa2 0xb6 */  "\313\232",
  /* 0xa2 0xb7 */  "\357\275\236",
  /* 0xa2 0xb8 */  "\316\204",
  /* 0xa2 0xb9 */  "\316\205",
  /* 0xa2 0xba (invalid) */  NULL,
  /* 0xa2 0xbb (invalid) */  NULL,
  /* 0xa2 0xbc (invalid) */  NULL,
  /* 0xa2 0xbd (invalid) */  NULL,
  /* 0xa2 0xbe (invalid) */  NULL,
  /* 0xa2 0xbf (invalid) */  NULL,
  /* 0xa2 0xc0 (invalid) */  NULL,
  /* 0xa2 0xc1 (invalid) */  NULL,
  /* 0xa2 0xc2 */  "\302\241",
  /* 0xa2 0xc3 */  "\357\277\244",
  /* 0xa2 0xc4 */  "\302\277",
  /* 0xa2 0xc5 (invalid) */  NULL,
  /* 0xa2 0xc6 (invalid) */  NULL,
  /* 0xa2 0xc7 (invalid) */  NULL,
  /* 0xa2 0xc8 (invalid) */  NULL,
  /* 0xa2 0xc9 (invalid) */  NULL,
  /* 0xa2 0xca (invalid) */  NULL,
  /* 0xa2 0xcb (invalid) */  NULL,
  /* 0xa2 0xcc (invalid) */  NULL,
  /* 0xa2 0xcd (invalid) */  NULL,
  /* 0xa2 0xce (invalid) */  NULL,
  /* 0xa2 0xcf (invalid) */  NULL,
  /* 0xa2 0xd0 (invalid) */  NULL,
  /* 0xa2 0xd1 (invalid) */  NULL,
  /* 0xa2 0xd2 (invalid) */  NULL,
  /* 0xa2 0xd3 (invalid) */  NULL,
  /* 0xa2 0xd4 (invalid) */  NULL,
  /* 0xa2 0xd5 (invalid) */  NULL,
  /* 0xa2 0xd6 (invalid) */  NULL,
  /* 0xa2 0xd7 (invalid) */  NULL,
  /* 0xa2 0xd8 (invalid) */  NULL,
  /* 0xa2 0xd9 (invalid) */  NULL,
  /* 0xa2 0xda (invalid) */  NULL,
  /* 0xa2 0xdb (invalid) */  NULL,
  /* 0xa2 0xdc (invalid) */  NULL,
  /* 0xa2 0xdd (invalid) */  NULL,
  /* 0xa2 0xde (invalid) */  NULL,
  /* 0xa2 0xdf (invalid) */  NULL,
  /* 0xa2 0xe0 (invalid) */  NULL,
  /* 0xa2 0xe1 (invalid) */  NULL,
  /* 0xa2 0xe2 (invalid) */  NULL,
  /* 0xa2 0xe3 (invalid) */  NULL,
  /* 0xa2 0xe4 (invalid) */  NULL,
  /* 0xa2 0xe5 (invalid) */  NULL,
  /* 0xa2 0xe6 (invalid) */  NULL,
  /* 0xa2 0xe7 (invalid) */  NULL,
  /* 0xa2 0xe8 (invalid) */  NULL,
  /* 0xa2 0xe9 (invalid) */  NULL,
  /* 0xa2 0xea (invalid) */  NULL,
  /* 0xa2 0xeb */  "\302\272",
  /* 0xa2 0xec */  "\302\252",
  /* 0xa2 0xed */  "\302\251",
  /* 0xa2 0xee */  "\302\256",
  /* 0xa2 0xef */  "\342\204\242",
  /* 0xa2 0xf0 */  "\302\244",
  /* 0xa2 0xf1 */  "\342\204\226",
};


/* 0xa6, 0xe1-0xfc */
static const char *eucjp_to_utf8_table_0xa6[] = {
  /* 0xa6 0xe1 */  "\316\206",
  /* 0xa6 0xe2 */  "\316\210",
  /* 0xa6 0xe3 */  "\316\211",
  /* 0xa6 0xe4 */  "\316\212",
  /* 0xa6 0xe5 */  "\316\252",
  /* 0xa6 0xe6 (invalid) */  NULL,
  /* 0xa6 0xe7 */  "\316\214",
  /* 0xa6 0xe8 (invalid) */  NULL,
  /* 0xa6 0xe9 */  "\316\216",
  /* 0xa6 0xea */  "\316\253",
  /* 0xa6 0xeb (invalid) */  NULL,
  /* 0xa6 0xec */  "\316\217",
  /* 0xa6 0xed (invalid) */  NULL,
  /* 0xa6 0xee (invalid) */  NULL,
  /* 0xa6 0xef (invalid) */  NULL,
  /* 0xa6 0xf0 (invalid) */  NULL,
  /* 0xa6 0xf1 */  "\316\254",
  /* 0xa6 0xf2 */  "\316\255",
  /* 0xa6 0xf3 */  "\316\256",
  /* 0xa6 0xf4 */  "\316\257",
  /* 0xa6 0xf5 */  "\317\212",
  /* 0xa6 0xf6 */  "\316\220",
  /* 0xa6 0xf7 */  "\317\214",
  /* 0xa6 0xf8 */  "\317\202",
  /* 0xa6 0xf9 */  "\317\215",
  /* 0xa6 0xfa */  "\317\213",
  /* 0xa6 0xfb */  "\316\260",
  /* 0xa6 0xfc */  "\317\216",
};


/* 0xa7, 0xc2-0xfe */
static const char *eucjp_to_utf8_table_0xa7[] = {
  /* 0xa7 0xc2 */  "\320\202",
  /* 0xa7 0xc3 */  "\320\203",
  /* 0xa7 0xc4 */  "\320\204",
  /* 0xa7 0xc5 */  "\320\205",
  /* 0xa7 0xc6 */  "\320\206",
  /* 0xa7 0xc7 */  "\320\207",
  /* 0xa7 0xc8 */  "\320\210",
  /* 0xa7 0xc9 */  "\320\211",
  /* 0xa7 0xca */  "\320\212",
  /* 0xa7 0xcb */  "\320\213",
  /* 0xa7 0xcc */  "\320\214",
  /* 0xa7 0xcd */  "\320\216",
  /* 0xa7 0xce */  "\320\217",
  /* 0xa7 0xcf (invalid) */  NULL,
  /* 0xa7 0xd0 (invalid) */  NULL,
  /* 0xa7 0xd1 (invalid) */  NULL,
  /* 0xa7 0xd2 (invalid) */  NULL,
  /* 0xa7 0xd3 (invalid) */  NULL,
  /* 0xa7 0xd4 (invalid) */  NULL,
  /* 0xa7 0xd5 (invalid) */  NULL,
  /* 0xa7 0xd6 (invalid) */  NULL,
  /* 0xa7 0xd7 (invalid) */  NULL,
  /* 0xa7 0xd8 (invalid) */  NULL,
  /* 0xa7 0xd9 (invalid) */  NULL,
  /* 0xa7 0xda (invalid) */  NULL,
  /* 0xa7 0xdb (invalid) */  NULL,
  /* 0xa7 0xdc (invalid) */  NULL,
  /* 0xa7 0xdd (invalid) */  NULL,
  /* 0xa7 0xde (invalid) */  NULL,
  /* 0xa7 0xdf (invalid) */  NULL,
  /* 0xa7 0xe0 (invalid) */  NULL,
  /* 0xa7 0xe1 (invalid) */  NULL,
  /* 0xa7 0xe2 (invalid) */  NULL,
  /* 0xa7 0xe3 (invalid) */  NULL,
  /* 0xa7 0xe4 (invalid) */  NULL,
  /* 0xa7 0xe5 (invalid) */  NULL,
  /* 0xa7 0xe6 (invalid) */  NULL,
  /* 0xa7 0xe7 (invalid) */  NULL,
  /* 0xa7 0xe8 (invalid) */  NULL,
  /* 0xa7 0xe9 (invalid) */  NULL,
  /* 0xa7 0xea (invalid) */  NULL,
  /* 0xa7 0xeb (invalid) */  NULL,
  /* 0xa7 0xec (invalid) */  NULL,
  /* 0xa7 0xed (invalid) */  NULL,
  /* 0xa7 0xee (invalid) */  NULL,
  /* 0xa7 0xef (invalid) */  NULL,
  /* 0xa7 0xf0 (invalid) */  NULL,
  /* 0xa7 0xf1 (invalid) */  NULL,
  /* 0xa7 0xf2 */  "\321\222",
  /* 0xa7 0xf3 */  "\321\223",
  /* 0xa7 0xf4 */  "\321\224",
  /* 0xa7 0xf5 */  "\321\225",
  /* 0xa7 0xf6 */  "\321\226",
  /* 0xa7 0xf7 */  "\321\227",
  /* 0xa7 0xf8 */  "\321\230",
  /* 0xa7 0xf9 */  "\321\231",
  /* 0xa7 0xfa */  "\321\232",
  /* 0xa7 0xfb */  "\321\233",
  /* 0xa7 0xfc */  "\321\234",
  /* 0xa7 0xfd */  "\321\236",
  /* 0xa7 0xfe */  "\321\237",
};


/* 0xa9, 0xa1-0xd0 */
static const char *eucjp_to_utf8_table_0xa9[] = {
  /* 0xa9 0xa1 */  "\303\206",
  /* 0xa9 0xa2 */  "\304\220",
  /* 0xa9 0xa3 (invalid) */  NULL,
  /* 0xa9 0xa4 */  "\304\246",
  /* 0xa9 0xa5 (invalid) */  NULL,
  /* 0xa9 0xa6 */  "\304\262",
  /* 0xa9 0xa7 (invalid) */  NULL,
  /* 0xa9 0xa8 */  "\305\201",
  /* 0xa9 0xa9 */  "\304\277",
  /* 0xa9 0xaa (invalid) */  NULL,
  /* 0xa9 0xab */  "\305\212",
  /* 0xa9 0xac */  "\303\230",
  /* 0xa9 0xad */  "\305\222",
  /* 0xa9 0xae (invalid) */  NULL,
  /* 0xa9 0xaf */  "\305\246",
  /* 0xa9 0xb0 */  "\303\236",
  /* 0xa9 0xb1 (invalid) */  NULL,
  /* 0xa9 0xb2 (invalid) */  NULL,
  /* 0xa9 0xb3 (invalid) */  NULL,
  /* 0xa9 0xb4 (invalid) */  NULL,
  /* 0xa9 0xb5 (invalid) */  NULL,
  /* 0xa9 0xb6 (invalid) */  NULL,
  /* 0xa9 0xb7 (invalid) */  NULL,
  /* 0xa9 0xb8 (invalid) */  NULL,
  /* 0xa9 0xb9 (invalid) */  NULL,
  /* 0xa9 0xba (invalid) */  NULL,
  /* 0xa9 0xbb (invalid) */  NULL,
  /* 0xa9 0xbc (invalid) */  NULL,
  /* 0xa9 0xbd (invalid) */  NULL,
  /* 0xa9 0xbe (invalid) */  NULL,
  /* 0xa9 0xbf (invalid) */  NULL,
  /* 0xa9 0xc0 (invalid) */  NULL,
  /* 0xa9 0xc1 */  "\303\246",
  /* 0xa9 0xc2 */  "\304\221",
  /* 0xa9 0xc3 */  "\303\260",
  /* 0xa9 0xc4 */  "\304\247",
  /* 0xa9 0xc5 */  "\304\261",
  /* 0xa9 0xc6 */  "\304\263",
  /* 0xa9 0xc7 */  "\304\270",
  /* 0xa9 0xc8 */  "\305\202",
  /* 0xa9 0xc9 */  "\305\200",
  /* 0xa9 0xca */  "\305\211",
  /* 0xa9 0xcb */  "\305\213",
  /* 0xa9 0xcc */  "\303\270",
  /* 0xa9 0xcd */  "\305\223",
  /* 0xa9 0xce */  "\303\237",
  /* 0xa9 0xcf */  "\305\247",
  /* 0xa9 0xd0 */  "\303\276",
};


/* 0xaa, 0xa1-0xf7 */
static const char *eucjp_to_utf8_table_0xaa[] = {
  /* 0xaa 0xa1 */  "\303\201",
  /* 0xaa 0xa2 */  "\303\200",
  /* 0xaa 0xa3 */  "\303\204",
  /* 0xaa 0xa4 */  "\303\202",
  /* 0xaa 0xa5 */  "\304\202",
  /* 0xaa 0xa6 */  "\307\215",
  /* 0xaa 0xa7 */  "\304\200",
  /* 0xaa 0xa8 */  "\304\204",
  /* 0xaa 0xa9 */  "\303\205",
  /* 0xaa 0xaa */  "\303\203",
  /* 0xaa 0xab */  "\304\206",
  /* 0xaa 0xac */  "\304\210",
  /* 0xaa 0xad */  "\304\214",
  /* 0xaa 0xae */  "\303\207",
  /* 0xaa 0xaf */  "\304\212",
  /* 0xaa 0xb0 */  "\304\216",
  /* 0xaa 0xb1 */  "\303\211",
  /* 0xaa 0xb2 */  "\303\210",
  /* 0xaa 0xb3 */  "\303\213",
  /* 0xaa 0xb4 */  "\303\212",
  /* 0xaa 0xb5 */  "\304\232",
  /* 0xaa 0xb6 */  "\304\226",
  /* 0xaa 0xb7 */  "\304\222",
  /* 0xaa 0xb8 */  "\304\230",
  /* 0xaa 0xb9 (invalid) */  NULL,
  /* 0xaa 0xba */  "\304\234",
  /* 0xaa 0xbb */  "\304\236",
  /* 0xaa 0xbc */  "\304\242",
  /* 0xaa 0xbd */  "\304\240",
  /* 0xaa 0xbe */  "\304\244",
  /* 0xaa 0xbf */  "\303\215",
  /* 0xaa 0xc0 */  "\303\214",
  /* 0xaa 0xc1 */  "\303\217",
  /* 0xaa 0xc2 */  "\303\216",
  /* 0xaa 0xc3 */  "\307\217",
  /* 0xaa 0xc4 */  "\304\260",
  /* 0xaa 0xc5 */  "\304\252",
  /* 0xaa 0xc6 */  "\304\256",
  /* 0xaa 0xc7 */  "\304\250",
  /* 0xaa 0xc8 */  "\304\264",
  /* 0xaa 0xc9 */  "\304\266",
  /* 0xaa 0xca */  "\304\271",
  /* 0xaa 0xcb */  "\304\275",
  /* 0xaa 0xcc */  "\304\273",
  /* 0xaa 0xcd */  "\305\203",
  /* 0xaa 0xce */  "\305\207",
  /* 0xaa 0xcf */  "\305\205",
  /* 0xaa 0xd0 */  "\303\221",
  /* 0xaa 0xd1 */  "\303\223",
  /* 0xaa 0xd2 */  "\303\222",
  /* 0xaa 0xd3 */  "\303\226",
  /* 0xaa 0xd4 */  "\303\224",
  /* 0xaa 0xd5 */  "\307\221",
  /* 0xaa 0xd6 */  "\305\220",
  /* 0xaa 0xd7 */  "\305\214",
  /* 0xaa 0xd8 */  "\303\225",
  /* 0xaa 0xd9 */  "\305\224",
  /* 0xaa 0xda */  "\305\230",
  /* 0xaa 0xdb */  "\305\226",
  /* 0xaa 0xdc */  "\305\232",
  /* 0xaa 0xdd */  "\305\234",
  /* 0xaa 0xde */  "\305\240",
  /* 0xaa 0xdf */  "\305\236",
  /* 0xaa 0xe0 */  "\305\244",
  /* 0xaa 0xe1 */  "\305\242",
  /* 0xaa 0xe2 */  "\303\232",
  /* 0xaa 0xe3 */  "\303\231",
  /* 0xaa 0xe4 */  "\303\234",
  /* 0xaa 0xe5 */  "\303\233",
  /* 0xaa 0xe6 */  "\305\254",
  /* 0xaa 0xe7 */  "\307\223",
  /* 0xaa 0xe8 */  "\305\260",
  /* 0xaa 0xe9 */  "\305\252",
  /* 0xaa 0xea */  "\305\262",
  /* 0xaa 0xeb */  "\305\256",
  /* 0xaa 0xec */  "\305\250",
  /* 0xaa 0xed */  "\307\227",
  /* 0xaa 0xee */  "\307\233",
  /* 0xaa 0xef */  "\307\231",
  /* 0xaa 0xf0 */  "\307\225",
  /* 0xaa 0xf1 */  "\305\264",
  /* 0xaa 0xf2 */  "\303\235",
  /* 0xaa 0xf3 */  "\305\270",
  /* 0xaa 0xf4 */  "\305\266",
  /* 0xaa 0xf5 */  "\305\271",
  /* 0xaa 0xf6 */  "\305\275",
  /* 0xaa 0xf7 */  "\305\273",
};


/* 0xab, 0xa1-0xf7 */
static const char *eucjp_to_utf8_table_0xab[] = {
  /* 0xab 0xa1 */  "\303\241",
  /* 0xab 0xa2 */  "\303\240",
  /* 0xab 0xa3 */  "\303\244",
  /* 0xab 0xa4 */  "\303\242",
  /* 0xab 0xa5 */  "\304\203",
  /* 0xab 0xa6 */  "\307\216",
  /* 0xab 0xa7 */  "\304\201",
  /* 0xab 0xa8 */  "\304\205",
  /* 0xab 0xa9 */  "\303\245",
  /* 0xab 0xaa */  "\303\243",
  /* 0xab 0xab */  "\304\207",
  /* 0xab 0xac */  "\304\211",
  /* 0xab 0xad */  "\304\215",
  /* 0xab 0xae */  "\303\247",
  /* 0xab 0xaf */  "\304\213",
  /* 0xab 0xb0 */  "\304\217",
  /* 0xab 0xb1 */  "\303\251",
  /* 0xab 0xb2 */  "\303\250",
  /* 0xab 0xb3 */  "\303\253",
  /* 0xab 0xb4 */  "\303\252",
  /* 0xab 0xb5 */  "\304\233",
  /* 0xab 0xb6 */  "\304\227",
  /* 0xab 0xb7 */  "\304\223",
  /* 0xab 0xb8 */  "\304\231",
  /* 0xab 0xb9 */  "\307\265",
  /* 0xab 0xba */  "\304\235",
  /* 0xab 0xbb */  "\304\237",
  /* 0xab 0xbc (invalid) */  NULL,
  /* 0xab 0xbd */  "\304\241",
  /* 0xab 0xbe */  "\304\245",
  /* 0xab 0xbf */  "\303\255",
  /* 0xab 0xc0 */  "\303\254",
  /* 0xab 0xc1 */  "\303\257",
  /* 0xab 0xc2 */  "\303\256",
  /* 0xab 0xc3 */  "\307\220",
  /* 0xab 0xc4 (invalid) */  NULL,
  /* 0xab 0xc5 */  "\304\253",
  /* 0xab 0xc6 */  "\304\257",
  /* 0xab 0xc7 */  "\304\251",
  /* 0xab 0xc8 */  "\304\265",
  /* 0xab 0xc9 */  "\304\267",
  /* 0xab 0xca */  "\304\272",
  /* 0xab 0xcb */  "\304\276",
  /* 0xab 0xcc */  "\304\274",
  /* 0xab 0xcd */  "\305\204",
  /* 0xab 0xce */  "\305\210",
  /* 0xab 0xcf */  "\305\206",
  /* 0xab 0xd0 */  "\303\261",
  /* 0xab 0xd1 */  "\303\263",
  /* 0xab 0xd2 */  "\303\262",
  /* 0xab 0xd3 */  "\303\266",
  /* 0xab 0xd4 */  "\303\264",
  /* 0xab 0xd5 */  "\307\222",
  /* 0xab 0xd6 */  "\305\221",
  /* 0xab 0xd7 */  "\305\215",
  /* 0xab 0xd8 */  "\303\265",
  /* 0xab 0xd9 */  "\305\225",
  /* 0xab 0xda */  "\305\231",
  /* 0xab 0xdb */  "\305\227",
  /* 0xab 0xdc */  "\305\233",
  /* 0xab 0xdd */  "\305\235",
  /* 0xab 0xde */  "\305\241",
  /* 0xab 0xdf */  "\305\237",
  /* 0xab 0xe0 */  "\305\245",
  /* 0xab 0xe1 */  "\305\243",
  /* 0xab 0xe2 */  "\303\272",
  /* 0xab 0xe3 */  "\303\271",
  /* 0xab 0xe4 */  "\303\274",
  /* 0xab 0xe5 */  "\303\273",
  /* 0xab 0xe6 */  "\305\255",
  /* 0xab 0xe7 */  "\307\224",
  /* 0xab 0xe8 */  "\305\261",
  /* 0xab 0xe9 */  "\305\253",
  /* 0xab 0xea */  "\305\263",
  /* 0xab 0xeb */  "\305\257",
  /* 0xab 0xec */  "\305\251",
  /* 0xab 0xed */  "\307\230",
  /* 0xab 0xee */  "\307\234",
  /* 0xab 0xef */  "\307\232",
  /* 0xab 0xf0 */  "\307\226",
  /* 0xab 0xf1 */  "\305\265",
  /* 0xab 0xf2 */  "\303\275",
  /* 0xab 0xf3 */  "\303\277",
  /* 0xab 0xf4 */  "\305\267",
  /* 0xab 0xf5 */  "\305\272",
  /* 0xab 0xf6 */  "\305\276",
  /* 0xab 0xf7 */  "\305\274",
};


/* 0xad, 0xa1-0xfc */
static const char *eucjp_to_utf8_table_0xad[] = {
  /* 0xad 0xa1 */  "\342\221\240",
  /* 0xad 0xa2 */  "\342\221\241",
  /* 0xad 0xa3 */  "\342\221\242",
  /* 0xad 0xa4 */  "\342\221\243",
  /* 0xad 0xa5 */  "\342\221\244",
  /* 0xad 0xa6 */  "\342\221\245",
  /* 0xad 0xa7 */  "\342\221\246",
  /* 0xad 0xa8 */  "\342\221\247",
  /* 0xad 0xa9 */  "\342\221\250",
  /* 0xad 0xaa */  "\342\221\251",
  /* 0xad 0xab */  "\342\221\252",
  /* 0xad 0xac */  "\342\221\253",
  /* 0xad 0xad */  "\342\221\254",
  /* 0xad 0xae */  "\342\221\255",
  /* 0xad 0xaf */  "\342\221\256",
  /* 0xad 0xb0 */  "\342\221\257",
  /* 0xad 0xb1 */  "\342\221\260",
  /* 0xad 0xb2 */  "\342\221\261",
  /* 0xad 0xb3 */  "\342\221\262",
  /* 0xad 0xb4 */  "\342\221\263",
  /* 0xad 0xb5 */  "\342\205\240",
  /* 0xad 0xb6 */  "\342\205\241",
  /* 0xad 0xb7 */  "\342\205\242",
  /* 0xad 0xb8 */  "\342\205\243",
  /* 0xad 0xb9 */  "\342\205\244",
  /* 0xad 0xba */  "\342\205\245",
  /* 0xad 0xbb */  "\342\205\246",
  /* 0xad 0xbc */  "\342\205\247",
  /* 0xad 0xbd */  "\342\205\250",
  /* 0xad 0xbe */  "\342\205\251",
  /* 0xad 0xbf (invalid) */  NULL,
  /* 0xad 0xc0 */  "\343\215\211",
  /* 0xad 0xc1 */  "\343\214\224",
  /* 0xad 0xc2 */  "\343\214\242",
  /* 0xad 0xc3 */  "\343\215\215",
  /* 0xad 0xc4 */  "\343\214\230",
  /* 0xad 0xc5 */  "\343\214\247",
  /* 0xad 0xc6 */  "\343\214\203",
  /* 0xad 0xc7 */  "\343\214\266",
  /* 0xad 0xc8 */  "\343\215\221",
  /* 0xad 0xc9 */  "\343\215\227",
  /* 0xad 0xca */  "\343\214\215",
  /* 0xad 0xcb */  "\343\214\246",
  /* 0xad 0xcc */  "\343\214\243",
  /* 0xad 0xcd */  "\343\214\253",
  /* 0xad 0xce */  "\343\215\212",
  /* 0xad 0xcf */  "\343\214\273",
  /* 0xad 0xd0 */  "\343\216\234",
  /* 0xad 0xd1 */  "\343\216\235",
  /* 0xad 0xd2 */  "\343\216\236",
  /* 0xad 0xd3 */  "\343\216\216",
  /* 0xad 0xd4 */  "\343\216\217",
  /* 0xad 0xd5 */  "\343\217\204",
  /* 0xad 0xd6 */  "\343\216\241",
  /* 0xad 0xd7 (invalid) */  NULL,
  /* 0xad 0xd8 (invalid) */  NULL,
  /* 0xad 0xd9 (invalid) */  NULL,
  /* 0xad 0xda (invalid) */  NULL,
  /* 0xad 0xdb (invalid) */  NULL,
  /* 0xad 0xdc (invalid) */  NULL,
  /* 0xad 0xdd (invalid) */  NULL,
  /* 0xad 0xde (invalid) */  NULL,
  /* 0xad 0xdf */  "\343\215\273",
  /* 0xad 0xe0 */  "\343\200\235",
  /* 0xad 0xe1 */  "\343\200\237",
  /* 0xad 0xe2 */  "\342\204\226",
  /* 0xad 0xe3 */  "\343\217\215",
  /* 0xad 0xe4 */  "\342\204\241",
  /* 0xad 0xe5 */  "\343\212\244",
  /* 0xad 0xe6 */  "\343\212\245",
  /* 0xad 0xe7 */  "\343\212\246",
  /* 0xad 0xe8 */  "\343\212\247",
  /* 0xad 0xe9 */  "\343\212\250",
  /* 0xad 0xea */  "\343\210\261",
  /* 0xad 0xeb */  "\343\210\262",
  /* 0xad 0xec */  "\343\210\271",
  /* 0xad 0xed */  "\343\215\276",
  /* 0xad 0xee */  "\343\215\275",
  /* 0xad 0xef */  "\343\215\274",
  /* 0xad 0xf0 */  "\342\211\222",
  /* 0xad 0xf1 */  "\342\211\241",
  /* 0xad 0xf2 */  "\342\210\253",
  /* 0xad 0xf3 */  "\342\210\256",
  /* 0xad 0xf4 */  "\342\210\221",
  /* 0xad 0xf5 */  "\342\210\232",
  /* 0xad 0xf6 */  "\342\212\245",
  /* 0xad 0xf7 */  "\342\210\240",
  /* 0xad 0xf8 */  "\342\210\237",
  /* 0xad 0xf9 */  "\342\212\277",
  /* 0xad 0xfa */  "\342\210\265",
  /* 0xad 0xfb */  "\342\210\251",
  /* 0xad 0xfc */  "\342\210\252",
};


/* 0xb0, 0xa1-0xfe */
static const char *eucjp_to_utf8_table_0xb0[] = {
  /* 0xb0 0xa1 */  "\344\270\202",
  /* 0xb0 0xa2 */  "\344\270\204",
  /* 0xb0 0xa3 */  "\344\270\205",
  /* 0xb0 0xa4 */  "\344\270\214",
  /* 0xb0 0xa5 */  "\344\270\222",
  /* 0xb0 0xa6 */  "\344\270\237",
  /* 0xb0 0xa7 */  "\344\270\243",
  /* 0xb0 0xa8 */  "\344\270\244",
  /* 0xb0 0xa9 */  "\344\270\250",
  /* 0xb0 0xaa */  "\344\270\253",
  /* 0xb0 0xab */  "\344\270\256",
  /* 0xb0 0xac */  "\344\270\257",
  /* 0xb0 0xad */  "\344\270\260",
  /* 0xb0 0xae */  "\344\270\265",
  /* 0xb0 0xaf */  "\344\271\200",
  /* 0xb0 0xb0 */  "\344\271\201",
  /* 0xb0 0xb1 */  "\344\271\204",
  /* 0xb0 0xb2 */  "\344\271\207",
  /* 0xb0 0xb3 */  "\344\271\221",
  /* 0xb0 0xb4 */  "\344\271\232",
  /* 0xb0 0xb5 */  "\344\271\234",
  /* 0xb0 0xb6 */  "\344\271\243",
  /* 0xb0 0xb7 */  "\344\271\250",
  /* 0xb0 0xb8 */  "\344\271\251",
  /* 0xb0 0xb9 */  "\344\271\264",
  /* 0xb0 0xba */  "\344\271\265",
  /* 0xb0 0xbb */  "\344\271\271",
  /* 0xb0 0xbc */  "\344\271\277",
  /* 0xb0 0xbd */  "\344\272\215",
  /* 0xb0 0xbe */  "\344\272\226",
  /* 0xb0 0xbf */  "\344\272\227",
  /* 0xb0 0xc0 */  "\344\272\235",
  /* 0xb0 0xc1 */  "\344\272\257",
  /* 0xb0 0xc2 */  "\344\272\271",
  /* 0xb0 0xc3 */  "\344\273\203",
  /* 0xb0 0xc4 */  "\344\273\220",
  /* 0xb0 0xc5 */  "\344\273\232",
  /* 0xb0 0xc6 */  "\344\273\233",
  /* 0xb0 0xc7 */  "\344\273\240",
  /* 0xb0 0xc8 */  "\344\273\241",
  /* 0xb0 0xc9 */  "\344\273\242",
  /* 0xb0 0xca */  "\344\273\250",
  /* 0xb0 0xcb */  "\344\273\257",
  /* 0xb0 0xcc */  "\344\273\261",
  /* 0xb0 0xcd */  "\344\273\263",
  /* 0xb0 0xce */  "\344\273\265",
  /* 0xb0 0xcf */  "\344\273\275",
  /* 0xb0 0xd0 */  "\344\273\276",
  /* 0xb0 0xd1 */  "\344\273\277",
  /* 0xb0 0xd2 */  "\344\274\200",
  /* 0xb0 0xd3 */  "\344\274\202",
  /* 0xb0 0xd4 */  "\344\274\203",
  /* 0xb0 0xd5 */  "\344\274\210",
  /* 0xb0 0xd6 */  "\344\274\213",
  /* 0xb0 0xd7 */  "\344\274\214",
  /* 0xb0 0xd8 */  "\344\274\222",
  /* 0xb0 0xd9 */  "\344\274\225",
  /* 0xb0 0xda */  "\344\274\226",
  /* 0xb0 0xdb */  "\344\274\227",
  /* 0xb0 0xdc */  "\344\274\231",
  /* 0xb0 0xdd */  "\344\274\256",
  /* 0xb0 0xde */  "\344\274\261",
  /* 0xb0 0xdf */  "\344\275\240",
  /* 0xb0 0xe0 */  "\344\274\263",
  /* 0xb0 0xe1 */  "\344\274\265",
  /* 0xb0 0xe2 */  "\344\274\267",
  /* 0xb0 0xe3 */  "\344\274\271",
  /* 0xb0 0xe4 */  "\344\274\273",
  /* 0xb0 0xe5 */  "\344\274\276",
  /* 0xb0 0xe6 */  "\344\275\200",
  /* 0xb0 0xe7 */  "\344\275\202",
  /* 0xb0 0xe8 */  "\344\275\210",
  /* 0xb0 0xe9 */  "\344\275\211",
  /* 0xb0 0xea */  "\344\275\213",
  /* 0xb0 0xeb */  "\344\275\214",
  /* 0xb0 0xec */  "\344\275\222",
  /* 0xb0 0xed */  "\344\275\224",
  /* 0xb0 0xee */  "\344\275\226",
  /* 0xb0 0xef */  "\344\275\230",
  /* 0xb0 0xf0 */  "\344\275\237",
  /* 0xb0 0xf1 */  "\344\275\243",
  /* 0xb0 0xf2 */  "\344\275\252",
  /* 0xb0 0xf3 */  "\344\275\254",
  /* 0xb0 0xf4 */  "\344\275\256",
  /* 0xb0 0xf5 */  "\344\275\261",
  /* 0xb0 0xf6 */  "\344\275\267",
  /* 0xb0 0xf7 */  "\344\275\270",
  /* 0xb0 0xf8 */  "\344\275\271",
  /* 0xb0 0xf9 */  "\344\275\272",
  /* 0xb0 0xfa */  "\344\275\275",
  /* 0xb0 0xfb */  "\344\275\276",
  /* 0xb0 0xfc */  "\344\276\201",
  /* 0xb0 0xfd */  "\344\276\202",
  /* 0xb0 0xfe */  "\344\276\204",
};


/* 0xb1, 0xa1-0xfe */
static const char *eucjp_to_utf8_table_0xb1[] = {
  /* 0xb1 0xa1 */  "\344\276\205",
  /* 0xb1 0xa2 */  "\344\276\211",
  /* 0xb1 0xa3 */  "\344\276\212",
  /* 0xb1 0xa4 */  "\344\276\214",
  /* 0xb1 0xa5 */  "\344\276\216",
  /* 0xb1 0xa6 */  "\344\276\220",
  /* 0xb1 0xa7 */  "\344\276\222",
  /* 0xb1 0xa8 */  "\344\276\223",
  /* 0xb1 0xa9 */  "\344\276\224",
  /* 0xb1 0xaa */  "\344\276\227",
  /* 0xb1 0xab */  "\344\276\231",
  /* 0xb1 0xac */  "\344\276\232",
  /* 0xb1 0xad */  "\344\276\236",
  /* 0xb1 0xae */  "\344\276\237",
  /* 0xb1 0xaf */  "\344\276\262",
  /* 0xb1 0xb0 */  "\344\276\267",
  /* 0xb1 0xb1 */  "\344\276\271",
  /* 0xb1 0xb2 */  "\344\276\273",
  /* 0xb1 0xb3 */  "\344\276\274",
  /* 0xb1 0xb4 */  "\344\276\275",
  /* 0xb1 0xb5 */  "\344\276\276",
  /* 0xb1 0xb6 */  "\344\277\200",
  /* 0xb1 0xb7 */  "\344\277\201",
  /* 0xb1 0xb8 */  "\344\277\205",
  /* 0xb1 0xb9 */  "\344\277\206",
  /* 0xb1 0xba */  "\344\277\210",
  /* 0xb1 0xbb */  "\344\277\211",
  /* 0xb1 0xbc */  "\344\277\213",
  /* 0xb1 0xbd */  "\344\277\214",
  /* 0xb1 0xbe */  "\344\277\215",
  /* 0xb1 0xbf */  "\344\277\217",
  /* 0xb1 0xc0 */  "\344\277\222",
  /* 0xb1 0xc1 */  "\344\277\234",
  /* 0xb1 0xc2 */  "\344\277\240",
  /* 0xb1 0xc3 */  "\344\277\242",
  /* 0xb1 0xc4 */  "\344\277\260",
  /* 0xb1 0xc5 */  "\344\277\262",
  /* 0xb1 0xc6 */  "\344\277\274",
  /* 0xb1 0xc7 */  "\344\277\275",
  /* 0xb1 0xc8 */  "\344\277\277",
  /* 0xb1 0xc9 */  "\345\200\200",
  /* 0xb1 0xca */  "\345\200\201",
  /* 0xb1 0xcb */  "\345\200\204",
  /* 0xb1 0xcc */  "\345\200\207",
  /* 0xb1 0xcd */  "\345\200\212",
  /* 0xb1 0xce */  "\345\200\214",
  /* 0xb1 0xcf */  "\345\200\216",
  /* 0xb1 0xd0 */  "\345\200\220",
  /* 0xb1 0xd1 */  "\345\200\223",
  /* 0xb1 0xd2 */  "\345\200\227",
  /* 0xb1 0xd3 */  "\345\200\230",
  /* 0xb1 0xd4 */  "\345\200\233",
  /* 0xb1 0xd5 */  "\345\200\234",
  /* 0xb1 0xd6 */  "\345\200\235",
  /* 0xb1 0xd7 */  "\345\200\236",
  /* 0xb1 0xd8 */  "\345\200\242",
  /* 0xb1 0xd9 */  "\345\200\247",
  /* 0xb1 0xda */  "\345\200\256",
  /* 0xb1 0xdb */  "\345\200\260",
  /* 0xb1 0xdc */  "\345\200\262",
  /* 0xb1 0xdd */  "\345\200\263",
  /* 0xb1 0xde */  "\345\200\265",
  /* 0xb1 0xdf */  "\345\201\200",
  /* 0xb1 0xe0 */  "\345\201\201",
  /* 0xb1 0xe1 */  "\345\201\202",
  /* 0xb1 0xe2 */  "\345\201\205",
  /* 0xb1 0xe3 */  "\345\201\206",
  /* 0xb1 0xe4 */  "\345\201\212",
  /* 0xb1 0xe5 */  "\345\201\214",
  /* 0xb1 0xe6 */  "\345\201\216",
  /* 0xb1 0xe7 */  "\345\201\221",
  /* 0xb1 0xe8 */  "\345\201\222",
  /* 0xb1 0xe9 */  "\345\201\223",
  /* 0xb1 0xea */  "\345\201\227",
  /* 0xb1 0xeb */  "\345\201\231",
  /* 0xb1 0xec */  "\345\201\237",
  /* 0xb1 0xed */  "\345\201\240",
  /* 0xb1 0xee */  "\345\201\242",
  /* 0xb1 0xef */  "\345\201\243",
  /* 0xb1 0xf0 */  "\345\201\246",
  /* 0xb1 0xf1 */  "\345\201\247",
  /* 0xb1 0xf2 */  "\345\201\252",
  /* 0xb1 0xf3 */  "\345\201\255",
  /* 0xb1 0xf4 */  "\345\201\260",
  /* 0xb1 0xf5 */  "\345\201\261",
  /* 0xb1 0xf6 */  "\345\200\273",
  /* 0xb1 0xf7 */  "\345\202\201",
  /* 0xb1 0xf8 */  "\345\202\203",
  /* 0xb1 0xf9 */  "\345\202\204",
  /* 0xb1 0xfa */  "\345\202\206",
  /* 0xb1 0xfb */  "\345\202\212",
  /* 0xb1 0xfc */  "\345\202\216",
  /* 0xb1 0xfd */  "\345\202\217",
  /* 0xb1 0xfe */  "\345\202\220",
};


/* 0xb2, 0xa1-0xfe */
static const char *eucjp_to_utf8_table_0xb2[] = {
  /* 0xb2 0xa1 */  "\345\202\222",
  /* 0xb2 0xa2 */  "\345\202\223",
  /* 0xb2 0xa3 */  "\345\202\224",
  /* 0xb2 0xa4 */  "\345\202\226",
  /* 0xb2 0xa5 */  "\345\202\233",
  /* 0xb2 0xa6 */  "\345\202\234",
  /* 0xb2 0xa7 */  "\345\202\236",
  /* 0xb2 0xa8 */  "\345\202\237",
  /* 0xb2 0xa9 */  "\345\202\240",
  /* 0xb2 0xaa */  "\345\202\241",
  /* 0xb2 0xab */  "\345\202\242",
  /* 0xb2 0xac */  "\345\202\252",
  /* 0xb2 0xad */  "\345\202\257",
  /* 0xb2 0xae */  "\345\202\260",
  /* 0xb2 0xaf */  "\345\202\271",
  /* 0xb2 0xb0 */  "\345\202\272",
  /* 0xb2 0xb1 */  "\345\202\275",
  /* 0xb2 0xb2 */  "\345\203\200",
  /* 0xb2 0xb3 */  "\345\203\203",
  /* 0xb2 0xb4 */  "\345\203\204",
  /* 0xb2 0xb5 */  "\345\203\207",
  /* 0xb2 0xb6 */  "\345\203\214",
  /* 0xb2 0xb7 */  "\345\203\216",
  /* 0xb2 0xb8 */  "\345\203\220",
  /* 0xb2 0xb9 */  "\345\203\223",
  /* 0xb2 0xba */  "\345\203\224",
  /* 0xb2 0xbb */  "\345\203\230",
  /* 0xb2 0xbc */  "\345\203\234",
  /* 0xb2 0xbd */  "\345\203\235",
  /* 0xb2 0xbe */  "\345\203\237",
  /* 0xb2 0xbf */  "\345\203\242",
  /* 0xb2 0xc0 */  "\345\203\244",
  /* 0xb2 0xc1 */  "\345\203\246",
  /* 0xb2 0xc2 */  "\345\203\250",
  /* 0xb2 0xc3 */  "\345\203\251",
  /* 0xb2 0xc4 */  "\345\203\257",
  /* 0xb2 0xc5 */  "\345\203\261",
  /* 0xb2 0xc6 */  "\345\203\266",
  /* 0xb2 0xc7 */  "\345\203\272",
  /* 0xb2 0xc8 */  "\345\203\276",
  /* 0xb2 0xc9 */  "\345\204\203",
  /* 0xb2 0xca */  "\345\204\206",
  /* 0xb2 0xcb */  "\345\204\207",
  /* 0xb2 0xcc */  "\345\204\210",
  /* 0xb2 0xcd */  "\345\204\213",
  /* 0xb2 0xce */  "\345\204\214",
  /* 0xb2 0xcf */  "\345\204\215",
  /* 0xb2 0xd0 */  "\345\204\216",
  /* 0xb2 0xd1 */  "\345\203\262",
  /* 0xb2 0xd2 */  "\345\204\220",
  /* 0xb2 0xd3 */  "\345\204\227",
  /* 0xb2 0xd4 */  "\345\204\231",
  /* 0xb2 0xd5 */  "\345\204\233",
  /* 0xb2 0xd6 */  "\345\204\234",
  /* 0xb2 0xd7 */  "\345\204\235",
  /* 0xb2 0xd8 */  "\345\204\236",
  /* 0xb2 0xd9 */  "\345\204\243",
  /* 0xb2 0xda */  "\345\204\247",
  /* 0xb2 0xdb */  "\345\204\250",
  /* 0xb2 0xdc */  "\345\204\254",
  /* 0xb2 0xdd */  "\345\204\255",
  /* 0xb2 0xde */  "\345\204\257",
  /* 0xb2 0xdf */  "\345\204\261",
  /* 0xb2 0xe0 */  "\345\204\263",
  /* 0xb2 0xe1 */  "\345\204\264",
  /* 0xb2 0xe2 */  "\345\204\265",
  /* 0xb2 0xe3 */  "\345\204\270",
  /* 0xb2 0xe4 */  "\345\204\271",
  /* 0xb2 0xe5 */  "\345\205\202",
  /* 0xb2 0xe6 */  "\345\205\212",
  /* 0xb2 0xe7 */  "\345\205\217",
  /* 0xb2 0xe8 */  "\345\205\223",
  /* 0xb2 0xe9 */  "\345\205\225",
  /* 0xb2 0xea */  "\345\205\227",
  /* 0xb2 0xeb */  "\345\205\230",
  /* 0xb2 0xec */  "\345\205\237",
  /* 0xb2 0xed */  "\345\205\244",
  /* 0xb2 0xee */  "\345\205\246",
  /* 0xb2 0xef */  "\345\205\276",
  /* 0xb2 0xf0 */  "\345\206\203",
  /* 0xb2 0xf1 */  "\345\206\204",
  /* 0xb2 0xf2 */  "\345\206\213",
  /* 0xb2 0xf3 */  "\345\206\216",
  /* 0xb2 0xf4 */  "\345\206\230",
  /* 0xb2 0xf5 */  "\345\206\235",
  /* 0xb2 0xf6 */  "\345\206\241",
  /* 0xb2 0xf7 */  "\345\206\243",
  /* 0xb2 0xf8 */  "\345\206\255",
  /* 0xb2 0xf9 */  "\345\206\270",
  /* 0xb2 0xfa */  "\345\206\272",
  /* 0xb2 0xfb */  "\345\206\274",
  /* 0xb2 0xfc */  "\345\206\276",
  /* 0xb2 0xfd */  "\345\206\277",
  /* 0xb2 0xfe */  "\345\207\202",
};


/* 0xb3, 0xa1-0xfe */
static const char *eucjp_to_utf8_table_0xb3[] = {
  /* 0xb3 0xa1 */  "\345\207\210",
  /* 0xb3 0xa2 */  "\345\207\217",
  /* 0xb3 0xa3 */  "\345\207\221",
  /* 0xb3 0xa4 */  "\345\207\222",
  /* 0xb3 0xa5 */  "\345\207\223",
  /* 0xb3 0xa6 */  "\345\207\225",
  /* 0xb3 0xa7 */  "\345\207\230",
  /* 0xb3 0xa8 */  "\345\207\236",
  /* 0xb3 0xa9 */  "\345\207\242",
  /* 0xb3 0xaa */  "\345\207\245",
  /* 0xb3 0xab */  "\345\207\256",
  /* 0xb3 0xac */  "\345\207\262",
  /* 0xb3 0xad */  "\345\207\263",
  /* 0xb3 0xae */  "\345\207\264",
  /* 0xb3 0xaf */  "\345\207\267",
  /* 0xb3 0xb0 */  "\345\210\201",
  /* 0xb3 0xb1 */  "\345\210\202",
  /* 0xb3 0xb2 */  "\345\210\205",
  /* 0xb3 0xb3 */  "\345\210\222",
  /* 0xb3 0xb4 */  "\345\210\223",
  /* 0xb3 0xb5 */  "\345\210\225",
  /* 0xb3 0xb6 */  "\345\210\226",
  /* 0xb3 0xb7 */  "\345\210\230",
  /* 0xb3 0xb8 */  "\345\210\242",
  /* 0xb3 0xb9 */  "\345\210\250",
  /* 0xb3 0xba */  "\345\210\261",
  /* 0xb3 0xbb */  "\345\210\262",
  /* 0xb3 0xbc */  "\345\210\265",
  /* 0xb3 0xbd */  "\345\210\274",
  /* 0xb3 0xbe */  "\345\211\205",
  /* 0xb3 0xbf */  "\345\211\211",
  /* 0xb3 0xc0 */  "\345\211\225",
  /* 0xb3 0xc1 */  "\345\211\227",
  /* 0xb3 0xc2 */  "\345\211\230",
  /* 0xb3 0xc3 */  "\345\211\232",
  /* 0xb3 0xc4 */  "\345\211\234",
  /* 0xb3 0xc5 */  "\345\211\237",
  /* 0xb3 0xc6 */  "\345\211\240",
  /* 0xb3 0xc7 */  "\345\211\241",
  /* 0xb3 0xc8 */  "\345\211\246",
  /* 0xb3 0xc9 */  "\345\211\256",
  /* 0xb3 0xca */  "\345\211\267",
  /* 0xb3 0xcb */  "\345\211\270",
  /* 0xb3 0xcc */  "\345\211\271",
  /* 0xb3 0xcd */  "\345\212\200",
  /* 0xb3 0xce */  "\345\212\202",
  /* 0xb3 0xcf */  "\345\212\205",
  /* 0xb3 0xd0 */  "\345\212\212",
  /* 0xb3 0xd1 */  "\345\212\214",
  /* 0xb3 0xd2 */  "\345\212\223",
  /* 0xb3 0xd3 */  "\345\212\225",
  /* 0xb3 0xd4 */  "\345\212\226",
  /* 0xb3 0xd5 */  "\345\212\227",
  /* 0xb3 0xd6 */  "\345\212\230",
  /* 0xb3 0xd7 */  "\345\212\232",
  /* 0xb3 0xd8 */  "\345\212\234",
  /* 0xb3 0xd9 */  "\345\212\244",
  /* 0xb3 0xda */  "\345\212\245",
  /* 0xb3 0xdb */  "\345\212\246",
  /* 0xb3 0xdc */  "\345\212\247",
  /* 0xb3 0xdd */  "\345\212\257",
  /* 0xb3 0xde */  "\345\212\260",
  /* 0xb3 0xdf */  "\345\212\266",
  /* 0xb3 0xe0 */  "\345\212\267",
  /* 0xb3 0xe1 */  "\345\212\270",
  /* 0xb3 0xe2 */  "\345\212\272",
  /* 0xb3 0xe3 */  "\345\212\273",
  /* 0xb3 0xe4 */  "\345\212\275",
  /* 0xb3 0xe5 */  "\345\213\200",
  /* 0xb3 0xe6 */  "\345\213\204",
  /* 0xb3 0xe7 */  "\345\213\206",
  /* 0xb3 0xe8 */  "\345\213\210",
  /* 0xb3 0xe9 */  "\345\213\214",
  /* 0xb3 0xea */  "\345\213\217",
  /* 0xb3 0xeb */  "\345\213\221",
  /* 0xb3 0xec */  "\345\213\224",
  /* 0xb3 0xed */  "\345\213\226",
  /* 0xb3 0xee */  "\345\213\233",
  /* 0xb3 0xef */  "\345\213\234",
  /* 0xb3 0xf0 */  "\345\213\241",
  /* 0xb3 0xf1 */  "\345\213\245",
  /* 0xb3 0xf2 */  "\345\213\250",
  /* 0xb3 0xf3 */  "\345\213\251",
  /* 0xb3 0xf4 */  "\345\213\252",
  /* 0xb3 0xf5 */  "\345\213\254",
  /* 0xb3 0xf6 */  "\345\213\260",
  /* 0xb3 0xf7 */  "\345\213\261",
  /* 0xb3 0xf8 */  "\345\213\264",
  /* 0xb3 0xf9 */  "\345\213\266",
  /* 0xb3 0xfa */  "\345\213\267",
  /* 0xb3 0xfb */  "\345\214\200",
  /* 0xb3 0xfc */  "\345\214\203",
  /* 0xb3 0xfd */  "\345\214\212",
  /* 0xb3 0xfe */  "\345\214\213",
};


/* 0xb4, 0xa1-0xfe */
static const char *eucjp_to_utf8_table_0xb4[] = {
  /* 0xb4 0xa1 */  "\345\214\214",
  /* 0xb4 0xa2 */  "\345\214\221",
  /* 0xb4 0xa3 */  "\345\214\223",
  /* 0xb4 0xa4 */  "\345\214\230",
  /* 0xb4 0xa5 */  "\345\214\233",
  /* 0xb4 0xa6 */  "\345\214\234",
  /* 0xb4 0xa7 */  "\345\214\236",
  /* 0xb4 0xa8 */  "\345\214\237",
  /* 0xb4 0xa9 */  "\345\214\245",
  /* 0xb4 0xaa */  "\345\214\247",
  /* 0xb4 0xab */  "\345\214\250",
  /* 0xb4 0xac */  "\345\214\251",
  /* 0xb4 0xad */  "\345\214\253",
  /* 0xb4 0xae */  "\345\214\254",
  /* 0xb4 0xaf */  "\345\214\255",
  /* 0xb4 0xb0 */  "\345\214\260",
  /* 0xb4 0xb1 */  "\345\214\262",
  /* 0xb4 0xb2 */  "\345\214\265",
  /* 0xb4 0xb3 */  "\345\214\274",
  /* 0xb4 0xb4 */  "\345\214\275",
  /* 0xb4 0xb5 */  "\345\214\276",
  /* 0xb4 0xb6 */  "\345\215\202",
  /* 0xb4 0xb7 */  "\345\215\214",
  /* 0xb4 0xb8 */  "\345\215\213",
  /* 0xb4 0xb9 */  "\345\215\231",
  /* 0xb4 0xba */  "\345\215\233",
  /* 0xb4 0xbb */  "\345\215\241",
  /* 0xb4 0xbc */  "\345\215\243",
  /* 0xb4 0xbd */  "\345\215\245",
  /* 0xb4 0xbe */  "\345\215\254",
  /* 0xb4 0xbf */  "\345\215\255",
  /* 0xb4 0xc0 */  "\345\215\262",
  /* 0xb4 0xc1 */  "\345\215\271",
  /* 0xb4 0xc2 */  "\345\215\276",
  /* 0xb4 0xc3 */  "\345\216\203",
  /* 0xb4 0xc4 */  "\345\216\207",
  /* 0xb4 0xc5 */  "\345\216\210",
  /* 0xb4 0xc6 */  "\345\216\216",
  /* 0xb4 0xc7 */  "\345\216\223",
  /* 0xb4 0xc8 */  "\345\216\224",
  /* 0xb4 0xc9 */  "\345\216\231",
  /* 0xb4 0xca */  "\345\216\235",
  /* 0xb4 0xcb */  "\345\216\241",
  /* 0xb4 0xcc */  "\345\216\244",
  /* 0xb4 0xcd */  "\345\216\252",
  /* 0xb4 0xce */  "\345\216\253",
  /* 0xb4 0xcf */  "\345\216\257",
  /* 0xb4 0xd0 */  "\345\216\262",
  /* 0xb4 0xd1 */  "\345\216\264",
  /* 0xb4 0xd2 */  "\345\216\265",
  /* 0xb4 0xd3 */  "\345\216\267",
  /* 0xb4 0xd4 */  "\345\216\270",
  /* 0xb4 0xd5 */  "\345\216\272",
  /* 0xb4 0xd6 */  "\345\216\275",
  /* 0xb4 0xd7 */  "\345\217\200",
  /* 0xb4 0xd8 */  "\345\217\205",
  /* 0xb4 0xd9 */  "\345\217\217",
  /* 0xb4 0xda */  "\345\217\222",
  /* 0xb4 0xdb */  "\345\217\223",
  /* 0xb4 0xdc */  "\345\217\225",
  /* 0xb4 0xdd */  "\345\217\232",
  /* 0xb4 0xde */  "\345\217\235",
  /* 0xb4 0xdf */  "\345\217\236",
  /* 0xb4 0xe0 */  "\345\217\240",
  /* 0xb4 0xe1 */  "\345\217\246",
  /* 0xb4 0xe2 */  "\345\217\247",
  /* 0xb4 0xe3 */  "\345\217\265",
  /* 0xb4 0xe4 */  "\345\220\202",
  /* 0xb4 0xe5 */  "\345\220\223",
  /* 0xb4 0xe6 */  "\345\220\232",
  /* 0xb4 0xe7 */  "\345\220\241",
  /* 0xb4 0xe8 */  "\345\220\247",
  /* 0xb4 0xe9 */  "\345\220\250",
  /* 0xb4 0xea */  "\345\220\252",
  /* 0xb4 0xeb */  "\345\220\257",
  /* 0xb4 0xec */  "\345\220\261",
  /* 0xb4 0xed */  "\345\220\264",
  /* 0xb4 0xee */  "\345\220\265",
  /* 0xb4 0xef */  "\345\221\203",
  /* 0xb4 0xf0 */  "\345\221\204",
  /* 0xb4 0xf1 */  "\345\221\207",
  /* 0xb4 0xf2 */  "\345\221\215",
  /* 0xb4 0xf3 */  "\345\221\217",
  /* 0xb4 0xf4 */  "\345\221\236",
  /* 0xb4 0xf5 */  "\345\221\242",
  /* 0xb4 0xf6 */  "\345\221\244",
  /* 0xb4 0xf7 */  "\345\221\246",
  /* 0xb4 0xf8 */  "\345\221\247",
  /* 0xb4 0xf9 */  "\345\221\251",
  /* 0xb4 0xfa */  "\345\221\253",
  /* 0xb4 0xfb */  "\345\221\255",
  /* 0xb4 0xfc */  "\345\221\256",
  /* 0xb4 0xfd */  "\345\221\264",
  /* 0xb4 0xfe */  "\345\221\277",
};


/* 0xb5, 0xa1-0xfe */
static const char *eucjp_to_utf8_table_0xb5[] = {
  /* 0xb5 0xa1 */  "\345\222\201",
  /* 0xb5 0xa2 */  "\345\222\203",
  /* 0xb5 0xa3 */  "\345\222\205",
  /* 0xb5 0xa4 */  "\345\222\210",
  /* 0xb5 0xa5 */  "\345\222\211",
  /* 0xb5 0xa6 */  "\345\222\215",
  /* 0xb5 0xa7 */  "\345\222\221",
  /* 0xb5 0xa8 */  "\345\222\225",
  /* 0xb5 0xa9 */  "\345\222\226",
  /* 0xb5 0xaa */  "\345\222\234",
  /* 0xb5 0xab */  "\345\222\237",
  /* 0xb5 0xac */  "\345\222\241",
  /* 0xb5 0xad */  "\345\222\246",
  /* 0xb5 0xae */  "\345\222\247",
  /* 0xb5 0xaf */  "\345\222\251",
  /* 0xb5 0xb0 */  "\345\222\252",
  /* 0xb5 0xb1 */  "\345\222\255",
  /* 0xb5 0xb2 */  "\345\222\256",
  /* 0xb5 0xb3 */  "\345\222\261",
  /* 0xb5 0xb4 */  "\345\222\267",
  /* 0xb5 0xb5 */  "\345\222\271",
  /* 0xb5 0xb6 */  "\345\222\272",
  /* 0xb5 0xb7 */  "\345\222\273",
  /* 0xb5 0xb8 */  "\345\222\277",
  /* 0xb5 0xb9 */  "\345\223\206",
  /* 0xb5 0xba */  "\345\223\212",
  /* 0xb5 0xbb */  "\345\223\215",
  /* 0xb5 0xbc */  "\345\223\216",
  /* 0xb5 0xbd */  "\345\223\240",
  /* 0xb5 0xbe */  "\345\223\252",
  /* 0xb5 0xbf */  "\345\223\254",
  /* 0xb5 0xc0 */  "\345\223\257",
  /* 0xb5 0xc1 */  "\345\223\266",
  /* 0xb5 0xc2 */  "\345\223\274",
  /* 0xb5 0xc3 */  "\345\223\276",
  /* 0xb5 0xc4 */  "\345\223\277",
  /* 0xb5 0xc5 */  "\345\224\200",
  /* 0xb5 0xc6 */  "\345\224\201",
  /* 0xb5 0xc7 */  "\345\224\205",
  /* 0xb5 0xc8 */  "\345\224\210",
  /* 0xb5 0xc9 */  "\345\224\211",
  /* 0xb5 0xca */  "\345\224\214",
  /* 0xb5 0xcb */  "\345\224\215",
  /* 0xb5 0xcc */  "\345\224\216",
  /* 0xb5 0xcd */  "\345\224\225",
  /* 0xb5 0xce */  "\345\224\252",
  /* 0xb5 0xcf */  "\345\224\253",
  /* 0xb5 0xd0 */  "\345\224\262",
  /* 0xb5 0xd1 */  "\345\224\265",
  /* 0xb5 0xd2 */  "\345\224\266",
  /* 0xb5 0xd3 */  "\345\224\273",
  /* 0xb5 0xd4 */  "\345\224\274",
  /* 0xb5 0xd5 */  "\345\224\275",
  /* 0xb5 0xd6 */  "\345\225\201",
  /* 0xb5 0xd7 */  "\345\225\207",
  /* 0xb5 0xd8 */  "\345\225\211",
  /* 0xb5 0xd9 */  "\345\225\212",
  /* 0xb5 0xda */  "\345\225\215",
  /* 0xb5 0xdb */  "\345\225\220",
  /* 0xb5 0xdc */  "\345\225\221",
  /* 0xb5 0xdd */  "\345\225\230",
  /* 0xb5 0xde */  "\345\225\232",
  /* 0xb5 0xdf */  "\345\225\233",
  /* 0xb5 0xe0 */  "\345\225\236",
  /* 0xb5 0xe1 */  "\345\225\240",
  /* 0xb5 0xe2 */  "\345\225\241",
  /* 0xb5 0xe3 */  "\345\225\244",
  /* 0xb5 0xe4 */  "\345\225\246",
  /* 0xb5 0xe5 */  "\345\225\277",
  /* 0xb5 0xe6 */  "\345\226\201",
  /* 0xb5 0xe7 */  "\345\226\202",
  /* 0xb5 0xe8 */  "\345\226\206",
  /* 0xb5 0xe9 */  "\345\226\210",
  /* 0xb5 0xea */  "\345\226\216",
  /* 0xb5 0xeb */  "\345\226\217",
  /* 0xb5 0xec */  "\345\226\221",
  /* 0xb5 0xed */  "\345\226\222",
  /* 0xb5 0xee */  "\345\226\223",
  /* 0xb5 0xef */  "\345\226\224",
  /* 0xb5 0xf0 */  "\345\226\227",
  /* 0xb5 0xf1 */  "\345\226\243",
  /* 0xb5 0xf2 */  "\345\226\244",
  /* 0xb5 0xf3 */  "\345\226\255",
  /* 0xb5 0xf4 */  "\345\226\262",
  /* 0xb5 0xf5 */  "\345\226\277",
  /* 0xb5 0xf6 */  "\345\227\201",
  /* 0xb5 0xf7 */  "\345\227\203",
  /* 0xb5 0xf8 */  "\345\227\206",
  /* 0xb5 0xf9 */  "\345\227\211",
  /* 0xb5 0xfa */  "\345\227\213",
  /* 0xb5 0xfb */  "\345\227\214",
  /* 0xb5 0xfc */  "\345\227\216",
  /* 0xb5 0xfd */  "\345\227\221",
  /* 0xb5 0xfe */  "\345\227\222",
};


/* 0xb6, 0xa1-0xb6 */
static const char *eucjp_to_utf8_table_0xb6[] = {
  /* 0xb6 0xa1 */  "\345\227\223",
  /* 0xb6 0xa2 */  "\345\227\227",
  /* 0xb6 0xa3 */  "\345\227\230",
  /* 0xb6 0xa4 */  "\345\227\233",
  /* 0xb6 0xa5 */  "\345\227\236",
  /* 0xb6 0xa6 */  "\345\227\242",
  /* 0xb6 0xa7 */  "\345\227\251",
  /* 0xb6 0xa8 */  "\345\227\266",
  /* 0xb6 0xa9 */  "\345\227\277",
  /* 0xb6 0xaa */  "\345\230\205",
  /* 0xb6 0xab */  "\345\230\210",
  /* 0xb6 0xac */  "\345\230\212",
  /* 0xb6 0xad */  "\345\230\215",
  /* 0xb6 0xae */  "\345\230\216",
  /* 0xb6 0xaf */  "\345\230\217",
  /* 0xb6 0xb0 */  "\345\230\220",
  /* 0xb6 0xb1 */  "\345\230\221",
  /* 0xb6 0xb2 */  "\345\230\222",
  /* 0xb6 0xb3 */  "\345\230\231",
  /* 0xb6 0xb4 */  "\345\230\254",
  /* 0xb6 0xb5 */  "\345\230\260",
  /* 0xb6 0xb6 */  "\126",
};


/* 0xf3, 0xf3-0xfe */
static const char *eucjp_to_utf8_table_0xf3[] = {
  /* 0xf3 0xf3 */  "\342\205\260",
  /* 0xf3 0xf4 */  "\342\205\261",
  /* 0xf3 0xf5 */  "\342\205\262",
  /* 0xf3 0xf6 */  "\342\205\263",
  /* 0xf3 0xf7 */  "\342\205\264",
  /* 0xf3 0xf8 */  "\342\205\265",
  /* 0xf3 0xf9 */  "\342\205\266",
  /* 0xf3 0xfa */  "\342\205\267",
  /* 0xf3 0xfb */  "\342\205\270",
  /* 0xf3 0xfc */  "\342\205\271",
  /* 0xf3 0xfd */  "\342\205\240",
  /* 0xf3 0xfe */  "\342\205\241",
};


/* 0xf4, 0xa1-0xfe */
static const char *eucjp_to_utf8_table_0xf4[] = {
  /* 0xf4 0xa1 */  "\342\205\242",
  /* 0xf4 0xa2 */  "\342\205\243",
  /* 0xf4 0xa3 */  "\342\205\244",
  /* 0xf4 0xa4 */  "\342\205\245",
  /* 0xf4 0xa5 */  "\342\205\246",
  /* 0xf4 0xa6 */  "\342\205\247",
  /* 0xf4 0xa7 */  "\342\205\250",
  /* 0xf4 0xa8 */  "\342\205\251",
  /* 0xf4 0xa9 */  "\357\274\207",
  /* 0xf4 0xaa */  "\357\274\202",
  /* 0xf4 0xab */  "\343\210\261",
  /* 0xf4 0xac */  "\342\204\226",
  /* 0xf4 0xad */  "\342\204\241",
  /* 0xf4 0xae */  "\347\202\273",
  /* 0xf4 0xaf */  "\344\273\274",
  /* 0xf4 0xb0 */  "\345\203\264",
  /* 0xf4 0xb1 */  "\345\207\254",
  /* 0xf4 0xb2 */  "\345\214\207",
  /* 0xf4 0xb3 */  "\345\214\244",
  /* 0xf4 0xb4 */  "\357\250\216",
  /* 0xf4 0xb5 */  "\345\222\212",
  /* 0xf4 0xb6 */  "\345\235\231",
  /* 0xf4 0xb7 */  "\357\250\217",
  /* 0xf4 0xb8 */  "\357\250\220",
  /* 0xf4 0xb9 */  "\345\242\236",
  /* 0xf4 0xba */  "\345\257\254",
  /* 0xf4 0xbb */  "\345\263\265",
  /* 0xf4 0xbc */  "\345\265\223",
  /* 0xf4 0xbd */  "\357\250\221",
  /* 0xf4 0xbe */  "\345\276\267",
  /* 0xf4 0xbf */  "\346\202\205",
  /* 0xf4 0xc0 */  "\346\204\240",
  /* 0xf4 0xc1 */  "\346\225\216",
  /* 0xf4 0xc2 */  "\346\230\273",
  /* 0xf4 0xc3 */  "\346\231\245",
  /* 0xf4 0xc4 */  "\357\250\222",
  /* 0xf4 0xc5 */  "\357\244\251",
  /* 0xf4 0xc6 */  "\346\240\201",
  /* 0xf4 0xc7 */  "\357\250\223",
  /* 0xf4 0xc8 */  "\357\250\224",
  /* 0xf4 0xc9 */  "\346\251\253",
  /* 0xf4 0xca */  "\346\253\242",
  /* 0xf4 0xcb */  "\346\267\270",
  /* 0xf4 0xcc */  "\346\267\262",
  /* 0xf4 0xcd */  "\347\200\250",
  /* 0xf4 0xce */  "\357\250\225",
  /* 0xf4 0xcf */  "\357\250\226",
  /* 0xf4 0xd0 */  "\347\224\201",
  /* 0xf4 0xd1 */  "\347\232\202",
  /* 0xf4 0xd2 */  "\347\232\236",
  /* 0xf4 0xd3 */  "\357\250\227",
  /* 0xf4 0xd4 */  "\347\244\260",
  /* 0xf4 0xd5 */  "\357\250\230",
  /* 0xf4 0xd6 */  "\357\250\231",
  /* 0xf4 0xd7 */  "\357\250\232",
  /* 0xf4 0xd8 */  "\357\250\233",
  /* 0xf4 0xd9 */  "\347\253\247",
  /* 0xf4 0xda */  "\357\250\234",
  /* 0xf4 0xdb */  "\357\250\235",
  /* 0xf4 0xdc */  "\347\266\240",
  /* 0xf4 0xdd */  "\347\267\226",
  /* 0xf4 0xde */  "\357\250\236",
  /* 0xf4 0xdf */  "\350\215\242",
  /* 0xf4 0xe0 */  "\357\250\237",
  /* 0xf4 0xe1 */  "\350\226\260",
  /* 0xf4 0xe2 */  "\357\250\240",
  /* 0xf4 0xe3 */  "\357\250\241",
  /* 0xf4 0xe4 */  "\350\240\207",
  /* 0xf4 0xe5 */  "\357\250\242",
  /* 0xf4 0xe6 */  "\350\255\277",
  /* 0xf4 0xe7 */  "\350\263\264",
  /* 0xf4 0xe8 */  "\350\265\266",
  /* 0xf4 0xe9 */  "\357\250\243",
  /* 0xf4 0xea */  "\357\250\244",
  /* 0xf4 0xeb */  "\357\250\245",
  /* 0xf4 0xec */  "\351\203\236",
  /* 0xf4 0xed */  "\357\250\246",
  /* 0xf4 0xee */  "\351\204\225",
  /* 0xf4 0xef */  "\357\250\247",
  /* 0xf4 0xf0 */  "\357\250\250",
  /* 0xf4 0xf1 */  "\351\226\222",
  /* 0xf4 0xf2 */  "\357\247\234",
  /* 0xf4 0xf3 */  "\357\250\251",
  /* 0xf4 0xf4 */  "\351\234\273",
  /* 0xf4 0xf5 */  "\351\235\215",
  /* 0xf4 0xf6 */  "\351\235\221",
  /* 0xf4 0xf7 */  "\357\250\252",
  /* 0xf4 0xf8 */  "\357\250\253",
  /* 0xf4 0xf9 */  "\357\250\254",
  /* 0xf4 0xfa */  "\351\246\236",
  /* 0xf4 0xfb */  "\351\253\231",
  /* 0xf4 0xfc */  "\351\255\262",
  /* 0xf4 0xfd */  "\357\250\255",
  /* 0xf4 0xfe */  "\351\273\221",
};


/* 0xf5, 0xa1-0xfe */
static const char *eucjp_to_utf8_table_0xf5[] = {
  /* 0xf5 0xa1 */  "\356\200\200",
  /* 0xf5 0xa2 */  "\356\200\201",
  /* 0xf5 0xa3 */  "\356\200\202",
  /* 0xf5 0xa4 */  "\356\200\203",
  /* 0xf5 0xa5 */  "\356\200\204",
  /* 0xf5 0xa6 */  "\356\200\205",
  /* 0xf5 0xa7 */  "\356\200\206",
  /* 0xf5 0xa8 */  "\356\200\207",
  /* 0xf5 0xa9 */  "\356\200\210",
  /* 0xf5 0xaa */  "\356\200\211",
  /* 0xf5 0xab */  "\356\200\212",
  /* 0xf5 0xac */  "\356\200\213",
  /* 0xf5 0xad */  "\356\200\214",
  /* 0xf5 0xae */  "\356\200\215",
  /* 0xf5 0xaf */  "\356\200\216",
  /* 0xf5 0xb0 */  "\356\200\217",
  /* 0xf5 0xb1 */  "\356\200\220",
  /* 0xf5 0xb2 */  "\356\200\221",
  /* 0xf5 0xb3 */  "\356\200\222",
  /* 0xf5 0xb4 */  "\356\200\223",
  /* 0xf5 0xb5 */  "\356\200\224",
  /* 0xf5 0xb6 */  "\356\200\225",
  /* 0xf5 0xb7 */  "\356\200\226",
  /* 0xf5 0xb8 */  "\356\200\227",
  /* 0xf5 0xb9 */  "\356\200\230",
  /* 0xf5 0xba */  "\356\200\231",
  /* 0xf5 0xbb */  "\356\200\232",
  /* 0xf5 0xbc */  "\356\200\233",
  /* 0xf5 0xbd */  "\356\200\234",
  /* 0xf5 0xbe */  "\356\200\235",
  /* 0xf5 0xbf */  "\356\200\236",
  /* 0xf5 0xc0 */  "\356\200\237",
  /* 0xf5 0xc1 */  "\356\200\240",
  /* 0xf5 0xc2 */  "\356\200\241",
  /* 0xf5 0xc3 */  "\356\200\242",
  /* 0xf5 0xc4 */  "\356\200\243",
  /* 0xf5 0xc5 */  "\356\200\244",
  /* 0xf5 0xc6 */  "\356\200\245",
  /* 0xf5 0xc7 */  "\356\200\246",
  /* 0xf5 0xc8 */  "\356\200\247",
  /* 0xf5 0xc9 */  "\356\200\250",
  /* 0xf5 0xca */  "\356\200\251",
  /* 0xf5 0xcb */  "\356\200\252",
  /* 0xf5 0xcc */  "\356\200\253",
  /* 0xf5 0xcd */  "\356\200\254",
  /* 0xf5 0xce */  "\356\200\255",
  /* 0xf5 0xcf */  "\356\200\256",
  /* 0xf5 0xd0 */  "\356\200\257",
  /* 0xf5 0xd1 */  "\356\200\260",
  /* 0xf5 0xd2 */  "\356\200\261",
  /* 0xf5 0xd3 */  "\356\200\262",
  /* 0xf5 0xd4 */  "\356\200\263",
  /* 0xf5 0xd5 */  "\356\200\264",
  /* 0xf5 0xd6 */  "\356\200\265",
  /* 0xf5 0xd7 */  "\356\200\266",
  /* 0xf5 0xd8 */  "\356\200\267",
  /* 0xf5 0xd9 */  "\356\200\270",
  /* 0xf5 0xda */  "\356\200\271",
  /* 0xf5 0xdb */  "\356\200\272",
  /* 0xf5 0xdc */  "\356\200\273",
  /* 0xf5 0xdd */  "\356\200\274",
  /* 0xf5 0xde */  "\356\200\275",
  /* 0xf5 0xdf */  "\356\200\276",
  /* 0xf5 0xe0 */  "\356\200\277",
  /* 0xf5 0xe1 */  "\356\201\200",
  /* 0xf5 0xe2 */  "\356\201\201",
  /* 0xf5 0xe3 */  "\356\201\202",
  /* 0xf5 0xe4 */  "\356\201\203",
  /* 0xf5 0xe5 */  "\356\201\204",
  /* 0xf5 0xe6 */  "\356\201\205",
  /* 0xf5 0xe7 */  "\356\201\206",
  /* 0xf5 0xe8 */  "\356\201\207",
  /* 0xf5 0xe9 */  "\356\201\210",
  /* 0xf5 0xea */  "\356\201\211",
  /* 0xf5 0xeb */  "\356\201\212",
  /* 0xf5 0xec */  "\356\201\213",
  /* 0xf5 0xed */  "\356\201\214",
  /* 0xf5 0xee */  "\356\201\215",
  /* 0xf5 0xef */  "\356\201\216",
  /* 0xf5 0xf0 */  "\356\201\217",
  /* 0xf5 0xf1 */  "\356\201\220",
  /* 0xf5 0xf2 */  "\356\201\221",
  /* 0xf5 0xf3 */  "\356\201\222",
  /* 0xf5 0xf4 */  "\356\201\223",
  /* 0xf5 0xf5 */  "\356\201\224",
  /* 0xf5 0xf6 */  "\356\201\225",
  /* 0xf5 0xf7 */  "\356\201\226",
  /* 0xf5 0xf8 */  "\356\201\227",
  /* 0xf5 0xf9 */  "\356\201\230",
  /* 0xf5 0xfa */  "\356\201\231",
  /* 0xf5 0xfb */  "\356\201\232",
  /* 0xf5 0xfc */  "\356\201\233",
  /* 0xf5 0xfd */  "\356\201\234",
  /* 0xf5 0xfe */  "\356\201\235",
};


/* 0xf6, 0xa1-0xfe */
static const char *eucjp_to_utf8_table_0xf6[] = {
  /* 0xf6 0xa1 */  "\356\201\236",
  /* 0xf6 0xa2 */  "\356\201\237",
  /* 0xf6 0xa3 */  "\356\201\240",
  /* 0xf6 0xa4 */  "\356\201\241",
  /* 0xf6 0xa5 */  "\356\201\242",
  /* 0xf6 0xa6 */  "\356\201\243",
  /* 0xf6 0xa7 */  "\356\201\244",
  /* 0xf6 0xa8 */  "\356\201\245",
  /* 0xf6 0xa9 */  "\356\201\246",
  /* 0xf6 0xaa */  "\356\201\247",
  /* 0xf6 0xab */  "\356\201\250",
  /* 0xf6 0xac */  "\356\201\251",
  /* 0xf6 0xad */  "\356\201\252",
  /* 0xf6 0xae */  "\356\201\253",
  /* 0xf6 0xaf */  "\356\201\254",
  /* 0xf6 0xb0 */  "\356\201\255",
  /* 0xf6 0xb1 */  "\356\201\256",
  /* 0xf6 0xb2 */  "\356\201\257",
  /* 0xf6 0xb3 */  "\356\201\260",
  /* 0xf6 0xb4 */  "\356\201\261",
  /* 0xf6 0xb5 */  "\356\201\262",
  /* 0xf6 0xb6 */  "\356\201\263",
  /* 0xf6 0xb7 */  "\356\201\264",
  /* 0xf6 0xb8 */  "\356\201\265",
  /* 0xf6 0xb9 */  "\356\201\266",
  /* 0xf6 0xba */  "\356\201\267",
  /* 0xf6 0xbb */  "\356\201\270",
  /* 0xf6 0xbc */  "\356\201\271",
  /* 0xf6 0xbd */  "\356\201\272",
  /* 0xf6 0xbe */  "\356\201\273",
  /* 0xf6 0xbf */  "\356\201\274",
  /* 0xf6 0xc0 */  "\356\201\275",
  /* 0xf6 0xc1 */  "\356\201\276",
  /* 0xf6 0xc2 */  "\356\201\277",
  /* 0xf6 0xc3 */  "\356\202\200",
  /* 0xf6 0xc4 */  "\356\202\201",
  /* 0xf6 0xc5 */  "\356\202\202",
  /* 0xf6 0xc6 */  "\356\202\203",
  /* 0xf6 0xc7 */  "\356\202\204",
  /* 0xf6 0xc8 */  "\356\202\205",
  /* 0xf6 0xc9 */  "\356\202\206",
  /* 0xf6 0xca */  "\356\202\207",
  /* 0xf6 0xcb */  "\356\202\210",
  /* 0xf6 0xcc */  "\356\202\211",
  /* 0xf6 0xcd */  "\356\202\212",
  /* 0xf6 0xce */  "\356\202\213",
  /* 0xf6 0xcf */  "\356\202\214",
  /* 0xf6 0xd0 */  "\356\202\215",
  /* 0xf6 0xd1 */  "\356\202\216",
  /* 0xf6 0xd2 */  "\356\202\217",
  /* 0xf6 0xd3 */  "\356\202\220",
  /* 0xf6 0xd4 */  "\356\202\221",
  /* 0xf6 0xd5 */  "\356\202\222",
  /* 0xf6 0xd6 */  "\356\202\223",
  /* 0xf6 0xd7 */  "\356\202\224",
  /* 0xf6 0xd8 */  "\356\202\225",
  /* 0xf6 0xd9 */  "\356\202\226",
  /* 0xf6 0xda */  "\356\202\227",
  /* 0xf6 0xdb */  "\356\202\230",
  /* 0xf6 0xdc */  "\356\202\231",
  /* 0xf6 0xdd */  "\356\202\232",
  /* 0xf6 0xde */  "\356\202\233",
  /* 0xf6 0xdf */  "\356\202\234",
  /* 0xf6 0xe0 */  "\356\202\235",
  /* 0xf6 0xe1 */  "\356\202\236",
  /* 0xf6 0xe2 */  "\356\202\237",
  /* 0xf6 0xe3 */  "\356\202\240",
  /* 0xf6 0xe4 */  "\356\202\241",
  /* 0xf6 0xe5 */  "\356\202\242",
  /* 0xf6 0xe6 */  "\356\202\243",
  /* 0xf6 0xe7 */  "\356\202\244",
  /* 0xf6 0xe8 */  "\356\202\245",
  /* 0xf6 0xe9 */  "\356\202\246",
  /* 0xf6 0xea */  "\356\202\247",
  /* 0xf6 0xeb */  "\356\202\250",
  /* 0xf6 0xec */  "\356\202\251",
  /* 0xf6 0xed */  "\356\202\252",
  /* 0xf6 0xee */  "\356\202\253",
  /* 0xf6 0xef */  "\356\202\254",
  /* 0xf6 0xf0 */  "\356\202\255",
  /* 0xf6 0xf1 */  "\356\202\256",
  /* 0xf6 0xf2 */  "\356\202\257",
  /* 0xf6 0xf3 */  "\356\202\260",
  /* 0xf6 0xf4 */  "\356\202\261",
  /* 0xf6 0xf5 */  "\356\202\262",
  /* 0xf6 0xf6 */  "\356\202\263",
  /* 0xf6 0xf7 */  "\356\202\264",
  /* 0xf6 0xf8 */  "\356\202\265",
  /* 0xf6 0xf9 */  "\356\202\266",
  /* 0xf6 0xfa */  "\356\202\267",
  /* 0xf6 0xfb */  "\356\202\270",
  /* 0xf6 0xfc */  "\356\202\271",
  /* 0xf6 0xfd */  "\356\202\272",
  /* 0xf6 0xfe */  "\356\202\273",
};


/* 0xf7, 0xa1-0xfe */
static const char *eucjp_to_utf8_table_0xf7[] = {
  /* 0xf7 0xa1 */  "\356\202\274",
  /* 0xf7 0xa2 */  "\356\202\275",
  /* 0xf7 0xa3 */  "\356\202\276",
  /* 0xf7 0xa4 */  "\356\202\277",
  /* 0xf7 0xa5 */  "\356\203\200",
  /* 0xf7 0xa6 */  "\356\203\201",
  /* 0xf7 0xa7 */  "\356\203\202",
  /* 0xf7 0xa8 */  "\356\203\203",
  /* 0xf7 0xa9 */  "\356\203\204",
  /* 0xf7 0xaa */  "\356\203\205",
  /* 0xf7 0xab */  "\356\203\206",
  /* 0xf7 0xac */  "\356\203\207",
  /* 0xf7 0xad */  "\356\203\210",
  /* 0xf7 0xae */  "\356\203\211",
  /* 0xf7 0xaf */  "\356\203\212",
  /* 0xf7 0xb0 */  "\356\203\213",
  /* 0xf7 0xb1 */  "\356\203\214",
  /* 0xf7 0xb2 */  "\356\203\215",
  /* 0xf7 0xb3 */  "\356\203\216",
  /* 0xf7 0xb4 */  "\356\203\217",
  /* 0xf7 0xb5 */  "\356\203\220",
  /* 0xf7 0xb6 */  "\356\203\221",
  /* 0xf7 0xb7 */  "\356\203\222",
  /* 0xf7 0xb8 */  "\356\203\223",
  /* 0xf7 0xb9 */  "\356\203\224",
  /* 0xf7 0xba */  "\356\203\225",
  /* 0xf7 0xbb */  "\356\203\226",
  /* 0xf7 0xbc */  "\356\203\227",
  /* 0xf7 0xbd */  "\356\203\230",
  /* 0xf7 0xbe */  "\356\203\231",
  /* 0xf7 0xbf */  "\356\203\232",
  /* 0xf7 0xc0 */  "\356\203\233",
  /* 0xf7 0xc1 */  "\356\203\234",
  /* 0xf7 0xc2 */  "\356\203\235",
  /* 0xf7 0xc3 */  "\356\203\236",
  /* 0xf7 0xc4 */  "\356\203\237",
  /* 0xf7 0xc5 */  "\356\203\240",
  /* 0xf7 0xc6 */  "\356\203\241",
  /* 0xf7 0xc7 */  "\356\203\242",
  /* 0xf7 0xc8 */  "\356\203\243",
  /* 0xf7 0xc9 */  "\356\203\244",
  /* 0xf7 0xca */  "\356\203\245",
  /* 0xf7 0xcb */  "\356\203\246",
  /* 0xf7 0xcc */  "\356\203\247",
  /* 0xf7 0xcd */  "\356\203\250",
  /* 0xf7 0xce */  "\356\203\251",
  /* 0xf7 0xcf */  "\356\203\252",
  /* 0xf7 0xd0 */  "\356\203\253",
  /* 0xf7 0xd1 */  "\356\203\254",
  /* 0xf7 0xd2 */  "\356\203\255",
  /* 0xf7 0xd3 */  "\356\203\256",
  /* 0xf7 0xd4 */  "\356\203\257",
  /* 0xf7 0xd5 */  "\356\203\260",
  /* 0xf7 0xd6 */  "\356\203\261",
  /* 0xf7 0xd7 */  "\356\203\262",
  /* 0xf7 0xd8 */  "\356\203\263",
  /* 0xf7 0xd9 */  "\356\203\264",
  /* 0xf7 0xda */  "\356\203\265",
  /* 0xf7 0xdb */  "\356\203\266",
  /* 0xf7 0xdc */  "\356\203\267",
  /* 0xf7 0xdd */  "\356\203\270",
  /* 0xf7 0xde */  "\356\203\271",
  /* 0xf7 0xdf */  "\356\203\272",
  /* 0xf7 0xe0 */  "\356\203\273",
  /* 0xf7 0xe1 */  "\356\203\274",
  /* 0xf7 0xe2 */  "\356\203\275",
  /* 0xf7 0xe3 */  "\356\203\276",
  /* 0xf7 0xe4 */  "\356\203\277",
  /* 0xf7 0xe5 */  "\356\204\200",
  /* 0xf7 0xe6 */  "\356\204\201",
  /* 0xf7 0xe7 */  "\356\204\202",
  /* 0xf7 0xe8 */  "\356\204\203",
  /* 0xf7 0xe9 */  "\356\204\204",
  /* 0xf7 0xea */  "\356\204\205",
  /* 0xf7 0xeb */  "\356\204\206",
  /* 0xf7 0xec */  "\356\204\207",
  /* 0xf7 0xed */  "\356\204\210",
  /* 0xf7 0xee */  "\356\204\211",
  /* 0xf7 0xef */  "\356\204\212",
  /* 0xf7 0xf0 */  "\356\204\213",
  /* 0xf7 0xf1 */  "\356\204\214",
  /* 0xf7 0xf2 */  "\356\204\215",
  /* 0xf7 0xf3 */  "\356\204\216",
  /* 0xf7 0xf4 */  "\356\204\217",
  /* 0xf7 0xf5 */  "\356\204\220",
  /* 0xf7 0xf6 */  "\356\204\221",
  /* 0xf7 0xf7 */  "\356\204\222",
  /* 0xf7 0xf8 */  "\356\204\223",
  /* 0xf7 0xf9 */  "\356\204\224",
  /* 0xf7 0xfa */  "\356\204\225",
  /* 0xf7 0xfb */  "\356\204\226",
  /* 0xf7 0xfc */  "\356\204\227",
  /* 0xf7 0xfd */  "\356\204\230",
  /* 0xf7 0xfe */  "\356\204\231",
};


/* 0xf8, 0xa1-0xfe */
static const char *eucjp_to_utf8_table_0xf8[] = {
  /* 0xf8 0xa1 */  "\356\204\232",
  /* 0xf8 0xa2 */  "\356\204\233",
  /* 0xf8 0xa3 */  "\356\204\234",
  /* 0xf8 0xa4 */  "\356\204\235",
  /* 0xf8 0xa5 */  "\356\204\236",
  /* 0xf8 0xa6 */  "\356\204\237",
  /* 0xf8 0xa7 */  "\356\204\240",
  /* 0xf8 0xa8 */  "\356\204\241",
  /* 0xf8 0xa9 */  "\356\204\242",
  /* 0xf8 0xaa */  "\356\204\243",
  /* 0xf8 0xab */  "\356\204\244",
  /* 0xf8 0xac */  "\356\204\245",
  /* 0xf8 0xad */  "\356\204\246",
  /* 0xf8 0xae */  "\356\204\247",
  /* 0xf8 0xaf */  "\356\204\250",
  /* 0xf8 0xb0 */  "\356\204\251",
  /* 0xf8 0xb1 */  "\356\204\252",
  /* 0xf8 0xb2 */  "\356\204\253",
  /* 0xf8 0xb3 */  "\356\204\254",
  /* 0xf8 0xb4 */  "\356\204\255",
  /* 0xf8 0xb5 */  "\356\204\256",
  /* 0xf8 0xb6 */  "\356\204\257",
  /* 0xf8 0xb7 */  "\356\204\260",
  /* 0xf8 0xb8 */  "\356\204\261",
  /* 0xf8 0xb9 */  "\356\204\262",
  /* 0xf8 0xba */  "\356\204\263",
  /* 0xf8 0xbb */  "\356\204\264",
  /* 0xf8 0xbc */  "\356\204\265",
  /* 0xf8 0xbd */  "\356\204\266",
  /* 0xf8 0xbe */  "\356\204\267",
  /* 0xf8 0xbf */  "\356\204\270",
  /* 0xf8 0xc0 */  "\356\204\271",
  /* 0xf8 0xc1 */  "\356\204\272",
  /* 0xf8 0xc2 */  "\356\204\273",
  /* 0xf8 0xc3 */  "\356\204\274",
  /* 0xf8 0xc4 */  "\356\204\275",
  /* 0xf8 0xc5 */  "\356\204\276",
  /* 0xf8 0xc6 */  "\356\204\277",
  /* 0xf8 0xc7 */  "\356\205\200",
  /* 0xf8 0xc8 */  "\356\205\201",
  /* 0xf8 0xc9 */  "\356\205\202",
  /* 0xf8 0xca */  "\356\205\203",
  /* 0xf8 0xcb */  "\356\205\204",
  /* 0xf8 0xcc */  "\356\205\205",
  /* 0xf8 0xcd */  "\356\205\206",
  /* 0xf8 0xce */  "\356\205\207",
  /* 0xf8 0xcf */  "\356\205\210",
  /* 0xf8 0xd0 */  "\356\205\211",
  /* 0xf8 0xd1 */  "\356\205\212",
  /* 0xf8 0xd2 */  "\356\205\213",
  /* 0xf8 0xd3 */  "\356\205\214",
  /* 0xf8 0xd4 */  "\356\205\215",
  /* 0xf8 0xd5 */  "\356\205\216",
  /* 0xf8 0xd6 */  "\356\205\217",
  /* 0xf8 0xd7 */  "\356\205\220",
  /* 0xf8 0xd8 */  "\356\205\221",
  /* 0xf8 0xd9 */  "\356\205\222",
  /* 0xf8 0xda */  "\356\205\223",
  /* 0xf8 0xdb */  "\356\205\224",
  /* 0xf8 0xdc */  "\356\205\225",
  /* 0xf8 0xdd */  "\356\205\226",
  /* 0xf8 0xde */  "\356\205\227",
  /* 0xf8 0xdf */  "\356\205\230",
  /* 0xf8 0xe0 */  "\356\205\231",
  /* 0xf8 0xe1 */  "\356\205\232",
  /* 0xf8 0xe2 */  "\356\205\233",
  /* 0xf8 0xe3 */  "\356\205\234",
  /* 0xf8 0xe4 */  "\356\205\235",
  /* 0xf8 0xe5 */  "\356\205\236",
  /* 0xf8 0xe6 */  "\356\205\237",
  /* 0xf8 0xe7 */  "\356\205\240",
  /* 0xf8 0xe8 */  "\356\205\241",
  /* 0xf8 0xe9 */  "\356\205\242",
  /* 0xf8 0xea */  "\356\205\243",
  /* 0xf8 0xeb */  "\356\205\244",
  /* 0xf8 0xec */  "\356\205\245",
  /* 0xf8 0xed */  "\356\205\246",
  /* 0xf8 0xee */  "\356\205\247",
  /* 0xf8 0xef */  "\356\205\250",
  /* 0xf8 0xf0 */  "\356\205\251",
  /* 0xf8 0xf1 */  "\356\205\252",
  /* 0xf8 0xf2 */  "\356\205\253",
  /* 0xf8 0xf3 */  "\356\205\254",
  /* 0xf8 0xf4 */  "\356\205\255",
  /* 0xf8 0xf5 */  "\356\205\256",
  /* 0xf8 0xf6 */  "\356\205\257",
  /* 0xf8 0xf7 */  "\356\205\260",
  /* 0xf8 0xf8 */  "\356\205\261",
  /* 0xf8 0xf9 */  "\356\205\262",
  /* 0xf8 0xfa */  "\356\205\263",
  /* 0xf8 0xfb */  "\356\205\264",
  /* 0xf8 0xfc */  "\356\205\265",
  /* 0xf8 0xfd */  "\356\205\266",
  /* 0xf8 0xfe */  "\356\205\267",
};


/* 0xf9, 0xa1-0xfe */
static const char *eucjp_to_utf8_table_0xf9[] = {
  /* 0xf9 0xa1 */  "\356\205\270",
  /* 0xf9 0xa2 */  "\356\205\271",
  /* 0xf9 0xa3 */  "\356\205\272",
  /* 0xf9 0xa4 */  "\356\205\273",
  /* 0xf9 0xa5 */  "\356\205\274",
  /* 0xf9 0xa6 */  "\356\205\275",
  /* 0xf9 0xa7 */  "\356\205\276",
  /* 0xf9 0xa8 */  "\356\205\277",
  /* 0xf9 0xa9 */  "\356\206\200",
  /* 0xf9 0xaa */  "\356\206\201",
  /* 0xf9 0xab */  "\356\206\202",
  /* 0xf9 0xac */  "\356\206\203",
  /* 0xf9 0xad */  "\356\206\204",
  /* 0xf9 0xae */  "\356\206\205",
  /* 0xf9 0xaf */  "\356\206\206",
  /* 0xf9 0xb0 */  "\356\206\207",
  /* 0xf9 0xb1 */  "\356\206\210",
  /* 0xf9 0xb2 */  "\356\206\211",
  /* 0xf9 0xb3 */  "\356\206\212",
  /* 0xf9 0xb4 */  "\356\206\213",
  /* 0xf9 0xb5 */  "\356\206\214",
  /* 0xf9 0xb6 */  "\356\206\215",
  /* 0xf9 0xb7 */  "\356\206\216",
  /* 0xf9 0xb8 */  "\356\206\217",
  /* 0xf9 0xb9 */  "\356\206\220",
  /* 0xf9 0xba */  "\356\206\221",
  /* 0xf9 0xbb */  "\356\206\222",
  /* 0xf9 0xbc */  "\356\206\223",
  /* 0xf9 0xbd */  "\356\206\224",
  /* 0xf9 0xbe */  "\356\206\225",
  /* 0xf9 0xbf */  "\356\206\226",
  /* 0xf9 0xc0 */  "\356\206\227",
  /* 0xf9 0xc1 */  "\356\206\230",
  /* 0xf9 0xc2 */  "\356\206\231",
  /* 0xf9 0xc3 */  "\356\206\232",
  /* 0xf9 0xc4 */  "\356\206\233",
  /* 0xf9 0xc5 */  "\356\206\234",
  /* 0xf9 0xc6 */  "\356\206\235",
  /* 0xf9 0xc7 */  "\356\206\236",
  /* 0xf9 0xc8 */  "\356\206\237",
  /* 0xf9 0xc9 */  "\356\206\240",
  /* 0xf9 0xca */  "\356\206\241",
  /* 0xf9 0xcb */  "\356\206\242",
  /* 0xf9 0xcc */  "\356\206\243",
  /* 0xf9 0xcd */  "\356\206\244",
  /* 0xf9 0xce */  "\356\206\245",
  /* 0xf9 0xcf */  "\356\206\246",
  /* 0xf9 0xd0 */  "\356\206\247",
  /* 0xf9 0xd1 */  "\356\206\250",
  /* 0xf9 0xd2 */  "\356\206\251",
  /* 0xf9 0xd3 */  "\356\206\252",
  /* 0xf9 0xd4 */  "\356\206\253",
  /* 0xf9 0xd5 */  "\356\206\254",
  /* 0xf9 0xd6 */  "\356\206\255",
  /* 0xf9 0xd7 */  "\356\206\256",
  /* 0xf9 0xd8 */  "\356\206\257",
  /* 0xf9 0xd9 */  "\356\206\260",
  /* 0xf9 0xda */  "\356\206\261",
  /* 0xf9 0xdb */  "\356\206\262",
  /* 0xf9 0xdc */  "\356\206\263",
  /* 0xf9 0xdd */  "\356\206\264",
  /* 0xf9 0xde */  "\356\206\265",
  /* 0xf9 0xdf */  "\356\206\266",
  /* 0xf9 0xe0 */  "\356\206\267",
  /* 0xf9 0xe1 */  "\356\206\270",
  /* 0xf9 0xe2 */  "\356\206\271",
  /* 0xf9 0xe3 */  "\356\206\272",
  /* 0xf9 0xe4 */  "\356\206\273",
  /* 0xf9 0xe5 */  "\356\206\274",
  /* 0xf9 0xe6 */  "\356\206\275",
  /* 0xf9 0xe7 */  "\356\206\276",
  /* 0xf9 0xe8 */  "\356\206\277",
  /* 0xf9 0xe9 */  "\356\207\200",
  /* 0xf9 0xea */  "\356\207\201",
  /* 0xf9 0xeb */  "\356\207\202",
  /* 0xf9 0xec */  "\356\207\203",
  /* 0xf9 0xed */  "\356\207\204",
  /* 0xf9 0xee */  "\356\207\205",
  /* 0xf9 0xef */  "\356\207\206",
  /* 0xf9 0xf0 */  "\356\207\207",
  /* 0xf9 0xf1 */  "\356\207\210",
  /* 0xf9 0xf2 */  "\356\207\211",
  /* 0xf9 0xf3 */  "\356\207\212",
  /* 0xf9 0xf4 */  "\356\207\213",
  /* 0xf9 0xf5 */  "\356\207\214",
  /* 0xf9 0xf6 */  "\356\207\215",
  /* 0xf9 0xf7 */  "\356\207\216",
  /* 0xf9 0xf8 */  "\356\207\217",
  /* 0xf9 0xf9 */  "\356\207\220",
  /* 0xf9 0xfa */  "\356\207\221",
  /* 0xf9 0xfb */  "\356\207\222",
  /* 0xf9 0xfc */  "\356\207\223",
  /* 0xf9 0xfd */  "\356\207\224",
  /* 0xf9 0xfe */  "\356\207\225",
};


/* 0xfa, 0xa1-0xfe */
static const char *eucjp_to_utf8_table_0xfa[] = {
  /* 0xfa 0xa1 */  "\356\207\226",
  /* 0xfa 0xa2 */  "\356\207\227",
  /* 0xfa 0xa3 */  "\356\207\230",
  /* 0xfa 0xa4 */  "\356\207\231",
  /* 0xfa 0xa5 */  "\356\207\232",
  /* 0xfa 0xa6 */  "\356\207\233",
  /* 0xfa 0xa7 */  "\356\207\234",
  /* 0xfa 0xa8 */  "\356\207\235",
  /* 0xfa 0xa9 */  "\356\207\236",
  /* 0xfa 0xaa */  "\356\207\237",
  /* 0xfa 0xab */  "\356\207\240",
  /* 0xfa 0xac */  "\356\207\241",
  /* 0xfa 0xad */  "\356\207\242",
  /* 0xfa 0xae */  "\356\207\243",
  /* 0xfa 0xaf */  "\356\207\244",
  /* 0xfa 0xb0 */  "\356\207\245",
  /* 0xfa 0xb1 */  "\356\207\246",
  /* 0xfa 0xb2 */  "\356\207\247",
  /* 0xfa 0xb3 */  "\356\207\250",
  /* 0xfa 0xb4 */  "\356\207\251",
  /* 0xfa 0xb5 */  "\356\207\252",
  /* 0xfa 0xb6 */  "\356\207\253",
  /* 0xfa 0xb7 */  "\356\207\254",
  /* 0xfa 0xb8 */  "\356\207\255",
  /* 0xfa 0xb9 */  "\356\207\256",
  /* 0xfa 0xba */  "\356\207\257",
  /* 0xfa 0xbb */  "\356\207\260",
  /* 0xfa 0xbc */  "\356\207\261",
  /* 0xfa 0xbd */  "\356\207\262",
  /* 0xfa 0xbe */  "\356\207\263",
  /* 0xfa 0xbf */  "\356\207\264",
  /* 0xfa 0xc0 */  "\356\207\265",
  /* 0xfa 0xc1 */  "\356\207\266",
  /* 0xfa 0xc2 */  "\356\207\267",
  /* 0xfa 0xc3 */  "\356\207\270",
  /* 0xfa 0xc4 */  "\356\207\271",
  /* 0xfa 0xc5 */  "\356\207\272",
  /* 0xfa 0xc6 */  "\356\207\273",
  /* 0xfa 0xc7 */  "\356\207\274",
  /* 0xfa 0xc8 */  "\356\207\275",
  /* 0xfa 0xc9 */  "\356\207\276",
  /* 0xfa 0xca */  "\356\207\277",
  /* 0xfa 0xcb */  "\356\210\200",
  /* 0xfa 0xcc */  "\356\210\201",
  /* 0xfa 0xcd */  "\356\210\202",
  /* 0xfa 0xce */  "\356\210\203",
  /* 0xfa 0xcf */  "\356\210\204",
  /* 0xfa 0xd0 */  "\356\210\205",
  /* 0xfa 0xd1 */  "\356\210\206",
  /* 0xfa 0xd2 */  "\356\210\207",
  /* 0xfa 0xd3 */  "\356\210\210",
  /* 0xfa 0xd4 */  "\356\210\211",
  /* 0xfa 0xd5 */  "\356\210\212",
  /* 0xfa 0xd6 */  "\356\210\213",
  /* 0xfa 0xd7 */  "\356\210\214",
  /* 0xfa 0xd8 */  "\356\210\215",
  /* 0xfa 0xd9 */  "\356\210\216",
  /* 0xfa 0xda */  "\356\210\217",
  /* 0xfa 0xdb */  "\356\210\220",
  /* 0xfa 0xdc */  "\356\210\221",
  /* 0xfa 0xdd */  "\356\210\222",
  /* 0xfa 0xde */  "\356\210\223",
  /* 0xfa 0xdf */  "\356\210\224",
  /* 0xfa 0xe0 */  "\356\210\225",
  /* 0xfa 0xe1 */  "\356\210\226",
  /* 0xfa 0xe2 */  "\356\210\227",
  /* 0xfa 0xe3 */  "\356\210\230",
  /* 0xfa 0xe4 */  "\356\210\231",
  /* 0xfa 0xe5 */  "\356\210\232",
  /* 0xfa 0xe6 */  "\356\210\233",
  /* 0xfa 0xe7 */  "\356\210\234",
  /* 0xfa 0xe8 */  "\356\210\235",
  /* 0xfa 0xe9 */  "\356\210\236",
  /* 0xfa 0xea */  "\356\210\237",
  /* 0xfa 0xeb */  "\356\210\240",
  /* 0xfa 0xec */  "\356\210\241",
  /* 0xfa 0xed */  "\356\210\242",
  /* 0xfa 0xee */  "\356\210\243",
  /* 0xfa 0xef */  "\356\210\244",
  /* 0xfa 0xf0 */  "\356\210\245",
  /* 0xfa 0xf1 */  "\356\210\246",
  /* 0xfa 0xf2 */  "\356\210\247",
  /* 0xfa 0xf3 */  "\356\210\250",
  /* 0xfa 0xf4 */  "\356\210\251",
  /* 0xfa 0xf5 */  "\356\210\252",
  /* 0xfa 0xf6 */  "\356\210\253",
  /* 0xfa 0xf7 */  "\356\210\254",
  /* 0xfa 0xf8 */  "\356\210\255",
  /* 0xfa 0xf9 */  "\356\210\256",
  /* 0xfa 0xfa */  "\356\210\257",
  /* 0xfa 0xfb */  "\356\210\260",
  /* 0xfa 0xfc */  "\356\210\261",
  /* 0xfa 0xfd */  "\356\210\262",
  /* 0xfa 0xfe */  "\356\210\263",
};


/* 0xfb, 0xa1-0xfe */
static const char *eucjp_to_utf8_table_0xfb[] = {
  /* 0xfb 0xa1 */  "\356\210\264",
  /* 0xfb 0xa2 */  "\356\210\265",
  /* 0xfb 0xa3 */  "\356\210\266",
  /* 0xfb 0xa4 */  "\356\210\267",
  /* 0xfb 0xa5 */  "\356\210\270",
  /* 0xfb 0xa6 */  "\356\210\271",
  /* 0xfb 0xa7 */  "\356\210\272",
  /* 0xfb 0xa8 */  "\356\210\273",
  /* 0xfb 0xa9 */  "\356\210\274",
  /* 0xfb 0xaa */  "\356\210\275",
  /* 0xfb 0xab */  "\356\210\276",
  /* 0xfb 0xac */  "\356\210\277",
  /* 0xfb 0xad */  "\356\211\200",
  /* 0xfb 0xae */  "\356\211\201",
  /* 0xfb 0xaf */  "\356\211\202",
  /* 0xfb 0xb0 */  "\356\211\203",
  /* 0xfb 0xb1 */  "\356\211\204",
  /* 0xfb 0xb2 */  "\356\211\205",
  /* 0xfb 0xb3 */  "\356\211\206",
  /* 0xfb 0xb4 */  "\356\211\207",
  /* 0xfb 0xb5 */  "\356\211\210",
  /* 0xfb 0xb6 */  "\356\211\211",
  /* 0xfb 0xb7 */  "\356\211\212",
  /* 0xfb 0xb8 */  "\356\211\213",
  /* 0xfb 0xb9 */  "\356\211\214",
  /* 0xfb 0xba */  "\356\211\215",
  /* 0xfb 0xbb */  "\356\211\216",
  /* 0xfb 0xbc */  "\356\211\217",
  /* 0xfb 0xbd */  "\356\211\220",
  /* 0xfb 0xbe */  "\356\211\221",
  /* 0xfb 0xbf */  "\356\211\222",
  /* 0xfb 0xc0 */  "\356\211\223",
  /* 0xfb 0xc1 */  "\356\211\224",
  /* 0xfb 0xc2 */  "\356\211\225",
  /* 0xfb 0xc3 */  "\356\211\226",
  /* 0xfb 0xc4 */  "\356\211\227",
  /* 0xfb 0xc5 */  "\356\211\230",
  /* 0xfb 0xc6 */  "\356\211\231",
  /* 0xfb 0xc7 */  "\356\211\232",
  /* 0xfb 0xc8 */  "\356\211\233",
  /* 0xfb 0xc9 */  "\356\211\234",
  /* 0xfb 0xca */  "\356\211\235",
  /* 0xfb 0xcb */  "\356\211\236",
  /* 0xfb 0xcc */  "\356\211\237",
  /* 0xfb 0xcd */  "\356\211\240",
  /* 0xfb 0xce */  "\356\211\241",
  /* 0xfb 0xcf */  "\356\211\242",
  /* 0xfb 0xd0 */  "\356\211\243",
  /* 0xfb 0xd1 */  "\356\211\244",
  /* 0xfb 0xd2 */  "\356\211\245",
  /* 0xfb 0xd3 */  "\356\211\246",
  /* 0xfb 0xd4 */  "\356\211\247",
  /* 0xfb 0xd5 */  "\356\211\250",
  /* 0xfb 0xd6 */  "\356\211\251",
  /* 0xfb 0xd7 */  "\356\211\252",
  /* 0xfb 0xd8 */  "\356\211\253",
  /* 0xfb 0xd9 */  "\356\211\254",
  /* 0xfb 0xda */  "\356\211\255",
  /* 0xfb 0xdb */  "\356\211\256",
  /* 0xfb 0xdc */  "\356\211\257",
  /* 0xfb 0xdd */  "\356\211\260",
  /* 0xfb 0xde */  "\356\211\261",
  /* 0xfb 0xdf */  "\356\211\262",
  /* 0xfb 0xe0 */  "\356\211\263",
  /* 0xfb 0xe1 */  "\356\211\264",
  /* 0xfb 0xe2 */  "\356\211\265",
  /* 0xfb 0xe3 */  "\356\211\266",
  /* 0xfb 0xe4 */  "\356\211\267",
  /* 0xfb 0xe5 */  "\356\211\270",
  /* 0xfb 0xe6 */  "\356\211\271",
  /* 0xfb 0xe7 */  "\356\211\272",
  /* 0xfb 0xe8 */  "\356\211\273",
  /* 0xfb 0xe9 */  "\356\211\274",
  /* 0xfb 0xea */  "\356\211\275",
  /* 0xfb 0xeb */  "\356\211\276",
  /* 0xfb 0xec */  "\356\211\277",
  /* 0xfb 0xed */  "\356\212\200",
  /* 0xfb 0xee */  "\356\212\201",
  /* 0xfb 0xef */  "\356\212\202",
  /* 0xfb 0xf0 */  "\356\212\203",
  /* 0xfb 0xf1 */  "\356\212\204",
  /* 0xfb 0xf2 */  "\356\212\205",
  /* 0xfb 0xf3 */  "\356\212\206",
  /* 0xfb 0xf4 */  "\356\212\207",
  /* 0xfb 0xf5 */  "\356\212\210",
  /* 0xfb 0xf6 */  "\356\212\211",
  /* 0xfb 0xf7 */  "\356\212\212",
  /* 0xfb 0xf8 */  "\356\212\213",
  /* 0xfb 0xf9 */  "\356\212\214",
  /* 0xfb 0xfa */  "\356\212\215",
  /* 0xfb 0xfb */  "\356\212\216",
  /* 0xfb 0xfc */  "\356\212\217",
  /* 0xfb 0xfd */  "\356\212\220",
  /* 0xfb 0xfe */  "\356\212\221",
};


/* 0xfc, 0xa1-0xfe */
static const char *eucjp_to_utf8_table_0xfc[] = {
  /* 0xfc 0xa1 */  "\356\212\222",
  /* 0xfc 0xa2 */  "\356\212\223",
  /* 0xfc 0xa3 */  "\356\212\224",
  /* 0xfc 0xa4 */  "\356\212\225",
  /* 0xfc 0xa5 */  "\356\212\226",
  /* 0xfc 0xa6 */  "\356\212\227",
  /* 0xfc 0xa7 */  "\356\212\230",
  /* 0xfc 0xa8 */  "\356\212\231",
  /* 0xfc 0xa9 */  "\356\212\232",
  /* 0xfc 0xaa */  "\356\212\233",
  /* 0xfc 0xab */  "\356\212\234",
  /* 0xfc 0xac */  "\356\212\235",
  /* 0xfc 0xad */  "\356\212\236",
  /* 0xfc 0xae */  "\356\212\237",
  /* 0xfc 0xaf */  "\356\212\240",
  /* 0xfc 0xb0 */  "\356\212\241",
  /* 0xfc 0xb1 */  "\356\212\242",
  /* 0xfc 0xb2 */  "\356\212\243",
  /* 0xfc 0xb3 */  "\356\212\244",
  /* 0xfc 0xb4 */  "\356\212\245",
  /* 0xfc 0xb5 */  "\356\212\246",
  /* 0xfc 0xb6 */  "\356\212\247",
  /* 0xfc 0xb7 */  "\356\212\250",
  /* 0xfc 0xb8 */  "\356\212\251",
  /* 0xfc 0xb9 */  "\356\212\252",
  /* 0xfc 0xba */  "\356\212\253",
  /* 0xfc 0xbb */  "\356\212\254",
  /* 0xfc 0xbc */  "\356\212\255",
  /* 0xfc 0xbd */  "\356\212\256",
  /* 0xfc 0xbe */  "\356\212\257",
  /* 0xfc 0xbf */  "\356\212\260",
  /* 0xfc 0xc0 */  "\356\212\261",
  /* 0xfc 0xc1 */  "\356\212\262",
  /* 0xfc 0xc2 */  "\356\212\263",
  /* 0xfc 0xc3 */  "\356\212\264",
  /* 0xfc 0xc4 */  "\356\212\265",
  /* 0xfc 0xc5 */  "\356\212\266",
  /* 0xfc 0xc6 */  "\356\212\267",
  /* 0xfc 0xc7 */  "\356\212\270",
  /* 0xfc 0xc8 */  "\356\212\271",
  /* 0xfc 0xc9 */  "\356\212\272",
  /* 0xfc 0xca */  "\356\212\273",
  /* 0xfc 0xcb */  "\356\212\274",
  /* 0xfc 0xcc */  "\356\212\275",
  /* 0xfc 0xcd */  "\356\212\276",
  /* 0xfc 0xce */  "\356\212\277",
  /* 0xfc 0xcf */  "\356\213\200",
  /* 0xfc 0xd0 */  "\356\213\201",
  /* 0xfc 0xd1 */  "\356\213\202",
  /* 0xfc 0xd2 */  "\356\213\203",
  /* 0xfc 0xd3 */  "\356\213\204",
  /* 0xfc 0xd4 */  "\356\213\205",
  /* 0xfc 0xd5 */  "\356\213\206",
  /* 0xfc 0xd6 */  "\356\213\207",
  /* 0xfc 0xd7 */  "\356\213\210",
  /* 0xfc 0xd8 */  "\356\213\211",
  /* 0xfc 0xd9 */  "\356\213\212",
  /* 0xfc 0xda */  "\356\213\213",
  /* 0xfc 0xdb */  "\356\213\214",
  /* 0xfc 0xdc */  "\356\213\215",
  /* 0xfc 0xdd */  "\356\213\216",
  /* 0xfc 0xde */  "\356\213\217",
  /* 0xfc 0xdf */  "\356\213\220",
  /* 0xfc 0xe0 */  "\356\213\221",
  /* 0xfc 0xe1 */  "\356\213\222",
  /* 0xfc 0xe2 */  "\356\213\223",
  /* 0xfc 0xe3 */  "\356\213\224",
  /* 0xfc 0xe4 */  "\356\213\225",
  /* 0xfc 0xe5 */  "\356\213\226",
  /* 0xfc 0xe6 */  "\356\213\227",
  /* 0xfc 0xe7 */  "\356\213\230",
  /* 0xfc 0xe8 */  "\356\213\231",
  /* 0xfc 0xe9 */  "\356\213\232",
  /* 0xfc 0xea */  "\356\213\233",
  /* 0xfc 0xeb */  "\356\213\234",
  /* 0xfc 0xec */  "\356\213\235",
  /* 0xfc 0xed */  "\356\213\236",
  /* 0xfc 0xee */  "\356\213\237",
  /* 0xfc 0xef */  "\356\213\240",
  /* 0xfc 0xf0 */  "\356\213\241",
  /* 0xfc 0xf1 */  "\356\213\242",
  /* 0xfc 0xf2 */  "\356\213\243",
  /* 0xfc 0xf3 */  "\356\213\244",
  /* 0xfc 0xf4 */  "\356\213\245",
  /* 0xfc 0xf5 */  "\356\213\246",
  /* 0xfc 0xf6 */  "\356\213\247",
  /* 0xfc 0xf7 */  "\356\213\250",
  /* 0xfc 0xf8 */  "\356\213\251",
  /* 0xfc 0xf9 */  "\356\213\252",
  /* 0xfc 0xfa */  "\356\213\253",
  /* 0xfc 0xfb */  "\356\213\254",
  /* 0xfc 0xfc */  "\356\213\255",
  /* 0xfc 0xfd */  "\356\213\256",
  /* 0xfc 0xfe */  "\356\213\257",
};


/* 0xfd, 0xa1-0xfe */
static const char *eucjp_to_utf8_table_0xfd[] = {
  /* 0xfd 0xa1 */  "\356\213\260",
  /* 0xfd 0xa2 */  "\356\213\261",
  /* 0xfd 0xa3 */  "\356\213\262",
  /* 0xfd 0xa4 */  "\356\213\263",
  /* 0xfd 0xa5 */  "\356\213\264",
  /* 0xfd 0xa6 */  "\356\213\265",
  /* 0xfd 0xa7 */  "\356\213\266",
  /* 0xfd 0xa8 */  "\356\213\267",
  /* 0xfd 0xa9 */  "\356\213\270",
  /* 0xfd 0xaa */  "\356\213\271",
  /* 0xfd 0xab */  "\356\213\272",
  /* 0xfd 0xac */  "\356\213\273",
  /* 0xfd 0xad */  "\356\213\274",
  /* 0xfd 0xae */  "\356\213\275",
  /* 0xfd 0xaf */  "\356\213\276",
  /* 0xfd 0xb0 */  "\356\213\277",
  /* 0xfd 0xb1 */  "\356\214\200",
  /* 0xfd 0xb2 */  "\356\214\201",
  /* 0xfd 0xb3 */  "\356\214\202",
  /* 0xfd 0xb4 */  "\356\214\203",
  /* 0xfd 0xb5 */  "\356\214\204",
  /* 0xfd 0xb6 */  "\356\214\205",
  /* 0xfd 0xb7 */  "\356\214\206",
  /* 0xfd 0xb8 */  "\356\214\207",
  /* 0xfd 0xb9 */  "\356\214\210",
  /* 0xfd 0xba */  "\356\214\211",
  /* 0xfd 0xbb */  "\356\214\212",
  /* 0xfd 0xbc */  "\356\214\213",
  /* 0xfd 0xbd */  "\356\214\214",
  /* 0xfd 0xbe */  "\356\214\215",
  /* 0xfd 0xbf */  "\356\214\216",
  /* 0xfd 0xc0 */  "\356\214\217",
  /* 0xfd 0xc1 */  "\356\214\220",
  /* 0xfd 0xc2 */  "\356\214\221",
  /* 0xfd 0xc3 */  "\356\214\222",
  /* 0xfd 0xc4 */  "\356\214\223",
  /* 0xfd 0xc5 */  "\356\214\224",
  /* 0xfd 0xc6 */  "\356\214\225",
  /* 0xfd 0xc7 */  "\356\214\226",
  /* 0xfd 0xc8 */  "\356\214\227",
  /* 0xfd 0xc9 */  "\356\214\230",
  /* 0xfd 0xca */  "\356\214\231",
  /* 0xfd 0xcb */  "\356\214\232",
  /* 0xfd 0xcc */  "\356\214\233",
  /* 0xfd 0xcd */  "\356\214\234",
  /* 0xfd 0xce */  "\356\214\235",
  /* 0xfd 0xcf */  "\356\214\236",
  /* 0xfd 0xd0 */  "\356\214\237",
  /* 0xfd 0xd1 */  "\356\214\240",
  /* 0xfd 0xd2 */  "\356\214\241",
  /* 0xfd 0xd3 */  "\356\214\242",
  /* 0xfd 0xd4 */  "\356\214\243",
  /* 0xfd 0xd5 */  "\356\214\244",
  /* 0xfd 0xd6 */  "\356\214\245",
  /* 0xfd 0xd7 */  "\356\214\246",
  /* 0xfd 0xd8 */  "\356\214\247",
  /* 0xfd 0xd9 */  "\356\214\250",
  /* 0xfd 0xda */  "\356\214\251",
  /* 0xfd 0xdb */  "\356\214\252",
  /* 0xfd 0xdc */  "\356\214\253",
  /* 0xfd 0xdd */  "\356\214\254",
  /* 0xfd 0xde */  "\356\214\255",
  /* 0xfd 0xdf */  "\356\214\256",
  /* 0xfd 0xe0 */  "\356\214\257",
  /* 0xfd 0xe1 */  "\356\214\260",
  /* 0xfd 0xe2 */  "\356\214\261",
  /* 0xfd 0xe3 */  "\356\214\262",
  /* 0xfd 0xe4 */  "\356\214\263",
  /* 0xfd 0xe5 */  "\356\214\264",
  /* 0xfd 0xe6 */  "\356\214\265",
  /* 0xfd 0xe7 */  "\356\214\266",
  /* 0xfd 0xe8 */  "\356\214\267",
  /* 0xfd 0xe9 */  "\356\214\270",
  /* 0xfd 0xea */  "\356\214\271",
  /* 0xfd 0xeb */  "\356\214\272",
  /* 0xfd 0xec */  "\356\214\273",
  /* 0xfd 0xed */  "\356\214\274",
  /* 0xfd 0xee */  "\356\214\275",
  /* 0xfd 0xef */  "\356\214\276",
  /* 0xfd 0xf0 */  "\356\214\277",
  /* 0xfd 0xf1 */  "\356\215\200",
  /* 0xfd 0xf2 */  "\356\215\201",
  /* 0xfd 0xf3 */  "\356\215\202",
  /* 0xfd 0xf4 */  "\356\215\203",
  /* 0xfd 0xf5 */  "\356\215\204",
  /* 0xfd 0xf6 */  "\356\215\205",
  /* 0xfd 0xf7 */  "\356\215\206",
  /* 0xfd 0xf8 */  "\356\215\207",
  /* 0xfd 0xf9 */  "\356\215\210",
  /* 0xfd 0xfa */  "\356\215\211",
  /* 0xfd 0xfb */  "\356\215\212",
  /* 0xfd 0xfc */  "\356\215\213",
  /* 0xfd 0xfd */  "\356\215\214",
  /* 0xfd 0xfe */  "\356\215\215",
};


/* 0xfe, 0xa1-0xfe */
static const char *eucjp_to_utf8_table_0xfe[] = {
  /* 0xfe 0xa1 */  "\356\215\216",
  /* 0xfe 0xa2 */  "\356\215\217",
  /* 0xfe 0xa3 */  "\356\215\220",
  /* 0xfe 0xa4 */  "\356\215\221",
  /* 0xfe 0xa5 */  "\356\215\222",
  /* 0xfe 0xa6 */  "\356\215\223",
  /* 0xfe 0xa7 */  "\356\215\224",
  /* 0xfe 0xa8 */  "\356\215\225",
  /* 0xfe 0xa9 */  "\356\215\226",
  /* 0xfe 0xaa */  "\356\215\227",
  /* 0xfe 0xab */  "\356\215\230",
  /* 0xfe 0xac */  "\356\215\231",
  /* 0xfe 0xad */  "\356\215\232",
  /* 0xfe 0xae */  "\356\215\233",
  /* 0xfe 0xaf */  "\356\215\234",
  /* 0xfe 0xb0 */  "\356\215\235",
  /* 0xfe 0xb1 */  "\356\215\236",
  /* 0xfe 0xb2 */  "\356\215\237",
  /* 0xfe 0xb3 */  "\356\215\240",
  /* 0xfe 0xb4 */  "\356\215\241",
  /* 0xfe 0xb5 */  "\356\215\242",
  /* 0xfe 0xb6 */  "\356\215\243",
  /* 0xfe 0xb7 */  "\356\215\244",
  /* 0xfe 0xb8 */  "\356\215\245",
  /* 0xfe 0xb9 */  "\356\215\246",
  /* 0xfe 0xba */  "\356\215\247",
  /* 0xfe 0xbb */  "\356\215\250",
  /* 0xfe 0xbc */  "\356\215\251",
  /* 0xfe 0xbd */  "\356\215\252",
  /* 0xfe 0xbe */  "\356\215\253",
  /* 0xfe 0xbf */  "\356\215\254",
  /* 0xfe 0xc0 */  "\356\215\255",
  /* 0xfe 0xc1 */  "\356\215\256",
  /* 0xfe 0xc2 */  "\356\215\257",
  /* 0xfe 0xc3 */  "\356\215\260",
  /* 0xfe 0xc4 */  "\356\215\261",
  /* 0xfe 0xc5 */  "\356\215\262",
  /* 0xfe 0xc6 */  "\356\215\263",
  /* 0xfe 0xc7 */  "\356\215\264",
  /* 0xfe 0xc8 */  "\356\215\265",
  /* 0xfe 0xc9 */  "\356\215\266",
  /* 0xfe 0xca */  "\356\215\267",
  /* 0xfe 0xcb */  "\356\215\270",
  /* 0xfe 0xcc */  "\356\215\271",
  /* 0xfe 0xcd */  "\356\215\272",
  /* 0xfe 0xce */  "\356\215\273",
  /* 0xfe 0xcf */  "\356\215\274",
  /* 0xfe 0xd0 */  "\356\215\275",
  /* 0xfe 0xd1 */  "\356\215\276",
  /* 0xfe 0xd2 */  "\356\215\277",
  /* 0xfe 0xd3 */  "\356\216\200",
  /* 0xfe 0xd4 */  "\356\216\201",
  /* 0xfe 0xd5 */  "\356\216\202",
  /* 0xfe 0xd6 */  "\356\216\203",
  /* 0xfe 0xd7 */  "\356\216\204",
  /* 0xfe 0xd8 */  "\356\216\205",
  /* 0xfe 0xd9 */  "\356\216\206",
  /* 0xfe 0xda */  "\356\216\207",
  /* 0xfe 0xdb */  "\356\216\210",
  /* 0xfe 0xdc */  "\356\216\211",
  /* 0xfe 0xdd */  "\356\216\212",
  /* 0xfe 0xde */  "\356\216\213",
  /* 0xfe 0xdf */  "\356\216\214",
  /* 0xfe 0xe0 */  "\356\216\215",
  /* 0xfe 0xe1 */  "\356\216\216",
  /* 0xfe 0xe2 */  "\356\216\217",
  /* 0xfe 0xe3 */  "\356\216\220",
  /* 0xfe 0xe4 */  "\356\216\221",
  /* 0xfe 0xe5 */  "\356\216\222",
  /* 0xfe 0xe6 */  "\356\216\223",
  /* 0xfe 0xe7 */  "\356\216\224",
  /* 0xfe 0xe8 */  "\356\216\225",
  /* 0xfe 0xe9 */  "\356\216\226",
  /* 0xfe 0xea */  "\356\216\227",
  /* 0xfe 0xeb */  "\356\216\230",
  /* 0xfe 0xec */  "\356\216\231",
  /* 0xfe 0xed */  "\356\216\232",
  /* 0xfe 0xee */  "\356\216\233",
  /* 0xfe 0xef */  "\356\216\234",
  /* 0xfe 0xf0 */  "\356\216\235",
  /* 0xfe 0xf1 */  "\356\216\236",
  /* 0xfe 0xf2 */  "\356\216\237",
  /* 0xfe 0xf3 */  "\356\216\240",
  /* 0xfe 0xf4 */  "\356\216\241",
  /* 0xfe 0xf5 */  "\356\216\242",
  /* 0xfe 0xf6 */  "\356\216\243",
  /* 0xfe 0xf7 */  "\356\216\244",
  /* 0xfe 0xf8 */  "\356\216\245",
  /* 0xfe 0xf9 */  "\356\216\246",
  /* 0xfe 0xfa */  "\356\216\247",
  /* 0xfe 0xfb */  "\356\216\250",
  /* 0xfe 0xfc */  "\356\216\251",
  /* 0xfe 0xfd */  "\356\216\252",
  /* 0xfe 0xfe */  "\356\216\253",
};


#define EUCJP_TO_UTF8_0X8E_MIN	0xa1
#define EUCJP_TO_UTF8_0X8E_MAX	0xdf
#define EUCJP_TO_UTF8_0XA2_MIN	0xaf
#define EUCJP_TO_UTF8_0XA2_MAX	0xf1
#define EUCJP_TO_UTF8_0XA6_MIN	0xe1
#define EUCJP_TO_UTF8_0XA6_MAX	0xfc
#define EUCJP_TO_UTF8_0XA7_MIN	0xc2
#define EUCJP_TO_UTF8_0XA7_MAX	0xfe
#define EUCJP_TO_UTF8_0XA9_MIN	0xa1
#define EUCJP_TO_UTF8_0XA9_MAX	0xd0
#define EUCJP_TO_UTF8_0XAA_MIN	0xa1
#define EUCJP_TO_UTF8_0XAA_MAX	0xf7
#define EUCJP_TO_UTF8_0XAB_MIN	0xa1
#define EUCJP_TO_UTF8_0XAB_MAX	0xf7
#define EUCJP_TO_UTF8_0XAD_MIN	0xa1
#define EUCJP_TO_UTF8_0XAD_MAX	0xfc
#define EUCJP_TO_UTF8_0XB0_MIN	0xa1
#define EUCJP_TO_UTF8_0XB0_MAX	0xfe
#define EUCJP_TO_UTF8_0XB1_MIN	0xa1
#define EUCJP_TO_UTF8_0XB1_MAX	0xfe
#define EUCJP_TO_UTF8_0XB2_MIN	0xa1
#define EUCJP_TO_UTF8_0XB2_MAX	0xfe
#define EUCJP_TO_UTF8_0XB3_MIN	0xa1
#define EUCJP_TO_UTF8_0XB3_MAX	0xfe
#define EUCJP_TO_UTF8_0XB4_MIN	0xa1
#define EUCJP_TO_UTF8_0XB4_MAX	0xfe
#define EUCJP_TO_UTF8_0XB5_MIN	0xa1
#define EUCJP_TO_UTF8_0XB5_MAX	0xfe
#define EUCJP_TO_UTF8_0XB6_MIN	0xa1
#define EUCJP_TO_UTF8_0XB6_MAX	0xb6
#define EUCJP_TO_UTF8_0XF3_MIN	0xf3
#define EUCJP_TO_UTF8_0XF3_MAX	0xfe
#define EUCJP_TO_UTF8_0XF4_MIN	0xa1
#define EUCJP_TO_UTF8_0XF4_MAX	0xfe
#define EUCJP_TO_UTF8_0XF5_MIN	0xa1
#define EUCJP_TO_UTF8_0XF5_MAX	0xfe
#define EUCJP_TO_UTF8_0XF6_MIN	0xa1
#define EUCJP_TO_UTF8_0XF6_MAX	0xfe
#define EUCJP_TO_UTF8_0XF7_MIN	0xa1
#define EUCJP_TO_UTF8_0XF7_MAX	0xfe
#define EUCJP_TO_UTF8_0XF8_MIN	0xa1
#define EUCJP_TO_UTF8_0XF8_MAX	0xfe
#define EUCJP_TO_UTF8_0XF9_MIN	0xa1
#define EUCJP_TO_UTF8_0XF9_MAX	0xfe
#define EUCJP_TO_UTF8_0XFA_MIN	0xa1
#define EUCJP_TO_UTF8_0XFA_MAX	0xfe
#define EUCJP_TO_UTF8_0XFB_MIN	0xa1
#define EUCJP_TO_UTF8_0XFB_MAX	0xfe
#define EUCJP_TO_UTF8_0XFC_MIN	0xa1
#define EUCJP_TO_UTF8_0XFC_MAX	0xfe
#define EUCJP_TO_UTF8_0XFD_MIN	0xa1
#define EUCJP_TO_UTF8_0XFD_MAX	0xfe
#define EUCJP_TO_UTF8_0XFE_MIN	0xa1
#define EUCJP_TO_UTF8_0XFE_MAX	0xfe


#if 0
static void
validate_utf8_table(const char * const *table, int size)
{
  int i;
  for (i = 0; i <= size; i++)
    {
      if (table[i] != NULL && !g_utf8_validate(table[i], -1, NULL))
	fprintf(stderr, "table[%d] is invalid UTF-8 string\n", i);
    }
}


static void
validate_eucjp_to_utf8_tables(void)
{
  fprintf(stderr, "validating eucjp_to_utf8_table_0x8e\n");
  validate_utf8_table(eucjp_to_utf8_table_0x8e,
		      EUCJP_TO_UTF8_0X8E_MAX - EUCJP_TO_UTF8_0X8E_MIN);

  fprintf(stderr, "validating eucjp_to_utf8_table_0xa2\n");
  validate_utf8_table(eucjp_to_utf8_table_0xa2,
		      EUCJP_TO_UTF8_0XA2_MAX - EUCJP_TO_UTF8_0XA2_MIN);

  fprintf(stderr, "validating eucjp_to_utf8_table_0xa6\n");
  validate_utf8_table(eucjp_to_utf8_table_0xa6,
		      EUCJP_TO_UTF8_0XA6_MAX - EUCJP_TO_UTF8_0XA6_MIN);

  fprintf(stderr, "validating eucjp_to_utf8_table_0xa7\n");
  validate_utf8_table(eucjp_to_utf8_table_0xa7,
		      EUCJP_TO_UTF8_0XA7_MAX - EUCJP_TO_UTF8_0XA7_MIN);

  fprintf(stderr, "validating eucjp_to_utf8_table_0xa9\n");
  validate_utf8_table(eucjp_to_utf8_table_0xa9,
		      EUCJP_TO_UTF8_0XA9_MAX - EUCJP_TO_UTF8_0XA9_MIN);

  fprintf(stderr, "validating eucjp_to_utf8_table_0xaa\n");
  validate_utf8_table(eucjp_to_utf8_table_0xaa,
		      EUCJP_TO_UTF8_0XAA_MAX - EUCJP_TO_UTF8_0XAA_MIN);

  fprintf(stderr, "validating eucjp_to_utf8_table_0xab\n");
  validate_utf8_table(eucjp_to_utf8_table_0xab,
		      EUCJP_TO_UTF8_0XAB_MAX - EUCJP_TO_UTF8_0XAB_MIN);

  fprintf(stderr, "validating eucjp_to_utf8_table_0xad\n");
  validate_utf8_table(eucjp_to_utf8_table_0xad,
		      EUCJP_TO_UTF8_0XAD_MAX - EUCJP_TO_UTF8_0XAD_MIN);

  fprintf(stderr, "validating eucjp_to_utf8_table_0xb0\n");
  validate_utf8_table(eucjp_to_utf8_table_0xb0,
		      EUCJP_TO_UTF8_0XB0_MAX - EUCJP_TO_UTF8_0XB0_MIN);

  fprintf(stderr, "validating eucjp_to_utf8_table_0xb1\n");
  validate_utf8_table(eucjp_to_utf8_table_0xb1,
		      EUCJP_TO_UTF8_0XB1_MAX - EUCJP_TO_UTF8_0XB1_MIN);

  fprintf(stderr, "validating eucjp_to_utf8_table_0xb2\n");
  validate_utf8_table(eucjp_to_utf8_table_0xb2,
		      EUCJP_TO_UTF8_0XB2_MAX - EUCJP_TO_UTF8_0XB2_MIN);

  fprintf(stderr, "validating eucjp_to_utf8_table_0xb3\n");
  validate_utf8_table(eucjp_to_utf8_table_0xb3,
		      EUCJP_TO_UTF8_0XB3_MAX - EUCJP_TO_UTF8_0XB3_MIN);

  fprintf(stderr, "validating eucjp_to_utf8_table_0xb4\n");
  validate_utf8_table(eucjp_to_utf8_table_0xb4,
		      EUCJP_TO_UTF8_0XB4_MAX - EUCJP_TO_UTF8_0XB4_MIN);

  fprintf(stderr, "validating eucjp_to_utf8_table_0xb5\n");
  validate_utf8_table(eucjp_to_utf8_table_0xb5,
		      EUCJP_TO_UTF8_0XB5_MAX - EUCJP_TO_UTF8_0XB5_MIN);

  fprintf(stderr, "validating eucjp_to_utf8_table_0xb6\n");
  validate_utf8_table(eucjp_to_utf8_table_0xb6,
		      EUCJP_TO_UTF8_0XB6_MAX - EUCJP_TO_UTF8_0XB6_MIN);

  fprintf(stderr, "validating eucjp_to_utf8_table_0xf3\n");
  validate_utf8_table(eucjp_to_utf8_table_0xf3,
		      EUCJP_TO_UTF8_0XF3_MAX - EUCJP_TO_UTF8_0XF3_MIN);

  fprintf(stderr, "validating eucjp_to_utf8_table_0xf4\n");
  validate_utf8_table(eucjp_to_utf8_table_0xf4,
		      EUCJP_TO_UTF8_0XF4_MAX - EUCJP_TO_UTF8_0XF4_MIN);

  fprintf(stderr, "validating eucjp_to_utf8_table_0xf5\n");
  validate_utf8_table(eucjp_to_utf8_table_0xf5,
		      EUCJP_TO_UTF8_0XF5_MAX - EUCJP_TO_UTF8_0XF5_MIN);

  fprintf(stderr, "validating eucjp_to_utf8_table_0xf6\n");
  validate_utf8_table(eucjp_to_utf8_table_0xf6,
		      EUCJP_TO_UTF8_0XF6_MAX - EUCJP_TO_UTF8_0XF6_MIN);

  fprintf(stderr, "validating eucjp_to_utf8_table_0xf7\n");
  validate_utf8_table(eucjp_to_utf8_table_0xf7,
		      EUCJP_TO_UTF8_0XF7_MAX - EUCJP_TO_UTF8_0XF7_MIN);

  fprintf(stderr, "validating eucjp_to_utf8_table_0xf8\n");
  validate_utf8_table(eucjp_to_utf8_table_0xf8,
		      EUCJP_TO_UTF8_0XF8_MAX - EUCJP_TO_UTF8_0XF8_MIN);

  fprintf(stderr, "validating eucjp_to_utf8_table_0xf9\n");
  validate_utf8_table(eucjp_to_utf8_table_0xf9,
		      EUCJP_TO_UTF8_0XF9_MAX - EUCJP_TO_UTF8_0XF9_MIN);

  fprintf(stderr, "validating eucjp_to_utf8_table_0xfa\n");
  validate_utf8_table(eucjp_to_utf8_table_0xfa,
		      EUCJP_TO_UTF8_0XFA_MAX - EUCJP_TO_UTF8_0XFA_MIN);

  fprintf(stderr, "validating eucjp_to_utf8_table_0xfb\n");
  validate_utf8_table(eucjp_to_utf8_table_0xfb,
		      EUCJP_TO_UTF8_0XFB_MAX - EUCJP_TO_UTF8_0XFB_MIN);

  fprintf(stderr, "validating eucjp_to_utf8_table_0xfc\n");
  validate_utf8_table(eucjp_to_utf8_table_0xfc,
		      EUCJP_TO_UTF8_0XFC_MAX - EUCJP_TO_UTF8_0XFC_MIN);

  fprintf(stderr, "validating eucjp_to_utf8_table_0xfd\n");
  validate_utf8_table(eucjp_to_utf8_table_0xfd,
		      EUCJP_TO_UTF8_0XFD_MAX - EUCJP_TO_UTF8_0XFD_MIN);

  fprintf(stderr, "validating eucjp_to_utf8_table_0xfe\n");
  validate_utf8_table(eucjp_to_utf8_table_0xfe,
		      EUCJP_TO_UTF8_0XFE_MAX - EUCJP_TO_UTF8_0XFE_MIN);
}
#endif


size_t
eucjp_to_utf8_helper(const char **inbuf, size_t *inbytesleft,
		     char **outbuf, size_t *outbytesleft)
{
  const char *utf8_string = NULL;
  unsigned char first;
  unsigned char second;

  if (*inbytesleft < 2)
    {
      errno = EINVAL;
      return -1;
    }

  first = (*inbuf)[0];
  second = (*inbuf)[1];

#if DEBUG_ICONV_MOST
  fprintf(stderr, "<1>first=%02x second=%02x: %c%c\n",
	  first, second, first, second);
#endif

  switch (first)
    {
    case 0x8e:
      if (second >= EUCJP_TO_UTF8_0X8E_MIN && second <= EUCJP_TO_UTF8_0X8E_MAX)
	utf8_string = eucjp_to_utf8_table_0x8e[second - EUCJP_TO_UTF8_0X8E_MIN];
      break;

    case 0xa2:
      if (second >= EUCJP_TO_UTF8_0XA2_MIN && second <= EUCJP_TO_UTF8_0XA2_MAX)
	utf8_string = eucjp_to_utf8_table_0xa2[second - EUCJP_TO_UTF8_0XA2_MIN];
      break;

    case 0xa6:
      if (second >= EUCJP_TO_UTF8_0XA6_MIN && second <= EUCJP_TO_UTF8_0XA6_MAX)
	utf8_string = eucjp_to_utf8_table_0xa6[second - EUCJP_TO_UTF8_0XA6_MIN];
      break;

    case 0xa7:
      if (second >= EUCJP_TO_UTF8_0XA7_MIN && second <= EUCJP_TO_UTF8_0XA7_MAX)
	utf8_string = eucjp_to_utf8_table_0xa7[second - EUCJP_TO_UTF8_0XA7_MIN];
      break;

    case 0xa9:
      if (second >= EUCJP_TO_UTF8_0XA9_MIN && second <= EUCJP_TO_UTF8_0XA9_MAX)
	utf8_string = eucjp_to_utf8_table_0xa9[second - EUCJP_TO_UTF8_0XA9_MIN];
      break;

    case 0xaa:
      if (second >= EUCJP_TO_UTF8_0XAA_MIN && second <= EUCJP_TO_UTF8_0XAA_MAX)
	utf8_string = eucjp_to_utf8_table_0xaa[second - EUCJP_TO_UTF8_0XAA_MIN];
      break;

    case 0xab:
      if (second >= EUCJP_TO_UTF8_0XAB_MIN && second <= EUCJP_TO_UTF8_0XAB_MAX)
	utf8_string = eucjp_to_utf8_table_0xab[second - EUCJP_TO_UTF8_0XAB_MIN];
      break;

    case 0xad:
      if (second >= EUCJP_TO_UTF8_0XAD_MIN && second <= EUCJP_TO_UTF8_0XAD_MAX)
	utf8_string = eucjp_to_utf8_table_0xad[second - EUCJP_TO_UTF8_0XAD_MIN];
      break;

    case 0xb0:
      if (second >= EUCJP_TO_UTF8_0XB0_MIN && second <= EUCJP_TO_UTF8_0XB0_MAX)
	utf8_string = eucjp_to_utf8_table_0xb0[second - EUCJP_TO_UTF8_0XB0_MIN];
      break;

    case 0xb1:
      if (second >= EUCJP_TO_UTF8_0XB1_MIN && second <= EUCJP_TO_UTF8_0XB1_MAX)
	utf8_string = eucjp_to_utf8_table_0xb1[second - EUCJP_TO_UTF8_0XB1_MIN];
      break;

    case 0xb2:
      if (second >= EUCJP_TO_UTF8_0XB2_MIN && second <= EUCJP_TO_UTF8_0XB2_MAX)
	utf8_string = eucjp_to_utf8_table_0xb2[second - EUCJP_TO_UTF8_0XB2_MIN];
      break;

    case 0xb3:
      if (second >= EUCJP_TO_UTF8_0XB3_MIN && second <= EUCJP_TO_UTF8_0XB3_MAX)
	utf8_string = eucjp_to_utf8_table_0xb3[second - EUCJP_TO_UTF8_0XB3_MIN];
      break;

    case 0xb4:
      if (second >= EUCJP_TO_UTF8_0XB4_MIN && second <= EUCJP_TO_UTF8_0XB4_MAX)
	utf8_string = eucjp_to_utf8_table_0xb4[second - EUCJP_TO_UTF8_0XB4_MIN];
      break;

    case 0xb5:
      if (second >= EUCJP_TO_UTF8_0XB5_MIN && second <= EUCJP_TO_UTF8_0XB5_MAX)
	utf8_string = eucjp_to_utf8_table_0xb5[second - EUCJP_TO_UTF8_0XB5_MIN];
      break;

    case 0xb6:
      if (second >= EUCJP_TO_UTF8_0XB6_MIN && second <= EUCJP_TO_UTF8_0XB6_MAX)
	utf8_string = eucjp_to_utf8_table_0xb6[second - EUCJP_TO_UTF8_0XB6_MIN];
      break;

    case 0xf3:
      if (second >= EUCJP_TO_UTF8_0XF3_MIN && second <= EUCJP_TO_UTF8_0XF3_MAX)
	utf8_string = eucjp_to_utf8_table_0xf3[second - EUCJP_TO_UTF8_0XF3_MIN];
      break;

    case 0xf4:
      if (second >= EUCJP_TO_UTF8_0XF4_MIN && second <= EUCJP_TO_UTF8_0XF4_MAX)
	utf8_string = eucjp_to_utf8_table_0xf4[second - EUCJP_TO_UTF8_0XF4_MIN];
      break;

    case 0xf5:
      if (second >= EUCJP_TO_UTF8_0XF5_MIN && second <= EUCJP_TO_UTF8_0XF5_MAX)
	utf8_string = eucjp_to_utf8_table_0xf5[second - EUCJP_TO_UTF8_0XF5_MIN];
      break;

    case 0xf6:
      if (second >= EUCJP_TO_UTF8_0XF6_MIN && second <= EUCJP_TO_UTF8_0XF6_MAX)
	utf8_string = eucjp_to_utf8_table_0xf6[second - EUCJP_TO_UTF8_0XF6_MIN];
      break;

    case 0xf7:
      if (second >= EUCJP_TO_UTF8_0XF7_MIN && second <= EUCJP_TO_UTF8_0XF7_MAX)
	utf8_string = eucjp_to_utf8_table_0xf7[second - EUCJP_TO_UTF8_0XF7_MIN];
      break;

    case 0xf8:
      if (second >= EUCJP_TO_UTF8_0XF8_MIN && second <= EUCJP_TO_UTF8_0XF8_MAX)
	utf8_string = eucjp_to_utf8_table_0xf8[second - EUCJP_TO_UTF8_0XF8_MIN];
      break;

    case 0xf9:
      if (second >= EUCJP_TO_UTF8_0XF9_MIN && second <= EUCJP_TO_UTF8_0XF9_MAX)
	utf8_string = eucjp_to_utf8_table_0xf9[second - EUCJP_TO_UTF8_0XF9_MIN];
      break;

    case 0xfa:
      if (second >= EUCJP_TO_UTF8_0XFA_MIN && second <= EUCJP_TO_UTF8_0XFA_MAX)
	utf8_string = eucjp_to_utf8_table_0xfa[second - EUCJP_TO_UTF8_0XFA_MIN];
      break;

    case 0xfb:
      if (second >= EUCJP_TO_UTF8_0XFB_MIN && second <= EUCJP_TO_UTF8_0XFB_MAX)
	utf8_string = eucjp_to_utf8_table_0xfb[second - EUCJP_TO_UTF8_0XFB_MIN];
      break;

    case 0xfc:
      if (second >= EUCJP_TO_UTF8_0XFC_MIN && second <= EUCJP_TO_UTF8_0XFC_MAX)
	utf8_string = eucjp_to_utf8_table_0xfc[second - EUCJP_TO_UTF8_0XFC_MIN];
      break;

    case 0xfd:
      if (second >= EUCJP_TO_UTF8_0XFD_MIN && second <= EUCJP_TO_UTF8_0XFD_MAX)
	utf8_string = eucjp_to_utf8_table_0xfd[second - EUCJP_TO_UTF8_0XFD_MIN];
      break;

    case 0xfe:
      if (second >= EUCJP_TO_UTF8_0XFE_MIN && second <= EUCJP_TO_UTF8_0XFE_MAX)
	utf8_string = eucjp_to_utf8_table_0xfe[second - EUCJP_TO_UTF8_0XFE_MIN];
      break;
    }

  if (utf8_string == NULL)
    {
      /* Ʀ(={ 0xe2, 0x96, 0xa1 })ɤ */
#if DEBUG_ICONV
      fprintf(stderr, "first=%02x second=%02x: %c%c\n",
	      first, second, first, second);
#endif
      utf8_string = "\342\226\241";
#if 0	/* 1 byteʤ褦ˤɤ */
      first = '\0';
#endif
    }

  if ((first & 0x80) != 0)
    {
      *inbuf += 2;
      *inbytesleft -= 2;
    }
  else
    {
      *inbuf += 1;
      *inbytesleft -= 1;
    }

  if ((utf8_string[0] & 0x80) == 0)
    {
      /* 1 byte */
      if (*outbytesleft < 1)
	{
	  errno = E2BIG;
	  return -1;
	}
      (*outbuf)[0] = utf8_string[0];
      *outbuf += 1;
      *outbytesleft -= 1;
    }
  else if ((utf8_string[0] & 0xe0) == 0xc0)
    {
      /* 2 byte */
      if (*outbytesleft < 2)
	{
	  errno = E2BIG;
	  return -1;
	}
      (*outbuf)[0] = utf8_string[0];
      (*outbuf)[1] = utf8_string[1];
      *outbuf += 2;
      *outbytesleft -= 2;
    }
  else
    {
      /* 3 byte */
      if (*outbytesleft < 3)
	{
	  errno = E2BIG;
	  return -1;
	}
      (*outbuf)[0] = utf8_string[0];
      (*outbuf)[1] = utf8_string[1];
      (*outbuf)[2] = utf8_string[2];
      *outbuf += 3;
      *outbytesleft -= 3;
    }

#if DEBUG_ICONV_MOST
  fprintf(stderr, "<2>first=%02x second=%02x: \"%c%c\" => \"%s\"\n",
	  first, second, first, second, utf8_string);
#endif

  return 0;
}


#if TRACE_MEMORY_USAGE
/*
 * ꡼åϢ
 */
static pthread_mutex_t memory_trace_lock;
static GHashTable *memory_trace;


#define MEMORY_TRACE_LOCK				\
  if (pthread_mutex_lock(&memory_trace_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't lock a mutex.\n");	\
      abort();						\
    }

#define MEMORY_TRACE_UNLOCK				\
  if (pthread_mutex_unlock(&memory_trace_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't unlock a mutex.\n");	\
      abort();						\
    }


void
init_memory_trace(void)
{
  if (pthread_mutex_init(&memory_trace_lock, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a mutex.\n");
      abort();
    }

  memory_trace = g_hash_table_new_full(g_direct_hash, g_direct_equal,
				       NULL, (GDestroyNotify)g_free);
}


void
register_pointer(gpointer pointer, const char *where)
{
  gpointer previous;
  MEMORY_TRACE_LOCK
    {
      previous = g_hash_table_lookup(memory_trace, pointer);
      g_hash_table_insert(memory_trace, pointer, g_strdup(where));
    }
  MEMORY_TRACE_UNLOCK;

  if (previous != NULL)
    fprintf(stderr, "%p has already been registered: \"%s\" -> \"%s\"\n",
	    pointer, (char *)previous, where);
}


void
unregister_pointer(gpointer pointer, const char *where)
{
  gpointer usage;
  MEMORY_TRACE_LOCK
    {
      usage = g_hash_table_lookup(memory_trace, pointer);
      g_hash_table_remove(memory_trace, pointer);
    }
  MEMORY_TRACE_UNLOCK;

  if (usage == NULL && pointer != NULL)
    fprintf(stderr, "%p not registered: %s\n", pointer, where);
}


static void
print_pointer_info(gpointer key, gpointer value, gpointer user_data)
{
  if (strstr((const char *)value, "g_str") != NULL)
    fprintf(stderr, "%p \"%s\": %s\n", key, (char *)key, (char *)value);
  else
    fprintf(stderr, "%p: %s\n", key, (char *)value);
}


void
dump_pointers()
{
  MEMORY_TRACE_LOCK
    {
      g_hash_table_foreach(memory_trace, print_pointer_info, NULL);
    }
  MEMORY_TRACE_UNLOCK;
}
#endif


#if TRACE_REFCOUNT
static pthread_mutex_t refcount_trace_lock;
static GHashTable *refcount_trace;
static GSList *consistent_object_list;


#define REFCOUNT_TRACE_LOCK				\
  if (pthread_mutex_lock(&refcount_trace_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't lock a mutex.\n");	\
      abort();						\
    }

#define REFCOUNT_TRACE_UNLOCK				\
  if (pthread_mutex_unlock(&refcount_trace_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't unlock a mutex.\n");	\
      abort();						\
    }


static void
delete_refcount_history(GSList *history)
{
  g_slist_foreach(history, (GFunc)g_free, NULL);
  g_slist_free(history);
}


void
init_refcount_trace(void)
{
  if (pthread_mutex_init(&refcount_trace_lock, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a mutex.\n");
      abort();
    }

  refcount_trace = g_hash_table_new(g_direct_hash, g_direct_equal);
  consistent_object_list = NULL;
}


void
trace_register_g_object(gpointer pointer, const char *file, int line)
{
  GSList *previous;
  char where[1024];
  GObject *object = (GObject *)pointer;
  snprintf(where, 1024, "register_g_object(%p) at %s, line %d: ref_count=%d",
	   object, file, line, object->ref_count);
  REFCOUNT_TRACE_LOCK
  {
    previous = g_hash_table_lookup(refcount_trace, object);
    if (previous != NULL)
      {
	if (g_slist_find(consistent_object_list, object) != NULL)
	  {
	    consistent_object_list = g_slist_remove(consistent_object_list,
						    object);
	    delete_refcount_history(previous);
	    previous = NULL;
	  }
      }
    g_hash_table_insert(refcount_trace, object,
			g_slist_append(previous, g_strdup(where)));
  }
  REFCOUNT_TRACE_UNLOCK;
}


static void
output_warning(GObject *object)
{
  fprintf(stderr, "%p seems not to be a valid pointer to GObject\n", object);
}


void
trace_g_object_ref(gpointer pointer, const char *file, int line)
{
  GSList *history;
  char where[1024];
  GObject *object = (GObject *)pointer;
  snprintf(where, 1024, "g_object_ref(%p) at %s, line %d: ref_count=%d",
	   object, file, line, object->ref_count);
  if (object->ref_count < 1 || object->ref_count >= 100)
    output_warning(object);
  REFCOUNT_TRACE_LOCK
  {
    history = g_hash_table_lookup(refcount_trace, object);
    g_hash_table_insert(refcount_trace, object,
			g_slist_append(history, g_strdup(where)));
  }
  REFCOUNT_TRACE_UNLOCK;
  g_object_ref(object);
}


void
trace_g_object_unref(gpointer pointer, const char *file, int line)
{
  GSList *history;
  char where[1024];
  GObject *object = (GObject *)pointer;
  snprintf(where, 1024, "g_object_unref(%p) at %s, line %d: ref_count=%d",
	   object, file, line, object->ref_count);
  if (object->ref_count < 1 || object->ref_count >= 100)
    output_warning(object);
  REFCOUNT_TRACE_LOCK
  {
    history = g_hash_table_lookup(refcount_trace, object);
    g_hash_table_insert(refcount_trace, object,
			g_slist_append(history, g_strdup(where)));
    if (object->ref_count == 1)
      consistent_object_list = g_slist_append(consistent_object_list, object);
    else if (object->ref_count == 0)
      consistent_object_list = g_slist_remove(consistent_object_list, object);
  }
  REFCOUNT_TRACE_UNLOCK;
  if (object->ref_count > 0 && object->ref_count < 100)
    g_object_unref(object);
}


static void
print_history(const char *where)
{
  fprintf(stderr, "%s\n", where);
}


static void
remove_consistent_object(gpointer object, gpointer unused)
{
  GSList *history = g_hash_table_lookup(refcount_trace, object);
#if 0
  fprintf(stderr, "*** Consistent object %p\n", object);
  g_slist_foreach(history, (GFunc)print_history, NULL);
  fprintf(stderr, "******\n\n");
#endif
  delete_refcount_history(history);
  g_hash_table_remove(refcount_trace, object);
}


static void
print_refcount_info(GObject *object, GSList *history, gpointer unused)
{
  fprintf(stderr, "*** Inconsistent object %p\n", object);
  g_slist_foreach(history, (GFunc)print_history, NULL);
  fprintf(stderr, "******\n\n");
  delete_refcount_history(history);
}


void
dump_objects(void)
{
  REFCOUNT_TRACE_LOCK
  {
    g_slist_foreach(consistent_object_list, remove_consistent_object, NULL);
    g_hash_table_foreach(refcount_trace, (GHFunc)print_refcount_info, NULL);
  }
  REFCOUNT_TRACE_UNLOCK;
}


void
ochu_object_ref(gpointer pointer)
{
  GObject *object = (GObject *)pointer;
  OCHU_OBJECT_REF(object);
}


void
ochu_object_unref(gpointer pointer)
{
  GObject *object = (GObject *)pointer;
  OCHU_OBJECT_UNREF(object);
}
#endif


