/*
 * Copyright (c) 2003-2004 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: bbs_thread_ui.c,v 1.101.2.18 2004/06/30 09:03:13 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "ochusha_thread_2ch.h"
#include "ochusha_utils_2ch.h"

#include "ochusha_ui.h"
#include "bbs_thread_ui.h"
#include "bbs_thread_view.h"
#include "boardlist_ui.h"
#include "bulletin_board_ui.h"
#include "download_ui.h"
#include "image_ui.h"
#include "icon_label.h"

#include "regex_utils.h"
#include "ugly_gtk2utils.h"

#include "htmlutils.h"
#include "worker.h"
#include "utils.h"

#include <pthread.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


#define DEBUG_POPUP		0

typedef struct _PopupContext PopupContext;

#if GTK_MINOR_VERSION <= 2
static void link_popup_menu_callback(gpointer data, guint action,
				     GtkWidget *widget);
#else
static void popup_menu_open_link_cb(GtkAction *action,
				    OchushaApplication *application);
static void popup_menu_open_link_in_tab_cb(GtkAction *action,
					   OchushaApplication *application);
static void popup_menu_open_link_with_web_browser_cb(GtkAction *action,
					OchushaApplication *application);
static void popup_menu_save_link_as_cb(GtkAction *action,
				       OchushaApplication *application);
static void popup_menu_copy_link_location_cb(GtkAction *action,
					     OchushaApplication *application);
static void popup_menu_tenure_link_image_cb(GtkAction *action,
					    OchushaApplication *application);
static void popup_menu_a_bone_link_image_cb(GtkAction *action,
					    OchushaApplication *application);
static void popup_menu_a_bone_link_server_cb(GtkAction *action,
					     OchushaApplication *application);
#endif

static void link_popup_menu_destructed_cb(gpointer data);

#if GTK_MINOR_VERSION <= 2
static void response_popup_menu_callback(gpointer data, guint action,
				     GtkWidget *widget);
#else
static void popup_menu_insert_bookmark_cb(GtkAction *action,
					  OchushaApplication *application);
static void popup_menu_write_response_cb(GtkAction *action,
					 OchushaApplication *application);
#endif
static void response_popup_menu_destructed_cb(gpointer data);

static gboolean link_mouse_over_cb(BBSThreadView *view, GdkEventMotion *event,
				   OchushaBBSThread *thread, const gchar *link,
				   OchushaApplication *application);
static gboolean link_mouse_out_cb(BBSThreadView *view, GdkEventAny *event,
				  OchushaBBSThread *thread,
				  OchushaApplication *application);
static gboolean link_mouse_press_cb(BBSThreadView *view, GdkEventButton *event,
				    OchushaBBSThread *thread,
				    const gchar *link,
				    OchushaApplication *application);
static gboolean link_mouse_release_cb(BBSThreadView *view,
				      GdkEventButton *event,
				      OchushaBBSThread *thread,
				      const gchar *link,
				      OchushaApplication *application);
static void copy_clipboard_cb(BBSThreadView *view,
			      OchushaApplication *application);
static void scroll_view_cb(GtkWidget *view, GtkScrollType scroll,
			   OchushaApplication *application);
static void write_response_cb(BBSThreadView *view,
			      OchushaApplication *application);
static void interactive_search_cb(BBSThreadView *view,
				  BBSThreadViewSearchAction action,
				  OchushaApplication *application);
static void save_initial_thread_view_position(BBSThreadView *view,
					      BBSThreadGUIInfo *info,
					      GtkTextMark *mark);

static void show_popup(OchushaApplication *application,
		       OchushaBBSThread *thread,
		       const gchar *link, int x_pos, int y_pos,
		       BBSThreadView *parent);
static gboolean popup_delay_timeout(PopupContext *popup_context);
static void show_popup_real(PopupContext *popup_context);
static void hide_popup(PopupContext *popup_context);
static void insert_bookmark(BBSThreadView *view, OchushaBBSThread *thread,
			    int res_num, OchushaApplication *application);
static void render_bbs_thread(WorkerThread *employee, gpointer args);
static gboolean start_thread_rendering(OchushaBBSThread *thread,
				       const gchar *title,
				       gpointer user_data);
static gboolean render_response(OchushaBBSThread *thread, int number,
				const OchushaBBSResponse *response,
				gpointer user_data);
static gboolean render_broken_response(OchushaBBSThread *thread, int number,
				       gpointer user_data);
static gboolean end_thread_rendering(OchushaBBSThread *thread,
				     gboolean finished,
				     gpointer user_data);
static void start_element_cb(void *context, const char *name,
			     const char *const attrs[]);
static void end_element_cb(void *context, const char *name);
static void characters_cb(void *context, const char *ch, int len);

static pthread_mutex_t bbs_thread_ui_lock;

/* ơС˥ᥤȤʤɤɽ뤿context ID */
static guint link_field_id;

/* Ǹ˥ơСɽåФmessage ID */
static guint message_id;

/*
 * ơС˥Фͥåȥξ֤ɽ뤿
 * context ID
 */
guint thread_netstat_id;


#define BBS_THREAD_UI_LOCK				\
  if (pthread_mutex_lock(&bbs_thread_ui_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't lock a mutex.\n");	\
      abort();						\
    }

#define BBS_THREAD_UI_UNLOCK				\
  if (pthread_mutex_unlock(&bbs_thread_ui_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't unlock a mutex.\n");	\
      abort();						\
    }


typedef struct _RendererContext
{
  ElementHandler handler;
  BBSThreadView *view;
  OchushaBBSThread *thread;
  BBSThreadGUIInfo *info;
  OchushaApplication *application;
  int last_read_number;
  int last_rendered_response;
  gboolean title_shown;

  gchar *link;
  gchar *link_text_buffer;
  guint link_text_buffer_size;
  guint link_text_length;

  int number_of_gt_characters;
  gboolean last_text_is_link;

  gboolean show_mailto_literally;

  GHashTable *tag_table;

  GtkTextTag *thread_title_tag;
  GtkTextTag *response_header_tag;
  GtkTextTag *response_name_tag;
  GtkTextTag *response_paragraph_tag;
  GtkTextTag *bold_tag;

  gboolean transparent_a_bone;
  gpointer a_bone_by_name;
  gpointer a_bone_by_id;
  gpointer a_bone_by_content;

  regex_t comma_separated_link_pattern;
  regex_t informal_link_pattern;
} RendererContext;


#define DEFAULT_LINK_BUFFER_SIZE	512
#define DEFAULT_BUFFER_SIZE		4096

static GHashTable *tag_table = NULL;
static int stack_depth = 0;


struct _PopupContext
{
  RendererContext renderer_context;

  GtkWidget *popup_window;
  GtkContainer *popup_frame;
  GtkWidget *image;

  PopupContext *parent_context;

  gchar *url;
  guint delay_id;
  guint close_delay_id;
  int x_pos;
  int y_pos;

  gboolean now_sticked;
  gboolean now_pointed;
};


/* PopupϢ */


#if GTK_MINOR_VERSION <= 2
enum {
  LINK_POPUP_MENU_OPEN_LINK = 1,
  LINK_POPUP_MENU_OPEN_LINK_IN_TAB,
  LINK_POPUP_MENU_OPEN_LINK_WITH_WEB_BROWSER,
  LINK_POPUP_MENU_SAVE_LINK_AS,
  LINK_POPUP_MENU_COPY_LINK_LOCATION,
  LINK_POPUP_MENU_TENURE_LINK_IMAGE,
  LINK_POPUP_MENU_A_BONE_LINK_IMAGE,
  LINK_POPUP_MENU_A_BONE_LINK_SERVER,
};


enum {
  RESPONSE_POPUP_MENU_INSERT_BOOKMARK = 1,
  RESPONSE_POPUP_MENU_WRITE_RESPONSE
};


static GtkItemFactory *link_popup_menu_item_factory = NULL;
static GtkItemFactory *response_popup_menu_item_factory = NULL;
#else
static GtkActionGroup *action_group = NULL;
static GtkUIManager *ui_manager = NULL;
#endif
static GQuark bbs_thread_gui_info_id;
static GQuark popup_context_id;
static GQuark popup_parent_context_id;
static gboolean popup_menu_shown = FALSE;


void
prepare_thread_ui_initialization(OchushaApplication *application)
{
  if (pthread_mutex_init(&bbs_thread_ui_lock, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a mutex.\n");
      abort();
    }

  bbs_thread_gui_info_id = g_quark_from_static_string("BBSThreadUI::GUIInfo");
  popup_context_id = g_quark_from_static_string("BBSThreadUI::PopupContext");
  popup_parent_context_id
    = g_quark_from_static_string("BBSThreadUI::PopupParentContext");
}


static GtkTextTag *thread_title_tag;
static GtkTextTag *response_header_tag;
static GtkTextTag *response_name_tag;
static GtkTextTag *response_paragraph_tag;
static GtkTextTag *bold_tag;


static PangoFontDescription *thread_title_font_description = NULL;
static PangoFontDescription *thread_body_font_description = NULL;
static GtkWidget *font_selection_dialog = NULL;


static void
font_selection_dialog_response_cb(GtkWidget *dialog, int response_id,
				  OchushaApplication *application)
{
  GtkFontSelectionDialog *font_selection;
  gchar *font_name = NULL;

  g_return_if_fail(GTK_IS_FONT_SELECTION_DIALOG(dialog));

  font_selection = GTK_FONT_SELECTION_DIALOG(dialog);

  switch (response_id)
    {
    case GTK_RESPONSE_APPLY:
      font_name = gtk_font_selection_dialog_get_font_name(font_selection);
      preview_thread_view_font(font_name);
      g_free(font_name);
      return;

    case GTK_RESPONSE_OK:
      font_name	= gtk_font_selection_dialog_get_font_name(font_selection);
      set_thread_view_font(font_name);
      if (application->thread_view_font_name != NULL)
	G_FREE(application->thread_view_font_name);
      application->thread_view_font_name = font_name;
      break;

    case GTK_RESPONSE_CANCEL:
      reset_thread_view_font();
      break;
    }

  gtk_widget_hide(dialog);
  gtk_widget_unrealize(dialog);
  gtk_widget_destroy(dialog);
}


void
select_thread_view_font(OchushaApplication *application)
{
  char *font_name;
  if (font_selection_dialog != NULL)
    return;

  font_selection_dialog
    = gtk_font_selection_dialog_new(_("Thread View Font Selection"));

  font_name = application->thread_view_font_name;

  if (font_name != NULL && *font_name != '\0')
    gtk_font_selection_dialog_set_font_name(GTK_FONT_SELECTION_DIALOG(font_selection_dialog),
					    font_name);

  g_signal_connect(G_OBJECT(font_selection_dialog), "response",
		   G_CALLBACK(font_selection_dialog_response_cb), application);
  g_signal_connect(G_OBJECT(font_selection_dialog), "destroy",
		   G_CALLBACK(gtk_widget_destroyed), &font_selection_dialog);

  gtk_widget_show(GTK_FONT_SELECTION_DIALOG(font_selection_dialog)->apply_button);
  gtk_widget_show_all(font_selection_dialog);
}


void
set_thread_view_font(const char *font_name)
{
  g_return_if_fail(thread_title_tag != NULL);
  g_return_if_fail(response_header_tag != NULL);
  g_return_if_fail(response_paragraph_tag != NULL);

  if (font_name == NULL || *font_name == '\0')
    return;

  if (thread_title_font_description != NULL)
    {
      pango_font_description_free(thread_title_font_description);
      thread_title_font_description = NULL;
    }
  if (thread_body_font_description != NULL)
    {
      pango_font_description_free(thread_body_font_description);
      thread_body_font_description = NULL;
    }

  thread_body_font_description = pango_font_description_from_string(font_name);
  g_object_set(G_OBJECT(response_paragraph_tag),
	       "font-desc", thread_body_font_description,
	       NULL);
  g_object_set(G_OBJECT(response_header_tag),
	       "font-desc", thread_body_font_description,
	       NULL);
  thread_title_font_description
    = pango_font_description_copy(thread_body_font_description);
  pango_font_description_set_size(thread_title_font_description,
				  pango_font_description_get_size(thread_body_font_description) * 1.2);
  g_object_set(G_OBJECT(thread_title_tag),
	       "font-desc", thread_title_font_description,
	       NULL);
}


void
reset_thread_view_font(void)
{
  g_return_if_fail(thread_title_tag != NULL);
  g_return_if_fail(response_header_tag != NULL);
  g_return_if_fail(response_paragraph_tag != NULL);

  if (thread_title_font_description == NULL
      || thread_body_font_description == NULL)
    {
      PangoFontDescription *default_desc
	= pango_font_description_new();

      g_object_set(G_OBJECT(response_paragraph_tag),
		   "font-desc", default_desc,
		   NULL);
      g_object_set(G_OBJECT(response_header_tag),
		   "font-desc", default_desc,
		   NULL);
      g_object_set(G_OBJECT(thread_title_tag),
		   "font-desc", default_desc,
		   NULL);
      pango_font_description_free(default_desc);

      return;
    }

  g_object_set(G_OBJECT(response_paragraph_tag),
	       "font-desc", thread_body_font_description,
	       NULL);
  g_object_set(G_OBJECT(response_header_tag),
	       "font-desc", thread_body_font_description,
	       NULL);
  g_object_set(G_OBJECT(thread_title_tag),
	       "font-desc", thread_title_font_description,
	       NULL);
}


void
preview_thread_view_font(const char *font_name)
{
  PangoFontDescription *font_desc;
  g_return_if_fail(thread_title_tag != NULL);
  g_return_if_fail(response_header_tag != NULL);
  g_return_if_fail(response_paragraph_tag != NULL);

  if (font_name == NULL || *font_name == '\0')
    return;

  font_desc = pango_font_description_from_string(font_name);
  g_object_set(G_OBJECT(response_paragraph_tag),
	       "font-desc", font_desc,
	       NULL);
  g_object_set(G_OBJECT(response_header_tag),
	       "font-desc", font_desc,
	       NULL);

  pango_font_description_set_size(font_desc,
			pango_font_description_get_size(font_desc) * 1.2);
  pango_font_description_set_weight(font_desc, PANGO_WEIGHT_BOLD);
  g_object_set(G_OBJECT(thread_title_tag),
	       "font-desc", font_desc,
	       NULL);
  pango_font_description_free(font_desc);
}


#if 0
static void
popup_window_size_allocate_cb(GtkWidget *widget, GtkAllocation *allocation,
			      PopupContext *popup_context)
{
  fprintf(stderr, "popup_window_size_allocate_cb(%p, %p, %p): w=%d, h=%d\n",
	  widget, allocation, popup_context,
	  allocation->width, allocation->height);
}


static void
popup_window_size_request_cb(GtkWidget *widget, GtkRequisition *requisition,
			     PopupContext *popup_context)
{
  fprintf(stderr, "popup_window_size_request_cb(%p, %p, %p): w=%d, h=%d\n",
	  widget, requisition, popup_context,
	  requisition->width, requisition->height);
}
#endif


static void
initialize_popup_context(PopupContext *popup_context,
			 OchushaApplication *application)
{
  GtkWidget *frame;
  GtkWindow *parent_window;

#if DEBUG_POPUP
  fprintf(stderr, "initialize_popup_context(%p, %p)\n",
	  popup_context, application);
#endif

  popup_context->popup_window = gtk_window_new(GTK_WINDOW_POPUP);
#if 0
  g_signal_connect(G_OBJECT(popup_context->popup_window), "size_allocate",
		   G_CALLBACK(popup_window_size_allocate_cb), popup_context);
  g_signal_connect(G_OBJECT(popup_context->popup_window), "size_request",
		   G_CALLBACK(popup_window_size_request_cb), popup_context);
#endif
  gtk_window_set_resizable(GTK_WINDOW(popup_context->popup_window),
			   FALSE);

  frame = gtk_frame_new(NULL);
  gtk_widget_show(frame);
  popup_context->popup_frame = GTK_CONTAINER(frame);
  gtk_container_add(GTK_CONTAINER(popup_context->popup_window), frame);
  
  popup_context->renderer_context.application = application;

  popup_context->renderer_context.thread_title_tag = thread_title_tag;
  popup_context->renderer_context.response_header_tag = response_header_tag;
  popup_context->renderer_context.response_name_tag = response_name_tag;
  popup_context->renderer_context.response_paragraph_tag
    = response_paragraph_tag;
  popup_context->renderer_context.bold_tag = bold_tag;
  popup_context->renderer_context.tag_table = tag_table;

  /* popup_context->now_sticked = FALSE; */
  /* popup_context->now_pointed = FALSE; */
  parent_window
    = (popup_context->parent_context == NULL
       ? popup_context->renderer_context.application->top_level
       : GTK_WINDOW(popup_context->parent_context->popup_window));
  gtk_window_set_transient_for(GTK_WINDOW(popup_context->popup_window),
			       parent_window);
}


void
initialize_thread_ui(OchushaApplication *application)
{
  GtkWidget *dummy_widget;
  BBSThreadView *dummy_view;

  /* ݥåץåץ˥塼ν */
#if GTK_MINOR_VERSION <= 2
  GtkItemFactoryEntry menu_items[] =
    {
      {
	_("/_Open Link"),				/* menu path */
	NULL,						/* accelerator */
	link_popup_menu_callback,			/* callback_func */
	LINK_POPUP_MENU_OPEN_LINK,			/* act */
	NULL						/* type */
      },
      {
	_("/Open Link in New _Tab"),			/* menu path */
	NULL,						/* accelerator */
	link_popup_menu_callback,			/* callback_func */
	LINK_POPUP_MENU_OPEN_LINK_IN_TAB,		/* act */
	NULL						/* type */
      },
      {
	_("/Open Link with _Web Browser"),		/* menu path */
	NULL,						/* accelerator */
	link_popup_menu_callback,			/* callback_func */
	LINK_POPUP_MENU_OPEN_LINK_WITH_WEB_BROWSER,	/* act */
	NULL						/* type */
      },
      {
	_("/-------"),
	NULL,
	NULL,
	0,
	(char *)"<Separator>"
      },
      {
	_("/_Save Link As..."),				/* menu path */
	NULL,						/* accelerator */
	link_popup_menu_callback,			/* callback_func */
	LINK_POPUP_MENU_SAVE_LINK_AS,			/* act */
	NULL						/* type */
      },
      {
	_("/_Copy Link Location"),			/* menu path */
	NULL,						/* accelerator */
	link_popup_menu_callback,			/* callback_func */
	LINK_POPUP_MENU_COPY_LINK_LOCATION,		/* act */
	NULL						/* type */
      },
      {
	_("/-------"),
	NULL,
	NULL,
	0,
	(char *)"<Separator>"
      },
      {
	_("/Tenure Link Image"),			/* menu path */
	NULL,						/* accelerator */
	link_popup_menu_callback,			/* callback_func */
	LINK_POPUP_MENU_TENURE_LINK_IMAGE,		/* act */
	NULL,						/* type */
      },
      {
	_("/-------"),
	NULL,
	NULL,
	0,
	(char *)"<Separator>"
      },
      {
	_("/A Bone Link Image"),			/* menu path */
	NULL,						/* accelerator */
	link_popup_menu_callback,			/* callback_func */
	LINK_POPUP_MENU_A_BONE_LINK_IMAGE,		/* act */
	NULL						/* type */
      },
      {
	_("/A Bone Link Server"),			/* menu path */
	NULL,						/* accelerator */
	link_popup_menu_callback,			/* callback_func */
	LINK_POPUP_MENU_A_BONE_LINK_SERVER,		/* act */
	NULL						/* type */
      },
    };
  int nmenu_items = sizeof(menu_items) / sizeof(menu_items[0]);

  GtkItemFactoryEntry response_menu_items[] =
    {
      {
	_("/Write Response"),				/* menu path */
	NULL,						/* accelerator */
	response_popup_menu_callback,			/* callback_func */
	RESPONSE_POPUP_MENU_WRITE_RESPONSE,		/* act */
	NULL						/* type */
      },
      {
	_("/Insert Bookmark"),				/* menu path */
	NULL,						/* accelerator */
	response_popup_menu_callback,			/* callback_func */
	RESPONSE_POPUP_MENU_INSERT_BOOKMARK,		/* act */
	NULL						/* type */
      },
    };
  int response_nmenu_items
    = sizeof(response_menu_items) / sizeof(response_menu_items[0]);

  link_popup_menu_item_factory = gtk_item_factory_new(GTK_TYPE_MENU,
						      "<ThreadLinkPopup>",
						      NULL);
  gtk_item_factory_create_items(link_popup_menu_item_factory,
				nmenu_items, menu_items, application);

  response_popup_menu_item_factory = gtk_item_factory_new(GTK_TYPE_MENU,
							  "<ResponsePopup>",
							  NULL);
  gtk_item_factory_create_items(response_popup_menu_item_factory,
				response_nmenu_items, response_menu_items,
				application);
#else
  static GtkActionEntry popup_menu_entries[] =
    {
      {
	"OpenLink", GTK_STOCK_OPEN,
	N_("_Open Link"), NULL,
	N_("Open Link"),
	(GCallback)popup_menu_open_link_cb
      },
      {
	"OpenLinkInTab", GTK_STOCK_OPEN,
	N_("Open Link in New _Tab"), NULL,
	N_("Open Link in New Tab"),
	(GCallback)popup_menu_open_link_in_tab_cb
      },
      {
	"OpenLinkWithWebBrowser", GTK_STOCK_EXECUTE,
	N_("Open Link with _Web Browser"), NULL,
	N_("Open Link with Web Browser"),
	(GCallback)popup_menu_open_link_with_web_browser_cb
      },
      {
	"SaveLinkAs", GTK_STOCK_SAVE_AS,
	N_("_Save Link as..."), NULL,
	N_("Save Link as..."),
	(GCallback)popup_menu_save_link_as_cb
      },
      {
	"CopyLinkLocation", GTK_STOCK_COPY,
	N_("_Copy Link Location"), NULL,
	N_("Copy Link Location"),
	(GCallback)popup_menu_copy_link_location_cb
      },
      {
	"TenureLinkImage", GTK_STOCK_SAVE,
	N_("Tenure Link Image"), NULL,
	N_("Tenure Link Image"),
	(GCallback)popup_menu_tenure_link_image_cb
      },
      {
	"ABoneLinkImage", GTK_STOCK_DELETE,
	N_("A Bone Link Image"), NULL,
	N_("A Bone Link Image"),
	(GCallback)popup_menu_a_bone_link_image_cb
      },
      {
	"ABoneLinkServer", GTK_STOCK_DELETE,
	N_("A Bone Link Server"), NULL,
	N_("A Bone Link Server"),
	(GCallback)popup_menu_a_bone_link_server_cb
      },
      {
	"InsertBookmark", GTK_STOCK_INDEX,
	N_("Insert Bookmark"), NULL,
	N_("Insert Bookmark"),
	(GCallback)popup_menu_insert_bookmark_cb
      },
      {
	"WriteResponse", OCHUSHA_STOCK_WRITE_RESPONSE,
	N_("_Write Response"), NULL,
	N_("Write Response"),
	(GCallback)popup_menu_write_response_cb
      }
    };

  static const char *popup_menu_ui =
    "<ui>"
    "  <popup name=\"ThreadLinkPopup\">"
    "    <menuitem action=\"OpenLink\"/>"
    "    <menuitem action=\"OpenLinkInTab\"/>"
    "    <menuitem action=\"OpenLinkWithWebBrowser\"/>"
    "    <separator/>"
    "    <menuitem action=\"SaveLinkAs\"/>"
    "    <menuitem action=\"CopyLinkLocation\"/>"
    "    <separator/>"
    "    <menuitem action=\"TenureLinkImage\"/>"
    "    <separator/>"
    "    <menuitem action=\"ABoneLinkImage\"/>"
    "    <menuitem action=\"ABoneLinkServer\"/>"
    "  </popup>"
    "  <popup name=\"ResponsePopup\">"
    "    <menuitem action=\"WriteResponse\"/>"
    "    <menuitem action=\"InsertBookmark\"/>"
    "  </popup>"
    "</ui>";

  GError *error = NULL;

  action_group = gtk_action_group_new("ThreadViewPopupActions");
  gtk_action_group_set_translation_domain(action_group, PACKAGE_NAME);
  gtk_action_group_add_actions(action_group,
			       popup_menu_entries,
			       G_N_ELEMENTS(popup_menu_entries),
			       application);

  ui_manager = gtk_ui_manager_new();
  gtk_ui_manager_insert_action_group(ui_manager, action_group, 0);

  if (!gtk_ui_manager_add_ui_from_string(ui_manager, popup_menu_ui, -1,
					 &error))
    {
      g_message("building menus failed: %s\n", error->message);
      g_error_free(error);
      exit(EXIT_FAILURE);
    }
#endif

  g_return_if_fail(tag_table == NULL);

  link_field_id = gtk_statusbar_get_context_id(application->statusbar,
					       "response-link-field");
  thread_netstat_id = gtk_statusbar_get_context_id(application->statusbar,
						   "thread-netstat");

  dummy_widget = bbs_thread_view_new(NULL);
  dummy_view = BBS_THREAD_VIEW(dummy_widget);

  thread_title_tag
    = bbs_thread_view_get_tag_by_name(dummy_view, "thread_title");
  response_header_tag
    = bbs_thread_view_get_tag_by_name(dummy_view, "response_header");
  response_name_tag
    = bbs_thread_view_get_tag_by_name(dummy_view, "response_name");
  response_paragraph_tag
    = bbs_thread_view_get_tag_by_name(dummy_view, "response_paragraph");
  bold_tag = bbs_thread_view_get_tag_by_name(dummy_view, "bold");

  /* Τ */
  gtk_object_sink(GTK_OBJECT(dummy_view));

  {
#if 0
    GtkTextTag *tag;
    GtkTextTagTable *default_tag_table = bbs_thread_view_class_get_default_tag_table(BBS_THREAD_VIEW_GET_CLASS(job_args->view));
#endif
    tag_table = g_hash_table_new(g_str_hash, g_str_equal);

    /* TODO: äȡ */
    g_hash_table_insert(tag_table, (char *)"b", bold_tag);
  }

  if (application->thread_view_font_name != NULL
      && *application->thread_view_font_name != '\0')
    set_thread_view_font(application->thread_view_font_name);
}


static void
bbs_thread_info_free(BBSThreadGUIInfo *info)
{
  if (info->offsets != NULL)
    {
      G_FREE(info->offsets);
      info->offsets = NULL;
    }

  if (info->last_name != NULL)
    {
      G_FREE(info->last_name);
      info->last_name = NULL;
    }

  if (info->last_mail != NULL)
    {
      G_FREE(info->last_mail);
      info->last_mail = NULL;
    }

  if (info->a_bone_by_name_pattern != NULL)
    {
      G_FREE(info->a_bone_by_name_pattern);
      info->a_bone_by_name_pattern = NULL;
    }

  if (info->a_bone_by_id_pattern != NULL)
    {
      G_FREE(info->a_bone_by_id_pattern);
      info->a_bone_by_id_pattern = NULL;
    }

  if (info->a_bone_by_content_pattern != NULL)
    {
      G_FREE(info->a_bone_by_content_pattern);
      info->a_bone_by_content_pattern = NULL;
    }

  if (info->write_dialog != NULL)
    {
      gtk_widget_hide(info->write_dialog);
      gtk_widget_unrealize(info->write_dialog);
      gtk_widget_destroy(info->write_dialog);
    }

  G_FREE(info);
}


BBSThreadGUIInfo *
ensure_bbs_thread_info(OchushaBBSThread *thread)
{
  BBSThreadGUIInfo *info = g_object_get_qdata(G_OBJECT(thread),
					      bbs_thread_gui_info_id);
  if (info == NULL)
    {
      info = G_NEW0(BBSThreadGUIInfo, 1);
      g_object_set_qdata_full(G_OBJECT(thread), bbs_thread_gui_info_id,
			      info, (GDestroyNotify)bbs_thread_info_free);
    }

  return info;
}


void
specify_thread_view_point(OchushaBBSThread *thread, PanedNotebook *board_view,
			  int res_num)
{
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);
  BBS_THREAD_UI_LOCK
  {
    int offset;
    info->res_num = res_num;
    if (info->rendering_done && info->last_read != NULL
	&& res_num < info->offsets_length
	&& (offset = info->offsets[res_num - 1]) > 0)
      {
	GtkWidget *scrolled_window
	  = paned_notebook_get_item_view(board_view, thread);
	if (scrolled_window != NULL)
	  {
	    BBSThreadView *thread_view
	      = BBS_THREAD_VIEW(gtk_bin_get_child(GTK_BIN(scrolled_window)));
	    GtkTextMark *mark
	      = bbs_thread_view_create_mark_at_offset(thread_view, offset);
	    save_initial_thread_view_position(thread_view, info, mark);
	  }
      }
  }
  BBS_THREAD_UI_UNLOCK
}


typedef struct _BBSThreadJobArgs
{
  OchushaApplication *application;
  OchushaBBSThread *thread;
  BBSThreadView *view;
  IconLabel *tab_label;
  OchushaAsyncBuffer *buffer;
  int start;
  int number_of_responses;
  gboolean refresh;
  gboolean auto_refresh;
} BBSThreadJobArgs;


static void
mark_deleted_cb(GtkTextBuffer *buffer, GtkTextMark *mark,
		BBSThreadGUIInfo *info)
{
  if (info->last_read == mark)
    info->last_read = NULL;
  if (info->next_mark == mark)
    info->next_mark = NULL;
}


GtkWidget *
open_bbs_thread(OchushaApplication *application, OchushaBBSThread *thread,
		IconLabel *tab_label)
{
  GtkWidget *thread_view;
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);

  if (info == NULL)
    {
      fprintf(stderr, "Out of memory.\n");
      return NULL;
    }

  BBS_THREAD_UI_LOCK
  {
    WorkerJob *job;
    BBSThreadJobArgs *job_args;
    OchushaAsyncBuffer *buffer;

    thread_view = bbs_thread_view_new(thread);

    job = G_NEW0(WorkerJob, 1);
    job_args = G_NEW0(BBSThreadJobArgs, 1);
    buffer = ochusha_bbs_thread_get_responses_source(thread,
				application->broker, NULL,
				OCHUSHA_NETWORK_BROKER_CACHE_TRY_UPDATE);

    if (buffer == NULL)
      {
#if DEBUG_NETWORK_MOST
	fprintf(stderr, "Couldn't access network!\n");
#endif
	G_FREE(job_args);
	G_FREE(job);
	gtk_object_sink(GTK_OBJECT(thread_view));

	BBS_THREAD_UI_UNLOCK;
	return NULL;
      }

    setup_for_tab_label_animation(buffer, tab_label);

    info->recent_view = thread_view;
    info->recent_tab_label = tab_label;
    info->response_source_buffer = buffer;
    info->rendering_done = FALSE;

    job_args->application = application;
    job_args->thread = thread;
    job_args->view = BBS_THREAD_VIEW(thread_view);
    job_args->buffer = buffer;
    job_args->start = 0;
    job_args->number_of_responses = -1;
    job_args->refresh = FALSE;
    job_args->auto_refresh = FALSE;
    job_args->tab_label = tab_label;

    job->canceled = FALSE;
    job->job = render_bbs_thread;
    job->args = job_args;

    g_signal_connect(G_OBJECT(thread_view), "link_mouse_over",
		     G_CALLBACK(link_mouse_over_cb), application);
    g_signal_connect(G_OBJECT(thread_view), "link_mouse_out",
		     G_CALLBACK(link_mouse_out_cb), application);
    g_signal_connect(G_OBJECT(thread_view), "link_mouse_press",
		     G_CALLBACK(link_mouse_press_cb), application);
    g_signal_connect(G_OBJECT(thread_view), "link_mouse_release",
		     G_CALLBACK(link_mouse_release_cb), application);
    g_signal_connect(G_OBJECT(thread_view), "copy_clipboard",
		     G_CALLBACK(copy_clipboard_cb), application);
    g_signal_connect(G_OBJECT(thread_view), "write_response",
		     G_CALLBACK(write_response_cb), application);
    g_signal_connect(G_OBJECT(thread_view), "interactive_search",
		     G_CALLBACK(interactive_search_cb), application);
    g_signal_connect(G_OBJECT(thread_view), "scroll_view",
		     G_CALLBACK(scroll_view_cb), application);

    g_signal_connect(G_OBJECT(job_args->view->text_buffer), "mark_deleted",
		     G_CALLBACK(mark_deleted_cb), info);

#if DEBUG_ASYNC_BUFFER_MOST
    fprintf(stderr, "open_bbs_thread: ref AsyncBuffer(%p)\n", buffer);
    fprintf(stderr, "* before ref: buffer->ref_count=%d\n",
	    G_OBJECT(buffer)->ref_count);
#endif
      /* öref_count䤹 */
    OCHU_OBJECT_REF(buffer);
    g_object_ref(tab_label);		/* бǧ */
    g_object_ref(thread_view);		/* бǧ */

    commit_job(job);
  }
  BBS_THREAD_UI_UNLOCK;

  return thread_view;
}


#if GTK_MINOR_VERSION <= 2
static void
link_popup_menu_callback(gpointer data, guint action, GtkWidget *widget)
{
  OchushaApplication *application = data;
  const char *url = gtk_item_factory_popup_data(link_popup_menu_item_factory);

  switch (action)
    {
    case LINK_POPUP_MENU_OPEN_LINK:
      ochusha_open_url(application, url, FALSE, FALSE);
      break;

    case LINK_POPUP_MENU_OPEN_LINK_IN_TAB:
      ochusha_open_url(application, url, TRUE, FALSE);
      break;
      
    case LINK_POPUP_MENU_OPEN_LINK_WITH_WEB_BROWSER:
      ochusha_open_url(application, url, FALSE, TRUE);
      break;

    case LINK_POPUP_MENU_COPY_LINK_LOCATION:
      ochusha_clipboard_set_text(application, url);
      break;

    case LINK_POPUP_MENU_SAVE_LINK_AS:
      ochusha_download_url(application, url);
      break;

    case LINK_POPUP_MENU_TENURE_LINK_IMAGE:
      ochusha_tenure_cache_image(application, url);
      break;

    case LINK_POPUP_MENU_A_BONE_LINK_IMAGE:
      ochusha_a_bone_image(application, url);
      break;

    case LINK_POPUP_MENU_A_BONE_LINK_SERVER:
      ochusha_a_bone_image_server(application, url);
      break;
    }
}


#else


static void
popup_menu_open_link_cb(GtkAction *action, OchushaApplication *application)
{
  const char *url = ochusha_ui_manager_get_popup_data(ui_manager);
  g_return_if_fail(url != NULL);

  ochusha_open_url(application, url, FALSE, FALSE);
}


static void
popup_menu_open_link_in_tab_cb(GtkAction *action,
			       OchushaApplication *application)
{
  const char *url = ochusha_ui_manager_get_popup_data(ui_manager);
  g_return_if_fail(url != NULL);

  ochusha_open_url(application, url, TRUE, FALSE);
}


static void
popup_menu_open_link_with_web_browser_cb(GtkAction *action,
					 OchushaApplication *application)
{
  const char *url = ochusha_ui_manager_get_popup_data(ui_manager);
  g_return_if_fail(url != NULL);

  ochusha_open_url(application, url, FALSE, TRUE);
}


static void
popup_menu_save_link_as_cb(GtkAction *action, OchushaApplication *application)
{
  const char *url = ochusha_ui_manager_get_popup_data(ui_manager);
  g_return_if_fail(url != NULL);

  ochusha_download_url(application, url);
}


static void
popup_menu_copy_link_location_cb(GtkAction *action,
				 OchushaApplication *application)
{
  const char *url = ochusha_ui_manager_get_popup_data(ui_manager);
  g_return_if_fail(url != NULL);

  ochusha_clipboard_set_text(application, url);
}


static void
popup_menu_tenure_link_image_cb(GtkAction *action,
				OchushaApplication *application)
{
  const char *url = ochusha_ui_manager_get_popup_data(ui_manager);
  g_return_if_fail(url != NULL);

  ochusha_tenure_cache_image(application, url);
}


static void
popup_menu_a_bone_link_image_cb(GtkAction *action,
				OchushaApplication *application)
{
  const char *url = ochusha_ui_manager_get_popup_data(ui_manager);
  g_return_if_fail(url != NULL);

  ochusha_a_bone_image(application, url);
}


static void
popup_menu_a_bone_link_server_cb(GtkAction *action,
				 OchushaApplication *application)
{
  const char *url = ochusha_ui_manager_get_popup_data(ui_manager);
  g_return_if_fail(url != NULL);

  ochusha_a_bone_image_server(application, url);
}
#endif


static void
link_popup_menu_destructed_cb(gpointer data)
{
  popup_menu_shown = FALSE;
  G_FREE(data);
}


typedef struct _ResponsePopupMenuData
{
  int res_num;
  BBSThreadView *view;
  OchushaBBSThread *thread;
} ResponsePopupMenuData;


#if GTK_MINOR_VERSION <= 2
static void
response_popup_menu_callback(gpointer data, guint action, GtkWidget *widget)
{
  ResponsePopupMenuData *menu_data
    = gtk_item_factory_popup_data(response_popup_menu_item_factory);

  switch (action)
    {
    case RESPONSE_POPUP_MENU_INSERT_BOOKMARK:
      insert_bookmark(menu_data->view, menu_data->thread,
		      menu_data->res_num, (OchushaApplication *)data);
      break;

    case RESPONSE_POPUP_MENU_WRITE_RESPONSE:
      write_response((OchushaApplication *)data, menu_data->res_num);
      break;
    }
}


#else


static void
popup_menu_insert_bookmark_cb(GtkAction *action,
			      OchushaApplication *application)
{
  ResponsePopupMenuData *menu_data
    = ochusha_ui_manager_get_popup_data(ui_manager);
  g_return_if_fail(menu_data != NULL);

  insert_bookmark(menu_data->view, menu_data->thread,
		  menu_data->res_num, application);
}


static void
popup_menu_write_response_cb(GtkAction *action,
			     OchushaApplication *application)
{
  ResponsePopupMenuData *menu_data
    = ochusha_ui_manager_get_popup_data(ui_manager);
  g_return_if_fail(menu_data != NULL);

  write_response(application, menu_data->res_num);
}
#endif


static void
response_popup_menu_destructed_cb(gpointer data)
{
  popup_menu_shown = FALSE;
  G_FREE(data);
}


static gboolean
link_mouse_over_cb(BBSThreadView *view, GdkEventMotion *event,
		   OchushaBBSThread *thread, const gchar *link,
		   OchushaApplication *application)
{
  GtkStatusbar *statusbar = application->statusbar;
  gchar *tmp_link;
  gchar tmp_buffer[PATH_MAX];

  if (message_id != 0)
    {
      gtk_statusbar_remove(statusbar, link_field_id, message_id);
      stack_depth--;
      message_id = 0;
    }

  if (strncmp(link, "<RES>", 5) == 0)
    return FALSE;

  g_strlcpy(tmp_buffer, link, PATH_MAX);

  tmp_link = strstr(tmp_buffer, "<FAKE>");
  if (application->enable_popup_response || application->enable_popup_image)
    {
      if (tmp_link != NULL)
	show_popup(application, thread,
		   tmp_link + 6, event->x_root, event->y_root, view);
      else
	{
	  int res_num = parse_int(tmp_buffer, strlen(tmp_buffer));
	  if (res_num > 0)
	    {
	      char *reslink;
	      if ((reslink = ochusha_bbs_thread_get_url_for_response(thread,
						res_num, res_num)) != NULL)
		{
		  show_popup(application, thread, reslink,
			     event->x_root, event->y_root, view);
		  G_FREE(reslink);
		}
	      else
		{
		  return FALSE;
		}
	    }
	  else
	    show_popup(application, thread,
		       tmp_buffer, event->x_root, event->y_root, view);
	}
    }

  if (tmp_link != NULL)
    *tmp_link = '\0';

  if (stack_depth == 0 && strchr(tmp_buffer, '\n') == NULL)
    {
      stack_depth++;
      message_id = gtk_statusbar_push(statusbar, link_field_id, tmp_buffer);
    }

  return FALSE;
}


static void
popup_context_free(PopupContext *popup_context)
{
#if DEBUG_POPUP
  fprintf(stderr, "popup_context_free(%p)\n", popup_context);
#endif

  if (popup_context->renderer_context.handler.converter != 0)
    {
      iconv_close(popup_context->renderer_context.handler.converter);
      popup_context->renderer_context.handler.converter = 0;
    }

  if (popup_context->renderer_context.link_text_buffer != NULL)
    {
      G_FREE(popup_context->renderer_context.link_text_buffer);
      popup_context->renderer_context.link_text_buffer = NULL;
    }

  /* "\357\274\214\343\200\201"  "" UTF-8ɽ */
  regfree(&popup_context->renderer_context.comma_separated_link_pattern);
  regfree(&popup_context->renderer_context.informal_link_pattern);

  if (popup_context->renderer_context.a_bone_by_name != NULL)
    {
      oniguruma_regex_free(popup_context->renderer_context.a_bone_by_name);
      popup_context->renderer_context.a_bone_by_name = NULL;
    }

  if (popup_context->renderer_context.a_bone_by_id != NULL)
    {
      oniguruma_regex_free(popup_context->renderer_context.a_bone_by_id);
      popup_context->renderer_context.a_bone_by_id = NULL;
    }

  if (popup_context->renderer_context.a_bone_by_content != NULL)
    {
      oniguruma_regex_free(popup_context->renderer_context.a_bone_by_content);
      popup_context->renderer_context.a_bone_by_content = NULL;
    }

  if (popup_context->url != NULL)
    {
      G_FREE(popup_context->url);
      popup_context->url = NULL;
    }

  if (popup_context->delay_id != 0)
    {
      g_source_remove(popup_context->delay_id);
      popup_context->delay_id = 0;
    }

  if (popup_context->close_delay_id != 0)
    {
      g_source_remove(popup_context->close_delay_id);
      popup_context->close_delay_id = 0;
    }

  if (popup_context->popup_window != NULL)
    {
      gtk_widget_destroy(popup_context->popup_window);
      popup_context->popup_window = NULL;
    }

  G_FREE(popup_context);
}


/*
 * MEMO: gdk_threads_enter()ĶƤӽФȡ
 */
static PopupContext *
ensure_popup_context(BBSThreadView *parent, OchushaBBSThread *thread,
		     OchushaApplication *application)
{
  RendererContext *renderer_context;
  PopupContext *popup_context = g_object_get_qdata(G_OBJECT(parent),
						   popup_context_id);

  if (popup_context != NULL)
    return popup_context;

  popup_context = G_NEW0(PopupContext, 1);

  renderer_context = &popup_context->renderer_context;
  renderer_context->info = ensure_bbs_thread_info(thread);

  /* renderer_context->handler.converter = 0; */
  renderer_context->handler.startElement = start_element_cb;
  renderer_context->handler.endElement = end_element_cb;
  /* renderer_context->handler.entityReference = NULL; */
  renderer_context->handler.characters = characters_cb;
  /* renderer_context->handler.in_anchor = FALSE; */

  /* renderer_context->view = NULL; */
  /* renderer_context->thread = NULL; */
  renderer_context->application = application;
  renderer_context->last_read_number = -1;

  /* renderer_context->link = NULL; */
  renderer_context->link_text_buffer = G_NEW0(gchar, DEFAULT_LINK_BUFFER_SIZE);
  renderer_context->link_text_buffer_size = DEFAULT_LINK_BUFFER_SIZE;
  /* renderer_context->link_text_length = 0; */

  popup_context->parent_context = g_object_get_qdata(G_OBJECT(parent),
						     popup_parent_context_id);
  initialize_popup_context(popup_context, application);
  g_object_set_qdata_full(G_OBJECT(parent), popup_context_id, popup_context,
			  (GDestroyNotify)popup_context_free);

  /* "\357\274\214\343\200\201\357\274\235"  "" UTF-8ɽ */
  /* "\357\274\220"""UTF-8ɽ */
  /* "\357\274\221"""UTF-8ɽ */
  /* "\357\274\231"""UTF-8ɽ */
  if (regcomp(&renderer_context->comma_separated_link_pattern,
	      "^[,=\357\274\214\343\200\201\357\274\235][ ]*([1-9\357\274\221-\357\274\231][0-9\357\274\220-\357\274\231]*)(-[1-9\357\274\221-\357\274\231][0-9\357\274\220-\357\274\231]*)?",
	      REG_EXTENDED))
    {
      fprintf(stderr, "invalid regular expression\n");
    }

  /* "\357\274\236\342\211\253\343\200\213"  "" UTF-8ɽ */
  if (regcomp(&renderer_context->informal_link_pattern,
	      "[>\357\274\236\342\211\253\343\200\213]+[ ]*([1-9\357\274\221-\357\274\231][0-9\357\274\220-\357\274\231]*)(-[1-9\357\274\221-\357\274\231][0-9\357\274\220-\357\274\231]*)?",
	      REG_EXTENDED))
    {
      fprintf(stderr, "invalid regular expression\n");
    }

  if (application->disable_popup_a_bone)
    {
      gboolean a_bone_by_name;
      gchar *a_bone_by_name_pattern;
      gboolean a_bone_by_id;
      gchar *a_bone_by_id_pattern;
      gboolean a_bone_by_content;
      gchar *a_bone_by_content_pattern;

      BBSThreadGUIInfo *thread_info = ensure_bbs_thread_info(thread);

      if (thread_info->thread_local_a_bone)
	{
	  a_bone_by_name = thread_info->a_bone_by_name;
	  a_bone_by_name_pattern = thread_info->a_bone_by_name_pattern;
	  a_bone_by_id = thread_info->a_bone_by_id;
	  a_bone_by_id_pattern = thread_info->a_bone_by_id_pattern;
	  a_bone_by_content = thread_info->a_bone_by_content;
	  a_bone_by_content_pattern = thread_info->a_bone_by_content_pattern;
	}
      else
	{
	  BulletinBoardGUIInfo *board_info
	    = ensure_bulletin_board_info(ochusha_bbs_thread_get_board(thread),
					 application);
	  if (board_info->board_local_a_bone)
	    {
	      a_bone_by_name = board_info->a_bone_by_name;
	      a_bone_by_name_pattern = board_info->a_bone_by_name_pattern;
	      a_bone_by_id = board_info->a_bone_by_id;
	      a_bone_by_id_pattern = board_info->a_bone_by_id_pattern;
	      a_bone_by_content = board_info->a_bone_by_content;
	      a_bone_by_content_pattern
		= board_info->a_bone_by_content_pattern;
	    }
	  else
	    {
	      a_bone_by_name = application->a_bone_by_name;
	      a_bone_by_name_pattern = application->a_bone_by_name_pattern;
	      a_bone_by_id = application->a_bone_by_id;
	      a_bone_by_id_pattern = application->a_bone_by_id_pattern;
	      a_bone_by_content = application->a_bone_by_content;
	      a_bone_by_content_pattern
		= application->a_bone_by_content_pattern;
	    }
	}

      renderer_context->transparent_a_bone = TRUE;
      if (a_bone_by_name || a_bone_by_id || a_bone_by_content)
	{
	  const char *encoding
	    = ochusha_bbs_thread_get_response_character_encoding(thread);
	  iconv_t converter = iconv_open(encoding, "UTF-8");
	  if (converter != (iconv_t)-1)
	    {
	      if (a_bone_by_name
		  && a_bone_by_name_pattern != NULL
		  && *a_bone_by_name_pattern != '\0')
		{
		  char *pattern = convert_string(converter, NULL,
						 a_bone_by_name_pattern, -1);
		  if (pattern != NULL)
		    {
		      if (*pattern != '\0')
			renderer_context->a_bone_by_name
			  = oniguruma_regex_new(pattern, encoding);
		      G_FREE(pattern);
		    }
		}
	      if (a_bone_by_id
		  && a_bone_by_id_pattern != NULL
		  && *a_bone_by_id_pattern != '\0')
		{
		  char *pattern = convert_string(converter, NULL,
						 a_bone_by_id_pattern, -1);
		  if (pattern != NULL)
		    {
		      if (*pattern != '\0')
			renderer_context->a_bone_by_id
			  = oniguruma_regex_new(pattern, encoding);
		      G_FREE(pattern);
		    }
		}
	      if (a_bone_by_content
		  && a_bone_by_content_pattern != NULL
		  && *a_bone_by_content_pattern != '\0')
		{
		  char *pattern = convert_string(converter, NULL,
						 a_bone_by_content_pattern,
						 -1);
		  if (pattern != NULL)
		    {
		      if (*pattern != '\0')
			renderer_context->a_bone_by_content
			  = oniguruma_regex_new(pattern, encoding);
		      G_FREE(pattern);
		    }
		}
	      iconv_close(converter);
	    }
	}
    }

  return popup_context;
}


static gboolean
link_mouse_out_cb(BBSThreadView *view, GdkEventAny *event,
		  OchushaBBSThread *thread, OchushaApplication *application)
{
  PopupContext *popup_context;
  GtkStatusbar *statusbar = application->statusbar;

  popup_context = ensure_popup_context(view, thread, application);

  if (message_id != 0)
    {
      gtk_statusbar_remove(statusbar, link_field_id, message_id);
      message_id = 0;
      stack_depth--;
    }

#if DEBUG_POPUP
  fprintf(stderr, "link_mouse_out_cb: hide_popup(%p)\n", popup_context);
#endif
  hide_popup(popup_context);

  return FALSE;
}


static gboolean
link_mouse_press_cb(BBSThreadView *view, GdkEventButton *event,
		    OchushaBBSThread *thread, const gchar *link,
		    OchushaApplication *application)
{
  gchar *tmp_link;

  if (link == NULL)
    return FALSE;

  if (strncmp(link, "<RES>", 5) == 0)
    {
      int res_num;
      if (!application->stick_bookmark)
	{
#if GTK_MINOR_VERSION <= 2
	  GtkWidget *tmp_widget = gtk_item_factory_get_widget_by_action(
					response_popup_menu_item_factory,
					RESPONSE_POPUP_MENU_INSERT_BOOKMARK);
	  gtk_widget_set_sensitive(tmp_widget, FALSE);
#else
	  GtkAction *action = gtk_action_group_get_action(action_group,
							  "InsertBookmark");
	  if (action != NULL)
	    g_object_set(G_OBJECT(action), "sensitive", FALSE, NULL);
#endif
	}

      res_num = 0;
      if (event->button == 3 && sscanf(link + 5, "%d", &res_num) == 1)
	{
	  ResponsePopupMenuData *popup_data
	    = G_NEW0(ResponsePopupMenuData, 1);
	  popup_data->res_num = res_num;
	  popup_data->view = view;
	  popup_data->thread = thread;

	  popup_menu_shown = TRUE;
#if GTK_MINOR_VERSION <= 2
	  gtk_item_factory_popup_with_data(response_popup_menu_item_factory,
					   popup_data,
					   response_popup_menu_destructed_cb,
					   event->x_root, event->y_root,
					   event->button,
					   gtk_get_current_event_time());
#else
	  ochusha_ui_manager_popup_with_data(ui_manager, "/ResponsePopup",
					     popup_data,
					     response_popup_menu_destructed_cb,
					     event->x_root, event->y_root,
					     event->button,
					     gtk_get_current_event_time());
#endif
	}
      return TRUE;
    }

  tmp_link = strstr(link, "<FAKE>");
  if (tmp_link != NULL)
    link = tmp_link + 6;

  if (event->button == 3)
    {
      gboolean sensitive = (g_str_has_suffix(link, ".jpg")
			    || g_str_has_suffix(link, ".png")
			    || g_str_has_suffix(link, ".gif")
			    || g_str_has_suffix(link, ".jpeg")
			    || g_str_has_suffix(link, ".JPG")
			    || g_str_has_suffix(link, ".PNG")
			    || g_str_has_suffix(link, ".GIF")
			    || g_str_has_suffix(link, ".JPEG"));
#if GTK_MINOR_VERSION <= 2
      GtkWidget *tmp_widget = gtk_item_factory_get_widget_by_action(
					link_popup_menu_item_factory,
					LINK_POPUP_MENU_TENURE_LINK_IMAGE);
      if (tmp_widget != NULL)
	gtk_widget_set_sensitive(tmp_widget, sensitive);

      tmp_widget = gtk_item_factory_get_widget_by_action(
					link_popup_menu_item_factory,
					LINK_POPUP_MENU_A_BONE_LINK_IMAGE);
      if (tmp_widget != NULL)
	gtk_widget_set_sensitive(tmp_widget, sensitive);

      tmp_widget = gtk_item_factory_get_widget_by_action(
					link_popup_menu_item_factory,
					LINK_POPUP_MENU_A_BONE_LINK_SERVER);
      if (tmp_widget != NULL)
	gtk_widget_set_sensitive(tmp_widget, sensitive);
      
#else
      GtkAction *action = gtk_action_group_get_action(action_group,
						      "TenureLinkImage");
      if (action != NULL)
	  g_object_set(G_OBJECT(action), "sensitive", sensitive, NULL);

      action = gtk_action_group_get_action(action_group,
						      "ABoneLinkImage");
      if (action != NULL)
	  g_object_set(G_OBJECT(action), "sensitive", sensitive, NULL);

      action = gtk_action_group_get_action(action_group, "ABoneLinkServer");
      if (action != NULL)
	  g_object_set(G_OBJECT(action), "sensitive", sensitive, NULL);
#endif

      popup_menu_shown = TRUE;

#if GTK_MINOR_VERSION <= 2
      gtk_item_factory_popup_with_data(link_popup_menu_item_factory,
				       G_STRDUP(link),
				       link_popup_menu_destructed_cb,
				       event->x_root, event->y_root,
				       event->button,
				       gtk_get_current_event_time());
#else
      ochusha_ui_manager_popup_with_data(ui_manager, "/ThreadLinkPopup",
					 G_STRDUP(link),
					 link_popup_menu_destructed_cb,
					 event->x_root, event->y_root,
					 event->button,
					 gtk_get_current_event_time());
#endif
    }
  else
    {
      unsigned int from;
      unsigned int to;

      if (ochusha_bbs_thread_check_url(thread, link, &from, &to))
	{
	  /* URL */
	  if (from == 0 && to > 0)
	    from = to;
	  if (from > 0)
	    {
	      BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);
	      GtkWidget *root_view;
	      PanedNotebook *paned_notebook
		= PANED_NOTEBOOK(application->contents_window);
	      paned_notebook
		= PANED_NOTEBOOK(paned_notebook_get_item_view(paned_notebook,
							      thread->board));
	      if (paned_notebook == NULL)
		return TRUE;

	      root_view = paned_notebook_get_item_view(paned_notebook, thread);
	      if (root_view == NULL)
		return TRUE;

	      root_view = gtk_bin_get_child(GTK_BIN(root_view));

	      if (info->offsets != NULL && info->offsets_length > from
		  && info->offsets[from] > 0)
		{
		  /*  */
		  save_thread_view_position(root_view, thread);
		  bbs_thread_view_scroll_to_offset(BBS_THREAD_VIEW(root_view),
						   info->offsets[from]);
		}
	    }
	}
      else
	ochusha_open_url(application, link,
			 application->default_open_in_tab
			 ? event->button == 1 : event->button == 2, FALSE);
    }

  return TRUE;
}


static gboolean
link_mouse_release_cb(BBSThreadView *view, GdkEventButton *event,
		      OchushaBBSThread *thread, const gchar *link,
		      OchushaApplication *application)
{
  return FALSE;
}


static void
copy_clipboard_cb(BBSThreadView *view, OchushaApplication *application)
{
  gchar *text = bbs_thread_view_get_selected_text(view);
  if (text != NULL)
    {
      ochusha_clipboard_set_text(application, text);
      g_free(text);
    }
}


static void
scroll_view_cb(GtkWidget *view, GtkScrollType scroll,
	       OchushaApplication *application)
{
  g_signal_emit_by_name(G_OBJECT(view->parent), "scroll_child", scroll, FALSE);
}


static void
write_response_cb(BBSThreadView *view, OchushaApplication *application)
{
  write_response(application, 0);
}


static void
interactive_search_cb(BBSThreadView *view, BBSThreadViewSearchAction action,
		      OchushaApplication *application)
{
  start_thread_search(application);
}


static gboolean
show_mailto(OchushaApplication *application, OchushaBBSThread *thread)
{
  BBSThreadGUIInfo *thread_info = ensure_bbs_thread_info(thread);
  BulletinBoardGUIInfo *board_info;
  if (thread_info->show_mailto_mode != THREAD_VIEW_MAILTO_MODE_DEFAULT)
    return (thread_info->show_mailto_mode
	    == THREAD_VIEW_MAILTO_MODE_SHOW);

  board_info = ensure_bulletin_board_info(ochusha_bbs_thread_get_board(thread),
					  application);
  if (board_info->show_mailto_mode != THREAD_VIEW_MAILTO_MODE_DEFAULT)
    return (board_info->show_mailto_mode
	    == THREAD_VIEW_MAILTO_MODE_SHOW);

  if (application->show_mailto_mode != THREAD_VIEW_MAILTO_MODE_DEFAULT)
    return (application->show_mailto_mode
	    == THREAD_VIEW_MAILTO_MODE_SHOW);

  return FALSE;
}


static void
update_threadlist_entry_real(OchushaApplication *application,
			     OchushaBBSThread *thread)
{
  PanedNotebook *board_view;
  GtkWidget *scrolled_window;
  ThreadlistView *threadlist_view;
  PanedNotebook *boardlist = (PanedNotebook *)application->contents_window;

  g_return_if_fail(boardlist != NULL && thread != NULL);

  board_view = (PanedNotebook *)paned_notebook_get_item_view(boardlist,
							     thread->board);

  if (board_view == NULL)
    return;	/* ĤɽƤʤ⤢ */

  scrolled_window = paned_notebook_get_selector(board_view);

  if (scrolled_window == NULL)
    {
      return;	/* ե륿ߤǥɽƤʤ
		 * ⤢롣
		 */
    }

  threadlist_view
    = THREADLIST_VIEW(gtk_bin_get_child(GTK_BIN(scrolled_window)));

  update_threadlist_entry_style(application, threadlist_view, thread);
}


static void
update_threadlist_entry(OchushaApplication *application,
			OchushaBBSThread *thread)
{
  PanedNotebook *boardlist = (PanedNotebook *)application->contents_window;

  g_return_if_fail(boardlist != NULL && thread != NULL);

  gdk_threads_enter();
  update_threadlist_entry_real(application, thread);
  gdk_threads_leave();
}


typedef struct _ScrollToMarkArgs
{
  BBSThreadView *view;
  BBSThreadGUIInfo *info;
  GtkTextMark *mark;
} ScrollToMarkArgs;


static gboolean
scroll_to_mark_on_idle(ScrollToMarkArgs *args)
{
  gdk_threads_enter();
  if (args->info->next_mark == args->mark
      && (BBSThreadView *)args->info->recent_view == args->view)
    {
      bbs_thread_view_scroll_to_mark(args->view, args->info->next_mark);
      args->info->next_mark = NULL;
      args->info->scroller_id = 0;
    }
  gdk_threads_leave();
  return FALSE;
}


static void
save_initial_thread_view_position(BBSThreadView *view, BBSThreadGUIInfo *info,
				  GtkTextMark *mark)
{
  ScrollToMarkArgs *args;
  g_return_if_fail(IS_BBS_THREAD_VIEW(view) && info != NULL && mark != NULL);

  args = g_new0(ScrollToMarkArgs, 1);
  info->next_mark = mark;
  args->view = view;
  args->info = info;
  args->mark = mark;
  info->scroller_id = g_idle_add_full(G_PRIORITY_HIGH_IDLE + 13,
				      (GSourceFunc)scroll_to_mark_on_idle,
				      args, (GDestroyNotify)g_free);
}


static void
bookmark_clicked_cb(GtkWidget *widget, BBSThreadView *view)
{
  OchushaApplication *application = g_object_get_data(G_OBJECT(widget),
						      "application");
  OchushaBBSThread *thread = g_object_get_data(G_OBJECT(widget),
					       "thread");
  BBSThreadGUIInfo *info;

  g_return_if_fail(application != NULL && OCHUSHA_IS_BBS_THREAD(thread));

  info = ensure_bbs_thread_info(thread);

  if (info->bookmark_response_number
      == ochusha_bbs_thread_get_number_of_responses_read(thread))
    return;

  insert_bookmark(view, thread,
		  ochusha_bbs_thread_get_number_of_responses_read(thread),
		  application);
}


static void
bookmark_destroy_cb(GtkWidget *widget, BBSThreadGUIInfo *info)
{
  if (info->bookmark == widget)
    info->bookmark = NULL;
#if 0
  fprintf(stderr, "bookmark(%p) destroyed.\n", widget);
#endif
}


static void
insert_bookmark(BBSThreadView *view, OchushaBBSThread *thread, int res_num,
		OchushaApplication *application)
{
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);
  guint *offsets = info->offsets;
  guint offsets_length = info->offsets_length;
  GtkWidget *bookmark;
  int i;
  gchar *bookmark_text;

  if (res_num <= 0 || res_num >= offsets_length)
    return;

  if (info->bookmark != NULL && info->bookmark_response_number == res_num)
    return;

  if (application->hide_bookmark)
    bookmark = NULL;
  else
    {
      bookmark_text = application->bookmark_text;
  
      if (bookmark_text == NULL || *bookmark_text == '\0')
	{
	  if (application->bookmark_text != NULL)
	    G_FREE(application->bookmark_text);
	  bookmark_text
	    = G_STRDUP(_("==== My bookmark: I have read above. ===="));
	  application->bookmark_text = bookmark_text;
	}

      bookmark = gtk_button_new_with_label(bookmark_text);
      GTK_WIDGET_UNSET_FLAGS(bookmark, GTK_CAN_FOCUS);

      gtk_widget_set_name(bookmark, "mybookmark");

      g_object_set_data(G_OBJECT(bookmark), "application", application);
      g_object_set_data(G_OBJECT(bookmark), "thread", thread);

      g_signal_connect(G_OBJECT(bookmark), "clicked",
		       G_CALLBACK(bookmark_clicked_cb), view);
      g_signal_connect(G_OBJECT(bookmark), "destroy",
		       G_CALLBACK(bookmark_destroy_cb), info);
    }

  if (info->bookmark != NULL)
    {
      bbs_thread_view_remove_widget(view, info->bookmark);
      info->bookmark = NULL;

      for (i = info->bookmark_response_number + 1; i < offsets_length; i++)
	{
	  if (offsets[i] == 0)
	    break;
	  offsets[i]--;
	}
    }

  if (info->last_read != NULL)
    bbs_thread_view_delete_mark(view, info->last_read);

  if (offsets[res_num + 1] == 0)
    {
      if (offsets[res_num] != 0)
	info->last_read
	  = bbs_thread_view_create_mark_at_offset(view, offsets[res_num]);
      else
	info->last_read = bbs_thread_view_create_mark(view);
      if (!application->hide_bookmark)
	bbs_thread_view_append_widget(view, bookmark, 30, 20);
    }
  else
    {
      info->last_read
	= bbs_thread_view_create_mark_at_offset(view, offsets[res_num]);
      if (!application->hide_bookmark)
	{
	  bbs_thread_view_insert_widget_at_offset(view, bookmark,
						  offsets[res_num + 1]);
	  for (i = res_num + 1; i < offsets_length; i++)
	    {
	      if (offsets[i] == 0)
		break;
	      offsets[i]++;
	    }
	}
    }

  info->bookmark_response_number = res_num;
  info->bookmark = bookmark;

  update_threadlist_entry_real(application, thread);
}


#if 0
typedef struct _ThreadRenderingSyncObject
{
  pthread_mutex_t lock;
  pthread_cond_t cond;
} ThreadRenderingSyncObject;


static gboolean
start_rendering(gpointer data)
{
  ThreadRenderingSyncObject *sync_object = (ThreadRenderingSyncObject *)data;
  if (pthread_mutex_lock(&sync_object->lock) != 0)
    {
      fprintf(stderr, "Couldn't lock a mutex.\n");
      abort();
    }
  if (pthread_cond_signal(&sync_object->cond) != 0)
    {
      fprintf(stderr, "Couldn't signal a condition.\n");
      abort();
    }
  if (pthread_mutex_unlock(&sync_object->lock) != 0)
    {
      fprintf(stderr, "Couldn't unlock a mutex.\n");
      abort();
    }
  return FALSE;
}
#endif


static void
render_bbs_thread(WorkerThread *employee, gpointer args)
{
  BBSThreadJobArgs *job_args = (BBSThreadJobArgs *)args;
  OchushaApplication *application = job_args->application;
  OchushaBBSThread *thread = job_args->thread;
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);
  OchushaAsyncBuffer *buffer = job_args->buffer;
  gchar default_text_buffer[DEFAULT_BUFFER_SIZE];
  iconv_helper *helper = ochusha_bbs_thread_get_response_iconv_helper(thread);
  gboolean a_bone_by_name;
  gchar *a_bone_by_name_pattern;
  gboolean a_bone_by_id;
  gchar *a_bone_by_id_pattern;
  gboolean a_bone_by_content;
  gchar *a_bone_by_content_pattern;
  RendererContext context =
    {
      {
	helper != NULL
	? iconv_open("UTF-8",
		     ochusha_bbs_thread_get_response_character_encoding(thread))
	: iconv_open("UTF-8//IGNORE",
		     ochusha_bbs_thread_get_response_character_encoding(thread)),
	helper,
	start_element_cb,
	end_element_cb,
	NULL,
	characters_cb,
	FALSE
      },					/* handler */
      job_args->view,				/* view */
      thread,					/* thread */
      info,					/* info */
      application,				/* application */
      ochusha_bbs_thread_get_number_of_responses_read(thread),
      						/* last_read_number */
      job_args->start,				/* last_rendered_response */
      job_args->start > 0,			/* title_shown */

      NULL,					/* link */
      default_text_buffer,			/* link_text_buffer */
      DEFAULT_BUFFER_SIZE,			/* link_text_buffer_size */
      0,					/* link_text_length */

      0,					/* number_of_gt_characters */
      FALSE,					/* last_text_is_link */
      show_mailto(application, thread),		/* show_mailto_literally */

      tag_table,				/* tag_table */

      bbs_thread_view_get_tag_by_name(job_args->view, "thread_title"),
      bbs_thread_view_get_tag_by_name(job_args->view, "response_header"),
      bbs_thread_view_get_tag_by_name(job_args->view, "response_name"),
      bbs_thread_view_get_tag_by_name(job_args->view, "response_paragraph"),
      bbs_thread_view_get_tag_by_name(job_args->view, "bold"),

      FALSE,					/* transparent_a_bone */
      NULL,					/* a_bone_by_name regex_obj */
      NULL,					/* a_bone_by_id */
      NULL,					/* a_bone_by_content */
      						/* regex_t */
      						/* regex_t */
  };
#if 0
  ThreadRenderingSyncObject sync_object;
#endif


  /* "\357\274\214\343\200\201\357\274\235"  "" UTF-8ɽ */
  /* "\357\274\220"""UTF-8ɽ */
  /* "\357\274\221"""UTF-8ɽ */
  /* "\357\274\231"""UTF-8ɽ */
  if (regcomp(&context.comma_separated_link_pattern,
	      "^[,=\357\274\214\343\200\201\357\274\235][ ]*([1-9\357\274\221-\357\274\231][0-9\357\274\220-\357\274\231]*)(-[1-9\357\274\221-\357\274\231][0-9\357\274\220-\357\274\231]*)?",
	      REG_EXTENDED))
    {
      fprintf(stderr, "invalid regular expression\n");
    }

  /* "\357\274\236\342\211\253\343\200\213"  "" UTF-8ɽ */
  if (regcomp(&context.informal_link_pattern,
	      "[>\357\274\236\342\211\253\343\200\213]+[ ]*([1-9\357\274\221-\357\274\231][0-9\357\274\220-\357\274\231]*)(-[1-9\357\274\221-\357\274\231][0-9\357\274\220-\357\274\231]*)?",
	      REG_EXTENDED))
    {
      fprintf(stderr, "invalid regular expression\n");
    }

  if (info->thread_local_a_bone)
    {
      a_bone_by_name = info->a_bone_by_name;
      a_bone_by_name_pattern = info->a_bone_by_name_pattern;
      a_bone_by_id = info->a_bone_by_id;
      a_bone_by_id_pattern = info->a_bone_by_id_pattern;
      a_bone_by_content = info->a_bone_by_content;
      a_bone_by_content_pattern = info->a_bone_by_content_pattern;
    }
  else
    {
      BulletinBoardGUIInfo *board_info
	= ensure_bulletin_board_info(ochusha_bbs_thread_get_board(thread),
				     application);
      if (board_info->board_local_a_bone)
	{
	  a_bone_by_name = board_info->a_bone_by_name;
	  a_bone_by_name_pattern = board_info->a_bone_by_name_pattern;
	  a_bone_by_id = board_info->a_bone_by_id;
	  a_bone_by_id_pattern = board_info->a_bone_by_id_pattern;
	  a_bone_by_content = board_info->a_bone_by_content;
	  a_bone_by_content_pattern = board_info->a_bone_by_content_pattern;
	}
      else
	{
	  a_bone_by_name = application->a_bone_by_name;
	  a_bone_by_name_pattern = application->a_bone_by_name_pattern;
	  a_bone_by_id = application->a_bone_by_id;
	  a_bone_by_id_pattern = application->a_bone_by_id_pattern;
	  a_bone_by_content = application->a_bone_by_content;
	  a_bone_by_content_pattern = application->a_bone_by_content_pattern;
	}
    }

  context.transparent_a_bone = application->enable_transparent_a_bone;
  if (a_bone_by_name || a_bone_by_id || a_bone_by_content)
    {
      const char *encoding
	= ochusha_bbs_thread_get_response_character_encoding(thread);
      iconv_t converter = iconv_open(encoding, "UTF-8");
      if (converter != (iconv_t)-1)
	{
	  if (a_bone_by_name
	      && a_bone_by_name_pattern != NULL
	      && *a_bone_by_name_pattern != '\0')
	    {
	      char *pattern = convert_string(converter, NULL,
					     a_bone_by_name_pattern, -1);
	      if (pattern != NULL)
		{
		  if (*pattern != '\0')
		    context.a_bone_by_name = oniguruma_regex_new(pattern,
								 encoding);
		  G_FREE(pattern);
		}
	    }
	  if (a_bone_by_id
	      && a_bone_by_id_pattern != NULL
	      && *a_bone_by_id_pattern != '\0')
	    {
	      char *pattern = convert_string(converter, NULL,
					     a_bone_by_id_pattern, -1);
	      if (pattern != NULL)
		{
		  if (*pattern != '\0')
		    context.a_bone_by_id = oniguruma_regex_new(pattern,
							       encoding);
		  G_FREE(pattern);
		}
	    }
	  if (a_bone_by_content
	      && a_bone_by_content_pattern != NULL
	      && *a_bone_by_content_pattern != '\0')
	    {
	      char *pattern
		= convert_string(converter, NULL,
				 a_bone_by_content_pattern, -1);
	      if (pattern != NULL)
		{
		  if (*pattern != '\0')
		    context.a_bone_by_content = oniguruma_regex_new(pattern,
								    encoding);
		  G_FREE(pattern);
		}
	    }

	  iconv_close(converter);
	}
    }

  if (info->offsets == NULL)
    {
      info->offsets = G_MALLOC0(sizeof(guint) * MAX_RESPONSE);
      info->offsets_length = MAX_RESPONSE;
    }

  if (context.handler.converter == (iconv_t)-1)
    {
      fprintf(stderr, "Out of memory\n");
      BBS_THREAD_UI_LOCK
      {
	info->rendering_done = TRUE;
	info->res_num = 0;
      }
      BBS_THREAD_UI_UNLOCK;
      return;
    }

#if DEBUG_GUI_MOST
  {
    gchar *title = convert_string(utf8_to_native, NULL, thread->title, -1);
    fprintf(stderr, "render_bbs_thread: %s\n", title);
    G_FREE(title);
  }
#endif

  if (job_args->start > 0 && !job_args->refresh)
    {
#if DEBUG_GUI
      fprintf(stderr, "XXX: Should this show thread title?");
#endif
    }

  if (job_args->refresh)
    {
      snprintf(default_text_buffer, DEFAULT_BUFFER_SIZE,
	       _("Updating thread %s@%s board"),
	       thread->title, ochusha_bbs_thread_get_board(thread)->name);

      gdk_threads_enter();

      if (!application->stick_bookmark && !job_args->auto_refresh)
	{
	  insert_bookmark(job_args->view, thread, job_args->start,
			  application);

	  save_initial_thread_view_position(job_args->view, info,
					    info->last_read);
	}

      if (info->message_id != 0)
	gtk_statusbar_remove(application->statusbar, thread_netstat_id,
			     info->message_id);
      info->message_id
	= gtk_statusbar_push(application->statusbar, thread_netstat_id,
			     default_text_buffer);
      gdk_threads_leave();
    }
  else
    {
      snprintf(default_text_buffer, DEFAULT_BUFFER_SIZE,
	       _("Getting thread %s@%s board"),
	       thread->title, ochusha_bbs_thread_get_board(thread)->name);

      gdk_threads_enter();

      if (info->message_id != 0)
	gtk_statusbar_remove(application->statusbar, thread_netstat_id,
			     info->message_id);
      info->message_id
	= gtk_statusbar_push(application->statusbar, thread_netstat_id,
			     default_text_buffer);

      gdk_threads_leave();
    }

#if 0
  if (pthread_mutex_init(&sync_object.lock, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a mutex.\n");
      abort();
    }

  if (pthread_cond_init(&sync_object.cond, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a condition variable.\n");
      abort();
    }

  if (pthread_mutex_lock(&sync_object.lock) != 0)
    {
      fprintf(stderr, "Couldn't lock a mutex.\n");
      abort();
    }

  /* idleؿϿ */
  g_idle_add_full(GDK_PRIORITY_REDRAW + 10,
		  start_rendering, &sync_object, NULL);

  if (pthread_cond_wait(&sync_object.cond, &sync_object.lock) != 0)
    {
      fprintf(stderr, "Couldn't wait a condition.\n");
      abort();
    }

  if (pthread_mutex_unlock(&sync_object.lock) != 0)
    {
      fprintf(stderr, "Couldn't unlock a mutex.\n");
      abort();
    }

  if (pthread_mutex_destroy(&sync_object.lock) != 0)
    {
      fprintf(stderr, "Couldn't destroy a mutex.\n");
      abort();
    }

  if (pthread_cond_destroy(&sync_object.cond) != 0)
    {
      fprintf(stderr, "Couldn't destroy a condition.\n");
      abort();
    }
#endif

  ochusha_bbs_thread_parse_responses(thread, buffer,
				     job_args->start,
				     job_args->number_of_responses,
				     FALSE,
				     start_thread_rendering,
				     render_response,
				     render_broken_response,
				     end_thread_rendering,
				     (StartParsingCallback *)gdk_threads_enter,
				     (BeforeWaitCallback *)gdk_threads_leave,
				     (AfterWaitCallback *)gdk_threads_enter,
				     (EndParsingCallback *)gdk_threads_leave,
				     &context);

  update_threadlist_entry(job_args->application, thread);

  iconv_close(context.handler.converter);

  OCHU_OBJECT_UNREF(buffer);

  gdk_threads_enter();

  if (job_args->auto_refresh)
    {
      GtkTextMark *mark = bbs_thread_view_create_mark(job_args->view);
      save_initial_thread_view_position(job_args->view, info, mark);
    }

  if (job_args->tab_label != NULL)
    g_object_unref(job_args->tab_label);	/* бǧ */
  g_object_unref(job_args->view);		/* бǧ */

  if (info->message_id != 0)
    {
      gtk_statusbar_remove(application->statusbar, thread_netstat_id,
			   info->message_id);
      info->message_id = 0;
    }

  if (ochusha_bbs_thread_get_number_of_responses_read(thread) > 0
      && ochusha_bbs_thread_get_board(thread)->bbs_type != OCHUSHA_BBS_TYPE_2CH_HEADLINE)
    bookmark_thread(application->all_threads, thread, application);

  gdk_threads_leave();

  BBS_THREAD_UI_LOCK
  {
    info->rendering_done = TRUE;
  }
  BBS_THREAD_UI_UNLOCK;

  regfree(&context.comma_separated_link_pattern);
  regfree(&context.informal_link_pattern);

  if (context.a_bone_by_name != NULL)
    {
      oniguruma_regex_free(context.a_bone_by_name);
      context.a_bone_by_name = NULL;
    }

  if (context.a_bone_by_id != NULL)
    {
      oniguruma_regex_free(context.a_bone_by_id);
      context.a_bone_by_id = NULL;
    }

  if (context.a_bone_by_content != NULL)
    {
      oniguruma_regex_free(context.a_bone_by_content);
      context.a_bone_by_content = NULL;
    }

  if (context.link_text_buffer != default_text_buffer)
    G_FREE(context.link_text_buffer);
  G_FREE(args);
}


static gboolean
start_thread_rendering(OchushaBBSThread *real_thread, const gchar *title,
		       gpointer user_data)
{
  RendererContext *context = (RendererContext *)user_data;
  BBSThreadView *view = context->view;
  BBSThreadGUIInfo *info = context->info;

#if DEBUG_GUI_MOST
  fprintf(stderr, "start_rendering\n");
#endif

  if (context->title_shown)
    return TRUE;

  context->title_shown = TRUE;

  if (context->last_read_number == 0 && info->res_num == 0)
    {
      /* ɤॹξ祿ȥߤˤΤǤ˥ޡդ롣*/
      info->last_read = bbs_thread_view_create_mark(view);
    }
  else
    info->last_read = NULL;

  bbs_thread_view_push_tag(view, context->thread_title_tag);

  parse_text(&context->handler, context, title, -1);
  bbs_thread_view_append_text(view, "\n", 1);

  bbs_thread_view_pop_tags(view, context->thread_title_tag);

  if (info->last_read != NULL)
    {
      /* bbs_thread_view_scroll_to_mark(...)ϥ뤵뤿
       * ǤϤʤŪ˥ƬŽդ뤿˸Ƥ֡
       */
      bbs_thread_view_scroll_to_mark(view, info->last_read);
    }

  return TRUE;	/* go ahead! */
}


#define TEXT_BUFFER_LEN		4096
#define LARGE_COLON		"\357\274\232"

static gboolean
render_response(OchushaBBSThread *real_thread, int number,
		const OchushaBBSResponse *response,
		gpointer user_data)
{
  RendererContext *context = (RendererContext *)user_data;
  OchushaBBSThread *thread = context->thread;
  BBSThreadView *view = context->view;
  BBSThreadGUIInfo *info = context->info;
  char buffer[18];
  gboolean a_bone;

#if DEBUG_GUI_MOST
  fprintf(stderr, "rendering response #%d, last_read_number=%d\n",
	  number, context->last_read_number);
#endif

  if (context->last_rendered_response >= number)
    return TRUE;

  context->last_rendered_response = number;

  if (context->a_bone_by_name != NULL
      && oniguruma_regex_match(context->a_bone_by_name, response->name))
    a_bone = TRUE;
  else if (context->a_bone_by_id != NULL
	   && oniguruma_regex_match(context->a_bone_by_id, response->date_id))
    a_bone = TRUE;
  else if (context->a_bone_by_content != NULL
	   && oniguruma_regex_match(context->a_bone_by_content,
				    response->content))
    a_bone = TRUE;
  else
    a_bone = FALSE;

  if (context->last_read_number >= 0)
    {
      if (number >= info->offsets_length)
	{
	  unsigned int new_len = info->offsets_length * 2;
	  info->offsets = G_REALLOC(info->offsets, new_len * sizeof(guint));
	  memset(info->offsets + info->offsets_length, 0,
		 (new_len - info->offsets_length) * sizeof(guint));
	  info->offsets_length = new_len;
	}
      info->offsets[number] = bbs_thread_view_get_current_offset(view);
    }

  if (number == info->res_num)
    {
      GtkTextMark *mark = bbs_thread_view_create_mark(view);
      save_initial_thread_view_position(view, info, mark);
    }

  if (ochusha_bbs_thread_get_number_of_responses_read(thread) < number)
    ochusha_bbs_thread_set_number_of_responses_read(thread, number);

  info->num_shown = number;
  update_threadlist_entry_real(context->application, thread);

  if (a_bone && context->transparent_a_bone)
    {
      goto after_response;
    }

  snprintf(buffer, 18, "<RES>%d", number);

  bbs_thread_view_push_tag(view, context->response_header_tag);
  bbs_thread_view_append_text(view, "\n", 1);
  if (context->last_read_number >= 0)
    bbs_thread_view_append_text_as_link(view, buffer + 5, -1, buffer);
  else
    bbs_thread_view_append_text(view, buffer + 5, -1);
  bbs_thread_view_append_text(view, LARGE_COLON, 3);

  if (!a_bone)
    {
      gchar internal_mailto[PATH_MAX];
      int res_num = 0;
      char *link = NULL;
      const gchar *tmp_name = response->name;

      if (response->name[0] != '\0'
	  && reverse_strpbrk(tmp_name, "0123456789 ") == NULL
	  && sscanf(tmp_name, "%d", &res_num) == 1
	  && (link = ochusha_bbs_thread_get_url_for_response(thread, res_num,
							     res_num)) != NULL)
	{
	  /* ̾󤬿ξ */
	  if (response->mailto == NULL)
	    snprintf(internal_mailto, PATH_MAX, "<FAKE>%s", link);
	  else
	    snprintf(internal_mailto, PATH_MAX,
		     "%s<FAKE>%s", response->mailto, link);
	  G_FREE(link);
	}
      else
	{
	  if (response->mailto == NULL)
	    internal_mailto[0] = '\0';
	  else
	    g_strlcpy(internal_mailto, response->mailto, PATH_MAX);
	}

      if (internal_mailto[0] == '\0')
	{
	  bbs_thread_view_push_tag(view, context->response_name_tag);
	  bbs_thread_view_push_tag(view, context->bold_tag);
	  parse_text(&context->handler, context, response->name, -1);
	  bbs_thread_view_pop_tags(view, context->bold_tag);
	  bbs_thread_view_pop_tags(view, context->response_name_tag);
	  if (context->show_mailto_literally)
	    bbs_thread_view_append_text(view, " []", 3);
	}
      else
	{
	  gchar *name = simple_string_canon(response->name, -1,
					    context->handler.converter,
					    context->handler.helper);
	  gchar *mailto = convert_string(context->handler.converter,
					 context->handler.helper,
					 internal_mailto, -1);
	  bbs_thread_view_push_tag(view, context->response_name_tag);
	  if (response->mailto == NULL)
	    {
	      bbs_thread_view_push_tag(view, context->bold_tag);
	      bbs_thread_view_append_text_as_hidden_link(view, name, -1,
						(mailto != NULL
						 ? mailto : _("broken here")));
	      bbs_thread_view_pop_tags(view, context->bold_tag);
	    }
	  else if (strcmp(response->mailto, "sage") != 0)
	    bbs_thread_view_append_text_as_link(view, name, -1,
						(mailto != NULL
						 ? mailto : _("broken here")));
	  else
	    bbs_thread_view_append_text_as_sage(view, name, -1,
						(mailto != NULL
						 ? mailto : _("broken here")));
	  bbs_thread_view_pop_tags(view, context->response_name_tag);
	  if (context->show_mailto_literally)
	    {
	      char *tmp_mailto;
	      bbs_thread_view_append_text(view, " [", 2);
	      
	      if (mailto != NULL)
		{
		  tmp_mailto = strstr(mailto, "<FAKE>");
		  if (tmp_mailto != NULL)
		    *tmp_mailto = '\0';
		  tmp_mailto = simple_string_canon(mailto, -1, NULL, NULL);
		}
	      else
		tmp_mailto = NULL;

	      if (tmp_mailto != NULL)
		{
		  bbs_thread_view_append_text(view, tmp_mailto, -1);
		  G_FREE(tmp_mailto);
		}
	      else
		bbs_thread_view_append_text(view, _("broken here"), -1);

	      bbs_thread_view_append_text(view, "]", 1);
	    }
	    
	  if (name != NULL)
	    G_FREE(name);
	  if (mailto != NULL)
	    G_FREE(mailto);
	}
    }
  else
    {
      /* ܡ */
      bbs_thread_view_push_tag(view, context->response_name_tag);
      bbs_thread_view_push_tag(view, context->bold_tag);
      bbs_thread_view_append_text_as_link(view, _("A Bone"), -1, _("A Bone"));
      bbs_thread_view_pop_tags(view, context->bold_tag);
      bbs_thread_view_pop_tags(view, context->response_name_tag);
      if (context->show_mailto_literally)
	{
	  bbs_thread_view_append_text(view, " [", 2);
	  bbs_thread_view_append_text(view, _("A Bone"), -1);
	  bbs_thread_view_append_text(view, "]", 1);
	}
    }

  bbs_thread_view_append_text(view, LARGE_COLON, 3);
  if (!a_bone)
    parse_text(&context->handler, context, response->date_id, -1);
  else
    bbs_thread_view_append_text(view, _("A Bone"), -1);

  bbs_thread_view_append_text(view, "\n", 1);
  bbs_thread_view_pop_tags(view, context->response_header_tag);

  bbs_thread_view_push_tag(view, context->response_paragraph_tag);

  if (!a_bone)
    {
      const char *cur_pos = response->content;
      while (cur_pos != NULL)
	{
	  cur_pos = parse_text(&context->handler, context, cur_pos, -1);
	  if (cur_pos != NULL)
	    {
	      if (*cur_pos == '&')
		{
		  parse_text(&context->handler, context, "&amp;", 5);
		  cur_pos++;
		}
	      else if (*cur_pos == '<')
		{
		  parse_text(&context->handler, context, "&lt;", 4);
		  cur_pos++;
		}
	      else
		break;	/*  */
	    }
	}
    }
  else
    {
      /* ܡ */
      gchar *link = ochusha_bbs_thread_get_url_for_response(context->thread,
							    number, number);
      bbs_thread_view_append_text_as_link(view, _("A Bone"), -1, link);
      G_FREE(link);
    }

  bbs_thread_view_pop_tags(view, context->response_paragraph_tag);

  bbs_thread_view_append_text(view, "\n", 1);

 after_response:
  if (context->application->stick_bookmark)
    {
      if (number == info->bookmark_response_number)
	{
	  insert_bookmark(view, context->thread, number, context->application);
	  if (info->res_num == 0)
	    save_initial_thread_view_position(view, info, info->last_read);
	}
    }
  else
    {
      if (number == context->last_read_number)
	{
	  insert_bookmark(view, context->thread, number, context->application);
	  if (info->res_num == 0)
	    save_initial_thread_view_position(view, info, info->last_read);
	}
    }

  return TRUE;
}


static gboolean
render_broken_response(OchushaBBSThread *real_thread, int number,
		       gpointer user_data)
{
  RendererContext *context = (RendererContext *)user_data;
  OchushaBBSThread *thread = context->thread;
  BBSThreadView *view = context->view;
  BBSThreadGUIInfo *info = context->info;
  char buffer[13];
#if DEBUG_GUI_MOST
  fprintf(stderr, "rendering response #%d, last_read_number=%d\n",
	  number, context->last_read_number);
#endif

  if (context->last_rendered_response >= number)
    return TRUE;

  context->last_rendered_response = number;

  if (context->last_read_number >= 0)
    {
      if (number >= info->offsets_length)
	{
	  unsigned int new_len = info->offsets_length * 2;
	  info->offsets = G_REALLOC(info->offsets, new_len);
	  memset(info->offsets + info->offsets_length, 0,
		 (new_len - info->offsets_length) * sizeof(guint));
	  info->offsets_length = new_len;
	}
      info->offsets[number] = bbs_thread_view_get_current_offset(view);
    }

  if (number == info->res_num)
    {
      GtkTextMark *mark = bbs_thread_view_create_mark(view);
      save_initial_thread_view_position(view, info, mark);
    }
  
  if (ochusha_bbs_thread_get_number_of_responses_read(thread) < number)
    ochusha_bbs_thread_set_number_of_responses_read(thread, number);

  info->num_shown = number;
  update_threadlist_entry_real(context->application, thread);

  snprintf(buffer, 13, "\n%d", number);

  bbs_thread_view_push_tag(view, context->response_header_tag);
  bbs_thread_view_append_text(view, buffer, -1);
  bbs_thread_view_append_text(view, LARGE_COLON, 3);
  bbs_thread_view_append_text(view, LARGE_COLON, 3);
  bbs_thread_view_append_text(view, _("[broken here]\n"), -1);
  bbs_thread_view_pop_tags(view, context->response_header_tag);

  bbs_thread_view_push_tag(view, context->response_paragraph_tag);
  bbs_thread_view_append_text(view, _("[broken here]\n"), -1);
  bbs_thread_view_pop_tags(view, context->response_paragraph_tag);

  if (context->application->stick_bookmark)
    {
      if (number == info->bookmark_response_number)
	{
	  insert_bookmark(view, context->thread, number, context->application);
	  if (info->res_num == 0)
	    save_initial_thread_view_position(view, info, info->last_read);
	}
    }
  else
    {
      if (number == context->last_read_number)
	{
	  insert_bookmark(view, context->thread, number, context->application);
	  if (info->res_num == 0)
	    save_initial_thread_view_position(view, info, info->last_read);
	}
    }

  return TRUE;
}


static gboolean
end_thread_rendering(OchushaBBSThread *real_thread, gboolean finished,
		     gpointer user_data)
{
  RendererContext *context = (RendererContext *)user_data;
  OchushaBBSThread *thread = context->thread;
  BBSThreadView *view = context->view;
  BBSThreadGUIInfo *info = context->info;

  if (finished)
    {
      if (info->last_read == NULL)
	{
	  /*
	   * MEMO: ꤨ͡Ĥꡣ
	   */
	  info->last_read = bbs_thread_view_create_mark(view);
	  save_initial_thread_view_position(view, info, info->last_read);
#if DEBUG_GUI
	  fprintf(stderr, "end_thread_rendering: What's happen?\n");
#endif
	}
      info->res_num = 0;
    }
  else
    {
      gchar message[DEFAULT_BUFFER_SIZE];
#if DEBUG_GUI_MOST
      fprintf(stderr, "end_thread_rendering: a bone happen!?\n");
#endif
      /*
       * MEMO: å夬äƤ(ܡ󤵤줿쥹)
       *       å夵Ƥ̤ͤ꾯ʤäʤɡ
       *       DATեΤμľȤ뤬
       *       DATեβϥ롼ϡꤷ쥹ְʹߤ
       *       ƤΤcallbackؿƤ֤Τǡι
       *       å夬äƤΤȽˡ
       *       äƤޤȡ쥹ɽʤȤˤʤ롣
       *       ʤΤǡΤˤȤʤɽä
       *       ʬϡܡ󤵤ƤƤ⸵Υ쥹ɽ뤳Ȥ
       *       롣
       */
      snprintf(message, DEFAULT_BUFFER_SIZE,
	       _("A Bone occurs in thread %s@%s board"),
	       thread->title, ochusha_bbs_thread_get_board(thread)->name);

      if (info->message_id != 0)
	gtk_statusbar_remove(context->application->statusbar,
			     thread_netstat_id, info->message_id);
      info->message_id = gtk_statusbar_push(context->application->statusbar,
					    thread_netstat_id, message);

      ochusha_bbs_thread_set_number_of_responses_read(thread, 1);
      /* 0ˤȡɤ߹ߤߤˡå夬ĤäƤΤ̤
       * ˤʤ륿ߥ󥰤Τǡ
       */
    }

#if DEBUG_GUI_MOST
  fprintf(stderr, "end_rendering\n");
#endif

  return TRUE;
}


static void
process_link(RendererContext *context)
{
#if 1	/* VIPк */
  unsigned int from;
  unsigned int to;
#endif
  gchar *text;
  if (context->link == NULL)
    return;

  text = simple_string_canon(context->link_text_buffer,
			     context->link_text_length,
			     NULL, NULL);

  if (context->application->enable_inline_image)
    {
      char *link = context->link;
      if (g_str_has_suffix(link, ".jpg") || g_str_has_suffix(link, ".png")
	  || g_str_has_suffix(link, ".gif") || g_str_has_suffix(link, ".jpeg")
	  || g_str_has_suffix(link, ".JPG") || g_str_has_suffix(link, ".PNG")
	  || g_str_has_suffix(link, ".GIF") || g_str_has_suffix(link, ".JPEG"))
	{
	  /* ᡼μ¸ */
	  OchushaApplication *application = context->application;
	  GtkWidget *image
	    = ochusha_download_image(application, link,
				     application->inline_image_width,
				     application->inline_image_height);
	  if (image != NULL)
	    bbs_thread_view_append_widget(context->view, image, 20, 0);
#if 0
	  fprintf(stderr, "insert image: %s\n", link);
	  fprintf(stderr, "text: %s\n", text);
#endif
	}
    }

#if 1	/* VIPк */
#if 0
  fprintf(stderr, "<1>context->link=\"%s\"\n", context->link);
#endif
  if (!ochusha_bbs_thread_check_url(context->thread, context->link,
				    &from, &to))
    {
      char *server = ochusha_utils_url_extract_http_server(context->link);
      if (server == NULL
	  && ochusha_utils_2ch_check_url(context->link, NULL, NULL,
					 NULL, NULL, &from, &to, NULL))
	{
	  G_FREE(context->link);
	  context->link
	    = ochusha_bbs_thread_get_url_for_response(context->thread,
						      from, to);
	  if (context->link == NULL)
	    {
	      if (text != NULL)
		{
		  bbs_thread_view_append_text(context->view, text, -1);
		  G_FREE(text);
		}
	      else
		bbs_thread_view_append_text(context->view,
					    context->link_text_buffer,
					    context->link_text_length);
	      context->last_text_is_link = TRUE;
	      return;
	    }
	}
      else
	{
	  if (server != NULL)
	    G_FREE(server);
	}
    }
#endif
#if 0
  fprintf(stderr, "<2>context->link=\"%s\"\n", context->link);
#endif

  if (text != NULL)
    bbs_thread_view_append_text_as_link(context->view, text, -1,
					context->link);
  else
    bbs_thread_view_append_text_as_link(context->view,
					context->link_text_buffer,
					context->link_text_length,
					context->link);

  if (text != NULL)
    G_FREE(text);
  G_FREE(context->link);
  context->link = NULL;
  context->last_text_is_link = TRUE;
}


static void
start_element_cb(void *user_data, const char *name, const char *const attrs[])
{
  RendererContext *context = (RendererContext *)user_data;
  BBSThreadView *view = context->view;
  int i;

  context->last_text_is_link = FALSE;
#if DEBUG_GUI_MOST
  fprintf(stderr, "<%s", name);
  i = 0;
  while (attrs[i * 2] != NULL)
    {
      fprintf(stderr, " %s=%s", attrs[i * 2], attrs[i * 2 + 1]);
      i++;
    }
  fprintf(stderr, ">\n");
#endif

  while (context->number_of_gt_characters > 0)
    {
      bbs_thread_view_append_text(context->view, ">", 1);
      context->number_of_gt_characters--;
    }

  bbs_thread_view_push_tag(view,
			   g_hash_table_lookup(context->tag_table, name));

  if (strcasecmp(name, "br") == 0)
    {
      bbs_thread_view_pop_tags(view, context->response_paragraph_tag);
      bbs_thread_view_append_text(view, "\n", 1);
      bbs_thread_view_push_tag(view, context->response_paragraph_tag);
    }
  else if (strcasecmp(name, "a") == 0)
    {
      gchar *link = NULL;
      if (context->link != NULL)
	{
#if DEBUG_GUI
	  fprintf(stderr, "previous link hasn't been closed.\n");
#endif
	  process_link(context);
	}
      context->link_text_buffer[0] = '\0';
      context->link_text_length = 0;

      i = 0;
      while (attrs[i * 2] != NULL)
	if (strcasecmp(attrs[i * 2], "href") == 0)
	  {
	    const char *tmp_pos = attrs[i * 2 + 1];
	    while (*tmp_pos != '\0' && (*tmp_pos & 0x80) == 0)
	      tmp_pos++;

	    if (*tmp_pos == '\0')
	      link = simple_string_canon(attrs[i * 2 + 1], -1, NULL, NULL);
	    else
	      link = convert_string(context->handler.converter,
				    context->handler.helper,
				    attrs[i * 2 + 1], -1);
#if DEBUG_GUI_MOST
	    fprintf(stderr, "href=\"%s\"\n", link);
#endif
	    break;
	  }
	else
	  i++;

      if (link == NULL)
	link = G_STRDUP(_("*empty link*"));

      context->link = link;
    }
}


static void
end_element_cb(void *user_data, const char *name)
{
  RendererContext *context = (RendererContext *)user_data;
  BBSThreadView *view = context->view;

#if DEBUG_GUI_MOST
  fprintf(stderr, "</%s>\n", name);
#endif
  while (context->number_of_gt_characters > 0)
    {
      bbs_thread_view_append_text(context->view, ">", 1);
      context->number_of_gt_characters--;
    }

  bbs_thread_view_pop_tags(view,
			   g_hash_table_lookup(context->tag_table, name));

  if (strcasecmp(name, "a") == 0 && context->link != NULL)
    process_link(context);
}


static void
characters_cb(void *user_data, const char *ch, int len)
{
  gchar *buffer;
  int buffer_size;
  RendererContext *context = (RendererContext *)user_data;

  if (context->link == NULL)
    {
      if (context->application->analyze_informal_links)
	{
	  regmatch_t match[3];
	  gchar *text;
	  gchar *text_top;

	  if (*ch == '>' && len == 1)
	    {
	      context->number_of_gt_characters++;
	      len = 0;
	      text = NULL;
	    }
	  else if (context->number_of_gt_characters > 0)
	    {
	      text = G_MALLOC(len + context->number_of_gt_characters + 1);
	      memset(text, '>', context->number_of_gt_characters);
	      memcpy(text + context->number_of_gt_characters, ch, len);
	      text_top = text;
	      len += context->number_of_gt_characters;
	      text[len] = '\0';
	      context->number_of_gt_characters = 0;
	    }
	  else
	    text = G_STRNDUP(ch, len);

	  text_top = text;

	  while (len > 0)
	    {
	      while (context->last_text_is_link
		     && regexec(&context->comma_separated_link_pattern,
				text_top, 2, match, 0) == 0)
		{
		  int match_len = match[0].rm_eo - match[0].rm_so;
		  int from = 0;
		  int to = 0;
		  char *link;

		  if (match[1].rm_so != -1 && match[1].rm_eo != match[1].rm_so)
		    {
		      from = parse_int(text_top + match[1].rm_so,
				       match[1].rm_eo - match[1].rm_so);
		    }

		  if (match[2].rm_so != -1 && match[2].rm_eo != match[2].rm_so)
		    {
		      to = parse_int(text_top + match[2].rm_so + 1,
				     match[2].rm_eo - match[2].rm_so - 1);
		    }
		  else
		    to = from;

		  if (from > 0)
		    link = ochusha_bbs_thread_get_url_for_response(context->thread,
								   from, to);
		  else
		    link = NULL;

		  if (link != NULL)
		    {
		      bbs_thread_view_append_text_as_link(context->view,
							  text_top + match[0].rm_so,
							  match_len,
							  link);
		      G_FREE(link);
		    }
		  else
		    bbs_thread_view_append_text(context->view,
						text_top + match[0].rm_so,
						match_len);

		  text_top += match_len;
		  len -= match_len;
		}

	      context->last_text_is_link = FALSE;

	      if (len == 0)
		break;

	      if (regexec(&context->informal_link_pattern,
			  text_top, 2, match, 0) == 0)
		{
		  int no_match_len = match[0].rm_so;
		  int match_len = match[0].rm_eo - match[0].rm_so;
		  int from = 0;
		  int to;
		  char *link;

		  if (no_match_len > 0)
		    bbs_thread_view_append_text(context->view, text_top,
						no_match_len);

		  if (match[1].rm_so != -1 && match[1].rm_eo != match[1].rm_so)
		    {
		      from = parse_int(text_top + match[1].rm_so,
				       match[1].rm_eo - match[1].rm_so);
		    }

		  if (match[2].rm_so != -1 && match[2].rm_eo != match[2].rm_so)
		    {
		      to = parse_int(text_top + match[2].rm_so + 1,
				     match[2].rm_eo - match[2].rm_so - 1);
		    }
		  else
		    to = from;

		  if (from > 0)
		    link = ochusha_bbs_thread_get_url_for_response(context->thread,
								   from, to);
		  else
		    link = NULL;

		  if (link != NULL)
		    {
		      bbs_thread_view_append_text_as_link(context->view,
							  text_top + match[0].rm_so,
							  match_len,
							  link);
		      G_FREE(link);
		      context->last_text_is_link = TRUE;
		    }
		  else
		    bbs_thread_view_append_text(context->view,
						text_top + match[0].rm_so,
						match_len);

		  text_top += no_match_len + match_len;
		  len -= (no_match_len + match_len);
		}
	      else
		{
		  if (text_top[len - 1] == '>')
		    {
		      context->number_of_gt_characters++;
		      len--;
		    }

		  if (len > 0)
		    {
		      bbs_thread_view_append_text(context->view,
						  text_top, len);
		    }

		  break;
		}
	    }

	  if (text != NULL)
	    G_FREE(text);
	}
      return;
    }

  buffer = context->link_text_buffer;
  buffer_size = context->link_text_buffer_size;

  while (buffer_size <= (context->link_text_length + len))
    {
#if DEBUG_GUI_MOST
      fprintf(stderr, "buffer_size=%d, requirement=%d\n",
	      buffer_size, (context->link_text_length + len));
#endif
      buffer_size *= 2;
      if (context->link_text_buffer_size == DEFAULT_BUFFER_SIZE)
	{
	  buffer = (gchar *)G_MALLOC(buffer_size);
	  memcpy(buffer, context->link_text_buffer, context->link_text_length);
	}
      else
	buffer = (gchar *)G_REALLOC(buffer, buffer_size);

      context->link_text_buffer = buffer;
      context->link_text_buffer_size = buffer_size;
    }

  if (buffer_size > context->link_text_length + len)
    {
      memcpy(buffer + context->link_text_length, ch, len);
      context->link_text_length += len;
    }
  else
    {
      int left = buffer_size - context->link_text_length - 1;
      memcpy(buffer + context->link_text_length, ch, left);
      context->link_text_length = buffer_size - 1;
    }

  buffer[context->link_text_length] = '\0';
}


static gboolean
prepare_buffer_for_popup(OchushaApplication *application,
			 OchushaBBSThread *thread,
			 const gchar *link,
			 unsigned int *response_from,
			 unsigned int *response_to,
			 OchushaAsyncBuffer **buffer)
{
  unsigned int from = 0;
  unsigned int to = 0;
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);

  if (application->maximum_number_of_popup_responses < 1)
    return FALSE;

  if (!ochusha_bbs_thread_check_url(thread, link, &from, &to))
    return FALSE;

#if DEBUG_GUI_MOST
  fprintf(stderr, "prepare_buffer_for_popup(): from=%d, to=%d\n", from, to);
#endif

  if ((from == 0 && to == 0)
      || from > ochusha_bbs_thread_get_number_of_responses_read(thread))
    return FALSE;

  if (to > ochusha_bbs_thread_get_number_of_responses_read(thread))
    to = ochusha_bbs_thread_get_number_of_responses_read(thread);

  if (to > 0 && from > to)
    {
      unsigned int tmp = to;
      to = from;
      from = tmp;
    }

  *response_from = from;
  *response_to = to;

  /*
   * ǶΥɤǤɽƤ륹ɬresponse_source_buffer˥Хåե
   * äƤ뤬⤷Ŭʥߥ󥰤ǲ٤⤷ʤΤǡ
   * Τ(å夫Τɤ߹)ɤĤƤ
   */
  if (info->response_source_buffer != NULL)
    {
      *buffer = info->response_source_buffer;
      return TRUE;
    }

  *buffer = ochusha_bbs_thread_get_responses_source(thread,
					application->broker, NULL,
					OCHUSHA_NETWORK_BROKER_CACHE_ONLY);
  info->response_source_buffer = *buffer;

  return (*buffer != NULL);
}


static gboolean
popup_window_button_press_event_cb(GtkWidget *widget, GdkEventButton *event,
				   PopupContext *popup_context)
{
  popup_context->now_sticked = TRUE;
  return FALSE;
}


static gboolean
popup_window_button_release_event_cb(GtkWidget *widget, GdkEventButton *event,
				     PopupContext *popup_context)
{
  popup_context->now_sticked = FALSE;
  return FALSE;
}


static gboolean
popup_window_enter_notify_event_cb(GtkWidget *widget,
				   GdkEventCrossing *event,
				   PopupContext *popup_context)
{
#if 0
  fprintf(stderr, "enter_notify: %p, %p\n", widget, popup_context);
#endif

  if (popup_context->close_delay_id != 0)
    {
      g_source_remove(popup_context->close_delay_id);
      popup_context->close_delay_id = 0;
    }

  popup_context->now_sticked = FALSE;
  popup_context->now_pointed = TRUE;

  return FALSE;
}


static gboolean
popup_window_leave_notify_event_cb(GtkWidget *widget,
				   GdkEventCrossing *event,
				   PopupContext *popup_context)
{
  popup_context->now_pointed = FALSE;
  hide_popup(popup_context);
  return FALSE;
}


static void
popup_view_size_request_cb(GtkWidget *widget, GtkRequisition *requisition,
			     PopupContext *popup_context)
{
  GdkScreen *screen;
  int x_pos;
  int y_pos;
#if 0
  fprintf(stderr, "popup_view_size_request_cb(%p, %p, %p): w=%d, h=%d\n",
	  widget, requisition, popup_context,
	  requisition->width, requisition->height);
#endif
  if (requisition->width <= 0 || requisition->height <= 0)
    return;

  gtk_widget_set_size_request(widget,
			      requisition->width
			      + THREAD_VIEW_EXTRA_WIDTH,
			      requisition->height);
  gtk_window_resize(GTK_WINDOW(popup_context->popup_window),
		    requisition->width + THREAD_VIEW_EXTRA_WIDTH,
		    requisition->height);

  screen = gtk_widget_get_screen(popup_context->popup_window);
  x_pos = popup_context->x_pos;
  y_pos = popup_context->y_pos - (requisition->height + POPUP_MARGIN_HEIGHT);
  y_pos = y_pos > 0 ? y_pos : 0;

  if (screen != NULL)
    {
      int screen_width = gdk_screen_get_width(screen);
      int widget_width = requisition->width + THREAD_VIEW_EXTRA_WIDTH;
      if (screen_width > 0 && (x_pos + widget_width) > screen_width)
	{
	  x_pos = screen_width - widget_width;
	  if (x_pos < 0)
	    x_pos = 0;
	}
    }

  gtk_window_move(GTK_WINDOW(popup_context->popup_window), x_pos, y_pos);
}


static void
render_popup_response(PopupContext *popup_context,
		      OchushaBBSThread *thread, OchushaAsyncBuffer *buffer,
		      int response_from, int num_responses)
{
  OchushaApplication *application
    = popup_context->renderer_context.application;
  GtkWidget *view = bbs_thread_view_new(thread);
  gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(view), GTK_WRAP_NONE);

  g_object_set_qdata(G_OBJECT(view), popup_parent_context_id,
		     popup_context);
#if DEBUG_POPUP
  fprintf(stderr, "render_popup_response(): view=%p\n", view);
#endif

  popup_context->renderer_context.show_mailto_literally
    = show_mailto(application, thread);

  g_signal_connect(G_OBJECT(view), "link_mouse_over",
		   G_CALLBACK(link_mouse_over_cb), application);
  g_signal_connect(G_OBJECT(view), "link_mouse_out",
		   G_CALLBACK(link_mouse_out_cb), application);
  g_signal_connect(G_OBJECT(view), "link_mouse_press",
		   G_CALLBACK(link_mouse_press_cb), application);
  g_signal_connect(G_OBJECT(view), "link_mouse_release",
		   G_CALLBACK(link_mouse_release_cb), application);
  g_signal_connect(G_OBJECT(view), "copy_clipboard",
		   G_CALLBACK(copy_clipboard_cb), application);

  g_signal_connect(G_OBJECT(view), "button_press_event",
		   G_CALLBACK(popup_window_button_press_event_cb),
		   popup_context);
  g_signal_connect(G_OBJECT(view), "button_release_event",
		   G_CALLBACK(popup_window_button_release_event_cb),
		   popup_context);
  g_signal_connect(G_OBJECT(view), "enter_notify_event",
		   G_CALLBACK(popup_window_enter_notify_event_cb),
		   popup_context);
  g_signal_connect(G_OBJECT(view), "leave_notify_event",
		   G_CALLBACK(popup_window_leave_notify_event_cb),
		   popup_context);

  g_signal_connect(G_OBJECT(view), "size_request",
		   G_CALLBACK(popup_view_size_request_cb), popup_context);

  gtk_widget_show_all(view);

  if (popup_context->renderer_context.view != NULL)
    gtk_container_remove(popup_context->popup_frame,
			 GTK_WIDGET(popup_context->renderer_context.view));
  else if (popup_context->image != NULL)
    {
      gtk_container_remove(popup_context->popup_frame,
			   popup_context->image);
      popup_context->image = NULL;
    }
  gtk_container_add(popup_context->popup_frame, view);
  gtk_widget_show(GTK_WIDGET(popup_context->popup_frame));

  popup_context->renderer_context.view = BBS_THREAD_VIEW(view);
  popup_context->renderer_context.link_text_length = 0;
  popup_context->renderer_context.thread = thread;

#if DEBUG_GUI_MOST
  if (num_responses > 1)
    fprintf(stderr, "from=%d, num=%d\n", response_from, num_responses);
#endif

  popup_context->renderer_context.handler.converter
    = iconv_open("UTF-8//IGNORE",
		 ochusha_bbs_thread_get_response_character_encoding(thread));
  if (popup_context->renderer_context.handler.converter == (iconv_t)-1)
    {
      fprintf(stderr, "Out of memory: Couldn't render popup\n");
      return;
    }

  popup_context->renderer_context.last_rendered_response = 0;

  gdk_threads_leave();
  ochusha_bbs_thread_parse_responses(thread, buffer,
				     response_from - 1, num_responses,
				     TRUE,
				     NULL, render_response,
				     render_broken_response, NULL,
				     (StartParsingCallback *)gdk_threads_enter,
				     (BeforeWaitCallback *)gdk_threads_leave,
				     (AfterWaitCallback *)gdk_threads_enter,
				     (EndParsingCallback *)gdk_threads_leave,
				     &popup_context->renderer_context);
  gdk_threads_enter();

  iconv_close(popup_context->renderer_context.handler.converter);
  popup_context->renderer_context.handler.converter = (iconv_t)0;
}


#if 0	/* ɬפʤäݤ */
static void
image_realize_cb(GtkWidget *widget, PopupContext *unused)
{
  GdkWindow *window = gtk_widget_get_parent_window(widget);
  fprintf(stderr, "realize_cb\n");
  gdk_window_set_events(window,
			(gdk_window_get_events(window)
			 & ~GDK_POINTER_MOTION_HINT_MASK)
			| GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK);
}
#endif


static void
popup_image_size_request_cb(GtkWidget *widget, GtkRequisition *requisition,
			    PopupContext *popup_context)
{
  GdkScreen *screen;
  int x_pos;
  int y_pos;
#if 0
  fprintf(stderr, "popup_view_size_request_cb(%p, %p, %p): w=%d, h=%d\n",
	  widget, requisition, popup_context,
	  requisition->width, requisition->height);
#endif
  if (requisition->width <= 0 || requisition->height <= 0)
    return;

  gtk_widget_set_size_request(widget,
			      requisition->width,
			      requisition->height);
  gtk_window_resize(GTK_WINDOW(popup_context->popup_window),
		    requisition->width, requisition->height);

  screen = gtk_widget_get_screen(popup_context->popup_window);
  x_pos = popup_context->x_pos;
  y_pos = popup_context->y_pos - (requisition->height + POPUP_MARGIN_HEIGHT);
  y_pos = y_pos > 0 ? y_pos : 0;

  if (screen != NULL)
    {
      int screen_width = gdk_screen_get_width(screen);
      int widget_width = requisition->width;
      if (screen_width > 0 && (x_pos + widget_width) > screen_width)
	{
	  x_pos = screen_width - widget_width;
	  if (x_pos < 0)
	    x_pos = 0;
	}
    }

  gtk_window_move(GTK_WINDOW(popup_context->popup_window), x_pos, y_pos);
}


static void
show_popup(OchushaApplication *application, OchushaBBSThread *thread,
	   const gchar *link, int x_pos, int y_pos, BBSThreadView *parent)
{
  /*
   * MEMO: gdk_threads_enter()줿ĶƤФƤ롣
   */
  PopupContext *popup_context;
  OchushaAsyncBuffer *buffer;
  int response_from = 0;
  int response_to = 0;
  int num_responses;

  popup_context = ensure_popup_context(parent, thread, application);

#if DEBUG_GUI_MOST
  fprintf(stderr, "show_popup: link=\"%s\", x_pos=%d, y_pos=%d\n",
	  link, x_pos, y_pos);
#endif
  if (application->enable_popup_image)
    {
      if (g_str_has_suffix(link, ".jpg") || g_str_has_suffix(link, ".png")
	  || g_str_has_suffix(link, ".gif") || g_str_has_suffix(link, ".jpeg")
	  || g_str_has_suffix(link, ".JPG") || g_str_has_suffix(link, ".PNG")
	  || g_str_has_suffix(link, ".GIF") || g_str_has_suffix(link, ".JPEG"))
	{
	  /* ᡼μ¸ */
	  GtkWidget *image
	    = ochusha_download_image(application, link,
				     application->popup_image_width,
				     application->popup_image_height);
	  if (image != NULL)
	    {
	      GtkWidget *event_box = gtk_event_box_new();

	      gtk_widget_show(image);
	      gtk_container_add(GTK_CONTAINER(event_box), image);
	      gtk_widget_show(event_box);
#if 0
	      g_signal_connect(G_OBJECT(event_box), "realize",
			       G_CALLBACK(image_realize_cb), popup_context);
#endif
	      g_signal_connect(G_OBJECT(event_box), "enter_notify_event",
			       G_CALLBACK(popup_window_enter_notify_event_cb),
			       popup_context);
	      g_signal_connect(G_OBJECT(event_box), "leave_notify_event",
			       G_CALLBACK(popup_window_leave_notify_event_cb),
			       popup_context);
	      g_signal_connect(G_OBJECT(image), "size_request",
			       G_CALLBACK(popup_image_size_request_cb),
			       popup_context);

	      if (popup_context->renderer_context.view != NULL)
		{
		  gtk_container_remove(popup_context->popup_frame,
				       GTK_WIDGET(popup_context->renderer_context.view));
		  popup_context->renderer_context.view = NULL;
		}
	      else if (popup_context->image != NULL)
		gtk_container_remove(popup_context->popup_frame,
				     popup_context->image);
	      gtk_container_add(popup_context->popup_frame, event_box);
	      gtk_widget_show(GTK_WIDGET(popup_context->popup_frame));
	      popup_context->image = event_box;
	      goto do_popup;
	    }
	}
    }

  if (!prepare_buffer_for_popup(application, thread, link,
				&response_from, &response_to, &buffer))
    {
#if DEBUG_POPUP
      fprintf(stderr, "show_popup(): from=%d to=%d\n",
	      response_from, response_to);
      fprintf(stderr, "Popup cannot be prepared...why?\n");
#endif
      return;
    }

  if (response_from == 0)
    response_from
      = response_to - application->maximum_number_of_popup_responses;
  if (response_to == 0)
    response_to = response_from;

  num_responses = response_to - response_from + 1;
  if (num_responses > application->maximum_number_of_popup_responses)
    num_responses = application->maximum_number_of_popup_responses;

#if DEBUG_POPUP
      fprintf(stderr, "show_popup(): from=%d num_responses=%d\n",
	      response_from, num_responses);
#endif
  render_popup_response(popup_context, thread, buffer,
			response_from, num_responses);

 do_popup:

  popup_context->x_pos = x_pos;
  popup_context->y_pos = y_pos;

  if (application->popup_response_delay == 0)
    show_popup_real(popup_context);
  else
    {
      if (popup_context->close_delay_id != 0)
	{
	  g_source_remove(popup_context->close_delay_id);
	  popup_context->close_delay_id = 0;
	}

      if (popup_context->delay_id != 0)
	g_source_remove(popup_context->delay_id);

      if (popup_context->url != NULL)
	G_FREE(popup_context->url);

      popup_context->url = G_STRDUP(link);
      popup_context->delay_id
	= g_timeout_add(application->popup_response_delay,
			(GSourceFunc)popup_delay_timeout, popup_context);
#if DEBUG_GUI_MOST
      fprintf(stderr, "timeout_add, id=%d, %p\n",
	      popup_context->delay_id, popup_context->url);
#endif
    }
}


static gboolean
popup_delay_timeout(PopupContext *popup_context)
{
  gboolean result = TRUE;

#if DEBUG_GUI_MOST
  fprintf(stderr, "popup_delay_timeout(%p)\n", data);
#endif

  gdk_threads_enter();
  if (popup_context->url != NULL)
    {
      if (popup_context->delay_id != 0)
	popup_context->delay_id = 0;

      G_FREE(popup_context->url);
      popup_context->url = NULL;

      show_popup_real(popup_context);
      result = FALSE;
    }
  gdk_threads_leave();

  return result;
}


static void
show_popup_real(PopupContext *popup_context)
{
#if DEBUG_POPUP
  fprintf(stderr, "show_popup_real(): popup_context=%p\n", popup_context);
#endif
  if (popup_menu_shown)
    return;

  if (popup_context->renderer_context.view != NULL
      || popup_context->image != NULL)
    {
      gtk_widget_show_all(popup_context->popup_window);
#if DEBUG_POPUP
      fprintf(stderr, "show_popup_real(): show_all, view=%p\n",
	      popup_context->renderer_context.view);
#endif
    }
}


static void
hide_popup_real(PopupContext *popup_context)
{
  /*
   * MEMO: gdk_threads_enter()줿ĶƤФ롣
   */
  if (popup_context->url != NULL)
    {
      G_FREE(popup_context->url);
      popup_context->url = NULL;
    }

  if (popup_context->delay_id != 0)
    {
      g_source_remove(popup_context->delay_id);
      popup_context->delay_id = 0;
    }

#if DEBUG_POPUP
  fprintf(stderr, "hide_popup_real(): view=%p\n",
	  popup_context->renderer_context.view);
#endif
  gtk_widget_hide_all(popup_context->popup_window);
  if (popup_context->renderer_context.view != NULL)
    {
      gtk_container_remove(popup_context->popup_frame,
			   GTK_WIDGET(popup_context->renderer_context.view));
      popup_context->renderer_context.view = NULL;

      if (popup_context->parent_context != NULL)
	{
	  if (popup_context->parent_context != popup_context)
	    hide_popup(popup_context->parent_context);
	}
    }
  else if (popup_context->image != NULL)
    {
      gtk_container_remove(popup_context->popup_frame,
			   popup_context->image);
      popup_context->image = NULL;

      if (popup_context->parent_context != NULL)
	{
	  if (popup_context->parent_context != popup_context)
	    hide_popup(popup_context->parent_context);
	}
    }
}


static gboolean
popup_close_delay_timeout(PopupContext *popup_context)
{
  PopupContext *child_context = NULL;
#if DEBUG_POPUP
  fprintf(stderr, "popup_close_delay_timeout: popup_context=%p\n",
	  popup_context);
#endif

  gdk_threads_enter();

  if (!popup_context->now_pointed && !popup_context->now_sticked
      && !(popup_context->renderer_context.view != NULL
	   && (child_context = g_object_get_qdata(G_OBJECT(popup_context->renderer_context.view), popup_context_id)) != NULL
	   && (child_context->renderer_context.view != NULL
	       || child_context->image != NULL)))
    {
#if DEBUG_POPUP
      fprintf(stderr, "popup_close_delay_timeout: now_pointed=%d child=%p\n",
	      popup_context->now_pointed, child_context);
#endif
      hide_popup_real(popup_context);
    }

  popup_context->close_delay_id = 0;

  gdk_threads_leave();

  return FALSE;
}


static void
hide_popup(PopupContext *popup_context)
{
  /*
   * MEMO: gdk_threads_enter()ʴĶƤФ롣
   */
  OchushaApplication *application
    = popup_context->renderer_context.application;

#if DEBUG_POPUP
  fprintf(stderr, "hide_popup: popup_context=%p\n", popup_context);
#endif

  if (application->popup_close_delay == 0)
    hide_popup_real(popup_context);
  else if (popup_context->delay_id == 0)
    {
      if (popup_context->close_delay_id != 0)
	g_source_remove(popup_context->close_delay_id); /* Ԥ֥ꥻå */
      popup_context->close_delay_id
	= g_timeout_add(application->popup_close_delay,
			(GSourceFunc)popup_close_delay_timeout, popup_context);
    }
  else
    {
      /* ɽԤΥݥåץåפɽΤ롣 */
      hide_popup_real(popup_context);
    }
}


static void
open_hisshidana_dialog(OchushaApplication *application, GtkWidget *widget,
		       OchushaBBSThread *thread, BBSThreadGUIInfo *info,
		       IconLabel *tab_label);


void
refresh_thread(OchushaApplication *application, GtkWidget *widget,
	       OchushaBBSThread *thread, IconLabel *tab_label,
	       gboolean automatic)
{
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);
  BBSThreadView *view = BBS_THREAD_VIEW(widget);
  WorkerJob *job = NULL;
  BBSThreadJobArgs *job_args = NULL;
#if DEBUG_GUI_MOST
  fprintf(stderr, "refresh_thread\n");
#endif

  BBS_THREAD_UI_LOCK
  {
    OchushaAsyncBuffer *buffer;
    OchushaAsyncBuffer *old_buffer;

    if (!info->rendering_done)
      {
#if DEBUG_GUI_MOST
	fprintf(stderr, "refresh_thread: refresh has already been requested.\n");
#endif
	info->hisshi_count++;
	goto finish_refresh;	/* 󥰤򤷤Ƥ */
      }
    info->hisshi_count = 0;
    job = G_NEW0(WorkerJob, 1);
    job_args = G_NEW0(BBSThreadJobArgs, 1);
    old_buffer = info->response_source_buffer;
    buffer = ochusha_bbs_thread_get_responses_source(thread,
				application->broker, old_buffer,
				OCHUSHA_NETWORK_BROKER_CACHE_TRY_UPDATE);

    if (buffer == NULL)
      {
#if DEBUG_NETWORK_MOST
	fprintf(stderr, "Couldn't access network!\n");
#endif
	BBS_THREAD_UI_UNLOCK;
	if (job != NULL)
	  G_FREE(job);
	if (job_args != NULL)
	  G_FREE(job_args);
	return;
      }

    if (buffer == old_buffer)
      old_buffer = NULL;
    else
      setup_for_tab_label_animation(buffer, tab_label);

    info->response_source_buffer = buffer;
    info->rendering_done = FALSE;

    job_args->application = application;
    job_args->thread = thread;
    job_args->view = view;
    job_args->buffer = buffer;
    job_args->start = ochusha_bbs_thread_get_number_of_responses_read(thread);
    job_args->number_of_responses = -1;
    job_args->refresh = TRUE;
    job_args->auto_refresh = automatic;
    job_args->tab_label = tab_label;

    job->canceled = FALSE;
    job->job = render_bbs_thread;
    job->args = job_args;

    OCHU_OBJECT_REF(buffer);
    g_object_ref(tab_label);	/* бǧ */
    OCHU_OBJECT_REF(view);
    commit_job(job);

    if (old_buffer != NULL)
      OCHU_OBJECT_UNREF(old_buffer);
  }

 finish_refresh:
  BBS_THREAD_UI_UNLOCK;

  if (info->hisshi_count >= 2 && info->hisshidana_dialog == NULL
      && !automatic)
    open_hisshidana_dialog(application, widget, thread, info, tab_label);
  return;
}


typedef struct _AutoRefresherArgs
{
  OchushaApplication *application;
  GtkWidget *widget;
  IconLabel *tab_label;

  guint previous_interval;
  int previous_number_of_responses;

  GdkColor *normal;
} AutoRefresherArgs;


static gboolean
thread_auto_refresher_timeout(gpointer data)
{
  OchushaBBSThread *thread = OCHUSHA_BBS_THREAD(data);
  AutoRefresherArgs *args = g_object_get_data(G_OBJECT(thread),
					      "auto_refresher_args");
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);

  gdk_threads_enter();
  if (args != NULL
      && info->recent_view == args->widget
      && info->recent_tab_label == args->tab_label)
    {
      int num_res = ochusha_bbs_thread_get_number_of_responses_read(thread);
      if (args->previous_number_of_responses != num_res)
	{
	  args->previous_interval -= 1000;
#if ENABLE_STRICT_CHECK
	  if (args->previous_interval < 3000)
	    args->previous_interval = 3000;
#else
	  if (args->previous_interval < 10000)
	    args->previous_interval = 10000;
#endif
	}
      else
	{
	  if (info->hisshi_count < 2)
	    args->previous_interval += 5000;
	  else
	    args->previous_interval += 15000;
	  if ((thread->flags & OCHUSHA_BBS_THREAD_STOPPED) != 0
	      || args->previous_interval > 300000)
	    {
	      /*  */
	      gtk_widget_modify_fg(GTK_WIDGET(info->recent_tab_label),
				   GTK_STATE_NORMAL, NULL);
	      gtk_widget_modify_fg(GTK_WIDGET(info->recent_tab_label),
				   GTK_STATE_ACTIVE, NULL);
	      args->previous_interval = 0;
	    }
	}
      args->previous_number_of_responses = num_res;
      refresh_thread(args->application, args->widget, thread, args->tab_label,
		     TRUE);

      if (args->previous_interval != 0)
	info->auto_refresher_id = g_timeout_add(args->previous_interval,
						thread_auto_refresher_timeout,
						thread);
      else
	{
	  info->auto_refresher_id = 0;
	  g_object_set_data(G_OBJECT(thread), "auto_refresher_args", NULL);
	}
    }
  gdk_threads_leave();

  return FALSE;
}


static void
hisshidana_dialog_response_cb(GtkWidget *dialog, int response_id,
			      OchushaApplication *application)
{
  OchushaBBSThread *thread = g_object_get_data(G_OBJECT(dialog), "thread");

  if (thread != NULL)
    {
      BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);
      GtkWidget *thread_view = info->recent_view;
      IconLabel *tab_label = info->recent_tab_label;

      if (thread_view != NULL && tab_label != NULL)
	{
	  GtkLabel *label = GTK_LABEL(tab_label);
	  AutoRefresherArgs *args;
	  static GdkColor red = { 0, 65535, 0, 0 };
	  
	  switch (response_id)
	    {
	    case GTK_RESPONSE_YES:
	      gtk_widget_modify_fg(GTK_WIDGET(label), GTK_STATE_NORMAL, &red);
	      gtk_widget_modify_fg(GTK_WIDGET(label), GTK_STATE_ACTIVE, &red);
	      args = g_new0(AutoRefresherArgs, 1);
	      args->application = application;
	      args->widget = thread_view;
	      args->tab_label = tab_label;

#if ENABLE_STRICT_CHECK
	      args->previous_interval = 5000;
#else
	      args->previous_interval = 15000;
#endif
	      args->previous_number_of_responses
		= ochusha_bbs_thread_get_number_of_responses_read(thread);

	      g_object_set_data_full(G_OBJECT(thread),
				     "auto_refresher_args", args,
				     (GDestroyNotify)g_free);
	      info->auto_refresher_id
		= g_timeout_add(args->previous_interval,
				thread_auto_refresher_timeout, thread);
	      break;

	    default:
	      gtk_widget_modify_fg(GTK_WIDGET(label), GTK_STATE_NORMAL, NULL);
	      gtk_widget_modify_fg(GTK_WIDGET(label), GTK_STATE_ACTIVE, NULL);
	      if (info->auto_refresher_id != 0)
		{
		  g_source_remove(info->auto_refresher_id);
		  info->auto_refresher_id = 0;
		}
	      g_object_set_data(G_OBJECT(thread), "auto_refresher_args", NULL);
	      break;
	    }
	}

      gtk_widget_hide(dialog);
      gtk_widget_unrealize(dialog);
      gtk_widget_destroy(dialog);
    }
}


static void
open_hisshidana_dialog(OchushaApplication *application, GtkWidget *widget,
		       OchushaBBSThread *thread, BBSThreadGUIInfo *info,
		       IconLabel *tab_label)
{
  GtkWidget *dialog
    = gtk_message_dialog_new(application->top_level,
			     GTK_DIALOG_DESTROY_WITH_PARENT,
			     GTK_MESSAGE_QUESTION,
			     GTK_BUTTONS_YES_NO,
			     _("Hisshi dana(w"));
  info->hisshidana_dialog = dialog;
  g_object_set_data(G_OBJECT(dialog), "thread", thread);

  g_signal_connect(G_OBJECT(dialog), "destroy",
		   G_CALLBACK(gtk_widget_destroyed), &info->hisshidana_dialog);
  g_signal_connect(G_OBJECT(dialog), "response",
		   G_CALLBACK(hisshidana_dialog_response_cb), application);
  gtk_widget_show(dialog);
}


void
save_thread_view_position(GtkWidget *widget, OchushaBBSThread *thread)
{
  BBSThreadView *view = BBS_THREAD_VIEW(widget);
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);
  info->backward_history
    = g_slist_prepend(info->backward_history,
		      (gpointer)bbs_thread_view_get_visible_offset(view));
  g_slist_free(info->forward_history);
  info->forward_history = NULL;
}


void
go_to_the_first_response(GtkWidget *widget, OchushaBBSThread *thread)
{
  BBSThreadView *view;

  g_return_if_fail(widget != NULL);

  view = BBS_THREAD_VIEW(widget);
  save_thread_view_position(widget, thread);
  bbs_thread_view_scroll_to_start(view);
}


void
go_to_the_last_response(GtkWidget *widget, OchushaBBSThread *thread)
{
  BBSThreadView *view;

  g_return_if_fail(widget != NULL);

  view = BBS_THREAD_VIEW(widget);
  save_thread_view_position(widget, thread);
  bbs_thread_view_scroll_to_end(view);
}


void
jump_to_bookmark(GtkWidget *widget, OchushaBBSThread *thread)
{
  BBSThreadView *view;
  BBSThreadGUIInfo *info;
  g_return_if_fail(widget != NULL && OCHUSHA_IS_BBS_THREAD(thread));

  view = BBS_THREAD_VIEW(widget);
  info = ensure_bbs_thread_info(thread);
  if (info->last_read != NULL)
    {
      save_thread_view_position(widget, thread);
      bbs_thread_view_scroll_to_mark(view, info->last_read);
    }
}
