/* Copyright 2013 Akira Ohta (akohta001@gmail.com)
    This file is part of ntch.

    The ntch is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    The ntch is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with ntch.  If not, see <http://www.gnu.org/licenses/>.
    
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <locale.h>
#include <assert.h>
#include <wchar.h>
#include <iconv.h>
#include <errno.h>

#include "env.h"
#include "utils/nt_std_t.h"
#include "utils/text.h"
#include "utils/file.h"
#include "utils/base64.h"
#include "utils/crypt.h"
#include "utils/nt_conv_char.h"
#include "net/nt_http.h"
#include "cloud/nt_cloud.h"


#define NT_CLOUD_CHK_SUM (1478428)

typedef struct tag_nt_cloud_t *nt_cloud_tp;
typedef struct tag_nt_cloud_t {
	nt_cloud_handle_t handle;
	int ref_count;
	char *usr_id;
	char *passwd;
	char *auth_url;
	char *query_url;
	char *home_url;
	char *token;
	int error_no;
}nt_cloud_t;


static nt_link_tp nt_split_line(char *data);
static int nt_get_cloud_result(const char *ptr);
static BOOL nt_cloud_edit_lines_file(nt_cloud_handle handle,
		const char *file_name, nt_link_tp lines, const char *php_file, int depth);
static BOOL nt_cloud_upload_file_local(nt_cloud_handle handle,
		const char *file_name, nt_link_tp lines, int depth);

nt_cloud_handle nt_cloud_init(
		const char *auth_url, const char *usr_id, const char *pass)
{
	nt_cloud_tp cloudp;
	
	assert(usr_id && pass);
	
	cloudp = malloc(sizeof(nt_cloud_t));
	if(!cloudp)
		return NULL;
	cloudp->handle.chk_sum = NT_CLOUD_CHK_SUM;
	cloudp->ref_count = 1;
	cloudp->usr_id = nt_str_clone(usr_id);
	if(!cloudp->usr_id){
		free(cloudp);
		return NULL;
	}
	cloudp->passwd = nt_str_clone(pass);
	if(!cloudp->passwd){
		free(cloudp->usr_id);
		free(cloudp);
		return NULL;
	}
	cloudp->auth_url = nt_str_clone(auth_url);
	if(!cloudp->auth_url){
		free(cloudp->passwd);
		free(cloudp->usr_id);
		free(cloudp);
		return NULL;
	}
	cloudp->query_url = NULL;
	cloudp->home_url = NULL;
	cloudp->token = NULL;
	cloudp->error_no = 0;
	if(!nt_cloud_chk_user(&cloudp->handle)){
		nt_cloud_release_ref(&cloudp->handle);
		return NULL;
	}
	
	return &cloudp->handle;
}

int nt_cloud_add_ref(nt_cloud_handle handle)
{
	nt_cloud_tp cloudp;
	assert(handle);
	assert(handle->chk_sum == NT_CLOUD_CHK_SUM);
	cloudp = (nt_cloud_tp)handle;
	assert(0 < cloudp->ref_count);
	return ++cloudp->ref_count;
}
int nt_cloud_release_ref(nt_cloud_handle handle)
{
	nt_cloud_tp cloudp;
	assert(handle);
	assert(handle->chk_sum == NT_CLOUD_CHK_SUM);
	cloudp = (nt_cloud_tp)handle;
	assert(0 < cloudp->ref_count);
	if(0 != --cloudp->ref_count)
		return cloudp->ref_count;
	assert(cloudp->usr_id);
	free(cloudp->usr_id);
	assert(cloudp->passwd);
	free(cloudp->passwd);
	assert(cloudp->auth_url);
	free(cloudp->auth_url);
	if(cloudp->query_url)
		free(cloudp->query_url);
	if(cloudp->home_url)
		free(cloudp->home_url);
	if(cloudp->token)
		free(cloudp->token);
	free(cloudp);
	return 0;
}

BOOL nt_cloud_chk_user(nt_cloud_handle handle)
{
	char buf[1024];
	char post_data[1024];
	nt_cloud_tp cloudp;
	nt_link_tp linkp, wrkp;
	int idx;
	
	assert(handle);
	assert(handle->chk_sum == NT_CLOUD_CHK_SUM);
	cloudp = (nt_cloud_tp)handle;
	assert(cloudp->usr_id);
	assert(cloudp->passwd);
	if(cloudp->query_url)
		free(cloudp->query_url);
	if(cloudp->home_url)
		free(cloudp->home_url);
	if(cloudp->token)
		free(cloudp->token);
	cloudp->query_url = NULL;
	cloudp->home_url = NULL;
	cloudp->token = NULL;
	
	sprintf(post_data, "user=%s&passwd=%s", 
		cloudp->usr_id, cloudp->passwd);
	
	if(!nt_http_post(cloudp->auth_url, post_data,
			buf, sizeof(buf), NULL, NULL, NULL, NULL)){
		return FALSE;
	}
	linkp = nt_split_line(buf);
	if(!linkp)
		return FALSE;
	idx = 0;
	wrkp = linkp;
	do{
		switch(idx){
		case 0:
			cloudp->error_no = atoi((const char*)wrkp->data);
			if(100 != cloudp->error_no)
				goto ERROR_TRAP;
			break;
		case 2:
			cloudp->query_url = nt_str_clone((const char*)wrkp->data);
			if(!cloudp->query_url)
				goto ERROR_TRAP;
			break;
		case 3:
			cloudp->home_url = nt_str_clone((const char*)wrkp->data);
			if(!cloudp->query_url)
				goto ERROR_TRAP;
			break;
		case 4:
			cloudp->token = nt_str_clone((const char*)wrkp->data);
			if(!cloudp->token)
				goto ERROR_TRAP;
			break;
		}
		idx++;
		wrkp = wrkp->next;
	}while(wrkp != linkp && idx < 5);
	nt_all_link_free(linkp, NULL);
	return TRUE;
ERROR_TRAP:
	if(cloudp->query_url)
		free(cloudp->query_url);
	if(cloudp->home_url)
		free(cloudp->home_url);
	if(cloudp->token)
		free(cloudp->token);
	cloudp->query_url = NULL;
	cloudp->home_url = NULL;
	cloudp->token = NULL;
	if(linkp)
		nt_all_link_free(linkp, NULL);
	return FALSE;
}

BOOL nt_cloud_insert_lines_into_file(nt_cloud_handle handle,
		const char *file_name, nt_link_tp lines)
{
	return nt_cloud_edit_lines_file(handle, file_name, lines, "nc_insert_lines_file.php", 0);
}
BOOL nt_cloud_delete_lines_from_file(nt_cloud_handle handle,
		const char *file_name, nt_link_tp lines)
{
	return nt_cloud_edit_lines_file(handle, file_name, lines, "nc_delete_lines_file.php", 0);
}

static BOOL nt_cloud_edit_lines_file(nt_cloud_handle handle,
		const char *file_name, nt_link_tp lines, const char *php_file, int depth)
{
	char *url;
	char buf1[1024*4];
	char buf2[1024];
	char *post_data;
	static const int buf_delta = 2048*2;
	nt_cloud_tp cloudp;
	nt_link_tp linkp;
	int len, num, idx;
	nt_crypt_handle h_crypt;
	iconv_t icd;
	int result_code;
	
	assert(lines);
	assert(handle);
	assert(handle->chk_sum == NT_CLOUD_CHK_SUM);
	cloudp = (nt_cloud_tp)handle;
	assert(cloudp->usr_id);
	assert(cloudp->token);
	
	num = nt_link_num(lines);
	if(num <= 0)
		return FALSE;
	len = strlen(cloudp->query_url);
	len += strlen(php_file);
	url = malloc(len+1);
	if(!url)
		return FALSE;
	strcpy(url, cloudp->query_url);
	strcat(url, php_file);
	
	post_data = malloc(buf_delta);
	if(!post_data){
		free(url);
		return FALSE;
	}
	
	h_crypt = nt_crypt_get_handle();
	if(!h_crypt){
		free(url);
		free(post_data);
		return FALSE;
	}
	icd =   iconv_open("cp932", "wchar_t");
	if(((iconv_t)-1) == icd){
		nt_crypt_release_ref(h_crypt);
		free(url);
		free(post_data);
		return FALSE;
	}
	sprintf(post_data, "user=%s&token=%s&file_name=%s&num=%d", 
		cloudp->usr_id, cloudp->token, file_name, num);
	linkp = lines;
	idx = 1;
	do{
		assert(idx <= num);
		if(nt_conv_wc2sjis(icd, (const wchar_t*)linkp->data,
					buf1, sizeof(buf1))){
			len = nt_crypt_encrypt(h_crypt,
						(const unsigned char*)buf1,strlen(buf1),
						(unsigned char*)buf2, sizeof(buf2));
			if(len != -1){
				if(nt_base64_url_encode((const unsigned char*)buf2,
						len, buf1, sizeof(buf1))){
					snprintf(buf2, sizeof(buf2), "&n%d=", idx++);
					strcat(post_data, buf2);
					strcat(post_data, buf1);
				}
			}
		}
		linkp = linkp->next;
	}while(linkp != lines);
	
	iconv_close(icd);
	nt_crypt_release_ref(h_crypt);
	
	if(!nt_http_post2(url, post_data,
			buf1, sizeof(buf1), NULL, NULL, NULL, NULL)){
		free(post_data);
		free(url);
		return FALSE;
	}

	result_code = nt_get_cloud_result(buf1);
	free(post_data);
	free(url);
	if(result_code == 301 && depth == 0){
		if(!nt_cloud_chk_user(handle))
			return FALSE;
		return nt_cloud_edit_lines_file(handle,
					file_name, lines, php_file, ++depth);
	}else if(result_code != 100){
		return FALSE;
	}
	return TRUE;
}


nt_link_tp nt_cloud_download_file(nt_cloud_handle handle,
		const char *file_name)
{
	char	*outp;
	char	*url;
	nt_cloud_tp cloudp;
	nt_link_tp linkp, wrkp;
	char buf1[1024];
	char buf2[1024];
	wchar_t wc[1024];
	wchar_t *wcptr;
	FILE	*fp_src;
	char *cptr;
	int len;
	nt_crypt_handle h_crypt;
	iconv_t icd = 0;
	
	
	assert(handle);
	assert(handle->chk_sum == NT_CLOUD_CHK_SUM);
	cloudp = (nt_cloud_tp)handle;
	assert(cloudp->home_url);
	
	len = strlen(cloudp->home_url);
	len += strlen(file_name);
	url = malloc(len+1);
	if(!url)
		return FALSE;
	strcpy(url, cloudp->home_url);
	strcat(url, file_name);
	
	if(!nt_make_sha1_path(LOG_PATH, url, &outp)){
		free(url);
		return	FALSE;
	}
	
	if(!nt_http_get(url, outp, NULL, NULL, NULL, FALSE, FALSE)){
		free(url);
		free(outp);
		return FALSE;
	}
	
	fp_src	=	fopen(outp, "r");
	if(!fp_src){
		free(url);
		free(outp);
		return FALSE;
	}
	
	h_crypt = nt_crypt_get_handle();
	if(!h_crypt){
		fclose(fp_src);
		free(url);
		free(outp);
		return FALSE;
	}
	
	icd =   iconv_open("wchar_t", "cp932");
	if(((iconv_t)-1) == icd){
		fclose(fp_src);
		free(url);
		free(outp);
		return FALSE;
	}
	
	linkp = NULL;
	while(1){
		if(feof(fp_src))
			break;
		if(!fgets(buf1, sizeof(buf1), fp_src))
			break;
		cptr = nt_trim2(buf1, buf2, sizeof(buf2));
		if(!cptr || *cptr == '\0')
			break;
		len = nt_base64_url_decode(cptr, strlen(cptr),
					(unsigned char*)buf1, sizeof(buf1));
		if(len == -1)
			continue;
		len = nt_crypt_decrypt(h_crypt,
					(const unsigned char*)buf1, len, 
					(unsigned char*)buf2, sizeof(buf2));
		if(len == -1)
			continue;
		buf2[len] = '\0';
		if(!nt_conv_sjis2wc(icd, (const char*)buf2,
					wc, sizeof(wc))){
			continue;
		}
		wcptr = nt_w_str_clone(wc);
		if(!wcptr)
			break;
		wrkp = nt_link_add_data(linkp, wcptr);
		if(!wrkp){
			free(wcptr);
			break;
		}
		if(!linkp)
			linkp = wrkp;
	}
	iconv_close(icd);
	nt_crypt_release_ref(h_crypt);
	fclose(fp_src);
	free(url);
	free(outp);
	return linkp;
}

BOOL nt_cloud_upload_file(nt_cloud_handle handle,
		const char *file_name, nt_link_tp lines)
{
	return nt_cloud_upload_file_local(handle,
			file_name, lines, 0);
}

static BOOL nt_cloud_upload_file_local(nt_cloud_handle handle,
		const char *file_name, nt_link_tp lines, int depth)
{
	char *url;
	char buf1[1024];
	char buf2[1024];
	char *post_data;
	static const int buf_delta = 2048*2;
	nt_cloud_tp cloudp;
	nt_link_tp linkp;
	int len;
	nt_crypt_handle h_crypt;
	iconv_t icd;
	int result_code;
	
	assert(handle);
	assert(handle->chk_sum == NT_CLOUD_CHK_SUM);
	cloudp = (nt_cloud_tp)handle;
	assert(cloudp->usr_id);
	assert(cloudp->token);
	
	len = strlen(cloudp->query_url);
	len += strlen("nc_write_file.php");
	url = malloc(len+1);
	if(!url)
		return FALSE;
	strcpy(url, cloudp->query_url);
	strcat(url, "nc_write_file.php");
	
	post_data = malloc(buf_delta);
	if(!post_data){
		free(url);
		return FALSE;
	}
	
	h_crypt = nt_crypt_get_handle();
	if(!h_crypt){
		free(url);
		free(post_data);
		return FALSE;
	}
	icd =   iconv_open("cp932", "wchar_t");
	if(((iconv_t)-1) == icd){
		nt_crypt_release_ref(h_crypt);
		free(url);
		free(post_data);
		return FALSE;
	}
	sprintf(post_data, "user=%s&token=%s&file_name=%s&data=", 
		cloudp->usr_id, cloudp->token, file_name);
	linkp = lines;
	do{
		if(nt_conv_wc2sjis(icd, (const wchar_t*)linkp->data,
					buf1, sizeof(buf1))){
			len = nt_crypt_encrypt(h_crypt,
						(const unsigned char*)buf1,strlen(buf1),
						(unsigned char*)buf2, sizeof(buf2));
			if(len != -1){
				if(nt_base64_url_encode((const unsigned char*)buf2,
						len, buf1, sizeof(buf1))){
					strcat(post_data, buf1);
					strcat(post_data, "\n");
				}
			}
		}
		linkp = linkp->next;
	}while(linkp != lines);
	
	iconv_close(icd);
	nt_crypt_release_ref(h_crypt);
	
	if(!nt_http_post(url, post_data,
			buf1, sizeof(buf1), NULL, NULL, NULL, NULL)){
		free(post_data);
		free(url);
		return FALSE;
	}
	
	result_code = nt_get_cloud_result(buf1);
	if(result_code == 301 && depth == 0){
		if(!nt_cloud_chk_user(handle))
			return FALSE;
		return nt_cloud_upload_file_local(handle,
					file_name, lines, ++depth);
	}else if(result_code != 100){
		free(post_data);
		free(url);
		return FALSE;
	}
	return TRUE;
}

static int nt_get_cloud_result(const char *ptr)
{
	int num;
	
	if(!ptr)
		return -1;
	if(*ptr < '0' || *ptr > '9')
		return -1;
	num = *ptr - '0';
	ptr++;
	while(*ptr){
		if(*ptr >= '0' && *ptr <= '9'){
			num *= 10;
			num += *ptr - '0';
			ptr++;
		}else{
			break;
		}
	}
	return num;
}

static nt_link_tp nt_split_line(char *data)
{
	nt_link_tp linkp, wrkp;
	char *cptr, *p;
	
	assert(data);
	
	linkp = NULL;
	cptr = data;
	do{
		wrkp = nt_link_add_data(linkp, cptr);
		if(!wrkp)
			goto ERROR_TRAP;
		if(!linkp)
			linkp = wrkp;
		p = strchr(cptr, '\n');
		if(p){
			*p = '\0';
			p++;
		}
		cptr = p;
	}while(cptr);
	
	return linkp;
ERROR_TRAP:
	if(linkp)
		nt_all_link_free(linkp, NULL);
	return NULL;
}



