/*
 * Copyright 2009-2010 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.nina;

import java.util.Collections;

import junit.framework.TestCase;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2013/10/13
 */
public class NinaMatcherTest extends TestCase {

	static void mat(NinaPattern m, String s) {
		NinaMatcher x;

		x = m.matcher(s);
		assertTrue(x.matches());
	}

	static void nmt(NinaPattern m, String s) {
		NinaMatcher x;

		x = m.matcher(s);
		assertFalse(x.matches());
	}

	static void lok(NinaPattern m, String s, String d) {
		NinaMatcher x;

		x = m.matcher(s);
		assertTrue(x.lookingAt());
		assertEquals(d, x.group());
	}

	static void fnd(NinaPattern m, String s, String d) {
		NinaMatcher x;

		x = m.matcher(s);
		assertTrue(x.find());
		assertEquals(d, x.group());
	}

	public void test0001() {
		String s =
				"#machine NFABuilder\n" +
				" =======                                @*****\n" +
				" =aaa  >-----------------a-------------->aa  *\n" +
				" ==^====                                **v***\n" +
				"   \\-------------------[a-z]--------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "a");  
		mat(m, "aaa");
		mat(m, "aaaaa");
		nmt(m, "");
		nmt(m, "aa");
		nmt(m, "aaaa");
		mat(m, "a");  
		mat(m, "aba");
		mat(m, "abada");

		lok(m, "aza", "aza");
		lok(m, "azaz", "aza");
		assertFalse(m.matcher("baza").lookingAt());

		fnd(m, "baza", "aza");
		fnd(m, "zbazaz", "aza");
		assertFalse(m.matcher("bbb").find());
	}

	public void test0002() {
		String s =
				"#machine DFABuilder\n" +
				" =======                                @*****\n" +
				" =aaa  >-----------------a-------------->aa  *\n" +
				" ==^====                                **v***\n" +
				"   \\-------------------[b-z]--------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		assertTrue(m.matcher("a").matches());  
		assertTrue(m.matcher("aba").matches());
		assertTrue(m.matcher("ababa").matches());
		assertFalse(m.matcher("").matches());
		assertFalse(m.matcher("ab").matches());
		assertFalse(m.matcher("abab").matches());
		assertTrue(m.matcher("a").matches());  
		assertTrue(m.matcher("aba").matches());
		assertTrue(m.matcher("abada").matches());

		lok(m, "aza", "aza");
		lok(m, "azaz", "aza");
		assertFalse(m.matcher("baza").lookingAt());

		fnd(m, "baza", "aza");
		fnd(m, "zbazaz", "aza");
		assertFalse(m.matcher("bbb").find());
	}

	public void test0003() {
		String s =
				"#machine NFABuilder\n" +
				" =======     *****                      @*****\n" +
				" =aaa  >----->b  >-------a-------------->aa  *\n" +
				" =======     **v**                      **^***\n" +
				"               \\---------b----------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "a");  
		mat(m, "b");
		nmt(m, "");
		nmt(m, "ab");
		nmt(m, "ba");
	}

	public void test0004() {
		String s =
				"#machine NFABuilder\n" +
				"#option enableRegexByBackquote\n" +
				" =======                                @*****\n" +
				" =aaa  >----------------`a+`------------>aa  *\n" +
				" ==^====                                **v***\n" +
				"   \\-------------------[c-z]--------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "a");
		mat(m, "aaca");
		mat(m, "aaacaa");
		nmt(m, "");
		nmt(m, "aac");
		nmt(m, "aaaac");

		lok(m, "aaza", "aaza");
		lok(m, "aazaz", "aaza");
		assertFalse(m.matcher("baza").lookingAt());

		fnd(m, "baaza", "aaza");
		fnd(m, "zbazaaz", "azaa");
		assertFalse(m.matcher("bbb").find());
	}

	public void test0005() {
		String s =
				"#machine NFABuilder\n" +
				"#option enableRegexByBackquote\n" +
				" =======                                @*****\n" +
				" =aaa  >-------------`(ab)+`------------>aa  *\n" +
				" ==^====                                **v***\n" +
				"   \\-------------------[c-z]--------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "ab");
		mat(m, "ababcab");
		mat(m, "abababcabab");
		nmt(m, "");
		nmt(m, "ababc");
		nmt(m, "ababababc");
		nmt(m, "abababacab");
	}

	public void test0006() {
		String s =
				"#machine DFABuilder\n" +
				"   /a------------------\\  \n" +
				" ==^====             @*v***\n" +
				" =aaa  =             *aa  *\n" +
				" ==^====             **v***\n" +
				"   \\------------------b/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "aba");
	}

	public void test0011() {
		String s =
				"#machine NFABuilder\n" +
				" =======                                @*****\n" +
				" =aaa  >----------------'\\n'------------>aa  *\n" +
				" ==^====                                **v***\n" +
				"   \\-------------------'\\n'---------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "\n");
		mat(m, "\n\n\n");
		mat(m, "\n\n\n\n\n");
		nmt(m, "");
		nmt(m, "\n\n");
		nmt(m, "\n\n\n\n");
	}

	public void test0012() {
		String s =
				"#machine NFABuilder\n" +
				" =======                                @*****\n" +
				" =aaa  >----------------'\\r'------------>aa  *\n" +
				" ==^====                                **v***\n" +
				"   \\-------------------'\\r'---------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "\r");
		mat(m, "\r\r\r");
		mat(m, "\r\r\r\r\r");
		nmt(m, "");
		nmt(m, "\r\r");
		nmt(m, "\r\r\r\r");
	}

	public void test0013() {
		String s =
				"#machine NFABuilder\n" +
				" =======                                @*****\n" +
				" =aaa  >----------------'\\t'------------>aa  *\n" +
				" ==^====                                **v***\n" +
				"   \\-------------------'\\t'---------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "\t");
		mat(m, "\t\t\t");
		mat(m, "\t\t\t\t\t");
		nmt(m, "");
		nmt(m, "\t\t");
		nmt(m, "\t\t\t\t");
	}

	public void test0014() {
		String s =
				"#machine NFABuilder\n" +
				" =======                                @*****\n" +
				" =aaa  >----------------'\\n\\r\\t'-------->aa  *\n" +
				" ==^====                                **v***\n" +
				"   \\-------------------'\\n\\r\\t'-----------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "\n\r\t");
		mat(m, "\n\r\t\n\r\t\n\r\t");
		mat(m, "\n\r\t\n\r\t\n\r\t\n\r\t\n\r\t");
		nmt(m, "");
		nmt(m, "\n\r\t\n\r\t");
		nmt(m, "\n\r\t\n\r\t\n\r\t\n\r\t");
	}

	public void test0015() {
		String s =
				"#machine DFABuilder\n" +
				"   /[\\d]---------------\\  \n" +
				" ==^====             @*v***\n" +
				" =aaa  =             *aa  *\n" +
				" =v^====             **v***\n" +
				"  |\\---------------[\\D]/  \n" +
				"  |     ****  \n" +
				"  \\[\\D]->  *  \n" +
				"        ****  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "0a1a2a3b4c5d6e7f8g9");
	}

	public void test0016() {
		String s =
				"#machine DFABuilder\n" +
				"   /[\\w]---------------\\  \n" +
				" ==^====             @*v***\n" +
				" =aaa  =             *aa  *\n" +
				" =v^====             **v***\n" +
				"  |\\---------------[\\W]/  \n" +
				"  |     ****  \n" +
				"  \\[\\W]->  *  \n" +
				"        ****  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "A,B,C,D,E,F,G,H,I,J,K,L,M,N,O,P,Q,R,S,T,U,V,W,X,Y,Z");
		mat(m, "a,b,c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,x,y,z");
		mat(m, "0,1,2,3,4,5,6,7,8,9,_");
	}

	public void test0017() {
		String s =
				"#machine DFABuilder\n" +
				"   /[\\s]---------------\\  \n" +
				" ==^====             @*v***\n" +
				" =aaa  =             *aa  *\n" +
				" =v^====             **v***\n" +
				"  |\\---------------[\\S]/  \n" +
				"  |     ****  \n" +
				"  \\[\\S]->  *  \n" +
				"        ****  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
//		mat(m, "\t:\r:\n:\u000b:\f:\r: ");
		mat(m, "\t:\n: ");
	}

	public void test0018() {
		String s =
				"#machine DFABuilder\n" +
				"   /[\\p{Space}]--------\\  \n" +
				" ==^====             @*v***\n" +
				" =aaa  =             *aa  *\n" +
				" =v^====             **v***\n" +
				"  |\\---------------[\\S]/  \n" +
				"  |     ****  \n" +
				"  \\[\\S]->  *  \n" +
				"        ****  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "\t:\r:\n:\u000b:\f:\r: ");
	}

	public void test0019() {
		String s =
				"#machine NFABuilder\n" +
				" =======                                @*****\n" +
				" =aaa  >----------------'\\0'------------>aa  *\n" +
				" ==^====                                **v***\n" +
				"   \\-------------------'\\0'---------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "\0");
		mat(m, "\0\0\0");
		mat(m, "\0\0\0\0\0");
		nmt(m, "");
		nmt(m, "\0\0");
		nmt(m, "\0\0\0\0");
	}

	public void test0021() {
		String s =
				"#machine DFABuilder\n" +
				" =====                      @*****\n" +
				" =b  >-------'/'------------>aa  *\n" +
				" ==v==                      **^***\n" +
				"   \\---------b----------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "/");  
		mat(m, "b");
		nmt(m, "");
		nmt(m, "/b");
		nmt(m, "ba");
	}

	public void test0031() {
		String s =
				"#machine NFABuilder\n" +
				" =======                                @*****\n" +
				" =aaa  >----------------'{1}'----------->aa  *\n" +
				" ==^====                                **v***\n" +
				"   \\-------------------'{2}'--------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "{1}");
		mat(m, "{1}{2}{1}");
		mat(m, "{1}{2}{1}{2}{1}");
		nmt(m, "");
		nmt(m, "{1}{2}");
		nmt(m, "{1}{2}{1}{2}");
	}

	public void test0032() {
		String s =
				"#machine NFABuilder\n" +
				" =======                                @*****\n" +
				" =aaa  >----------------'$1'------------>aa  *\n" +
				" ==^====                                **v***\n" +
				"   \\-------------------'$2'---------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "$1");
		mat(m, "$1$2$1");
		mat(m, "$1$2$1$2$1");
		nmt(m, "");
		nmt(m, "$1$2");
		nmt(m, "$1$2$1$2");
	}

	public void test0033() {
		String s =
				"#machine NFABuilder\n" +
				" =======                                @*****\n" +
				" =aaa  >----------------'${1&}'--------->aa  *\n" +
				" ==^====                                **v***\n" +
				"   \\-------------------'${2&}'------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "${1&}");
		mat(m, "${1&}${2&}${1&}");
		mat(m, "${1&}${2&}${1&}${2&}${1&}");
		nmt(m, "");
		nmt(m, "${1&}${2&}");
		nmt(m, "${1&}${2&}${1&}${2&}");
	}

	public void test0041() {
		String s =
				"#machine DFABuilder\n" +
				" =======                                @*****\n" +
				" =aaa  >----------------'abc'----------->aa  *\n" +
				" ==^====                                **v***\n" +
				"   \\-------------------'efg'--------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "abc");
		mat(m, "abcefgabc");
		mat(m, "abcefgabcefgabc");
		nmt(m, "");
		nmt(m, "abcefg");
		nmt(m, "abcefgabcefg");
		nmt(m, "abcefgab");
		nmt(m, "abcefabc");
		nmt(m, "acefgabc");
	}

	public void test0042() {
		String s =
				"#machine DFABuilder\n" +
				" =======                              @*****\n" +
				" =aaa  >----------------'a'----------->aa  *\n" +
				" ==^====                              **v***\n" +
				"   \\-------------------'e'--------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "a");
		mat(m, "aea");
		mat(m, "aeaea");
		nmt(m, "");
		nmt(m, "ae");
		nmt(m, "aeae");
	}

	public void test0062() {
		String s =
				"#machine DFABuilder\n" +
				" =======                              @*****\n" +
				" =aaa  >-----------.------------------>aa  *\n" +
				" ==^====                              **v***\n" +
				"   \\--------------.---------------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "a");
		mat(m, "xyz");
		mat(m, "aaaaa");
		nmt(m, "");
		nmt(m, "az");
		nmt(m, "aaaa");
	}

	public void test0061() {
		String s =
				"#machine DFABuilder\n" +
				" =======                      @*****\n" +
				" =b    >-------'z'------------>aa  *\n" +
				" =v===^=                      **^***\n" +
				"  \\-.-/\n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "z");  
		mat(m, "aaaaaaaz");
		nmt(m, "");
		nmt(m, "za");
	}

	public void test0071() {
		String s =
				"#machine DFABuilder\n" +
				"   +-a-----------------+   \n" +
				" ==^====             @*v***\n" +
				" =aaa  =             *aa  *\n" +
				" ==^====             **v***\n" +
				"   +-----------------b-+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "aba");
	}

	public void test0072() {
		String s =
				"#machine NFABuilder\n" +
				"        +-.-+    +-.-+    +-.-+    +-.-+ \n" +
				" ===   *v*  |   *v*  |   *v*  |   *v*  |   @@@\n" +
				" = >-a->A>--+-o->O>--+-i->I>--+-e->E>--+-u-> @\n" +
				" ===   ***      ***      ***  |   ***      @@@\n" +
				"                              |    +-.-+\n" +
				"                              |   *v*  |   @@@\n" +
				"                              +-u->U>--+-e-> @\n" +
				"                                  ***      @@@\n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "aokiume");
	}

	public void test0073() {
		String s =
				"#machine DFABuilder\n" +
				"   +-c------------------+  \n" +
				"   +-a-----------------+|  \n" +
				" ==^====             @*vv**\n" +
				" =aaa  =             *aa  *\n" +
				" ==^====             **v***\n" +
				"   +-----------------b-+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "abc");
	}

	public void test0074() {
		String s =
				"#machine NFABuilder\n" +
				"   +-a-----------------+   \n" +
				" ==^====             @*v***\n" +
				" =aaa  =             *aa  *\n" +
				" ==^====             **v***\n" +
				"   +-----------------.-+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "aba");
	}

	public void test0081() {
		String s =
				"#machine DFABuilder\n" +
				"   +-a-----------------+   \n" +
				" ==^=========        @*v************\n" +
				" =a = a > 1;=        *if(b < 1)  1;*\n" +
				" ==^=========        **v************\n" +
				"   +-----------------b-+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "aba");
	}

	public void test0082() {
		String s =
				"#machine DFABuilder\n" +
				"   +-a-----------------+   \n" +
				" @@^@@@@@@@@@        ==v============\n" +
				" @a = a > 1;@        =if(b < 1)  1;=\n" +
				" @@^@@@@@@@@@        ==v============\n" +
				"   +-----------------b-+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "bab");
	}

	public void test0083() {
		String s =
				"#machine DFABuilder\n" +
				" @@@@@@@@@@@@\n" +
				" @a = a > 1;@\n" +
				" @@^@v@@@@@@@\n" +
				"   | |       \n" +
				"   b a       \n" +
				"   | |       \n" +
				" ==^=v=======\n" +
				" =a = a > 1;=\n" +
				" ============\n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "bab");
	}

	public void test0084() {
		String s =
				"#machine DFABuilder\n" +
				"   +-a-----------------+   \n" +
				" @@^@@@@@@@@@        ==v============\n" +
				" @a = a % 1;@        =if(b % 1)  1;=\n" +
				" @@^@@@@@@@@@        ==v============\n" +
				"   +-----------------v-+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "vav");
	}

	public void test0085() {
		String s =
				"#machine DFABuilder\n" +
				" @@@@@@@@@@@@\n" +
				" @a = a % 1;@\n" +
				" @@^@v@@@@@@@\n" +
				"   | |       \n" +
				"   v a       \n" +
				"   | |       \n" +
				" ==^=v=======\n" +
				" =a = a % 1;=\n" +
				" ============\n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "vav");
	}

	public void test0091() {
		String s =
				"#machine DFABuilder\n" +
				"   +-a--(1)      (1)---+   \n" +
				" ==^=========        @*v************\n" +
				" =a = a > 1;=        *if(b < 1)  1;*\n" +
				" ==^=========        **v************\n" +
				"   +----(2)      (2)-b-+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "aba");
	}

	public void test0092() {
		String s =
				"#machine DFABuilder\n" +
				"   +-a--(1)            +---(1)   \n" +
				" ==^=========        @*v************\n" +
				" =a = a > 1;=        *if(b < 1)  1;*\n" +
				" ====^=======        *v*************\n" +
				" (2)-+                +-';'-(2)  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "a;a");
	}

	public void test0093() {
		String s =
				"#machine DFABuilder\n" +
				"   +-'\\''-(2)          +-'\\''--(1)   \n" +
				" ==^=========        @*^************\n" +
				" =a = a > 1;=        *if(b < 1)  1;*\n" +
				" ====^=======        *^*************\n" +
				" (1)-+            (2)-+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "'''");
	}

	public void test0094() {
		String s =
				"#machine DFABuilder\n" +
				"   +-'\\''-(2)          +-'\\''--(1)   \n" +
				" ==^=========        @*^************\n" +
				" =a = a > 1;=        *if(b < 1)  1;*\n" +
				" ====^=======        *^*************\n" +
				" (1)-+                +-(2)  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "'''");
	}

	public void test0101() {
		String s =
				"#machine NFABuilder\n" +
				" *S*****                                @*****\n" +
				" *aaa  >-----------------a-------------->aa  *\n" +
				" **^****                                **v***\n" +
				"   \\-------------------[a-z]--------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "a");  
		mat(m, "aaa");
		mat(m, "aaaaa");
	}

	public void test0102() {
		String s =
				"#machine DFABuilder\n" +
				"   +-a--(E)                \n" +
				" *S^*********        @E@@@@@@@@@@@@@\n" +
				" *a = a > 1;*        @if(b < 1)  1;@\n" +
				" ************        @@v@@@@@@@@@@@@\n" +
				"                 (S)-b-+  \n" +
				"                       |  \n" +
				"                 (S)-c-+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "aba");
		mat(m, "abaca");
	}

	public void test0103() {
		String s =
				"#machine DFABuilder\n" +
				"   +-a--(E)                \n" +
				" *S^*********        @E@@@@@@@@@@@@@\n" +
				" *a = a > 1;*        @if(b < 1)  1;@\n" +
				" ************        @@v@@@@@@@@@@@@\n" +
				"                       +-b-(S)  \n" +
				"                       |  \n" +
				"                       +-c-(S)  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "aba");
		mat(m, "abaca");
	}

	public void test0111() {
		String s =
				"#machine NFABuilder\n" +
				" =======     *****                      @*****\n" +
				" =aaa  >----->b  >-------97------------->aa  *\n" +
				" =======     **^**                      **v***\n" +
				"               +---------98---------------+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "aba");  
	}

	public void test0112() {
		String s =
				"#machine NFABuilder\n" +
				" =======     *****                      @*****\n" +
				" =aaa  >----->b  >-------0x61----------->aa  *\n" +
				" =======     **^**                      **v***\n" +
				"               +---------0x62-------------+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "aba");  
	}

	public void test0113() {
		String s =
				"#machine NFABuilder\n" +
				" =======     *****                      @*****\n" +
				" =aaa  >----->b  >-------0141----------->aa  *\n" +
				" =======     **^**                      **v***\n" +
				"               +---------0142-------------+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "aba");  
	}

	public void test0121() {
		String s =
				"#machine NFABuilder\n" +
				" =======                                @*****\n" +
				" =aaa  >----------------[\\[\\]\\&]-------->aa  *\n" +
				" ==^====                                **v***\n" +
				"   \\-------------------[\\[\\]\\&]-----------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "&");  
		mat(m, "&]&");
		mat(m, "&[&]&");
	}

	public void test0122() {
		String s =
				"#machine NFABuilder\n" +
				" =======                                @*****\n" +
				" =aaa  >----------------[a-z&[^f]]------>aa  *\n" +
				" ==^====                                **v***\n" +
				"   \\-------------------[a-z&[^f]]---------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "a");  
		mat(m, "abc");
		mat(m, "abcde");
		nmt(m, "f");
		nmt(m, "abf");
	}

	public void test0131() {
		String s =
				"#machine NFABuilder\n" +
				" =======                     \n" +
				" =aaa  >---'a'---*---'b'--@  \n" +
				" ==^====                  |  \n" +
				"   +-------'c'------------+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "ab");
		mat(m, "abcab");
		mat(m, "abcabcab");
	}

	public void test0132() {
		String s =
				"#machine NFABuilder\n" +
				"  +--------'d'---+           \n" +
				" =v=====         |           \n" +
				" =aaa  >---'a'---*---'b'--@  \n" +
				" ==^====                  |  \n" +
				"   +-------'c'------------+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "ab");
		mat(m, "adab");
		mat(m, "abcadab");
	}

	public void test0133() {
		String s =
				"#machine DFABuilder\n" +
				" =======                     \n" +
				" =aaa  >---'a'---*---'b'--@  \n" +
				" ==^====                  |  \n" +
				"   +-------'c'------------+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "ab");
		mat(m, "abcab");
		mat(m, "abcabcab");
	}

	public void test0141() {
		String s =
				"#machine NFABuilder\n" +
				"#label lb1='aa'\n" +
				"#label lb2='bb'\n" +
				" =======     *****                      @*****\n" +
				" =aaa  >----->b  >-------${lb1}--------->aa  *\n" +
				" =======     **^**                      **v***\n" +
				"               +---------${lb2}-----------+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "aabbaa");  
	}

//	public void test0142() {
//		String s =
//				"#machine DFABuilder\n" +
//				"#label lb1=\"aa\"\n" +
//				"#label lb2=\"bb\"\n" +
//				" *S***                      @*****\n" +
//				" *b  >-------${lb1}--------->aa  *\n" +
//				" **^**                      **v***\n" +
//				"   +---------${lb2}-----------+  \n" +
//				"";
//		NinaPattern m;
//
//		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
//				new NinaSubautomata("a"),
//				new NinaConsoleStub());
//		mat(m, "aabbaa");  
//	}

	public void test0143() {
		String s =
				"#machine DFABuilder\n" +
				"#label lb1=[a-z]\n" +
				"#label lb2=[A-Z]\n" +
				" *S***                      @*****\n" +
				" *b  >-------${lb1}--------->aa  *\n" +
				" **^**                      **v***\n" +
				"   +---------${lb2}-----------+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "aBc");  
	}

	public void test0144() {
		String s =
				"#machine NFABuilder\n" +
				"#option enableRegexByBackquote\n" +
				"#label lb1=`a+`\n" +
				"#label lb2=`b+`\n" +
				" =======     *****                      @*****\n" +
				" =aaa  >----->b  >-------${lb1}--------->aa  *\n" +
				" =======     **^**                      **v***\n" +
				"               +---------${lb2}-----------+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "abbbaaa");  
	}

	public void test0145() {
		String s =
				"#machine DFABuilder\n" +
				"#label lb1=${lb3}\n" +
				"#label lb2='bb'\n" +
				"#label lb3='aa'\n" +
				" *S***                      @*****\n" +
				" *b  >-------${lb1}--------->aa  *\n" +
				" **^**                      **v***\n" +
				"   +---------${lb2}-----------+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "aabbaa");  
	}

	public void test0151() {
		String s =
				"#machine DFABuilder\n" +
				"#define LB1 97\n" +
				"#define LB2	98\n" +
				" *S***                      @*****\n" +
				" *b  >-------${LB1}--------->aa  *\n" +
				" **^**                      **v***\n" +
				"   +---------${LB2}-----------+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "aba");  
	}

	public void test0152() {
		String s =
				"#machine DFABuilder\n" +
				"#define LB1 0x61\n" +
				"#define LB2 0x62\n" +
				" *S***                      @*****\n" +
				" *b  >-------${LB1}--------->aa  *\n" +
				" **^**                      **v***\n" +
				"   +---------${LB2}-----------+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "aba");  
	}

	public void test0153() {
		String s =
				"#machine DFABuilder\n" +
				"#define LB1 0141\n" +
				"#define LB2 0142\n" +
				" *S***                      @*****\n" +
				" *b  >-------${LB1}--------->aa  *\n" +
				" **^**                      **v***\n" +
				"   +---------${LB2}-----------+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "aba");  
	}

	public void test0154() {
		String s =
				"#machine DFABuilder\n" +
				"#define LB1 LB3\n" +
				"#define LB2 98\n" +
				"#define LB3 97\n" +
				" *S***                      @*****\n" +
				" *b  >-------${LB1}--------->aa  *\n" +
				" **^**                      **v***\n" +
				"   +---------${LB2}-----------+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "aba");  
	}

	public void test0161() {
		String s =
				"#machine DFABuilder\n" +
				" *S*** 		             @*****\n" +
				" *b  >-------'a'------------>aa  *\n" +
				" **^**                      **v***\n" +
				"   +---------'b'--------------+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "aba");  
	}

	public void test0162() {
		String s =
				"#machine DFABuilder\n" +
				" *S*** 	             @*****\n" +
				" *b  >-------'a'------------>aa  *\n" +
				" **^**                      **v***\n" +
				"   +---------'b'--------------+  \n" +
				"";
		NinaPattern m;

		Quadro.setTab(8);
		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "aba");  
	}

	public void test0171() {
		String s =
				"#machine NFA\n" +
				" =======                                @*****\n" +
				" =aaa  >-----------------a-------------->aa  *\n" +
				" ==^====                                **v***\n" +
				"   \\-------------------[a-z]--------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "a");  
		mat(m, "aaa");
		mat(m, "aaaaa");
	}

	public void test0172() {
		String s =
				"#machine full\n" +
				" =======                                @*****\n" +
				" =aaa  >-----------------a-------------->aa  *\n" +
				" ==^====                                **v***\n" +
				"   \\-------------------[a-z]--------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "a");  
		mat(m, "aaa");
		mat(m, "aaaaa");
	}

	public void test0181() {
		String s =
				"#machine full\n" +
				" =======       \n" +
				" =aaa  >-a-(A) \n" +
				" =     =       \n" +
				" =     >-b-(A) \n" +
				" =======       \n" +
				" \n" +
				" @A@@@@@       \n" +
				" @aaa  @       \n" +
				" @@@@@@@       \n" +
				" ";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "a");  
		mat(m, "b");
	}

	public void test0182() {
		String s =
				"#machine full\n" +
				" =======       \n" +
				" =aaa  >-a-(A) \n" +
				" =     =       \n" +
				" =     >-b-(A) \n" +
				" =======       \n" +
				" \n" +
				" @A@@@@@       \n" +
				" @aaa  >-a-(S) \n" +
				" @@@@@@@       \n" +
				" ";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "aaa");  
		mat(m, "bab");
	}

	public void test0191() {
		String s =
				"#machine DFABuilder\n" +
				" *S***                      @*****\n" +
				" *b  >-------~a~------------>aa  *\n" +
				" **^**                      **v***\n" +
				"   +---------~b~--------------+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "a");
		mat(m, "aba");
		mat(m, " a b a ");
		mat(m, "  a  b  a  ");
		mat(m, "a  b  a");
		mat(m, "ab  a");
	}

	public void test0201() {
		String s =
				"#machine NFABuilder\n" +
				" =======                                @*****\n" +
				" =aaa  >-------------/(ab)+/------------>aa  *\n" +
				" ==^====                                **v***\n" +
				"   \\-----------------/[c-z]/--------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "ab");
		mat(m, "ababcab");
		mat(m, "abababcabab");
		nmt(m, "");
		nmt(m, "ababc");
		nmt(m, "ababababc");
		nmt(m, "abababacab");
	}

	public void test0211() {
		String s =
				"#machine NFABuilder\n" +
				"#option disableTurnBySlash\n" +
				" =======                                @*****\n" +
				" =aaa  >-------------/ +/--------------->aa  *\n" +
				" ==^====                                **v***\n" +
				"   +----------------/[c-z]/---------------+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "  ");
		mat(m, "    c  ");
		mat(m, "      c   ");
		nmt(m, "");
		nmt(m, "    c");
	}

	public void test0212() {
		String s =
				"#machine NFABuilder\n" +
				"#option disableTurnBySlash\n" +
				" =======                                @*****\n" +
				" =aaa  >-------------/[a-z]+/----------->aa  *\n" +
				" ==^====                                **v***\n" +
				"   +----------------/ /-------------------+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "aaaa");
		mat(m, "aaaa aaa");
		mat(m, "aaa aaa aaa");
		nmt(m, "");
		nmt(m, "aaaa ");
		nmt(m, "Aaa aAa aaA");
	}

	public void test0213() {
		String s =
				"#machine NFABuilder\n" +
				"#option disableTurnBySlash\n" +
				" =======                                @*****\n" +
				" =aaa  >------------/(?i)[a-z]+/-------->aa  *\n" +
				" ==^====                                **v***\n" +
				"   +----------------/ /-------------------+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "aBCa");
		mat(m, "aaCa aAa");
		mat(m, "Aaa aAa aaA");
		nmt(m, "");
		nmt(m, "aaaa ");
	}

	public void test0214() {
		String s =
				"#machine NFABuilder\n" +
				"#option disableTurnBySlash\n" +
				" =======                                @*****\n" +
				" =aaa  >------------/\\/+/--------------->aa  *\n" +
				" ==^====                                **v***\n" +
				"   +----------------/ /-------------------+  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern("A", s, Collections.<String, String>emptyMap(),
				new NinaSubautomata("a"),
				new NinaConsoleStub());
		mat(m, "////");
		mat(m, "//// ///");
		mat(m, "/// /// ///");
		nmt(m, "");
		nmt(m, "//// ");
	}

}
