/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk.nano.expr;

import java.math.BigInteger;

import net.morilib.awk.nano.io.AwkFiles;
import net.morilib.awk.nano.namespace.AwkNamespace;
import net.morilib.awk.nano.value.AwkUndefined;
import net.morilib.awk.nano.value.AwkValue;

/**
 * フィールド参照を表す中間表現です。
 * 
 * @author MORIGUCHI, Yuichiro 2013/03
 */
public class AwkReferField extends AwkExpression {

	private BigInteger MI = BigInteger.valueOf(Integer.MAX_VALUE);

	private AwkExpression field;

	/**
	 * 中間表現を生成します。
	 * 
	 * @param field フィールド番号
	 */
	public AwkReferField(AwkExpression field) {
		this.field = field;
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.expr.AwkExpression#eval(net.morilib.awk.namespace.AwkNamespace, net.morilib.awk.io.AwkFiles)
	 */
	@Override
	public AwkValue eval(AwkNamespace ns, AwkFiles o) {
		BigInteger b = field.eval(ns, o).toInteger();

		if(b.compareTo(MI) > 0 && b.signum() < 0) {
			return AwkUndefined.UNDEF;
		}
		return ns.referField(b.intValue());
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.expr.AwkExpression#assign(net.morilib.awk.namespace.AwkNamespace, net.morilib.awk.value.AwkValue, net.morilib.awk.io.AwkFiles)
	 */
	@Override
	public void assign(AwkNamespace ns, AwkValue v, AwkFiles f) {
		BigInteger b = field.eval(ns, f).toInteger();

		if(b.compareTo(MI) <= 0 && b.signum() >= 0) {
			ns.setField(b.intValue(), v.toString(ns));
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.expr.AwkExpression#isLvalue()
	 */
	@Override
	public boolean isLvalue() {
		return true;
	}

}
