/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp.nano;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.UnsupportedEncodingException;

import net.morilib.lisp.nano.util.PeekableReader;

/**
 * 
 *
 *
 * @author MORIGUCHI, Yuichiro 2009
 */
public class InputPort extends Datum {

	//
	private PeekableReader reader;
	private boolean standard;
	private boolean closed = false;
	private LispMessage msg;

	//
	private InputPort(Reader rd, boolean std, LispMessage msg) {
		this.reader     = new PeekableReader(rd);
		this.standard   = std;
		this.msg        = msg;
	}

	/**
	 * 
	 * @param rd
	 * @param msg
	 */
	public InputPort(Reader rd, LispMessage msg) {
		this(rd, false, msg);
	}

	/**
	 * 
	 * @param fname
	 * @param msg
	 */
	public InputPort(File fname, LispMessage msg) {
		try {
			Reader rd;

			rd = new InputStreamReader(new FileInputStream(fname));
			this.reader     = new PeekableReader(rd);
			this.standard   = false;
			this.msg        = msg;
		} catch (FileNotFoundException e) {
			throw new LispIOException(e);
		}

	}

	/**
	 * 
	 * @param fname
	 * @param msg
	 * @throws UnsupportedEncodingException 
	 */
	public InputPort(String fname, String enconding,
			LispMessage msg) throws UnsupportedEncodingException {
		try {
			Reader rd;

			rd = new InputStreamReader(new FileInputStream(fname),
					enconding);
			this.reader     = new PeekableReader(rd);
			this.standard   = false;
			this.msg        = msg;
		} catch (FileNotFoundException e) {
			throw new LispIOException(e);
		}

	}

	/**
	 * 
	 * @param msg
	 * @return
	 */
	public static InputPort getStandard(LispMessage msg) {
		return new InputPort(PeekableReader.in, true, msg);
	}

	/**
	 * 
	 * @return
	 */
	public Datum readChar() {
		int c;

		try {
			if(closed) {
				throw msg.getError("err.port.closed");
			} else if(reader.peek() < 0) {
				return EOFObject.EOF;
			} else if((c = reader.read()) < 0) {
				return EOFObject.EOF;
			} else {
				return new LispCharacter((char)c);
			}
		} catch (IOException e) {
			throw new LispIOException(e);
		}
	}

	/**
	 * 
	 * @return
	 */
	public Datum peekChar() {
		int c;

		try {
			if(closed) {
				throw msg.getError("err.port.closed");
			} else if((c = reader.peek()) < 0) {
				return EOFObject.EOF;
			}
			return new LispCharacter((char)c);
		} catch(IOException e) {
			throw new LispIOException(e);
		}
	}

	/**
	 * 
	 */
	public Datum readS() {
		try {
			if(closed) {
				throw msg.getError("err.port.closed");
			} else if(reader.peek() < 0) {
				return EOFObject.EOF;
			}
			return NanoParser.read(reader);
		} catch(IOException e) {
			throw new LispIOException(e);
		}
	}

	/**
	 * 
	 */
	public void close() {
		try {
			if(!closed && !standard) {
				reader.close();
				closed = true;
			}
		} catch (IOException e) {
			throw new LispIOException(e);
		}
	}

	/**
	 * 
	 * @return
	 */
	public boolean isStandard() {
		return standard;
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lisp.Datum#isTypePort()
	 */
	public boolean isTypePort() {
		return true;
	}

	/**
	 * 
	 * @return
	 */
	public boolean ready() {
		try {
			if(closed) {
				throw msg.getError("err.port.closed");
			} else if(reader.peek() < 0) {
				return false;
			} else {
				try {
					return reader.ready();
				} catch (IOException e) {
					throw new LispIOException(e);
				}
			}
		} catch(IOException e) {
			throw new LispIOException(e);
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.nano.Datum#toDisplay()
	 */
	@Override
	public String toDisplay() {
		return "#<iport>";
	}

}
