/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk.nano.io;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;

/**
 * awkiumで使用する仮想ファイルを管理します。<br />
 * awkiumには物理的なファイル名のほかに標準出力をあらわす
 * /dev/stdoutなどの仮想ファイル名があります。
 *
 * @author MORIGUCHI, Yuichiro 2013/04/02
 */
public final class AwkVirtualFile {

	static final String STDIN  = "/dev/stdin";
	static final String STDIN2 = "/dev/fd/0";

	static final String STDOUT = "/dev/stdout";
	static final String STDERR = "/dev/stderr";
	static final String STDOT2 = "/dev/fd/1";
	static final String STDER2 = "/dev/fd/2";
	static final String NULL   = "/dev/null";

	/**
	 * デバッグ用の仮想ファイル名です。
	 */
	public static final String DEBUG  = "/dev/debug";

	static final LineWriter NULLWRITER = new LineWriter() {

		public void print(String s) {
		}

		public void println(String s) {
		}

		public void printf(String format, Object... args) {
		}

		public void flush() {
		}

		public void close() throws IOException {
		}

	};

	/**
	 * デバッグ用のバイナリIOです。
	 */
	public static final ByteArrayOutputStream DEBUG_BIN_IO =
		new ByteArrayOutputStream();

	/**
	 * 書き込み用の仮想ファイルを取得します。
	 * 
	 * @param fname ファイル名
	 * @param fs ファイル管理クラス
	 */
	public static LineWriter out(String fname, AwkFiles fs) {
		if(fname.equals(STDOUT) || fname.equals(STDOT2)) {
			return fs.getStdout();
		} else if(fname.equals(STDERR) || fname.equals(STDER2)) {
			return fs.getStderr();
		} else if(fname.equals(NULL)) {
			return NULLWRITER;
		} else {
			return null;
		}
	}

	/**
	 * 読み込み用の仮想ファイルを取得します。
	 * 
	 * @param fname ファイル名
	 * @param fs ファイル管理クラス
	 */
	public static RecordReader in(String fname, AwkFiles fs) {
		if(fname.equals(STDIN) || fname.equals(STDIN2)) {
			return fs.getStdin();
		} else {
			return null;
		}
	}

	/**
	 * 書き込み用の仮想バイナリファイルを取得します。
	 * 
	 * @param fname ファイル名
	 * @param fs ファイル管理クラス
	 */
	public static OutputStream outBinary(String fname, AwkFiles fs) {
		if(fname.equals(DEBUG)) {
			return DEBUG_BIN_IO;
		} else {
			return null;
		}
	}

}
