/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk.nano.value;

import java.math.BigInteger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import net.morilib.awk.nano.namespace.AwkNamespace;

/**
 * awkium文字列のクラスです。
 * 
 * @author MORIGUCHI, Yuichiro 2013/03
 */
public class AwkString extends AwkValue {

	/**
	 * 空文字列です。
	 */
	public static final AwkString EMPTY = new AwkString("");

	private static final Pattern PFL = Pattern.compile(
			"[-+]?([0-9]+(\\.[0-9]+)?|\\.[0-9]+)([eE][-+]?[0-9]+)?");
	private static final Pattern PIT = Pattern.compile("[-+]?[0-9]+");
	private static final Pattern PDC = Pattern.compile(
			"[-+]?([0-9]*\\.[0-9]+)([eE][-+]?[0-9]+)?");
	private static final Pattern PFI = Pattern.compile(
			"([-+]?([0-9]+(\\.[0-9]+)?|\\.[0-9]+)([eE][-+]?[0-9]+)?)" +
			"([-+]([0-9]+(\\.[0-9]+)?|\\.[0-9]+)([eE][-+]?[0-9]+)?)" +
			"[ij]");

	//
	private String value;

	private AwkString(String s) {
		value = s;
	}

	/**
	 * 文字列を取得します。
	 * 
	 * @param s 文字列
	 * @return awkium文字列
	 */
	public static AwkString valueOf(String s) {
		return new AwkString(s);
	}

	public double toFloat() {
		Matcher m = PFL.matcher(value);

		if(m.lookingAt()) {
			return Double.parseDouble(
					m.group().replaceFirst("^\\+", ""));
		} else {
			return 0.0;
		}
	}

	public BigInteger toInteger() {
		Matcher m = PIT.matcher(value);

		if(m.lookingAt()) {
			return new BigInteger(m.group().replaceFirst("^\\+", ""));
		} else {
			return BigInteger.ZERO;
		}
	}

	public Number toReal() {
		Matcher m;

		if((m = PIT.matcher(value)).lookingAt()) {
			return new BigInteger(m.group().replaceFirst("^\\+", ""));
		} else {
			return toFloat();
		}
	}

	public double toImaginary() {
		Matcher m;

		if((m = PFI.matcher(value)).lookingAt()) {
			return Double.parseDouble(
					m.group(5).replaceFirst("^\\+", ""));
		} else {
			return 0.0;
		}
	}

	public boolean toBoolean(AwkNamespace ns) {
		return !value.equals("");
	}

	public AwkValue toNumber() {
		Matcher m;

		if((m = PFL.matcher(value)).lookingAt()) {
			if(PIT.matcher(m.group()).matches()) {
				return AwkInteger.valueOf(
						new BigInteger(
								m.group().replaceFirst("^\\+", "")));
			} else {
				return AwkFloat.valueOf(
						Double.parseDouble(
								m.group().replaceFirst("^\\+", "")));
			}
		} else {
			return AwkInteger.ZERO;
		}
	}

	public Pattern toRegex() {
		return Pattern.compile(value);
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.value.AwkValue#toObject()
	 */
	public Object toObject() {
		return value;
	}

	public boolean isFloat() {
		return false;
	}

	public boolean isInteger() {
		return false;
	}

	public boolean isReal() {
		return false;
	}

	public boolean isFloatValue() {
		return PFL.matcher(value).lookingAt();
	}

	public boolean isIntegerValue() {
		return (PIT.matcher(value).lookingAt() &&
				!PDC.matcher(value).lookingAt());
	}

	public boolean isRealValue() {
		return isFloatValue() || isIntegerValue();
	}

	public int signum() {
		Matcher m;

		if((m = PIT.matcher(value)).lookingAt()) {
			return new BigInteger(
					m.group().replaceFirst("^\\+", "")).signum();
		} else {
			return (int)Math.signum(toFloat());
		}
	}

	@Override
	public boolean isNegativeValue() {
		return signum() < 0;
	}

	@Override
	public boolean isPositiveValue() {
		return signum() > 0;
	}

	@Override
	public boolean isZeroValue() {
		return signum() == 0;
	}

	public boolean isString() {
		return true;
	}

	public boolean isRegex() {
		return false;
	}

	public boolean isFunction() {
		return false;
	}

	@Override
	public boolean isEmpty() {
		return value.equals("");
	}

	public int hashCode() {
		return value.hashCode();
	}

	public boolean equals(Object o) {
		if(o instanceof AwkString) {
			return value.equals(((AwkString)o).value);
		}
		return false;
	}

	public String toString(AwkNamespace ns) {
		return value;
	}

	public int compareTo(AwkValue o, AwkNamespace ns) {
		if(o instanceof AwkUndefined) {
			return 1;
		} else if(ns.isIgnoreCase()) {
			return toString(ns).compareToIgnoreCase(o.toString(ns));
		} else {
			return toString(ns).compareTo(o.toString(ns));
		}
	}

}
