/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.nina.translate.sh;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.util.Collection;
import java.util.Collections;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import net.morilib.nina.Quadro;
import net.morilib.nina.cmd.NinatOptions;
import net.morilib.sh.AbstractShFileSystem;
import net.morilib.sh.DefaultShVirtualDirectory;
import net.morilib.sh.ShEnvironment;
import net.morilib.sh.ShFile;
import net.morilib.sh.ShProcess;
import net.morilib.sh.ShSecurityPolicy;
import net.morilib.sh.ShStat;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2013/10/25
 */
public class ShNinatFileSystem extends AbstractShFileSystem {

	private static class StF implements ShFile {

		private byte[] buf;
		private String name;

		StF(String name, String f) {
			this.name = name;
			this.buf  = f.getBytes();
		}

		@Override
		public boolean isDirectory() {
			return false;
		}

		@Override
		public boolean isExecutable() {
			return false;
		}

		@Override
		public boolean isExist() {
			return true;
		}

		@Override
		public boolean isFile() {
			return true;
		}

		@Override
		public boolean isReadable() {
			return true;
		}

		@Override
		public boolean isWritable() {
			return false;
		}

		@Override
		public boolean isHidden() {
			return false;
		}

		@Override
		public boolean isZeroFile() {
			return buf.length == 0;
		}

		@Override
		public long getLastModified() {
			return 0;
		}

		@Override
		public String getName() {
			return name;
		}

		@Override
		public Collection<ShFile> getFiles() {
			return Collections.emptySet();
		}

		@Override
		public InputStream getInputStream() throws IOException {
			return new ByteArrayInputStream(buf);
		}

		@Override
		public PrintStream getPrintStream(
				boolean append) throws IOException {
			return null;
		}

		@Override
		public ShProcess getProcess() {
			return null;
		}

		@Override
		public ShFile toAbsolute() {
			return this;
		}

		@Override
		public ShFile toCanonical() {
			return this;
		}

		@Override
		public boolean mkdir() {
			return false;
		}

		@Override
		public boolean renameTo(ShFile f) {
			return false;
		}

		@Override
		public void setLastModified(long time) {
			// do nothing
		}

		@Override
		public boolean rmdir() {
			return false;
		}

		@Override
		public boolean deleteFile() {
			return false;
		}

		@Override
		public ShStat getStat() {
			return new ShStat(toString(), 0, buf.length,
					ShStat.READABLE);
		}

		@Override
		public String toString() {
			return name;
		}

	}

	private static final Pattern PTN1 = Pattern.compile(
			"/output/([^/]+)");

	private DefaultShVirtualDirectory root;
	private ShFile outputDir;
	private ShFile fragmentFile, definitionFile, fieldFile, headerFile;
	private ShFile readFile;
	private NinatOptions options;
	private File outputDirName;

	/**
	 * 
	 * @param f
	 * @param d
	 * @param opts
	 */
	public ShNinatFileSystem(String f, Quadro q, NinatOptions opts) {
		String s;

		root = new DefaultShVirtualDirectory(null, null);
		options = opts;
		fragmentFile = new StF("fragment", f);
		definitionFile = new StF("definition", q.getDefinition());
		fieldFile = new StF("field", q.getFragmentField());
		headerFile = new StF("header", q.getFragmentByName("header"));
		readFile = new StF("read", q.getFragmentByName("read"));

		root.addFile(fragmentFile);
		root.addFile(definitionFile);
		root.addFile(fieldFile);
		root.addFile(headerFile);
		root.addFile(readFile);

		s = (s = options.getOption("output")).equals("") ? "." : s;
		outputDirName = new File(s);
		outputDir = new ShNinaRealFile(this, "/output", outputDirName);
		root.addFile(outputDir);
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShFileSystem#getRoot()
	 */
	@Override
	public ShFile getRoot() {
		return root;
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShFileSystem#getFile(java.lang.String)
	 */
	@Override
	public ShFile getFile(String s) {
		Matcher m;

		if(s.equals("/")) {
			return root;
		} else if(s.equals("fragment")) {
			return fragmentFile;
		} else if(s.equals("definition")) {
			return definitionFile;
		} else if(s.equals("field")) {
			return fieldFile;
		} else if(s.equals("header")) {
			return headerFile;
		} else if(s.equals("read")) {
			return readFile;
		} else if(s.equals("/output")) {
			return outputDir;
		} else if((m = PTN1.matcher(s)).matches()) {
			return new ShNinaRealFile(this, s,
					new File(outputDirName, m.group(1)));
		} else {
			return null;
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShFileSystem#getFile(java.lang.String, java.lang.String)
	 */
	@Override
	public ShFile getFile(String dir, String name) {
		return getFile(getFile(dir), name);
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShFileSystem#getFile(net.morilib.sh.ShFile, java.lang.String)
	 */
	@Override
	public ShFile getFile(ShFile dir, String name) {
		return null;
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShFileSystem#getNativeFile(java.lang.String)
	 */
	@Override
	public ShFile getNativeFile(String s) {
		return getFile(s);
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShFileSystem#getSecurityPolicy()
	 */
	@Override
	public ShSecurityPolicy getSecurityPolicy() {
		return ShSecurityPolicy.ALL_PERMITTED;
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShFileSystem#getProfile()
	 */
	@Override
	public InputStream getProfile() {
		return null;
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShFileSystem#getRc()
	 */
	@Override
	public InputStream getRc() {
		return null;
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShFileSystem#getHome()
	 */
	@Override
	public ShFile getHome() {
		return getRoot();
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShFileSystem#searchPath(net.morilib.sh.ShEnvironment, java.lang.String)
	 */
	@Override
	public ShFile searchPath(ShEnvironment env, String name) {
		return null;
	}

}
