/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.nina;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import net.morilib.automata.AlphabetsNotDiscreteException;
import net.morilib.automata.DFA;
import net.morilib.automata.DFAState;
import net.morilib.automata.NFA;
import net.morilib.automata.NFAEdges;
import net.morilib.automata.NFAState;
import net.morilib.automata.TextBound;
import net.morilib.automata.dfa.DFAs;
import net.morilib.range.CharSets;
import net.morilib.range.Interval;
import net.morilib.range.Range;
import net.morilib.range.RangeAdder;
import net.morilib.util.TreeSectionMap;
import net.morilib.util.Tuple2;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2013/10/14
 */
public class DFABuilder implements NinaAction {

	private static class W {

		private CharSequence wrap;

		W(CharSequence s) {
			wrap = s;
		}

	}

	private static class RT {

		private String name;

		RT(String s) {
			name = s;
		}

	}

	/**
	 * 
	 *
	 *
	 * @author MORIGUCHI, Yuichiro 2013/10/14
	 */
	public static class DBS
	implements DFAState<Object, Object, Void>,
	NFAState, NFAEdges<Object> {

		TreeSectionMap<Interval, Object, DBS> edges;
		TreeSectionMap<Interval, Object, Object> mealy;
		boolean initial, accepted, moore;
		String nomFichier, label, recursiveName = null;
		Object mealyOthers = null;
		Object mealyEnd = null;
		DBS others = null;
		DBS end = null;
		DBS recursive = null;

		PrintWriter out = null;
		int devoirFermer = 0;

		Collection<DBS> identical = null;

		private void _linkRange(DBS d, Range o, Interval r, Object m) {
			if(d.identical != null) {
				throw new NinaException("invaliddfasubautomaton");
			} else if(edges == null) {
				edges = new TreeSectionMap<Interval, Object, DBS>(
						r, d);
				mealy = new TreeSectionMap<Interval, Object, Object>(
						r, m);
			} else {
				for(Tuple2<Interval, DBS> w : edges.entries()) {
					if(!o.independentOf(w.getA()) &&
							!d.equals(w.getB())) {
						throw new NinaNondeterministicException(
								o + "," + d + "," + w);
					}
				}
				edges.insert(r, d);
				mealy.insert(r, m);
			}
		}

		void linkAlphabet(DBS d, Object o, Object m) {
			Interval r;

			if(identical != null) {
				for(DBS x : identical) {
					x.linkAlphabet(d, o, m);
				}
			} else {
				r = Interval.newPoint(o);
				_linkRange(d, r, r, m);
			}
		}

		void linkAlphabet(DBS d, Range o, Object m) {
			if(identical != null) {
				for(DBS x : identical) {
					x.linkAlphabet(d, o, m);
				}
			} else {
				for(Interval v : o.intervals()) {
					_linkRange(d, o, v, m);
				}
			}
		}

		void linkOthers(DBS d, Object m) {
			if(others != null || recursive != null) {
				throw new NinaNondeterministicException("others");
			} else {
				others = d;
				mealyOthers = m;
			}
		}

		void linkEnd(DBS d, Object m) {
			if(end != null || recursive != null) {
				throw new NinaNondeterministicException("END");
			} else {
				end = d;
				mealyEnd = m;
			}
		}

		void linkRecursive(DBS d, String name) {
			if(others != null || end != null) {
				throw new NinaNondeterministicException("others");
			} else {
				recursive = d;
				recursiveName = name;
			}
		}

		void sortir(Object o) {
			String s;

			try {
				if(nomFichier == null || nomFichier.equals("")) {
					return;
				} else if(out == null) {
					if(nomFichier.equals("/dev/stdout")) {
						out = new PrintWriter(System.out, true);
						devoirFermer = -1;
					} else if(nomFichier.equals("/dev/stderr")) {
						out = new PrintWriter(System.err, true);
						devoirFermer = -1;
					} else if(nomFichier.equals("/dev/null")) {
						out = null;
					} else {
						out = new PrintWriter(new FileOutputStream(
								nomFichier, devoirFermer > 0), true);
						devoirFermer = 1;
					}
				}
		
				if(out != null && o != null &&
						!(s = o.toString().trim()).equals("")) {
					out.print(s);
					out.flush();
				}
				if(devoirFermer > 0)  out.close();
			} catch(IOException e) {
				// ignore it
				throw new RuntimeException(e);
			}
		}

		void sortirMoore() {
			if(moore)  sortir(label.trim());
		}

		/**
		 * 
		 * @return
		 */
		public DBS getOthers() {
			return others;
		}

		/**
		 * 
		 * @return
		 */
		public Object getMealyOthers() {
			return mealyOthers;
		}

		/**
		 * 
		 * @return
		 */
		public DBS getEnd() {
			return end;
		}

		/**
		 * 
		 * @return
		 */
		public Object getMealyEnd() {
			return mealyEnd;
		}

		/**
		 * 
		 * @return
		 */
		public DBS getRecursive() {
			return recursive;
		}

		/**
		 * 
		 * @return
		 */
		public String getRecursiveName() {
			return recursiveName;
		}

		@Override
		public DFAState<Object, Object, Void> go(Object a) {
			DFAState<Object, Object, Void> d;

			if(edges != null && (d = edges.map(a)) != null) {
				sortir(mealy.map(a));
				if(d instanceof DBS)  ((DBS)d).sortirMoore();
				return d;
			} else if(others == null) {
				return DFAs.<Object, Object, Void>deadState();
			} else {
				sortir(mealyOthers);
				return others;
			}
		}

		@Override
		public DFAState<Object, Object, Void> goInt(int x) {
			return go(Integer.valueOf(x));
		}

		@Override
		public DFAState<Object, Object, Void> goChar(char x) {
			return go(Integer.valueOf(x));
		}

		@Override
		public DFAState<Object, Object, Void> goBound(TextBound b) {
			return DFAs.deadState();
		}

		@Override
		public boolean isInitialState() {
			return initial;
		}

		@Override
		public Set<Object> getAccepted() {
			return accepted ?
					Collections.<Object>singleton(label) :
						Collections.emptySet();
		}

		@Override
		public boolean isDead() {
			return edges == null && others == null;
		}

		@Override
		public boolean isAccepted() {
			return accepted;
		}

		@Override
		public Set<Object> getAlphabets() {
			Set<Object> r = new HashSet<Object>();
			Object a, b;
			int x;

			if(edges != null) {
				for(Interval v : edges.keys()) {
					a = v.getInfimumBound();
					b = v.getSupremumBound();
					if(a.equals(b)) {
						r.add(a);
					} else if(a instanceof Integer) {
						x = ((Integer)a).intValue();
						for(int i = x; v.contains(i); i++) {
							r.add(Integer.valueOf(i));
						}
					} else {
						throw new NinaException("invalidalphabet",
								a);
					}
				}
			}
			return r;
		}

		@Override
		public Iterable<Interval> getAlphabetRanges() {
			return edges != null ?
					edges.keys() : Collections.<Interval>emptySet();
		}

		@Override
		public Object getLabel(Object a) {
			return mealy.map(a);
		}

		@Override
		public Object getLabelInt(int x) {
			return getLabel(Integer.valueOf(x));
		}

		@Override
		public Object getLabelChar(char x) {
			return getLabel(Integer.valueOf(x));
		}

		@Override
		public Set<NFAState> goNext(Object a) {
			DBS d;

			if(edges != null && (d = edges.map(a)) != null) {
				return Collections.singleton((NFAState)d);
			} else if(others == null) {
				return Collections.emptySet();
			} else {
				return Collections.singleton((NFAState)others);
			}
		}

		@Override
		public Set<NFAState> goNext(int a) {
			return goNext(Integer.valueOf(a));
		}

		@Override
		public Set<NFAState> goNext(char a) {
			return goNext(Integer.valueOf(a));
		}

		@Override
		public Set<NFAState> goNextEpsilon() {
			return Collections.emptySet();
		}

		@Override
		public Set<? extends Range> nextAlphabets() {
			Set<Range> x = new HashSet<Range>();
			RangeAdder a = new RangeAdder();

			for(Tuple2<Interval, DBS> t : edges.entries()) {
				x.add(t.getA());
				a.add(t.getA());
			}

			if(others == null) {
				// do nothing
			} else {
				x.add(CharSets.complement(a.toRange()));
			}
			return x;
		}

		@Override
		public boolean isNextEpsilon() {
			return false;
		}

		@Override
		public String toString() {
			return label;
		}

	};

	private static class FA
	implements DFA<Object, Object, Void>, NFA<Object, Object, Void> {

		private String name;
		private DBS initial;

		private transient Set<Object> allStates;

		FA(String name, DBS init) {
			this.name = name;
			initial   = init;
			allStates = new HashSet<Object>(DFAs.allStateRanges(init));
		}

		@Override
		public boolean isState(NFAState o) {
			return allStates.contains(o);
		}

		@Override
		public Set<NFAState> getStates(NFAState s, Object a) {
			DFAState<Object, Object, Void> d;

			if(!isState(s)) {
				return Collections.emptySet();
			} else if((d = ((DBS)s).go(a)).isDead()) {
				return Collections.emptySet();
			} else {
				return Collections.singleton((NFAState)d);
			}
		}

		@Override
		public Set<NFAState> getStates(NFAState s, Range r) {
			Set<NFAState> x;
			DBS d;

			if(!isState(s)) {
				return Collections.emptySet();
			} else {
				d = (DBS)s;
				x = new HashSet<NFAState>();
				for(Tuple2<Interval, DBS> t : d.edges.entries()) {
					for(Interval v : r.intervals()) {
						if(t.getA().equals(v)) {
							x.add(t.getB());
						}
					}
				}
				return x;
			}
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.NFA#getStates(net.morilib.automata.NFAState, java.util.EnumSet)
		 */
		@Override
		public Set<NFAState> getStates(NFAState s,
				EnumSet<TextBound> b) {
			return Collections.emptySet();
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.NFA#getStatesEpsilon(net.morilib.automata.NFAState)
		 */
		@Override
		public Set<NFAState> getStatesEpsilon(NFAState s) {
			return Collections.emptySet();
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.NFA#getStatesBound(net.morilib.automata.NFAState, java.util.EnumSet)
		 */
		@Override
		public Set<NFAState> getStatesBound(NFAState s,
				EnumSet<TextBound> b) {
			return Collections.emptySet();
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.NFA#getInitialStates()
		 */
		@Override
		public Set<NFAState> getInitialStates() {
			return Collections.singleton((NFAState)initial);
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.NFA#isInitialState(net.morilib.automata.NFAState)
		 */
		@Override
		public boolean isInitialState(NFAState o) {
			return initial.equals(o);
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.NFA#isFinal(net.morilib.automata.NFAState)
		 */
		@Override
		public boolean isFinal(NFAState s) {
			return isState(s) && ((DBS)s).isAccepted();
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.NFA#isFinalAny(java.util.Set)
		 */
		@Override
		public boolean isFinalAny(Set<NFAState> ss) {
			for(NFAState s : ss) {
				if(isFinal(s))  return true;
			}
			return false;
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.NFA#getEdges(net.morilib.automata.NFAState)
		 */
		@Override
		public NFAEdges<Object> getEdges(NFAState s) {
			return isState(s) ? (DBS)s : null;
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.NFA#nextAlphabets(net.morilib.automata.NFAState)
		 */
		@Override
		public Set<Interval> nextAlphabets(NFAState s) {
			Set<Interval> x = new HashSet<Interval>();
			DBS d;

			if(isState(s)) {
				d = (DBS)s;
				for(Tuple2<Interval, DBS> t : d.edges.entries()) {
					x.add(t.getA());
				}
				return x;
			} else {
				return Collections.emptySet();
			}
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.NFA#nextAlphabets(java.util.Set)
		 */
		@Override
		public Iterable<Interval> nextAlphabets(Set<NFAState> ss) {
			Set<Interval> x = new HashSet<Interval>();

			for(NFAState s : ss) {
				x.addAll(nextAlphabets(s));
			}
			return x;
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.NFA#nextDiscreteAlphabets(net.morilib.automata.NFAState)
		 */
		@Override
		public Set<Object> nextDiscreteAlphabets(NFAState s) {
			throw new AlphabetsNotDiscreteException();
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.NFA#nextDiscreteAlphabets(java.util.Set)
		 */
		@Override
		public Iterable<Object> nextDiscreteAlphabets(
				Set<NFAState> s) {
			throw new AlphabetsNotDiscreteException();
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.NFA#getAcceptedStates()
		 */
		@Override
		public Set<NFAState> getAcceptedStates() {
			Set<NFAState> x = new HashSet<NFAState>();

			for(Object s : allStates) {
				if(((DBS)s).isAccepted())  x.add((DBS)s);
			}
			return x;
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.NFA#getMatchTag(net.morilib.automata.NFAState)
		 */
		@Override
		public Set<Void> getMatchTag(NFAState s) {
			return Collections.emptySet();
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.NFA#getMatchTagEnd(net.morilib.automata.NFAState)
		 */
		@Override
		public Set<Void> getMatchTagEnd(NFAState s) {
			return Collections.emptySet();
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.NFA#getAccept(net.morilib.automata.NFAState)
		 */
		@Override
		public Set<Object> getAccept(NFAState s) {
			if(isAccepted(s)) {
				return Collections.singleton((Object)s);
			} else {
				return Collections.emptySet();
			}
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.NFA#isAccepted(net.morilib.automata.NFAState)
		 */
		@Override
		public boolean isAccepted(NFAState s) {
			return isState(s) && ((DBS)s).isAccepted();
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.DFA#getInitialState()
		 */
		@Override
		public DFAState<Object, Object, Void> getInitialState() {
			initial.sortirMoore();
			return initial;
		}

	}

	//
	private static final Object _END = new Object();

	private DBS initial, vertex;
	private Object mealy;
	private FA accepted = null;

	private DBS prendEtat(NinaEvent q) {
		DBS s;

		s = new DBS();
		s.nomFichier = q.getOption("out");
		s.moore = q.getOptionBoolean("moore");
		return s;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#labelAdded(net.morilib.nina.NinaEvent, java.lang.String, boolean)
	 */
	@Override
	public void labelAdded(NinaEvent q, NinaFrameReader rd,
			boolean accept) {
		StringWriter wr = new StringWriter();
		char[] a = new char[1024];
		int l;

		while((l = rd.read(a)) >= 0)  wr.write(a, 0, l);
		if(vertex == null) {
			// set initial state if it is not set
			vertex = prendEtat(q);
			vertex.initial = true;
			initial = vertex;
		}

		if(vertex.identical == null) {
			vertex.accepted = accept;
		} else {
			for(DBS x : vertex.identical)  x.accepted = accept;
		}
		vertex.label = wr.toString();
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#link(net.morilib.nina.NinaEvent, java.lang.Object)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public void link(NinaEvent q, Object ver) {
		Collection<DFAState<Object, Object, Void>> c;
		DFA<Object, Object, Void> d;
		DBS v = vertex, x, a;
		CharSequence s;
		RangeAdder r;
		Range z;
		int k;

		vertex = (ver != null) ? (DBS)ver : prendEtat(q);
		if(q.getEdge() instanceof FA) {
			v.linkRecursive(vertex, ((FA)q.getEdge()).name);
		} else if(q.getEdge() instanceof RT) {
			v.linkRecursive(vertex, ((RT)q.getEdge()).name);
		} else if(q.getEdge() instanceof DFA) {
			d = (DFA<Object, Object, Void>)q.getEdge();

			// check transition
			c = DFAs.allStateRanges(initial);
			for(DFAState<Object, Object, Void> y : c) {
				if(!(y instanceof DBS)) {
					throw new RuntimeException();
				} else if((x = (DBS)y).edges != null) {
					for(Tuple2<Interval, DBS> w : x.edges.entries()) {
						if(w.getB().equals(vertex)) {
							throw new NinaException(
									"invaliddfasubautomaton");
						}
					}
				}
			}

			// add initial state
			x = (DBS)d.getInitialState();
			if(x.edges != null) {
				for(Tuple2<Interval, DBS> w : x.edges.entries()) {
					v._linkRange(w.getB(), w.getA(), w.getA(),
							x.mealy.map(w.getA()));
				}
			}

			// change accept states
			c = DFAs.allAcceptedStateRanges(d);
			r = new RangeAdder();
			vertex.identical = new ArrayList<DBS>();
			if(vertex.edges != null) {
				for(Tuple2<Interval, DBS> w : vertex.edges.entries()) {
					r.add(w.getA());
				}
			}
			z = r.toRange();

			for(DFAState<Object, Object, Void> y : c) {
				if(!(y instanceof DBS)) {
					throw new RuntimeException();
				} else if(((DBS)y).identical != null) {
					throw new NinaException("invaliddfasubautomaton");
				}

				if((x = (DBS)y).edges != null) {
					for(Tuple2<Interval, DBS> w : x.edges.entries()) {
						x._linkRange(w.getB(), z, w.getA(),
								x.mealy.map(w.getA()));
					}
				}
				x.accepted = false;
				vertex.identical.add(x);
			}
		} else if(q.getEdge() instanceof W) {
			s = ((W)q.getEdge()).wrap;
			for(x = v, k = 0; k < s.length() - 1; k++, x = a) {
				a = prendEtat(q);
				x.linkAlphabet(a,
						Interval.newPoint(Integer.valueOf(s.charAt(k))),
						null);
			}
			x.linkAlphabet(vertex,
					Interval.newPoint(Integer.valueOf(s.charAt(k))),
					null);
		} else if(q.getEdge().equals(CharSets.ALL_CHAR)) {
			v.linkOthers(vertex, mealy);
		} else if(q.getEdge() instanceof Range) {
			v.linkAlphabet(vertex, (Range)q.getEdge(), mealy);
		} else if(q.getEdge() == _END) {
			v.linkEnd(vertex, mealy);
		} else {
			v.linkAlphabet(vertex, q.getEdge(), mealy);
		}
		mealy = null;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#getLabel()
	 */
	@Override
	public String getLabel() {
		return vertex.label;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#getVertex()
	 */
	@Override
	public Object getVertex() {
		return vertex;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#setVertex(java.lang.Object)
	 */
	@Override
	public void setVertex(Object o) {
		vertex = (DBS)o;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#doneBlockSearching(net.morilib.nina.NinaEvent)
	 */
	@Override
	public void doneBlockSearching(NinaEvent q) {
		vertex = (DBS)q.getScratch();
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#setEdge(net.morilib.nina.NinaEvent, java.lang.Object)
	 */
	@Override
	public void setEdge(NinaEvent q, Object o) {
		q.setEdge(o);
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#setEdgeCharSequence(net.morilib.nina.NinaEvent, java.lang.CharSequence)
	 */
	@Override
	public void setEdgeCharSequence(NinaEvent q, CharSequence o) {
		q.setEdge(new W(o));
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#setEdgeResource(net.morilib.nina.NinaEvent, net.morilib.nina.NinaParser, java.lang.String)
	 */
	@Override
	public void setEdgeResource(NinaEvent q,
			NinaParser p,
			String s,
			Map<String, String> map,
			Map<String, Object> sub) {
		String t = Nina.getFAName(s);

		if(sub.containsKey(t)) {
			q.setEdge(new RT(t));
		} else {
			sub.put(t, null);
			q.setEdge(p._compileRes(s, new DFABuilder(), map, sub));
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#setEdgeNFA(net.morilib.nina.NinaEvent, net.morilib.nina.NinaParser, net.morilib.automata.NFA)
	 */
	@Override
	public void setEdgeNFA(NinaEvent q, NinaParser p,
			NFA<Object, NFAState, Void> s) {
		throw new NinaException("cannotaddnfa");
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#setEdgeEnd(net.morilib.nina.NinaEvent)
	 */
	@Override
	public void setEdgeEnd(NinaEvent q) {
		q.setEdge(_END);
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#accept()
	 */
	@Override
	public Object accept(String name) {
		if(accepted == null) {
			accepted = new FA(name, initial);
		}
		return accepted;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#setMealyEdge(int)
	 */
	@Override
	public void setMealyEdge(int c) {
		mealy = Character.valueOf((char)c);
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#setMealyEdge(java.lang.Object)
	 */
	@Override
	public void setMealyEdge(Object o) {
		mealy = o;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#setPriority(int)
	 */
	@Override
	public void setPriority(int p) {
		// do nothing
	}

}
