#!/usr/bin/env ruby
# -*- mode: ruby; coding: euc-jp-unix; -*-
# $Id: nihohi.cgi,v 1.1.1.1 2003/12/21 22:36:22 takai Exp $

require 'date'
require 'cgi'
require 'rexml/document'

CONTEXT_XML = "#{File.dirname(__FILE__)}/context.xml"

class Template
  def initialize hash
    @regexp = Regexp.new('\\{(' << hash.keys.join('|') << ')\\}')
    @hash   = hash
  end
  def expand str, hash = nil
    result = str.gsub(@regexp){|matched|
      matched = @hash[$1] || '{' << $1 << '}'
    } || str
    if hash
      template = Template.new hash
      result = template.expand(result)
    end
    return result
  end
end

class Registry
  attr_accessor :properties
  def store key, value
    properties[key] = value
    properties
  end
  def fetch key
    return properties[key]
  end
  def to_hash
    return properties
  end
end

class Flavor
  attr_accessor :skin_dir
  DEFAULT_HTML = 
         {:head  =>
          '<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01 Transitional//EN">'+
          '<html><head><title>{site_title}</title></head>' +
          '<body><h1>{site_title}</h1>',
          :date  => '<h2>{date}</h2>',
          :story =>
          '<h3>{title}</h3><p>{body}</p>' +
          '<p>posted at {time} | ' + 
          'path: [<a href="{url}/view{path}/">{path}</a>] | ' + 
          '<a href="{url}/view{path}/{file}{suffix}">' + 
          'permanent link to this entry</a></p>',
          :foot  => '<hr><p>powered by nihohi</p></body></html>'}
  def initialize
    @html = nil
  end
  def init
    if skin_dir
      unless @html
        @html = {}
        @html[:head]  = File.open("#{skin_dir}/head.html"){|f| f.read}
        @html[:date]  = File.open("#{skin_dir}/date.html"){|f| f.read}
        @html[:story] = File.open("#{skin_dir}/story.html"){|f| f.read}
        @html[:foot]  = File.open("#{skin_dir}/foot.html"){|f| f.read}
      end
    else
      @html = DEFAULT_HTML
    end
  end
  def head_html
    init
    return @html[:head]
  end
  def date_html
    init
    return @html[:date]
  end
  def story_html
    init
    return @html[:story]
  end
  def foot_html
    init
    return @html[:foot]
  end
end

class HtmlView
  attr_accessor :registry, :flavor
  
  def render resources
    charset      = registry.fetch('charset')      || 'UTF-8'
    content_type = registry.fetch('content-type') || 'text/html'
    date_format  = registry.fetch('date_format')  || '%a, %d %b %Y'
    head_html    = flavor.head_html
    date_html    = flavor.date_html
    story_html   = flavor.story_html
    foot_html    = flavor.foot_html

    template = Template.new(registry.to_hash)
    
    head = template.expand(head_html)

    prev_date = Date.new
    body = ""
    resources.each do |resource|
      unless resource.date === prev_date
        body << template.expand(date_html,
                                {'date' => 
                                 resource.time.strftime(date_format)})
      end

      prev_date = resource.date
      body << template.expand(story_html, resource.to_hash)
    end
    foot = template.expand(foot_html)

    last_modified = resources.map{|res| res.time }.sort.last
    
    return [{'type' => content_type,
             'charset' => charset,
             'Last-Modified' => CGI.rfc1123_date(last_modified)},
            head + body + foot]
  end
end

class Resource
  attr_accessor :title, :time, :body, :path, :file
  def date
    return Date.new(time.year, time.month, time.day)
  end
  def to_hash
    return {'title' => title,
            'time' => time.strftime("%H:%M %Z"),
            'body' => body,
            'path' => path,
            'file' => file}
  end
end

class PlainTextGenerator
  def generate path, file
    res = Resource.new
    text = File.open(file){|f| f.read }
    res.title, res.body  = text.split(/\n/, 2)
    res.path = path
    res.file = File.basename(file, File.extname(file))
    res.time = File.mtime file
    return res
  end
end

class FileSystemFinder
  attr_accessor :directory, :limit, :generators, :registry
  
  def find path
    check_path(path)

    if is_file_request?(path)
      dirname  = File.dirname(path)
      basename = File.basename(path, suffix)
      file_suffix.each do |s|
        file_path = "#{directory}/#{dirname}/#{basename}#{s}"
        if FileTest.exists?(file_path)
          return [generators[s].generate(dirname, file_path)]
        end
      end
      raise NotFoundError.new("The requested path #{path} not found")
    else
      pattern = "#{directory}/#{path}/**/*{#{file_suffix.join(',')}}"
      files = Dir.glob(pattern).sort_by{ |file|
        File.mtime(file)
      }.reverse

      unless files.size > 0
        raise NotFoundError.new("The requested path #{path} not found")
      end
      
      if(limit > 0)
        files = files[0..(limit - 1)]
      end
      return files.map{|file|
        dirname = File.dirname(file)
        file_path = dirname[directory.length .. dirname.length]
        file_path.gsub!(%r|/+|, '/')
        generators[File.extname(file)].generate(file_path, file)
      }
    end
  end

  private
  def suffix
    return registry.fetch('suffix')
  end

  def file_suffix
    return generators.keys
  end

  def check_path path
    if /\.\./ =~ path
      raise ApplicationError.new("path must NOT contain '..'")
    end
  end

  def is_file_request? path
    @suffix_regexp ||= Regexp.new("(.*)#{suffix}$") 
    return @suffix_regexp =~ path
  end
end

class ViewerApplication
  attr_accessor :finder, :view
  def run path, command = nil
    resources = finder.find(path)
    return view.render(resources)
  end
end

class ApplicationLoader
  attr_accessor :applications
  def load name
    return applications[name]
  end
end

class XmlFileContext
  def initialize file
    doc = REXML::Document.new(File.open(file))
    @hash = {}
    doc.elements.each("/context/object") do |object|
      @hash[object.attributes["id"]] = \
        load_class(object.attributes["class"]).new
    end
    doc.elements.each("/context/object") do |object|
      obj = @hash.fetch(object.attributes["id"])
      object.elements.each("property") do |property|
        name = property.attributes["name"]
        if property.elements.empty?
          value = if /^\d+$/ =~ property.text
                    property.text.to_i
                  elsif  /^\d+\.\d+$/ =~ property.text
                    property.text.to_f
                  else
                    property.text
                  end
          obj.__send__ "#{name}=", value
        else
          case property.elements[1].name
          when 'hash'
            hash = {}
            property.elements.each("hash/entry") do |entry|
              value = if entry.elements.empty?
                        if /^\d+$/ =~ entry.text
                          entry.text.to_i
                        elsif  /^\d+\.\d+$/ =~ entry.text
                          entry.text.to_f
                        else
                          entry.text
                        end
                      else
                        @hash[entry.elements["ref"].attributes["object"]]
                      end
              hash[entry.attributes["key"]] = value
            end
            obj.__send__ "#{name}=", hash
          when 'list'
            raise NotImplementedError.new('<list> is not yet implemented.')
          when 'ref'
            refobj = property.elements[1].attributes["object"]
            obj.__send__ "#{name}=", @hash.fetch(refobj)
          end
        end
      end
    end
  end
  def get name
    return @hash.fetch(name)
  end
  def load_class(class_name)
    ObjectSpace.each_object(Class){|c|
      return c if c.name == class_name
    }
    raise "Class #{class_name} not found"
  end
end

class ApplicationError < RuntimeError
end

class NotFoundError < ApplicationError
end

begin
  cgi = CGI.new

  path_info = cgi.path_info || ""

  null, appname, path = path_info.split("/", 3)
  appname = "view" unless appname.to_s.size > 0

  context = XmlFileContext.new(CONTEXT_XML)
  registry = context.get("registry")
  registry.store('url', cgi.script_name)
  registry.store('basedir', File.dirname(cgi.script_name))
  loader = context.get("application_loader")
  app = loader.load(appname) or
    raise ApplicationError.new("Applicnation #{appname} not found")
  header, body = app.run(path)
  cgi.out(header){body}

rescue Exception => e
  cgi = CGI.new("html4Tr")
  cgi.out({'type'    => 'text/html',
           'charset' => 'iso8859-1'}){
    error_class     = CGI.escapeHTML(e.class.to_s)
    error_message   = CGI.escapeHTML(e.message)
    error_backtrace = e.backtrace.map{|str| CGI.escapeHTML(str)}.join("<BR>")
    
    cgi.html do
      cgi.head do
        cgi.title do "Error! #{error_class}" end
      end +
      cgi.body() do
        cgi.h1 do "Error! #{error_class}" end +
        cgi.p do error_message end +
        cgi.blockquote do error_backtrace end
      end
    end
  }
end
