/*
 * Copyright (c) 2012, Alexey Badalov
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of Tomboy Exporter nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software  * without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED  TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE. 
 */

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintStream;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.xml.sax.SAXException;


public class Exporter {

	/**
	 * @param args
	 */
	public static void main(String[] args) {
		if (args.length < 2) {
			System.err.println("Not enough parameters. The syntax is:\njava Exporter output-dir note-files");
		}
		
		new File(args[0]).mkdir();
		
		try {
			SAXParser parser = SAXParserFactory.newInstance().newSAXParser();

			// Make a pass through all the notes to find which notebooks they belong to.
			TomboyNotebookScanner notebookScanner = new TomboyNotebookScanner();
			for (int i = 1; i < args.length; ++i) {
				notebookScanner.setCurrentNoteName(args[i]);
				try {
					parser.parse(new File(args[i]), notebookScanner);
				} catch (IOException e) {
					System.err.println("Error: could not read note '" + args[i] + "'.");
					e.printStackTrace();
				} catch (SAXException e) {
					System.err.println("Error: could not read note '" + args[i] + "'.");
					e.printStackTrace();
				}
			}
			
			// Create an .nnex file for each notebook.
			for (String notebook : notebookScanner.getNotebooks()) {
				File outputFile = new File(args[0], makeFriendlyNotebookName(notebook) + ".nnex");
				try {
					PrintStream outputStream = new PrintStream(outputFile);
					TomboyXmlHandler xmlHandler = new TomboyXmlHandler(outputStream);
					for (String note : notebookScanner.GetNotebookNotes(notebook)) {
						try {
							parser.parse(new File(note), xmlHandler);
						} catch (IOException e) {
							System.err.println("Error: could not read note '" + note + "'.");
							e.printStackTrace();
						}
					}
					xmlHandler.end();
				} catch (FileNotFoundException e) {
					System.err.println("Error: could not write '" + outputFile + "'.");
				} catch (SAXException e) {
					System.err.println("Error: could not write '" + outputFile + "'.");
				}
			}
		} catch (ParserConfigurationException e) {
			System.err.println("Error: could not initialize.");
		} catch (SAXException e) {
			System.err.println("Error: could not initialize.");
		}
	}
	
	private static String makeFriendlyNotebookName(String name) {
		StringBuffer result = new StringBuffer();
		
		for (char c : name.toCharArray()) {
			if (Character.isDigit(c)) {
				result.append(c);
				continue;
			}
			if (Character.isLetter(c)) {
				result.append(Character.toLowerCase(c));
				continue;
			}
			if (Character.isWhitespace(c)) {
				result.append('-');
				continue;
			}
		}
		
		return result.toString();
	}

}
