# N2::Data
# 2008/4/9 v0.1
# H.OHARA
use strict;
use warnings;
use File::Basename;
use File::Path;
use Ho::A;
use N2::Prefs;
use N2::Deny;
use N2::Spam;
use N2::Group;

# Model class
package N2::Data;
use Carp;

# constructor
sub new {
    my $class = shift;
    my $self = {
        db => 'N2Wiki_db',
        temppage => '/_template',
        resdb => 'data',
        extns => {
            jpg => '', jpeg => '', gif => '', png => '',
            bmp => '', pch => '', spch => '', ico => '',
            txt => '', css => '', bak => '', rdf => '',
            xml => '',
            zip => '', gz => '', rar => '', lzh => '', tar => '',
            tbz => '', tbz2 => '',
            tgz => '', cab => '', '7z' => '', bz => '', bz2 => '',
            z => '', Z => '',
            sit => '', sitx => '', dmg => '',
            mp3 => '', mp4 => '', m4a => '', aif => '', aiff => '',
            wav => '', ogg => '', wma => '',
            mpg => '', avi => '', wmv => '', rm => '', mov => '',
            mid => '', mmf => '', amc => '', m3u => '',
            '3gp' => '', '3gp2' => '', m4p => '',
            flv => '', swf => '', fla => '',
            pdf => '', doc => '', ppt => '', xls => '',
            docx => '', pptx => '', xlsx => ''
        },
        extnpage => '/_preferences/extensions',
        ext => undef,
        freeze => '/_preferences/freeze',
        freezedelay => undef,
        deny => '/_preferences/deny',
        denylist => undef,
        denynew => '/_preferences/denynew',
        denynewlist => undef,
        denyread => '/_preferences/denyread',
        denyreadlist => undef,
        usefreezedelay => 1,
        user => '_preferences/user',
        userpwd => undef,
        session => '_preferences/session',
        usersession => undef,
        group => '_preferences/group',
        grouplist => undef,
        spam => '_spam/ip',
        spamlist => undef,
        ngword => '_spam/ngword',
        ngwordlist => undef,
        modfile => 'moddate.bak',
        logpage => '/_spam/accesslog',
        statpage => '/_spam/stat',
        pgpage => '/_spam/pages',
        refspage => '/_spam/refs',
        uaspage => '/_spam/uas',
        spampages => undef,
        spamrefs => undef,
        spamuas => undef,
        session_expire => 3600,
        recentdeleted => '/_RecentDeleted',
        _lockdir => 'lock',
        _lockdir1 => 'lock/n2lock1',
        _lockdir2 => 'lock/n2lock2',
        _lockretry => 5,
        @_
    };
    if($self->{db} !~ /\/$/) {
        $self->{db} = $self->{db}.'/';
    }
    Ho::A->make_dir($self->{db}, 0777);
    Ho::A->make_dir($self->{_lockdir}, 0777);
    Ho::A->make_dir($self->{resdb}, 0777);
    $self = bless $self, $class;
    $self->init();
    return $self;
}

# private methods
sub init {
    my $self = shift;
    $self->{spamlist} = $self->init_spamres($self->{spam});
    $self->{spampages} = $self->init_spamres($self->{pgpage});
    $self->{spamrefs} = $self->init_spamres($self->{refspage});
    $self->{spamuas} = $self->init_spamres($self->{uaspage});
}

sub deny_prefs {
    my $self = shift;
    my ($page,$key) = @_;
    if(!defined($self->{$key})) {
        $self->{$key} = N2::Deny->new(
            page => $page, db => $self,
            rex => '^([^# ].*)\s*$', #'
        );
    }
    return $self->{$key};
}

sub init_spamres {
    my $self = shift;
    my ($pg) = @_;
    return N2::Spam->new(page => $pg, db => $self);
}

sub is_denied {
    my $self = shift;
    my ($page) = @_;
    my $dny = $self->deny_prefs($self->{deny},'denylist');
    return $dny->get($page);
}

sub is_denied_ip {
    my $self = shift;
    my ($ip) = @_;
    my $prefs = $self->{spamlist}->get();
    foreach my $dp (@$prefs) {
        my $p = substr($dp,-1,1);
        if($ip eq $dp 
            || ($p eq '.' && substr($ip,0,length($dp)) eq $dp)) {
            return 1;
        }
    }
    return 0;
}

sub is_denynew {
    my $self = shift;
    my ($page) = @_;
    my $dny = $self->deny_prefs($self->{denynew},'denynewlist');
    return $dny->get($page);
}

sub is_denyread {
    my $self = shift;
    my ($page) = @_;
    my $dny = $self->deny_prefs($self->{denyread},'denyreadlist');
    return $dny->get($page);
}

sub delay_prefs {
    my $self = shift;
    if(!defined($self->{freezedelay})) {
        $self->{freezedelay} = N2::PathPrefs->new(
            page => $self->{freeze}, db => $self,
            rex => '^(\/.*)\s+=\s+([\w-]+)\s*$', #'
        );
    }
    return $self->{freezedelay};
}

sub delay_setting {
    my $self = shift;
    my ($page) = @_;
    my $df = $self->delay_prefs();
    $df->get_pref($page);
}

sub curr_delay {
    my $self = shift;
    my ($page) = @_;
    if(!$self->{usefreezedelay}) { return -1; }
    my $df = $self->delay_prefs();
    my $n = $df->get($page);
    my %units = (
        s => 86400, m => 1440, h => 24, d => 1
    );
    if($n) {
        my $delta = substr($n,0,-1);
        my $unit = substr($n,-1,1);
        if(defined($units{$unit})) {
            return $delta / $units{$unit};
        }
        else {
            return $n/24;
        }
    }
    else {
        return -1;
    }
}

sub set_delay {
    my $self = shift;
    my ($key, $value) = @_;
    my $df = $self->delay_prefs();
    $df->set($key, $value);
}

sub is_frozen {
    my $self = shift;
    my ($page) = @_;
    my $delay = $self->curr_delay($page);
    my $mod = $self->filemod($page);
    if($page eq '/') { return 1; }
    if($page =~ m|/_[^/]*$|) { return 1; }
    if($mod == 0 && $self->is_denynew($page)) { return 1; }
    if($delay < 0) { return 0; }
    return ($mod > $delay);
}

sub user_prefs {
    my $self = shift;
    my ($page,$key) = @_;
    if(!defined($self->{$key})) {
        $self->{$key} = N2::Prefs->new(
            page => $page, db => $self,
        );
    }
    return $self->{$key};
}

sub get_userpwd {
    my $self = shift;
    my ($user) = @_;
    my $upwd = $self->user_prefs($self->{user},'userpwd');
    return $upwd->get($user);
}

sub set_userpwd {
    my $self = shift;
    my ($user,$epwd) = @_;
    my $upwd = $self->user_prefs($self->{user},'userpwd');
    return $upwd->set($user,$epwd);
}

sub check_userpwd {
    my $self = shift;
    my ($page,$pwd) = @_;
    my $epwd = $self->get_userpwd($page);
    $epwd = defined($epwd) ? $epwd : '' ;
    if($page eq '/' && $epwd eq '') {
        return 1;
    }
    if($epwd eq '') { return 0; }
    return $epwd eq crypt($pwd, $epwd);
}

sub curr_userdir {
    my $self = shift;
    my ($pg) = @_;
    my $user = $self->user_prefs($self->{user},'userpwd');
    my $opg;
    while(!defined($user->get($pg))) {
        $opg = $pg;
        $pg = $self->dirname($pg);
        if($pg eq $opg) { return ''; }
    }
    return $pg;
}

sub group_prefs {
    my $self = shift;
    if(!defined($self->{grouplist})) {
        $self->{grouplist} = N2::Group->new(
            page => $self->{group}, db => $self
        );
    }
    return $self->{grouplist};
}

sub is_member {
    my $self = shift;
    my ($user,$page,$pwd) = @_;
    my $glist = $self->group_prefs()->prefs();
    my $mlist = $glist->{$page};
    my $epwd = $self->get_userpwd($page);
    $epwd = defined($epwd) ? $epwd : '' ;
    my $opg;
    if(!$self->check_userpwd($user,$pwd)) { return ''; }
    while(1) {
        my $af = ($epwd ne '') && ($epwd eq crypt($pwd, $epwd));
        my $n = grep { $_ eq $user } @$mlist;
        if($af || $n != 0 ) { return $page; }
        $opg = $page;
        $page = $self->dirname($page);
        if($page eq $opg) { return ''; }
        $epwd = $self->get_userpwd($page);
        $epwd = defined($epwd) ? $epwd : '' ;
        $mlist = $glist->{$page};
    }
}

sub sidcrypt {
    my $self = shift;
    my ($sid,$salt) = @_;
    return Ho::A->sidcrypt($sid,$salt);
}

sub is_membersid {
    my $self = shift;
    my ($user,$page,$sid) = @_;
    my $glist = $self->group_prefs()->prefs();
    my $mlist = $glist->{$page};
    my $esid = $self->get_usersession($page);
    $esid = defined($esid) ? $esid : '' ;
    my $opg;
    if(!$self->check_usersession($user,$sid)) { return ''; }
    while(1) {
        my $af = ($esid ne '') && ($esid eq $self->sidcrypt($sid, $esid));
        my $n = grep { $_ eq $user } @$mlist;
        if($af || $n != 0 ) { return $page; }
        $opg = $page;
        $page = $self->dirname($page);
        if($page eq $opg) { return ''; }
        $esid = $self->get_usersession($page);
        $esid = defined($esid) ? $esid : '' ;
        $mlist = $glist->{$page};
    }
}

sub get_usersession {
    my $self = shift;
    my ($page) = @_;
    my $uss = $self->user_prefs($self->{session},'usersession');
    my $sdata = $uss->get($page);
    $sdata = $sdata ? $sdata : '' ;
    my ($esid,$tm) = split(/,/,$sdata);
    $esid = $esid ? $esid : '' ;
    $tm = $tm ? $tm : 0 ;
    if($tm > 0 && time() > $tm+$self->{session_expire}) {
        $self->set_usersession($page,'');
        return '';
    }
    else {
        return $esid;
    }
}

sub set_usersession {
    my $self = shift;
    my ($page,$epwd) = @_;
    my $tm = time();
    my $uss = $self->user_prefs($self->{session},'usersession');
    if($epwd eq '') {
        return $uss->set($page,'');
    }
    else {
        return $uss->set($page,"$epwd,$tm");
    }
}

sub update_usersession {
    my $self = shift;
    my ($page,$epwd) = @_;
    my $tm = time();
    if($epwd) {
        my $uss = $self->user_prefs($self->{session},'usersession');
        $uss->set($page,"$epwd,$tm");
    }
}

sub check_usersession {
    my $self = shift;
    my ($page,$pwd) = @_;
    my $epwd = $self->get_usersession($page);
    $epwd = defined($epwd) ? $epwd : '' ;
    if($epwd eq '') { return 0; }
    my $b = $epwd eq $self->sidcrypt($pwd, $epwd);
    if($b) {
        $self->update_usersession($page,$epwd);
    }
    return $b;
}

sub check_sid {
    my $self = shift;
    my ($esid,$sid) = @_;
    $esid = defined($esid) ? $esid : '' ;
    $sid = defined($sid) ? $sid : '' ;
    if($esid eq '') { return 0; }
    return $esid eq $self->sidcrypt($sid, $esid);
}

sub is_currsession {
    my $self = shift;
    my ($page,$pwd) = @_;
    my $epwd = $self->get_usersession($page);
    $epwd = defined($epwd) ? $epwd : '' ;
    my $opg;
    while($epwd eq '' || $epwd ne $self->sidcrypt($pwd, $epwd)) {
        $opg = $page;
        $page = $self->dirname($page);
        if($page eq $opg) { return ''; }
        $epwd = $self->get_usersession($page);
        $epwd = defined($epwd) ? $epwd : '' ;
    }
    return $page;
}

sub filename_of {
    my $self = shift;
    my ($key) = @_;
    return $self->{db}.Ho::A->encode_uri($self->as_pagename($key));
}

sub lock {
    my $self = shift;
    Ho::A->lock($self->{_lockretry},$self->{_lockdir1},$self->{_lockdir2});
}

sub unlock {
    my $self = shift;
    Ho::A->unlock($self->{_lockdir1});
}

sub read_db {
    my $self = shift;
    my ($fname) = @_;
    my @data;
    $fname = Ho::A->check_filename($fname,'[\w\/%]+');
    open(DB,$fname) || die $!;
    @data = <DB>;
    close(DB);
    return Ho::A->unescape(join('',@data));
}

sub write_db {
    my $self = shift;
    my ($fname,$data,$touch,$add) = @_;
    $touch = defined($touch) ? $touch : '' ;
    $add = defined($add) ? $add : '' ;
    my $mod = '';
    $fname = Ho::A->check_filename($fname,'[\w\/%]+');
    if(length($fname) > 255) { return 1; }
    my $err = sub { $self->unlock(); die $!; };
    $self->lock();
    if($touch) { $mod = (-f $fname) ? (stat $fname)[9] : '' ; };
    if($add) {
        open(DB,">> $fname") || $err->();
    }
    else {
        open(DB,"> $fname") || $err->();
    }
    print DB Ho::A->escape($data);
    close(DB);
    if($touch && $mod ne '') { utime($mod, $mod, $fname); }
    $self->unlock();
    return 0;
}

sub read_res {
    my $self = shift;
    my ($fname) = @_;
    my $data;
    $fname=Ho::A->check_filename($fname,q([^&;\\`'"|*?~<>^()\[\]{}$\r\n]+));
    # '
    open(RES,$fname) || die $!;
    binmode(RES);
    read(RES,$data,-s $fname);
    close(RES);
    return $data;
}

sub write_res {
    my $self = shift;
    my ($fname,$data,$add) = @_;
    $add = defined($add) ? $add : '' ;
    $fname=Ho::A->check_filename($fname,q([^&;\\`'"|*?~<>^()\[\]{}$\r\n]+));
    # '
    my $err = sub { $self->unlock(); die $!; };
    $self->lock();
    if($add) {
        open(RES,">> $fname") || $err->();
    }
    else {
        open(RES,"> $fname") || $err->();
    }
    binmode(RES);
    print RES $data;
    close(RES);
    $self->unlock();
    return 0;
}

sub search_pagetemplate {
    my $self = shift;
    my ($key) = @_;
    my $tmp = $self->{temppage};
    my $parent = $self->dirname($tmp.$self->as_pagename($key));
    while(not $self->exists_page($parent) and $parent ne $tmp) {
        $parent = $self->dirname($parent);
    }
    return $parent;
}

# public methods
sub as_pagename {
    my $self = shift;
    my ($s) = @_;
    return ($s !~ /^\//) ? '/'.$s : $s ;
}

sub pagename {
    my $self = shift;
    my ($s,$page) = @_;
    $s = $self->as_pagename($s);
    $page = $self->as_pagename($page);
    if($s =~ m|^/\.\./(.*)$|) {
        $page = $self->dirname($page);
        $page = $page eq '/' ? '' : $page ;
        if($1) {
            $s =~ s|^/\.\./|$page/|;
        }
        else {
            $s = $page eq '' ? '/' : $page ;
        }
    }
    elsif($s =~ m|^/\./(.*)$|) {
        $page = $page eq '/' ? '' : $page ;
        if($1) {
            $s =~ s|^/\./|$page/|;
        }
        else {
            $s = $page eq '' ? '/' : $page ;
        }
    }
    elsif($s =~ m|^/~/(.*)$|) {
        $page = $self->curr_userdir($page);
        $page = $page eq '/' ? '' : $page ;
        if($1) {
            $s =~ s|^/~/|$page/|;
        }
        else {
            $s = $page eq '' ? '/' : $page ;
        }
    }
    return $s;
}

sub longpagename {
    my $self = shift;
    my ($page) = @_;
    $page = $self->as_pagename($page);
    return (length($self->filename_of($page)) > 255);
}

sub exists_page {
    my $self = shift;
    my ($key) = @_;
    my $fname = $self->filename_of($key);
    return (-f $fname);
}

sub get_page_info {
    my $self = shift;
    my ($key) = @_;
    my $fname = $self->filename_of($key);
    return (-f $fname) ? Ho::A->file_info($fname) : ( ) ;
}

sub moddate {
    my $self = shift;
    my ($key) = @_;
    my %info = $self->get_page_info($key);
    my $mod = $info{mod} ? $info{mod} : 0 ;
    return $mod;
}

sub filemod {
    my $self = shift;
    my ($key) = @_;
    my $fname = $self->filename_of($key);
    return (-f $fname) ? (-M $fname) : 0 ;
}

sub filesiz {
    my $self = shift;
    my ($key) = @_;
    my $fname = $self->filename_of($key);
    return (-s $fname);
}

sub get {
    my $self = shift;
    my ($key) = @_;
    my $fname = $self->filename_of($key);
    if(-f $fname) {
        return $self->read_db($fname);
    }
    else {
        my $tmp = $self->search_pagetemplate($key);
        return $self->read_db($self->filename_of($tmp));
    }
}

sub set {
    my $self = shift;
    my ($key,$data,$moddate,$touch,$add) = @_;
    $data = Ho::A->replace_newline($data);
    if(defined($moddate) && $moddate ne '') {
        if($moddate < $self->editdate($key)) {
            return 2;
        }
    }
    my $err = $self->write_db($self->filename_of($key),$data,$touch,$add);
    if(!$err) {
        $self->set_editdate($key);
    }
    return $err;
}

sub copy {
    my $self = shift;
    my ($src,$dst) = @_;
    my $data = $self->get($src);
    return $self->write_db($self->filename_of($dst),$data);
}

sub move {
    my $self = shift;
    my ($src,$dst) = @_;
    my $err = $self->copy($src,$dst);
    if($err) { return $err; }
    $self->remove($src);
    return 0;
}

sub remove {
    my $self = shift;
    my ($key) = @_;
    my $fname = $self->filename_of($key);
    $fname = Ho::A->check_filename($fname,'[\w\/%]+');
    $self->lock();
    unlink($fname);
    $self->unlock();
}

sub rename {
    my $self = shift;
    my ($old,$new) = @_;
    my $oldn = $self->filename_of($old);
    my $newn = $self->filename_of($new);
    $oldn = Ho::A->check_filename($oldn,'[\w\/%]+');
    $newn = Ho::A->check_filename($newn,'[\w\/%]+');
    $self->lock();
    rename($oldn,$newn);
    $self->unlock();
}

sub copy_tree {
    my $self = shift;
    my ($src,$dst) = @_;
    my $err;
    if(($err=$self->copy($src,$dst))) { return $err; } 
    my $cond = sub { 1; };
    my $proc = sub { 
        my ($x) = @_;
        if($x =~ m|^\Q$src\E(/.*)?|) {
            $self->copy($x,$dst.$1);
        }
    };
    $self->filter_and_each('-d',$src,$cond,$proc);
}

sub move_tree {
    my $self = shift;
    my ($src,$dst) = @_;
    my $cond = sub { 1; };
    my $proc = sub { 
        my ($x) = @_;
        if($x =~ m|^\Q$src\E(/.*)|) {
            $self->rename($x,$dst.$1);
        }
    };
    $self->rename($src,$dst);
    $self->filter_and_each('-d',$src,$cond,$proc);
}

sub remove_tree {
    my $self = shift;
    my ($key) = @_;
    my $cond = sub { 1; };
    my $proc = sub { 
        my ($x) = @_;
        if($x =~ m|^\Q$key\E/.*|) {
            $self->remove($x);
        }
    };
    $self->remove($key);
    $self->filter_and_each('-d',$key,$cond,$proc);
}

sub eachfiles {
    my $self = shift;
    my ($dir,$options,$pg) = @_;
    my @pages;
    $options = defined($options) ? $options : "" ;
    
    opendir(DIR,$dir);
    @pages = grep {-f "$dir$_"} readdir(DIR);
    closedir(DIR);
    
    if($options =~ /a/) {
        @pages = grep {/^%2[Ff]/} @pages;
    }
    else {
        @pages = grep {/^%2[Ff]([^_]|$)/} @pages;
    }

    if(defined($pg)) {
        $pg = $options =~ /u/ ? $self->curr_userdir($pg) : $pg ;
        $pg = Ho::A->encode_uri($pg);
        if($options =~ /d/) {
            @pages = grep {/^$pg%2[Ff]/} @pages; 
        }
        elsif($options =~ /[eu]/) {
            @pages = grep {/^($pg$)|($pg%2[Ff])/} @pages; 
        }
    }
    
    if($options =~ /[MAC]/) {
        if($options =~ /M/) {
            @pages = map  { [$_, -M "$dir$_"] } @pages;
        }
        elsif($options =~ /A/) {
            @pages = map  { [$_, -A "$dir$_"] } @pages;
        }
        elsif($options =~ /C/) {
            @pages = map  { [$_, -C "$dir$_"] } @pages;
        }
        @pages = map  { Ho::A->decode_uri($_) }
                 map  { $_->[0] }
                 sort { $a->[1] <=> $b->[1] } @pages;
    }
    else {
        @pages = sort {$a cmp $b}
                 map  { Ho::A->decode_uri($_) } @pages;
    }
    
    if($options =~ /r/) {
        @pages = reverse(@pages);
    }
    
    return @pages;
}

sub each {
    my $self = shift;
    my ($options,$pg) = @_;
    my $dir = $self->{db};
    return $self->eachfiles($dir,$options,$pg);
}

sub filter_and_eachfiles {
    my $self = shift;
    my ($e,$options,$page,$cond,$proc) = @_;
    my @pages = $e->();
    my $count = 0;
    my $limit = 0;
    my $rev = ($options =~ /R/);
    my $con = $cond;
    my @cache;
    if($options =~ /L([0-9]*)/) {
        if($1) {
            $limit = $1;
        }
        else {
            $limit = 10;
        }
    }

    if($options =~ /n/) {
        $con = sub {
            my ($x) = @_;
            return !$cond->($x);
        };
    }
    foreach my $x (@pages) {
        if($con->($x)) {
            if($rev) {
                push(@cache,$x);
            }
            else {
                $proc->($x);
            }
            if($limit > 0 && ++$count >= $limit) {
                last; 
            }
        }
    }
    if($rev) {
        foreach my $x (reverse(@cache)) {
            $proc->($x);
        }
    }
}

sub filter_and_each {
    my $self = shift;
    my ($op,$pg,$cond,$proc) = @_;
    my $e = sub { $self->eachfiles($self->{db},$op,$pg) };
    return $self->filter_and_eachfiles($e,$op,$pg,$cond,$proc);
}

sub is_resource {
    my $self = shift;
    my ($page) = @_;
    if($page =~ /\.(\w{1,4})$/) {
        my $e = lc($1);
        if(!defined($self->{ext})) {
            $self->{ext} = N2::Prefs->new(
                page => $self->{extnpage}, db => $self,
                rex => '^(\w+)\s*=\s*(.+)\s*$', #'
                _prefs => $self->{extns}
            );
        }
        return $self->{ext}->get($e);
    }
    else {
        return undef;
    }
}

sub respath {
    my $self = shift;
    my ($page) = @_;
    my $path = $self->as_pagename($page);
    $path =~ s/([^\/])/unpack('H2', $1)/eg;
    if($path ne '/') {
        return $self->{resdb}.$path;
    }
    else {
        return $self->{resdb};
    }
}

sub dirname {
    my $self = shift;
    my ($page) = @_;
    return File::Basename::dirname($self->as_pagename($page));
}

sub basename {
    my $self = shift;
    my ($page) = @_;
    return File::Basename::basename($self->as_pagename($page));
}

sub resname {
    my $self = shift;
    my ($page,$res) = @_;
    if($page eq '/') {
        return $page.$res;
    }
    else {
        return $page.'/'.$res;
    }
}

sub resuri {
    my $self = shift;
    my ($path,$page) = @_;
    $page = $self->as_pagename($page);
    my $dirname = $self->dirname($page);
    my $basename = $self->basename($page);
    my $uri = $self->respath($dirname);
    return $path.'/'.$uri.'/'.$basename;
}

sub res_get {
    my $self = shift;
    my ($key) = @_;
    my $fname = $self->resuri('.',$key);
    if(-f $fname) {
        return $self->read_res($fname);
    }
    else {
        return '';
    }
}

sub res_set {
    my $self = shift;
    my ($key,$data,$add) = @_;
    my $fname = $self->resuri('.',$key);
    return $self->write_res($fname,$data,$add);
}

sub res_remove {
    my $self = shift;
    my ($key) = @_;
    my $fname = $self->resuri('.',$key);
    $fname = 
        Ho::A->check_filename($fname,q([^&;\\`'"|*?~<>^()\[\]{}$\r\n]+));
    # '
    unlink($fname);
}

sub resdir_copy {
    my $self = shift;
    my ($src,$dst) = @_;
    my $srcd = './'.$self->respath($src);
    my $dstd = './'.$self->respath($dst);
    $srcd = Ho::A->check_filename($srcd,'[\w\/.]+');
    $dstd = Ho::A->check_filename($dstd,'[\w\/.]+');
    return system('/bin/cp','-r',$srcd,$dstd);
}

sub resdir_move {
    my $self = shift;
    my ($src,$dst) = @_;
    my $srcd = './'.$self->respath($src);
    my $dstd = './'.$self->respath($dst);
    $srcd = Ho::A->check_filename($srcd,'[\w\/.]+');
    $dstd = Ho::A->check_filename($dstd,'[\w\/.]+');
    return system('/bin/mv','-f',$srcd,$dstd);
}

sub resdir_swap {
    my $self = shift;
    my ($src,$dst) = @_;
    my $srcd = './'.$self->respath($src);
    my $dstd = './'.$self->respath($dst);
    $srcd = Ho::A->check_filename($srcd,'[\w\/.]+');
    $dstd = Ho::A->check_filename($dstd,'[\w\/.]+');
    my $fn = 'tmp'.time();
    my $dirname = $self->dirname($src);
    my $tmp = './'.$self->respath($dirname).'/'.$fn;
    $tmp = Ho::A->check_filename($tmp,'[\w\/.]+');
    system('/bin/mv','-f',$srcd,$tmp);
    system('/bin/mv','-f',$dstd,$srcd);
    system('/bin/mv','-f',$tmp,$dstd);
    system('/bin/rm','-rf',$tmp);
}

sub res_exists {
    my $self = shift;
    my ($key) = @_;
    my $fname = $self->resuri('.',$key);
    return (-f $fname);
}

sub resdir_exists {
    my $self = shift;
    my ($page) = @_;
    $page = $self->as_pagename($page);
    my $path = $self->respath($page);
    $path = Ho::A->check_filename($path,'[\w\/]+');
    return (-d $path);
}

sub res_prepare_path {
    my $self = shift;
    my ($page) = @_;
    $page = $self->as_pagename($page);
    my $dirname = $self->dirname($page);
    my $path = $self->respath($dirname);
    $path = Ho::A->check_filename($path,'[\w\/]+');
    if(not -d $path) {
        File::Path::mkpath($path,0,0777);
    }
}

sub res_get_info {
    my $self = shift;
    my ($key) = @_;
    my $fname = $self->resuri('.',$key);
    return (-f $fname) ? Ho::A->file_info($fname) : ( ) ;
}

sub res_moddate {
    my $self = shift;
    my ($key) = @_;
    my %info = $self->res_get_info($key);
    return $info{mod};
}

sub editdate {
    my $self = shift;
    my ($page) = @_;
    my $key = $self->resname($page,$self->{modfile});
    if(!$self->res_exists($key)) {
        $self->res_prepare_path($key);
        $self->res_set($key,"\n");
    }
    return $self->res_moddate($key);
}

sub set_editdate {
    my $self = shift;
    my ($page) = @_;
    my $key = $self->resname($page,$self->{modfile});
    my $fname = $self->resuri('.',$key);
    if(!$self->res_exists($key)) {
        $self->res_prepare_path($key);
        $self->res_set($key,"\n");
    }
    else {
        my $mod = time;
        $fname = 
         Ho::A->check_filename($fname,q([^&;\\`'"|*?~<>^()\[\]{}$\r\n]+)); #'
        utime($mod, $mod, $fname);
    }
}

sub filter_and_eachres {
    my $self = shift;
    my ($op,$pg,$cond,$proc) = @_;
    $pg = $self->as_pagename($pg);
    my $e = sub { $self->eachres($self->respath($pg),$op) };
    return $self->filter_and_eachfiles($e,$op,$pg,$cond,$proc);
}

sub filter_and_eachrestrash {
    my $self = shift;
    my ($op,$pg,$cond,$proc) = @_;
    $pg = $self->as_pagename($pg);
    my $e = sub { $self->eachres($self->resuri('.',$pg.'/trash'),$op) };
    return $self->filter_and_eachfiles($e,$op,$pg,$cond,$proc);
}

sub eachres {
    my $self = shift;
    my ($dir,$options) = @_;
    my @pages;
    opendir(DIR,$dir);
    @pages = grep {-f "$dir/$_"} readdir(DIR);
    closedir(DIR);
        
    if($options =~ /[MAC]/) {
        if($options =~ /M/) {
            @pages = map  { [$_, -M "$dir/$_"] } @pages;
        }
        elsif($options =~ /A/) {
            @pages = map  { [$_, -A "$dir/$_"] } @pages;
        }
        elsif($options =~ /C/) {
            @pages = map  { [$_, -C "$dir/$_"] } @pages;
        }
        @pages = map  { $_->[0] }
                 sort { $a->[1] <=> $b->[1] } @pages;
    }
    else {
        @pages = sort {$a cmp $b} @pages;
    }
    
    if($options =~ /r/) {
        @pages = reverse(@pages);
    }
    
    return @pages;
}

sub is_spambot {
    my $self = shift;
    my ($pg,$v) = @_;
    my $q = $v->{query};
    my $ip = $ENV{REMOTE_ADDR};
    if($self->is_denied_ip($ip)) { return 1; }
    if($v->spam_check($pg,$self)) {
        $self->{spamlist}->add($ip);
        if(defined($q)) { $q->spamlog(); }
        return 1;
    }
    return 0;
}

sub set_spambot {
    my $self = shift;
    my ($q) = @_;
    my $ip = $ENV{REMOTE_ADDR};
    if(!$self->is_denied_ip($ip)) {
        $self->{spamlist}->add($ip);
        if(defined($q)) { $q->spamlog(); }
    }
}

sub is_ngword {
    my $self = shift;
    my ($data,$pg) = @_;
    if(!defined($self->{ngwordlist})) {
        $self->{ngwordlist} = $self->init_spamres($self->{ngword});
    }
    my $ngw = $self->{ngwordlist}->get();
    if($pg eq $self->{ngword}) {
        return 0;
    }
    foreach my $r (@$ngw) {
        if($data =~ m/\Q$r\E/i) { return 1; }
    }
    return 0;
}

sub setspamatrb {
    my $self = shift;
    my ($pg,$rf,$ua) = @_;
    $self->{spampages}->add($pg);
    $self->{spamrefs}->add($rf);
    $self->{spamuas}->add($ua);
}

sub backup {
    my $self = shift;
    my ($page,$q) = @_;
    my $data = $self->get($page);
    my $fname = $self->resname($page,'backup'.time().'.bak');
    my @rec = $self->eachres($self->respath($page),'-M');
    my ($recent) = grep {$_ =~ /^backup[0-9]+\.bak$/} @rec ;
    my $rdata = $self->res_get($self->resname($page,$recent));
    if($data eq $rdata) { return ''; }
    $self->res_prepare_path($fname);
    $self->res_set($fname,$q->escape($data));
    return $fname;
}

sub latest_backup_utime {
    my $self = shift;
    my ($page) = @_;
    my $ut = 0;
    my @bak = $self->eachres($self->respath($page),'-M');
    @bak = grep {$_ =~ /^backup\d+\.bak$/} @bak ;
    if(@bak > 0) { ($ut) = $bak[0] =~ /^backup(\d+)\.bak$/; }
    return $ut;
}

sub autobackup {
    my $self = shift;
    my ($page,$view) = @_;
    if($view->{autobackup}) {
        my $ct = time();
#       my $et = $self->editdate($page);
        my $bt = $self->latest_backup_utime($page);
        my $delta = ($ct - $bt);
        my $delay = ($view->{backupdelay} * 60);
        if($delta > $delay) {
            $self->backup($page,$view->{query});
        }
    }
}

sub add_recentdeleted {
    my $self = shift;
    my ($page,$q) = @_;
    my $data = $q->curr_date_str()." [[$page]]\n";
    my $add = 1;
    if(!$self->exists_page($self->{recentdeleted})) {
        $data = "#?header\n#?body\n$data";
        $add = 0;
    }
    $self->set($self->{recentdeleted},$data,'',0,$add);
}

1;
