<?php
/**
 * モデル用の抽象クラスです。
 */
abstract class MyModel
{
	/** 主キー名 */
	protected $pk;

	/** MySQLラッパー */
	protected $db = null;

	/**
	 * コンストラクタです。
	 * @param		(MySqlWrapper)$db	MySQLラッパー
	 */
	public function __construct($db)
	{
		$this->db = $db;
	}

	/**
	 * コミットを実施します。
	 */
	public function commit()
	{
		$this->db->commit();
	}

	/**
	 * ロールバックを実施します。
	 */
	public function rollback()
	{
		$this->db->rollback();
	}

	/**
	 * コードと名称の一覧を取得します。
	 * [$order]には順序を指定しますが省略可能です。
	 * 省略時には"code ASC"となります。
	 * @param		(string)$decrypt	名前の復号化
	 * @param		(string)$order		ORDER BY句
	 * @return		(array)				コードと名称の配列
	 */
	public function getCodeName($decrypt = false, $order = 'code ASC')
	{
		$name = 'name';
		if ($decrypt) {
			$name = 'decrypt_' . $name;
		}
		$sql = "SELECT `code`, `name` AS `" . $name . "` FROM `" . get_class($this) . "`";
				if (isset($order) && strlen($order) > 0) {
			$sql .= " ORDER BY " . $order;
		}
		return $this->db->execSelectSql($sql);
	}

	/**
	 * コード値から主キーを取得します。
	 * 値が取得できなかった場合には<b>false</b>を返します。
	 * [$name]にはコードカラムの名前を指定しますが省略可能です。
	 * 省略時には"code"となります。
	 * @param		(string)$code		コード
	 * @param		(string)$name		コードカラム名
	 * @return		(int)				主キー
	 */
	public function code2pk($code, $name = 'code')
	{
		$sql = "SELECT `" . $this->pk . "` FROM `" . get_class($this) . "` WHERE `" . $name . "` = :code";
		$param = array('code' => $code);
		$assoc = $this->db->execSelectSql($sql, $param, true);
		if (isset($assoc[$this->pk])) {
			return $assoc[$this->pk];
		}
		return false;
	}

	/**
	 * コード値から名称を取得します。
	 * 値が取得できなかった場合には<b>false</b>を返します。
	 * [$name1]にはコードカラムの名前、[$name2]には名称カラムの名前を指定しますが省略可能です。
	 * 省略時には"code"、"name"となります。
	 * @param		(string)$code		コード
	 * @param		(string)$name1		コードカラム名
	 * @param		(string)$name2		名称カラム名
	 * @param		(bool)				復号化
	 * @return		(int)				名称
	 */
	public function code2name($code, $name1 = 'code', $name2 = 'name', $decrypt = false)
	{
		$alias = $name2;
		if ($decrypt) {
			$alias = 'decrypt_' . $alias;
		}
		$sql = "SELECT `{$name2}` AS `{$alias}` FROM `" . get_class($this) . "` WHERE `{$name1}` = :code";
		$param = array('code' => $code);
		$assoc = $this->db->execSelectSql($sql, $param, true);
		if (isset($assoc[$name2])) {
			return $assoc[$name2];
		}
		return false;
	}

	/**
	 * 連想配列の配列をhtml_options等で使えるコードと名称の連想配列に変換します。
	 * 実際には<b>code</b>と<b>name</b>という要素から作成するのでその要素が必要になります。
	 * @param		(array)$array		連想配列の配列
	 * @param		(bool)$blank		空行の作成要否
	 * @param		(bool)$decrypt		名前の復号化
	 * @return		(assoc)				連想配列
	 */
	public function getOptions($blank = false, $decrypt = false)
	{
		// 全レコードの取得
		$array = $this->getCodeName($decrypt);

		// 戻り値
		$assoc = array();

		// 配列のループ
		foreach ($array as $value) {
			$assoc[$value['code']] = $value['code'] . '：' . $value['name'];
		}

		// 空行を作成する
		if ($blank) {
			$assoc = array('' => '') + $assoc;
		}

		// 返す
		return $assoc;
	}

	/**
	 * 一覧を取得します。
	 * @return		(array)				一覧表示の配列
	 */
	public function getList($where = '', $param = null, $order = '')
	{
		$sql = $this->getListSql() . $where;
		if (strlen($order) > 0) {
			$sql .= " ORDER BY {$order}";
		}
		return $this->db->execSelectSql($sql, $param);
	}

	/**
	 * 一覧表示用のSQLを取得します。
	 * @return		(string)			一覧表示用SQL
	 */
	protected abstract function getListSql();

	/**
	 * 詳細を取得します。
	 * @param		(int)				プライマリキー
	 * @return		(assoc)				詳細の連想配列
	 */
	public function getDetail($pk)
	{
		$sql = $this->getListSql() . "AND `" . get_class($this) . "`.`" . $this->pk . "` = :pk";
		$param = array('pk' => $pk);
		return $this->db->execSelectSql($sql, $param, true);
	}
}