<?php
/**
 * コントローラ用の抽象クラスです。
 */
abstract class MyController
{
	/** コンフィグ */
	protected $config = null;
	/** ディレクトリ */
	private $_tplDir = null;
	/** テンプレートファイル */
	private $_tplFile = null;
	/** セット値 */
	private $_setValues = array();

	/** パラメータ */
	protected $param = null;

	/** Smartyオブジェクト */
	private $_smarty = null;
	/** 利用するデータベース */
	protected $dbName = 'default';
	/** DB接続リスト */
	protected $dbs = array();
	/** デフォルトDB接続 */
	protected $db = null;
	/** ロガー */
	protected $logger = null;
	/** セッション管理 */
	protected $session = null;
	/** 認証マネージャ */
	protected $authManager = null;
	/** モデル */
	protected $models = array();
	/** セッションの利用 */
	protected $useSession = false;
	/** ログインチェック */
	protected $authCheck = false;

	/**
	 * インスタンスの初期化を行います。
	 * このメソッドはmy.phpによってのみ呼び出されるべきです。
	 * @param		(assoc)$config		コンフィグ連想配列
	 * @return		(bool)				初期化成功
	 */
	public function init($config)
	{
		// コンフィグを格納
		$this->config = $config;

		// ロガーを開始する
		$this->logger = new MyLogger($config['log']);

		// GET/POST値の変換と取得
		$pt = new ParamTransrator($config['env']);
		$this->param = $pt->transrate($_GET);
		$this->param += $pt->transrate($_POST);
		$this->param += $pt->transrate(($_FILES));

		// Smartyラッパーを取得
		$this->_smarty = new SmartyWrapper($config);

		// MySQLラッパーを取得
		foreach ($config['db'] as $key => $value) {
			$this->dbs[$key] = new MySqlWrapper($config, $key);
		}
		$this->db = $this->dbs[$this->dbName];

		// 利用するモデルをロードしておく
		foreach ($this->models as $value) {
			$this->$value = new $value($this->db);
		}

		// セッションを使う場合セッションを開始しておく
		if ($this->useSession) {
			$this->session = new SessionManager();
		}

		// 認証チェック処理
		$this->authManager = new AuthManager($this->config['auth'], $this->db);
		if ($this->authCheck) {
			$this->logger->info('認証を開始しました。');
			if ($this->authManager->authorization() === false) {
				$this->logger->info('認証に失敗しました。');
				$this->redirect($this->config['auth']['failed_action'], $this->config['auth']['failed_controller']);
				return false;
			}
			$this->logger->info('認証に成功しました。');
		}

		// OKを返す
		return true;
	}

	/**
	 * 実行中のクラス名とメソッド名を取得します。
	 * @param		(string)$funcName	メソッド名
	 * @return		(string)			<clasName>.<methodName>
	 */
	protected function getClassMethod($funcName)
	{
		return get_class($this) . '.' . $funcName;
	}

	/**
	 * 不正実行ログを出力します。
	 * @param		(string)$funcName	メソッド名
	 * @param		(string)$uid		ユーザID
	 */
	protected function writeInvalidLog($funcName, $uid)
	{
		$classFunc = $this->getClassMethod($funcName);
		$this->logger->info("権限のないユーザによって実行されました。[{$funcName}], uid=[{$uid}]");
	}

	/**
	 * テンプレートディレクトリを設定します。
	 * @param		(string)$dir		テンプレートディレクトリ
	 */
	public function setTplDir($dir)
	{
		$this->_tplDir = $dir;
	}

	/**
	 * テンプレートファイル名を設定します。
	 * @param		(string)$file		テンプレートファイル名
	 */
	public function setTplFile($file)
	{
		$this->_tplFile = $file;
	}

	/**
	 * 内部Smartyのテンプレートを表示します。
	 */
	public function show()
	{
		$this->set('TITLE', $this->TITLE, true);
		$this->set('BASE_URL', BASE_URL, true);
		$this->set('CONTEXT_ROOT', CONTEXT_ROOT, true);
		foreach ($this->_setValues as $key => $value) {
			$this->_smarty->assign($key, $value);
		}
		$this->_smarty->display('file:' . $this->_tplDir . DS . $this->_tplFile);
	}

	/**
	 * 内部Smartyに値をセットします。
	 * @param		(string)$key		キー
	 * @param		(mixed)$value		値
	 * @param		(bool)$override		値が存在する場合の上書き
	 */
	protected function set($key, $value, $override = false)
	{
		if ($override) {
			$this->_setValues[$key] = $value;
		}
		else {
			if (!isset($this->_setValues[$key])) {
				$this->_setValues[$key] = $value;
			}
		}
	}

	/**
	 * 指定アクションにリダイレクトします。
	 * @param		(string)$action		アクション名
	 * @param		(string)$controller	コントローラ名
	 */
	protected function redirect($action, $controller = '')
	{
		$location = $action;
		if (strlen($controller) > 0) $location = BASE_URL . '/' . $controller . '/';
		header('Location: ' . $location);
	}

	/**
	 * AJAXとして動作した場合に値を返して終了します。
	 * @param		(string)$value		返す値
	 * @return		(bool)				false
	 */
	protected function ajaxReturn($value)
	{
		echo mb_convert_encoding($value, $this->config['env']['tpl_encode'], $this->config['env']['php_encode']);
		return false;
	}

	/**
	 * AJAXとして動作した場合に空の結果を返して終了します。
	 * @return		(bool)				false
	 */
	protected function ajaxNoReturn()
	{
		return $this->ajaxReturn('');
	}
}