#!/usr/bin/env bash

################################################################################
# xtrfm
# RFM分析
# 2.0 シナリオバージョン新規作成 2004/11/09 Y.Hamuro
################################################################################

#===============================================================================
#ヘルプメッセージ( -hが与えられたときに表示される )
#-------------------------------------------------------------------------------
# -hが与えられたときに表示される。
# "cat >/dev/stderr <<EOR"から"EOF"の行までのテキストがそのまま画面に表示される
#===============================================================================
version="2.0"

function help {
cat >/dev/stderr <<EOF
---------------------
MUSASHI-SCENARIO
mssRFM.sh version 2.0
---------------------
概要) RFM分析
環境変数) 以下の環境変数を設定した後にこのコマンドを起動する。
  InDat		顧客購買履歴データファイルを指定する
  OutDatDir	結果データを保存するディレクトリを指定する
  OutHtmlDir	HTMLによる結果データを保存するディレクトリを指定する
  CustFldName	InDat上の顧客ID項目名(もしくは顧客IDに相当する項目名)
  DateFldName	InDat上の日付項目名
  SelPeriodFrom	InDatで利用するレコードの日付選択開始条件
  SelPeriodTo	InDatで利用するレコードの日付選択終了条件
  MoneyFldName	InDat上の金額項目名(Monetaryを計算する項目)
  DateFldName 	Recencyを計算するための基準日(Recency=基準日-最近来店日)
		latestを指定すればInDat上で観測されるもっとも新しい日付
		todayを指定すれば本日
  RecencyCnt	Recencyをいくつのクラスに分割するか
  FreqCnt	Frequencyをいくつのクラスに分割するか
  MonetaryCnt	Monetaryをいくつのクラスに分割するか
  BucketMethodR	Recencyのクラス分割基準(cnt:件数均等化分割,rng:範囲均等化分割)
  BucketMethodF	Frequencyのクラス分割基準(cnt:件数均等化分割,rng:範囲均等化分割)
  BucketMethodM	Monetaryのクラス分割基準(cnt:件数均等化分割,rng:範囲均等化分割)
  RangeR	BucketMethodR="usr"の時のユーザ指定による範囲
  RangeF	BucketMethodF="usr"の時のユーザ指定による範囲
  RangeM	BucketMethodM="usr"の時のユーザ指定による範囲
  mssQuiet	途中経過の完了メッセージを出力するなら0、そうでなければ1を指定
  TmpDir	作業ディレクトリ
  HtmlComment	生成されるindex.htmlのトップに表示されるコメントの内容
EOF
exit 1
}

#完了メッセージ用にコマンドラインを保存
cmdLine="$0 $*"

#コマンド名を保存
cmdName="$0"

#シグナルによる終了
function endByInt {
  rm -f $TD-xx*
  echo "#ERROR# $$" \"$cmdLine\" \""end by signal(ctr^C)"\" >/dev/stderr
  exit 1
}

#-------------------------------------------------------------------------------
# 環境変数のチェック
#-------------------------------------------------------------------------------
if [ "$1" = "-h" ] ; then
  help
fi

function checkNull {
if eval [ -z \"\$$1\" ] ; then
  echo "#ERROR# $$" \"$0\" \""environment variable [$1] is mandatory"\" >/dev/stderr
  exit 1
fi
}

echo "#MSG# $$" "$cmdName" Checking Environment Variable >/dev/stderr

checkNull InDat
checkNull OutDatDir
checkNull CustFldName
checkNull DateFldName
checkNull MoneyFldName
checkNull DateFldName 
checkNull RecencyCnt
checkNull FreqCnt
checkNull RecencyCnt
checkNull BucketMethodR
checkNull BucketMethodF
checkNull BucketMethodM
checkNull TmpDir

if [ "$BucketMethodR" = "usr" ] ; then
  checkNull RangeR
fi
if [ "$BucketMethodF" = "usr" ] ; then
  checkNull RangeF
fi
if [ "$BucketMethodM" = "usr" ] ; then
  checkNull RangeM
fi

#-------------------------------------------------------------------------------
# 各種前処理
#-------------------------------------------------------------------------------
# ワークファイルのプレフィックスの設定
# $$はプロセス番号
wf=$TmpDir/xtrfm-$$

# 出力先ディレクトリ作成
mkdir -p $OutDatDir

#トラップ発動
trap endByInt INT QUIT TERM HUP

#-------------------------------------------------------------------------------
# メイン処理処理
#-------------------------------------------------------------------------------
# 1. 入力データを併合し、
# 2. 必要な項目(顧客,日付,金額)を抜きだし、
# 3. null値を含む行を削除し
# 4. データを選択し、
# 5. 顧客別日別金額合計を求める。
echo "#MSG# $$" "$cmdName" Creating Basic Data for RFM analysis >/dev/stderr

if [ -z "$SelPeriodFrom" ]  && [ -z "$SelPeriodTo" ] ; then
  xtcat -i "$InDat" |
  xtdelnul -f $CustFldName,$DateFldName,$MoneyFldName |
  xtcut -f $CustFldName,$DateFldName,$MoneyFldName |
  xtagg -k $CustFldName,$DateFldName -f $MoneyFldName -c sum -o $wf-xxbase
else
  if [ -z "$SelPeriodFrom" ] ; then
    SelPeriodFrom="00000000"
  fi
  if [ -z "$SelPeriodTo" ] ; then
    SelPeriodTo="99999999"
  fi
  xtcat -i "$InDat" |
  xtdelnul -f $CustFldName,$DateFldName,$MoneyFldName |
  xtsel -c "\$$DateFldName>$SelPeriodFrom && \$$DateFldName<$SelPeriodTo" |
  xtcut -f $CustFldName,$DateFldName,$MoneyFldName |
  xtagg -k $CustFldName,$DateFldName -f $MoneyFldName -c sum -o $wf-xxbase
fi

#来店頻度クラスを求める
echo "#MSG# $$" "$cmdName" Calculating Frequency >/dev/stderr

xtcut -f $CustFldName -i $wf-xxbase |
xtcount -k $CustFldName -a Frequency -o $wf-xxf

if [ "$BucketMethodF" = "usr" ] ; then
  echo "$RangeF" | grep -v '^$' |
  txt2xt -a bucketNo,rangeFrom,rangeTo -o  $wf-xxrngF
  xtnrjoin -N -v Frequency%n -m $wf-xxrngF -R rangeFrom,rangeTo -f bucketNo:Fclass -i $wf-xxf -o $wf-xxF
else
  xtbucket -n $FreqCnt -c $BucketMethodF -f Frequency:Fclass -i $wf-xxf -o $wf-xxF -O $wf-xxrngF
fi

#金額クラスを求める
echo "#MSG# $$" "$cmdName" Calculating Monetary >/dev/stderr

xtcut -f $CustFldName,$MoneyFldName -i $wf-xxbase |
xtagg -k $CustFldName -f $MoneyFldName:Monetary -c sum -o $wf-xxm

if [ "$BucketMethodM" = "usr" ] ; then
  echo "$RangeM" | grep -v '^$' |
  txt2xt -a bucketNo,rangeFrom,rangeTo -o $wf-xxrngM
  xtnrjoin -N -v Monetary%n -m $wf-xxrngM -R rangeFrom,rangeTo -f bucketNo:Mclass -i $wf-xxm -o $wf-xxM
else
  xtbucket -n $MonetaryCnt -c $BucketMethodM -f Monetary:Mclass -i $wf-xxm -o $wf-xxM -O $wf-xxrngM
fi

#Recencyクラスを求める
echo "#MSG# $$" "$cmdName" Calculating Recency >/dev/stderr

if [ "$BaseDate" = "today" ] ; then
  BaseDate=$(date +%Y%m%d)
fi
if [ "$BaseDate" = "latest" ] ; then
  BaseDate=$(xtcut -f $DateFldName -i $wf-xxbase | xtagg -f $DateFldName -c max | xt2txt)
fi

xtsetchr -v "$BaseDate" -a today -i $wf-xxbase |
xtcut -f $CustFldName,$DateFldName,today |
xtagg -k $CustFldName -f $DateFldName -c max |
xtcal -c 'day($today,$'$DateFldName')' -a Recency -o $wf-xxr

if [ "$BucketMethodR" = "usr" ] ; then
  echo "$RangeR" | grep -v '^$' |
  txt2xt -a bucketNo,rangeFrom,rangeTo -o $wf-xxrngR
  xtnrjoin -N -v Recency%n -m $wf-xxrngR -R rangeFrom,rangeTo -f bucketNo:Rclass -i $wf-xxr -o $wf-xxR
else
  xtbucket -r -n $RecencyCnt -c $BucketMethodR -f Recency:Rclass -i $wf-xxr -o $wf-xxR -O $wf-xxrngR
fi

echo "#MSG# $$" "$cmdName" Output Final Results >/dev/stderr
#三つのファイルを結合
xtjoin -nN -k $CustFldName -m $wf-xxF -f Frequency,Fclass -i $wf-xxR |
xtjoin -nN -k $CustFldName -m $wf-xxM -f Monetary,Mclass |
xtnulto -f Rclass,Fclass,Mclass -v 0 |
xtcut -f $CustFldName,Recency,Frequency,Monetary,Rclass,Fclass,Mclass -o $OutDatDir/dat.xt

#クラス別件数表の作成
xtcut -f Rclass,Fclass,Mclass -i $OutDatDir/dat.xt |
xtcount -k Rclass,Fclass,Mclass -a 人数 -o $OutDatDir/RFMcnt.xt

#Recency範囲表の作成
xtcut -f Rclass,人数 -i $OutDatDir/RFMcnt.xt |
xtagg -k Rclass -f 人数 -c sum |
xtjoin -k Rclass -m $wf-xxrngR -K bucketNo -f rangeFrom,rangeTo |
xtcut -f Rclass,rangeFrom:開始範囲,rangeTo:終了範囲,人数 -o $OutDatDir/RecencyRng.xt


#Frequency範囲表の作成
xtcut -f Fclass,人数 -i $OutDatDir/RFMcnt.xt |
xtagg -k Fclass -f 人数 -c sum |
xtjoin -k Fclass -m $wf-xxrngF -K bucketNo -f rangeFrom,rangeTo |
xtcut -f Fclass,rangeFrom:開始範囲,rangeTo:終了範囲,人数 -o $OutDatDir/FrequencyRng.xt

#Monetary範囲表の作成
xtcut -f Mclass,人数 -i $OutDatDir/RFMcnt.xt |
xtagg -k Mclass -f 人数 -c sum |
xtjoin -k Mclass -m $wf-xxrngM -K bucketNo -f rangeFrom,rangeTo |
xtcut -f Mclass,rangeFrom:開始範囲,rangeTo:終了範囲,人数 -o $OutDatDir/MonetaryRng.xt

#====================
# HTMLファイル生成
#====================
if eval [ ! -z "$OutHtmlDir" ] ; then

echo "#MSG# $$" "$cmdName" Creating HTML documents >/dev/stderr

# 出力先ディレクトリ作成
mkdir -p $OutHtmlDir

# 各種データをCSVに変換
xt2csv -F -i $OutDatDir/dat.xt -o $OutHtmlDir/dat.csv

xt2csv -F -i $OutDatDir/RFMcnt.xt -o $OutHtmlDir/RFMcnt.csv

xt2csv -F -i $OutDatDir/RecencyRng.xt -o $OutHtmlDir/RecencyRng.csv
xt2html -k Rclass -R 開始範囲,終了範囲,人数 -i $OutDatDir/RecencyRng.xt -o $OutHtmlDir/RecencyRng.html

xt2csv -F -i $OutDatDir/FrequencyRng.xt -o $OutHtmlDir/FrequencyRng.csv
xt2html -k Fclass -R 開始範囲,終了範囲,人数 -i $OutDatDir/FrequencyRng.xt -o $OutHtmlDir/FrequencyRng.html

xt2csv -F -i $OutDatDir/MonetaryRng.xt -o $OutHtmlDir/MonetaryRng.csv
xt2html -k Mclass -R 開始範囲,終了範囲,人数 -i $OutDatDir/MonetaryRng.xt -o $OutHtmlDir/MonetaryRng.html

# RFM別人数分布グラフ
xtbar -k Rclass%n -r Fclass%n -c Mclass%n -f 人数 -i $OutDatDir/RFMcnt.xt -o $OutHtmlDir/FMRbar.svg --title 'FM別に見たRecencyの人数分布' --fix
xtbar -k Fclass%n -r Mclass%n -c Rclass%n -f 人数 -i $OutDatDir/RFMcnt.xt -o $OutHtmlDir/MRFbar.svg --title 'MR別に見たFrequencyの人数分布' --fix
xtbar -k Mclass%n -r Rclass%n -c Fclass%n -f 人数 -i $OutDatDir/RFMcnt.xt -o $OutHtmlDir/RFMbar.svg --title 'RF別に見たMonetaryの人数分布' --fix

# R,F,M別人数分布グラフ
xtagg -k Rclass,Fclass -f 人数 -c sum -i $OutDatDir/RFMcnt.xt |
xtbar -k Rclass%n -c Fclass%n -f 人数 -o $OutHtmlDir/FRbar.svg --title 'F別に見たRecencyの人数分布' -H 400 --fix
xtagg -k Rclass,Mclass -f 人数 -c sum -i $OutDatDir/RFMcnt.xt |
xtbar -k Rclass%n -c Mclass%n -f 人数 -o $OutHtmlDir/MRbar.svg --title 'M別に見たRecencyの人数分布' -H 400 --fix
xtagg -k Fclass,Mclass -f 人数 -c sum -i $OutDatDir/RFMcnt.xt |
xtbar -k Fclass%n -c Mclass%n -f 人数 -o $OutHtmlDir/MFbar.svg --title 'M別に見たFrequencyの人数分布' -H 400 --fix
xtagg -k Fclass,Rclass -f 人数 -c sum -i $OutDatDir/RFMcnt.xt |
xtbar -k Fclass%n -c Rclass%n -f 人数 -o $OutHtmlDir/RFbar.svg --title 'R別に見たFrequencyの人数分布' -H 400 --fix
xtagg -k Mclass,Rclass -f 人数 -c sum -i $OutDatDir/RFMcnt.xt |
xtbar -k Mclass%n -c Rclass%n -f 人数 -o $OutHtmlDir/RMbar.svg --title 'R別に見たMonetaryの人数分布' -H 400 --fix
xtagg -k Mclass,Fclass -f 人数 -c sum -i $OutDatDir/RFMcnt.xt |
xtbar -k Mclass%n -c Fclass%n -f 人数 -o $OutHtmlDir/FMbar.svg --title 'F別に見たMonetaryの人数分布' -H 400 --fix

# R,F,Mの分布グラフ作成

# 顧客人数
CustCnt=`xtcut -f $CustFldName -i $OutDatDir/dat.xt |
xtcount -a 顧客人数 |
xtcut -f 顧客人数 |
xt2txt`

# 作成日付
update=`date`

# index.html
cat >$OutHtmlDir/index.html <<EOF
<html>
<H1>RFM分析結果</H1>
<p>作成日時:${update} (version:$version)
<HR>
${HtmlComment}
<HR>
<H2>基本データ</H2>
<p>
顧客人数:<B>${CustCnt}</B>人
<p>RFMクラス分割数:<BR>
&nbsp;&nbsp;Recency:<B>${RecencyCnt}</B><BR>
&nbsp;&nbsp;Frequency:<B>${FreqCnt}</B><BR>
&nbsp;&nbsp;Monetary:<B>${MonetaryCnt}</B><BR>

<p>RFMクラス分割基準:(cnt:件数均等化, rng:範囲均等化, usr:ユーザ指定)<BR>
&nbsp;&nbsp;Recency:<B>${BucketMethodR}</B><BR>
&nbsp;&nbsp;Frequency:<B>${BucketMethodF}</B><BR>
&nbsp;&nbsp;Monetary:<B>${BucketMethodM}</B><BR>

<H2>RFMクラス範囲表</H2>
&nbsp;&nbsp;<a href="./RecencyRng.html">Recencyクラスの範囲表</a>(<a href="./RecencyRng.csv">CVS形式</a>)<BR>
&nbsp;&nbsp;<a href="./FrequencyRng.html">Frequencyクラスの範囲表</a>(<a href="./FrequencyRng.csv">CVS形式</a>)<BR>
&nbsp;&nbsp;<a href="./MonetaryRng.html">Monetaryクラスの範囲表</a>(<a href="./MonetaryRng.csv">CVS形式</a>)
<p><a href="./dat.csv">顧客別RFMクラス表(CSV形式)</a>
<H2>RFMの人数分布</H2>
<table CELLPADDING=3 BORDER=2>
<TR ALIGN="center">
<TH BGCOLOR="#00FFFF">Recency</TH>
<TH BGCOLOR="#00FFFF">Frequency</TH>
<TH BGCOLOR="#00FFFF">Monetary</TH>
</TR>
<TR ALIGN="center">
<TD><a href="./FMRbar.svg">FM別</a></TD>
<TD><a href="./MRFbar.svg">MR別</a></TD>
<TD><a href="./RFMbar.svg">RF別</a></TD>
</TR>
<TR ALIGN="center">
<TD><a href="./FRbar.svg">F別</a><BR><a href="./MRbar.svg">M別</a></TD>
<TD><a href="./MFbar.svg">M別</a><BR><a href="./RFbar.svg">R別</a></TD>
<TD><a href="./RMbar.svg">R別</a><BR><a href="./FMbar.svg">F別</a></TD>
</TR>
</table>
<p><a href="./RFMcnt.csv">RFM人数分布表(CSV形式)</a>
</html>
EOF

fi

#-------------------------------------------------------------------------------
# 後処理
#-------------------------------------------------------------------------------
# ワークファイルの削除
rm $wf-xx*

#完了メッセージ表示
echo "#END# $$" \"$cmdLine\" >/dev/stderr
exit 0
#===============================================================

