/*
 * ext2_fs.c
 *
 * Copyright 2003, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * EXT2ե륷ƥɥ饤С
 *
 */


#include"types.h"
#include"lib.h"
#include"mm.h"
#include"fs.h"
#include"device.h"
#include"test.h"


enum{
	BYTE_BIT=8,		/* 1ХȤΥӥåȿ */
};

/***********************************************************************
 *
 * < Common class >
 *
 ***********************************************************************/

/*****************************************

sample block location.

offset #	of blocks	description
--------	-----------	-----------
0			1			boot record
-- block group 0 --
(1024 bytes)1			superblock
2			1			group descriptors
3			1			block bitmap
4			1			inode bitmap
5			214			inode table
219			7974		data blocks
-- block group 1 --
8193		1			superblock backup
8194		1			group descriptors backup
8195		1			block bitmap
8196		1			inode bitmap
8197		214			inode table
8408		7974		data blocks
-- block group 2 --
16385		1			block bitmap
16386		1			inode bitmap
16387		214			inode table
16601		3879		data blocks

******************************************/

/*
 * super block.
 */
enum{
	EXT2_BLOCK_BASE=1024,			/* ֥å׻뤿Υ١Хȿ */
	EXT2_BLOCK_BASE_LOG=10,			/* ١֥åΣ򺬤Ȥп */

	EXT2_SUPER_BLK_POSITION=1024,	/* ѡ֥åΰ֡ƬΥХȿˡ */
	
	EXT2_GROUP_DESC_BLK=1,	/* group descriptor block number. */

	/* File system states at "s_state". */
	EXT2_VALID_FS=0x0001,			/* Unmounted cleanly */
	EXT2_ERROR_FS=0x0002,			/* Errors detected */

	/* Maximal mount counts. */
	EXT2_DFL_MAX_MNT_COUNT=20,
	EXT2_DFL_CHECKINTERVAL=0,

	/* Magic number at "s_magic". */
	EXT2_SUPER_MAGIC=0xEF53,

	/* Behaviour when detecting errors. */
	EXT2_ERRORS_CONTINUE=1,			/* continue as if nothing happened. */
	EXT2_ERRORS_RO=2,				/* remount read-only. */
	EXT2_ERRORS_PANIC=3,			/* cause a kernel panic. */
	EXT2_ERRORS_DEFAULT=EXT2_ERRORS_CONTINUE,

	/* revision level value. */
	EXT2_GOOD_OLD_REV=0,			/* original format. */
	EXT2_DYNAMIC_REV=1,				/* V2 format with dynamic inode sizes. */

	/* EXT2_*_INO values at "s_first_ino". */
	EXT2_BAD_INO=0x01, 				/* bad blocks inode */
	EXT2_ROOT_INO=0x02, 			/* root directory inode */
	EXT2_ACL_IDX_INO=0x03, 			/* ACL index inode (deprecated?) */
	EXT2_ACL_DATA_INO=0x04, 		/* ACL data inode (deprecated?) */
	EXT2_BOOT_LOADER_INO=0x05,		/* boot loader inode */
	EXT2_UNDEL_DIR_INO=0x06,		/* undelete directory inode */
};


typedef struct{
	uint s_inodes_count;			/* 00,total number of inodes. */
	uint s_blocks_count;			/* 04,total number of blocks. */
	uint s_r_blocks_count;			/* 08,total number of blocks reserved for the usage of the super user. */
	uint s_free_blocks_count;		/* 0c,total number of free blocks. */
	uint s_free_inodes_count;		/* 10,total number of free inodes. */
	uint s_first_data_block;		/* 14,the block containing the superblock structure.block size larger than 1KB=0,block size of 1KB=1. */
	uint s_log_block_size;			/* 18,number of bits to shift left the value 1024 for computing block size. */
	int s_log_frag_size;			/* 1c,number of bits to shift the value 1024 for computing fragment size. */
	uint s_blocks_per_group;		/* 20,total number of blocks per group. */
	uint s_frags_per_group;			/* 24,total number of fragments per group. */
	uint s_inodes_per_group;		/* 28,total number of inodes per group. */
	uint s_mtime;					/* 2c,Unix time,last time the file system was mounted. */
	uint s_wtime;					/* 30,Unix time,last write access to the file system. */
	ushort s_mnt_count;				/* 34,how many time the file system was mounted. */
	short s_max_mnt_count;			/* 36,the maximum number of mount times. */
	ushort s_magic;					/* 38,magic number 0xEF53. */
	ushort s_state;					/* 3a,file system state. */
	ushort s_errors;				/* 3c,behaviour when an error is detected. */
	ushort s_minor_rev_level;		/* 3e,minor revision level. */
	uint s_lastcheck;				/* 40,Unix time,last file system check. */
	uint s_checkinterval;			/* 44,between file system checks. */
	uint s_creator_os;				/* 48,os that created the file system. */
	uint s_rev_level;				/* 4c,revision level value. */
	ushort s_def_resuid;			/* 4e,default user id for reserved blocks. */
	ushort s_def_resgid;			/* 50,default group id for reserved blocks. */

	/* EXT2_DYNAMIC_REV Specific */
	uint s_first_ino;				/* 54,index to the first inode useable for standard files. */
	ushort s_inode_size;			/* 58,size of the inode structure. */
	ushort s_block_group_nr;		/* 5a,block group number hosting this superblock. */
	uint s_feature_compat;			/* 5c,bitmask of compatible features. */
	uint s_feature_incompat;		/* 60,bitmask of incompatible features. */
	uint s_feature_ro_compat;		/* 64,bitmask of read-only features. */
	uchar s_uuid[16];				/* 68,volume id. */
	char s_volume_name[16];			/* 78,volume name,mostly unusued. */
	char s_last_mounted[64];		/* 88,last mounted directory path,not normally used. */
	uint s_algo_bitmap;				/* compression algorithms. */

	/* Performance Hints */
	uchar s_prealloc_blocks;		/* nmber of blocks to try to preallocate. */
	uchar s_prealloc_dir_blocks;	/* number to preallocate for dirs. */
	ushort s_padding1;

	/* Journaling Support */
	uchar s_journal_uuid[16];		/*  */
	uint s_journal_inum;			/*  */
	uint s_journal_dev;				/*  */
	uint s_last_orphan;				/*  */

	/* ꡼¸ȼ */
	int blockSize;					/* ֥å */
	int sectorSize;					/*  */
	int sectors;					/* 1֥åΥ */
}SUPER_BLOCK;


/*
 * groupe descriptor.
 */
typedef struct{
	uint bg_block_bitmap;			/* the first block of the "block bitmap". */
	uint bg_inode_bitmap;			/* the first block of the "inode bitmap". */
	uint bg_inode_table;			/* the first block of the "inode table". */
	ushort bg_free_blocks_count;	/* total number of free blocks. */
	ushort bg_free_inodes_count;	/* total number of free inodes. */
	ushort bg_used_dirs_count;		/* number of inodes allocated to directories. */
	ushort bg_pad;
	uint bg_reserved[3];
}GROUP_DESC;


/* ޥȻ롣 */
static SUPER_BLOCK *superBlock[MAX_DEVICE_OPEN];
static GROUP_DESC *groupDesc[MAX_DEVICE_OPEN];
static int groupDescSize[MAX_DEVICE_OPEN];			/* ֥åñ̤Υ */


/***********************************************************************
 *
 * < Block class >
 *
 ***********************************************************************/

/* PRIVATE,ӥåȥޥåפǥХ֥ͤåΰ֤롣 */
static const uchar emptyBlock[]={
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,5,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,6,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,5,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,7,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,5,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,6,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,5,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,0,
};

static uchar *blockBitmap[MAX_DEVICE_OPEN];		/* PRIVATE ӥåȥޥåס */
static int currentBitmap[MAX_DEVICE_OPEN];		/* ߤζӥåȥޥåפΥХȥեåȡ */
static int currentGd[MAX_DEVICE_OPEN];			/* PUBLIC ߤΥ롼ץǥץ */


/*
 * PUBLIC
 * ֥å饻򻻽Ф롣
 * parameters : device inode,block
 * return : secter number
 */
static inline uint blockToSect(int din,uint blk)
{
	return superBlock[din]->blockSize*blk/superBlock[din]->sectorSize;
}


/*
 * PUBLIC
 * ֥å롣
 * parameters : device inode
 * return : empty block or error=0;
 */
static uint getBlock(int din)
{
	GROUP_DESC *gd=groupDesc[din];
	int block;
	int i,last;


	/* ߤΥ롼פ˶֥åʤ */
	if(gd[currentGd[din]].bg_free_blocks_count==0)
	{
		int sectors=superBlock[din]->sectors;

		for(i=currentGd[din]+1;;++i)
		{
			if(gd[i].bg_block_bitmap==0)i=0;
			if(i==currentGd[din])return 0;
			if(gd[i].bg_free_blocks_count!=0)break;
		}
		if(read_cache(din,blockBitmap[din],sectors,gd[i].bg_block_bitmap)!=sectors)
			return -1;
		currentGd[din]=i;
	}

	/* ӥåȥޥåפθ */
	i=currentBitmap[din];
	last=superBlock[din]->s_blocks_per_group/BYTE_BIT;
	while(blockBitmap[din][i]==0xff)
		if(++i>=last)i=0;
	currentBitmap[din]=i;
	block=i*BYTE_BIT+emptyBlock[blockBitmap[din][i]];
	blockBitmap[din][i]|=1<<emptyBlock[blockBitmap[din][i]];

	return block+currentGd[din]*superBlock[din]->s_blocks_per_group+superBlock[din]->s_first_data_block;
}


/*
 * PUBLIC
 * ֥å֤
 */
static void releaseBlock(int din,uint block)
{
	uchar *bitmap;
	int gd;
	SUPER_BLOCK *sb=superBlock[din];
	int a;


	gd=(block-sb->s_first_data_block)/(sb->s_blocks_per_group);
	if(gd!=currentGd[din])
	{
		if((bitmap=(uchar*)kmalloc(sb->blockSize))==NULL)return;
		if(read_cache(din,bitmap,blockToSect(din,groupDesc[din]->bg_block_bitmap),sb->sectors)!=sb->sectors)
			return;
	}
	else bitmap=blockBitmap[din];
	a=block-gd*sb->s_blocks_per_group-sb->s_first_data_block;
	bitmap[a/BYTE_BIT]&=~(1<<a%BYTE_BIT);
	if(gd!=currentGd[din])
	{
		if(write_cache(din,bitmap,blockToSect(din,groupDesc[din]->bg_block_bitmap),sb->sectors)!=sb->sectors)
			return;
		kfree(bitmap);
	}
}


/*
 * PUBLIC
 * ޥȻν
 * parameters : device inode
 * return : groupe descriptor number or error=-1
 */
static int initBlock(int din)
{
	int i;
	GROUP_DESC *gd;


	if((blockBitmap[din]=(uchar*)kmalloc(superBlock[din]->blockSize))==NULL)return -1;

	i=0;
	for(gd=groupDesc[din];gd[i].bg_block_bitmap!=0;++i)
		if(gd[i].bg_free_blocks_count>0)
			if(read_direct(din,blockBitmap[din],superBlock[din]->sectors,
				blockToSect(din,gd[i].bg_block_bitmap))!=superBlock[din]->sectors)return -1;
	currentGd[din]=i;

	for(i=0;blockBitmap[din][i]==0xff;++i);
	currentBitmap[din]=i;

	return i;
}


/*
 * PUBLIC
 * ޥȻν
 * parameters : device inode
 */
static void closeBlock(int din)
{
	kfree(blockBitmap[din]);
	blockBitmap[din]=NULL;
}


/***********************************************************************
 *
 * < Inode class >
 *
 ***********************************************************************/

/*
 * inode definition
 */
enum{
	EXT2_ROOT_INODE_INDEX=2,		/* 롼inodeΥǥå */

	/* at i_mode */
	EXT2_S_IFMT=0xF000,		/* format mask */
	EXT2_S_IFSOCK=0xC000,	/* socket */
	EXT2_S_IFLNK=0xA000,	/* symbolic link */
	EXT2_S_IFREG=0x8000,	/* regular file */
	EXT2_S_IFBLK=0x6000,	/* block device */
	EXT2_S_IFDIR=0x4000,	/* directory */
	EXT2_S_IFCHR=0x2000,	/* character device */
	EXT2_S_IFIFO=0x1000,	/* fifo */
	EXT2_S_ISUID=0x0800,	/* SUID */
	EXT2_S_ISGID=0x0400,	/* SGID */
	EXT2_S_ISVTX=0x0200,	/* sticky bit */
	EXT2_S_IRWXU=0x01C0,	/* user access rights mask */
	EXT2_S_IRUSR=0x0100,	/* read */
	EXT2_S_IWUSR=0x0080,	/* write */
	EXT2_S_IXUSR=0x0040,	/* execute */
	EXT2_S_IRWXG=0x0038,	/* group access rights mask */
	EXT2_S_IRGRP=0x0020,	/* read */
	EXT2_S_IWGRP=0x0010,	/* write */
	EXT2_S_IXGRP=0x0008,	/* execute */
	EXT2_S_IRWXO=0x0007,	/* others access rights mask */
	EXT2_S_IROTH=0x0004,	/* read */
	EXT2_S_IWOTH=0x0002,	/* write */
	EXT2_S_IXOTH=0x0001,	/* execute */

	/* behaviour control flags at "i_flags". */
	EXT2_SECRM_FL=0x00000001,		/* secure deletion */
	EXT2_UNRM_FL=0x00000002,		/* record for undelete */
	EXT2_COMPR_FL=0x00000004,		/* compressed file */
	EXT2_SYNC_FL=0x00000008,		/* synchronous updates */
	EXT2_IMMUTABLE_FL=0x00000010,	/* immutable file */
	EXT2_APPEND_FL=0x00000020,		/* append only */
	EXT2_NODUMP_FL=0x00000040,		/* do not dump/delete file */
	EXT2_NOATIME_FL=0x00000080,		/* do not update .i_atime */
	EXT2_DIRTY_FL=0x00000100,		/* dirty (file is in use?) */
	EXT2_COMPRBLK_FL=0x00000200,	/* compressed blocks */
	EXT2_NOCOMPR_FL=0x00000400,		/* access raw compressed data */
	EXT2_ECOMPR_FL=0x00000800,		/* compression error */
	EXT2_BTREE_FL=0x00010000,		/* b-tree format directory */
	EXT2_INDEX_FL=0x00010000,		/* Hash indexed directory */

	/* indirect data block index. */
	INODE_DATA_INDIRECT1=12,		/* ܻȥ֥å */
	INODE_DATA_INDIRECT2=13,		/* Ŵܻȥ֥å */
	INODE_DATA_INDIRECT3=14,		/* Ŵܻȥ֥å */

	INODE_DATA_BLOCK_NUM=15,		/* ǡ֥å */
};


typedef struct{
	ushort i_mode;			/* 00,format of the described file and the access rights. */
	ushort i_uid;			/* 02,user id. */
	uint i_size;			/* 04,size of the file in bytes. */
	uint i_atime;			/* 08,Unix time,last time this file was accessed. */
	uint i_ctime;			/* 0c,Unix time,when file was created. */
	uint i_mtime;			/* 10,Unix time,last time this file was modified. */
	uint i_dtime;			/* 14,Unix time,when file was deleted,this value is always 0. */
	ushort i_gid;			/* 18,group id. */
	ushort i_links_count;	/* 1a,how many times this inode is linked. */
	uint i_blocks;			/* 1c,both currently in used and currently reserved blocks. */
	uint i_flags;			/* 20,behave when accessing the data for this inode. */
	uint i_number;			/* 24,OS dependent 1,inode number(open) */

	/*
	 * data block number.
	 * 112 direct block number.
	 * 13 indirect block number.
	 * 14 bi-indirect block number.
	 * 15 tri-indirect block number.
	 * Each indirect block array contains as many entries of 32bit block numbers as possible
	 * (to fill one entire block).
	 */
	uint i_block[INODE_DATA_BLOCK_NUM];		/* 28 */
	uint i_generation;		/* 64,file version (used by NFS). */
	uint i_file_acl;		/* 68,block number containing the extended attributes, */
	uint i_dir_acl;			/* 6c,"high size" of the file. */
	uint i_faddr;			/* 70,location of the last file fragment. */
	uint i_osd2[3];			/* 74,OS dependent 2 */
}INODE;


static uchar *inodeBitmap[MAX_DEVICE_OPEN];		/* PUBLIC inodeӥåȥޥåס */
static INODE *rootInode[MAX_DEVICE_OPEN];		/* PUBLIC root inode. */


/*
 * PRIVATE
 * get data from indeirect block
 * ɹ߳ϥХͤϴܥ֥åǡǤγϥХ͡
 * parameters : device inode,block buffer,last block index,buffer,read size,begin byte,indirect node(=1,Ŵ=2,Ŵ=3)
 * return : ɹߥ or error=-1
 */
static int readIndirectBlock(int din,uint* block_buf,int last,void *buf,size_t size,size_t begin,int node)
{
	char *data_buf=NULL;
	uint *next_buf;
	uint rest=size;								/* ɤ߹ĤΥ */
	int sectors=superBlock[din]->sectors;		/* 1֥åΥ */
	int block_size=superBlock[din]->blockSize;
	int blk,beg,siz;


	if(node==1)
	{
		blk=begin/block_size;

		/* Ϥ᤬֥åǤʤ硣 */
		if((beg=begin%block_size)!=0)
		{
			if((data_buf=(char*)kmalloc(block_size))==NULL)goto ERR;
			if(read_cache(din,data_buf,sectors,blockToSect(din,block_buf[blk]))!=sectors)goto ERR;
			siz=((block_size-beg)>rest)?rest:block_size-beg;
			memcpy(buf,data_buf+beg,siz);
			rest-=siz;
			++blk;
			(char*)buf+=siz;
		}

		for(;blk<last;++blk)
		{
			/* Ǹ夬֥åǤʤ硣 */
			if(rest<block_size)
			{
				if(rest==0)break;

				if(data_buf==NULL)
					if((data_buf=(char*)kmalloc(block_size))==NULL)goto ERR;
				if(read_cache(din,data_buf,sectors,blockToSect(din,block_buf[blk]))!=sectors)goto ERR;
				memcpy(buf,data_buf,rest);
				rest=0;

				break;
			}

			if(read_cache(din,buf,sectors,blockToSect(din,block_buf[blk]))!=sectors)goto ERR;
			(char*)buf+=block_size;
			rest-=block_size;
		}
	}
	else
	{
		if((next_buf=(uint*)kmalloc(block_size))==NULL)return -1;

		for(blk=begin/(1<<(EXT2_BLOCK_BASE_LOG+superBlock[din]->s_log_block_size)*node);blk<last;++blk)
		{
			if(read_cache(din,next_buf,sectors,blockToSect(din,block_buf[blk]))!=sectors)goto ERR;

			if((siz=readIndirectBlock(din,next_buf,block_size/sizeof(uint),buf,rest,begin,node-1))==-1)goto ERR;
			if((rest-=siz)==0)break;;
			buf+=siz;
			begin=0;
		}

		kfree(next_buf);
	}

	kfree(data_buf);

	return size-rest;

ERR:
	kfree(data_buf);

	return -1;
}


/*
 * PUBLIC
 * get data block from inode
 * parameters : device inode,inode,buffer,read size,begin byte
 * return : read size or error=-1
 */
static int readData(int din,INODE *inode,void *buf,size_t size,size_t begin)
{
	uint *block_buf=NULL;						/* ֥åХåե */
	uint rest=size;								/* ɤ߹ĤΥ */
	int b_entry;								/* 1֥åΥ֥åȥ꡼ */
	int sectors=superBlock[din]->sectors;		/* 1֥åΥ */
	int block_size=superBlock[din]->blockSize;
	uint blk;
	int siz;


	blk=begin/block_size;
	{

		/* ľǡ֥å */
		if(blk<INODE_DATA_INDIRECT1)
		{
			if((siz=readIndirectBlock(din,inode->i_block,INODE_DATA_INDIRECT1,buf,rest,begin,1))==-1)return -1;
			if((rest-=siz)==0)goto END;
			buf+=siz;

			begin=block_size*INODE_DATA_INDIRECT1;	/* δܥ֥åǳϥХͤ0ˤʤ褦Ĵ */
		}

		b_entry=block_size/sizeof(uint);

		/* ܥ֥å */
		if(blk<b_entry+INODE_DATA_INDIRECT1)
		{
			if((block_buf=(uint*)kmalloc(block_size))==NULL)return -1;
			if(read_cache(din,block_buf,sectors,blockToSect(din,inode->i_block[INODE_DATA_INDIRECT1]))!=sectors)goto ERR;

			begin-=block_size*INODE_DATA_INDIRECT1;
			if((siz=readIndirectBlock(din,block_buf,block_size/sizeof(uint),buf,rest,begin,1))==-1)goto ERR;
			if((rest-=siz)==0)goto END;
			buf+=siz;

			begin=block_size*(b_entry+INODE_DATA_INDIRECT1);	/* δܥ֥åǳϥХͤ0ˤʤ褦Ĵ */
		}

		/* Ŵܥ֥å */
		if(blk<b_entry*b_entry+b_entry+INODE_DATA_INDIRECT1)
		{
			if(block_buf==NULL)
				if((block_buf=(uint*)kmalloc(block_size))==NULL)return -1;
			if(read_cache(din,block_buf,sectors,blockToSect(din,inode->i_block[INODE_DATA_INDIRECT2]))!=sectors)goto ERR;

			begin-=block_size*(b_entry+INODE_DATA_INDIRECT1);
			if((siz=readIndirectBlock(din,block_buf,block_size/sizeof(uint),buf,rest,begin,2))==-1)goto ERR;
			if((rest-=siz)==0)goto END;
			buf+=siz;

			begin=block_size*(b_entry*b_entry+b_entry+INODE_DATA_INDIRECT1);	/* δܥ֥åǳϥХͤ0ˤʤ褦Ĵ */
		}

		/* Ŵܥ֥å */
		{
			if(block_buf==NULL)
				if((block_buf=(uint*)kmalloc(block_size))==NULL)return -1;
			if(read_cache(din,block_buf,sectors,blockToSect(din,inode->i_block[INODE_DATA_INDIRECT2]))!=sectors)goto ERR;

			begin-=block_size*(b_entry*b_entry+b_entry+INODE_DATA_INDIRECT1);
			if((siz=readIndirectBlock(din,block_buf,block_size/sizeof(uint),buf,rest,begin,3))==-1)goto ERR;
		}
	}
END:

	kfree(block_buf);

	return size;

ERR:
	kfree(block_buf);

	return -1;
}


/*
 * PUBLIC
 * write date.
 * parameters : device inode,inode,buffer,read size,begin byte
 * return : read size or error=-1
 */
static int writeData(int din,INODE *inode,void *buf,size_t size,size_t begin)
{
	return 0;
}


/*
 * PUBLIC
 * Υ֥åХåեinodeơ֥ɤ߹ǡinodeХåե˥ԡ롣
 * get inode.
 * parameters : device inode,inode number,block buffer,inode_buf
 */
static int getInode(int din,uint i_number,char *block_buf,INODE *inode_buf)
{
	int gdesc;
	int index;
	uint sector;


	gdesc=(i_number-1)/superBlock[din]->s_inodes_per_group;
	index=(i_number-1)%superBlock[din]->s_inodes_per_group;
	sector=blockToSect(din,groupDesc[din][gdesc].bg_inode_table)+index*sizeof(INODE)/superBlock[din]->sectorSize;
	if(read_cache(din,block_buf,1,sector)!=1)return -1;
	memcpy(inode_buf,block_buf+index*sizeof(INODE)%superBlock[din]->sectorSize,sizeof(INODE));

	return 0;
}


/*
 * PUBLIC
 * ޥȻν
 * parameters : device inode,򳫻Ϥ֥å롼ץʥС
 * return : 0 or error=-1
 */
static int initInode(int din,int bg)
{
	void *buf;
	int i;
	GROUP_DESC *gd;


	/* inodeӥåȥޥåפɤ߹ࡣ */
	gd=groupDesc[din];
	i=bg;
	do
	{
		if(gd[i].bg_free_inodes_count>0)break;
		if(gd[++i].bg_block_bitmap==0)i=0;
	}while(i!=bg);
	if(read_direct(din,inodeBitmap[din],superBlock[din]->sectors,blockToSect(din,gd[i].bg_inode_bitmap))!=
		superBlock[din]->sectors)return -1;

	/* 롼inodeФ롣 */
	if((buf=kmalloc(superBlock[din]->sectorSize))==NULL)return -1;
	if((rootInode[din]=(INODE*)kmalloc(sizeof(INODE)))==NULL){kfree(buf);return -1;}
	if(read_direct(din,buf,1,blockToSect(din,groupDesc[din][0].bg_inode_table))!=1)
		{kfree(buf);kfree(rootInode[din]);return -1;}
	memcpy(rootInode[din],&((INODE*)buf)[EXT2_ROOT_INODE_INDEX-1],sizeof(INODE));
	kfree(buf);
	rootInode[din]->i_number=EXT2_ROOT_INODE_INDEX;

	return 0;
}


/*
 * PUBLIC
 * ޥȻν
 * parameters : device inode
 */
static void closeInode(int din)
{
	kfree(rootInode[din]);
	rootInode[din]=NULL;
	kfree(inodeBitmap[din]);
	inodeBitmap[din]=NULL;
}


/***********************************************************************
 *
 * < Directory class >
 *
 ***********************************************************************/

/************************************************************

Sampl directory

$ ls -1a /home/eks
.
..
.bash_profile
.bashrc
mbox
public_html
tmp

offset	size	description
------- ------- -----------
0		4		inode number (783362)
4		2		record length (9)
6		1		name length (1)
7		1		file type (EXT2_FT_DIR)
8		1		name (.)

9		4		inode number (1109761)
13		2		record length (10)
15		1		name length (2)
16		1		file type (EXT2_FT_DIR)
17		2		name (..)

19		4		inode number (783364)
23		2		record length (21)
25		1		name length (13)
26		1		file type (EXT2_FT_REG_FILE)
27		13		name (.bash_profile)

40		4		inode number (783363)
44		2		record length (15)
46		1		name length (7)
47		1		file type (EXT2_FT_REG_FILE)
48		7		name (.bashrc)

55		4		inode number (783377)
59		2		record length (12)
61		1		name length (4)
62		1		file type (EXT2_FT_REG_FILE)
63		4		name (mbox)

67		4		inode number (783545)
71		2		record length (19)
73		1		name length (11)
74		1		file type (EXT2_FT_DIR)
75		11		name (public_html)

86		4		inode number (669354)
90		2		record length (11)
92		1		name length (3)
93		1		file type (EXT2_FT_DIR)
94		3		name (tmp)

97		4		inode number (0)
101		2		record length (3999)
103		1		name length (0)
104		1		file type (EXT2_FT_UNKNOWN)
105		0		name ()

************************************************************/


enum{
	EXT2_NAME_LEN=255,

	/* directory file types. */
	EXT2_FT_UNKNOWN=0,
	EXT2_FT_REG_FILE,
	EXT2_FT_DIR,
	EXT2_FT_CHRDEV,
	EXT2_FT_BLKDEV,
	EXT2_FT_FIFO,
	EXT2_FT_SOCK,
	EXT2_FT_SYMLINK,
	EXT2_FT_MAX,
};

typedef struct{
	uint inode;				/* inode number. */
	ushort rec_len;			/* directory entry length. */
	uchar name_len;			/* name length. */
	uchar file_type;		/* file type. */
	char name[0];			/* File name. */
}DIR_ENTRY;


/*
 * PRIVATE
 * search entry.
 * parameters : path,device inode,directory block,directory block buffer
 * return : directory entry address or error=NULL
 */
static inline DIR_ENTRY *searchEntry(const char *path,int din,char *block_buf)
{
	char *last;


	last=block_buf+superBlock[din]->blockSize;

	for(;block_buf<last;block_buf+=((DIR_ENTRY*)block_buf)->rec_len)
	{
		if(cmpPathLength(((DIR_ENTRY*)block_buf)->name,path,((DIR_ENTRY*)block_buf)->name_len)==0)
			return (DIR_ENTRY*)block_buf;
	}

	return NULL;
}


/*
 * PRIVATE
 * ܻȥǡ֥å饨ȥ꡼õ
 * ΥХåե˥ǥ쥯ȥɤ߹ǥȥ꡼ɥ쥹֤
 * parameters : const char *path,int din,block number,base buffer(block size),indirect node(=1,Ŵ=2,Ŵ=3)
 * return : directory entry address or error=NULL
 */
static DIR_ENTRY *searchIndirect(const char *path,int din,uint block,char *block_buf,int node)
{
	uint *buf;
	int sectors=superBlock[din]->sectors;
	DIR_ENTRY *entry=NULL;
	int i,last;


	if((buf=(uint*)kmalloc(superBlock[din]->blockSize))==NULL)return NULL;
	if(read_cache(din,buf,sectors,blockToSect(din,block))!=sectors)
	{
		kfree(buf);
		return NULL;
	}

	if(node==1)
	{
		for(i=0,last=superBlock[din]->blockSize/sizeof(uint);buf[i]!=0;)
		{
			if(read_cache(din,block_buf,sectors,blockToSect(din,buf[i]))!=sectors)break;
			if((entry=searchEntry(path,din,block_buf))!=NULL)break;

			if(++i>=last)break;
		}
	}
	else
		for(i=0,last=superBlock[din]->blockSize/sizeof(uint);buf[i]!=0;)
		{
			if((entry=searchIndirect(path,din,buf[i],block_buf,node-1))!=NULL)break;

			if(++i>=last)break;
		}

	kfree(buf);

	return entry;
}


/*
 * PUBLIC
 * search directory.
 * ͤϡͿ줿١Хåեinodeɥ쥹ˤʤ롣
 * parameters : path,device inode,directory inode,inode buffer
 * return : 0 or error=-1
 */
static int searchPath(const char *path,int din,INODE *inode,INODE *inode_buf)
{
	char *block_buf;
	int sectors=superBlock[din]->sectors;
	uint number;
	DIR_ENTRY *entry;
	int i;


	if(*path=='\0')return -1;

	if((block_buf=(char*)kmalloc(superBlock[din]->blockSize))==NULL)return -1;

	for(;;)
	{
		{
			i=0;

			/* ľܻȡ */
			for(;i<INODE_DATA_INDIRECT1;++i)
			{
				if(inode->i_block[i]==0)goto ERR;
				if(read_cache(din,block_buf,sectors,blockToSect(din,inode->i_block[i]))!=sectors)goto ERR;
				if((entry=searchEntry(path,din,block_buf))!=NULL)goto EXIT;
			}

			/* ܻȡ */
			for(;i<INODE_DATA_BLOCK_NUM;++i)
			{
				if(inode->i_block[i]==0)goto ERR;
				if((entry=searchIndirect(path,din,inode->i_block[i],block_buf,i-INODE_DATA_INDIRECT1+1))!=NULL)
					goto EXIT;
			}

			goto ERR;
		}
EXIT:
		path+=entry->name_len;
		if(*path++=='\0')break;
		if(entry->file_type!=EXT2_FT_DIR)goto ERR;

		/* inode롣 */
		if(getInode(din,entry->inode,block_buf,inode_buf)==-1)goto ERR;
		inode=inode_buf;
	}

	/* inode롣 */
	number=entry->inode;
	if(getInode(din,number,block_buf,inode_buf)==-1)goto ERR;
	inode_buf->i_number=number;

	kfree(block_buf);

	return 0;

ERR:
	kfree(block_buf);

	return -1;
}


/***********************************************************************
 *
 * < system call >
 *
 ***********************************************************************/

/*
 * return : root inode address in memory.
 */
static uint mount(int din)
{
	void *buf;
	DEV_STAT dev_stat;
	int sectors;
	int gd;
	int a;


	/* Ǥ˥ޥȤƤ뤫ǧ */
	if(superBlock[din]!=NULL)return -1;

	/* ѡ֥åɤ߹ࡣ */
	if(get_dev_stat(din,&dev_stat)==-1)return -1;
	if((buf=kmalloc(PAGE_SIZE))==NULL)return -1;
	if(read_direct(din,buf,PAGE_SIZE/dev_stat.sect_size,0)!=PAGE_SIZE/dev_stat.sect_size)
		{kfree(buf);return -1;}
	if((superBlock[din]=(SUPER_BLOCK*)kmalloc(sizeof(SUPER_BLOCK)))==NULL)
		{kfree(buf);return -1;}
	memcpy(superBlock[din],(char*)buf+EXT2_SUPER_BLK_POSITION,sizeof(SUPER_BLOCK));
	kfree(buf);

	/* ޥåʥСγǧ */
	if(superBlock[din]->s_magic!=EXT2_SUPER_MAGIC)goto ERR;

	/* ȼѡ֥å */
	superBlock[din]->blockSize=EXT2_BLOCK_BASE<<superBlock[din]->s_log_block_size;
	superBlock[din]->sectorSize=dev_stat.sect_size;
	superBlock[din]->sectors=superBlock[din]->blockSize/superBlock[din]->sectorSize;

	sectors=superBlock[din]->sectors;

	/* 롼ץǥץɤ߹ࡣ */
	a=ROUNDUP_DIV(superBlock[din]->s_blocks_count,superBlock[din]->s_blocks_per_group)*sizeof(GROUP_DESC);
	groupDescSize[din]=ROUNDUP_DIV(a,superBlock[din]->blockSize);
	if((groupDesc[din]=(GROUP_DESC*)kmalloc(superBlock[din]->blockSize*groupDescSize[din]))==NULL)goto ERR;
	if(read_direct(din,groupDesc[din],sectors*groupDescSize[din],blockToSect(din,EXT2_GROUP_DESC_BLK+superBlock[din]->s_first_data_block))!=
		sectors*groupDescSize[din])goto ERR;

	/* ֥åطν */
	if((gd=initBlock(din))==-1)goto ERR;

	/* inodeطν */
	if(initInode(din,gd)==-1)goto ERR;

	return (int)rootInode[din];

ERR:
	kfree(groupDesc[din]);
	groupDesc[din]=NULL;
	kfree(superBlock[din]);
	superBlock[din]=NULL;

	return -1;
}


static int umount(int din)
{
	SUPER_BLOCK *sb=superBlock[din];
	void *buf;


	closeInode(din);
	closeBlock(din);

	write_cache(din,groupDesc[din],blockToSect(din,EXT2_GROUP_DESC_BLK+sb->s_first_data_block),groupDescSize[din]*sb->sectors);
	write_cache(din,groupDesc[din],blockToSect(din,EXT2_GROUP_DESC_BLK+sb->s_first_data_block+sb->s_blocks_per_group),
		groupDescSize[din]*sb->sectors);
	kfree(groupDesc[din]);
	groupDesc[din]=NULL;

	if((buf=kmalloc(sb->blockSize))==NULL)return -1;
	read_cache(din,buf,blockToSect(din,sb->s_first_data_block),sb->sectors);
	if(sb->s_first_data_block==1)memcpy(buf,sb,sizeof(sb));
	else memcpy((char*)buf+EXT2_SUPER_BLK_POSITION,sb,sizeof(sb));
	write_cache(din,buf,blockToSect(din,sb->s_first_data_block),sb->sectors);
	kfree(superBlock[din]);
	superBlock[din]=NULL;

	return 0;
}


/*
 * parameters : path,device inode,directory inode address in memory
 */
static int open(const char *path,int din,uint dir_inode)
{
	INODE *inode_buf;


	if((inode_buf=(INODE*)kmalloc(sizeof(INODE)))==NULL)return -1;

	if(searchPath(path,din,(INODE*)dir_inode,inode_buf)==-1)goto ERR;

	return (int)inode_buf;

ERR:
	kfree(inode_buf);

	return -1;
}


static int close(int din,uint inode)
{
	kfree((void*)inode);

	return 0;
}


static int read(int din,uint inode,void *buf,size_t size,size_t begin)
{
	/* γǧ */
	if(begin>=((INODE*)inode)->i_size)return 0;
	if(begin+size>=((INODE*)inode)->i_size)size=((INODE*)inode)->i_size-begin;

	/* Хåեɤ߹ࡣ */
	return readData(din,(INODE*)inode,buf,size,begin);
}

/*********************************** ̤ ***************************************/
static int write(int din,uint inode,void *buf,size_t size,size_t begin)
{
	/* γǧ */
	if(begin>((INODE*)inode)->i_size)return -1;
	if(begin+size-((INODE*)inode)->i_size>superBlock[din]->s_free_blocks_count*superBlock[din]->blockSize)
		return -1;

	return 0;
}


static int mkdir(const char *path,int din,uint dir_blk)
{
	return 0;
}


static int creat(const char *path,int din,uint dir_blk)
{
	return 0;
}


static int opendir(const char *path,int din,uint dir_blk)
{
	return 0;
}


static int ioctr(int din,uint inode,int cmd,void *param)
{
	return 0;
}


static int rename(int din,uint inode,const char *name)
{
	return 0;
}


/***********************************************************************
 *
 * < Init class >
 *
 ***********************************************************************/

 static FS ext2Fs={
	"ext2",mount,umount,open,close,read,write,ioctr,rename,creat,opendir,mkdir
};


/*
 * GLOBAL
 */
int initExt2tFs()
{
	int i;


	for(i=0;i<MAX_DEVICE_OPEN;++i)
	{
		rootInode[i]=NULL;
		groupDesc[i]=NULL;
		superBlock[i]=NULL;
		blockBitmap[i]=NULL;
		inodeBitmap[i]=NULL;
	}

	return regist_fs(&ext2Fs);
}
/**************************************************************************************/
void test_ext2()
{
	printk("superBlock[1]=%x,&superBlock[1]->sectorSize=%x\n",superBlock[1],&superBlock[1]->sectorSize);
	printk("superBlock[1]->sectorSize=%d\n",superBlock[1]->sectorSize);
}
/**************************************************************************************/
