/*
 * device.c
 *
 * Copyright 2002, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 */


#include"config.h"
#include"types.h"
#include"lib.h"
#include"mm.h"
#include"interrupt.h"
#include"proc.h"
#include"fs.h"
#include"errno.h"
#include"lock.h"
#include"device.h"
#include"test.h"


enum{
	MAX_NAME=16,			/* Device name size */
	HASH_NUM=CACHE_NUM,		/* Number of cache hash table */
	SECTOR_SIZE=512
};


/* ǥХȤ */
typedef struct{
	uchar pt_num;		/* Partition number */
	uchar pt_pres;		/* Partition presence flag */
	uchar pt_type;		/* Partition type */
	char ref_count;		/* Reference count */
	DEV_INFO *dev;		/* Device infomation pointer */
	uint begin_blk;		/* Begin block number */
	uint blocks;		/* Number of blocks */
}DEV_INODE;

/* Partition table */
typedef struct{
	uchar active;			/* Boot active flag=0x80 */
	uchar chs_begin[3];		/* CHS begin block */
	uchar type;				/* Partition type */
	uchar chs_last[3];		/* CHS last block */
	uint lba_begin;			/* LBA begin block */
	uint size;				/* Partition size by sector */
}PARTITON_TBL;

typedef struct CACHE_TABLE{
	struct CACHE_TABLE *hnext;		/* Hash next */
	struct CACHE_TABLE *hprev;		/* Hash previous */
	struct CACHE_TABLE *next;		/* Cache chain next */
	struct CACHE_TABLE *prev;		/* Cache chain previous */
	struct CACHE_TABLE *write_next;	/* ǥХؽ񤭹߹ꥹ */
	struct CACHE_TABLE *write_prev; /* ǥХؽ񤭹߹ꥹ */
	void *addr;						/* Cache address */
	uint block;						/* Device block index */
	uchar din;						/* Device inode number */
	uchar ref_count;				/* ȥ */
	uchar update;					/* ե饰 */
	uchar writing;					/* ǥХؽ񤭹߹ե饰 */
	int gate;						/* ԥå */
}CACHE_TABLE;


static DEV_INFO *device[MAX_DEVICES];
static DEV_INODE dinode[MAX_DEVICE_OPEN];

/* Cache */
static CACHE_TABLE cache_tbl[CACHE_NUM];			/* Cache table */
static CACHE_TABLE *cache_hash[HASH_NUM];			/* Hash table for searching cache table */
static CACHE_TABLE *cache_tbl_old=&cache_tbl[0];	/* Cache table old point */
static int cache_gate=0;							/* åơ֥ѥԥåե饰 */
static int cache_ref_count=0;						/* åơ֥뻲ȥ */
static int cache_update=0;							/* åơ֥빹ե饰 */

static PROC *write_task=NULL;						/* ٱ񤭹ѥ */
static CACHE_TABLE delay_list=						/* ٱ񤭹ߥꥹȤƬ */
	{NULL,NULL,NULL,NULL,&delay_list,&delay_list,NULL,0,0,0,0,0,0};


static int init_cache();
static void start_delay_write_task();
static void add_delay_list(CACHE_TABLE*);
static void delay_write();
static int open(const char*,int,uint);
static int close(int,uint);
static int read(int,uint,void*,size_t,size_t);
static int write(int,uint,void*,size_t,size_t);
static int ioctl(int,uint,int,void*);


static FS dev_fs={
	"dev_fs",NULL,NULL,open,close,read,write,ioctl,NULL,NULL,NULL,NULL
};


 /*
  * ǥХѥ̾Ӥ롣
  * parameters : destination path,sorce path
  * return :  פΥݥ,԰ NULL
  */
static inline const char *cmp_devpath(const char *s1,const char *s2)
{
	while(*s1!='\0')
		if(*s1++!=*s2++)return NULL;

	return s2;
}


/************************************************************************************************
 *
 * ؿ
 *
 ************************************************************************************************/


/*
 * Init device system
 * return : 0 or error=-1
 */
int init_device()
{
	int fs;
	int i;


	for(i=0;i<MAX_DEVICES;++i)device[i]=NULL;
	for(i=0;i<MAX_DEVICES;++i)dinode[i].ref_count=0;

	/* IOå */
	init_cache();

	/* ٱ񤭹ѥư롣 */
	start_delay_write_task();

	/* ե륷ƥϿ롣 */
	if((fs=regist_fs(&dev_fs))==-1)return -1;
	if(mount_dev_fs(fs)==-1)return -1;

	return 0;
}


/*
 * Register device
 * parameters : Device infomation struct address
 * return : 0 or Error number
 */
int regist_device(DEV_INFO *info)
{
	int i;


	for(i=0;i<MAX_DEVICES;++i)
		if(device[i]==NULL)
		{
			device[i]=info;
			return 0;
		}

	return -EMDEV;
}


/*
 * Delete from regist table
 * parameters : name
 * return : 0,error -1
 */
int delete_device(const char *name)
{
	return 0;
}


 /***********************************************************************************
 *
 * ¾δؿ
 *
 ***********************************************************************************/

/*
 * ǥХμ
 * parameters : device inode,stat store buffer
 * return : 0 or error=-1
 */
int get_dev_stat(int din,DEV_STAT *buf)
{
	if(dinode[din].ref_count==0)return -1;

	buf->pt_pres=dinode[din].pt_pres;
	buf->prt_type=dinode[din].pt_type;
	buf->sect_size=dinode[din].dev->sector_size;
	buf->all_sect=dinode[din].blocks;

	return 0;
}


/************************************************************************************************
 *
 * ľžؿ
 *
 ************************************************************************************************/

/*
 * parameters : device inode number,buffer address,transfer sectors,begin sector
 * return : transfer sectors or error=-1
 */
int read_direct(int din,void *buf,size_t sectors,size_t begin)
{
	/* ǥ¤Υå */
	if(begin>dinode[din].blocks)return -1;
	if(begin+sectors>dinode[din].blocks)sectors=dinode[din].blocks-begin;

	return dinode[din].dev->read(buf,sectors,begin+dinode[din].begin_blk);
}


/*
 * parameters : device inode number,buffer address,transfer sectors,begin sector
 * return : transfer sectors or error=-1
 */
int write_direct(int din,void *buf,size_t sectors,size_t begin)
{
	/* ǥ¤Υå */
	if(begin>dinode[din].blocks)return -1;
	if(begin+sectors>dinode[din].blocks)sectors=dinode[din].blocks-begin;

	return dinode[din].dev->write(buf,sectors,begin+dinode[din].begin_blk);
}


/***********************************************************************************
 *
 * IOåؿ
 *  å֥åñ̤ɤ߽񤭤롣
 *
 ***********************************************************************************/

/*
 * ϥåͤ׻롣
 * return : hash value
 */
static inline int calc_hash(int value)
{
	return (value/8)%HASH_NUM;
}


static inline int cache_rounddown(int value)
{
	return ~(CACHE_BLOCK_SIZE-1)/value;
}


/*
 * ɤ߼궦ͭåûξѡ
 * parameters : ɤ߼껲ȥ,ե饰ݥ
 */
static inline void enter_read_lock(uchar *refcnt,uchar *update,int *gate)
{
	enter_spinlock(gate);
	{
		while(*update==1);
		++*refcnt;
	}
	exit_spinlock(gate);
}


/*
 * ɤ߼궦ͭåФ롣
 * parameters : ɤ߼껲ȥ
 */
static inline void exit_read_lock(uchar *refcnt)
{
	--*refcnt;
}

static inline void set_update(uchar *refcnt,uchar *update,int *gate)
{
	/* ʣãƤɤ褦ˡȥȤö餹 */
	--*refcnt;
	enter_spinlock(gate);
	{
		while(*refcnt!=0);
		*update=1;
		++*refcnt;
	}
	exit_spinlock(gate);
}

static inline void reset_update(uchar *update)
{
	*update=0;
}


/*
 * Init cache
 * return : 0 or Error number
 */
int init_cache()
{
	int i;


	/* Init cache address */
	for(i=0;i<CACHE_NUM;++i)
	{
		memset(&cache_tbl[i],0,sizeof(CACHE_TABLE));
		if((cache_tbl[i].addr=kmalloc(CACHE_BLOCK_SIZE))==NULL)return -ENOMEM;
		cache_tbl[i].next=&cache_tbl[i+1];
		cache_tbl[i].prev=&cache_tbl[i-1];
		cache_tbl[i].hprev=&cache_tbl[i];
	}
	cache_tbl[CACHE_NUM-1].next=&cache_tbl[0];
	cache_tbl[0].prev=&cache_tbl[CACHE_NUM-1];

	/* Init hash table */
	for(i=0;i<HASH_NUM;++i)cache_hash[i]=NULL;

	return 0;
}


/*
 * åơ֥뤫鳺ǥХ֥åõ
 * ʤХåơ֥ɲä롣
 * parameters : cache hash,device inode,device block
 * return : cache table addres or error=NULL
 */
CACHE_TABLE *search_cache(int hash,int din,size_t begin)
{
	int sect_num;		/* å֥åˤϤ륻 */
	CACHE_TABLE *p;


	sect_num=CACHE_BLOCK_SIZE/dinode[din].dev->sector_size;

	enter_read_lock((uchar*)&cache_ref_count,(uchar*)&cache_update,&cache_gate);

	/* å򸡺롣 */
	for(p=cache_hash[hash];;p=p->hnext)
	{
		/* åˤʤ */
		if(p==NULL)
		{
			set_update((uchar*)&cache_ref_count,(uchar*)&cache_update,&cache_gate);

			/* ָŤå򲣼 */
			p=cache_tbl_old;
			cache_tbl_old=p->next;

			/* Delete old hash link */
			p->hprev->hnext=p->hnext;
			if(p->hnext!=NULL)p->hnext->hprev=p->hprev;

			/* Add new hash link */
			p->hnext=cache_hash[hash];
			p->hprev=(CACHE_TABLE*)&cache_hash[hash];
			if(cache_hash[hash]!=NULL)cache_hash[hash]->hprev=p;
			cache_hash[hash]=p;

			/* updateå */
			enter_spinlock(&p->gate);
			{
				/* ٱ񤭹Ԥ */
				while(p->writing!=0);

				reset_update((uchar*)&cache_update);
				--cache_ref_count;
				while(cache_ref_count!=0);
				p->update=1;
				++cache_ref_count;
			}
			exit_spinlock(&p->gate);

			p->din=din;
			p->block=begin;

			/* ǥХ֥åɤ߼롣*/
			if(dinode[din].dev->read(p->addr,sect_num,begin+dinode[din].begin_blk)<sect_num)
			{
				exit_read_lock((uchar*)&cache_ref_count);
				return NULL;
			}

			reset_update(&p->update);

			break;
		}

		/* åˤ */
		if((p->block==begin)&&(p->din==din))break;
	}

	exit_read_lock((uchar*)&cache_ref_count);

	return p;
}


/*
 * Read from cache
 * parameters : device inode number,buffer address,transfer sectors,begin sector
 * return : transfer sectors or error=-1
 */
int read_cache(int din,void *buf,size_t sectors,size_t begin)
{
	int cache_mask;		/* åƬ򻻽Ф뤿Υӥåȥޥ */
	int sect_num;		/* å֥åˤϤ륻 */
	int sect_size;		/* ǥХ */
	int rest;			/* Ĥž */
	CACHE_TABLE *p;
	int i,s,t;


	/* ǥ¤Υå */
	if(begin>dinode[din].blocks)return 0;
	if(begin+sectors>dinode[din].blocks)sectors=dinode[din].blocks-begin;

	sect_size=dinode[din].dev->sector_size;
	sect_num=CACHE_BLOCK_SIZE/sect_size;
	cache_mask=~(sect_num-1);
	rest=sectors;

	/* žå֥å椫Ϥޤ硣 */
	if((begin&~cache_mask)!=0)
	{
		s=begin&~cache_mask;

		/* å򸡺 */
		if((p=search_cache(calc_hash(begin&cache_mask),din,begin&cache_mask))==NULL)return -1;

		if((t=sect_num-s)>sectors)
		{
			memcpy(buf,(char*)p->addr+s*sect_size,sectors*sect_size);
			
			return sectors;
		}

		memcpy(buf,(char*)p->addr+s*sect_size,t*sect_size);
		(char*)buf+=t*sect_size;
		rest-=t;
		begin+=t;
	}

	for(i=0,s=rest-sect_num;i<=s;i+=sect_num)
	{
		/* å򸡺 */
		if((p=search_cache(calc_hash(begin+i),din,begin+i))==NULL)return -1;

		memcpy(buf,p->addr,CACHE_BLOCK_SIZE);
		(char*)buf+=CACHE_BLOCK_SIZE;
	}

	/* žå֥åǽ硣 */
	if(i!=rest)
	{
		/* å򸡺 */
		if((p=search_cache(calc_hash(begin+i),din,begin+i))==NULL)return -1;

		memcpy(buf,p->addr,sect_size*(rest-i));
	}

	return sectors;
}


/*
 * Write to cache
 * parameters : device inode number,buffer address,transfer sectors,begin sector
 * return : transfer sectors or error=-1
 */
int write_cache(int din,void *buf,size_t sectors,size_t begin)
{
	int cache_mask;		/* åƬ򻻽Ф뤿Υӥåȥޥ */
	int sect_num;		/* å֥åˤϤ륻 */
	int sect_size;		/* ǥХ */
	int rest;			/* Ĥž */
	CACHE_TABLE *p;
	int i,s,t;


	/* ǥ¤Υå */
	if(begin>dinode[din].blocks)return 0;
	if(begin+sectors>dinode[din].blocks)sectors=dinode[din].blocks-begin;

	sect_size=dinode[din].dev->sector_size;
	sect_num=CACHE_BLOCK_SIZE/sect_size;
	cache_mask=~(sect_num-1);
	rest=sectors;

	/* žå֥å椫Ϥޤ硣 */
	if((begin&~cache_mask)!=0)
	{
		s=begin&~cache_mask;

		/* å򸡺 */
		if((p=search_cache(calc_hash(begin&cache_mask),din,begin&cache_mask))==NULL)return -1;

		if((t=sect_num-s)>sectors)
		{
			memcpy((char*)p->addr+s*sect_size,buf,sectors*sect_size);
			add_delay_list(p);		/* ٱ񤭹ߥꥹȤ˲ä롣 */
			return sectors;
		}

		memcpy((char*)p->addr+s*sect_size,buf,t*sect_size);
		(char*)buf+=t*sect_size;
		rest-=t;
		begin+=t;

		/* ٱ񤭹ߥꥹȤ˲ä롣 */
		add_delay_list(p);
	}

	for(i=0,s=rest-sect_num;i<=s;i+=sect_num)
	{
		/* å򸡺 */
		if((p=search_cache(calc_hash(begin+i),din,begin+i))==NULL)return -1;

		memcpy(p->addr,buf,CACHE_BLOCK_SIZE);
		(char*)buf+=CACHE_BLOCK_SIZE;

		/* ٱ񤭹ߥꥹȤ˲ä롣 */
		add_delay_list(p);
	}

	/* žå֥åǽ硣 */
	if(i!=rest)
	{
		/* å򸡺 */
		if((p=search_cache(calc_hash(begin+i),din,begin+i))==NULL)return -1;

		memcpy(p->addr,buf,sect_size*(rest-i));

		/* ٱ񤭹ߥꥹȤ˲ä롣 */
		add_delay_list(p);
	}

	return sectors;
}


/************************************************************************************************
 *
 * ǥХٱ񤭹
 *
 ************************************************************************************************/

/*
 * ٱ񤭹ߥεư
 */
void start_delay_write_task()
{
	int proc;


	switch(proc=sys_fork())
    {
		case -1:
			printk("Faild start_delay_write_task\n");
			break;
		case 0:
			delay_write();
		default:
			write_task=(PROC*)proc;
	}
}


/*
 * ٱ񤭹ߥꥹȤ˲ä롣
 * parameters : device inode number,񤭹ߥ֥å,֥åλϤޤ
 */
void add_delay_list(CACHE_TABLE *p)
{
	if(p->writing==1)return;

	enter_spinlock(&p->gate);
	{
		p->writing=1;
	}
	exit_spinlock(&p->gate);

	enter_spinlock(&delay_list.gate);
	{
		p->write_next=delay_list.write_next;
		p->write_prev=&delay_list;
		p->write_next->write_prev=p;
		delay_list.write_next=p;
	}
	exit_spinlock(&delay_list.gate);

	/* ٱ񤭹ߥ򵯤 */
	if(p->write_next==&delay_list)add_to_schedule(write_task);
}


/*
 * ǥХؤٱ񤭹ߥѴؿ
 */
void delay_write()
{
	CACHE_TABLE *p;


	while(write_task==NULL);
	del_from_schedule(write_task);
	wait_task();

	for(;;)
	{
		enter_spinlock(&delay_list.gate);

		if(delay_list.write_next!=&delay_list)
		{
			p=delay_list.write_prev;
			p->write_prev->write_next=&delay_list;
			delay_list.write_prev=p->write_prev;

			exit_spinlock(&delay_list.gate);

			/* ǥХ˽񤭹ࡣ */
			dinode[p->din].dev->write(
				p->addr,CACHE_BLOCK_SIZE/dinode[p->din].dev->sector_size,p->block+dinode[p->din].begin_blk);

			p->writing=0;
		}
		else
		{
			exit_spinlock(&delay_list.gate);

			/* ꡼פ롣 */
			del_from_schedule(write_task);
			wait_task();
		}
	}
}


/************************************************************************************************
 *
 * System call interface
 *
 ************************************************************************************************/

/*
 * Open device
 * device inode0ϤǤ˥ǥХե륷ƥ˻ȤƤΤǡ1鳫Ϥ롣
 * parameters : Device name,device inode=0,directory block=0
 * return : device inode number or error=-1
 */
static int open(const char *dev_name,int dev,uint dir)
{
	enum{
		/* Partition table */
		PT_NUM=4,				/* 1Υѡƥơ֥ */
		PT_BEGIN=0x1be,			/* ѡƥơ֥볫ϥեå */
		PT_MAGIC_ADDR=0x1fe,	/* ѡƥơ֥ޥåʥСեå */
		PT_MAGIC_NUMBER=0xaa55,	/* ѡƥơ֥ޥåʥС */

		/* Partition type */
		PT_TYPE_NULL=0,			/* NULL partition */
		PT_TYPE_EXT=0x5,		/* ext partition */
		PT_TYPE_LEXT=0xf,		/* ext(LBA) partition */
	};

	const char *str;
	char *buf;
	int inode_num;
	int pt_num;
	uint beg_blk,next_blk;
	PARTITON_TBL *pt;
	int i;


	/* ϿǥХ̾Ǹ */
	for(i=0;(str=cmp_devpath(device[i]->name,dev_name))==NULL;++i)
		if(i>=MAX_DEVICES)return -1;
	dev=i;

	/* ̾˥ѡƥֹ椬СѴ */
	pt_num=0;
	if(*str!='\0')
		if((pt_num=atoi(str))<=0)return -1;		/* ѡƥֹ1ʾ */

	/* Ǥ˥ץ󤵤Ƥ뤫ǧ */
	for(i=1;i<MAX_DEVICE_OPEN;++i)
		if((dinode[i].dev==device[dev])&&(dinode[i].pt_num==pt_num))
		{
			++dinode[i].ref_count;
			return i;
		}

	/* ǥХinode򤹤롣 */
	for(i=1;dinode[i].ref_count!=0;++i)
		if(i>=MAX_DEVICE_OPEN)return -1;
	inode_num=i;

	/* Open device */
	if(device[dev]->open()<0)return -1;


	if(device[dev]->sector_size==0)
	{
		/* 饯ǥХ */
		dinode[inode_num].pt_pres=0;
		dinode[inode_num].pt_num=0;
		dinode[inode_num].pt_type=0;
		dinode[inode_num].begin_blk=0;
		dinode[inode_num].blocks=0;
	}
	else
	{
		/* ѡƥơ֥򸡺롣 */
		if((buf=(char*)kmalloc(device[dev]->sector_size))==NULL)return -1;
		if(device[dev]->read(buf,1,0)!=1)goto ERR;
		if(pt_num==0)
		{
			if(*(ushort*)(buf+PT_MAGIC_ADDR)!=PT_MAGIC_NUMBER)dinode[inode_num].pt_pres=1;
			else dinode[inode_num].pt_pres=0;

			dinode[inode_num].pt_num=0;
			dinode[inode_num].pt_type=0;
			dinode[inode_num].begin_blk=device[dev]->begin_blk;
			dinode[inode_num].blocks=device[dev]->last_blk-device[dev]->begin_blk+1;
		}
		else
		{
			if(*(ushort*)(buf+PT_MAGIC_ADDR)!=PT_MAGIC_NUMBER)goto ERR;

			pt=(PARTITON_TBL*)(buf+PT_BEGIN);
			if(pt_num<=PT_NUM)
			{
				if(pt[pt_num].type==PT_TYPE_EXT)goto ERR;	/* ΰγĥѡƥϳʤ */

				dinode[inode_num].pt_num=pt_num;
				--pt_num;
				dinode[inode_num].pt_type=pt[pt_num].type;
				dinode[inode_num].begin_blk=pt[pt_num].lba_begin;
				dinode[inode_num].blocks=pt[pt_num].size;
			}
			else
			{
				for(i=0;pt[i].type!=PT_TYPE_EXT;++i)
					if(i>=PT_NUM)goto ERR;			/* ĥѡƥ󤬤ʤ */

				/* ĥѡƥ󥯤򤿤ɤ롣 */
				beg_blk=next_blk=pt[i].lba_begin;
				for(i=PT_NUM;;)
				{
					if(device[dev]->read(buf,1,next_blk)!=1)goto ERR;
					if(++i==pt_num)break;
					if(pt[1].type!=PT_TYPE_EXT)goto ERR;
					next_blk=beg_blk+pt[1].lba_begin;
				}

				dinode[inode_num].pt_num=pt_num;
				dinode[inode_num].pt_type=pt[0].type;
				dinode[inode_num].begin_blk=pt[0].lba_begin+next_blk;
				dinode[inode_num].blocks=pt[0].size;
			}

			dinode[inode_num].pt_pres=1;
		}

		kfree(buf);
	}

	dinode[inode_num].dev=device[dev];
	dinode[inode_num].ref_count=1;

	return inode_num;

ERR:
	kfree(buf);

	return -1;
}

/*
 * parameters : device inode,inode block
 */
static int close(int din,uint inode)
{
	if(--dinode[inode].ref_count<0)dinode[inode].ref_count=0;

	return 0;
}

/*
 * parameters : device inode,inode,Buffer,Transfer bytes,begin byte
 */
static int read(int din,uint inode,void *buf,size_t size,size_t begin)
{
	char *tmp=NULL;
	int sect_siz;		/*  */
	int rest=size;		/* ĤžХȿ */
	int beg_sect;		/* žϤΥֹ */
	int s,t;


	if((sect_siz=dinode[inode].dev->sector_size)==0)
	{
		/* 饯ǥХ */
		return dinode[inode].dev->read(buf,size,0);
	}

	beg_sect=begin/sect_siz;

	/* ǽΥ̤ž */
	if((s=begin%sect_siz)!=0)
	{
		if((tmp=(char*)kmalloc(sect_siz))==NULL)return -1;
		if(read_cache(din,tmp,1,beg_sect)!=1)return 0;
		t=sect_siz-s;
		if(t>size)
		{
			memcpy(buf,tmp+s,size);
			kfree(tmp);
			return size;
		}
		memcpy(buf,tmp+s,t);
		(char*)buf+=t;
		beg_sect+=1;
		rest-=t;
	}

	s=rest/sect_siz;
	if((t=read_cache(din,buf,s,beg_sect))<s)
	{
		kfree(tmp);
		return size-rest+t*sect_siz;
	}

	/* Ĥ꤬̤ž */
	if(s*sect_siz<rest)
	{
		t=s*sect_siz;
		if((tmp==NULL)&&((tmp=(char*)kmalloc(sect_siz))==NULL))return -1;
		if(read_cache(din,tmp,1,beg_sect+s)!=1)
		{
			kfree(tmp);
			return size-rest+t;
		}
		memcpy((char*)buf+t,tmp,rest-t);
	}

	kfree(tmp);

	return size;
}

/*
 * parameters : device inode,inode,Buffer,Transfer bytes,begin byte
 */
static int write(int din,uint inode,void *buf,size_t size,size_t begin)
{
	char *tmp=NULL;
	int sect_siz;		/*  */
	int rest=size;		/* ĤžХȿ */
	int beg_sect;		/* žϤΥֹ */
	int s,t;


	if((sect_siz=dinode[inode].dev->sector_size)==0)
	{
		/* 饯ǥХ */
		return dinode[inode].dev->write(buf,size,0);
	}

	beg_sect=begin/sect_siz;

	/* ǽΥ̤ž */
	if((s=begin%sect_siz)!=0)
	{
		if((tmp=(char*)kmalloc(sect_siz))==NULL)return -1;
		if(read_cache(din,tmp,1,beg_sect)!=1)return 0;
		if((t=sect_siz-s)>size)
		{
			memcpy(tmp+s,buf,size);
			if(write_cache(din,tmp,1,beg_sect)!=1)return 0;
			kfree(tmp);
			return size;
		}
		memcpy(tmp+s,buf,t);
		if(write_cache(din,tmp,1,beg_sect)!=1)return 0;
		(char*)buf+=t;
		beg_sect+=1;
		rest-=t;
	}

	s=rest/sect_siz;
	if((t=write_cache(din,buf,s,beg_sect))<s)
	{
		kfree(tmp);
		return size-rest+t*sect_siz;
	}

	/* Ĥ꤬̤ž */
	if(s*sect_siz<rest)
	{
		t=s*sect_siz;
		if((tmp==NULL)&&((tmp=(char*)kmalloc(sect_siz))==NULL))return -1;
		if(read_cache(din,tmp,1,beg_sect+s)!=1)
		{
			kfree(tmp);
			return size-rest+t;
		}
		memcpy(tmp,(char*)buf+t,rest-t);
		if(write_cache(din,tmp,1,beg_sect+s)!=1)
		{
			kfree(tmp);
			return size-rest+t;
		}
	}

	kfree(tmp);

	return size;
}

/*
 * parameters : device inode,inode,Command,Parameter struct pointer
 */
static int ioctl(int din,uint ionode,int com,void *prm)
{
	return 0;
}
/****************************************************************************/
void test_device()
{
	uint *buf=(uint*)0x90000;
	uint *buf2=(uint*)0x80000;
	int din;


	din=open("hdb",0,0);

	read_cache(din,buf,1,2000);
	printk("buf[0]=%x\n",buf[0]);

	buf[0]=0xabababab;
	write_cache(din,buf,1,2000);

	mili_timer(2000);

	read_direct(din,buf2,1,2000);
	printk("buf2[0]=%x\n",buf[0]);
}
/*****************************************************************************/
