/*
 * device.c
 *
 * Copyright 2002, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 */


#include <config.h>
#include <types.h>
#include <lib.h>
#include <mm.h>
#include <interrupt.h>
#include <proc.h>
#include <fs.h>
#include <lock.h>
#include <device.h>
#include <except_i386.h>
#include <errno.h>
#include <lock.h>
#include <time.h>
#include <signal.h>
#include <sys/param.h>
#include <sys/conf.h>
#include <sys/systm.h>
#include <test.h>
#include <debug.h>


//#define DEBUG_DEVICE 1
#ifdef DEBUG_DEVICE
	#define STATIC
	#define INLINE
#else
	#define STATIC static
	#define INLINE	inline
#endif


/******************************************************************************************************
 *
 * ǥХڥ졼
 *
 *******************************************************************************************************/


STATIC INLINE int readDev(dev_t dev, void *buf, size_t len, size_t begin, int ioflag)
{
	struct iovec iov;
	struct uio uio;
	
	iov.iov_base = buf;
	iov.iov_len = len;
	uio.uio_iov = &iov;
	uio.uio_iovcnt = 1;
	uio.uio_offset = begin;
	uio.uio_resid = len;
	uio.uio_segflg = UIO_USERSPACE;
	uio.uio_rw = UIO_READ;
	uio.uio_procp = NULL;

	return dev->si_devsw->d_read(dev, &uio, ioflag);
}


STATIC INLINE int writeDev(dev_t dev, void *buf, size_t len, size_t begin, int ioflag)
{
	struct iovec iov;
	struct uio uio;
	
	iov.iov_base = buf;
	iov.iov_len = len;
	uio.uio_iov = &iov;
	uio.uio_iovcnt = 1;
	uio.uio_offset = begin;
	uio.uio_resid = len;
	uio.uio_segflg = UIO_USERSPACE;
	uio.uio_rw = UIO_WRITE;
	uio.uio_procp = NULL;

	return dev->si_devsw->d_write(dev, &uio, ioflag);

}


/******************************************************************************************************
 *
 * ǥХԤ
 *
 *******************************************************************************************************/


//================================== PRIVATE ============================================


// Ԥץꤹ˳ߤȯȤꤵ͡
#define WAIT_INTR_ACCESS ((PROC*)-1)


//================================== PUBLIC =============================================


void wait_intr(WAIT_INTR *wait, uint time, int procStat)
{
	int eflag;

	enterCli(&eflag);
	enter_spinlock(&wait->lock);
	if (wait->proc == NULL){
		wait->proc = get_current_task();

		exit_spinlock(&wait->lock);
		exitCli(&eflag);

		setWake(time);
		del_from_schedule(procStat, wait->isDel);
		wait_task();
		delWakeTimer(wait->proc->timer);
	}
	else{
		exit_spinlock(&wait->lock);
		exitCli(&eflag);
	}

	wait->proc = NULL;
	wait->isDel = PROC_SCHEDULE_DEL;
}


int wake_intr(WAIT_INTR *wait, int procStat)
{
	enter_spinlock(&wait->lock);
	if (wait->proc != NULL){
		exit_spinlock(&wait->lock);
		add_to_schedule(wait->proc, procStat, &wait->isDel);
		return 1;
	}
	else{
		wait->proc = WAIT_INTR_ACCESS;
		exit_spinlock(&wait->lock);
		return 0;
	}
}


/***********************************************************************************
 *
 * ֥åǥХ
 *
 ***********************************************************************************/


enum{
	/* Partition table */
	PARTITION_TBL_MAX		= 4,		/* 1κѡƥơ֥ */
	PARTITION_BEGIN			= 0x1be,	/* ѡƥơ֥볫ϥեå */
	PARTITION_MAGIC_ADDR	= 0x1fe,	/* ѡƥơ֥ޥåʥСեå */
	PARTITION_MAGIC_NUMBER	= 0xaa55,	/* ѡƥơ֥ޥåʥС */

	/* Partition type */
	PARTITION_TYPE_NULL		= 0,	/* NULL partition */
	PARTITION_TYPE_EXT		= 0x5,	/* ĥ */
};


/* Partition table */
typedef struct{
	uint8_t		active;		/* Boot active flag=0x80 */
	uint8_t		chsBeg[3];	/* CHS begin block */
	uint8_t		type;		/* Partition type */
	uint8_t		chsEnd[3];	/* CHS last block */
	uint32_t	lbaBeg;		/* LBA begin block */
	uint32_t	size;		/* Partition size by sector */
}PARTITON_TBL;


//================================== PROTECTED ==========================================


// ѡƥơ֥򸡺
// return : NOERR
//          ENODEV=Ѥ or ĥѡƥ 
//          ENOENT=ĥѡƥ󥯤κǸã
static INLINE dev_t getDevInfo(int inodeNum);
STATIC int readPartition(const int inodeNum, const int partNum, PARTITON_TBL *o_partTbl)
{
	char *sectBuf;
	dev_t dev;
	PARTITON_TBL *partTbl;
	uint extBeg, extNext;
	int c_part;
	int i, rest;

	// ǥХץ
	dev = getDevInfo(inodeNum);
	if (dev == NULL){
		return -ENOENT;
	}
	if (dev->si_devsw->d_open(dev, O_RDONLY, 0, NULL) < 0){
		return -EIO;
	}

	/* MBRɤ */
	sectBuf = kmalloc(dev->si_devsw->sector_size);
	if (sectBuf == NULL){
		return -ENOMEM;
	}
	rest = readDev(dev, sectBuf, 1, 0, 0);
	if (rest < 0){
		goto END;
	}

	/* ѡƥơ֥ǧ */
	if (*(uint16_t*)(sectBuf + PARTITION_MAGIC_ADDR) != PARTITION_MAGIC_NUMBER){
		rest = -ENODEV;
		goto END;
	}
	partTbl = (PARTITON_TBL*)(sectBuf + PARTITION_BEGIN);

	if (partNum <= PARTITION_TBL_MAX){
		/* ܥѡƥ򸡺 */
		c_part = partNum - 1;

		switch (partTbl[c_part].type){
			/* ΰγĥѡƥϳʤ */
			case PARTITION_TYPE_EXT:
				// 
			case PARTITION_TYPE_NULL:
				rest = -ENODEV;
				goto END;
			default:
				/* ѥѡƥơ֥ */
				o_partTbl->type		= partTbl[c_part].type;
				o_partTbl->lbaBeg	= partTbl[c_part].lbaBeg;
				o_partTbl->size		= partTbl[c_part].size;
		}
	}
	else{
		/* ĥѡƥ򸡺 */
		/*
		 * MBRγĥѡƥ򸡺
		 * out : c_part = ĥѡƥֹ
		 */
		for (i = 0;; ++i){
			if (PARTITION_TBL_MAX <= i){
				rest = -ENOENT;
				goto END;
			}
			if (partTbl[i].type == PARTITION_TYPE_EXT){
				c_part = i;
				break;
			}
		}

		/*
		 * ĥѡƥ󥯤򤿤ɤ
		 * out : extNext = ĥΰϤޤ꤫Υ֥å sectBuf = ĥΰΥѡƥ󥻥
		 */
		extBeg = extNext = partTbl[c_part].lbaBeg;
		for (i = PARTITION_TBL_MAX; ;){
			rest = readDev(dev, sectBuf, 1, extNext, 0);
			if (rest < 0){
				goto END;
			}

			/* ѡƥȯ */
			if (++i == partNum){
				break;
			}

			/* ĥѡƥ󥯤νλ */
			if (partTbl[1].type != PARTITION_TYPE_EXT){
				rest = -ENOENT;
				goto END;
			}

			extNext = extBeg + partTbl[1].lbaBeg;
		}

		/* Ѥߥѡƥ */
		if (partTbl[0].type == PARTITION_TYPE_NULL){
			rest = -ENODEV;
			goto END;
		}
		else{
			/* ѥѡƥơ֥ */
			o_partTbl->type		= partTbl[0].type;
			o_partTbl->lbaBeg	= partTbl[0].lbaBeg + extNext;
			o_partTbl->size		= partTbl[0].size;
		}
	}

	rest = NOERR;
END:
	kfree(sectBuf);

	return rest;
}


/***********************************************************************************
 *
 * ǥХե륷ƥ
 *
 * ¤
 *	inode¤Τǳ
 *		root inodeϣΩ
 *
 * ========== PRIVATE
 * void setRootInode()
 * void partNumToStr(int value, char *m_str)
 * int getMajorNumber()
 * dev_t allocSpecinfo()
 * void freeSpecinfo(dev_t dev)
 * int getInodeNum(DEV_INODE *devInode)
 * void linkInode(int inodeNum)
 * int getFreeInode()
 * int setInode(struct cdevsw *cdevsw, uint blockBeg, uint size, int partType, int major, int partNum, int fflag)
 * ========== PROTECTED
 * void unlinkInode(int inodeNum)
 * int lookupDevInode(const char *path)
 * int getNextInode(int inodeNum)
 * int getStat(int inodeNum, struct stat *m_stat)
 * dev_t getDevInfo(int inodeNum)
 * uint getBlocks(int inodeNum)
 * uint getBegBlock(int inodeNum)
 * uint getMode(int inodeNum)
 * const char *getName(int inodeNum)
 * void initDevFs()
 * ========== PUBLIC
 * dev_t make_dev(struct cdevsw *devsw, int minor, uid_t uid, gid_t gid, int perms, const char *fmt, ...)
 * int get_devstat(int inodeNum, DEV_STAT *m_devStat)
 * int delete_device(const char *name)
 * int registBlockPartition(const char *name)
 ***********************************************************************************/


enum{
	DEVICE_INODE_NUM	= MAX_DEVICE_OPEN,	// ǥХinode
	MAX_DEVICE_NAME		= 8,				// ǥХ̾ʸ
	MAX_PART_NUM_DIGIT	= 4,				// ֥åѡƥֹ
};


/* ǥХinode */
typedef struct DEV_INODE{
	dev_t				dev;					// struct specinfo*
	ushort				partType;				// ѡƥ󥿥
	ushort				major;					// ᥸㡼ֹ
	ushort				partNum;				// ѡƥֹޥʡֹ
	short				mode;					/* ե⡼ */
	int 				refCount;				/* ȥ */
	uint				blockBeg;				/* ǥǤΥѡƥλϤΥ֥å */
	uint				blocks;					/* ѡƥ֥å */
	uint				creatTime;				/*  */
	char				name[MAX_DEVICE_NAME];	/* ǥХ̾ */
}DEV_INODE;


//================================== PRIVATE ============================================


/* ǥХե륷ƥinodeơ֥ */
static DEV_INODE devInodeTbl[DEVICE_INODE_NUM];	/* inodeơ֥ */

// dev_tŪΰ
struct devData{
	struct specinfo specinfo[DEVICE_INODE_NUM];
	char busy[DEVICE_INODE_NUM];		// ʤ飱
}devData;


// root inode
STATIC void setRootInode()
{
	/* inode */
	devInodeTbl[DEVICE_ROOT_INODE].partType = 0;
	devInodeTbl[DEVICE_ROOT_INODE].partNum = 0;
	devInodeTbl[DEVICE_ROOT_INODE].refCount = 2;
	devInodeTbl[DEVICE_ROOT_INODE].dev = NULL;
	devInodeTbl[DEVICE_ROOT_INODE].blockBeg = 0;
	devInodeTbl[DEVICE_ROOT_INODE].blocks = 0;
	devInodeTbl[DEVICE_ROOT_INODE].creatTime = sys_time(NULL);
	devInodeTbl[DEVICE_ROOT_INODE].mode = S_IFDIR | S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH;
}


// ѡƥֹʸѴ롣
// ʸ"MAX_PART_NUM_DIGIT"ڤ롣
STATIC void partNumToStr(int value, char *m_str)
{
	char str[MAX_PART_NUM_DIGIT + 1];
	int i;

	for (i = 0; i < MAX_PART_NUM_DIGIT; ++i){
		str[i] = value % 10 + '0';
		if ((value /= 10) == 0){
			break;
		}
	}
	m_str[i + 1] = '\0';
	while (i >= 0){
		*m_str++ = str[i--];
	}
}


// ᥸㡼ֹ롣
STATIC int getMajorNumber()
{
	static int majorNum = 0;
	
	return majorNum++;
}


// dev_tƤ
// return : dev_t ro NULL
STATIC dev_t allocSpecinfo()
{
	dev_t dev = NULL;
	int i;
	
	for (i = 0; i < DEVICE_INODE_NUM; ++i){
		if (devData.busy[i] == NO){
			devData.busy[i] = YES;
			dev = &devData.specinfo[i];
			break;
		}
	}
	
	return dev;
}


// dev_t
STATIC void freeSpecinfo(dev_t dev)
{
	int i;

	for (i = 0; i < DEVICE_INODE_NUM; ++i){
		if (dev == &devData.specinfo[i]){
			devData.busy[i] = NO;
			break;
		}
	}
}


// ǥХinodeinodeֹ
STATIC INLINE int getInodeNum(DEV_INODE *devInode)
{
	return ((uint)devInode - (uint)devInodeTbl) / sizeof(*devInode);
}


// dev_tinodeֹ򸡺롣
STATIC int searchInodeNum(dev_t dev)
{
	int i;
	
	for (i = 0; i < DEVICE_INODE_NUM; ++i){
		if (0 < devInodeTbl[i].refCount){
			if (devInodeTbl[i].dev == dev){
				return i;
			}
		}
	}
	
	return -ENOENT;
}


// inodeλȥ󥿤䤹
STATIC void linkInode(int inodeNum)
{
	ASSERT(0 <= devInodeTbl[inodeNum].refCount);
	devInodeTbl[inodeNum].refCount += 1;
}


// inode
// return : inode number or error number
STATIC int getFreeInode()
{
	int i;
	
	for (i = 0; i < DEVICE_INODE_NUM; ++i){
		if (devInodeTbl[i].refCount == 0){
			return i;
		}
	}

	return -EMFILE;
}


// ǥХinodeꤹ롣
// return : device inode or error number
static int lookupDevInode(const char *);
STATIC int setInode(
	struct cdevsw *cdevsw,
	uint blockBeg,
	uint size,
	int partType,
	int major,
	int partNum,
	int fflag)
{
	char numStr[MAX_PART_NUM_DIGIT + 1] = {'\0'};
	char name[MAX_DEVICE_NAME + MAX_PART_NUM_DIGIT];
	int inodeNum;

	/* ѡƥֹ礷ǥХ͡ʸγǧ */
	if (0 < partNum){
		partNumToStr(partNum, numStr);
	}
	strcpy(name, cdevsw->d_name);
	strcat(name, numStr);
	if (MAX_DEVICE_NAME <= strlen(name)){
		return -ENAMETOOLONG;
	}

	/* ǤϿ뤫 */
	inodeNum = lookupDevInode(name);
	if (inodeNum < 0){
		// Ͽʤ
		/* inode */
		inodeNum = getFreeInode();
		if (inodeNum < 0){
			return inodeNum;
		}

		/* inode */
		devInodeTbl[inodeNum].dev = allocSpecinfo();
		if (devInodeTbl[inodeNum].dev == NULL){
			return -ENFILE;
		}
		devInodeTbl[inodeNum].partType = partType;
		devInodeTbl[inodeNum].major = major;
		devInodeTbl[inodeNum].partNum = partNum;
		devInodeTbl[inodeNum].blockBeg = blockBeg;
		devInodeTbl[inodeNum].blocks = size;
		devInodeTbl[inodeNum].creatTime = sys_time(NULL);
		if (cdevsw->d_flags & (D_TAPE | D_TTY | D_MEM)){
			// 饯ǥХ
			devInodeTbl[inodeNum].mode = S_IFCHR | fflag;
		}
		else if (cdevsw->d_flags & D_DISK){
			// ֥åǥХ
			devInodeTbl[inodeNum].mode = S_IFBLK | fflag;
		}
		else{
			// Ȥꤢ饯ǥХˤƤ
			devInodeTbl[inodeNum].mode = S_IFCHR | fflag;
		}
		/* ǥХ̾ */
		strcpy(devInodeTbl[inodeNum].name, name);
		linkInode(inodeNum);

		// dev
		devInodeTbl[inodeNum].dev->si_udev = makeudev(major, partNum);
		devInodeTbl[inodeNum].dev->si_devsw = cdevsw;
	}
	else{
		return -EEXIST;
	}

	return inodeNum;
}


// cdevsw¤ΤNULLؿѴ롣
STATIC void compile_devsw(struct cdevsw *devsw)
{
	if (devsw->d_open == NULL)
		devsw->d_open = noopen;
	if (devsw->d_close == NULL)
		devsw->d_close = noclose;
	if (devsw->d_read == NULL)
		devsw->d_read = noread;
	if (devsw->d_write == NULL)
		devsw->d_write = nowrite;
	if (devsw->d_ioctl == NULL)
		devsw->d_ioctl = noioctl;
	if (devsw->d_poll == NULL)
		devsw->d_poll = nopoll;
	if (devsw->d_mmap == NULL)
		devsw->d_mmap = nommap;
	if (devsw->d_strategy == NULL)
		devsw->d_strategy = nostrategy;
	if (devsw->d_dump == NULL)
		devsw->d_dump = nodump;
	if (devsw->d_psize == NULL)
		devsw->d_psize = nopsize;
	if (devsw->d_kqfilter == NULL)
		devsw->d_kqfilter = nokqfilter;
}


//================================== PROTECTED ==========================================


// inodeλȥ󥿤򣱸餹
// return : ȿ
STATIC void unlinkInode(int inodeNum)
{
	ASSERT(0 <= devInodeTbl[inodeNum].refCount);
	// ȥȤ򸺤餹
	devInodeTbl[inodeNum].refCount -= 1;
	if (devInodeTbl[inodeNum].refCount == 0){
		// dev_t
		freeSpecinfo(devInodeTbl[inodeNum].dev);
	}
}


// ǥХθ
// return : inode number or error number
STATIC int lookupDevInode(const char *path)
{
	const char *devName;
	int i;
	
	// ѥθ
	// out : devName = "path"ФǥХ̾
	for (devName = path;;){
		/* "."".."ν */
		if (cmpPath(".",devName,1) == 0){
			if (path[1] == '/'){
				devName = &path[2];
			}
			else{
				devName = &path[1];
			}
		}
		/* ".."̤б */
		else if (cmpPath("..",devName,2) == 0){
			return -ENOENT;
		}
		/* root inode? */
		else if (cmpPath("\0",devName,0) == 0){
			if (path[1] == '/'){
				devName = &path[1];
			}
			else{
				return DEVICE_ROOT_INODE;
			}
		}
		else{
			break;
		}
	}

	// ǥХinodeơ֥򸡺
	for (i = 0; i < DEVICE_INODE_NUM; ++i){
		if (strcmp(devInodeTbl[i].name, devName) == 0){
			if (0 < devInodeTbl[i].refCount){
				return i;
			}
		}
	}

	return -ENOENT;
}


// Ϳ줿inodeμΥ󥯤inode֤
// Ϳ줿inode-1ʤȥåץ󥯤inode֤
// return : next device inode or ERR
STATIC int getNextInode(int inodeNum)
{
	int i;
	
	if (inodeNum == -1){
		i = DEVICE_ROOT_INODE;
	}
	else{
		i = inodeNum;
	}
	for (++i; i < DEVICE_INODE_NUM; ++i){
		if (0 < devInodeTbl[i].refCount){
			return i;
		}
	}
	
	return ERR;
}


// statꤹ
STATIC int getStat(int inodeNum, struct stat *m_stat)
{
	ASSERT((0 <= inodeNum) && (inodeNum < DEVICE_INODE_NUM));
	if (devInodeTbl[inodeNum].refCount == 0){
		return -ENOENT;
	}

	m_stat->st_dev			= devInodeTbl[inodeNum].dev->si_udev;
	m_stat->st_ino			= inodeNum;
	m_stat->st_mode			= devInodeTbl[inodeNum].mode;
	m_stat->st_nlink		= devInodeTbl[inodeNum].refCount;
	m_stat->st_uid			= 0;
	m_stat->st_gid			= 0;
	m_stat->st_size			= 0;
	m_stat->st_atime		= devInodeTbl[inodeNum].creatTime;
	m_stat->st_mtime		= devInodeTbl[inodeNum].creatTime;
	m_stat->st_ctime		= devInodeTbl[inodeNum].creatTime;
	m_stat->st_crtpos		= 0;
	if (devInodeTbl[inodeNum].dev == NULL){
		m_stat->st_blksize	= 0;
	}
	else{
		m_stat->st_blksize	= devInodeTbl[inodeNum].dev->si_devsw->sector_size;
	}
	m_stat->st_rdev			= devInodeTbl[inodeNum].dev->si_udev;
	m_stat->st_flags		= 0;
	m_stat->st_blocks		= devInodeTbl[inodeNum].blocks;

	return NOERR;
}


// dev_t֤
STATIC INLINE dev_t getDevInfo(int inodeNum)
{
	ASSERT((0 <= inodeNum) && (inodeNum < DEVICE_INODE_NUM));
	return devInodeTbl[inodeNum].dev;
}


// ֥å֤
STATIC INLINE uint getBlocks(int inodeNum)
{
	ASSERT((0 <= inodeNum) && (inodeNum < DEVICE_INODE_NUM));
	return devInodeTbl[inodeNum].blocks;
}


// ֥åλϤޤ֤
STATIC INLINE uint getBegBlock(int inodeNum)
{
	ASSERT((0 <= inodeNum) && (inodeNum < DEVICE_INODE_NUM));
	return devInodeTbl[inodeNum].blockBeg;
}


// ե⡼ɤ֤
STATIC INLINE uint getMode(int inodeNum)
{
	ASSERT((0 <= inodeNum) && (inodeNum < DEVICE_INODE_NUM));
	return devInodeTbl[inodeNum].mode;
}


// ǥХ̾
STATIC const char *getName(int inodeNum)
{
	ASSERT((0 <= inodeNum) && (inodeNum < DEVICE_INODE_NUM));
	return devInodeTbl[inodeNum].name;
}


// Init device filesystem.
STATIC void initDevFs()
{
	// root inode
	setRootInode();
}


//================================== PUBLIC =============================================


int major(dev_t x)
{
	if (x == NODEV){
		return NOUDEV;
	}

	return((x->si_udev >> 8) & 0xff);
}


int minor(dev_t x)
{
	if (x == NODEV){
		return NOUDEV;
	}

	return(x->si_udev & 0xffff00ff);
}


udev_t makeudev(int x, int y)
{
	return ((x << 8) | y);
}


// Regist device.
// parameters : cdevsw¤, ޥʡֹ, uid, gid, ե⡼, Ͽ̾
// return : dev_t or NODEV
dev_t make_dev(struct cdevsw *devsw, int minor, uid_t uid, gid_t gid, int perms, const char *fmt, ...)
{
	int inodeNum, nextInodeNum;
	int i;
	dev_t dev;
	va_list ap;

	/* Name sizeγǧ  */
	if (MAX_DEVICE_NAME <= strlen(devsw->d_name)){
		return NODEV;
	}

	// ƱǥХ̾ʤ
	for (nextInodeNum = -1; (nextInodeNum = getNextInode(nextInodeNum)) != ERR;){
		if (strcmp(devInodeTbl[nextInodeNum].dev->si_name, devsw->d_name) == 0){
			return NODEV;
		}
	}

	// cdevsw¤ΤNULLؿ
	compile_devsw(devsw);
	
	// ǥХե륷ƥϿ롣
	inodeNum = setInode(
		devsw,
		devsw->begin_blk,
		devsw->last_blk - devsw->begin_blk + 1,
		0,
		getMajorNumber(),
		minor,
		perms
	);
	if (inodeNum < 0){
		return NODEV;
	}
	dev = devInodeTbl[inodeNum].dev;
	
	va_start(ap, fmt);
	i = kvprintf(fmt, NULL, dev->si_name, 32, ap);
	dev->si_name[i] = '\0';
	va_end(ap);

	return dev;
}


// Delete from regist table
// parameters : name
// return : error number
void destroy_dev(dev_t dev)
{
	int inodeNum;

	inodeNum = searchInodeNum(dev);
	if (inodeNum < 0){
		return;
	}

	unlinkInode(inodeNum);
}


// ǥХinodeμ
// return : error number
int get_devstat(int inodeNum, DEV_STAT *m_devStat)
{
	if (devInodeTbl[inodeNum].refCount == 0){
		return -ENODEV;
	}

	m_devStat->prt_type = devInodeTbl[inodeNum].partType;
	m_devStat->major = devInodeTbl[inodeNum].major;
	m_devStat->prt_num = devInodeTbl[inodeNum].partNum;
	m_devStat->sect_size = devInodeTbl[inodeNum].dev->si_devsw->sector_size;
	m_devStat->all_sect = devInodeTbl[inodeNum].blocks;
	m_devStat->name = devInodeTbl[inodeNum].dev->si_name;

	return NOERR;
}


// ֥åǥХѡƥǥХե륷ƥϿ
// parmeters : ǥХ̵ֹ̾
int registBlockPartition(const char *name)
{
	int inodeNum;
	dev_t dev;
	PARTITON_TBL partTbl;
	int partNum;
	
	/* inode */
	inodeNum = lookupDevInode(name);
	if (inodeNum < 0){
		return inodeNum;
	}
	
	/* struct cdevsw¤Τ */
	dev = getDevInfo(inodeNum);
	if (dev == NULL){
		return -ENODEV;
	}
	
	/* ΰθ */
	for (partNum = 1; partNum <= PARTITION_TBL_MAX; ++partNum){
		if (readPartition(inodeNum, partNum, &partTbl) == NOERR){
			setInode(
				dev->si_devsw,
				partTbl.lbaBeg,
				partTbl.size,
				partTbl.type,
				devInodeTbl[inodeNum].major,
				partNum,
				S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP
			);
		}
	}

	/* ĥΰθ */
	for (partNum = PARTITION_TBL_MAX + 1;; ++partNum){
		int rest;

		switch (rest = readPartition(inodeNum, partNum, &partTbl)){
			case NOERR:
				setInode(
					dev->si_devsw,
					partTbl.lbaBeg,
					partTbl.size,
					partTbl.type,
					devInodeTbl[inodeNum].major,
					partNum,
					S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP
				);
				break;
			case -ENODEV:
				continue;
			case -ENOENT:
				goto END;
			default:
				printk("%s registBlockPartition() unkown return value!:%d", __FILE__, rest);
		}
	}
END:
	return NOERR;
}


int noopen(dev_t dev, int flags, int fmt, struct proc *p)
{
	return (-ENODEV);
}

int noclose(dev_t dev, int flags, int fmt, struct proc *p)
{
	return (-ENODEV);
}

int noread(dev_t dev, struct uio *uio, int ioflag)
{
	return (-ENODEV);
}

int nowrite(dev_t dev, struct uio *uio, int ioflag)
{
	return (-ENODEV);
}

int noioctl(dev_t dev, u_long cmd, caddr_t data, int flags, struct proc *p)
{
	return (-ENODEV);
}

int nokqfilter(dev_t dev, struct knote *kn)
{
	return (-ENODEV);
}

int nommap(dev_t dev, vm_offset_t offset, int nprot)
{
	/* Don't return ENODEV.  That would allow mapping address ENODEV! */
	return (-1);
}

int nodump(dev_t dev)
{
	return (-ENODEV);
}

// nopoll
int seltrue(dev_t dev, int events, struct proc *p)
{
	return (events & (POLLIN | POLLOUT | POLLRDNORM | POLLWRNORM));
}


/************************************************************************************************
 *
 * ľžؿ
 *
 ************************************************************************************************/


//================================== PUBLIC =============================================


// return : transfer sectors or error number
int read_direct(int inodeNum, void *buf, size_t sectors, size_t begin)
{
	/* ǥ¤Υå */
	if (getBlocks(inodeNum) < begin){
		return -1;
	}
	if (getBlocks(inodeNum) < begin + sectors){
		sectors = getBlocks(inodeNum) - begin;
	}

	return readDev(getDevInfo(inodeNum), buf, sectors, begin + getBegBlock(inodeNum), 0);
}


// return : transfer sectors or error number
int write_direct(int inodeNum, void *buf, size_t sectors, size_t begin)
{
	/* ǥ¤Υå */
	if (getBlocks(inodeNum) < begin){
		return -1;
	}
	if (getBlocks(inodeNum) < begin + sectors){
		sectors = getBlocks(inodeNum) - begin;
	}

	return writeDev(getDevInfo(inodeNum), buf, sectors, begin + getBegBlock(inodeNum), 0);
}


/************************************************************************************************
 *
 * ǥХå
 *
 * ա
 *  ǥХϥ512ХȤˤƤΤǡ¾Υ
 *  ǥХоݤˤˤϡɬס
 *
 ************************************************************************************************/


enum{
	CACHE_NUM = 8,				/* IOå֥å */
	CACHE_BLOCK_SIZE = 0x1000,	/* IOå֥å */
	HASH_NUM = CACHE_NUM,		/* Number of cache hash table */
	CACHE_UNIT_SIZE = 512,		/* å֥åΥӥåȥޥåñ̥Хȥ */
};

typedef struct CACHE_TABLE{
	struct CACHE_TABLE			*hashNext;	// Hash next
	struct CACHE_TABLE			*hashPrev;
	struct CACHE_TABLE			*cacheNext;	// Cache chain next
	struct CACHE_TABLE			*cachePrev;
	struct CACHE_TABLE volatile	*writeNext;	// ٱ񤭹ߥꥹ
	struct CACHE_TABLE		 	*writePrev;
	char						*cache;		// Cache address
	uint						block;		// Device block index
	uchar						inodeNum;	// Device inode number
	uchar volatile				bitmap;		// ǡ¸ߥӥåȥޥå
	uchar volatile				writing;	// ٱԤե饰
	uchar volatile				busy;		// ե饰
	int							gate;		// ԥå
}CACHE_TABLE;


//================================== PRIVATE ============================================


static CACHE_TABLE cache_tbl[CACHE_NUM];			// Cache table
static CACHE_TABLE *cache_hash[HASH_NUM];			// Hash table for searching cache table
static CACHE_TABLE *cacheOldest = &cache_tbl[0];	// old chache block.nextֿ֥å
static int cacheLock;								// å󥯹ѥå
static PROC *writeTask;								// ٱ񤭹ѥ
static CACHE_TABLE delayTop;						// ٱ񤭹ߥꥹȤƬ
static int delayLock;								// å󥯹ѥå
static uchar bitSects[] = {							// ȤΥӥåȥޥå
	0,0x1,0x3,0x7,0xf,0x1f,0x3f,0x7f,0xff
};


// ǥХ饭åɤ߹ࡣ
// return : 0 or error number
STATIC int readDevice(CACHE_TABLE *cacheTbl, uchar bitmap)
{
	//ɤ߹ɬפɤ߹ޤƤʤӥåȤΤߣˤ
	uchar map = ~bitmap | cacheTbl->bitmap;
	dev_t dev = getDevInfo(cacheTbl->inodeNum);
	int start, last, allLast;
	int rest;

	for (start = 0, allLast = CACHE_BLOCK_SIZE / dev->si_devsw->sector_size; start < allLast; ++start){
		if ((map & (1 << start)) == 0){
			for (last = start + 1; last < allLast; ++last){
				if (map & (1 << last)){
					break;
				}
			}
			rest = readDev(dev, cacheTbl->cache + start * CACHE_UNIT_SIZE,
				last - start, start + cacheTbl->block + getBegBlock(cacheTbl->inodeNum), 0);
			if (rest < 0){
				return rest;
			}
			start = last;
		}
	}

	cacheTbl->bitmap |= bitmap;
	
	return 0;
}


// å夫ǥХ˽᤹
// return : 0 or error number
STATIC int writeDevice(CACHE_TABLE *cacheTbl)
{
	dev_t dev = getDevInfo(cacheTbl->inodeNum);
	int start, last;
	int rest;

	if (cacheTbl->bitmap == 0xff){
		rest = writeDev(dev, cacheTbl->cache, CACHE_BLOCK_SIZE / CACHE_UNIT_SIZE, cacheTbl->block + getBegBlock(cacheTbl->inodeNum), 0);
		if (rest < 0){
			return rest;
		}
	}
	else{
		// սǥХ˽񤭹ࡣ
		for (start = 0; start < CACHE_BLOCK_SIZE / CACHE_UNIT_SIZE; ++start){
			if (cacheTbl->bitmap & (1 << start)){
				for (last = start + 1; last < CACHE_BLOCK_SIZE / CACHE_UNIT_SIZE; ++last){
					if ((cacheTbl->bitmap & (1 << last)) == 0){
						break;
					}
				}
				rest = writeDev(
					dev, cacheTbl->cache + start * CACHE_UNIT_SIZE, last - start, cacheTbl->block + start + getBegBlock(cacheTbl->inodeNum), 0);
				if (rest < 0){
					return rest;
				}
				start = last;
			}
		}
	}

	return 0;
}


// ǥХؤٱ񤭹ߥ
STATIC void delayWriteTask()
{
	PROC **q;
	CACHE_TABLE *cacheTbl = NULL;

	// ƥץڤΥ
	for (q = &writeTask->parent->child; *q != writeTask; q = &(*q)->brother){;
	}
	*q = writeTask->brother;
	
	// λ륷ʥ򤹤٤ƥޥ
	setSigAllMask();

	// å
	enter_spinlock(&delayLock);
	for (;;){
		// 񤭹ԤФ
		if (delayTop.writeNext == &delayTop){
			if (cacheTbl != NULL){
				cacheTbl->writing = NO;
			}
			del_from_schedule(TASK_WAIT, PROC_SCHEDULE_DEL);

			// å
			exit_spinlock(&delayLock);

			wait_task();

			// å
			enter_spinlock(&delayLock);
		}
		else{
			cacheTbl = delayTop.writePrev;
			cacheTbl->writeNext->writePrev = cacheTbl->writePrev;
			cacheTbl->writePrev->writeNext = cacheTbl->writeNext;
			cacheTbl->writeNext = cacheTbl->writePrev = cacheTbl;

			// å
			exit_spinlock(&delayLock);

			if (writeDevice(cacheTbl) < 0){
				printk("Device error in delay write function!\n");
			}

			// å
			enter_spinlock(&delayLock);
			
			if (cacheTbl->writeNext == cacheTbl){
				cacheTbl->writing = NO;
			}
		}
	}
}


//ադδؿ"cacheLock"Ƕ礷ʤؿƤӽФ뤳
// ٱ񤭹ߥꥹȤ˲ä롣
STATIC void addWriteList(CACHE_TABLE *cacheTbl)
{
	enter_spinlock(&delayLock);
	{
		// Ǥ˥󥯤³Ƥ⤢Τǰöǽ򤹤
		cacheTbl->writeNext->writePrev = cacheTbl->writePrev;
		cacheTbl->writePrev->writeNext = cacheTbl->writeNext;
		
		cacheTbl->writeNext = delayTop.writeNext;
		cacheTbl->writePrev = &delayTop;
		delayTop.writeNext->writePrev = cacheTbl;
		delayTop.writeNext = cacheTbl;
		cacheTbl->writing = YES;

		// ٱ񤭹ߥ򵯤
		add_to_schedule(writeTask, TASK_WAIT, NULL);
	}
	exit_spinlock(&delayLock);

}


// ϥåͤ׻롣
STATIC INLINE int calcHash(int value)
{
	return (value / 8) % HASH_NUM;
}


// åơ֥뤫鳺ǥХ֥åõ
// return : block addres or NULL
STATIC CACHE_TABLE *searchCache(int inodeNum, size_t begin)
{
	int hash = calcHash(begin);
	CACHE_TABLE *cacheFind;

	for (cacheFind = cache_hash[hash]; cacheFind != NULL; cacheFind = cacheFind->hashNext){
		// åˤ
		if ((cacheFind->block == begin) && (cacheFind->inodeNum == inodeNum)){
			// 󥯤ֿ֤ˤ
			if (cacheFind == cacheOldest){
				cacheOldest = cacheOldest->cachePrev;
			}
			else{
				cacheFind->cacheNext->cachePrev = cacheFind->cachePrev;
				cacheFind->cachePrev->cacheNext = cacheFind->cacheNext;
				cacheFind->cacheNext = cacheOldest->cacheNext;
				cacheFind->cachePrev = cacheOldest;
				cacheOldest->cacheNext->cachePrev = cacheFind;
				cacheOldest->cacheNext = cacheFind;
			}

			return cacheFind;
		}
	}

	return NULL;
}


// ˥å֥åƤ롣
// return : cache table
STATIC CACHE_TABLE *getNewCache(int inodeNum, size_t begin)
{
	int hash = calcHash(begin);
	CACHE_TABLE *newCache;

	// ָŤå򲣼
	newCache = cacheOldest;
	cacheOldest = cacheOldest->cachePrev;

	// ϥå󥯤
	newCache->hashPrev->hashNext = newCache->hashNext;
	if (newCache->hashNext != NULL){
		newCache->hashNext->hashPrev = newCache->hashPrev;
	}

	// ϥå󥯤˲ä
	newCache->hashNext = cache_hash[hash];
	newCache->hashPrev = (CACHE_TABLE*)&cache_hash[hash];
	if (cache_hash[hash] != NULL){
		cache_hash[hash]->hashPrev = newCache;
	}
	cache_hash[hash] = newCache;

	return newCache;
}


// ٱ񤭹ߥεư
STATIC void startDelayWriteTask()
{
	switch (sys_fork()){
		case -1:
			printk("Faild startDelayWriteTask\n");
			break;
		case 0:
			writeTask = get_current_task();
			delayWriteTask();
			// äƤʤ
	}
}


//================================== PROTECTED ==========================================


// ߤꥭå֥å򤹤٤ٱߤ롣
STATIC void writeBackAllCache()
{
	// ٱ񤭹Ԥ
	for (;;){
		enter_spinlock(&delayLock);
		if (delayTop.writeNext != &delayTop){
			exit_spinlock(&delayLock);
			wait_task();
		}
		else{
			exit_spinlock(&delayLock);
			break;
		}
	}
}


// return : 0 or Error number
STATIC int initCache()
{
	int i;

	// åơ֥ν
	for (i = 0; i < CACHE_NUM; ++i){
		memset(&cache_tbl[i], 0, sizeof(CACHE_TABLE));
		cache_tbl[i].cache = kmalloc(CACHE_BLOCK_SIZE);
		if (cache_tbl[i].cache == NULL){
			return -ENOMEM;
		}
		cache_tbl[i].cacheNext = &cache_tbl[i + 1];
		cache_tbl[i].cachePrev = &cache_tbl[i - 1];
		cache_tbl[i].hashPrev = &cache_tbl[i];
		cache_tbl[i].writeNext = cache_tbl[i].writePrev = &cache_tbl[i];
	}
	cache_tbl[CACHE_NUM - 1].cacheNext = &cache_tbl[0];
	cache_tbl[0].cachePrev = &cache_tbl[CACHE_NUM - 1];

	// ϥå󥯤ν
	for (i = 0; i < HASH_NUM; ++i){
		cache_hash[i] = NULL;
	}
	
	// ٱ񤭹ߥ󥯤ν
	delayTop.writeNext = delayTop.writePrev = &delayTop;
	
	return 0;
}


//================================== PUBLIC =============================================


//աճߥϥɥ餫θƤӽФԲ
// return : transfer sectors or error number
int read_cache(int inodeNum, void *buf, size_t sector_num, size_t begin)
{
	char *readBuf = (char*)buf;
	int sectSize = getDevInfo(inodeNum)->si_devsw->sector_size;	/* ǥХ */
	int cacheSects = CACHE_BLOCK_SIZE / sectSize;				/* å֥åˤϤ륻 */
	int restSects = sector_num;									/* Ĥž */
	int start, cacheStart, copySects;
	CACHE_TABLE *cacheTbl;

	ASSERT(0 <= restSects);

	/* ǥ¤Υå */
	if (getBlocks(inodeNum) < begin){
		return -EIO;
	}
	if (getBlocks(inodeNum) < begin + cacheSects){
		return -EIO;
	}
	start = begin % cacheSects;
	cacheStart = ROUNDDOWN(begin, cacheSects);
	copySects = (restSects <= cacheSects - start)? restSects : cacheSects - start;
	for(;;){
		enter_spinlock(&cacheLock);
		{
			// å򸡺
			if ((cacheTbl = searchCache(inodeNum, cacheStart)) != NULL){
				// åԤ
				while (cacheTbl->busy == YES){;
				}
			}
			else{
				cacheTbl = getNewCache(inodeNum, cacheStart);

				// åԤ
				while (cacheTbl->busy == YES){;
				}
				// ٱ񤭹Ԥ
				while (cacheTbl->writing == YES){;
				}

				cacheTbl->inodeNum = inodeNum;
				cacheTbl->block = cacheStart;
				cacheTbl->bitmap = 0;
			}

			// å
			cacheTbl->busy = YES;
		}
		exit_spinlock(&cacheLock);

		// å׵᥻ʤХǥХɤ߹
		if ((cacheTbl->bitmap & (bitSects[copySects] << start)) != (bitSects[copySects] << start)){
			if (readDevice(cacheTbl, bitSects[copySects] << start) < 0){
				// ---> åλ
				cacheTbl->busy = NO;
				return -EIO;
			}
		}

		// Хåե˥ԡ
		memcpy(readBuf, cacheTbl->cache + start * sectSize, copySects * sectSize);

		// åλ
		cacheTbl->busy = NO;

		if ((restSects -= copySects) == 0){
			break;
		}
		readBuf += copySects * sectSize;
		cacheStart += cacheSects;
		copySects = (restSects < cacheSects)? restSects : cacheSects;
		start = 0;
	}

	return sector_num;
}


/*
 *  ߥϥɥ餫θƤӽФԲ
 * return : transfer sectors or error number
 */
int write_cache(int inodeNum, void *wbuf, size_t sector_num, size_t begin)
{
	char *writeBuf = (char*)wbuf;
	int sectSize = getDevInfo(inodeNum)->si_devsw->sector_size;	/* ǥХ */
	int cacheSects = CACHE_BLOCK_SIZE / sectSize;				/* å֥åˤϤ륻 */
	int restSects = sector_num;									/* Ĥž */
	int start, cacheStart, copySects;
	CACHE_TABLE *cacheTbl;

	ASSERT(0 <= restSects);

	/* ǥ¤Υå */
	if (getBlocks(inodeNum) < begin){
		return -EIO;
	}
	if (getBlocks(inodeNum) < begin + cacheSects){
		return -EIO;
	}

	start = begin % cacheSects;
	cacheStart = ROUNDDOWN(begin, cacheSects);
	copySects = (restSects <= cacheSects - start)? restSects : cacheSects - start;
	for(;;){
		enter_spinlock(&cacheLock);
		{
			// å򸡺
			if ((cacheTbl = searchCache(inodeNum, cacheStart)) != NULL){
				// åԤ
				while (cacheTbl->busy == YES){;
				}
			}
			else{
				cacheTbl = getNewCache(inodeNum, cacheStart);

				// åԤ
				while (cacheTbl->busy == YES){;
				}
				// ٱ񤭹Ԥ
				while (cacheTbl->writing == YES){;
				}

				cacheTbl->inodeNum = inodeNum;
				cacheTbl->block = cacheStart;
				cacheTbl->bitmap = 0;
			}

			// å
			cacheTbl->busy = YES;
		}
		exit_spinlock(&cacheLock);

		// å˥ԡ
		memcpy(cacheTbl->cache + start * sectSize, writeBuf, copySects * sectSize);
		writeBuf += copySects * sectSize;
		cacheTbl->bitmap |= bitSects[copySects] << start;

		// ٱ񤭹
		addWriteList(cacheTbl);

		// åλ
		cacheTbl->busy = NO;

		if ((restSects -= copySects) == 0){
			break;
		}
		cacheStart += cacheSects;
		copySects = (restSects < cacheSects)? restSects : cacheSects;
		start = 0;
	}

	return sector_num;
}


int initBlockCache()
{
	/* IOå */
	initCache();

	/* ٱ񤭹ѥư롣 */
	startDelayWriteTask();

	return 0;
}


/************************************************************************************************
 *
 * ե륵ӥؿ
 *
 ************************************************************************************************/


STATIC int lookup(const char *path, const int inodeNum, void *inodeObj, uint *o_inodeNum)
{
	int rest;
	
	rest = lookupDevInode(path);
	if (rest < 0){
		return rest;
	}
	*o_inodeNum = rest;

	return NOERR;
}


STATIC int creat(const char *path, const int inodeNum, void *inodeObj, const int mode, uint *o_inodeNum)
{
	return -EOPNOTSUPP;
}


STATIC int open(const int din, const uint inodeNum, int oflag, void **o_inodeObj, int *o_type)
{
	dev_t dev = getDevInfo(inodeNum);
	
	// ǥХopen
	if (inodeNum != DEVICE_ROOT_INODE){
		if (dev->si_devsw->d_open(dev, oflag, 0, NULL) < 0){
			return -EIO;
		}
	}

	*o_inodeObj = (void*)inodeNum;
	*o_type = getMode(inodeNum) & S_IFMT;

	return NOERR;
}


STATIC int close(const int din, void *inodeObj)
{
	int inodeNum = (int)inodeObj;
	dev_t dev = getDevInfo(inodeNum);

	/* ǥХclose */
	if (inodeNum != DEVICE_ROOT_INODE){
		if (dev->si_devsw->d_close(dev, 0, 0, NULL) < 0){
			return -EIO;
		}
	}

	return NOERR;
}


STATIC int read(const int din, void *inodeObj, void *buf, size_t size, size_t begin)
{
	char *readBuf = buf;
	int inodeNum = (int)inodeObj;
	dev_t dev = getDevInfo(inodeNum);
	int rest;

	switch (getMode(inodeNum) & S_IFMT){
		/* 饯ǥХ */
		case S_IFCHR:
			return readDev(dev, readBuf, size, 0, 0);
		/* ֥åǥХ */
		case S_IFBLK:{
			char *sectBuf = NULL;
			int sectSize = getDevInfo(inodeNum)->si_devsw->sector_size;	/*  */
			int sectBeg = begin / sectSize;								/* žϤΥֹ */
			int restSize = size;										/* ĤžХȿ */
			int sects;													/*  */
			int bytes, restBytes;										/* Хȿ */

			/* ǽΥ̤ž */
			if ((bytes = begin % sectSize) != 0){
				if ((sectBuf = kmalloc(sectSize)) == NULL){
					return -ENOMEM;
				}
				rest = read_cache(inodeNum, sectBuf, 1, sectBeg);
				if (rest != 1){
					return rest;
				}
				restBytes = sectSize - bytes;
				if (size < restBytes){
					memcpy(readBuf, sectBuf + bytes, size);
					kfree(sectBuf);
					return size;
				}

				memcpy(readBuf, sectBuf + bytes, restBytes);
				readBuf += restBytes;
				sectBeg += 1;
				restSize -= restBytes;
			}

			sects = restSize / sectSize;
			rest = read_cache(inodeNum, readBuf, sects, sectBeg);
			if (rest < sects){
				kfree(sectBuf);
				return rest;
			}

			/* Ĥ꤬̤ž */
			if (sects * sectSize < restSize){
				bytes = sects * sectSize;
				if ((sectBuf == NULL) && ((sectBuf = kmalloc(sectSize)) == NULL)){
					return -ENOMEM;
				}
				if (read_cache(inodeNum, sectBuf, 1, sectBeg + sects) != 1){
					kfree(sectBuf);
					return size - restSize + bytes;
				}
				memcpy(readBuf + bytes, sectBuf, restSize - bytes);
			}

			kfree(sectBuf);
			
			return size;
		}
		/* root directory */
		case S_IFDIR:
			return 0;
		default:
			ASSERT(0);
			return -ENOENT;
	}
}


STATIC int write(const int din, void *inodeObj, void *buf, size_t size, size_t begin)
{
	char *writeBuf = buf;
	int inodeNum = (int)inodeObj;
	dev_t dev = getDevInfo(inodeNum);
	int rest;

	switch (getMode(inodeNum) & S_IFMT){
		/* 饯ǥХ */
		case S_IFCHR:
			return writeDev(dev, writeBuf, size, 0, 0);
		/* ֥åǥХ */
		case S_IFBLK:{
			char *sectBuf = NULL;
			int sectSize = getDevInfo(inodeNum)->si_devsw->sector_size;	/*  */
			int sectBeg = begin / sectSize;								/* žϤΥֹ */
			int restSize = size;										/* ĤžХȿ */
			int sects;													/*  */
			int bytes, restBytes;										/* Хȿ */

			/* ǽΥ̤ž */
			if ((bytes = begin % sectSize) != 0){
				if ((sectBuf = kmalloc(sectSize)) == NULL){
					return -ENOMEM;
				}
				rest = read_cache(inodeNum, sectBuf, 1, sectBeg);
				if (rest != 1){
					return rest;
				}
				if (size < (restBytes = sectSize - bytes)){
					memcpy(sectBuf + bytes, writeBuf, size);
					if (write_cache(inodeNum, sectBuf, 1, sectBeg) != 1){
						return 0;
					}
					kfree(sectBuf);
					return size;
				}
				memcpy(sectBuf + bytes, writeBuf, restBytes);
				if (write_cache(inodeNum, sectBuf, 1, sectBeg) != 1){
					return 0;
				}
				writeBuf += restBytes;
				sectBeg += 1;
				restSize -= restBytes;
			}

			sects = restSize / sectSize;
			rest = write_cache(inodeNum, writeBuf, sects, sectBeg);
			if (rest < sects){
				kfree(sectBuf);
				return rest;
			}

			/* Ĥ꤬̤ž */
			if (sects * sectSize < restSize){
				bytes = sects * sectSize;
				if ((sectBuf == NULL) && ((sectBuf = kmalloc(sectSize)) == NULL)){
					return -ENOMEM;
				}
				if (read_cache(inodeNum, sectBuf, 1, sectBeg + sects) != 1){
					kfree(sectBuf);
					return size - restSize + bytes;
				}
				memcpy(sectBuf, writeBuf + bytes, restSize - bytes);
				rest = write_cache(inodeNum, sectBuf, 1, sectBeg + sects);
				if (rest != 1){
					kfree(sectBuf);
					return rest;
				}
			}

			kfree(sectBuf);

			return size;
		}
		/* root directory */
		case S_IFDIR:
			return 0;
		default:
			ASSERT(0);
			return -ENOENT;
	}
}


STATIC int ioctl(const int din, void *inodeObj, int cmd, caddr_t param, int fflag)
{
	int inodeNum = (int)inodeObj;
	dev_t dev = getDevInfo(inodeNum);

	if (inodeNum == DEVICE_ROOT_INODE){
		return -ENOENT;
	}

	return dev->si_devsw->d_ioctl(dev, cmd, param, fflag, NULL);
}


STATIC int poll(void *inodeObj, int events)
{
	int inodeNum = (int)inodeObj;
	dev_t dev = getDevInfo(inodeNum);

	if (inodeNum == DEVICE_ROOT_INODE){
		return -ENOENT;
	}

	return dev->si_devsw->d_poll(dev, events, NULL);
}


STATIC int opendir(const char *path, const int din, void *dirInodeObj, void **o_inodeObj, uint *o_block, int *o_index)
{
	int inodeNum;

	/* ѥθ */
	inodeNum = lookupDevInode(path);
	if (inodeNum < 0){
		return inodeNum;
	}
	
	*o_inodeObj = (void*)inodeNum;
	*o_block = 0;
	*o_index = -1;

	return NOERR;
}


STATIC int readdir(int din, void *inodeObj, uint *m_block, int *m_index, char *o_name)
{
	const char *name;
	int nodeNum = (int)inodeObj;
	int nextInode;

	if (nodeNum != DEVICE_ROOT_INODE){
		return -ENOENT;
	}

	/* Υȥ꡼򸡺 */
	nextInode = getNextInode(*m_index);
	if (nextInode == ERR){
		return -ENOENT;
	}
	
	*m_index = nextInode;

	/* ̾ */
	name = getName(nextInode);
	strcpy(o_name, name);
	o_name[strlen(name)] = '\0';

	return 0;
}


STATIC int stat(int din, void *inodeObj, struct stat *m_stat)
{
	int inodeNum = (int)inodeObj;

	return getStat(inodeNum, m_stat);
}


STATIC int mount(const int din, void **o_rootInodeObj)
{
	return -EOPNOTSUPP;
}


STATIC int umount(const int din, void *rootInodeObj)
{
	return -EOPNOTSUPP;
}


STATIC int rename(int din, void *OldDirInodeObj, const char *oldPath, void *newDirInodeObj, const char *newPath)
{
	return -EOPNOTSUPP;
}


STATIC int mkdir(const char *path, const int din, void *dirInodeObj, int mode)
{
	return -EOPNOTSUPP;
}

STATIC int delete(const char *path, const int din, void *dirInodeObj, int type)
{
	return -EOPNOTSUPP;
}

STATIC int chattr(const int din, void *inodeObj, uint mode, uint uid, uint gid, uint atime, uint mtime)
{
	return -EOPNOTSUPP;
}

STATIC int link(const int din, void *dstInodeObj, void *srcInodeObj, const char *srcPath)
{
	return -EOPNOTSUPP;
}

STATIC int symlink(const char *filePath, const int din, void *dirInodeObj, char *linkPath)
{
	return -EOPNOTSUPP;
}

STATIC int statfs(const int din, struct statfs *m_statfs)
{
	return -EOPNOTSUPP;
}


static FS devFsFunc = {
	DEV_FS_NAME,
	mount,
	umount,
	lookup,
	creat,
	open,
	close,
	read,
	write,
	ioctl,
	poll,
	opendir,
	readdir,
	rename,
	mkdir,
	stat,
	delete,
	chattr,
	link,
	symlink,
	statfs,
};


/*
 * å夫ǥХ˥ǡ᤹
 */
int sys_sync()
{
	writeBackAllCache();

	return 0;
}


/*
 * եǥץü̾롣
 */
int sys_ttyname(int fdnum, char *buf)
{
	F_DSC *fd;
	dev_t dev;
	int inodeNum;
	const char *tty;

	if ((fd = getFd(fdnum)) == NULL){
		return -EBADF;
	}

	// ǥХե뤫
	if (fd->vent->din != 0){
		return -ENOTTY;
	}

	// struct cdevswΥ饹ȥ֥åʥС0Υ饯ǥХǧ롣
	inodeNum = (int)fd->vent->inodeObj;
	dev = getDevInfo(inodeNum);
	if (dev->si_devsw->sector_size != 0){
		return -ENOTTY;
	}

	tty = getName(inodeNum);
	strcpy(buf, tty);

	return 0;
}


/************************************************************************************************
 *
 * 
 *
 ************************************************************************************************/


//================================== PUBLIC =============================================


int initDeviceFs()
{
	/* ǥХե륷ƥν */
	initDevFs();
	
	return 0;
}


int mountDeviceFs()
{
	int rest;


	/* ե륷ƥϿ롣 */
	rest = regist_fs(&devFsFunc);
	if (rest < 0)
		return rest;

	/* ǥХե륷ƥޥȤ롣 */
	rest = mount_dev_fs(&devFsFunc);
	if (rest < 0)
		return rest;

	return 0;
}


/****************************************************************************/
void test_device()
{
	printk("buf=%x\n",*(uint*)0x158f58);
}
/*****************************************************************************/
